/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.qbusiness.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information on boosting <code>STRING</code> type document attributes.
 * </p>
 * <note>
 * <p>
 * For <code>STRING</code> and <code>STRING_LIST</code> type document attributes to be used for boosting on the console
 * and the API, they must be enabled for search using the <a
 * href="https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DocumentAttributeConfiguration.html"
 * >DocumentAttributeConfiguration</a> object of the <a
 * href="https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateIndex.html">UpdateIndex</a> API. If you
 * haven't enabled searching on these attributes, you can't boost attributes of these data types on either the console
 * or the API.
 * </p>
 * </note>
 * <p>
 * For more information on how boosting document attributes work in Amazon Q Business, see <a
 * href="https://docs.aws.amazon.com/amazonq/latest/business-use-dg/metadata-boosting.html">Boosting using document
 * attributes</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StringAttributeBoostingConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<StringAttributeBoostingConfiguration.Builder, StringAttributeBoostingConfiguration> {
    private static final SdkField<String> BOOSTING_LEVEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("boostingLevel").getter(getter(StringAttributeBoostingConfiguration::boostingLevelAsString))
            .setter(setter(Builder::boostingLevel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("boostingLevel").build()).build();

    private static final SdkField<Map<String, String>> ATTRIBUTE_VALUE_BOOSTING_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("attributeValueBoosting")
            .getter(getter(StringAttributeBoostingConfiguration::attributeValueBoostingAsStrings))
            .setter(setter(Builder::attributeValueBoostingWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("attributeValueBoosting").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BOOSTING_LEVEL_FIELD,
            ATTRIBUTE_VALUE_BOOSTING_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String boostingLevel;

    private final Map<String, String> attributeValueBoosting;

    private StringAttributeBoostingConfiguration(BuilderImpl builder) {
        this.boostingLevel = builder.boostingLevel;
        this.attributeValueBoosting = builder.attributeValueBoosting;
    }

    /**
     * <p>
     * Specifies the priority tier ranking of boosting applied to document attributes. For version 2, this parameter
     * indicates the relative ranking between boosted fields (ONE being highest priority, TWO being second highest,
     * etc.) and determines the order in which attributes influence document ranking in search results. For version 1,
     * this parameter specifies the boosting intensity. For version 2, boosting intensity (VERY HIGH, HIGH, MEDIUM, LOW,
     * NONE) are not supported. Note that in version 2, you are not allowed to boost on only one field and make this
     * value TWO.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #boostingLevel}
     * will return {@link DocumentAttributeBoostingLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #boostingLevelAsString}.
     * </p>
     * 
     * @return Specifies the priority tier ranking of boosting applied to document attributes. For version 2, this
     *         parameter indicates the relative ranking between boosted fields (ONE being highest priority, TWO being
     *         second highest, etc.) and determines the order in which attributes influence document ranking in search
     *         results. For version 1, this parameter specifies the boosting intensity. For version 2, boosting
     *         intensity (VERY HIGH, HIGH, MEDIUM, LOW, NONE) are not supported. Note that in version 2, you are not
     *         allowed to boost on only one field and make this value TWO.
     * @see DocumentAttributeBoostingLevel
     */
    public final DocumentAttributeBoostingLevel boostingLevel() {
        return DocumentAttributeBoostingLevel.fromValue(boostingLevel);
    }

    /**
     * <p>
     * Specifies the priority tier ranking of boosting applied to document attributes. For version 2, this parameter
     * indicates the relative ranking between boosted fields (ONE being highest priority, TWO being second highest,
     * etc.) and determines the order in which attributes influence document ranking in search results. For version 1,
     * this parameter specifies the boosting intensity. For version 2, boosting intensity (VERY HIGH, HIGH, MEDIUM, LOW,
     * NONE) are not supported. Note that in version 2, you are not allowed to boost on only one field and make this
     * value TWO.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #boostingLevel}
     * will return {@link DocumentAttributeBoostingLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #boostingLevelAsString}.
     * </p>
     * 
     * @return Specifies the priority tier ranking of boosting applied to document attributes. For version 2, this
     *         parameter indicates the relative ranking between boosted fields (ONE being highest priority, TWO being
     *         second highest, etc.) and determines the order in which attributes influence document ranking in search
     *         results. For version 1, this parameter specifies the boosting intensity. For version 2, boosting
     *         intensity (VERY HIGH, HIGH, MEDIUM, LOW, NONE) are not supported. Note that in version 2, you are not
     *         allowed to boost on only one field and make this value TWO.
     * @see DocumentAttributeBoostingLevel
     */
    public final String boostingLevelAsString() {
        return boostingLevel;
    }

    /**
     * <p>
     * Specifies specific values of a <code>STRING</code> type document attribute being boosted. When using
     * <code>NativeIndexConfiguration</code> version 2, you can specify up to five values in order of priority.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributeValueBoosting} method.
     * </p>
     * 
     * @return Specifies specific values of a <code>STRING</code> type document attribute being boosted. When using
     *         <code>NativeIndexConfiguration</code> version 2, you can specify up to five values in order of priority.
     */
    public final Map<String, StringAttributeValueBoostingLevel> attributeValueBoosting() {
        return StringAttributeValueBoostingCopier.copyStringToEnum(attributeValueBoosting);
    }

    /**
     * For responses, this returns true if the service returned a value for the AttributeValueBoosting property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAttributeValueBoosting() {
        return attributeValueBoosting != null && !(attributeValueBoosting instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies specific values of a <code>STRING</code> type document attribute being boosted. When using
     * <code>NativeIndexConfiguration</code> version 2, you can specify up to five values in order of priority.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributeValueBoosting} method.
     * </p>
     * 
     * @return Specifies specific values of a <code>STRING</code> type document attribute being boosted. When using
     *         <code>NativeIndexConfiguration</code> version 2, you can specify up to five values in order of priority.
     */
    public final Map<String, String> attributeValueBoostingAsStrings() {
        return attributeValueBoosting;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(boostingLevelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributeValueBoosting() ? attributeValueBoostingAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StringAttributeBoostingConfiguration)) {
            return false;
        }
        StringAttributeBoostingConfiguration other = (StringAttributeBoostingConfiguration) obj;
        return Objects.equals(boostingLevelAsString(), other.boostingLevelAsString())
                && hasAttributeValueBoosting() == other.hasAttributeValueBoosting()
                && Objects.equals(attributeValueBoostingAsStrings(), other.attributeValueBoostingAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StringAttributeBoostingConfiguration").add("BoostingLevel", boostingLevelAsString())
                .add("AttributeValueBoosting", hasAttributeValueBoosting() ? attributeValueBoostingAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "boostingLevel":
            return Optional.ofNullable(clazz.cast(boostingLevelAsString()));
        case "attributeValueBoosting":
            return Optional.ofNullable(clazz.cast(attributeValueBoostingAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("boostingLevel", BOOSTING_LEVEL_FIELD);
        map.put("attributeValueBoosting", ATTRIBUTE_VALUE_BOOSTING_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<StringAttributeBoostingConfiguration, T> g) {
        return obj -> g.apply((StringAttributeBoostingConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StringAttributeBoostingConfiguration> {
        /**
         * <p>
         * Specifies the priority tier ranking of boosting applied to document attributes. For version 2, this parameter
         * indicates the relative ranking between boosted fields (ONE being highest priority, TWO being second highest,
         * etc.) and determines the order in which attributes influence document ranking in search results. For version
         * 1, this parameter specifies the boosting intensity. For version 2, boosting intensity (VERY HIGH, HIGH,
         * MEDIUM, LOW, NONE) are not supported. Note that in version 2, you are not allowed to boost on only one field
         * and make this value TWO.
         * </p>
         * 
         * @param boostingLevel
         *        Specifies the priority tier ranking of boosting applied to document attributes. For version 2, this
         *        parameter indicates the relative ranking between boosted fields (ONE being highest priority, TWO being
         *        second highest, etc.) and determines the order in which attributes influence document ranking in
         *        search results. For version 1, this parameter specifies the boosting intensity. For version 2,
         *        boosting intensity (VERY HIGH, HIGH, MEDIUM, LOW, NONE) are not supported. Note that in version 2, you
         *        are not allowed to boost on only one field and make this value TWO.
         * @see DocumentAttributeBoostingLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentAttributeBoostingLevel
         */
        Builder boostingLevel(String boostingLevel);

        /**
         * <p>
         * Specifies the priority tier ranking of boosting applied to document attributes. For version 2, this parameter
         * indicates the relative ranking between boosted fields (ONE being highest priority, TWO being second highest,
         * etc.) and determines the order in which attributes influence document ranking in search results. For version
         * 1, this parameter specifies the boosting intensity. For version 2, boosting intensity (VERY HIGH, HIGH,
         * MEDIUM, LOW, NONE) are not supported. Note that in version 2, you are not allowed to boost on only one field
         * and make this value TWO.
         * </p>
         * 
         * @param boostingLevel
         *        Specifies the priority tier ranking of boosting applied to document attributes. For version 2, this
         *        parameter indicates the relative ranking between boosted fields (ONE being highest priority, TWO being
         *        second highest, etc.) and determines the order in which attributes influence document ranking in
         *        search results. For version 1, this parameter specifies the boosting intensity. For version 2,
         *        boosting intensity (VERY HIGH, HIGH, MEDIUM, LOW, NONE) are not supported. Note that in version 2, you
         *        are not allowed to boost on only one field and make this value TWO.
         * @see DocumentAttributeBoostingLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentAttributeBoostingLevel
         */
        Builder boostingLevel(DocumentAttributeBoostingLevel boostingLevel);

        /**
         * <p>
         * Specifies specific values of a <code>STRING</code> type document attribute being boosted. When using
         * <code>NativeIndexConfiguration</code> version 2, you can specify up to five values in order of priority.
         * </p>
         * 
         * @param attributeValueBoosting
         *        Specifies specific values of a <code>STRING</code> type document attribute being boosted. When using
         *        <code>NativeIndexConfiguration</code> version 2, you can specify up to five values in order of
         *        priority.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeValueBoostingWithStrings(Map<String, String> attributeValueBoosting);

        /**
         * <p>
         * Specifies specific values of a <code>STRING</code> type document attribute being boosted. When using
         * <code>NativeIndexConfiguration</code> version 2, you can specify up to five values in order of priority.
         * </p>
         * 
         * @param attributeValueBoosting
         *        Specifies specific values of a <code>STRING</code> type document attribute being boosted. When using
         *        <code>NativeIndexConfiguration</code> version 2, you can specify up to five values in order of
         *        priority.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeValueBoosting(Map<String, StringAttributeValueBoostingLevel> attributeValueBoosting);
    }

    static final class BuilderImpl implements Builder {
        private String boostingLevel;

        private Map<String, String> attributeValueBoosting = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(StringAttributeBoostingConfiguration model) {
            boostingLevel(model.boostingLevel);
            attributeValueBoostingWithStrings(model.attributeValueBoosting);
        }

        public final String getBoostingLevel() {
            return boostingLevel;
        }

        public final void setBoostingLevel(String boostingLevel) {
            this.boostingLevel = boostingLevel;
        }

        @Override
        public final Builder boostingLevel(String boostingLevel) {
            this.boostingLevel = boostingLevel;
            return this;
        }

        @Override
        public final Builder boostingLevel(DocumentAttributeBoostingLevel boostingLevel) {
            this.boostingLevel(boostingLevel == null ? null : boostingLevel.toString());
            return this;
        }

        public final Map<String, String> getAttributeValueBoosting() {
            if (attributeValueBoosting instanceof SdkAutoConstructMap) {
                return null;
            }
            return attributeValueBoosting;
        }

        public final void setAttributeValueBoosting(Map<String, String> attributeValueBoosting) {
            this.attributeValueBoosting = StringAttributeValueBoostingCopier.copy(attributeValueBoosting);
        }

        @Override
        public final Builder attributeValueBoostingWithStrings(Map<String, String> attributeValueBoosting) {
            this.attributeValueBoosting = StringAttributeValueBoostingCopier.copy(attributeValueBoosting);
            return this;
        }

        @Override
        public final Builder attributeValueBoosting(Map<String, StringAttributeValueBoostingLevel> attributeValueBoosting) {
            this.attributeValueBoosting = StringAttributeValueBoostingCopier.copyEnumToString(attributeValueBoosting);
            return this;
        }

        @Override
        public StringAttributeBoostingConfiguration build() {
            return new StringAttributeBoostingConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
