/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.qbusiness.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the authentication configuration details for a data accessor. This structure defines how the ISV
 * authenticates when accessing data through the data accessor.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataAccessorAuthenticationDetail implements SdkPojo, Serializable,
        ToCopyableBuilder<DataAccessorAuthenticationDetail.Builder, DataAccessorAuthenticationDetail> {
    private static final SdkField<String> AUTHENTICATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("authenticationType").getter(getter(DataAccessorAuthenticationDetail::authenticationTypeAsString))
            .setter(setter(Builder::authenticationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authenticationType").build())
            .build();

    private static final SdkField<DataAccessorAuthenticationConfiguration> AUTHENTICATION_CONFIGURATION_FIELD = SdkField
            .<DataAccessorAuthenticationConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("authenticationConfiguration")
            .getter(getter(DataAccessorAuthenticationDetail::authenticationConfiguration))
            .setter(setter(Builder::authenticationConfiguration))
            .constructor(DataAccessorAuthenticationConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authenticationConfiguration")
                    .build()).build();

    private static final SdkField<List<String>> EXTERNAL_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("externalIds")
            .getter(getter(DataAccessorAuthenticationDetail::externalIds))
            .setter(setter(Builder::externalIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("externalIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AUTHENTICATION_TYPE_FIELD,
            AUTHENTICATION_CONFIGURATION_FIELD, EXTERNAL_IDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String authenticationType;

    private final DataAccessorAuthenticationConfiguration authenticationConfiguration;

    private final List<String> externalIds;

    private DataAccessorAuthenticationDetail(BuilderImpl builder) {
        this.authenticationType = builder.authenticationType;
        this.authenticationConfiguration = builder.authenticationConfiguration;
        this.externalIds = builder.externalIds;
    }

    /**
     * <p>
     * The type of authentication to use for the data accessor. This determines how the ISV authenticates when accessing
     * data. You can use one of two authentication types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS_IAM_IDC_TTI</code> - Authentication using IAM Identity Center Trusted Token Issuer (TTI). This
     * authentication type allows the ISV to use a trusted token issuer to generate tokens for accessing the data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWS_IAM_IDC_AUTH_CODE</code> - Authentication using IAM Identity Center authorization code flow. This
     * authentication type uses the standard OAuth 2.0 authorization code flow for authentication.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #authenticationType} will return {@link DataAccessorAuthenticationType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #authenticationTypeAsString}.
     * </p>
     * 
     * @return The type of authentication to use for the data accessor. This determines how the ISV authenticates when
     *         accessing data. You can use one of two authentication types:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AWS_IAM_IDC_TTI</code> - Authentication using IAM Identity Center Trusted Token Issuer (TTI). This
     *         authentication type allows the ISV to use a trusted token issuer to generate tokens for accessing the
     *         data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWS_IAM_IDC_AUTH_CODE</code> - Authentication using IAM Identity Center authorization code flow.
     *         This authentication type uses the standard OAuth 2.0 authorization code flow for authentication.
     *         </p>
     *         </li>
     * @see DataAccessorAuthenticationType
     */
    public final DataAccessorAuthenticationType authenticationType() {
        return DataAccessorAuthenticationType.fromValue(authenticationType);
    }

    /**
     * <p>
     * The type of authentication to use for the data accessor. This determines how the ISV authenticates when accessing
     * data. You can use one of two authentication types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS_IAM_IDC_TTI</code> - Authentication using IAM Identity Center Trusted Token Issuer (TTI). This
     * authentication type allows the ISV to use a trusted token issuer to generate tokens for accessing the data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWS_IAM_IDC_AUTH_CODE</code> - Authentication using IAM Identity Center authorization code flow. This
     * authentication type uses the standard OAuth 2.0 authorization code flow for authentication.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #authenticationType} will return {@link DataAccessorAuthenticationType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #authenticationTypeAsString}.
     * </p>
     * 
     * @return The type of authentication to use for the data accessor. This determines how the ISV authenticates when
     *         accessing data. You can use one of two authentication types:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AWS_IAM_IDC_TTI</code> - Authentication using IAM Identity Center Trusted Token Issuer (TTI). This
     *         authentication type allows the ISV to use a trusted token issuer to generate tokens for accessing the
     *         data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWS_IAM_IDC_AUTH_CODE</code> - Authentication using IAM Identity Center authorization code flow.
     *         This authentication type uses the standard OAuth 2.0 authorization code flow for authentication.
     *         </p>
     *         </li>
     * @see DataAccessorAuthenticationType
     */
    public final String authenticationTypeAsString() {
        return authenticationType;
    }

    /**
     * <p>
     * The specific authentication configuration based on the authentication type.
     * </p>
     * 
     * @return The specific authentication configuration based on the authentication type.
     */
    public final DataAccessorAuthenticationConfiguration authenticationConfiguration() {
        return authenticationConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExternalIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExternalIds() {
        return externalIds != null && !(externalIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of external identifiers associated with this authentication configuration. These are used to correlate the
     * data accessor with external systems.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExternalIds} method.
     * </p>
     * 
     * @return A list of external identifiers associated with this authentication configuration. These are used to
     *         correlate the data accessor with external systems.
     */
    public final List<String> externalIds() {
        return externalIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(authenticationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(authenticationConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasExternalIds() ? externalIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataAccessorAuthenticationDetail)) {
            return false;
        }
        DataAccessorAuthenticationDetail other = (DataAccessorAuthenticationDetail) obj;
        return Objects.equals(authenticationTypeAsString(), other.authenticationTypeAsString())
                && Objects.equals(authenticationConfiguration(), other.authenticationConfiguration())
                && hasExternalIds() == other.hasExternalIds() && Objects.equals(externalIds(), other.externalIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataAccessorAuthenticationDetail").add("AuthenticationType", authenticationTypeAsString())
                .add("AuthenticationConfiguration", authenticationConfiguration())
                .add("ExternalIds", hasExternalIds() ? externalIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "authenticationType":
            return Optional.ofNullable(clazz.cast(authenticationTypeAsString()));
        case "authenticationConfiguration":
            return Optional.ofNullable(clazz.cast(authenticationConfiguration()));
        case "externalIds":
            return Optional.ofNullable(clazz.cast(externalIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("authenticationType", AUTHENTICATION_TYPE_FIELD);
        map.put("authenticationConfiguration", AUTHENTICATION_CONFIGURATION_FIELD);
        map.put("externalIds", EXTERNAL_IDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DataAccessorAuthenticationDetail, T> g) {
        return obj -> g.apply((DataAccessorAuthenticationDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataAccessorAuthenticationDetail> {
        /**
         * <p>
         * The type of authentication to use for the data accessor. This determines how the ISV authenticates when
         * accessing data. You can use one of two authentication types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AWS_IAM_IDC_TTI</code> - Authentication using IAM Identity Center Trusted Token Issuer (TTI). This
         * authentication type allows the ISV to use a trusted token issuer to generate tokens for accessing the data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWS_IAM_IDC_AUTH_CODE</code> - Authentication using IAM Identity Center authorization code flow. This
         * authentication type uses the standard OAuth 2.0 authorization code flow for authentication.
         * </p>
         * </li>
         * </ul>
         * 
         * @param authenticationType
         *        The type of authentication to use for the data accessor. This determines how the ISV authenticates
         *        when accessing data. You can use one of two authentication types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AWS_IAM_IDC_TTI</code> - Authentication using IAM Identity Center Trusted Token Issuer (TTI).
         *        This authentication type allows the ISV to use a trusted token issuer to generate tokens for accessing
         *        the data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWS_IAM_IDC_AUTH_CODE</code> - Authentication using IAM Identity Center authorization code flow.
         *        This authentication type uses the standard OAuth 2.0 authorization code flow for authentication.
         *        </p>
         *        </li>
         * @see DataAccessorAuthenticationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataAccessorAuthenticationType
         */
        Builder authenticationType(String authenticationType);

        /**
         * <p>
         * The type of authentication to use for the data accessor. This determines how the ISV authenticates when
         * accessing data. You can use one of two authentication types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AWS_IAM_IDC_TTI</code> - Authentication using IAM Identity Center Trusted Token Issuer (TTI). This
         * authentication type allows the ISV to use a trusted token issuer to generate tokens for accessing the data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWS_IAM_IDC_AUTH_CODE</code> - Authentication using IAM Identity Center authorization code flow. This
         * authentication type uses the standard OAuth 2.0 authorization code flow for authentication.
         * </p>
         * </li>
         * </ul>
         * 
         * @param authenticationType
         *        The type of authentication to use for the data accessor. This determines how the ISV authenticates
         *        when accessing data. You can use one of two authentication types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AWS_IAM_IDC_TTI</code> - Authentication using IAM Identity Center Trusted Token Issuer (TTI).
         *        This authentication type allows the ISV to use a trusted token issuer to generate tokens for accessing
         *        the data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWS_IAM_IDC_AUTH_CODE</code> - Authentication using IAM Identity Center authorization code flow.
         *        This authentication type uses the standard OAuth 2.0 authorization code flow for authentication.
         *        </p>
         *        </li>
         * @see DataAccessorAuthenticationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataAccessorAuthenticationType
         */
        Builder authenticationType(DataAccessorAuthenticationType authenticationType);

        /**
         * <p>
         * The specific authentication configuration based on the authentication type.
         * </p>
         * 
         * @param authenticationConfiguration
         *        The specific authentication configuration based on the authentication type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authenticationConfiguration(DataAccessorAuthenticationConfiguration authenticationConfiguration);

        /**
         * <p>
         * The specific authentication configuration based on the authentication type.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link DataAccessorAuthenticationConfiguration.Builder} avoiding the need to create one manually via
         * {@link DataAccessorAuthenticationConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataAccessorAuthenticationConfiguration.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #authenticationConfiguration(DataAccessorAuthenticationConfiguration)}.
         * 
         * @param authenticationConfiguration
         *        a consumer that will call methods on {@link DataAccessorAuthenticationConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #authenticationConfiguration(DataAccessorAuthenticationConfiguration)
         */
        default Builder authenticationConfiguration(
                Consumer<DataAccessorAuthenticationConfiguration.Builder> authenticationConfiguration) {
            return authenticationConfiguration(DataAccessorAuthenticationConfiguration.builder()
                    .applyMutation(authenticationConfiguration).build());
        }

        /**
         * <p>
         * A list of external identifiers associated with this authentication configuration. These are used to correlate
         * the data accessor with external systems.
         * </p>
         * 
         * @param externalIds
         *        A list of external identifiers associated with this authentication configuration. These are used to
         *        correlate the data accessor with external systems.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder externalIds(Collection<String> externalIds);

        /**
         * <p>
         * A list of external identifiers associated with this authentication configuration. These are used to correlate
         * the data accessor with external systems.
         * </p>
         * 
         * @param externalIds
         *        A list of external identifiers associated with this authentication configuration. These are used to
         *        correlate the data accessor with external systems.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder externalIds(String... externalIds);
    }

    static final class BuilderImpl implements Builder {
        private String authenticationType;

        private DataAccessorAuthenticationConfiguration authenticationConfiguration;

        private List<String> externalIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DataAccessorAuthenticationDetail model) {
            authenticationType(model.authenticationType);
            authenticationConfiguration(model.authenticationConfiguration);
            externalIds(model.externalIds);
        }

        public final String getAuthenticationType() {
            return authenticationType;
        }

        public final void setAuthenticationType(String authenticationType) {
            this.authenticationType = authenticationType;
        }

        @Override
        public final Builder authenticationType(String authenticationType) {
            this.authenticationType = authenticationType;
            return this;
        }

        @Override
        public final Builder authenticationType(DataAccessorAuthenticationType authenticationType) {
            this.authenticationType(authenticationType == null ? null : authenticationType.toString());
            return this;
        }

        public final DataAccessorAuthenticationConfiguration.Builder getAuthenticationConfiguration() {
            return authenticationConfiguration != null ? authenticationConfiguration.toBuilder() : null;
        }

        public final void setAuthenticationConfiguration(
                DataAccessorAuthenticationConfiguration.BuilderImpl authenticationConfiguration) {
            this.authenticationConfiguration = authenticationConfiguration != null ? authenticationConfiguration.build() : null;
        }

        @Override
        public final Builder authenticationConfiguration(DataAccessorAuthenticationConfiguration authenticationConfiguration) {
            this.authenticationConfiguration = authenticationConfiguration;
            return this;
        }

        public final Collection<String> getExternalIds() {
            if (externalIds instanceof SdkAutoConstructList) {
                return null;
            }
            return externalIds;
        }

        public final void setExternalIds(Collection<String> externalIds) {
            this.externalIds = DataAccessorExternalIdsCopier.copy(externalIds);
        }

        @Override
        public final Builder externalIds(Collection<String> externalIds) {
            this.externalIds = DataAccessorExternalIdsCopier.copy(externalIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder externalIds(String... externalIds) {
            externalIds(Arrays.asList(externalIds));
            return this;
        }

        @Override
        public DataAccessorAuthenticationDetail build() {
            return new DataAccessorAuthenticationDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
