/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.qconnect.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains filter strengths for harmful content. AI Guardrail's support the following content filters to detect and
 * filter harmful user inputs and FM-generated outputs.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>Hate</b>: Describes input prompts and model responses that discriminate, criticize, insult, denounce, or
 * dehumanize a person or group on the basis of an identity (such as race, ethnicity, gender, religion, sexual
 * orientation, ability, and national origin).
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Insults</b>: Describes input prompts and model responses that includes demeaning, humiliating, mocking, insulting,
 * or belittling language. This type of language is also labeled as bullying.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Sexual</b>: Describes input prompts and model responses that indicates sexual interest, activity, or arousal using
 * direct or indirect references to body parts, physical traits, or sex.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Violence</b>: Describes input prompts and model responses that includes glorification of, or threats to inflict
 * physical pain, hurt, or injury toward a person, group, or thing.
 * </p>
 * </li>
 * </ul>
 * <p>
 * Content filtering depends on the confidence classification of user inputs and FM responses across each of the four
 * harmful categories. All input and output statements are classified into one of four confidence levels (NONE, LOW,
 * MEDIUM, HIGH) for each harmful category. For example, if a statement is classified as <i>Hate</i> with HIGH
 * confidence, the likelihood of the statement representing hateful content is high. A single statement can be
 * classified across multiple categories with varying confidence levels. For example, a single statement can be
 * classified as <i>Hate</i> with HIGH confidence, <i> Insults</i> with LOW confidence, <i>Sexual</i> with NONE
 * confidence, and <i>Violence</i> with MEDIUM confidence.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GuardrailContentFilterConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<GuardrailContentFilterConfig.Builder, GuardrailContentFilterConfig> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(GuardrailContentFilterConfig::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> INPUT_STRENGTH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("inputStrength").getter(getter(GuardrailContentFilterConfig::inputStrengthAsString))
            .setter(setter(Builder::inputStrength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inputStrength").build()).build();

    private static final SdkField<String> OUTPUT_STRENGTH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("outputStrength").getter(getter(GuardrailContentFilterConfig::outputStrengthAsString))
            .setter(setter(Builder::outputStrength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("outputStrength").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD,
            INPUT_STRENGTH_FIELD, OUTPUT_STRENGTH_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String inputStrength;

    private final String outputStrength;

    private GuardrailContentFilterConfig(BuilderImpl builder) {
        this.type = builder.type;
        this.inputStrength = builder.inputStrength;
        this.outputStrength = builder.outputStrength;
    }

    /**
     * <p>
     * The harmful category that the content filter is applied to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link GuardrailContentFilterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #typeAsString}.
     * </p>
     * 
     * @return The harmful category that the content filter is applied to.
     * @see GuardrailContentFilterType
     */
    public final GuardrailContentFilterType type() {
        return GuardrailContentFilterType.fromValue(type);
    }

    /**
     * <p>
     * The harmful category that the content filter is applied to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link GuardrailContentFilterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #typeAsString}.
     * </p>
     * 
     * @return The harmful category that the content filter is applied to.
     * @see GuardrailContentFilterType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The strength of the content filter to apply to prompts. As you increase the filter strength, the likelihood of
     * filtering harmful content increases and the probability of seeing harmful content in your application reduces.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #inputStrength}
     * will return {@link GuardrailFilterStrength#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #inputStrengthAsString}.
     * </p>
     * 
     * @return The strength of the content filter to apply to prompts. As you increase the filter strength, the
     *         likelihood of filtering harmful content increases and the probability of seeing harmful content in your
     *         application reduces.
     * @see GuardrailFilterStrength
     */
    public final GuardrailFilterStrength inputStrength() {
        return GuardrailFilterStrength.fromValue(inputStrength);
    }

    /**
     * <p>
     * The strength of the content filter to apply to prompts. As you increase the filter strength, the likelihood of
     * filtering harmful content increases and the probability of seeing harmful content in your application reduces.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #inputStrength}
     * will return {@link GuardrailFilterStrength#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #inputStrengthAsString}.
     * </p>
     * 
     * @return The strength of the content filter to apply to prompts. As you increase the filter strength, the
     *         likelihood of filtering harmful content increases and the probability of seeing harmful content in your
     *         application reduces.
     * @see GuardrailFilterStrength
     */
    public final String inputStrengthAsString() {
        return inputStrength;
    }

    /**
     * <p>
     * The strength of the content filter to apply to model responses. As you increase the filter strength, the
     * likelihood of filtering harmful content increases and the probability of seeing harmful content in your
     * application reduces.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #outputStrength}
     * will return {@link GuardrailFilterStrength#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #outputStrengthAsString}.
     * </p>
     * 
     * @return The strength of the content filter to apply to model responses. As you increase the filter strength, the
     *         likelihood of filtering harmful content increases and the probability of seeing harmful content in your
     *         application reduces.
     * @see GuardrailFilterStrength
     */
    public final GuardrailFilterStrength outputStrength() {
        return GuardrailFilterStrength.fromValue(outputStrength);
    }

    /**
     * <p>
     * The strength of the content filter to apply to model responses. As you increase the filter strength, the
     * likelihood of filtering harmful content increases and the probability of seeing harmful content in your
     * application reduces.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #outputStrength}
     * will return {@link GuardrailFilterStrength#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #outputStrengthAsString}.
     * </p>
     * 
     * @return The strength of the content filter to apply to model responses. As you increase the filter strength, the
     *         likelihood of filtering harmful content increases and the probability of seeing harmful content in your
     *         application reduces.
     * @see GuardrailFilterStrength
     */
    public final String outputStrengthAsString() {
        return outputStrength;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(inputStrengthAsString());
        hashCode = 31 * hashCode + Objects.hashCode(outputStrengthAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GuardrailContentFilterConfig)) {
            return false;
        }
        GuardrailContentFilterConfig other = (GuardrailContentFilterConfig) obj;
        return Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(inputStrengthAsString(), other.inputStrengthAsString())
                && Objects.equals(outputStrengthAsString(), other.outputStrengthAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GuardrailContentFilterConfig")
                .add("Type", typeAsString() == null ? null : "*** Sensitive Data Redacted ***")
                .add("InputStrength", inputStrengthAsString() == null ? null : "*** Sensitive Data Redacted ***")
                .add("OutputStrength", outputStrengthAsString() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "inputStrength":
            return Optional.ofNullable(clazz.cast(inputStrengthAsString()));
        case "outputStrength":
            return Optional.ofNullable(clazz.cast(outputStrengthAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("type", TYPE_FIELD);
        map.put("inputStrength", INPUT_STRENGTH_FIELD);
        map.put("outputStrength", OUTPUT_STRENGTH_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GuardrailContentFilterConfig, T> g) {
        return obj -> g.apply((GuardrailContentFilterConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GuardrailContentFilterConfig> {
        /**
         * <p>
         * The harmful category that the content filter is applied to.
         * </p>
         * 
         * @param type
         *        The harmful category that the content filter is applied to.
         * @see GuardrailContentFilterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailContentFilterType
         */
        Builder type(String type);

        /**
         * <p>
         * The harmful category that the content filter is applied to.
         * </p>
         * 
         * @param type
         *        The harmful category that the content filter is applied to.
         * @see GuardrailContentFilterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailContentFilterType
         */
        Builder type(GuardrailContentFilterType type);

        /**
         * <p>
         * The strength of the content filter to apply to prompts. As you increase the filter strength, the likelihood
         * of filtering harmful content increases and the probability of seeing harmful content in your application
         * reduces.
         * </p>
         * 
         * @param inputStrength
         *        The strength of the content filter to apply to prompts. As you increase the filter strength, the
         *        likelihood of filtering harmful content increases and the probability of seeing harmful content in
         *        your application reduces.
         * @see GuardrailFilterStrength
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailFilterStrength
         */
        Builder inputStrength(String inputStrength);

        /**
         * <p>
         * The strength of the content filter to apply to prompts. As you increase the filter strength, the likelihood
         * of filtering harmful content increases and the probability of seeing harmful content in your application
         * reduces.
         * </p>
         * 
         * @param inputStrength
         *        The strength of the content filter to apply to prompts. As you increase the filter strength, the
         *        likelihood of filtering harmful content increases and the probability of seeing harmful content in
         *        your application reduces.
         * @see GuardrailFilterStrength
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailFilterStrength
         */
        Builder inputStrength(GuardrailFilterStrength inputStrength);

        /**
         * <p>
         * The strength of the content filter to apply to model responses. As you increase the filter strength, the
         * likelihood of filtering harmful content increases and the probability of seeing harmful content in your
         * application reduces.
         * </p>
         * 
         * @param outputStrength
         *        The strength of the content filter to apply to model responses. As you increase the filter strength,
         *        the likelihood of filtering harmful content increases and the probability of seeing harmful content in
         *        your application reduces.
         * @see GuardrailFilterStrength
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailFilterStrength
         */
        Builder outputStrength(String outputStrength);

        /**
         * <p>
         * The strength of the content filter to apply to model responses. As you increase the filter strength, the
         * likelihood of filtering harmful content increases and the probability of seeing harmful content in your
         * application reduces.
         * </p>
         * 
         * @param outputStrength
         *        The strength of the content filter to apply to model responses. As you increase the filter strength,
         *        the likelihood of filtering harmful content increases and the probability of seeing harmful content in
         *        your application reduces.
         * @see GuardrailFilterStrength
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailFilterStrength
         */
        Builder outputStrength(GuardrailFilterStrength outputStrength);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String inputStrength;

        private String outputStrength;

        private BuilderImpl() {
        }

        private BuilderImpl(GuardrailContentFilterConfig model) {
            type(model.type);
            inputStrength(model.inputStrength);
            outputStrength(model.outputStrength);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(GuardrailContentFilterType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getInputStrength() {
            return inputStrength;
        }

        public final void setInputStrength(String inputStrength) {
            this.inputStrength = inputStrength;
        }

        @Override
        public final Builder inputStrength(String inputStrength) {
            this.inputStrength = inputStrength;
            return this;
        }

        @Override
        public final Builder inputStrength(GuardrailFilterStrength inputStrength) {
            this.inputStrength(inputStrength == null ? null : inputStrength.toString());
            return this;
        }

        public final String getOutputStrength() {
            return outputStrength;
        }

        public final void setOutputStrength(String outputStrength) {
            this.outputStrength = outputStrength;
        }

        @Override
        public final Builder outputStrength(String outputStrength) {
            this.outputStrength = outputStrength;
            return this;
        }

        @Override
        public final Builder outputStrength(GuardrailFilterStrength outputStrength) {
            this.outputStrength(outputStrength == null ? null : outputStrength.toString());
            return this;
        }

        @Override
        public GuardrailContentFilterConfig build() {
            return new GuardrailContentFilterConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
