/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.qldb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The encryption settings that are used by a journal export job to write data in an Amazon Simple Storage Service
 * (Amazon S3) bucket.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3EncryptionConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<S3EncryptionConfiguration.Builder, S3EncryptionConfiguration> {
    private static final SdkField<String> OBJECT_ENCRYPTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3EncryptionConfiguration::objectEncryptionTypeAsString))
            .setter(setter(Builder::objectEncryptionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectEncryptionType").build())
            .build();

    private static final SdkField<String> KMS_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3EncryptionConfiguration::kmsKeyArn)).setter(setter(Builder::kmsKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OBJECT_ENCRYPTION_TYPE_FIELD,
            KMS_KEY_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String objectEncryptionType;

    private final String kmsKeyArn;

    private S3EncryptionConfiguration(BuilderImpl builder) {
        this.objectEncryptionType = builder.objectEncryptionType;
        this.kmsKeyArn = builder.kmsKeyArn;
    }

    /**
     * <p>
     * The Amazon S3 object encryption type.
     * </p>
     * <p>
     * To learn more about server-side encryption options in Amazon S3, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/serv-side-encryption.html">Protecting Data Using
     * Server-Side Encryption</a> in the <i>Amazon S3 Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #objectEncryptionType} will return {@link S3ObjectEncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #objectEncryptionTypeAsString}.
     * </p>
     * 
     * @return The Amazon S3 object encryption type.</p>
     *         <p>
     *         To learn more about server-side encryption options in Amazon S3, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/serv-side-encryption.html">Protecting Data Using
     *         Server-Side Encryption</a> in the <i>Amazon S3 Developer Guide</i>.
     * @see S3ObjectEncryptionType
     */
    public S3ObjectEncryptionType objectEncryptionType() {
        return S3ObjectEncryptionType.fromValue(objectEncryptionType);
    }

    /**
     * <p>
     * The Amazon S3 object encryption type.
     * </p>
     * <p>
     * To learn more about server-side encryption options in Amazon S3, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/serv-side-encryption.html">Protecting Data Using
     * Server-Side Encryption</a> in the <i>Amazon S3 Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #objectEncryptionType} will return {@link S3ObjectEncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #objectEncryptionTypeAsString}.
     * </p>
     * 
     * @return The Amazon S3 object encryption type.</p>
     *         <p>
     *         To learn more about server-side encryption options in Amazon S3, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/serv-side-encryption.html">Protecting Data Using
     *         Server-Side Encryption</a> in the <i>Amazon S3 Developer Guide</i>.
     * @see S3ObjectEncryptionType
     */
    public String objectEncryptionTypeAsString() {
        return objectEncryptionType;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for a customer master key (CMK) in AWS Key Management Service (AWS KMS).
     * </p>
     * <p>
     * You must provide a <code>KmsKeyArn</code> if you specify <code>SSE_KMS</code> as the
     * <code>ObjectEncryptionType</code>.
     * </p>
     * <p>
     * <code>KmsKeyArn</code> is not required if you specify <code>SSE_S3</code> as the
     * <code>ObjectEncryptionType</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for a customer master key (CMK) in AWS Key Management Service (AWS
     *         KMS).</p>
     *         <p>
     *         You must provide a <code>KmsKeyArn</code> if you specify <code>SSE_KMS</code> as the
     *         <code>ObjectEncryptionType</code>.
     *         </p>
     *         <p>
     *         <code>KmsKeyArn</code> is not required if you specify <code>SSE_S3</code> as the
     *         <code>ObjectEncryptionType</code>.
     */
    public String kmsKeyArn() {
        return kmsKeyArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(objectEncryptionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyArn());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3EncryptionConfiguration)) {
            return false;
        }
        S3EncryptionConfiguration other = (S3EncryptionConfiguration) obj;
        return Objects.equals(objectEncryptionTypeAsString(), other.objectEncryptionTypeAsString())
                && Objects.equals(kmsKeyArn(), other.kmsKeyArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("S3EncryptionConfiguration").add("ObjectEncryptionType", objectEncryptionTypeAsString())
                .add("KmsKeyArn", kmsKeyArn()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ObjectEncryptionType":
            return Optional.ofNullable(clazz.cast(objectEncryptionTypeAsString()));
        case "KmsKeyArn":
            return Optional.ofNullable(clazz.cast(kmsKeyArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3EncryptionConfiguration, T> g) {
        return obj -> g.apply((S3EncryptionConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3EncryptionConfiguration> {
        /**
         * <p>
         * The Amazon S3 object encryption type.
         * </p>
         * <p>
         * To learn more about server-side encryption options in Amazon S3, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/serv-side-encryption.html">Protecting Data Using
         * Server-Side Encryption</a> in the <i>Amazon S3 Developer Guide</i>.
         * </p>
         * 
         * @param objectEncryptionType
         *        The Amazon S3 object encryption type.</p>
         *        <p>
         *        To learn more about server-side encryption options in Amazon S3, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/serv-side-encryption.html">Protecting Data Using
         *        Server-Side Encryption</a> in the <i>Amazon S3 Developer Guide</i>.
         * @see S3ObjectEncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3ObjectEncryptionType
         */
        Builder objectEncryptionType(String objectEncryptionType);

        /**
         * <p>
         * The Amazon S3 object encryption type.
         * </p>
         * <p>
         * To learn more about server-side encryption options in Amazon S3, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/serv-side-encryption.html">Protecting Data Using
         * Server-Side Encryption</a> in the <i>Amazon S3 Developer Guide</i>.
         * </p>
         * 
         * @param objectEncryptionType
         *        The Amazon S3 object encryption type.</p>
         *        <p>
         *        To learn more about server-side encryption options in Amazon S3, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/serv-side-encryption.html">Protecting Data Using
         *        Server-Side Encryption</a> in the <i>Amazon S3 Developer Guide</i>.
         * @see S3ObjectEncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3ObjectEncryptionType
         */
        Builder objectEncryptionType(S3ObjectEncryptionType objectEncryptionType);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for a customer master key (CMK) in AWS Key Management Service (AWS KMS).
         * </p>
         * <p>
         * You must provide a <code>KmsKeyArn</code> if you specify <code>SSE_KMS</code> as the
         * <code>ObjectEncryptionType</code>.
         * </p>
         * <p>
         * <code>KmsKeyArn</code> is not required if you specify <code>SSE_S3</code> as the
         * <code>ObjectEncryptionType</code>.
         * </p>
         * 
         * @param kmsKeyArn
         *        The Amazon Resource Name (ARN) for a customer master key (CMK) in AWS Key Management Service (AWS
         *        KMS).</p>
         *        <p>
         *        You must provide a <code>KmsKeyArn</code> if you specify <code>SSE_KMS</code> as the
         *        <code>ObjectEncryptionType</code>.
         *        </p>
         *        <p>
         *        <code>KmsKeyArn</code> is not required if you specify <code>SSE_S3</code> as the
         *        <code>ObjectEncryptionType</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyArn(String kmsKeyArn);
    }

    static final class BuilderImpl implements Builder {
        private String objectEncryptionType;

        private String kmsKeyArn;

        private BuilderImpl() {
        }

        private BuilderImpl(S3EncryptionConfiguration model) {
            objectEncryptionType(model.objectEncryptionType);
            kmsKeyArn(model.kmsKeyArn);
        }

        public final String getObjectEncryptionTypeAsString() {
            return objectEncryptionType;
        }

        @Override
        public final Builder objectEncryptionType(String objectEncryptionType) {
            this.objectEncryptionType = objectEncryptionType;
            return this;
        }

        @Override
        public final Builder objectEncryptionType(S3ObjectEncryptionType objectEncryptionType) {
            this.objectEncryptionType(objectEncryptionType == null ? null : objectEncryptionType.toString());
            return this;
        }

        public final void setObjectEncryptionType(String objectEncryptionType) {
            this.objectEncryptionType = objectEncryptionType;
        }

        public final String getKmsKeyArn() {
            return kmsKeyArn;
        }

        @Override
        public final Builder kmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
            return this;
        }

        public final void setKmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
        }

        @Override
        public S3EncryptionConfiguration build() {
            return new S3EncryptionConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
