/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.qldb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Amazon Simple Storage Service (Amazon S3) bucket location in which a journal export job writes the journal
 * contents.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3ExportConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<S3ExportConfiguration.Builder, S3ExportConfiguration> {
    private static final SdkField<String> BUCKET_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Bucket")
            .getter(getter(S3ExportConfiguration::bucket)).setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Bucket").build()).build();

    private static final SdkField<String> PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Prefix")
            .getter(getter(S3ExportConfiguration::prefix)).setter(setter(Builder::prefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Prefix").build()).build();

    private static final SdkField<S3EncryptionConfiguration> ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<S3EncryptionConfiguration> builder(MarshallingType.SDK_POJO).memberName("EncryptionConfiguration")
            .getter(getter(S3ExportConfiguration::encryptionConfiguration)).setter(setter(Builder::encryptionConfiguration))
            .constructor(S3EncryptionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_FIELD, PREFIX_FIELD,
            ENCRYPTION_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String bucket;

    private final String prefix;

    private final S3EncryptionConfiguration encryptionConfiguration;

    private S3ExportConfiguration(BuilderImpl builder) {
        this.bucket = builder.bucket;
        this.prefix = builder.prefix;
        this.encryptionConfiguration = builder.encryptionConfiguration;
    }

    /**
     * <p>
     * The Amazon S3 bucket name in which a journal export job writes the journal contents.
     * </p>
     * <p>
     * The bucket name must comply with the Amazon S3 bucket naming conventions. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html">Bucket Restrictions and
     * Limitations</a> in the <i>Amazon S3 Developer Guide</i>.
     * </p>
     * 
     * @return The Amazon S3 bucket name in which a journal export job writes the journal contents.</p>
     *         <p>
     *         The bucket name must comply with the Amazon S3 bucket naming conventions. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html">Bucket Restrictions and
     *         Limitations</a> in the <i>Amazon S3 Developer Guide</i>.
     */
    public final String bucket() {
        return bucket;
    }

    /**
     * <p>
     * The prefix for the Amazon S3 bucket in which a journal export job writes the journal contents.
     * </p>
     * <p>
     * The prefix must comply with Amazon S3 key naming rules and restrictions. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html">Object Key and Metadata</a> in the
     * <i>Amazon S3 Developer Guide</i>.
     * </p>
     * <p>
     * The following are examples of valid <code>Prefix</code> values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>JournalExports-ForMyLedger/Testing/</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>JournalExports</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>My:Tests/</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The prefix for the Amazon S3 bucket in which a journal export job writes the journal contents.</p>
     *         <p>
     *         The prefix must comply with Amazon S3 key naming rules and restrictions. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html">Object Key and Metadata</a> in
     *         the <i>Amazon S3 Developer Guide</i>.
     *         </p>
     *         <p>
     *         The following are examples of valid <code>Prefix</code> values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>JournalExports-ForMyLedger/Testing/</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>JournalExports</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>My:Tests/</code>
     *         </p>
     *         </li>
     */
    public final String prefix() {
        return prefix;
    }

    /**
     * <p>
     * The encryption settings that are used by a journal export job to write data in an Amazon S3 bucket.
     * </p>
     * 
     * @return The encryption settings that are used by a journal export job to write data in an Amazon S3 bucket.
     */
    public final S3EncryptionConfiguration encryptionConfiguration() {
        return encryptionConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(prefix());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3ExportConfiguration)) {
            return false;
        }
        S3ExportConfiguration other = (S3ExportConfiguration) obj;
        return Objects.equals(bucket(), other.bucket()) && Objects.equals(prefix(), other.prefix())
                && Objects.equals(encryptionConfiguration(), other.encryptionConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3ExportConfiguration").add("Bucket", bucket()).add("Prefix", prefix())
                .add("EncryptionConfiguration", encryptionConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "Prefix":
            return Optional.ofNullable(clazz.cast(prefix()));
        case "EncryptionConfiguration":
            return Optional.ofNullable(clazz.cast(encryptionConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3ExportConfiguration, T> g) {
        return obj -> g.apply((S3ExportConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3ExportConfiguration> {
        /**
         * <p>
         * The Amazon S3 bucket name in which a journal export job writes the journal contents.
         * </p>
         * <p>
         * The bucket name must comply with the Amazon S3 bucket naming conventions. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html">Bucket Restrictions and
         * Limitations</a> in the <i>Amazon S3 Developer Guide</i>.
         * </p>
         * 
         * @param bucket
         *        The Amazon S3 bucket name in which a journal export job writes the journal contents.</p>
         *        <p>
         *        The bucket name must comply with the Amazon S3 bucket naming conventions. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html">Bucket Restrictions and
         *        Limitations</a> in the <i>Amazon S3 Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * The prefix for the Amazon S3 bucket in which a journal export job writes the journal contents.
         * </p>
         * <p>
         * The prefix must comply with Amazon S3 key naming rules and restrictions. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html">Object Key and Metadata</a> in the
         * <i>Amazon S3 Developer Guide</i>.
         * </p>
         * <p>
         * The following are examples of valid <code>Prefix</code> values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>JournalExports-ForMyLedger/Testing/</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>JournalExports</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>My:Tests/</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param prefix
         *        The prefix for the Amazon S3 bucket in which a journal export job writes the journal contents.</p>
         *        <p>
         *        The prefix must comply with Amazon S3 key naming rules and restrictions. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html">Object Key and Metadata</a>
         *        in the <i>Amazon S3 Developer Guide</i>.
         *        </p>
         *        <p>
         *        The following are examples of valid <code>Prefix</code> values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>JournalExports-ForMyLedger/Testing/</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>JournalExports</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>My:Tests/</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder prefix(String prefix);

        /**
         * <p>
         * The encryption settings that are used by a journal export job to write data in an Amazon S3 bucket.
         * </p>
         * 
         * @param encryptionConfiguration
         *        The encryption settings that are used by a journal export job to write data in an Amazon S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfiguration(S3EncryptionConfiguration encryptionConfiguration);

        /**
         * <p>
         * The encryption settings that are used by a journal export job to write data in an Amazon S3 bucket.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3EncryptionConfiguration.Builder}
         * avoiding the need to create one manually via {@link S3EncryptionConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3EncryptionConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #encryptionConfiguration(S3EncryptionConfiguration)}.
         * 
         * @param encryptionConfiguration
         *        a consumer that will call methods on {@link S3EncryptionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionConfiguration(S3EncryptionConfiguration)
         */
        default Builder encryptionConfiguration(Consumer<S3EncryptionConfiguration.Builder> encryptionConfiguration) {
            return encryptionConfiguration(S3EncryptionConfiguration.builder().applyMutation(encryptionConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String bucket;

        private String prefix;

        private S3EncryptionConfiguration encryptionConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(S3ExportConfiguration model) {
            bucket(model.bucket);
            prefix(model.prefix);
            encryptionConfiguration(model.encryptionConfiguration);
        }

        public final String getBucket() {
            return bucket;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final String getPrefix() {
            return prefix;
        }

        public final void setPrefix(String prefix) {
            this.prefix = prefix;
        }

        @Override
        public final Builder prefix(String prefix) {
            this.prefix = prefix;
            return this;
        }

        public final S3EncryptionConfiguration.Builder getEncryptionConfiguration() {
            return encryptionConfiguration != null ? encryptionConfiguration.toBuilder() : null;
        }

        public final void setEncryptionConfiguration(S3EncryptionConfiguration.BuilderImpl encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration != null ? encryptionConfiguration.build() : null;
        }

        @Override
        public final Builder encryptionConfiguration(S3EncryptionConfiguration encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration;
            return this;
        }

        @Override
        public S3ExportConfiguration build() {
            return new S3ExportConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
