/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.qldb;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.qldb.model.CancelJournalKinesisStreamRequest;
import software.amazon.awssdk.services.qldb.model.CancelJournalKinesisStreamResponse;
import software.amazon.awssdk.services.qldb.model.CreateLedgerRequest;
import software.amazon.awssdk.services.qldb.model.CreateLedgerResponse;
import software.amazon.awssdk.services.qldb.model.DeleteLedgerRequest;
import software.amazon.awssdk.services.qldb.model.DeleteLedgerResponse;
import software.amazon.awssdk.services.qldb.model.DescribeJournalKinesisStreamRequest;
import software.amazon.awssdk.services.qldb.model.DescribeJournalKinesisStreamResponse;
import software.amazon.awssdk.services.qldb.model.DescribeJournalS3ExportRequest;
import software.amazon.awssdk.services.qldb.model.DescribeJournalS3ExportResponse;
import software.amazon.awssdk.services.qldb.model.DescribeLedgerRequest;
import software.amazon.awssdk.services.qldb.model.DescribeLedgerResponse;
import software.amazon.awssdk.services.qldb.model.ExportJournalToS3Request;
import software.amazon.awssdk.services.qldb.model.ExportJournalToS3Response;
import software.amazon.awssdk.services.qldb.model.GetBlockRequest;
import software.amazon.awssdk.services.qldb.model.GetBlockResponse;
import software.amazon.awssdk.services.qldb.model.GetDigestRequest;
import software.amazon.awssdk.services.qldb.model.GetDigestResponse;
import software.amazon.awssdk.services.qldb.model.GetRevisionRequest;
import software.amazon.awssdk.services.qldb.model.GetRevisionResponse;
import software.amazon.awssdk.services.qldb.model.InvalidParameterException;
import software.amazon.awssdk.services.qldb.model.LimitExceededException;
import software.amazon.awssdk.services.qldb.model.ListJournalKinesisStreamsForLedgerRequest;
import software.amazon.awssdk.services.qldb.model.ListJournalKinesisStreamsForLedgerResponse;
import software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerRequest;
import software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerResponse;
import software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsRequest;
import software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsResponse;
import software.amazon.awssdk.services.qldb.model.ListLedgersRequest;
import software.amazon.awssdk.services.qldb.model.ListLedgersResponse;
import software.amazon.awssdk.services.qldb.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.qldb.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.qldb.model.QldbException;
import software.amazon.awssdk.services.qldb.model.QldbRequest;
import software.amazon.awssdk.services.qldb.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.qldb.model.ResourceInUseException;
import software.amazon.awssdk.services.qldb.model.ResourceNotFoundException;
import software.amazon.awssdk.services.qldb.model.ResourcePreconditionNotMetException;
import software.amazon.awssdk.services.qldb.model.StreamJournalToKinesisRequest;
import software.amazon.awssdk.services.qldb.model.StreamJournalToKinesisResponse;
import software.amazon.awssdk.services.qldb.model.TagResourceRequest;
import software.amazon.awssdk.services.qldb.model.TagResourceResponse;
import software.amazon.awssdk.services.qldb.model.UntagResourceRequest;
import software.amazon.awssdk.services.qldb.model.UntagResourceResponse;
import software.amazon.awssdk.services.qldb.model.UpdateLedgerPermissionsModeRequest;
import software.amazon.awssdk.services.qldb.model.UpdateLedgerPermissionsModeResponse;
import software.amazon.awssdk.services.qldb.model.UpdateLedgerRequest;
import software.amazon.awssdk.services.qldb.model.UpdateLedgerResponse;
import software.amazon.awssdk.services.qldb.paginators.ListJournalKinesisStreamsForLedgerIterable;
import software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsForLedgerIterable;
import software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsIterable;
import software.amazon.awssdk.services.qldb.paginators.ListLedgersIterable;
import software.amazon.awssdk.services.qldb.transform.CancelJournalKinesisStreamRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.CreateLedgerRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.DeleteLedgerRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.DescribeJournalKinesisStreamRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.DescribeJournalS3ExportRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.DescribeLedgerRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.ExportJournalToS3RequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.GetBlockRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.GetDigestRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.GetRevisionRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.ListJournalKinesisStreamsForLedgerRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.ListJournalS3ExportsForLedgerRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.ListJournalS3ExportsRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.ListLedgersRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.StreamJournalToKinesisRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.UpdateLedgerPermissionsModeRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.UpdateLedgerRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link QldbClient}.
 *
 * @see QldbClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultQldbClient implements QldbClient {
    private static final Logger log = Logger.loggerFor(DefaultQldbClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultQldbClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Ends a given Amazon QLDB journal stream. Before a stream can be canceled, its current status must be
     * <code>ACTIVE</code>.
     * </p>
     * <p>
     * You can't restart a stream after you cancel it. Canceled QLDB stream resources are subject to a 7-day retention
     * period, so they are automatically deleted after this limit expires.
     * </p>
     *
     * @param cancelJournalKinesisStreamRequest
     * @return Result of the CancelJournalKinesisStream operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourcePreconditionNotMetException
     *         The operation failed because a condition wasn't satisfied in advance.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.CancelJournalKinesisStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/CancelJournalKinesisStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CancelJournalKinesisStreamResponse cancelJournalKinesisStream(
            CancelJournalKinesisStreamRequest cancelJournalKinesisStreamRequest) throws InvalidParameterException,
            ResourceNotFoundException, ResourcePreconditionNotMetException, AwsServiceException, SdkClientException,
            QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelJournalKinesisStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CancelJournalKinesisStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJournalKinesisStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJournalKinesisStream");

            return clientHandler
                    .execute(new ClientExecutionParams<CancelJournalKinesisStreamRequest, CancelJournalKinesisStreamResponse>()
                            .withOperationName("CancelJournalKinesisStream").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(cancelJournalKinesisStreamRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CancelJournalKinesisStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new ledger in your Amazon Web Services account in the current Region.
     * </p>
     *
     * @param createLedgerRequest
     * @return Result of the CreateLedger operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceAlreadyExistsException
     *         The specified resource already exists.
     * @throws LimitExceededException
     *         You have reached the limit on the maximum number of resources allowed.
     * @throws ResourceInUseException
     *         The specified resource can't be modified at this time.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.CreateLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/CreateLedger" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateLedgerResponse createLedger(CreateLedgerRequest createLedgerRequest) throws InvalidParameterException,
            ResourceAlreadyExistsException, LimitExceededException, ResourceInUseException, AwsServiceException,
            SdkClientException, QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateLedgerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateLedgerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLedgerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLedger");

            return clientHandler.execute(new ClientExecutionParams<CreateLedgerRequest, CreateLedgerResponse>()
                    .withOperationName("CreateLedger").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createLedgerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateLedgerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a ledger and all of its contents. This action is irreversible.
     * </p>
     * <p>
     * If deletion protection is enabled, you must first disable it before you can delete the ledger. You can disable it
     * by calling the <code>UpdateLedger</code> operation to set the flag to <code>false</code>.
     * </p>
     *
     * @param deleteLedgerRequest
     * @return Result of the DeleteLedger operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourceInUseException
     *         The specified resource can't be modified at this time.
     * @throws ResourcePreconditionNotMetException
     *         The operation failed because a condition wasn't satisfied in advance.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.DeleteLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/DeleteLedger" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteLedgerResponse deleteLedger(DeleteLedgerRequest deleteLedgerRequest) throws InvalidParameterException,
            ResourceNotFoundException, ResourceInUseException, ResourcePreconditionNotMetException, AwsServiceException,
            SdkClientException, QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLedgerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteLedgerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLedgerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLedger");

            return clientHandler.execute(new ClientExecutionParams<DeleteLedgerRequest, DeleteLedgerResponse>()
                    .withOperationName("DeleteLedger").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteLedgerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteLedgerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns detailed information about a given Amazon QLDB journal stream. The output includes the Amazon Resource
     * Name (ARN), stream name, current status, creation time, and the parameters of the original stream creation
     * request.
     * </p>
     * <p>
     * This action does not return any expired journal streams. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/streams.create.html#streams.create.states.expiration"
     * >Expiration for terminal streams</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     *
     * @param describeJournalKinesisStreamRequest
     * @return Result of the DescribeJournalKinesisStream operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourcePreconditionNotMetException
     *         The operation failed because a condition wasn't satisfied in advance.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.DescribeJournalKinesisStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/DescribeJournalKinesisStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeJournalKinesisStreamResponse describeJournalKinesisStream(
            DescribeJournalKinesisStreamRequest describeJournalKinesisStreamRequest) throws InvalidParameterException,
            ResourceNotFoundException, ResourcePreconditionNotMetException, AwsServiceException, SdkClientException,
            QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeJournalKinesisStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeJournalKinesisStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJournalKinesisStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJournalKinesisStream");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeJournalKinesisStreamRequest, DescribeJournalKinesisStreamResponse>()
                            .withOperationName("DescribeJournalKinesisStream").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeJournalKinesisStreamRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeJournalKinesisStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a journal export job, including the ledger name, export ID, creation time, current
     * status, and the parameters of the original export creation request.
     * </p>
     * <p>
     * This action does not return any expired export jobs. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/export-journal.request.html#export-journal.request.expiration"
     * >Export job expiration</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * <p>
     * If the export job with the given <code>ExportId</code> doesn't exist, then throws
     * <code>ResourceNotFoundException</code>.
     * </p>
     * <p>
     * If the ledger with the given <code>Name</code> doesn't exist, then throws <code>ResourceNotFoundException</code>.
     * </p>
     *
     * @param describeJournalS3ExportRequest
     * @return Result of the DescribeJournalS3Export operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.DescribeJournalS3Export
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/DescribeJournalS3Export" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeJournalS3ExportResponse describeJournalS3Export(DescribeJournalS3ExportRequest describeJournalS3ExportRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeJournalS3ExportResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeJournalS3ExportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJournalS3ExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJournalS3Export");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeJournalS3ExportRequest, DescribeJournalS3ExportResponse>()
                            .withOperationName("DescribeJournalS3Export").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeJournalS3ExportRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeJournalS3ExportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a ledger, including its state, permissions mode, encryption at rest settings, and when
     * it was created.
     * </p>
     *
     * @param describeLedgerRequest
     * @return Result of the DescribeLedger operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.DescribeLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/DescribeLedger" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeLedgerResponse describeLedger(DescribeLedgerRequest describeLedgerRequest) throws InvalidParameterException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeLedgerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeLedgerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLedgerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLedger");

            return clientHandler.execute(new ClientExecutionParams<DescribeLedgerRequest, DescribeLedgerResponse>()
                    .withOperationName("DescribeLedger").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeLedgerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeLedgerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Exports journal contents within a date and time range from a ledger into a specified Amazon Simple Storage
     * Service (Amazon S3) bucket. A journal export job can write the data objects in either the text or binary
     * representation of Amazon Ion format, or in <i>JSON Lines</i> text format.
     * </p>
     * <p>
     * In JSON Lines format, each journal block in the exported data object is a valid JSON object that is delimited by
     * a newline. You can use this format to easily integrate JSON exports with analytics tools such as Glue and Amazon
     * Athena because these services can parse newline-delimited JSON automatically. For more information about the
     * format, see <a href="https://jsonlines.org/">JSON Lines</a>.
     * </p>
     * <p>
     * If the ledger with the given <code>Name</code> doesn't exist, then throws <code>ResourceNotFoundException</code>.
     * </p>
     * <p>
     * If the ledger with the given <code>Name</code> is in <code>CREATING</code> status, then throws
     * <code>ResourcePreconditionNotMetException</code>.
     * </p>
     * <p>
     * You can initiate up to two concurrent journal export requests for each ledger. Beyond this limit, journal export
     * requests throw <code>LimitExceededException</code>.
     * </p>
     *
     * @param exportJournalToS3Request
     * @return Result of the ExportJournalToS3 operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourcePreconditionNotMetException
     *         The operation failed because a condition wasn't satisfied in advance.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.ExportJournalToS3
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ExportJournalToS3" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ExportJournalToS3Response exportJournalToS3(ExportJournalToS3Request exportJournalToS3Request)
            throws ResourceNotFoundException, ResourcePreconditionNotMetException, AwsServiceException, SdkClientException,
            QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExportJournalToS3Response> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ExportJournalToS3Response::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportJournalToS3Request
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportJournalToS3");

            return clientHandler.execute(new ClientExecutionParams<ExportJournalToS3Request, ExportJournalToS3Response>()
                    .withOperationName("ExportJournalToS3").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(exportJournalToS3Request)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExportJournalToS3RequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a block object at a specified address in a journal. Also returns a proof of the specified block for
     * verification if <code>DigestTipAddress</code> is provided.
     * </p>
     * <p>
     * For information about the data contents in a block, see <a
     * href="https://docs.aws.amazon.com/qldb/latest/developerguide/journal-contents.html">Journal contents</a> in the
     * <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * <p>
     * If the specified ledger doesn't exist or is in <code>DELETING</code> status, then throws
     * <code>ResourceNotFoundException</code>.
     * </p>
     * <p>
     * If the specified ledger is in <code>CREATING</code> status, then throws
     * <code>ResourcePreconditionNotMetException</code>.
     * </p>
     * <p>
     * If no block exists with the specified address, then throws <code>InvalidParameterException</code>.
     * </p>
     *
     * @param getBlockRequest
     * @return Result of the GetBlock operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourcePreconditionNotMetException
     *         The operation failed because a condition wasn't satisfied in advance.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.GetBlock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/GetBlock" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetBlockResponse getBlock(GetBlockRequest getBlockRequest) throws InvalidParameterException,
            ResourceNotFoundException, ResourcePreconditionNotMetException, AwsServiceException, SdkClientException,
            QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBlockResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetBlockResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBlockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBlock");

            return clientHandler.execute(new ClientExecutionParams<GetBlockRequest, GetBlockResponse>()
                    .withOperationName("GetBlock").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getBlockRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetBlockRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the digest of a ledger at the latest committed block in the journal. The response includes a 256-bit hash
     * value and a block address.
     * </p>
     *
     * @param getDigestRequest
     * @return Result of the GetDigest operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourcePreconditionNotMetException
     *         The operation failed because a condition wasn't satisfied in advance.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.GetDigest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/GetDigest" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDigestResponse getDigest(GetDigestRequest getDigestRequest) throws InvalidParameterException,
            ResourceNotFoundException, ResourcePreconditionNotMetException, AwsServiceException, SdkClientException,
            QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDigestResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDigestResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDigestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDigest");

            return clientHandler.execute(new ClientExecutionParams<GetDigestRequest, GetDigestResponse>()
                    .withOperationName("GetDigest").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDigestRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetDigestRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a revision data object for a specified document ID and block address. Also returns a proof of the
     * specified revision for verification if <code>DigestTipAddress</code> is provided.
     * </p>
     *
     * @param getRevisionRequest
     * @return Result of the GetRevision operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourcePreconditionNotMetException
     *         The operation failed because a condition wasn't satisfied in advance.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.GetRevision
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/GetRevision" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetRevisionResponse getRevision(GetRevisionRequest getRevisionRequest) throws InvalidParameterException,
            ResourceNotFoundException, ResourcePreconditionNotMetException, AwsServiceException, SdkClientException,
            QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRevisionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetRevisionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRevisionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRevision");

            return clientHandler.execute(new ClientExecutionParams<GetRevisionRequest, GetRevisionResponse>()
                    .withOperationName("GetRevision").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getRevisionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRevisionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of all Amazon QLDB journal stream descriptors for a given ledger. The output of each stream
     * descriptor includes the same details that are returned by <code>DescribeJournalKinesisStream</code>.
     * </p>
     * <p>
     * This action does not return any expired journal streams. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/streams.create.html#streams.create.states.expiration"
     * >Expiration for terminal streams</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * <p>
     * This action returns a maximum of <code>MaxResults</code> items. It is paginated so that you can retrieve all the
     * items by calling <code>ListJournalKinesisStreamsForLedger</code> multiple times.
     * </p>
     *
     * @param listJournalKinesisStreamsForLedgerRequest
     * @return Result of the ListJournalKinesisStreamsForLedger operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourcePreconditionNotMetException
     *         The operation failed because a condition wasn't satisfied in advance.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.ListJournalKinesisStreamsForLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListJournalKinesisStreamsForLedger"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListJournalKinesisStreamsForLedgerResponse listJournalKinesisStreamsForLedger(
            ListJournalKinesisStreamsForLedgerRequest listJournalKinesisStreamsForLedgerRequest)
            throws InvalidParameterException, ResourceNotFoundException, ResourcePreconditionNotMetException,
            AwsServiceException, SdkClientException, QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListJournalKinesisStreamsForLedgerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListJournalKinesisStreamsForLedgerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listJournalKinesisStreamsForLedgerRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJournalKinesisStreamsForLedger");

            return clientHandler
                    .execute(new ClientExecutionParams<ListJournalKinesisStreamsForLedgerRequest, ListJournalKinesisStreamsForLedgerResponse>()
                            .withOperationName("ListJournalKinesisStreamsForLedger").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listJournalKinesisStreamsForLedgerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListJournalKinesisStreamsForLedgerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of all Amazon QLDB journal stream descriptors for a given ledger. The output of each stream
     * descriptor includes the same details that are returned by <code>DescribeJournalKinesisStream</code>.
     * </p>
     * <p>
     * This action does not return any expired journal streams. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/streams.create.html#streams.create.states.expiration"
     * >Expiration for terminal streams</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * <p>
     * This action returns a maximum of <code>MaxResults</code> items. It is paginated so that you can retrieve all the
     * items by calling <code>ListJournalKinesisStreamsForLedger</code> multiple times.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listJournalKinesisStreamsForLedger(software.amazon.awssdk.services.qldb.model.ListJournalKinesisStreamsForLedgerRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListJournalKinesisStreamsForLedgerIterable responses = client.listJournalKinesisStreamsForLedgerPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.qldb.paginators.ListJournalKinesisStreamsForLedgerIterable responses = client
     *             .listJournalKinesisStreamsForLedgerPaginator(request);
     *     for (software.amazon.awssdk.services.qldb.model.ListJournalKinesisStreamsForLedgerResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListJournalKinesisStreamsForLedgerIterable responses = client.listJournalKinesisStreamsForLedgerPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJournalKinesisStreamsForLedger(software.amazon.awssdk.services.qldb.model.ListJournalKinesisStreamsForLedgerRequest)}
     * operation.</b>
     * </p>
     *
     * @param listJournalKinesisStreamsForLedgerRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourcePreconditionNotMetException
     *         The operation failed because a condition wasn't satisfied in advance.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.ListJournalKinesisStreamsForLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListJournalKinesisStreamsForLedger"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListJournalKinesisStreamsForLedgerIterable listJournalKinesisStreamsForLedgerPaginator(
            ListJournalKinesisStreamsForLedgerRequest listJournalKinesisStreamsForLedgerRequest)
            throws InvalidParameterException, ResourceNotFoundException, ResourcePreconditionNotMetException,
            AwsServiceException, SdkClientException, QldbException {
        return new ListJournalKinesisStreamsForLedgerIterable(this,
                applyPaginatorUserAgent(listJournalKinesisStreamsForLedgerRequest));
    }

    /**
     * <p>
     * Returns an array of journal export job descriptions for all ledgers that are associated with the current Amazon
     * Web Services account and Region.
     * </p>
     * <p>
     * This action returns a maximum of <code>MaxResults</code> items, and is paginated so that you can retrieve all the
     * items by calling <code>ListJournalS3Exports</code> multiple times.
     * </p>
     * <p>
     * This action does not return any expired export jobs. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/export-journal.request.html#export-journal.request.expiration"
     * >Export job expiration</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     *
     * @param listJournalS3ExportsRequest
     * @return Result of the ListJournalS3Exports operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.ListJournalS3Exports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListJournalS3Exports" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJournalS3ExportsResponse listJournalS3Exports(ListJournalS3ExportsRequest listJournalS3ExportsRequest)
            throws AwsServiceException, SdkClientException, QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListJournalS3ExportsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListJournalS3ExportsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJournalS3ExportsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJournalS3Exports");

            return clientHandler.execute(new ClientExecutionParams<ListJournalS3ExportsRequest, ListJournalS3ExportsResponse>()
                    .withOperationName("ListJournalS3Exports").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listJournalS3ExportsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListJournalS3ExportsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of journal export job descriptions for all ledgers that are associated with the current Amazon
     * Web Services account and Region.
     * </p>
     * <p>
     * This action returns a maximum of <code>MaxResults</code> items, and is paginated so that you can retrieve all the
     * items by calling <code>ListJournalS3Exports</code> multiple times.
     * </p>
     * <p>
     * This action does not return any expired export jobs. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/export-journal.request.html#export-journal.request.expiration"
     * >Export job expiration</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listJournalS3Exports(software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsIterable responses = client.listJournalS3ExportsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsIterable responses = client
     *             .listJournalS3ExportsPaginator(request);
     *     for (software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsIterable responses = client.listJournalS3ExportsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJournalS3Exports(software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listJournalS3ExportsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.ListJournalS3Exports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListJournalS3Exports" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJournalS3ExportsIterable listJournalS3ExportsPaginator(ListJournalS3ExportsRequest listJournalS3ExportsRequest)
            throws AwsServiceException, SdkClientException, QldbException {
        return new ListJournalS3ExportsIterable(this, applyPaginatorUserAgent(listJournalS3ExportsRequest));
    }

    /**
     * <p>
     * Returns an array of journal export job descriptions for a specified ledger.
     * </p>
     * <p>
     * This action returns a maximum of <code>MaxResults</code> items, and is paginated so that you can retrieve all the
     * items by calling <code>ListJournalS3ExportsForLedger</code> multiple times.
     * </p>
     * <p>
     * This action does not return any expired export jobs. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/export-journal.request.html#export-journal.request.expiration"
     * >Export job expiration</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     *
     * @param listJournalS3ExportsForLedgerRequest
     * @return Result of the ListJournalS3ExportsForLedger operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.ListJournalS3ExportsForLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListJournalS3ExportsForLedger"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListJournalS3ExportsForLedgerResponse listJournalS3ExportsForLedger(
            ListJournalS3ExportsForLedgerRequest listJournalS3ExportsForLedgerRequest) throws AwsServiceException,
            SdkClientException, QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListJournalS3ExportsForLedgerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListJournalS3ExportsForLedgerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listJournalS3ExportsForLedgerRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJournalS3ExportsForLedger");

            return clientHandler
                    .execute(new ClientExecutionParams<ListJournalS3ExportsForLedgerRequest, ListJournalS3ExportsForLedgerResponse>()
                            .withOperationName("ListJournalS3ExportsForLedger").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listJournalS3ExportsForLedgerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListJournalS3ExportsForLedgerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of journal export job descriptions for a specified ledger.
     * </p>
     * <p>
     * This action returns a maximum of <code>MaxResults</code> items, and is paginated so that you can retrieve all the
     * items by calling <code>ListJournalS3ExportsForLedger</code> multiple times.
     * </p>
     * <p>
     * This action does not return any expired export jobs. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/export-journal.request.html#export-journal.request.expiration"
     * >Export job expiration</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listJournalS3ExportsForLedger(software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsForLedgerIterable responses = client.listJournalS3ExportsForLedgerPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsForLedgerIterable responses = client
     *             .listJournalS3ExportsForLedgerPaginator(request);
     *     for (software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsForLedgerIterable responses = client.listJournalS3ExportsForLedgerPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJournalS3ExportsForLedger(software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerRequest)}
     * operation.</b>
     * </p>
     *
     * @param listJournalS3ExportsForLedgerRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.ListJournalS3ExportsForLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListJournalS3ExportsForLedger"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListJournalS3ExportsForLedgerIterable listJournalS3ExportsForLedgerPaginator(
            ListJournalS3ExportsForLedgerRequest listJournalS3ExportsForLedgerRequest) throws AwsServiceException,
            SdkClientException, QldbException {
        return new ListJournalS3ExportsForLedgerIterable(this, applyPaginatorUserAgent(listJournalS3ExportsForLedgerRequest));
    }

    /**
     * <p>
     * Returns an array of ledger summaries that are associated with the current Amazon Web Services account and Region.
     * </p>
     * <p>
     * This action returns a maximum of 100 items and is paginated so that you can retrieve all the items by calling
     * <code>ListLedgers</code> multiple times.
     * </p>
     *
     * @param listLedgersRequest
     * @return Result of the ListLedgers operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.ListLedgers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListLedgers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListLedgersResponse listLedgers(ListLedgersRequest listLedgersRequest) throws AwsServiceException, SdkClientException,
            QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLedgersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListLedgersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLedgersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLedgers");

            return clientHandler.execute(new ClientExecutionParams<ListLedgersRequest, ListLedgersResponse>()
                    .withOperationName("ListLedgers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listLedgersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListLedgersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of ledger summaries that are associated with the current Amazon Web Services account and Region.
     * </p>
     * <p>
     * This action returns a maximum of 100 items and is paginated so that you can retrieve all the items by calling
     * <code>ListLedgers</code> multiple times.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listLedgers(software.amazon.awssdk.services.qldb.model.ListLedgersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListLedgersIterable responses = client.listLedgersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.qldb.paginators.ListLedgersIterable responses = client.listLedgersPaginator(request);
     *     for (software.amazon.awssdk.services.qldb.model.ListLedgersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListLedgersIterable responses = client.listLedgersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLedgers(software.amazon.awssdk.services.qldb.model.ListLedgersRequest)} operation.</b>
     * </p>
     *
     * @param listLedgersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.ListLedgers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListLedgers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListLedgersIterable listLedgersPaginator(ListLedgersRequest listLedgersRequest) throws AwsServiceException,
            SdkClientException, QldbException {
        return new ListLedgersIterable(this, applyPaginatorUserAgent(listLedgersRequest));
    }

    /**
     * <p>
     * Returns all tags for a specified Amazon QLDB resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidParameterException, ResourceNotFoundException, AwsServiceException, SdkClientException, QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a journal stream for a given Amazon QLDB ledger. The stream captures every document revision that is
     * committed to the ledger's journal and delivers the data to a specified Amazon Kinesis Data Streams resource.
     * </p>
     *
     * @param streamJournalToKinesisRequest
     * @return Result of the StreamJournalToKinesis operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourcePreconditionNotMetException
     *         The operation failed because a condition wasn't satisfied in advance.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.StreamJournalToKinesis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/StreamJournalToKinesis" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StreamJournalToKinesisResponse streamJournalToKinesis(StreamJournalToKinesisRequest streamJournalToKinesisRequest)
            throws InvalidParameterException, ResourceNotFoundException, ResourcePreconditionNotMetException,
            AwsServiceException, SdkClientException, QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StreamJournalToKinesisResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StreamJournalToKinesisResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, streamJournalToKinesisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StreamJournalToKinesis");

            return clientHandler
                    .execute(new ClientExecutionParams<StreamJournalToKinesisRequest, StreamJournalToKinesisResponse>()
                            .withOperationName("StreamJournalToKinesis").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(streamJournalToKinesisRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StreamJournalToKinesisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more tags to a specified Amazon QLDB resource.
     * </p>
     * <p>
     * A resource can have up to 50 tags. If you try to create more than 50 tags for a resource, your request fails and
     * returns an error.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidParameterException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from a specified Amazon QLDB resource. You can specify up to 50 tag keys to remove.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidParameterException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates properties on a ledger.
     * </p>
     *
     * @param updateLedgerRequest
     * @return Result of the UpdateLedger operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.UpdateLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/UpdateLedger" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateLedgerResponse updateLedger(UpdateLedgerRequest updateLedgerRequest) throws InvalidParameterException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateLedgerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateLedgerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLedgerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLedger");

            return clientHandler.execute(new ClientExecutionParams<UpdateLedgerRequest, UpdateLedgerResponse>()
                    .withOperationName("UpdateLedger").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateLedgerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateLedgerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the permissions mode of a ledger.
     * </p>
     * <important>
     * <p>
     * Before you switch to the <code>STANDARD</code> permissions mode, you must first create all required IAM policies
     * and table tags to avoid disruption to your users. To learn more, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/ledger-management.basics.html#ledger-mgmt.basics.update-permissions.migrating"
     * >Migrating to the standard permissions mode</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * </important>
     *
     * @param updateLedgerPermissionsModeRequest
     * @return Result of the UpdateLedgerPermissionsMode operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in the request aren't valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QldbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QldbClient.UpdateLedgerPermissionsMode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/UpdateLedgerPermissionsMode"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateLedgerPermissionsModeResponse updateLedgerPermissionsMode(
            UpdateLedgerPermissionsModeRequest updateLedgerPermissionsModeRequest) throws InvalidParameterException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, QldbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateLedgerPermissionsModeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateLedgerPermissionsModeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLedgerPermissionsModeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLedgerPermissionsMode");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateLedgerPermissionsModeRequest, UpdateLedgerPermissionsModeResponse>()
                            .withOperationName("UpdateLedgerPermissionsMode").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateLedgerPermissionsModeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateLedgerPermissionsModeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(QldbException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourcePreconditionNotMetException")
                                .exceptionBuilderSupplier(ResourcePreconditionNotMetException::builder).httpStatusCode(412)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends QldbRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
