/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.qldb.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateLedgerRequest extends QldbRequest implements
        ToCopyableBuilder<CreateLedgerRequest.Builder, CreateLedgerRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CreateLedgerRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(CreateLedgerRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> PERMISSIONS_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PermissionsMode").getter(getter(CreateLedgerRequest::permissionsModeAsString))
            .setter(setter(Builder::permissionsMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PermissionsMode").build()).build();

    private static final SdkField<Boolean> DELETION_PROTECTION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DeletionProtection").getter(getter(CreateLedgerRequest::deletionProtection))
            .setter(setter(Builder::deletionProtection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeletionProtection").build())
            .build();

    private static final SdkField<String> KMS_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("KmsKey")
            .getter(getter(CreateLedgerRequest::kmsKey)).setter(setter(Builder::kmsKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKey").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, TAGS_FIELD,
            PERMISSIONS_MODE_FIELD, DELETION_PROTECTION_FIELD, KMS_KEY_FIELD));

    private final String name;

    private final Map<String, String> tags;

    private final String permissionsMode;

    private final Boolean deletionProtection;

    private final String kmsKey;

    private CreateLedgerRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.tags = builder.tags;
        this.permissionsMode = builder.permissionsMode;
        this.deletionProtection = builder.deletionProtection;
        this.kmsKey = builder.kmsKey;
    }

    /**
     * <p>
     * The name of the ledger that you want to create. The name must be unique among all of the ledgers in your Amazon
     * Web Services account in the current Region.
     * </p>
     * <p>
     * Naming constraints for ledger names are defined in <a
     * href="https://docs.aws.amazon.com/qldb/latest/developerguide/limits.html#limits.naming">Quotas in Amazon QLDB</a>
     * in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * 
     * @return The name of the ledger that you want to create. The name must be unique among all of the ledgers in your
     *         Amazon Web Services account in the current Region.</p>
     *         <p>
     *         Naming constraints for ledger names are defined in <a
     *         href="https://docs.aws.amazon.com/qldb/latest/developerguide/limits.html#limits.naming">Quotas in Amazon
     *         QLDB</a> in the <i>Amazon QLDB Developer Guide</i>.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The key-value pairs to add as tags to the ledger that you want to create. Tag keys are case sensitive. Tag values
     * are case sensitive and can be null.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The key-value pairs to add as tags to the ledger that you want to create. Tag keys are case sensitive.
     *         Tag values are case sensitive and can be null.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The permissions mode to assign to the ledger that you want to create. This parameter can have one of the
     * following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ALLOW_ALL</code>: A legacy permissions mode that enables access control with API-level granularity for
     * ledgers.
     * </p>
     * <p>
     * This mode allows users who have the <code>SendCommand</code> API permission for this ledger to run all PartiQL
     * commands (hence, <code>ALLOW_ALL</code>) on any tables in the specified ledger. This mode disregards any
     * table-level or command-level IAM permissions policies that you create for the ledger.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STANDARD</code>: (<i>Recommended</i>) A permissions mode that enables access control with finer granularity
     * for ledgers, tables, and PartiQL commands.
     * </p>
     * <p>
     * By default, this mode denies all user requests to run any PartiQL commands on any tables in this ledger. To allow
     * PartiQL commands to run, you must create IAM permissions policies for specific table resources and PartiQL
     * actions, in addition to the <code>SendCommand</code> API permission for the ledger. For information, see <a
     * href="https://docs.aws.amazon.com/qldb/latest/developerguide/getting-started-standard-mode.html">Getting started
     * with the standard permissions mode</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * We strongly recommend using the <code>STANDARD</code> permissions mode to maximize the security of your ledger
     * data.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permissionsMode}
     * will return {@link PermissionsMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #permissionsModeAsString}.
     * </p>
     * 
     * @return The permissions mode to assign to the ledger that you want to create. This parameter can have one of the
     *         following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ALLOW_ALL</code>: A legacy permissions mode that enables access control with API-level granularity
     *         for ledgers.
     *         </p>
     *         <p>
     *         This mode allows users who have the <code>SendCommand</code> API permission for this ledger to run all
     *         PartiQL commands (hence, <code>ALLOW_ALL</code>) on any tables in the specified ledger. This mode
     *         disregards any table-level or command-level IAM permissions policies that you create for the ledger.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STANDARD</code>: (<i>Recommended</i>) A permissions mode that enables access control with finer
     *         granularity for ledgers, tables, and PartiQL commands.
     *         </p>
     *         <p>
     *         By default, this mode denies all user requests to run any PartiQL commands on any tables in this ledger.
     *         To allow PartiQL commands to run, you must create IAM permissions policies for specific table resources
     *         and PartiQL actions, in addition to the <code>SendCommand</code> API permission for the ledger. For
     *         information, see <a
     *         href="https://docs.aws.amazon.com/qldb/latest/developerguide/getting-started-standard-mode.html">Getting
     *         started with the standard permissions mode</a> in the <i>Amazon QLDB Developer Guide</i>.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         We strongly recommend using the <code>STANDARD</code> permissions mode to maximize the security of your
     *         ledger data.
     *         </p>
     * @see PermissionsMode
     */
    public final PermissionsMode permissionsMode() {
        return PermissionsMode.fromValue(permissionsMode);
    }

    /**
     * <p>
     * The permissions mode to assign to the ledger that you want to create. This parameter can have one of the
     * following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ALLOW_ALL</code>: A legacy permissions mode that enables access control with API-level granularity for
     * ledgers.
     * </p>
     * <p>
     * This mode allows users who have the <code>SendCommand</code> API permission for this ledger to run all PartiQL
     * commands (hence, <code>ALLOW_ALL</code>) on any tables in the specified ledger. This mode disregards any
     * table-level or command-level IAM permissions policies that you create for the ledger.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STANDARD</code>: (<i>Recommended</i>) A permissions mode that enables access control with finer granularity
     * for ledgers, tables, and PartiQL commands.
     * </p>
     * <p>
     * By default, this mode denies all user requests to run any PartiQL commands on any tables in this ledger. To allow
     * PartiQL commands to run, you must create IAM permissions policies for specific table resources and PartiQL
     * actions, in addition to the <code>SendCommand</code> API permission for the ledger. For information, see <a
     * href="https://docs.aws.amazon.com/qldb/latest/developerguide/getting-started-standard-mode.html">Getting started
     * with the standard permissions mode</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * We strongly recommend using the <code>STANDARD</code> permissions mode to maximize the security of your ledger
     * data.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permissionsMode}
     * will return {@link PermissionsMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #permissionsModeAsString}.
     * </p>
     * 
     * @return The permissions mode to assign to the ledger that you want to create. This parameter can have one of the
     *         following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ALLOW_ALL</code>: A legacy permissions mode that enables access control with API-level granularity
     *         for ledgers.
     *         </p>
     *         <p>
     *         This mode allows users who have the <code>SendCommand</code> API permission for this ledger to run all
     *         PartiQL commands (hence, <code>ALLOW_ALL</code>) on any tables in the specified ledger. This mode
     *         disregards any table-level or command-level IAM permissions policies that you create for the ledger.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STANDARD</code>: (<i>Recommended</i>) A permissions mode that enables access control with finer
     *         granularity for ledgers, tables, and PartiQL commands.
     *         </p>
     *         <p>
     *         By default, this mode denies all user requests to run any PartiQL commands on any tables in this ledger.
     *         To allow PartiQL commands to run, you must create IAM permissions policies for specific table resources
     *         and PartiQL actions, in addition to the <code>SendCommand</code> API permission for the ledger. For
     *         information, see <a
     *         href="https://docs.aws.amazon.com/qldb/latest/developerguide/getting-started-standard-mode.html">Getting
     *         started with the standard permissions mode</a> in the <i>Amazon QLDB Developer Guide</i>.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         We strongly recommend using the <code>STANDARD</code> permissions mode to maximize the security of your
     *         ledger data.
     *         </p>
     * @see PermissionsMode
     */
    public final String permissionsModeAsString() {
        return permissionsMode;
    }

    /**
     * <p>
     * The flag that prevents a ledger from being deleted by any user. If not provided on ledger creation, this feature
     * is enabled (<code>true</code>) by default.
     * </p>
     * <p>
     * If deletion protection is enabled, you must first disable it before you can delete the ledger. You can disable it
     * by calling the <code>UpdateLedger</code> operation to set the flag to <code>false</code>.
     * </p>
     * 
     * @return The flag that prevents a ledger from being deleted by any user. If not provided on ledger creation, this
     *         feature is enabled (<code>true</code>) by default.</p>
     *         <p>
     *         If deletion protection is enabled, you must first disable it before you can delete the ledger. You can
     *         disable it by calling the <code>UpdateLedger</code> operation to set the flag to <code>false</code>.
     */
    public final Boolean deletionProtection() {
        return deletionProtection;
    }

    /**
     * <p>
     * The key in Key Management Service (KMS) to use for encryption of data at rest in the ledger. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/qldb/latest/developerguide/encryption-at-rest.html">Encryption at rest</a> in
     * the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * <p>
     * Use one of the following options to specify this parameter:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS_OWNED_KMS_KEY</code>: Use an KMS key that is owned and managed by Amazon Web Services on your behalf.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Undefined</b>: By default, use an Amazon Web Services owned KMS key.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>A valid symmetric customer managed KMS key</b>: Use the specified KMS key in your account that you create,
     * own, and manage.
     * </p>
     * <p>
     * Amazon QLDB does not support asymmetric keys. For more information, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/symmetric-asymmetric.html">Using symmetric and
     * asymmetric keys</a> in the <i>Key Management Service Developer Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To specify a customer managed KMS key, you can use its key ID, Amazon Resource Name (ARN), alias name, or alias
     * ARN. When using an alias name, prefix it with <code>"alias/"</code>. To specify a key in a different Amazon Web
     * Services account, you must use the key ARN or alias ARN.
     * </p>
     * <p>
     * For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name: <code>alias/ExampleAlias</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#key-id">Key identifiers (KeyId)</a> in
     * the <i>Key Management Service Developer Guide</i>.
     * </p>
     * 
     * @return The key in Key Management Service (KMS) to use for encryption of data at rest in the ledger. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/qldb/latest/developerguide/encryption-at-rest.html">Encryption at
     *         rest</a> in the <i>Amazon QLDB Developer Guide</i>.</p>
     *         <p>
     *         Use one of the following options to specify this parameter:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AWS_OWNED_KMS_KEY</code>: Use an KMS key that is owned and managed by Amazon Web Services on your
     *         behalf.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Undefined</b>: By default, use an Amazon Web Services owned KMS key.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>A valid symmetric customer managed KMS key</b>: Use the specified KMS key in your account that you
     *         create, own, and manage.
     *         </p>
     *         <p>
     *         Amazon QLDB does not support asymmetric keys. For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/symmetric-asymmetric.html">Using symmetric
     *         and asymmetric keys</a> in the <i>Key Management Service Developer Guide</i>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To specify a customer managed KMS key, you can use its key ID, Amazon Resource Name (ARN), alias name, or
     *         alias ARN. When using an alias name, prefix it with <code>"alias/"</code>. To specify a key in a
     *         different Amazon Web Services account, you must use the key ARN or alias ARN.
     *         </p>
     *         <p>
     *         For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias name: <code>alias/ExampleAlias</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#key-id">Key identifiers
     *         (KeyId)</a> in the <i>Key Management Service Developer Guide</i>.
     */
    public final String kmsKey() {
        return kmsKey;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(permissionsModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(deletionProtection());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKey());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateLedgerRequest)) {
            return false;
        }
        CreateLedgerRequest other = (CreateLedgerRequest) obj;
        return Objects.equals(name(), other.name()) && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(permissionsModeAsString(), other.permissionsModeAsString())
                && Objects.equals(deletionProtection(), other.deletionProtection()) && Objects.equals(kmsKey(), other.kmsKey());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateLedgerRequest").add("Name", name()).add("Tags", hasTags() ? tags() : null)
                .add("PermissionsMode", permissionsModeAsString()).add("DeletionProtection", deletionProtection())
                .add("KmsKey", kmsKey()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "PermissionsMode":
            return Optional.ofNullable(clazz.cast(permissionsModeAsString()));
        case "DeletionProtection":
            return Optional.ofNullable(clazz.cast(deletionProtection()));
        case "KmsKey":
            return Optional.ofNullable(clazz.cast(kmsKey()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateLedgerRequest, T> g) {
        return obj -> g.apply((CreateLedgerRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends QldbRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateLedgerRequest> {
        /**
         * <p>
         * The name of the ledger that you want to create. The name must be unique among all of the ledgers in your
         * Amazon Web Services account in the current Region.
         * </p>
         * <p>
         * Naming constraints for ledger names are defined in <a
         * href="https://docs.aws.amazon.com/qldb/latest/developerguide/limits.html#limits.naming">Quotas in Amazon
         * QLDB</a> in the <i>Amazon QLDB Developer Guide</i>.
         * </p>
         * 
         * @param name
         *        The name of the ledger that you want to create. The name must be unique among all of the ledgers in
         *        your Amazon Web Services account in the current Region.</p>
         *        <p>
         *        Naming constraints for ledger names are defined in <a
         *        href="https://docs.aws.amazon.com/qldb/latest/developerguide/limits.html#limits.naming">Quotas in
         *        Amazon QLDB</a> in the <i>Amazon QLDB Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The key-value pairs to add as tags to the ledger that you want to create. Tag keys are case sensitive. Tag
         * values are case sensitive and can be null.
         * </p>
         * 
         * @param tags
         *        The key-value pairs to add as tags to the ledger that you want to create. Tag keys are case sensitive.
         *        Tag values are case sensitive and can be null.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The permissions mode to assign to the ledger that you want to create. This parameter can have one of the
         * following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ALLOW_ALL</code>: A legacy permissions mode that enables access control with API-level granularity for
         * ledgers.
         * </p>
         * <p>
         * This mode allows users who have the <code>SendCommand</code> API permission for this ledger to run all
         * PartiQL commands (hence, <code>ALLOW_ALL</code>) on any tables in the specified ledger. This mode disregards
         * any table-level or command-level IAM permissions policies that you create for the ledger.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STANDARD</code>: (<i>Recommended</i>) A permissions mode that enables access control with finer
         * granularity for ledgers, tables, and PartiQL commands.
         * </p>
         * <p>
         * By default, this mode denies all user requests to run any PartiQL commands on any tables in this ledger. To
         * allow PartiQL commands to run, you must create IAM permissions policies for specific table resources and
         * PartiQL actions, in addition to the <code>SendCommand</code> API permission for the ledger. For information,
         * see <a
         * href="https://docs.aws.amazon.com/qldb/latest/developerguide/getting-started-standard-mode.html">Getting
         * started with the standard permissions mode</a> in the <i>Amazon QLDB Developer Guide</i>.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * We strongly recommend using the <code>STANDARD</code> permissions mode to maximize the security of your
         * ledger data.
         * </p>
         * </note>
         * 
         * @param permissionsMode
         *        The permissions mode to assign to the ledger that you want to create. This parameter can have one of
         *        the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ALLOW_ALL</code>: A legacy permissions mode that enables access control with API-level
         *        granularity for ledgers.
         *        </p>
         *        <p>
         *        This mode allows users who have the <code>SendCommand</code> API permission for this ledger to run all
         *        PartiQL commands (hence, <code>ALLOW_ALL</code>) on any tables in the specified ledger. This mode
         *        disregards any table-level or command-level IAM permissions policies that you create for the ledger.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STANDARD</code>: (<i>Recommended</i>) A permissions mode that enables access control with finer
         *        granularity for ledgers, tables, and PartiQL commands.
         *        </p>
         *        <p>
         *        By default, this mode denies all user requests to run any PartiQL commands on any tables in this
         *        ledger. To allow PartiQL commands to run, you must create IAM permissions policies for specific table
         *        resources and PartiQL actions, in addition to the <code>SendCommand</code> API permission for the
         *        ledger. For information, see <a
         *        href="https://docs.aws.amazon.com/qldb/latest/developerguide/getting-started-standard-mode.html"
         *        >Getting started with the standard permissions mode</a> in the <i>Amazon QLDB Developer Guide</i>.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        We strongly recommend using the <code>STANDARD</code> permissions mode to maximize the security of
         *        your ledger data.
         *        </p>
         * @see PermissionsMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionsMode
         */
        Builder permissionsMode(String permissionsMode);

        /**
         * <p>
         * The permissions mode to assign to the ledger that you want to create. This parameter can have one of the
         * following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ALLOW_ALL</code>: A legacy permissions mode that enables access control with API-level granularity for
         * ledgers.
         * </p>
         * <p>
         * This mode allows users who have the <code>SendCommand</code> API permission for this ledger to run all
         * PartiQL commands (hence, <code>ALLOW_ALL</code>) on any tables in the specified ledger. This mode disregards
         * any table-level or command-level IAM permissions policies that you create for the ledger.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STANDARD</code>: (<i>Recommended</i>) A permissions mode that enables access control with finer
         * granularity for ledgers, tables, and PartiQL commands.
         * </p>
         * <p>
         * By default, this mode denies all user requests to run any PartiQL commands on any tables in this ledger. To
         * allow PartiQL commands to run, you must create IAM permissions policies for specific table resources and
         * PartiQL actions, in addition to the <code>SendCommand</code> API permission for the ledger. For information,
         * see <a
         * href="https://docs.aws.amazon.com/qldb/latest/developerguide/getting-started-standard-mode.html">Getting
         * started with the standard permissions mode</a> in the <i>Amazon QLDB Developer Guide</i>.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * We strongly recommend using the <code>STANDARD</code> permissions mode to maximize the security of your
         * ledger data.
         * </p>
         * </note>
         * 
         * @param permissionsMode
         *        The permissions mode to assign to the ledger that you want to create. This parameter can have one of
         *        the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ALLOW_ALL</code>: A legacy permissions mode that enables access control with API-level
         *        granularity for ledgers.
         *        </p>
         *        <p>
         *        This mode allows users who have the <code>SendCommand</code> API permission for this ledger to run all
         *        PartiQL commands (hence, <code>ALLOW_ALL</code>) on any tables in the specified ledger. This mode
         *        disregards any table-level or command-level IAM permissions policies that you create for the ledger.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STANDARD</code>: (<i>Recommended</i>) A permissions mode that enables access control with finer
         *        granularity for ledgers, tables, and PartiQL commands.
         *        </p>
         *        <p>
         *        By default, this mode denies all user requests to run any PartiQL commands on any tables in this
         *        ledger. To allow PartiQL commands to run, you must create IAM permissions policies for specific table
         *        resources and PartiQL actions, in addition to the <code>SendCommand</code> API permission for the
         *        ledger. For information, see <a
         *        href="https://docs.aws.amazon.com/qldb/latest/developerguide/getting-started-standard-mode.html"
         *        >Getting started with the standard permissions mode</a> in the <i>Amazon QLDB Developer Guide</i>.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        We strongly recommend using the <code>STANDARD</code> permissions mode to maximize the security of
         *        your ledger data.
         *        </p>
         * @see PermissionsMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionsMode
         */
        Builder permissionsMode(PermissionsMode permissionsMode);

        /**
         * <p>
         * The flag that prevents a ledger from being deleted by any user. If not provided on ledger creation, this
         * feature is enabled (<code>true</code>) by default.
         * </p>
         * <p>
         * If deletion protection is enabled, you must first disable it before you can delete the ledger. You can
         * disable it by calling the <code>UpdateLedger</code> operation to set the flag to <code>false</code>.
         * </p>
         * 
         * @param deletionProtection
         *        The flag that prevents a ledger from being deleted by any user. If not provided on ledger creation,
         *        this feature is enabled (<code>true</code>) by default.</p>
         *        <p>
         *        If deletion protection is enabled, you must first disable it before you can delete the ledger. You can
         *        disable it by calling the <code>UpdateLedger</code> operation to set the flag to <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletionProtection(Boolean deletionProtection);

        /**
         * <p>
         * The key in Key Management Service (KMS) to use for encryption of data at rest in the ledger. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/qldb/latest/developerguide/encryption-at-rest.html">Encryption at rest</a>
         * in the <i>Amazon QLDB Developer Guide</i>.
         * </p>
         * <p>
         * Use one of the following options to specify this parameter:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AWS_OWNED_KMS_KEY</code>: Use an KMS key that is owned and managed by Amazon Web Services on your
         * behalf.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Undefined</b>: By default, use an Amazon Web Services owned KMS key.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>A valid symmetric customer managed KMS key</b>: Use the specified KMS key in your account that you create,
         * own, and manage.
         * </p>
         * <p>
         * Amazon QLDB does not support asymmetric keys. For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/symmetric-asymmetric.html">Using symmetric and
         * asymmetric keys</a> in the <i>Key Management Service Developer Guide</i>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * To specify a customer managed KMS key, you can use its key ID, Amazon Resource Name (ARN), alias name, or
         * alias ARN. When using an alias name, prefix it with <code>"alias/"</code>. To specify a key in a different
         * Amazon Web Services account, you must use the key ARN or alias ARN.
         * </p>
         * <p>
         * For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias name: <code>alias/ExampleAlias</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#key-id">Key identifiers (KeyId)</a>
         * in the <i>Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param kmsKey
         *        The key in Key Management Service (KMS) to use for encryption of data at rest in the ledger. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/qldb/latest/developerguide/encryption-at-rest.html">Encryption at
         *        rest</a> in the <i>Amazon QLDB Developer Guide</i>.</p>
         *        <p>
         *        Use one of the following options to specify this parameter:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AWS_OWNED_KMS_KEY</code>: Use an KMS key that is owned and managed by Amazon Web Services on
         *        your behalf.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Undefined</b>: By default, use an Amazon Web Services owned KMS key.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>A valid symmetric customer managed KMS key</b>: Use the specified KMS key in your account that you
         *        create, own, and manage.
         *        </p>
         *        <p>
         *        Amazon QLDB does not support asymmetric keys. For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/symmetric-asymmetric.html">Using symmetric
         *        and asymmetric keys</a> in the <i>Key Management Service Developer Guide</i>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To specify a customer managed KMS key, you can use its key ID, Amazon Resource Name (ARN), alias name,
         *        or alias ARN. When using an alias name, prefix it with <code>"alias/"</code>. To specify a key in a
         *        different Amazon Web Services account, you must use the key ARN or alias ARN.
         *        </p>
         *        <p>
         *        For example:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias name: <code>alias/ExampleAlias</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#key-id">Key identifiers
         *        (KeyId)</a> in the <i>Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKey(String kmsKey);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends QldbRequest.BuilderImpl implements Builder {
        private String name;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String permissionsMode;

        private Boolean deletionProtection;

        private String kmsKey;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateLedgerRequest model) {
            super(model);
            name(model.name);
            tags(model.tags);
            permissionsMode(model.permissionsMode);
            deletionProtection(model.deletionProtection);
            kmsKey(model.kmsKey);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        public final String getPermissionsMode() {
            return permissionsMode;
        }

        public final void setPermissionsMode(String permissionsMode) {
            this.permissionsMode = permissionsMode;
        }

        @Override
        public final Builder permissionsMode(String permissionsMode) {
            this.permissionsMode = permissionsMode;
            return this;
        }

        @Override
        public final Builder permissionsMode(PermissionsMode permissionsMode) {
            this.permissionsMode(permissionsMode == null ? null : permissionsMode.toString());
            return this;
        }

        public final Boolean getDeletionProtection() {
            return deletionProtection;
        }

        public final void setDeletionProtection(Boolean deletionProtection) {
            this.deletionProtection = deletionProtection;
        }

        @Override
        public final Builder deletionProtection(Boolean deletionProtection) {
            this.deletionProtection = deletionProtection;
            return this;
        }

        public final String getKmsKey() {
            return kmsKey;
        }

        public final void setKmsKey(String kmsKey) {
            this.kmsKey = kmsKey;
        }

        @Override
        public final Builder kmsKey(String kmsKey) {
            this.kmsKey = kmsKey;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateLedgerRequest build() {
            return new CreateLedgerRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
