/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.qldb;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.qldb.model.CancelJournalKinesisStreamRequest;
import software.amazon.awssdk.services.qldb.model.CancelJournalKinesisStreamResponse;
import software.amazon.awssdk.services.qldb.model.CreateLedgerRequest;
import software.amazon.awssdk.services.qldb.model.CreateLedgerResponse;
import software.amazon.awssdk.services.qldb.model.DeleteLedgerRequest;
import software.amazon.awssdk.services.qldb.model.DeleteLedgerResponse;
import software.amazon.awssdk.services.qldb.model.DescribeJournalKinesisStreamRequest;
import software.amazon.awssdk.services.qldb.model.DescribeJournalKinesisStreamResponse;
import software.amazon.awssdk.services.qldb.model.DescribeJournalS3ExportRequest;
import software.amazon.awssdk.services.qldb.model.DescribeJournalS3ExportResponse;
import software.amazon.awssdk.services.qldb.model.DescribeLedgerRequest;
import software.amazon.awssdk.services.qldb.model.DescribeLedgerResponse;
import software.amazon.awssdk.services.qldb.model.ExportJournalToS3Request;
import software.amazon.awssdk.services.qldb.model.ExportJournalToS3Response;
import software.amazon.awssdk.services.qldb.model.GetBlockRequest;
import software.amazon.awssdk.services.qldb.model.GetBlockResponse;
import software.amazon.awssdk.services.qldb.model.GetDigestRequest;
import software.amazon.awssdk.services.qldb.model.GetDigestResponse;
import software.amazon.awssdk.services.qldb.model.GetRevisionRequest;
import software.amazon.awssdk.services.qldb.model.GetRevisionResponse;
import software.amazon.awssdk.services.qldb.model.InvalidParameterException;
import software.amazon.awssdk.services.qldb.model.LimitExceededException;
import software.amazon.awssdk.services.qldb.model.ListJournalKinesisStreamsForLedgerRequest;
import software.amazon.awssdk.services.qldb.model.ListJournalKinesisStreamsForLedgerResponse;
import software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerRequest;
import software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerResponse;
import software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsRequest;
import software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsResponse;
import software.amazon.awssdk.services.qldb.model.ListLedgersRequest;
import software.amazon.awssdk.services.qldb.model.ListLedgersResponse;
import software.amazon.awssdk.services.qldb.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.qldb.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.qldb.model.QldbException;
import software.amazon.awssdk.services.qldb.model.QldbRequest;
import software.amazon.awssdk.services.qldb.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.qldb.model.ResourceInUseException;
import software.amazon.awssdk.services.qldb.model.ResourceNotFoundException;
import software.amazon.awssdk.services.qldb.model.ResourcePreconditionNotMetException;
import software.amazon.awssdk.services.qldb.model.StreamJournalToKinesisRequest;
import software.amazon.awssdk.services.qldb.model.StreamJournalToKinesisResponse;
import software.amazon.awssdk.services.qldb.model.TagResourceRequest;
import software.amazon.awssdk.services.qldb.model.TagResourceResponse;
import software.amazon.awssdk.services.qldb.model.UntagResourceRequest;
import software.amazon.awssdk.services.qldb.model.UntagResourceResponse;
import software.amazon.awssdk.services.qldb.model.UpdateLedgerPermissionsModeRequest;
import software.amazon.awssdk.services.qldb.model.UpdateLedgerPermissionsModeResponse;
import software.amazon.awssdk.services.qldb.model.UpdateLedgerRequest;
import software.amazon.awssdk.services.qldb.model.UpdateLedgerResponse;
import software.amazon.awssdk.services.qldb.paginators.ListJournalKinesisStreamsForLedgerPublisher;
import software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsForLedgerPublisher;
import software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsPublisher;
import software.amazon.awssdk.services.qldb.paginators.ListLedgersPublisher;
import software.amazon.awssdk.services.qldb.transform.CancelJournalKinesisStreamRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.CreateLedgerRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.DeleteLedgerRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.DescribeJournalKinesisStreamRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.DescribeJournalS3ExportRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.DescribeLedgerRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.ExportJournalToS3RequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.GetBlockRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.GetDigestRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.GetRevisionRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.ListJournalKinesisStreamsForLedgerRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.ListJournalS3ExportsForLedgerRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.ListJournalS3ExportsRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.ListLedgersRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.StreamJournalToKinesisRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.UpdateLedgerPermissionsModeRequestMarshaller;
import software.amazon.awssdk.services.qldb.transform.UpdateLedgerRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link QldbAsyncClient}.
 *
 * @see QldbAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultQldbAsyncClient implements QldbAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultQldbAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultQldbAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Ends a given Amazon QLDB journal stream. Before a stream can be canceled, its current status must be
     * <code>ACTIVE</code>.
     * </p>
     * <p>
     * You can't restart a stream after you cancel it. Canceled QLDB stream resources are subject to a 7-day retention
     * period, so they are automatically deleted after this limit expires.
     * </p>
     *
     * @param cancelJournalKinesisStreamRequest
     * @return A Java Future containing the result of the CancelJournalKinesisStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourcePreconditionNotMetException The operation failed because a condition wasn't satisfied in
     *         advance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.CancelJournalKinesisStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/CancelJournalKinesisStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelJournalKinesisStreamResponse> cancelJournalKinesisStream(
            CancelJournalKinesisStreamRequest cancelJournalKinesisStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJournalKinesisStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJournalKinesisStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelJournalKinesisStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelJournalKinesisStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelJournalKinesisStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelJournalKinesisStreamRequest, CancelJournalKinesisStreamResponse>()
                            .withOperationName("CancelJournalKinesisStream")
                            .withMarshaller(new CancelJournalKinesisStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelJournalKinesisStreamRequest));
            CompletableFuture<CancelJournalKinesisStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new ledger in your Amazon Web Services account in the current Region.
     * </p>
     *
     * @param createLedgerRequest
     * @return A Java Future containing the result of the CreateLedger operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceAlreadyExistsException The specified resource already exists.</li>
     *         <li>LimitExceededException You have reached the limit on the maximum number of resources allowed.</li>
     *         <li>ResourceInUseException The specified resource can't be modified at this time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.CreateLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/CreateLedger" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLedgerResponse> createLedger(CreateLedgerRequest createLedgerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLedgerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLedger");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLedgerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateLedgerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLedgerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLedgerRequest, CreateLedgerResponse>()
                            .withOperationName("CreateLedger").withMarshaller(new CreateLedgerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLedgerRequest));
            CompletableFuture<CreateLedgerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a ledger and all of its contents. This action is irreversible.
     * </p>
     * <p>
     * If deletion protection is enabled, you must first disable it before you can delete the ledger. You can disable it
     * by calling the <code>UpdateLedger</code> operation to set the flag to <code>false</code>.
     * </p>
     *
     * @param deleteLedgerRequest
     * @return A Java Future containing the result of the DeleteLedger operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourceInUseException The specified resource can't be modified at this time.</li>
     *         <li>ResourcePreconditionNotMetException The operation failed because a condition wasn't satisfied in
     *         advance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.DeleteLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/DeleteLedger" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLedgerResponse> deleteLedger(DeleteLedgerRequest deleteLedgerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLedgerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLedger");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLedgerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteLedgerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLedgerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLedgerRequest, DeleteLedgerResponse>()
                            .withOperationName("DeleteLedger").withMarshaller(new DeleteLedgerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteLedgerRequest));
            CompletableFuture<DeleteLedgerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns detailed information about a given Amazon QLDB journal stream. The output includes the Amazon Resource
     * Name (ARN), stream name, current status, creation time, and the parameters of the original stream creation
     * request.
     * </p>
     * <p>
     * This action does not return any expired journal streams. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/streams.create.html#streams.create.states.expiration"
     * >Expiration for terminal streams</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     *
     * @param describeJournalKinesisStreamRequest
     * @return A Java Future containing the result of the DescribeJournalKinesisStream operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourcePreconditionNotMetException The operation failed because a condition wasn't satisfied in
     *         advance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.DescribeJournalKinesisStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/DescribeJournalKinesisStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJournalKinesisStreamResponse> describeJournalKinesisStream(
            DescribeJournalKinesisStreamRequest describeJournalKinesisStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJournalKinesisStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJournalKinesisStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJournalKinesisStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeJournalKinesisStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeJournalKinesisStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJournalKinesisStreamRequest, DescribeJournalKinesisStreamResponse>()
                            .withOperationName("DescribeJournalKinesisStream")
                            .withMarshaller(new DescribeJournalKinesisStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeJournalKinesisStreamRequest));
            CompletableFuture<DescribeJournalKinesisStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a journal export job, including the ledger name, export ID, creation time, current
     * status, and the parameters of the original export creation request.
     * </p>
     * <p>
     * This action does not return any expired export jobs. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/export-journal.request.html#export-journal.request.expiration"
     * >Export job expiration</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * <p>
     * If the export job with the given <code>ExportId</code> doesn't exist, then throws
     * <code>ResourceNotFoundException</code>.
     * </p>
     * <p>
     * If the ledger with the given <code>Name</code> doesn't exist, then throws <code>ResourceNotFoundException</code>.
     * </p>
     *
     * @param describeJournalS3ExportRequest
     * @return A Java Future containing the result of the DescribeJournalS3Export operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.DescribeJournalS3Export
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/DescribeJournalS3Export" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJournalS3ExportResponse> describeJournalS3Export(
            DescribeJournalS3ExportRequest describeJournalS3ExportRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJournalS3ExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJournalS3Export");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJournalS3ExportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeJournalS3ExportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeJournalS3ExportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJournalS3ExportRequest, DescribeJournalS3ExportResponse>()
                            .withOperationName("DescribeJournalS3Export")
                            .withMarshaller(new DescribeJournalS3ExportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeJournalS3ExportRequest));
            CompletableFuture<DescribeJournalS3ExportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a ledger, including its state, permissions mode, encryption at rest settings, and when
     * it was created.
     * </p>
     *
     * @param describeLedgerRequest
     * @return A Java Future containing the result of the DescribeLedger operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.DescribeLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/DescribeLedger" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLedgerResponse> describeLedger(DescribeLedgerRequest describeLedgerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLedgerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLedger");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLedgerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLedgerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLedgerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLedgerRequest, DescribeLedgerResponse>()
                            .withOperationName("DescribeLedger")
                            .withMarshaller(new DescribeLedgerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeLedgerRequest));
            CompletableFuture<DescribeLedgerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports journal contents within a date and time range from a ledger into a specified Amazon Simple Storage
     * Service (Amazon S3) bucket. A journal export job can write the data objects in either the text or binary
     * representation of Amazon Ion format, or in <i>JSON Lines</i> text format.
     * </p>
     * <p>
     * In JSON Lines format, each journal block in the exported data object is a valid JSON object that is delimited by
     * a newline. You can use this format to easily integrate JSON exports with analytics tools such as Glue and Amazon
     * Athena because these services can parse newline-delimited JSON automatically. For more information about the
     * format, see <a href="https://jsonlines.org/">JSON Lines</a>.
     * </p>
     * <p>
     * If the ledger with the given <code>Name</code> doesn't exist, then throws <code>ResourceNotFoundException</code>.
     * </p>
     * <p>
     * If the ledger with the given <code>Name</code> is in <code>CREATING</code> status, then throws
     * <code>ResourcePreconditionNotMetException</code>.
     * </p>
     * <p>
     * You can initiate up to two concurrent journal export requests for each ledger. Beyond this limit, journal export
     * requests throw <code>LimitExceededException</code>.
     * </p>
     *
     * @param exportJournalToS3Request
     * @return A Java Future containing the result of the ExportJournalToS3 operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourcePreconditionNotMetException The operation failed because a condition wasn't satisfied in
     *         advance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.ExportJournalToS3
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ExportJournalToS3" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ExportJournalToS3Response> exportJournalToS3(ExportJournalToS3Request exportJournalToS3Request) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportJournalToS3Request
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportJournalToS3");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportJournalToS3Response> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExportJournalToS3Response::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportJournalToS3Response> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportJournalToS3Request, ExportJournalToS3Response>()
                            .withOperationName("ExportJournalToS3")
                            .withMarshaller(new ExportJournalToS3RequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(exportJournalToS3Request));
            CompletableFuture<ExportJournalToS3Response> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a block object at a specified address in a journal. Also returns a proof of the specified block for
     * verification if <code>DigestTipAddress</code> is provided.
     * </p>
     * <p>
     * For information about the data contents in a block, see <a
     * href="https://docs.aws.amazon.com/qldb/latest/developerguide/journal-contents.html">Journal contents</a> in the
     * <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * <p>
     * If the specified ledger doesn't exist or is in <code>DELETING</code> status, then throws
     * <code>ResourceNotFoundException</code>.
     * </p>
     * <p>
     * If the specified ledger is in <code>CREATING</code> status, then throws
     * <code>ResourcePreconditionNotMetException</code>.
     * </p>
     * <p>
     * If no block exists with the specified address, then throws <code>InvalidParameterException</code>.
     * </p>
     *
     * @param getBlockRequest
     * @return A Java Future containing the result of the GetBlock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourcePreconditionNotMetException The operation failed because a condition wasn't satisfied in
     *         advance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.GetBlock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/GetBlock" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetBlockResponse> getBlock(GetBlockRequest getBlockRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBlockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBlock");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBlockResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetBlockResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBlockResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBlockRequest, GetBlockResponse>().withOperationName("GetBlock")
                            .withMarshaller(new GetBlockRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getBlockRequest));
            CompletableFuture<GetBlockResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the digest of a ledger at the latest committed block in the journal. The response includes a 256-bit hash
     * value and a block address.
     * </p>
     *
     * @param getDigestRequest
     * @return A Java Future containing the result of the GetDigest operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourcePreconditionNotMetException The operation failed because a condition wasn't satisfied in
     *         advance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.GetDigest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/GetDigest" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDigestResponse> getDigest(GetDigestRequest getDigestRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDigestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDigest");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDigestResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDigestResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDigestResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDigestRequest, GetDigestResponse>().withOperationName("GetDigest")
                            .withMarshaller(new GetDigestRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDigestRequest));
            CompletableFuture<GetDigestResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a revision data object for a specified document ID and block address. Also returns a proof of the
     * specified revision for verification if <code>DigestTipAddress</code> is provided.
     * </p>
     *
     * @param getRevisionRequest
     * @return A Java Future containing the result of the GetRevision operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourcePreconditionNotMetException The operation failed because a condition wasn't satisfied in
     *         advance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.GetRevision
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/GetRevision" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetRevisionResponse> getRevision(GetRevisionRequest getRevisionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRevisionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRevision");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRevisionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetRevisionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRevisionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRevisionRequest, GetRevisionResponse>()
                            .withOperationName("GetRevision").withMarshaller(new GetRevisionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getRevisionRequest));
            CompletableFuture<GetRevisionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of all Amazon QLDB journal stream descriptors for a given ledger. The output of each stream
     * descriptor includes the same details that are returned by <code>DescribeJournalKinesisStream</code>.
     * </p>
     * <p>
     * This action does not return any expired journal streams. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/streams.create.html#streams.create.states.expiration"
     * >Expiration for terminal streams</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * <p>
     * This action returns a maximum of <code>MaxResults</code> items. It is paginated so that you can retrieve all the
     * items by calling <code>ListJournalKinesisStreamsForLedger</code> multiple times.
     * </p>
     *
     * @param listJournalKinesisStreamsForLedgerRequest
     * @return A Java Future containing the result of the ListJournalKinesisStreamsForLedger operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourcePreconditionNotMetException The operation failed because a condition wasn't satisfied in
     *         advance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.ListJournalKinesisStreamsForLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListJournalKinesisStreamsForLedger"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListJournalKinesisStreamsForLedgerResponse> listJournalKinesisStreamsForLedger(
            ListJournalKinesisStreamsForLedgerRequest listJournalKinesisStreamsForLedgerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listJournalKinesisStreamsForLedgerRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJournalKinesisStreamsForLedger");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJournalKinesisStreamsForLedgerResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListJournalKinesisStreamsForLedgerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListJournalKinesisStreamsForLedgerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJournalKinesisStreamsForLedgerRequest, ListJournalKinesisStreamsForLedgerResponse>()
                            .withOperationName("ListJournalKinesisStreamsForLedger")
                            .withMarshaller(new ListJournalKinesisStreamsForLedgerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listJournalKinesisStreamsForLedgerRequest));
            CompletableFuture<ListJournalKinesisStreamsForLedgerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of all Amazon QLDB journal stream descriptors for a given ledger. The output of each stream
     * descriptor includes the same details that are returned by <code>DescribeJournalKinesisStream</code>.
     * </p>
     * <p>
     * This action does not return any expired journal streams. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/streams.create.html#streams.create.states.expiration"
     * >Expiration for terminal streams</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * <p>
     * This action returns a maximum of <code>MaxResults</code> items. It is paginated so that you can retrieve all the
     * items by calling <code>ListJournalKinesisStreamsForLedger</code> multiple times.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listJournalKinesisStreamsForLedger(software.amazon.awssdk.services.qldb.model.ListJournalKinesisStreamsForLedgerRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListJournalKinesisStreamsForLedgerPublisher publisher = client.listJournalKinesisStreamsForLedgerPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListJournalKinesisStreamsForLedgerPublisher publisher = client.listJournalKinesisStreamsForLedgerPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.qldb.model.ListJournalKinesisStreamsForLedgerResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.qldb.model.ListJournalKinesisStreamsForLedgerResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJournalKinesisStreamsForLedger(software.amazon.awssdk.services.qldb.model.ListJournalKinesisStreamsForLedgerRequest)}
     * operation.</b>
     * </p>
     *
     * @param listJournalKinesisStreamsForLedgerRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourcePreconditionNotMetException The operation failed because a condition wasn't satisfied in
     *         advance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.ListJournalKinesisStreamsForLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListJournalKinesisStreamsForLedger"
     *      target="_top">AWS API Documentation</a>
     */
    public ListJournalKinesisStreamsForLedgerPublisher listJournalKinesisStreamsForLedgerPaginator(
            ListJournalKinesisStreamsForLedgerRequest listJournalKinesisStreamsForLedgerRequest) {
        return new ListJournalKinesisStreamsForLedgerPublisher(this,
                applyPaginatorUserAgent(listJournalKinesisStreamsForLedgerRequest));
    }

    /**
     * <p>
     * Returns an array of journal export job descriptions for all ledgers that are associated with the current Amazon
     * Web Services account and Region.
     * </p>
     * <p>
     * This action returns a maximum of <code>MaxResults</code> items, and is paginated so that you can retrieve all the
     * items by calling <code>ListJournalS3Exports</code> multiple times.
     * </p>
     * <p>
     * This action does not return any expired export jobs. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/export-journal.request.html#export-journal.request.expiration"
     * >Export job expiration</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     *
     * @param listJournalS3ExportsRequest
     * @return A Java Future containing the result of the ListJournalS3Exports operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.ListJournalS3Exports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListJournalS3Exports" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListJournalS3ExportsResponse> listJournalS3Exports(
            ListJournalS3ExportsRequest listJournalS3ExportsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJournalS3ExportsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJournalS3Exports");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJournalS3ExportsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListJournalS3ExportsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListJournalS3ExportsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJournalS3ExportsRequest, ListJournalS3ExportsResponse>()
                            .withOperationName("ListJournalS3Exports")
                            .withMarshaller(new ListJournalS3ExportsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listJournalS3ExportsRequest));
            CompletableFuture<ListJournalS3ExportsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of journal export job descriptions for a specified ledger.
     * </p>
     * <p>
     * This action returns a maximum of <code>MaxResults</code> items, and is paginated so that you can retrieve all the
     * items by calling <code>ListJournalS3ExportsForLedger</code> multiple times.
     * </p>
     * <p>
     * This action does not return any expired export jobs. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/export-journal.request.html#export-journal.request.expiration"
     * >Export job expiration</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     *
     * @param listJournalS3ExportsForLedgerRequest
     * @return A Java Future containing the result of the ListJournalS3ExportsForLedger operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.ListJournalS3ExportsForLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListJournalS3ExportsForLedger"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListJournalS3ExportsForLedgerResponse> listJournalS3ExportsForLedger(
            ListJournalS3ExportsForLedgerRequest listJournalS3ExportsForLedgerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listJournalS3ExportsForLedgerRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJournalS3ExportsForLedger");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJournalS3ExportsForLedgerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListJournalS3ExportsForLedgerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListJournalS3ExportsForLedgerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJournalS3ExportsForLedgerRequest, ListJournalS3ExportsForLedgerResponse>()
                            .withOperationName("ListJournalS3ExportsForLedger")
                            .withMarshaller(new ListJournalS3ExportsForLedgerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listJournalS3ExportsForLedgerRequest));
            CompletableFuture<ListJournalS3ExportsForLedgerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of journal export job descriptions for a specified ledger.
     * </p>
     * <p>
     * This action returns a maximum of <code>MaxResults</code> items, and is paginated so that you can retrieve all the
     * items by calling <code>ListJournalS3ExportsForLedger</code> multiple times.
     * </p>
     * <p>
     * This action does not return any expired export jobs. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/export-journal.request.html#export-journal.request.expiration"
     * >Export job expiration</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listJournalS3ExportsForLedger(software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsForLedgerPublisher publisher = client.listJournalS3ExportsForLedgerPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsForLedgerPublisher publisher = client.listJournalS3ExportsForLedgerPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJournalS3ExportsForLedger(software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerRequest)}
     * operation.</b>
     * </p>
     *
     * @param listJournalS3ExportsForLedgerRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.ListJournalS3ExportsForLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListJournalS3ExportsForLedger"
     *      target="_top">AWS API Documentation</a>
     */
    public ListJournalS3ExportsForLedgerPublisher listJournalS3ExportsForLedgerPaginator(
            ListJournalS3ExportsForLedgerRequest listJournalS3ExportsForLedgerRequest) {
        return new ListJournalS3ExportsForLedgerPublisher(this, applyPaginatorUserAgent(listJournalS3ExportsForLedgerRequest));
    }

    /**
     * <p>
     * Returns an array of journal export job descriptions for all ledgers that are associated with the current Amazon
     * Web Services account and Region.
     * </p>
     * <p>
     * This action returns a maximum of <code>MaxResults</code> items, and is paginated so that you can retrieve all the
     * items by calling <code>ListJournalS3Exports</code> multiple times.
     * </p>
     * <p>
     * This action does not return any expired export jobs. For more information, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/export-journal.request.html#export-journal.request.expiration"
     * >Export job expiration</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listJournalS3Exports(software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsPublisher publisher = client.listJournalS3ExportsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsPublisher publisher = client.listJournalS3ExportsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJournalS3Exports(software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listJournalS3ExportsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.ListJournalS3Exports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListJournalS3Exports" target="_top">AWS API
     *      Documentation</a>
     */
    public ListJournalS3ExportsPublisher listJournalS3ExportsPaginator(ListJournalS3ExportsRequest listJournalS3ExportsRequest) {
        return new ListJournalS3ExportsPublisher(this, applyPaginatorUserAgent(listJournalS3ExportsRequest));
    }

    /**
     * <p>
     * Returns an array of ledger summaries that are associated with the current Amazon Web Services account and Region.
     * </p>
     * <p>
     * This action returns a maximum of 100 items and is paginated so that you can retrieve all the items by calling
     * <code>ListLedgers</code> multiple times.
     * </p>
     *
     * @param listLedgersRequest
     * @return A Java Future containing the result of the ListLedgers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.ListLedgers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListLedgers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListLedgersResponse> listLedgers(ListLedgersRequest listLedgersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLedgersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLedgers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLedgersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListLedgersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLedgersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLedgersRequest, ListLedgersResponse>()
                            .withOperationName("ListLedgers").withMarshaller(new ListLedgersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLedgersRequest));
            CompletableFuture<ListLedgersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of ledger summaries that are associated with the current Amazon Web Services account and Region.
     * </p>
     * <p>
     * This action returns a maximum of 100 items and is paginated so that you can retrieve all the items by calling
     * <code>ListLedgers</code> multiple times.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listLedgers(software.amazon.awssdk.services.qldb.model.ListLedgersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListLedgersPublisher publisher = client.listLedgersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.qldb.paginators.ListLedgersPublisher publisher = client.listLedgersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.qldb.model.ListLedgersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.qldb.model.ListLedgersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLedgers(software.amazon.awssdk.services.qldb.model.ListLedgersRequest)} operation.</b>
     * </p>
     *
     * @param listLedgersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.ListLedgers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListLedgers" target="_top">AWS API
     *      Documentation</a>
     */
    public ListLedgersPublisher listLedgersPaginator(ListLedgersRequest listLedgersRequest) {
        return new ListLedgersPublisher(this, applyPaginatorUserAgent(listLedgersRequest));
    }

    /**
     * <p>
     * Returns all tags for a specified Amazon QLDB resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a journal stream for a given Amazon QLDB ledger. The stream captures every document revision that is
     * committed to the ledger's journal and delivers the data to a specified Amazon Kinesis Data Streams resource.
     * </p>
     *
     * @param streamJournalToKinesisRequest
     * @return A Java Future containing the result of the StreamJournalToKinesis operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourcePreconditionNotMetException The operation failed because a condition wasn't satisfied in
     *         advance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.StreamJournalToKinesis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/StreamJournalToKinesis" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StreamJournalToKinesisResponse> streamJournalToKinesis(
            StreamJournalToKinesisRequest streamJournalToKinesisRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, streamJournalToKinesisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StreamJournalToKinesis");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StreamJournalToKinesisResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StreamJournalToKinesisResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StreamJournalToKinesisResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StreamJournalToKinesisRequest, StreamJournalToKinesisResponse>()
                            .withOperationName("StreamJournalToKinesis")
                            .withMarshaller(new StreamJournalToKinesisRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(streamJournalToKinesisRequest));
            CompletableFuture<StreamJournalToKinesisResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to a specified Amazon QLDB resource.
     * </p>
     * <p>
     * A resource can have up to 50 tags. If you try to create more than 50 tags for a resource, your request fails and
     * returns an error.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from a specified Amazon QLDB resource. You can specify up to 50 tag keys to remove.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates properties on a ledger.
     * </p>
     *
     * @param updateLedgerRequest
     * @return A Java Future containing the result of the UpdateLedger operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.UpdateLedger
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/UpdateLedger" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLedgerResponse> updateLedger(UpdateLedgerRequest updateLedgerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLedgerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLedger");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLedgerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateLedgerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLedgerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLedgerRequest, UpdateLedgerResponse>()
                            .withOperationName("UpdateLedger").withMarshaller(new UpdateLedgerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLedgerRequest));
            CompletableFuture<UpdateLedgerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the permissions mode of a ledger.
     * </p>
     * <important>
     * <p>
     * Before you switch to the <code>STANDARD</code> permissions mode, you must first create all required IAM policies
     * and table tags to avoid disruption to your users. To learn more, see <a href=
     * "https://docs.aws.amazon.com/qldb/latest/developerguide/ledger-management.basics.html#ledger-mgmt.basics.update-permissions.migrating"
     * >Migrating to the standard permissions mode</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * </important>
     *
     * @param updateLedgerPermissionsModeRequest
     * @return A Java Future containing the result of the UpdateLedgerPermissionsMode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in the request aren't valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample QldbAsyncClient.UpdateLedgerPermissionsMode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qldb-2019-01-02/UpdateLedgerPermissionsMode"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLedgerPermissionsModeResponse> updateLedgerPermissionsMode(
            UpdateLedgerPermissionsModeRequest updateLedgerPermissionsModeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLedgerPermissionsModeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLedgerPermissionsMode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLedgerPermissionsModeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLedgerPermissionsModeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLedgerPermissionsModeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLedgerPermissionsModeRequest, UpdateLedgerPermissionsModeResponse>()
                            .withOperationName("UpdateLedgerPermissionsMode")
                            .withMarshaller(new UpdateLedgerPermissionsModeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLedgerPermissionsModeRequest));
            CompletableFuture<UpdateLedgerPermissionsModeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(QldbException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourcePreconditionNotMetException")
                                .exceptionBuilderSupplier(ResourcePreconditionNotMetException::builder).httpStatusCode(412)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends QldbRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
