/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.qldb.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the encryption of data at rest in an Amazon QLDB ledger. This includes the current status, the key
 * in Key Management Service (KMS), and when the key became inaccessible (in the case of an error).
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/qldb/latest/developerguide/encryption-at-rest.html">Encryption at rest</a> in the
 * <i>Amazon QLDB Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LedgerEncryptionDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<LedgerEncryptionDescription.Builder, LedgerEncryptionDescription> {
    private static final SdkField<String> KMS_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyArn").getter(getter(LedgerEncryptionDescription::kmsKeyArn)).setter(setter(Builder::kmsKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyArn").build()).build();

    private static final SdkField<String> ENCRYPTION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EncryptionStatus").getter(getter(LedgerEncryptionDescription::encryptionStatusAsString))
            .setter(setter(Builder::encryptionStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionStatus").build()).build();

    private static final SdkField<Instant> INACCESSIBLE_KMS_KEY_DATE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("InaccessibleKmsKeyDateTime")
            .getter(getter(LedgerEncryptionDescription::inaccessibleKmsKeyDateTime))
            .setter(setter(Builder::inaccessibleKmsKeyDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InaccessibleKmsKeyDateTime").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KMS_KEY_ARN_FIELD,
            ENCRYPTION_STATUS_FIELD, INACCESSIBLE_KMS_KEY_DATE_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String kmsKeyArn;

    private final String encryptionStatus;

    private final Instant inaccessibleKmsKeyDateTime;

    private LedgerEncryptionDescription(BuilderImpl builder) {
        this.kmsKeyArn = builder.kmsKeyArn;
        this.encryptionStatus = builder.encryptionStatus;
        this.inaccessibleKmsKeyDateTime = builder.inaccessibleKmsKeyDateTime;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the customer managed KMS key that the ledger uses for encryption at rest. If
     * this parameter is undefined, the ledger uses an Amazon Web Services owned KMS key for encryption. It will display
     * <code>AWS_OWNED_KMS_KEY</code> when updating the ledger's encryption configuration to the Amazon Web Services
     * owned KMS key.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the customer managed KMS key that the ledger uses for encryption at
     *         rest. If this parameter is undefined, the ledger uses an Amazon Web Services owned KMS key for
     *         encryption. It will display <code>AWS_OWNED_KMS_KEY</code> when updating the ledger's encryption
     *         configuration to the Amazon Web Services owned KMS key.
     */
    public final String kmsKeyArn() {
        return kmsKeyArn;
    }

    /**
     * <p>
     * The current state of encryption at rest for the ledger. This can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ENABLED</code>: Encryption is fully enabled using the specified key.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code>: The ledger is actively processing the specified key change.
     * </p>
     * <p>
     * Key changes in QLDB are asynchronous. The ledger is fully accessible without any performance impact while the key
     * change is being processed. The amount of time it takes to update a key varies depending on the ledger size.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KMS_KEY_INACCESSIBLE</code>: The specified customer managed KMS key is not accessible, and the ledger is
     * impaired. Either the key was disabled or deleted, or the grants on the key were revoked. When a ledger is
     * impaired, it is not accessible and does not accept any read or write requests.
     * </p>
     * <p>
     * An impaired ledger automatically returns to an active state after you restore the grants on the key, or re-enable
     * the key that was disabled. However, deleting a customer managed KMS key is irreversible. After a key is deleted,
     * you can no longer access the ledgers that are protected with that key, and the data becomes unrecoverable
     * permanently.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionStatus}
     * will return {@link EncryptionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionStatusAsString}.
     * </p>
     * 
     * @return The current state of encryption at rest for the ledger. This can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ENABLED</code>: Encryption is fully enabled using the specified key.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code>: The ledger is actively processing the specified key change.
     *         </p>
     *         <p>
     *         Key changes in QLDB are asynchronous. The ledger is fully accessible without any performance impact while
     *         the key change is being processed. The amount of time it takes to update a key varies depending on the
     *         ledger size.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KMS_KEY_INACCESSIBLE</code>: The specified customer managed KMS key is not accessible, and the
     *         ledger is impaired. Either the key was disabled or deleted, or the grants on the key were revoked. When a
     *         ledger is impaired, it is not accessible and does not accept any read or write requests.
     *         </p>
     *         <p>
     *         An impaired ledger automatically returns to an active state after you restore the grants on the key, or
     *         re-enable the key that was disabled. However, deleting a customer managed KMS key is irreversible. After
     *         a key is deleted, you can no longer access the ledgers that are protected with that key, and the data
     *         becomes unrecoverable permanently.
     *         </p>
     *         </li>
     * @see EncryptionStatus
     */
    public final EncryptionStatus encryptionStatus() {
        return EncryptionStatus.fromValue(encryptionStatus);
    }

    /**
     * <p>
     * The current state of encryption at rest for the ledger. This can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ENABLED</code>: Encryption is fully enabled using the specified key.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code>: The ledger is actively processing the specified key change.
     * </p>
     * <p>
     * Key changes in QLDB are asynchronous. The ledger is fully accessible without any performance impact while the key
     * change is being processed. The amount of time it takes to update a key varies depending on the ledger size.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KMS_KEY_INACCESSIBLE</code>: The specified customer managed KMS key is not accessible, and the ledger is
     * impaired. Either the key was disabled or deleted, or the grants on the key were revoked. When a ledger is
     * impaired, it is not accessible and does not accept any read or write requests.
     * </p>
     * <p>
     * An impaired ledger automatically returns to an active state after you restore the grants on the key, or re-enable
     * the key that was disabled. However, deleting a customer managed KMS key is irreversible. After a key is deleted,
     * you can no longer access the ledgers that are protected with that key, and the data becomes unrecoverable
     * permanently.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionStatus}
     * will return {@link EncryptionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionStatusAsString}.
     * </p>
     * 
     * @return The current state of encryption at rest for the ledger. This can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ENABLED</code>: Encryption is fully enabled using the specified key.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code>: The ledger is actively processing the specified key change.
     *         </p>
     *         <p>
     *         Key changes in QLDB are asynchronous. The ledger is fully accessible without any performance impact while
     *         the key change is being processed. The amount of time it takes to update a key varies depending on the
     *         ledger size.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KMS_KEY_INACCESSIBLE</code>: The specified customer managed KMS key is not accessible, and the
     *         ledger is impaired. Either the key was disabled or deleted, or the grants on the key were revoked. When a
     *         ledger is impaired, it is not accessible and does not accept any read or write requests.
     *         </p>
     *         <p>
     *         An impaired ledger automatically returns to an active state after you restore the grants on the key, or
     *         re-enable the key that was disabled. However, deleting a customer managed KMS key is irreversible. After
     *         a key is deleted, you can no longer access the ledgers that are protected with that key, and the data
     *         becomes unrecoverable permanently.
     *         </p>
     *         </li>
     * @see EncryptionStatus
     */
    public final String encryptionStatusAsString() {
        return encryptionStatus;
    }

    /**
     * <p>
     * The date and time, in epoch time format, when the KMS key first became inaccessible, in the case of an error.
     * (Epoch time format is the number of seconds that have elapsed since 12:00:00 AM January 1, 1970 UTC.)
     * </p>
     * <p>
     * This parameter is undefined if the KMS key is accessible.
     * </p>
     * 
     * @return The date and time, in epoch time format, when the KMS key first became inaccessible, in the case of an
     *         error. (Epoch time format is the number of seconds that have elapsed since 12:00:00 AM January 1, 1970
     *         UTC.)</p>
     *         <p>
     *         This parameter is undefined if the KMS key is accessible.
     */
    public final Instant inaccessibleKmsKeyDateTime() {
        return inaccessibleKmsKeyDateTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyArn());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(inaccessibleKmsKeyDateTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LedgerEncryptionDescription)) {
            return false;
        }
        LedgerEncryptionDescription other = (LedgerEncryptionDescription) obj;
        return Objects.equals(kmsKeyArn(), other.kmsKeyArn())
                && Objects.equals(encryptionStatusAsString(), other.encryptionStatusAsString())
                && Objects.equals(inaccessibleKmsKeyDateTime(), other.inaccessibleKmsKeyDateTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LedgerEncryptionDescription").add("KmsKeyArn", kmsKeyArn())
                .add("EncryptionStatus", encryptionStatusAsString())
                .add("InaccessibleKmsKeyDateTime", inaccessibleKmsKeyDateTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "KmsKeyArn":
            return Optional.ofNullable(clazz.cast(kmsKeyArn()));
        case "EncryptionStatus":
            return Optional.ofNullable(clazz.cast(encryptionStatusAsString()));
        case "InaccessibleKmsKeyDateTime":
            return Optional.ofNullable(clazz.cast(inaccessibleKmsKeyDateTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LedgerEncryptionDescription, T> g) {
        return obj -> g.apply((LedgerEncryptionDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LedgerEncryptionDescription> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the customer managed KMS key that the ledger uses for encryption at rest.
         * If this parameter is undefined, the ledger uses an Amazon Web Services owned KMS key for encryption. It will
         * display <code>AWS_OWNED_KMS_KEY</code> when updating the ledger's encryption configuration to the Amazon Web
         * Services owned KMS key.
         * </p>
         * 
         * @param kmsKeyArn
         *        The Amazon Resource Name (ARN) of the customer managed KMS key that the ledger uses for encryption at
         *        rest. If this parameter is undefined, the ledger uses an Amazon Web Services owned KMS key for
         *        encryption. It will display <code>AWS_OWNED_KMS_KEY</code> when updating the ledger's encryption
         *        configuration to the Amazon Web Services owned KMS key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyArn(String kmsKeyArn);

        /**
         * <p>
         * The current state of encryption at rest for the ledger. This can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ENABLED</code>: Encryption is fully enabled using the specified key.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code>: The ledger is actively processing the specified key change.
         * </p>
         * <p>
         * Key changes in QLDB are asynchronous. The ledger is fully accessible without any performance impact while the
         * key change is being processed. The amount of time it takes to update a key varies depending on the ledger
         * size.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KMS_KEY_INACCESSIBLE</code>: The specified customer managed KMS key is not accessible, and the ledger
         * is impaired. Either the key was disabled or deleted, or the grants on the key were revoked. When a ledger is
         * impaired, it is not accessible and does not accept any read or write requests.
         * </p>
         * <p>
         * An impaired ledger automatically returns to an active state after you restore the grants on the key, or
         * re-enable the key that was disabled. However, deleting a customer managed KMS key is irreversible. After a
         * key is deleted, you can no longer access the ledgers that are protected with that key, and the data becomes
         * unrecoverable permanently.
         * </p>
         * </li>
         * </ul>
         * 
         * @param encryptionStatus
         *        The current state of encryption at rest for the ledger. This can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ENABLED</code>: Encryption is fully enabled using the specified key.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code>: The ledger is actively processing the specified key change.
         *        </p>
         *        <p>
         *        Key changes in QLDB are asynchronous. The ledger is fully accessible without any performance impact
         *        while the key change is being processed. The amount of time it takes to update a key varies depending
         *        on the ledger size.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KMS_KEY_INACCESSIBLE</code>: The specified customer managed KMS key is not accessible, and the
         *        ledger is impaired. Either the key was disabled or deleted, or the grants on the key were revoked.
         *        When a ledger is impaired, it is not accessible and does not accept any read or write requests.
         *        </p>
         *        <p>
         *        An impaired ledger automatically returns to an active state after you restore the grants on the key,
         *        or re-enable the key that was disabled. However, deleting a customer managed KMS key is irreversible.
         *        After a key is deleted, you can no longer access the ledgers that are protected with that key, and the
         *        data becomes unrecoverable permanently.
         *        </p>
         *        </li>
         * @see EncryptionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionStatus
         */
        Builder encryptionStatus(String encryptionStatus);

        /**
         * <p>
         * The current state of encryption at rest for the ledger. This can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ENABLED</code>: Encryption is fully enabled using the specified key.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code>: The ledger is actively processing the specified key change.
         * </p>
         * <p>
         * Key changes in QLDB are asynchronous. The ledger is fully accessible without any performance impact while the
         * key change is being processed. The amount of time it takes to update a key varies depending on the ledger
         * size.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KMS_KEY_INACCESSIBLE</code>: The specified customer managed KMS key is not accessible, and the ledger
         * is impaired. Either the key was disabled or deleted, or the grants on the key were revoked. When a ledger is
         * impaired, it is not accessible and does not accept any read or write requests.
         * </p>
         * <p>
         * An impaired ledger automatically returns to an active state after you restore the grants on the key, or
         * re-enable the key that was disabled. However, deleting a customer managed KMS key is irreversible. After a
         * key is deleted, you can no longer access the ledgers that are protected with that key, and the data becomes
         * unrecoverable permanently.
         * </p>
         * </li>
         * </ul>
         * 
         * @param encryptionStatus
         *        The current state of encryption at rest for the ledger. This can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ENABLED</code>: Encryption is fully enabled using the specified key.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code>: The ledger is actively processing the specified key change.
         *        </p>
         *        <p>
         *        Key changes in QLDB are asynchronous. The ledger is fully accessible without any performance impact
         *        while the key change is being processed. The amount of time it takes to update a key varies depending
         *        on the ledger size.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KMS_KEY_INACCESSIBLE</code>: The specified customer managed KMS key is not accessible, and the
         *        ledger is impaired. Either the key was disabled or deleted, or the grants on the key were revoked.
         *        When a ledger is impaired, it is not accessible and does not accept any read or write requests.
         *        </p>
         *        <p>
         *        An impaired ledger automatically returns to an active state after you restore the grants on the key,
         *        or re-enable the key that was disabled. However, deleting a customer managed KMS key is irreversible.
         *        After a key is deleted, you can no longer access the ledgers that are protected with that key, and the
         *        data becomes unrecoverable permanently.
         *        </p>
         *        </li>
         * @see EncryptionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionStatus
         */
        Builder encryptionStatus(EncryptionStatus encryptionStatus);

        /**
         * <p>
         * The date and time, in epoch time format, when the KMS key first became inaccessible, in the case of an error.
         * (Epoch time format is the number of seconds that have elapsed since 12:00:00 AM January 1, 1970 UTC.)
         * </p>
         * <p>
         * This parameter is undefined if the KMS key is accessible.
         * </p>
         * 
         * @param inaccessibleKmsKeyDateTime
         *        The date and time, in epoch time format, when the KMS key first became inaccessible, in the case of an
         *        error. (Epoch time format is the number of seconds that have elapsed since 12:00:00 AM January 1, 1970
         *        UTC.)</p>
         *        <p>
         *        This parameter is undefined if the KMS key is accessible.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inaccessibleKmsKeyDateTime(Instant inaccessibleKmsKeyDateTime);
    }

    static final class BuilderImpl implements Builder {
        private String kmsKeyArn;

        private String encryptionStatus;

        private Instant inaccessibleKmsKeyDateTime;

        private BuilderImpl() {
        }

        private BuilderImpl(LedgerEncryptionDescription model) {
            kmsKeyArn(model.kmsKeyArn);
            encryptionStatus(model.encryptionStatus);
            inaccessibleKmsKeyDateTime(model.inaccessibleKmsKeyDateTime);
        }

        public final String getKmsKeyArn() {
            return kmsKeyArn;
        }

        public final void setKmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
        }

        @Override
        public final Builder kmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
            return this;
        }

        public final String getEncryptionStatus() {
            return encryptionStatus;
        }

        public final void setEncryptionStatus(String encryptionStatus) {
            this.encryptionStatus = encryptionStatus;
        }

        @Override
        public final Builder encryptionStatus(String encryptionStatus) {
            this.encryptionStatus = encryptionStatus;
            return this;
        }

        @Override
        public final Builder encryptionStatus(EncryptionStatus encryptionStatus) {
            this.encryptionStatus(encryptionStatus == null ? null : encryptionStatus.toString());
            return this;
        }

        public final Instant getInaccessibleKmsKeyDateTime() {
            return inaccessibleKmsKeyDateTime;
        }

        public final void setInaccessibleKmsKeyDateTime(Instant inaccessibleKmsKeyDateTime) {
            this.inaccessibleKmsKeyDateTime = inaccessibleKmsKeyDateTime;
        }

        @Override
        public final Builder inaccessibleKmsKeyDateTime(Instant inaccessibleKmsKeyDateTime) {
            this.inaccessibleKmsKeyDateTime = inaccessibleKmsKeyDateTime;
            return this;
        }

        @Override
        public LedgerEncryptionDescription build() {
            return new LedgerEncryptionDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
