/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.qldbsession;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.qldbsession.model.BadRequestException;
import software.amazon.awssdk.services.qldbsession.model.InvalidSessionException;
import software.amazon.awssdk.services.qldbsession.model.LimitExceededException;
import software.amazon.awssdk.services.qldbsession.model.OccConflictException;
import software.amazon.awssdk.services.qldbsession.model.QldbSessionException;
import software.amazon.awssdk.services.qldbsession.model.RateExceededException;
import software.amazon.awssdk.services.qldbsession.model.SendCommandRequest;
import software.amazon.awssdk.services.qldbsession.model.SendCommandResponse;
import software.amazon.awssdk.services.qldbsession.transform.SendCommandRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link QldbSessionAsyncClient}.
 *
 * @see QldbSessionAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultQldbSessionAsyncClient implements QldbSessionAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultQldbSessionAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultQldbSessionAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Sends a command to an Amazon QLDB ledger.
     * </p>
     * <note>
     * <p>
     * Instead of interacting directly with this API, we recommend that you use the Amazon QLDB Driver or the QLDB Shell
     * to execute data transactions on a ledger.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you are working with an AWS SDK, use the QLDB Driver. The driver provides a high-level abstraction layer above
     * this <code>qldbsession</code> data plane and manages <code>SendCommand</code> API calls for you. For information
     * and a list of supported programming languages, see <a
     * href="https://docs.aws.amazon.com/qldb/latest/developerguide/getting-started-driver.html">Getting started with
     * the driver</a> in the <i>Amazon QLDB Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you are working with the AWS Command Line Interface (AWS CLI), use the QLDB Shell. The shell is a command line
     * interface that uses the QLDB Driver to interact with a ledger. For information, see <a
     * href="https://docs.aws.amazon.com/qldb/latest/developerguide/data-shell.html">Accessing Amazon QLDB using the
     * QLDB Shell</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param sendCommandRequest
     * @return A Java Future containing the result of the SendCommand operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InvalidSessionException Returned if the session doesn't exist anymore because it timed out or
     *         expired.</li>
     *         <li>OccConflictException Returned when a transaction cannot be written to the journal due to a failure in
     *         the verification phase of <i>optimistic concurrency control</i> (OCC).</li>
     *         <li>RateExceededException Returned when the rate of requests exceeds the allowed throughput.</li>
     *         <li>LimitExceededException Returned if a resource limit such as number of active sessions is exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QldbSessionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QldbSessionAsyncClient.SendCommand
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/qldb-session-2019-07-11/SendCommand" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SendCommandResponse> sendCommand(SendCommandRequest sendCommandRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QLDB Session");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendCommand");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendCommandResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SendCommandResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendCommandResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendCommandRequest, SendCommandResponse>()
                            .withOperationName("SendCommand").withMarshaller(new SendCommandRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(sendCommandRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = sendCommandRequest.overrideConfiguration().orElse(null);
            CompletableFuture<SendCommandResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendCommandRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(QldbSessionException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSessionException")
                                .exceptionBuilderSupplier(InvalidSessionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RateExceededException")
                                .exceptionBuilderSupplier(RateExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OccConflictException")
                                .exceptionBuilderSupplier(OccConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
