/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.quicksight;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.quicksight.model.AccessDeniedException;
import software.amazon.awssdk.services.quicksight.model.CreateGroupMembershipRequest;
import software.amazon.awssdk.services.quicksight.model.CreateGroupMembershipResponse;
import software.amazon.awssdk.services.quicksight.model.CreateGroupRequest;
import software.amazon.awssdk.services.quicksight.model.CreateGroupResponse;
import software.amazon.awssdk.services.quicksight.model.DeleteGroupMembershipRequest;
import software.amazon.awssdk.services.quicksight.model.DeleteGroupMembershipResponse;
import software.amazon.awssdk.services.quicksight.model.DeleteGroupRequest;
import software.amazon.awssdk.services.quicksight.model.DeleteGroupResponse;
import software.amazon.awssdk.services.quicksight.model.DeleteUserRequest;
import software.amazon.awssdk.services.quicksight.model.DeleteUserResponse;
import software.amazon.awssdk.services.quicksight.model.DescribeGroupRequest;
import software.amazon.awssdk.services.quicksight.model.DescribeGroupResponse;
import software.amazon.awssdk.services.quicksight.model.DescribeUserRequest;
import software.amazon.awssdk.services.quicksight.model.DescribeUserResponse;
import software.amazon.awssdk.services.quicksight.model.DomainNotWhitelistedException;
import software.amazon.awssdk.services.quicksight.model.GetDashboardEmbedUrlRequest;
import software.amazon.awssdk.services.quicksight.model.GetDashboardEmbedUrlResponse;
import software.amazon.awssdk.services.quicksight.model.IdentityTypeNotSupportedException;
import software.amazon.awssdk.services.quicksight.model.InternalFailureException;
import software.amazon.awssdk.services.quicksight.model.InvalidNextTokenException;
import software.amazon.awssdk.services.quicksight.model.InvalidParameterValueException;
import software.amazon.awssdk.services.quicksight.model.LimitExceededException;
import software.amazon.awssdk.services.quicksight.model.ListGroupMembershipsRequest;
import software.amazon.awssdk.services.quicksight.model.ListGroupMembershipsResponse;
import software.amazon.awssdk.services.quicksight.model.ListGroupsRequest;
import software.amazon.awssdk.services.quicksight.model.ListGroupsResponse;
import software.amazon.awssdk.services.quicksight.model.ListUserGroupsRequest;
import software.amazon.awssdk.services.quicksight.model.ListUserGroupsResponse;
import software.amazon.awssdk.services.quicksight.model.ListUsersRequest;
import software.amazon.awssdk.services.quicksight.model.ListUsersResponse;
import software.amazon.awssdk.services.quicksight.model.PreconditionNotMetException;
import software.amazon.awssdk.services.quicksight.model.QuickSightException;
import software.amazon.awssdk.services.quicksight.model.QuickSightUserNotFoundException;
import software.amazon.awssdk.services.quicksight.model.RegisterUserRequest;
import software.amazon.awssdk.services.quicksight.model.RegisterUserResponse;
import software.amazon.awssdk.services.quicksight.model.ResourceExistsException;
import software.amazon.awssdk.services.quicksight.model.ResourceNotFoundException;
import software.amazon.awssdk.services.quicksight.model.ResourceUnavailableException;
import software.amazon.awssdk.services.quicksight.model.SessionLifetimeInMinutesInvalidException;
import software.amazon.awssdk.services.quicksight.model.ThrottlingException;
import software.amazon.awssdk.services.quicksight.model.UnsupportedUserEditionException;
import software.amazon.awssdk.services.quicksight.model.UpdateGroupRequest;
import software.amazon.awssdk.services.quicksight.model.UpdateGroupResponse;
import software.amazon.awssdk.services.quicksight.model.UpdateUserRequest;
import software.amazon.awssdk.services.quicksight.model.UpdateUserResponse;
import software.amazon.awssdk.services.quicksight.transform.CreateGroupMembershipRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.CreateGroupRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.DeleteGroupMembershipRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.DeleteGroupRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.DeleteUserRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.DescribeGroupRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.DescribeUserRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.GetDashboardEmbedUrlRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.ListGroupMembershipsRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.ListGroupsRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.ListUserGroupsRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.ListUsersRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.RegisterUserRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.UpdateGroupRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.UpdateUserRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link QuickSightAsyncClient}.
 *
 * @see QuickSightAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultQuickSightAsyncClient implements QuickSightAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultQuickSightAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultQuickSightAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an Amazon QuickSight group.
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;relevant-aws-account-id&gt;</i>:group/default/<i>&lt;group-name&gt;</i> </code>
     * .
     * </p>
     * <p>
     * The response is a group object.
     * </p>
     *
     * @param createGroupRequest
     *        The request object for this operation.
     * @return A Java Future containing the result of the CreateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have access to this.</li>
     *         <li>InvalidParameterValueException One or more parameters don't have a valid value.</li>
     *         <li>ResourceExistsException The resource specified doesn't exist.</li>
     *         <li>ResourceNotFoundException One or more resources couldn't be found.</li>
     *         <li>ThrottlingException Access is throttled.</li>
     *         <li>PreconditionNotMetException One or more preconditions aren't met.</li>
     *         <li>LimitExceededException A limit is exceeded.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceUnavailableException This resource is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QuickSightException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QuickSightAsyncClient.CreateGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/CreateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGroupResponse> createGroup(CreateGroupRequest createGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<CreateGroupRequest, CreateGroupResponse>()
                    .withMarshaller(new CreateGroupRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createGroupRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds an Amazon QuickSight user to an Amazon QuickSight group.
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:group/default/<i>&lt;group-name&gt;</i> </code>.
     * </p>
     * <p>
     * The condition resource is the user name.
     * </p>
     * <p>
     * The condition key is <code>quicksight:UserName</code>.
     * </p>
     * <p>
     * The response is the group member object.
     * </p>
     *
     * @param createGroupMembershipRequest
     * @return A Java Future containing the result of the CreateGroupMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have access to this.</li>
     *         <li>InvalidParameterValueException One or more parameters don't have a valid value.</li>
     *         <li>ResourceNotFoundException One or more resources couldn't be found.</li>
     *         <li>ThrottlingException Access is throttled.</li>
     *         <li>PreconditionNotMetException One or more preconditions aren't met.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceUnavailableException This resource is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QuickSightException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QuickSightAsyncClient.CreateGroupMembership
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/CreateGroupMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGroupMembershipResponse> createGroupMembership(
            CreateGroupMembershipRequest createGroupMembershipRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGroupMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateGroupMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<CreateGroupMembershipRequest, CreateGroupMembershipResponse>()
                    .withMarshaller(new CreateGroupMembershipRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(createGroupMembershipRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a user group from Amazon QuickSight.
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:group/default/<i>&lt;group-name&gt;</i> </code>.
     * </p>
     *
     * @param deleteGroupRequest
     * @return A Java Future containing the result of the DeleteGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have access to this.</li>
     *         <li>InvalidParameterValueException One or more parameters don't have a valid value.</li>
     *         <li>ResourceNotFoundException One or more resources couldn't be found.</li>
     *         <li>ThrottlingException Access is throttled.</li>
     *         <li>PreconditionNotMetException One or more preconditions aren't met.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceUnavailableException This resource is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QuickSightException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QuickSightAsyncClient.DeleteGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/DeleteGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGroupResponse> deleteGroup(DeleteGroupRequest deleteGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DeleteGroupRequest, DeleteGroupResponse>()
                    .withMarshaller(new DeleteGroupRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteGroupRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a user from a group so that the user is no longer a member of the group.
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:group/default/<i>&lt;group-name&gt;</i> </code>.
     * </p>
     * <p>
     * The condition resource is the user name.
     * </p>
     * <p>
     * The condition key is <code>quicksight:UserName</code>.
     * </p>
     *
     * @param deleteGroupMembershipRequest
     * @return A Java Future containing the result of the DeleteGroupMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have access to this.</li>
     *         <li>InvalidParameterValueException One or more parameters don't have a valid value.</li>
     *         <li>ResourceNotFoundException One or more resources couldn't be found.</li>
     *         <li>ThrottlingException Access is throttled.</li>
     *         <li>PreconditionNotMetException One or more preconditions aren't met.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceUnavailableException This resource is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QuickSightException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QuickSightAsyncClient.DeleteGroupMembership
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/DeleteGroupMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGroupMembershipResponse> deleteGroupMembership(
            DeleteGroupMembershipRequest deleteGroupMembershipRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGroupMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteGroupMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DeleteGroupMembershipRequest, DeleteGroupMembershipResponse>()
                    .withMarshaller(new DeleteGroupMembershipRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(deleteGroupMembershipRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the Amazon QuickSight user that is associated with the identity of the AWS Identity and Access Management
     * (IAM) user or role that's making the call. The IAM user isn't deleted as a result of this call.
     * </p>
     * <p>
     * The permission resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:user/default/<i>&lt;user-name&gt; </i> </code>.
     * </p>
     *
     * @param deleteUserRequest
     * @return A Java Future containing the result of the DeleteUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have access to this.</li>
     *         <li>InvalidParameterValueException One or more parameters don't have a valid value.</li>
     *         <li>ResourceNotFoundException One or more resources couldn't be found.</li>
     *         <li>ThrottlingException Access is throttled.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceUnavailableException This resource is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QuickSightException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QuickSightAsyncClient.DeleteUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteUserResponse> deleteUser(DeleteUserRequest deleteUserRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DeleteUserRequest, DeleteUserResponse>()
                    .withMarshaller(new DeleteUserRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteUserRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an Amazon QuickSight group's description and Amazon Resource Name (ARN).
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;relevant-aws-account-id&gt;</i>:group/default/<i>&lt;group-name&gt;</i> </code>
     * .
     * </p>
     * <p>
     * The response is the group object.
     * </p>
     *
     * @param describeGroupRequest
     * @return A Java Future containing the result of the DescribeGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have access to this.</li>
     *         <li>InvalidParameterValueException One or more parameters don't have a valid value.</li>
     *         <li>ResourceNotFoundException One or more resources couldn't be found.</li>
     *         <li>ThrottlingException Access is throttled.</li>
     *         <li>PreconditionNotMetException One or more preconditions aren't met.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceUnavailableException This resource is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QuickSightException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QuickSightAsyncClient.DescribeGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/DescribeGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGroupResponse> describeGroup(DescribeGroupRequest describeGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DescribeGroupRequest, DescribeGroupResponse>()
                    .withMarshaller(new DescribeGroupRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeGroupRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a user, given the user name.
     * </p>
     * <p>
     * The permission resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:user/default/<i>&lt;user-name&gt;</i> </code>.
     * </p>
     * <p>
     * The response is a user object that contains the user's Amazon Resource Name (ARN), AWS Identity and Access
     * Management (IAM) role, and email address.
     * </p>
     *
     * @param describeUserRequest
     * @return A Java Future containing the result of the DescribeUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have access to this.</li>
     *         <li>InvalidParameterValueException One or more parameters don't have a valid value.</li>
     *         <li>ResourceNotFoundException One or more resources couldn't be found.</li>
     *         <li>ThrottlingException Access is throttled.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceUnavailableException This resource is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QuickSightException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QuickSightAsyncClient.DescribeUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/DescribeUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeUserResponse> describeUser(DescribeUserRequest describeUserRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DescribeUserRequest, DescribeUserResponse>()
                    .withMarshaller(new DescribeUserRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeUserRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates an embedded URL and authorization code. Before this can work properly, you need to configure the
     * dashboards and user permissions first.
     * </p>
     *
     * @param getDashboardEmbedUrlRequest
     * @return A Java Future containing the result of the GetDashboardEmbedUrl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have access to this.</li>
     *         <li>InvalidParameterValueException One or more parameters don't have a valid value.</li>
     *         <li>ResourceExistsException The resource specified doesn't exist.</li>
     *         <li>ResourceNotFoundException One or more resources couldn't be found.</li>
     *         <li>ThrottlingException Access is throttled.</li>
     *         <li>PreconditionNotMetException One or more preconditions aren't met.</li>
     *         <li>DomainNotWhitelistedException The domain specified is not on the allowlist. All domains for embedded
     *         dashboards must be added to the approved list by an Amazon QuickSight admin.</li>
     *         <li>QuickSightUserNotFoundException The user is not found. This could happen in any operation that
     *         requires finding a user based on the provided user name, such as DeleteUser, DescribeUser, and so on.</li>
     *         <li>IdentityTypeNotSupportedException The identity type specified is not supported. Supported identity
     *         types include: IAM and QUICKSIGHT.</li>
     *         <li>SessionLifetimeInMinutesInvalidException The number of minutes specified for the lifetime of a
     *         session is invalid. The session lifetime must be between 15 and 600 minutes.</li>
     *         <li>UnsupportedUserEditionException Indicates that you are calling an operation on an Amazon QuickSight
     *         subscription where the edition does not include support for that operation. Amazon QuickSight currently
     *         has Standard Edition and Enterprise Edition. Not every operation and capability is available in every
     *         edition.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceUnavailableException This resource is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QuickSightException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QuickSightAsyncClient.GetDashboardEmbedUrl
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/GetDashboardEmbedUrl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDashboardEmbedUrlResponse> getDashboardEmbedUrl(
            GetDashboardEmbedUrlRequest getDashboardEmbedUrlRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDashboardEmbedUrlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDashboardEmbedUrlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<GetDashboardEmbedUrlRequest, GetDashboardEmbedUrlResponse>()
                    .withMarshaller(new GetDashboardEmbedUrlRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(getDashboardEmbedUrlRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists member users in a group.
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:group/default/<i>&lt;group-name&gt;</i> </code>.
     * </p>
     * <p>
     * The response is a list of group member objects.
     * </p>
     *
     * @param listGroupMembershipsRequest
     * @return A Java Future containing the result of the ListGroupMemberships operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have access to this.</li>
     *         <li>InvalidParameterValueException One or more parameters don't have a valid value.</li>
     *         <li>ResourceNotFoundException One or more resources couldn't be found.</li>
     *         <li>ThrottlingException Access is throttled.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value isn't valid.</li>
     *         <li>PreconditionNotMetException One or more preconditions aren't met.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceUnavailableException This resource is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QuickSightException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QuickSightAsyncClient.ListGroupMemberships
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/ListGroupMemberships"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListGroupMembershipsResponse> listGroupMemberships(
            ListGroupMembershipsRequest listGroupMembershipsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGroupMembershipsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGroupMembershipsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListGroupMembershipsRequest, ListGroupMembershipsResponse>()
                    .withMarshaller(new ListGroupMembershipsRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(listGroupMembershipsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all user groups in Amazon QuickSight.
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:group/default/*</code>.
     * </p>
     * <p>
     * The response is a list of group objects.
     * </p>
     *
     * @param listGroupsRequest
     * @return A Java Future containing the result of the ListGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have access to this.</li>
     *         <li>InvalidParameterValueException One or more parameters don't have a valid value.</li>
     *         <li>ResourceNotFoundException One or more resources couldn't be found.</li>
     *         <li>ThrottlingException Access is throttled.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value isn't valid.</li>
     *         <li>PreconditionNotMetException One or more preconditions aren't met.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceUnavailableException This resource is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QuickSightException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QuickSightAsyncClient.ListGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/ListGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListGroupsResponse> listGroups(ListGroupsRequest listGroupsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListGroupsRequest, ListGroupsResponse>()
                    .withMarshaller(new ListGroupsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listGroupsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Amazon QuickSight groups that a user is part of.
     * </p>
     *
     * @param listUserGroupsRequest
     * @return A Java Future containing the result of the ListUserGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have access to this.</li>
     *         <li>InvalidParameterValueException One or more parameters don't have a valid value.</li>
     *         <li>ResourceNotFoundException One or more resources couldn't be found.</li>
     *         <li>ThrottlingException Access is throttled.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceUnavailableException This resource is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QuickSightException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QuickSightAsyncClient.ListUserGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/ListUserGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListUserGroupsResponse> listUserGroups(ListUserGroupsRequest listUserGroupsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListUserGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListUserGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListUserGroupsRequest, ListUserGroupsResponse>()
                    .withMarshaller(new ListUserGroupsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listUserGroupsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all of the Amazon QuickSight users belonging to this account.
     * </p>
     * <p>
     * The permission resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:user/default/<i>*</i> </code>.
     * </p>
     * <p>
     * The response is a list of user objects, containing each user's Amazon Resource Name (ARN), AWS Identity and
     * Access Management (IAM) role, and email address.
     * </p>
     *
     * @param listUsersRequest
     * @return A Java Future containing the result of the ListUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have access to this.</li>
     *         <li>InvalidParameterValueException One or more parameters don't have a valid value.</li>
     *         <li>ResourceNotFoundException One or more resources couldn't be found.</li>
     *         <li>ThrottlingException Access is throttled.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value isn't valid.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceUnavailableException This resource is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QuickSightException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QuickSightAsyncClient.ListUsers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListUsersResponse> listUsers(ListUsersRequest listUsersRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListUsersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListUsersRequest, ListUsersResponse>()
                    .withMarshaller(new ListUsersRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listUsersRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon QuickSight user, whose identity is associated with the AWS Identity and Access Management (IAM)
     * identity or role specified in the request.
     * </p>
     * <p>
     * The permission resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:user/default/<i>&lt;user-name&gt;</i> </code>.
     * </p>
     * <p>
     * The condition resource is the Amazon Resource Name (ARN) for the IAM user or role, and the session name.
     * </p>
     * <p>
     * The condition keys are <code>quicksight:IamArn</code> and <code>quicksight:SessionName</code>.
     * </p>
     *
     * @param registerUserRequest
     * @return A Java Future containing the result of the RegisterUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have access to this.</li>
     *         <li>InvalidParameterValueException One or more parameters don't have a valid value.</li>
     *         <li>ResourceNotFoundException One or more resources couldn't be found.</li>
     *         <li>ThrottlingException Access is throttled.</li>
     *         <li>LimitExceededException A limit is exceeded.</li>
     *         <li>ResourceExistsException The resource specified doesn't exist.</li>
     *         <li>PreconditionNotMetException One or more preconditions aren't met.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceUnavailableException This resource is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QuickSightException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QuickSightAsyncClient.RegisterUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/RegisterUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterUserResponse> registerUser(RegisterUserRequest registerUserRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RegisterUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<RegisterUserRequest, RegisterUserResponse>()
                    .withMarshaller(new RegisterUserRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(registerUserRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Changes a group description.
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:group/default/<i>&lt;group-name&gt;</i> </code>.
     * </p>
     * <p>
     * The response is a group object.
     * </p>
     *
     * @param updateGroupRequest
     * @return A Java Future containing the result of the UpdateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have access to this.</li>
     *         <li>InvalidParameterValueException One or more parameters don't have a valid value.</li>
     *         <li>ResourceNotFoundException One or more resources couldn't be found.</li>
     *         <li>ThrottlingException Access is throttled.</li>
     *         <li>PreconditionNotMetException One or more preconditions aren't met.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceUnavailableException This resource is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QuickSightException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QuickSightAsyncClient.UpdateGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/UpdateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGroupResponse> updateGroup(UpdateGroupRequest updateGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<UpdateGroupRequest, UpdateGroupResponse>()
                    .withMarshaller(new UpdateGroupRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateGroupRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an Amazon QuickSight user.
     * </p>
     *
     * @param updateUserRequest
     * @return A Java Future containing the result of the UpdateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have access to this.</li>
     *         <li>InvalidParameterValueException One or more parameters don't have a valid value.</li>
     *         <li>ResourceNotFoundException One or more resources couldn't be found.</li>
     *         <li>ThrottlingException Access is throttled.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceUnavailableException This resource is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QuickSightException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QuickSightAsyncClient.UpdateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateUserResponse> updateUser(UpdateUserRequest updateUserRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<UpdateUserRequest, UpdateUserResponse>()
                    .withMarshaller(new UpdateUserRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateUserRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(QuickSightException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DomainNotWhitelistedException")
                                .exceptionBuilderSupplier(DomainNotWhitelistedException::builder).httpStatusCode(407).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("QuickSightUserNotFoundException")
                                .exceptionBuilderSupplier(QuickSightUserNotFoundException::builder).httpStatusCode(412).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdentityTypeNotSupportedException")
                                .exceptionBuilderSupplier(IdentityTypeNotSupportedException::builder).httpStatusCode(416).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceUnavailableException")
                                .exceptionBuilderSupplier(ResourceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceExistsException")
                                .exceptionBuilderSupplier(ResourceExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SessionLifetimeInMinutesInvalidException")
                                .exceptionBuilderSupplier(SessionLifetimeInMinutesInvalidException::builder).httpStatusCode(418)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedUserEditionException")
                                .exceptionBuilderSupplier(UnsupportedUserEditionException::builder).httpStatusCode(419).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PreconditionNotMetException")
                                .exceptionBuilderSupplier(PreconditionNotMetException::builder).httpStatusCode(400).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
