/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.quicksight.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A <i>logical table</i> is a unit that joins and that data transformations operate on. A logical table has a source,
 * which can be either a physical table or result of a join. When a logical table points to a physical table, the
 * logical table acts as a mutable copy of that physical table through transform operations.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LogicalTable implements SdkPojo, Serializable, ToCopyableBuilder<LogicalTable.Builder, LogicalTable> {
    private static final SdkField<String> ALIAS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Alias")
            .getter(getter(LogicalTable::alias)).setter(setter(Builder::alias))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Alias").build()).build();

    private static final SdkField<List<TransformOperation>> DATA_TRANSFORMS_FIELD = SdkField
            .<List<TransformOperation>> builder(MarshallingType.LIST)
            .memberName("DataTransforms")
            .getter(getter(LogicalTable::dataTransforms))
            .setter(setter(Builder::dataTransforms))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataTransforms").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TransformOperation> builder(MarshallingType.SDK_POJO)
                                            .constructor(TransformOperation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<LogicalTableSource> SOURCE_FIELD = SdkField
            .<LogicalTableSource> builder(MarshallingType.SDK_POJO).memberName("Source").getter(getter(LogicalTable::source))
            .setter(setter(Builder::source)).constructor(LogicalTableSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Source").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALIAS_FIELD,
            DATA_TRANSFORMS_FIELD, SOURCE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String alias;

    private final List<TransformOperation> dataTransforms;

    private final LogicalTableSource source;

    private LogicalTable(BuilderImpl builder) {
        this.alias = builder.alias;
        this.dataTransforms = builder.dataTransforms;
        this.source = builder.source;
    }

    /**
     * <p>
     * A display name for the logical table.
     * </p>
     * 
     * @return A display name for the logical table.
     */
    public String alias() {
        return alias;
    }

    /**
     * Returns true if the DataTransforms property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDataTransforms() {
        return dataTransforms != null && !(dataTransforms instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Transform operations that act on this logical table.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDataTransforms()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Transform operations that act on this logical table.
     */
    public List<TransformOperation> dataTransforms() {
        return dataTransforms;
    }

    /**
     * <p>
     * Source of this logical table.
     * </p>
     * 
     * @return Source of this logical table.
     */
    public LogicalTableSource source() {
        return source;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(alias());
        hashCode = 31 * hashCode + Objects.hashCode(dataTransforms());
        hashCode = 31 * hashCode + Objects.hashCode(source());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LogicalTable)) {
            return false;
        }
        LogicalTable other = (LogicalTable) obj;
        return Objects.equals(alias(), other.alias()) && Objects.equals(dataTransforms(), other.dataTransforms())
                && Objects.equals(source(), other.source());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("LogicalTable").add("Alias", alias()).add("DataTransforms", dataTransforms())
                .add("Source", source()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Alias":
            return Optional.ofNullable(clazz.cast(alias()));
        case "DataTransforms":
            return Optional.ofNullable(clazz.cast(dataTransforms()));
        case "Source":
            return Optional.ofNullable(clazz.cast(source()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LogicalTable, T> g) {
        return obj -> g.apply((LogicalTable) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LogicalTable> {
        /**
         * <p>
         * A display name for the logical table.
         * </p>
         * 
         * @param alias
         *        A display name for the logical table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alias(String alias);

        /**
         * <p>
         * Transform operations that act on this logical table.
         * </p>
         * 
         * @param dataTransforms
         *        Transform operations that act on this logical table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataTransforms(Collection<TransformOperation> dataTransforms);

        /**
         * <p>
         * Transform operations that act on this logical table.
         * </p>
         * 
         * @param dataTransforms
         *        Transform operations that act on this logical table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataTransforms(TransformOperation... dataTransforms);

        /**
         * <p>
         * Transform operations that act on this logical table.
         * </p>
         * This is a convenience that creates an instance of the {@link List<TransformOperation>.Builder} avoiding the
         * need to create one manually via {@link List<TransformOperation>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TransformOperation>.Builder#build()} is called immediately
         * and its result is passed to {@link #dataTransforms(List<TransformOperation>)}.
         * 
         * @param dataTransforms
         *        a consumer that will call methods on {@link List<TransformOperation>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataTransforms(List<TransformOperation>)
         */
        Builder dataTransforms(Consumer<TransformOperation.Builder>... dataTransforms);

        /**
         * <p>
         * Source of this logical table.
         * </p>
         * 
         * @param source
         *        Source of this logical table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(LogicalTableSource source);

        /**
         * <p>
         * Source of this logical table.
         * </p>
         * This is a convenience that creates an instance of the {@link LogicalTableSource.Builder} avoiding the need to
         * create one manually via {@link LogicalTableSource#builder()}.
         *
         * When the {@link Consumer} completes, {@link LogicalTableSource.Builder#build()} is called immediately and its
         * result is passed to {@link #source(LogicalTableSource)}.
         * 
         * @param source
         *        a consumer that will call methods on {@link LogicalTableSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #source(LogicalTableSource)
         */
        default Builder source(Consumer<LogicalTableSource.Builder> source) {
            return source(LogicalTableSource.builder().applyMutation(source).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String alias;

        private List<TransformOperation> dataTransforms = DefaultSdkAutoConstructList.getInstance();

        private LogicalTableSource source;

        private BuilderImpl() {
        }

        private BuilderImpl(LogicalTable model) {
            alias(model.alias);
            dataTransforms(model.dataTransforms);
            source(model.source);
        }

        public final String getAlias() {
            return alias;
        }

        @Override
        public final Builder alias(String alias) {
            this.alias = alias;
            return this;
        }

        public final void setAlias(String alias) {
            this.alias = alias;
        }

        public final Collection<TransformOperation.Builder> getDataTransforms() {
            return dataTransforms != null ? dataTransforms.stream().map(TransformOperation::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder dataTransforms(Collection<TransformOperation> dataTransforms) {
            this.dataTransforms = TransformOperationListCopier.copy(dataTransforms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dataTransforms(TransformOperation... dataTransforms) {
            dataTransforms(Arrays.asList(dataTransforms));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dataTransforms(Consumer<TransformOperation.Builder>... dataTransforms) {
            dataTransforms(Stream.of(dataTransforms).map(c -> TransformOperation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDataTransforms(Collection<TransformOperation.BuilderImpl> dataTransforms) {
            this.dataTransforms = TransformOperationListCopier.copyFromBuilder(dataTransforms);
        }

        public final LogicalTableSource.Builder getSource() {
            return source != null ? source.toBuilder() : null;
        }

        @Override
        public final Builder source(LogicalTableSource source) {
            this.source = source;
            return this;
        }

        public final void setSource(LogicalTableSource.BuilderImpl source) {
            this.source = source != null ? source.build() : null;
        }

        @Override
        public LogicalTable build() {
            return new LogicalTable(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
