/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.quicksight.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The structure of a data source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataSource implements SdkPojo, Serializable, ToCopyableBuilder<DataSource.Builder, DataSource> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(DataSource::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> DATA_SOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataSourceId").getter(getter(DataSource::dataSourceId)).setter(setter(Builder::dataSourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataSourceId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(DataSource::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(DataSource::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(DataSource::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime").getter(getter(DataSource::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedTime").getter(getter(DataSource::lastUpdatedTime)).setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTime").build()).build();

    private static final SdkField<DataSourceParameters> DATA_SOURCE_PARAMETERS_FIELD = SdkField
            .<DataSourceParameters> builder(MarshallingType.SDK_POJO).memberName("DataSourceParameters")
            .getter(getter(DataSource::dataSourceParameters)).setter(setter(Builder::dataSourceParameters))
            .constructor(DataSourceParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataSourceParameters").build())
            .build();

    private static final SdkField<List<DataSourceParameters>> ALTERNATE_DATA_SOURCE_PARAMETERS_FIELD = SdkField
            .<List<DataSourceParameters>> builder(MarshallingType.LIST)
            .memberName("AlternateDataSourceParameters")
            .getter(getter(DataSource::alternateDataSourceParameters))
            .setter(setter(Builder::alternateDataSourceParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlternateDataSourceParameters")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataSourceParameters> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataSourceParameters::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<VpcConnectionProperties> VPC_CONNECTION_PROPERTIES_FIELD = SdkField
            .<VpcConnectionProperties> builder(MarshallingType.SDK_POJO).memberName("VpcConnectionProperties")
            .getter(getter(DataSource::vpcConnectionProperties)).setter(setter(Builder::vpcConnectionProperties))
            .constructor(VpcConnectionProperties::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConnectionProperties").build())
            .build();

    private static final SdkField<SslProperties> SSL_PROPERTIES_FIELD = SdkField
            .<SslProperties> builder(MarshallingType.SDK_POJO).memberName("SslProperties")
            .getter(getter(DataSource::sslProperties)).setter(setter(Builder::sslProperties)).constructor(SslProperties::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SslProperties").build()).build();

    private static final SdkField<DataSourceErrorInfo> ERROR_INFO_FIELD = SdkField
            .<DataSourceErrorInfo> builder(MarshallingType.SDK_POJO).memberName("ErrorInfo")
            .getter(getter(DataSource::errorInfo)).setter(setter(Builder::errorInfo)).constructor(DataSourceErrorInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorInfo").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD,
            DATA_SOURCE_ID_FIELD, NAME_FIELD, TYPE_FIELD, STATUS_FIELD, CREATED_TIME_FIELD, LAST_UPDATED_TIME_FIELD,
            DATA_SOURCE_PARAMETERS_FIELD, ALTERNATE_DATA_SOURCE_PARAMETERS_FIELD, VPC_CONNECTION_PROPERTIES_FIELD,
            SSL_PROPERTIES_FIELD, ERROR_INFO_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String dataSourceId;

    private final String name;

    private final String type;

    private final String status;

    private final Instant createdTime;

    private final Instant lastUpdatedTime;

    private final DataSourceParameters dataSourceParameters;

    private final List<DataSourceParameters> alternateDataSourceParameters;

    private final VpcConnectionProperties vpcConnectionProperties;

    private final SslProperties sslProperties;

    private final DataSourceErrorInfo errorInfo;

    private DataSource(BuilderImpl builder) {
        this.arn = builder.arn;
        this.dataSourceId = builder.dataSourceId;
        this.name = builder.name;
        this.type = builder.type;
        this.status = builder.status;
        this.createdTime = builder.createdTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
        this.dataSourceParameters = builder.dataSourceParameters;
        this.alternateDataSourceParameters = builder.alternateDataSourceParameters;
        this.vpcConnectionProperties = builder.vpcConnectionProperties;
        this.sslProperties = builder.sslProperties;
        this.errorInfo = builder.errorInfo;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the data source.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the data source.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The ID of the data source. This ID is unique per Region; for each Amazon Web Services account;.
     * </p>
     * 
     * @return The ID of the data source. This ID is unique per Region; for each Amazon Web Services account;.
     */
    public final String dataSourceId() {
        return dataSourceId;
    }

    /**
     * <p>
     * A display name for the data source.
     * </p>
     * 
     * @return A display name for the data source.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The type of the data source. This type indicates which database engine the data source connects to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DataSourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the data source. This type indicates which database engine the data source connects to.
     * @see DataSourceType
     */
    public final DataSourceType type() {
        return DataSourceType.fromValue(type);
    }

    /**
     * <p>
     * The type of the data source. This type indicates which database engine the data source connects to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DataSourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the data source. This type indicates which database engine the data source connects to.
     * @see DataSourceType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The HTTP status of the request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The HTTP status of the request.
     * @see ResourceStatus
     */
    public final ResourceStatus status() {
        return ResourceStatus.fromValue(status);
    }

    /**
     * <p>
     * The HTTP status of the request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The HTTP status of the request.
     * @see ResourceStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The time that this data source was created.
     * </p>
     * 
     * @return The time that this data source was created.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The last time that this data source was updated.
     * </p>
     * 
     * @return The last time that this data source was updated.
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    /**
     * <p>
     * The parameters that Amazon QuickSight uses to connect to your underlying source. This is a variant type
     * structure. For this structure to be valid, only one of the attributes can be non-null.
     * </p>
     * 
     * @return The parameters that Amazon QuickSight uses to connect to your underlying source. This is a variant type
     *         structure. For this structure to be valid, only one of the attributes can be non-null.
     */
    public final DataSourceParameters dataSourceParameters() {
        return dataSourceParameters;
    }

    /**
     * Returns true if the AlternateDataSourceParameters property was specified by the sender (it may be empty), or
     * false if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender
     * is the AWS service.
     */
    public final boolean hasAlternateDataSourceParameters() {
        return alternateDataSourceParameters != null && !(alternateDataSourceParameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of alternate data source parameters that you want to share for the credentials stored with this data
     * source. The credentials are applied in tandem with the data source parameters when you copy a data source by
     * using a create or update request. The API operation compares the <code>DataSourceParameters</code> structure
     * that's in the request with the structures in the <code>AlternateDataSourceParameters</code> allow list. If the
     * structures are an exact match, the request is allowed to use the credentials from this existing data source. If
     * the <code>AlternateDataSourceParameters</code> list is null, the <code>Credentials</code> originally used with
     * this <code>DataSourceParameters</code> are automatically allowed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAlternateDataSourceParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A set of alternate data source parameters that you want to share for the credentials stored with this
     *         data source. The credentials are applied in tandem with the data source parameters when you copy a data
     *         source by using a create or update request. The API operation compares the
     *         <code>DataSourceParameters</code> structure that's in the request with the structures in the
     *         <code>AlternateDataSourceParameters</code> allow list. If the structures are an exact match, the request
     *         is allowed to use the credentials from this existing data source. If the
     *         <code>AlternateDataSourceParameters</code> list is null, the <code>Credentials</code> originally used
     *         with this <code>DataSourceParameters</code> are automatically allowed.
     */
    public final List<DataSourceParameters> alternateDataSourceParameters() {
        return alternateDataSourceParameters;
    }

    /**
     * <p>
     * The VPC connection information. You need to use this parameter only when you want QuickSight to use a VPC
     * connection when connecting to your underlying source.
     * </p>
     * 
     * @return The VPC connection information. You need to use this parameter only when you want QuickSight to use a VPC
     *         connection when connecting to your underlying source.
     */
    public final VpcConnectionProperties vpcConnectionProperties() {
        return vpcConnectionProperties;
    }

    /**
     * <p>
     * Secure Socket Layer (SSL) properties that apply when QuickSight connects to your underlying source.
     * </p>
     * 
     * @return Secure Socket Layer (SSL) properties that apply when QuickSight connects to your underlying source.
     */
    public final SslProperties sslProperties() {
        return sslProperties;
    }

    /**
     * <p>
     * Error information from the last update or the creation of the data source.
     * </p>
     * 
     * @return Error information from the last update or the creation of the data source.
     */
    public final DataSourceErrorInfo errorInfo() {
        return errorInfo;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(dataSourceId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(dataSourceParameters());
        hashCode = 31 * hashCode + Objects.hashCode(hasAlternateDataSourceParameters() ? alternateDataSourceParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vpcConnectionProperties());
        hashCode = 31 * hashCode + Objects.hashCode(sslProperties());
        hashCode = 31 * hashCode + Objects.hashCode(errorInfo());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataSource)) {
            return false;
        }
        DataSource other = (DataSource) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(dataSourceId(), other.dataSourceId())
                && Objects.equals(name(), other.name()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime())
                && Objects.equals(dataSourceParameters(), other.dataSourceParameters())
                && hasAlternateDataSourceParameters() == other.hasAlternateDataSourceParameters()
                && Objects.equals(alternateDataSourceParameters(), other.alternateDataSourceParameters())
                && Objects.equals(vpcConnectionProperties(), other.vpcConnectionProperties())
                && Objects.equals(sslProperties(), other.sslProperties()) && Objects.equals(errorInfo(), other.errorInfo());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("DataSource")
                .add("Arn", arn())
                .add("DataSourceId", dataSourceId())
                .add("Name", name())
                .add("Type", typeAsString())
                .add("Status", statusAsString())
                .add("CreatedTime", createdTime())
                .add("LastUpdatedTime", lastUpdatedTime())
                .add("DataSourceParameters", dataSourceParameters())
                .add("AlternateDataSourceParameters", hasAlternateDataSourceParameters() ? alternateDataSourceParameters() : null)
                .add("VpcConnectionProperties", vpcConnectionProperties()).add("SslProperties", sslProperties())
                .add("ErrorInfo", errorInfo()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "DataSourceId":
            return Optional.ofNullable(clazz.cast(dataSourceId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "LastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        case "DataSourceParameters":
            return Optional.ofNullable(clazz.cast(dataSourceParameters()));
        case "AlternateDataSourceParameters":
            return Optional.ofNullable(clazz.cast(alternateDataSourceParameters()));
        case "VpcConnectionProperties":
            return Optional.ofNullable(clazz.cast(vpcConnectionProperties()));
        case "SslProperties":
            return Optional.ofNullable(clazz.cast(sslProperties()));
        case "ErrorInfo":
            return Optional.ofNullable(clazz.cast(errorInfo()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DataSource, T> g) {
        return obj -> g.apply((DataSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataSource> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the data source.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The ID of the data source. This ID is unique per Region; for each Amazon Web Services account;.
         * </p>
         * 
         * @param dataSourceId
         *        The ID of the data source. This ID is unique per Region; for each Amazon Web Services account;.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSourceId(String dataSourceId);

        /**
         * <p>
         * A display name for the data source.
         * </p>
         * 
         * @param name
         *        A display name for the data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of the data source. This type indicates which database engine the data source connects to.
         * </p>
         * 
         * @param type
         *        The type of the data source. This type indicates which database engine the data source connects to.
         * @see DataSourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataSourceType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the data source. This type indicates which database engine the data source connects to.
         * </p>
         * 
         * @param type
         *        The type of the data source. This type indicates which database engine the data source connects to.
         * @see DataSourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataSourceType
         */
        Builder type(DataSourceType type);

        /**
         * <p>
         * The HTTP status of the request.
         * </p>
         * 
         * @param status
         *        The HTTP status of the request.
         * @see ResourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The HTTP status of the request.
         * </p>
         * 
         * @param status
         *        The HTTP status of the request.
         * @see ResourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceStatus
         */
        Builder status(ResourceStatus status);

        /**
         * <p>
         * The time that this data source was created.
         * </p>
         * 
         * @param createdTime
         *        The time that this data source was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The last time that this data source was updated.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The last time that this data source was updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);

        /**
         * <p>
         * The parameters that Amazon QuickSight uses to connect to your underlying source. This is a variant type
         * structure. For this structure to be valid, only one of the attributes can be non-null.
         * </p>
         * 
         * @param dataSourceParameters
         *        The parameters that Amazon QuickSight uses to connect to your underlying source. This is a variant
         *        type structure. For this structure to be valid, only one of the attributes can be non-null.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSourceParameters(DataSourceParameters dataSourceParameters);

        /**
         * <p>
         * The parameters that Amazon QuickSight uses to connect to your underlying source. This is a variant type
         * structure. For this structure to be valid, only one of the attributes can be non-null.
         * </p>
         * This is a convenience that creates an instance of the {@link DataSourceParameters.Builder} avoiding the need
         * to create one manually via {@link DataSourceParameters#builder()}.
         *
         * When the {@link Consumer} completes, {@link DataSourceParameters.Builder#build()} is called immediately and
         * its result is passed to {@link #dataSourceParameters(DataSourceParameters)}.
         * 
         * @param dataSourceParameters
         *        a consumer that will call methods on {@link DataSourceParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataSourceParameters(DataSourceParameters)
         */
        default Builder dataSourceParameters(Consumer<DataSourceParameters.Builder> dataSourceParameters) {
            return dataSourceParameters(DataSourceParameters.builder().applyMutation(dataSourceParameters).build());
        }

        /**
         * <p>
         * A set of alternate data source parameters that you want to share for the credentials stored with this data
         * source. The credentials are applied in tandem with the data source parameters when you copy a data source by
         * using a create or update request. The API operation compares the <code>DataSourceParameters</code> structure
         * that's in the request with the structures in the <code>AlternateDataSourceParameters</code> allow list. If
         * the structures are an exact match, the request is allowed to use the credentials from this existing data
         * source. If the <code>AlternateDataSourceParameters</code> list is null, the <code>Credentials</code>
         * originally used with this <code>DataSourceParameters</code> are automatically allowed.
         * </p>
         * 
         * @param alternateDataSourceParameters
         *        A set of alternate data source parameters that you want to share for the credentials stored with this
         *        data source. The credentials are applied in tandem with the data source parameters when you copy a
         *        data source by using a create or update request. The API operation compares the
         *        <code>DataSourceParameters</code> structure that's in the request with the structures in the
         *        <code>AlternateDataSourceParameters</code> allow list. If the structures are an exact match, the
         *        request is allowed to use the credentials from this existing data source. If the
         *        <code>AlternateDataSourceParameters</code> list is null, the <code>Credentials</code> originally used
         *        with this <code>DataSourceParameters</code> are automatically allowed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alternateDataSourceParameters(Collection<DataSourceParameters> alternateDataSourceParameters);

        /**
         * <p>
         * A set of alternate data source parameters that you want to share for the credentials stored with this data
         * source. The credentials are applied in tandem with the data source parameters when you copy a data source by
         * using a create or update request. The API operation compares the <code>DataSourceParameters</code> structure
         * that's in the request with the structures in the <code>AlternateDataSourceParameters</code> allow list. If
         * the structures are an exact match, the request is allowed to use the credentials from this existing data
         * source. If the <code>AlternateDataSourceParameters</code> list is null, the <code>Credentials</code>
         * originally used with this <code>DataSourceParameters</code> are automatically allowed.
         * </p>
         * 
         * @param alternateDataSourceParameters
         *        A set of alternate data source parameters that you want to share for the credentials stored with this
         *        data source. The credentials are applied in tandem with the data source parameters when you copy a
         *        data source by using a create or update request. The API operation compares the
         *        <code>DataSourceParameters</code> structure that's in the request with the structures in the
         *        <code>AlternateDataSourceParameters</code> allow list. If the structures are an exact match, the
         *        request is allowed to use the credentials from this existing data source. If the
         *        <code>AlternateDataSourceParameters</code> list is null, the <code>Credentials</code> originally used
         *        with this <code>DataSourceParameters</code> are automatically allowed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alternateDataSourceParameters(DataSourceParameters... alternateDataSourceParameters);

        /**
         * <p>
         * A set of alternate data source parameters that you want to share for the credentials stored with this data
         * source. The credentials are applied in tandem with the data source parameters when you copy a data source by
         * using a create or update request. The API operation compares the <code>DataSourceParameters</code> structure
         * that's in the request with the structures in the <code>AlternateDataSourceParameters</code> allow list. If
         * the structures are an exact match, the request is allowed to use the credentials from this existing data
         * source. If the <code>AlternateDataSourceParameters</code> list is null, the <code>Credentials</code>
         * originally used with this <code>DataSourceParameters</code> are automatically allowed.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DataSourceParameters>.Builder} avoiding the
         * need to create one manually via {@link List<DataSourceParameters>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DataSourceParameters>.Builder#build()} is called immediately
         * and its result is passed to {@link #alternateDataSourceParameters(List<DataSourceParameters>)}.
         * 
         * @param alternateDataSourceParameters
         *        a consumer that will call methods on {@link List<DataSourceParameters>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #alternateDataSourceParameters(List<DataSourceParameters>)
         */
        Builder alternateDataSourceParameters(Consumer<DataSourceParameters.Builder>... alternateDataSourceParameters);

        /**
         * <p>
         * The VPC connection information. You need to use this parameter only when you want QuickSight to use a VPC
         * connection when connecting to your underlying source.
         * </p>
         * 
         * @param vpcConnectionProperties
         *        The VPC connection information. You need to use this parameter only when you want QuickSight to use a
         *        VPC connection when connecting to your underlying source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConnectionProperties(VpcConnectionProperties vpcConnectionProperties);

        /**
         * <p>
         * The VPC connection information. You need to use this parameter only when you want QuickSight to use a VPC
         * connection when connecting to your underlying source.
         * </p>
         * This is a convenience that creates an instance of the {@link VpcConnectionProperties.Builder} avoiding the
         * need to create one manually via {@link VpcConnectionProperties#builder()}.
         *
         * When the {@link Consumer} completes, {@link VpcConnectionProperties.Builder#build()} is called immediately
         * and its result is passed to {@link #vpcConnectionProperties(VpcConnectionProperties)}.
         * 
         * @param vpcConnectionProperties
         *        a consumer that will call methods on {@link VpcConnectionProperties.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConnectionProperties(VpcConnectionProperties)
         */
        default Builder vpcConnectionProperties(Consumer<VpcConnectionProperties.Builder> vpcConnectionProperties) {
            return vpcConnectionProperties(VpcConnectionProperties.builder().applyMutation(vpcConnectionProperties).build());
        }

        /**
         * <p>
         * Secure Socket Layer (SSL) properties that apply when QuickSight connects to your underlying source.
         * </p>
         * 
         * @param sslProperties
         *        Secure Socket Layer (SSL) properties that apply when QuickSight connects to your underlying source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sslProperties(SslProperties sslProperties);

        /**
         * <p>
         * Secure Socket Layer (SSL) properties that apply when QuickSight connects to your underlying source.
         * </p>
         * This is a convenience that creates an instance of the {@link SslProperties.Builder} avoiding the need to
         * create one manually via {@link SslProperties#builder()}.
         *
         * When the {@link Consumer} completes, {@link SslProperties.Builder#build()} is called immediately and its
         * result is passed to {@link #sslProperties(SslProperties)}.
         * 
         * @param sslProperties
         *        a consumer that will call methods on {@link SslProperties.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sslProperties(SslProperties)
         */
        default Builder sslProperties(Consumer<SslProperties.Builder> sslProperties) {
            return sslProperties(SslProperties.builder().applyMutation(sslProperties).build());
        }

        /**
         * <p>
         * Error information from the last update or the creation of the data source.
         * </p>
         * 
         * @param errorInfo
         *        Error information from the last update or the creation of the data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorInfo(DataSourceErrorInfo errorInfo);

        /**
         * <p>
         * Error information from the last update or the creation of the data source.
         * </p>
         * This is a convenience that creates an instance of the {@link DataSourceErrorInfo.Builder} avoiding the need
         * to create one manually via {@link DataSourceErrorInfo#builder()}.
         *
         * When the {@link Consumer} completes, {@link DataSourceErrorInfo.Builder#build()} is called immediately and
         * its result is passed to {@link #errorInfo(DataSourceErrorInfo)}.
         * 
         * @param errorInfo
         *        a consumer that will call methods on {@link DataSourceErrorInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #errorInfo(DataSourceErrorInfo)
         */
        default Builder errorInfo(Consumer<DataSourceErrorInfo.Builder> errorInfo) {
            return errorInfo(DataSourceErrorInfo.builder().applyMutation(errorInfo).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String dataSourceId;

        private String name;

        private String type;

        private String status;

        private Instant createdTime;

        private Instant lastUpdatedTime;

        private DataSourceParameters dataSourceParameters;

        private List<DataSourceParameters> alternateDataSourceParameters = DefaultSdkAutoConstructList.getInstance();

        private VpcConnectionProperties vpcConnectionProperties;

        private SslProperties sslProperties;

        private DataSourceErrorInfo errorInfo;

        private BuilderImpl() {
        }

        private BuilderImpl(DataSource model) {
            arn(model.arn);
            dataSourceId(model.dataSourceId);
            name(model.name);
            type(model.type);
            status(model.status);
            createdTime(model.createdTime);
            lastUpdatedTime(model.lastUpdatedTime);
            dataSourceParameters(model.dataSourceParameters);
            alternateDataSourceParameters(model.alternateDataSourceParameters);
            vpcConnectionProperties(model.vpcConnectionProperties);
            sslProperties(model.sslProperties);
            errorInfo(model.errorInfo);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        @Transient
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getDataSourceId() {
            return dataSourceId;
        }

        public final void setDataSourceId(String dataSourceId) {
            this.dataSourceId = dataSourceId;
        }

        @Override
        @Transient
        public final Builder dataSourceId(String dataSourceId) {
            this.dataSourceId = dataSourceId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        @Transient
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        @Transient
        public final Builder type(DataSourceType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        @Transient
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        @Transient
        public final Builder status(ResourceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        @Transient
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        @Transient
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final DataSourceParameters.Builder getDataSourceParameters() {
            return dataSourceParameters != null ? dataSourceParameters.toBuilder() : null;
        }

        public final void setDataSourceParameters(DataSourceParameters.BuilderImpl dataSourceParameters) {
            this.dataSourceParameters = dataSourceParameters != null ? dataSourceParameters.build() : null;
        }

        @Override
        @Transient
        public final Builder dataSourceParameters(DataSourceParameters dataSourceParameters) {
            this.dataSourceParameters = dataSourceParameters;
            return this;
        }

        public final List<DataSourceParameters.Builder> getAlternateDataSourceParameters() {
            List<DataSourceParameters.Builder> result = DataSourceParametersListCopier
                    .copyToBuilder(this.alternateDataSourceParameters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAlternateDataSourceParameters(
                Collection<DataSourceParameters.BuilderImpl> alternateDataSourceParameters) {
            this.alternateDataSourceParameters = DataSourceParametersListCopier.copyFromBuilder(alternateDataSourceParameters);
        }

        @Override
        @Transient
        public final Builder alternateDataSourceParameters(Collection<DataSourceParameters> alternateDataSourceParameters) {
            this.alternateDataSourceParameters = DataSourceParametersListCopier.copy(alternateDataSourceParameters);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder alternateDataSourceParameters(DataSourceParameters... alternateDataSourceParameters) {
            alternateDataSourceParameters(Arrays.asList(alternateDataSourceParameters));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder alternateDataSourceParameters(
                Consumer<DataSourceParameters.Builder>... alternateDataSourceParameters) {
            alternateDataSourceParameters(Stream.of(alternateDataSourceParameters)
                    .map(c -> DataSourceParameters.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final VpcConnectionProperties.Builder getVpcConnectionProperties() {
            return vpcConnectionProperties != null ? vpcConnectionProperties.toBuilder() : null;
        }

        public final void setVpcConnectionProperties(VpcConnectionProperties.BuilderImpl vpcConnectionProperties) {
            this.vpcConnectionProperties = vpcConnectionProperties != null ? vpcConnectionProperties.build() : null;
        }

        @Override
        @Transient
        public final Builder vpcConnectionProperties(VpcConnectionProperties vpcConnectionProperties) {
            this.vpcConnectionProperties = vpcConnectionProperties;
            return this;
        }

        public final SslProperties.Builder getSslProperties() {
            return sslProperties != null ? sslProperties.toBuilder() : null;
        }

        public final void setSslProperties(SslProperties.BuilderImpl sslProperties) {
            this.sslProperties = sslProperties != null ? sslProperties.build() : null;
        }

        @Override
        @Transient
        public final Builder sslProperties(SslProperties sslProperties) {
            this.sslProperties = sslProperties;
            return this;
        }

        public final DataSourceErrorInfo.Builder getErrorInfo() {
            return errorInfo != null ? errorInfo.toBuilder() : null;
        }

        public final void setErrorInfo(DataSourceErrorInfo.BuilderImpl errorInfo) {
            this.errorInfo = errorInfo != null ? errorInfo.build() : null;
        }

        @Override
        @Transient
        public final Builder errorInfo(DataSourceErrorInfo errorInfo) {
            this.errorInfo = errorInfo;
            return this;
        }

        @Override
        public DataSource build() {
            return new DataSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
