/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.quicksight;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.quicksight.model.AccessDeniedException;
import software.amazon.awssdk.services.quicksight.model.CreateGroupMembershipRequest;
import software.amazon.awssdk.services.quicksight.model.CreateGroupMembershipResponse;
import software.amazon.awssdk.services.quicksight.model.CreateGroupRequest;
import software.amazon.awssdk.services.quicksight.model.CreateGroupResponse;
import software.amazon.awssdk.services.quicksight.model.DeleteGroupMembershipRequest;
import software.amazon.awssdk.services.quicksight.model.DeleteGroupMembershipResponse;
import software.amazon.awssdk.services.quicksight.model.DeleteGroupRequest;
import software.amazon.awssdk.services.quicksight.model.DeleteGroupResponse;
import software.amazon.awssdk.services.quicksight.model.DeleteUserRequest;
import software.amazon.awssdk.services.quicksight.model.DeleteUserResponse;
import software.amazon.awssdk.services.quicksight.model.DescribeGroupRequest;
import software.amazon.awssdk.services.quicksight.model.DescribeGroupResponse;
import software.amazon.awssdk.services.quicksight.model.DescribeUserRequest;
import software.amazon.awssdk.services.quicksight.model.DescribeUserResponse;
import software.amazon.awssdk.services.quicksight.model.DomainNotWhitelistedException;
import software.amazon.awssdk.services.quicksight.model.GetDashboardEmbedUrlRequest;
import software.amazon.awssdk.services.quicksight.model.GetDashboardEmbedUrlResponse;
import software.amazon.awssdk.services.quicksight.model.IdentityTypeNotSupportedException;
import software.amazon.awssdk.services.quicksight.model.InternalFailureException;
import software.amazon.awssdk.services.quicksight.model.InvalidNextTokenException;
import software.amazon.awssdk.services.quicksight.model.InvalidParameterValueException;
import software.amazon.awssdk.services.quicksight.model.LimitExceededException;
import software.amazon.awssdk.services.quicksight.model.ListGroupMembershipsRequest;
import software.amazon.awssdk.services.quicksight.model.ListGroupMembershipsResponse;
import software.amazon.awssdk.services.quicksight.model.ListGroupsRequest;
import software.amazon.awssdk.services.quicksight.model.ListGroupsResponse;
import software.amazon.awssdk.services.quicksight.model.ListUserGroupsRequest;
import software.amazon.awssdk.services.quicksight.model.ListUserGroupsResponse;
import software.amazon.awssdk.services.quicksight.model.ListUsersRequest;
import software.amazon.awssdk.services.quicksight.model.ListUsersResponse;
import software.amazon.awssdk.services.quicksight.model.PreconditionNotMetException;
import software.amazon.awssdk.services.quicksight.model.QuickSightException;
import software.amazon.awssdk.services.quicksight.model.QuickSightUserNotFoundException;
import software.amazon.awssdk.services.quicksight.model.RegisterUserRequest;
import software.amazon.awssdk.services.quicksight.model.RegisterUserResponse;
import software.amazon.awssdk.services.quicksight.model.ResourceExistsException;
import software.amazon.awssdk.services.quicksight.model.ResourceNotFoundException;
import software.amazon.awssdk.services.quicksight.model.ResourceUnavailableException;
import software.amazon.awssdk.services.quicksight.model.SessionLifetimeInMinutesInvalidException;
import software.amazon.awssdk.services.quicksight.model.ThrottlingException;
import software.amazon.awssdk.services.quicksight.model.UnsupportedUserEditionException;
import software.amazon.awssdk.services.quicksight.model.UpdateGroupRequest;
import software.amazon.awssdk.services.quicksight.model.UpdateGroupResponse;
import software.amazon.awssdk.services.quicksight.model.UpdateUserRequest;
import software.amazon.awssdk.services.quicksight.model.UpdateUserResponse;
import software.amazon.awssdk.services.quicksight.transform.CreateGroupMembershipRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.CreateGroupRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.DeleteGroupMembershipRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.DeleteGroupRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.DeleteUserRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.DescribeGroupRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.DescribeUserRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.GetDashboardEmbedUrlRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.ListGroupMembershipsRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.ListGroupsRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.ListUserGroupsRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.ListUsersRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.RegisterUserRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.UpdateGroupRequestMarshaller;
import software.amazon.awssdk.services.quicksight.transform.UpdateUserRequestMarshaller;

/**
 * Internal implementation of {@link QuickSightClient}.
 *
 * @see QuickSightClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultQuickSightClient implements QuickSightClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultQuickSightClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an Amazon QuickSight group.
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;relevant-aws-account-id&gt;</i>:group/default/<i>&lt;group-name&gt;</i> </code>
     * .
     * </p>
     * <p>
     * The response is a group object.
     * </p>
     * <p>
     * <b>CLI Sample:</b>
     * </p>
     * <p>
     * <code>aws quicksight create-group --aws-account-id=111122223333 --namespace=default --group-name="Sales-Management" --description="Sales Management - Forecasting" </code>
     * </p>
     *
     * @param createGroupRequest
     *        The request object for this operation.
     * @return Result of the CreateGroup operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have access to this. The provided credentials couldn't be validated. You might not be
     *         authorized to carry out the request. Ensure that your account is authorized to use the Amazon QuickSight
     *         service, that your policies have the correct permissions, and that you are using the correct access keys.
     * @throws InvalidParameterValueException
     *         One or more parameters don't have a valid value.
     * @throws ResourceExistsException
     *         The resource specified doesn't exist.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ThrottlingException
     *         Access is throttled.
     * @throws PreconditionNotMetException
     *         One or more preconditions aren't met.
     * @throws LimitExceededException
     *         A limit is exceeded.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceUnavailableException
     *         This resource is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QuickSightException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QuickSightClient.CreateGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/CreateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateGroupResponse createGroup(CreateGroupRequest createGroupRequest) throws AccessDeniedException,
            InvalidParameterValueException, ResourceExistsException, ResourceNotFoundException, ThrottlingException,
            PreconditionNotMetException, LimitExceededException, InternalFailureException, ResourceUnavailableException,
            AwsServiceException, SdkClientException, QuickSightException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateGroupRequest, CreateGroupResponse>()
                .withOperationName("CreateGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createGroupRequest)
                .withMarshaller(new CreateGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds an Amazon QuickSight user to an Amazon QuickSight group.
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:group/default/<i>&lt;group-name&gt;</i> </code>.
     * </p>
     * <p>
     * The condition resource is the user name.
     * </p>
     * <p>
     * The condition key is <code>quicksight:UserName</code>.
     * </p>
     * <p>
     * The response is the group member object.
     * </p>
     * <p>
     * <b>CLI Sample:</b>
     * </p>
     * <p>
     * <code>aws quicksight create-group-membership --aws-account-id=111122223333 --namespace=default --group-name=Sales --member-name=Pat </code>
     * </p>
     *
     * @param createGroupMembershipRequest
     * @return Result of the CreateGroupMembership operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have access to this. The provided credentials couldn't be validated. You might not be
     *         authorized to carry out the request. Ensure that your account is authorized to use the Amazon QuickSight
     *         service, that your policies have the correct permissions, and that you are using the correct access keys.
     * @throws InvalidParameterValueException
     *         One or more parameters don't have a valid value.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ThrottlingException
     *         Access is throttled.
     * @throws PreconditionNotMetException
     *         One or more preconditions aren't met.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceUnavailableException
     *         This resource is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QuickSightException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QuickSightClient.CreateGroupMembership
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/CreateGroupMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateGroupMembershipResponse createGroupMembership(CreateGroupMembershipRequest createGroupMembershipRequest)
            throws AccessDeniedException, InvalidParameterValueException, ResourceNotFoundException, ThrottlingException,
            PreconditionNotMetException, InternalFailureException, ResourceUnavailableException, AwsServiceException,
            SdkClientException, QuickSightException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGroupMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateGroupMembershipResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateGroupMembershipRequest, CreateGroupMembershipResponse>()
                .withOperationName("CreateGroupMembership").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createGroupMembershipRequest)
                .withMarshaller(new CreateGroupMembershipRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes a user group from Amazon QuickSight.
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:group/default/<i>&lt;group-name&gt;</i> </code>.
     * </p>
     * <p>
     * <b>CLI Sample:</b>
     * </p>
     * <p>
     * <code>aws quicksight delete-group -\-aws-account-id=111122223333 -\-namespace=default -\-group-name=Sales-Management </code>
     * </p>
     *
     * @param deleteGroupRequest
     * @return Result of the DeleteGroup operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have access to this. The provided credentials couldn't be validated. You might not be
     *         authorized to carry out the request. Ensure that your account is authorized to use the Amazon QuickSight
     *         service, that your policies have the correct permissions, and that you are using the correct access keys.
     * @throws InvalidParameterValueException
     *         One or more parameters don't have a valid value.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ThrottlingException
     *         Access is throttled.
     * @throws PreconditionNotMetException
     *         One or more preconditions aren't met.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceUnavailableException
     *         This resource is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QuickSightException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QuickSightClient.DeleteGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/DeleteGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteGroupResponse deleteGroup(DeleteGroupRequest deleteGroupRequest) throws AccessDeniedException,
            InvalidParameterValueException, ResourceNotFoundException, ThrottlingException, PreconditionNotMetException,
            InternalFailureException, ResourceUnavailableException, AwsServiceException, SdkClientException, QuickSightException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteGroupRequest, DeleteGroupResponse>()
                .withOperationName("DeleteGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteGroupRequest)
                .withMarshaller(new DeleteGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes a user from a group so that the user is no longer a member of the group.
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:group/default/<i>&lt;group-name&gt;</i> </code>.
     * </p>
     * <p>
     * The condition resource is the user name.
     * </p>
     * <p>
     * The condition key is <code>quicksight:UserName</code>.
     * </p>
     * <p>
     * <b>CLI Sample:</b>
     * </p>
     * <p>
     * <code>aws quicksight delete-group-membership --aws-account-id=111122223333 --namespace=default --group-name=Sales-Management --member-name=Charlie </code>
     * </p>
     *
     * @param deleteGroupMembershipRequest
     * @return Result of the DeleteGroupMembership operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have access to this. The provided credentials couldn't be validated. You might not be
     *         authorized to carry out the request. Ensure that your account is authorized to use the Amazon QuickSight
     *         service, that your policies have the correct permissions, and that you are using the correct access keys.
     * @throws InvalidParameterValueException
     *         One or more parameters don't have a valid value.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ThrottlingException
     *         Access is throttled.
     * @throws PreconditionNotMetException
     *         One or more preconditions aren't met.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceUnavailableException
     *         This resource is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QuickSightException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QuickSightClient.DeleteGroupMembership
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/DeleteGroupMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteGroupMembershipResponse deleteGroupMembership(DeleteGroupMembershipRequest deleteGroupMembershipRequest)
            throws AccessDeniedException, InvalidParameterValueException, ResourceNotFoundException, ThrottlingException,
            PreconditionNotMetException, InternalFailureException, ResourceUnavailableException, AwsServiceException,
            SdkClientException, QuickSightException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGroupMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteGroupMembershipResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteGroupMembershipRequest, DeleteGroupMembershipResponse>()
                .withOperationName("DeleteGroupMembership").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteGroupMembershipRequest)
                .withMarshaller(new DeleteGroupMembershipRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the Amazon QuickSight user that is associated with the identity of the AWS Identity and Access Management
     * (IAM) user or role that's making the call. The IAM user isn't deleted as a result of this call.
     * </p>
     * <p>
     * The permission resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:user/default/<i>&lt;user-name&gt; </i> </code>.
     * </p>
     * <p>
     * <b>CLI Sample:</b>
     * </p>
     * <p>
     * <code>aws quicksight delete-user --aws-account-id=111122223333 --namespace=default --user-name=Pat </code>
     * </p>
     *
     * @param deleteUserRequest
     * @return Result of the DeleteUser operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have access to this. The provided credentials couldn't be validated. You might not be
     *         authorized to carry out the request. Ensure that your account is authorized to use the Amazon QuickSight
     *         service, that your policies have the correct permissions, and that you are using the correct access keys.
     * @throws InvalidParameterValueException
     *         One or more parameters don't have a valid value.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ThrottlingException
     *         Access is throttled.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceUnavailableException
     *         This resource is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QuickSightException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QuickSightClient.DeleteUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteUserResponse deleteUser(DeleteUserRequest deleteUserRequest) throws AccessDeniedException,
            InvalidParameterValueException, ResourceNotFoundException, ThrottlingException, InternalFailureException,
            ResourceUnavailableException, AwsServiceException, SdkClientException, QuickSightException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteUserRequest, DeleteUserResponse>()
                .withOperationName("DeleteUser").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteUserRequest)
                .withMarshaller(new DeleteUserRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns an Amazon QuickSight group's description and Amazon Resource Name (ARN).
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;relevant-aws-account-id&gt;</i>:group/default/<i>&lt;group-name&gt;</i> </code>
     * .
     * </p>
     * <p>
     * The response is the group object.
     * </p>
     * <p>
     * <b>CLI Sample:</b>
     * </p>
     * <p>
     * <code>aws quicksight describe-group -\-aws-account-id=11112222333 -\-namespace=default -\-group-name=Sales </code>
     * </p>
     *
     * @param describeGroupRequest
     * @return Result of the DescribeGroup operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have access to this. The provided credentials couldn't be validated. You might not be
     *         authorized to carry out the request. Ensure that your account is authorized to use the Amazon QuickSight
     *         service, that your policies have the correct permissions, and that you are using the correct access keys.
     * @throws InvalidParameterValueException
     *         One or more parameters don't have a valid value.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ThrottlingException
     *         Access is throttled.
     * @throws PreconditionNotMetException
     *         One or more preconditions aren't met.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceUnavailableException
     *         This resource is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QuickSightException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QuickSightClient.DescribeGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/DescribeGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeGroupResponse describeGroup(DescribeGroupRequest describeGroupRequest) throws AccessDeniedException,
            InvalidParameterValueException, ResourceNotFoundException, ThrottlingException, PreconditionNotMetException,
            InternalFailureException, ResourceUnavailableException, AwsServiceException, SdkClientException, QuickSightException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeGroupRequest, DescribeGroupResponse>()
                .withOperationName("DescribeGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeGroupRequest)
                .withMarshaller(new DescribeGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns information about a user, given the user name.
     * </p>
     * <p>
     * The permission resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:user/default/<i>&lt;user-name&gt;</i> </code>.
     * </p>
     * <p>
     * The response is a user object that contains the user's Amazon Resource Name (ARN), AWS Identity and Access
     * Management (IAM) role, and email address.
     * </p>
     * <p>
     * <b>CLI Sample:</b>
     * </p>
     * <p>
     * <code>aws quicksight describe-user --aws-account-id=111122223333 --namespace=default --user-name=Pat </code>
     * </p>
     *
     * @param describeUserRequest
     * @return Result of the DescribeUser operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have access to this. The provided credentials couldn't be validated. You might not be
     *         authorized to carry out the request. Ensure that your account is authorized to use the Amazon QuickSight
     *         service, that your policies have the correct permissions, and that you are using the correct access keys.
     * @throws InvalidParameterValueException
     *         One or more parameters don't have a valid value.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ThrottlingException
     *         Access is throttled.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceUnavailableException
     *         This resource is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QuickSightException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QuickSightClient.DescribeUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/DescribeUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeUserResponse describeUser(DescribeUserRequest describeUserRequest) throws AccessDeniedException,
            InvalidParameterValueException, ResourceNotFoundException, ThrottlingException, InternalFailureException,
            ResourceUnavailableException, AwsServiceException, SdkClientException, QuickSightException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeUserRequest, DescribeUserResponse>()
                .withOperationName("DescribeUser").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeUserRequest)
                .withMarshaller(new DescribeUserRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Generates a server-side embeddable URL and authorization code. Before this can work properly, first you need to
     * configure the dashboards and user permissions. For more information, see <a
     * href="https://docs.aws.amazon.com/en_us/quicksight/latest/user/embedding.html"> Embedding Amazon QuickSight
     * Dashboards</a>.
     * </p>
     * <p>
     * Currently, you can use <code>GetDashboardEmbedURL</code> only from the server, not from the user’s browser.
     * </p>
     * <p>
     * <b>CLI Sample:</b>
     * </p>
     * <p>
     * Assume the role with permissions enabled for actions: <code>quickSight:RegisterUser</code> and
     * <code>quicksight:GetDashboardEmbedURL</code>. You can use assume-role, assume-role-with-web-identity, or
     * assume-role-with-saml.
     * </p>
     * <p>
     * <code>aws sts assume-role --role-arn "arn:aws:iam::111122223333:role/embedding_quicksight_dashboard_role" --role-session-name embeddingsession</code>
     * </p>
     * <p>
     * If the user does not exist in QuickSight, register the user:
     * </p>
     * <p>
     * <code>aws quicksight register-user --aws-account-id 111122223333 --namespace default --identity-type IAM --iam-arn "arn:aws:iam::111122223333:role/embedding_quicksight_dashboard_role" --user-role READER --session-name "embeddingsession" --email user123@example.com --region us-east-1</code>
     * </p>
     * <p>
     * Get the URL for the embedded dashboard
     * </p>
     * <p>
     * <code>aws quicksight get-dashboard-embed-url --aws-account-id 111122223333 --dashboard-id 1a1ac2b2-3fc3-4b44-5e5d-c6db6778df89 --identity-type IAM</code>
     * </p>
     *
     * @param getDashboardEmbedUrlRequest
     * @return Result of the GetDashboardEmbedUrl operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have access to this. The provided credentials couldn't be validated. You might not be
     *         authorized to carry out the request. Ensure that your account is authorized to use the Amazon QuickSight
     *         service, that your policies have the correct permissions, and that you are using the correct access keys.
     * @throws InvalidParameterValueException
     *         One or more parameters don't have a valid value.
     * @throws ResourceExistsException
     *         The resource specified doesn't exist.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ThrottlingException
     *         Access is throttled.
     * @throws PreconditionNotMetException
     *         One or more preconditions aren't met.
     * @throws DomainNotWhitelistedException
     *         The domain specified is not on the allowlist. All domains for embedded dashboards must be added to the
     *         approved list by an Amazon QuickSight admin.
     * @throws QuickSightUserNotFoundException
     *         The user is not found. This error can happen in any operation that requires finding a user based on a
     *         provided user name, such as <code>DeleteUser</code>, <code>DescribeUser</code>, and so on.
     * @throws IdentityTypeNotSupportedException
     *         The identity type specified is not supported. Supported identity types include IAM and QUICKSIGHT.
     * @throws SessionLifetimeInMinutesInvalidException
     *         The number of minutes specified for the lifetime of a session is not valid. The session lifetime must be
     *         from 15 to 600 minutes.
     * @throws UnsupportedUserEditionException
     *         This error indicates that you are calling an operation on an Amazon QuickSight subscription where the
     *         edition doesn't include support for that operation. Amazon QuickSight currently has Standard Edition and
     *         Enterprise Edition. Not every operation and capability is available in every edition.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceUnavailableException
     *         This resource is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QuickSightException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QuickSightClient.GetDashboardEmbedUrl
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/GetDashboardEmbedUrl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDashboardEmbedUrlResponse getDashboardEmbedUrl(GetDashboardEmbedUrlRequest getDashboardEmbedUrlRequest)
            throws AccessDeniedException, InvalidParameterValueException, ResourceExistsException, ResourceNotFoundException,
            ThrottlingException, PreconditionNotMetException, DomainNotWhitelistedException, QuickSightUserNotFoundException,
            IdentityTypeNotSupportedException, SessionLifetimeInMinutesInvalidException, UnsupportedUserEditionException,
            InternalFailureException, ResourceUnavailableException, AwsServiceException, SdkClientException, QuickSightException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDashboardEmbedUrlResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDashboardEmbedUrlResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetDashboardEmbedUrlRequest, GetDashboardEmbedUrlResponse>()
                .withOperationName("GetDashboardEmbedUrl").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getDashboardEmbedUrlRequest)
                .withMarshaller(new GetDashboardEmbedUrlRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists member users in a group.
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:group/default/<i>&lt;group-name&gt;</i> </code>.
     * </p>
     * <p>
     * The response is a list of group member objects.
     * </p>
     * <p>
     * <b>CLI Sample:</b>
     * </p>
     * <p>
     * <code>aws quicksight list-group-memberships -\-aws-account-id=111122223333 -\-namespace=default </code>
     * </p>
     *
     * @param listGroupMembershipsRequest
     * @return Result of the ListGroupMemberships operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have access to this. The provided credentials couldn't be validated. You might not be
     *         authorized to carry out the request. Ensure that your account is authorized to use the Amazon QuickSight
     *         service, that your policies have the correct permissions, and that you are using the correct access keys.
     * @throws InvalidParameterValueException
     *         One or more parameters don't have a valid value.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ThrottlingException
     *         Access is throttled.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value isn't valid.
     * @throws PreconditionNotMetException
     *         One or more preconditions aren't met.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceUnavailableException
     *         This resource is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QuickSightException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QuickSightClient.ListGroupMemberships
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/ListGroupMemberships"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListGroupMembershipsResponse listGroupMemberships(ListGroupMembershipsRequest listGroupMembershipsRequest)
            throws AccessDeniedException, InvalidParameterValueException, ResourceNotFoundException, ThrottlingException,
            InvalidNextTokenException, PreconditionNotMetException, InternalFailureException, ResourceUnavailableException,
            AwsServiceException, SdkClientException, QuickSightException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGroupMembershipsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListGroupMembershipsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListGroupMembershipsRequest, ListGroupMembershipsResponse>()
                .withOperationName("ListGroupMemberships").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listGroupMembershipsRequest)
                .withMarshaller(new ListGroupMembershipsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists all user groups in Amazon QuickSight.
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:group/default/*</code>.
     * </p>
     * <p>
     * The response is a list of group objects.
     * </p>
     * <p>
     * <b>CLI Sample:</b>
     * </p>
     * <p>
     * <code>aws quicksight list-groups -\-aws-account-id=111122223333 -\-namespace=default </code>
     * </p>
     *
     * @param listGroupsRequest
     * @return Result of the ListGroups operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have access to this. The provided credentials couldn't be validated. You might not be
     *         authorized to carry out the request. Ensure that your account is authorized to use the Amazon QuickSight
     *         service, that your policies have the correct permissions, and that you are using the correct access keys.
     * @throws InvalidParameterValueException
     *         One or more parameters don't have a valid value.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ThrottlingException
     *         Access is throttled.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value isn't valid.
     * @throws PreconditionNotMetException
     *         One or more preconditions aren't met.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceUnavailableException
     *         This resource is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QuickSightException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QuickSightClient.ListGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/ListGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGroupsResponse listGroups(ListGroupsRequest listGroupsRequest) throws AccessDeniedException,
            InvalidParameterValueException, ResourceNotFoundException, ThrottlingException, InvalidNextTokenException,
            PreconditionNotMetException, InternalFailureException, ResourceUnavailableException, AwsServiceException,
            SdkClientException, QuickSightException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListGroupsRequest, ListGroupsResponse>()
                .withOperationName("ListGroups").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listGroupsRequest)
                .withMarshaller(new ListGroupsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the Amazon QuickSight groups that an Amazon QuickSight user is a member of.
     * </p>
     * <p>
     * The permission resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:user/default/<i>&lt;user-name&gt;</i> </code>.
     * </p>
     * <p>
     * The response is a one or more group objects.
     * </p>
     * <p>
     * <b>CLI Sample:</b>
     * </p>
     * <p>
     * <code>aws quicksight list-user-groups -\-user-name=Pat -\-aws-account-id=111122223333 -\-namespace=default -\-region=us-east-1 </code>
     * </p>
     *
     * @param listUserGroupsRequest
     * @return Result of the ListUserGroups operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have access to this. The provided credentials couldn't be validated. You might not be
     *         authorized to carry out the request. Ensure that your account is authorized to use the Amazon QuickSight
     *         service, that your policies have the correct permissions, and that you are using the correct access keys.
     * @throws InvalidParameterValueException
     *         One or more parameters don't have a valid value.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ThrottlingException
     *         Access is throttled.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceUnavailableException
     *         This resource is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QuickSightException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QuickSightClient.ListUserGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/ListUserGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListUserGroupsResponse listUserGroups(ListUserGroupsRequest listUserGroupsRequest) throws AccessDeniedException,
            InvalidParameterValueException, ResourceNotFoundException, ThrottlingException, InternalFailureException,
            ResourceUnavailableException, AwsServiceException, SdkClientException, QuickSightException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListUserGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListUserGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListUserGroupsRequest, ListUserGroupsResponse>()
                .withOperationName("ListUserGroups").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listUserGroupsRequest)
                .withMarshaller(new ListUserGroupsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of all of the Amazon QuickSight users belonging to this account.
     * </p>
     * <p>
     * The permission resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:user/default/<i>*</i> </code>.
     * </p>
     * <p>
     * The response is a list of user objects, containing each user's Amazon Resource Name (ARN), AWS Identity and
     * Access Management (IAM) role, and email address.
     * </p>
     * <p>
     * <b>CLI Sample:</b>
     * </p>
     * <p>
     * <code>aws quicksight list-users --aws-account-id=111122223333 --namespace=default </code>
     * </p>
     *
     * @param listUsersRequest
     * @return Result of the ListUsers operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have access to this. The provided credentials couldn't be validated. You might not be
     *         authorized to carry out the request. Ensure that your account is authorized to use the Amazon QuickSight
     *         service, that your policies have the correct permissions, and that you are using the correct access keys.
     * @throws InvalidParameterValueException
     *         One or more parameters don't have a valid value.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ThrottlingException
     *         Access is throttled.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value isn't valid.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceUnavailableException
     *         This resource is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QuickSightException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QuickSightClient.ListUsers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListUsersResponse listUsers(ListUsersRequest listUsersRequest) throws AccessDeniedException,
            InvalidParameterValueException, ResourceNotFoundException, ThrottlingException, InvalidNextTokenException,
            InternalFailureException, ResourceUnavailableException, AwsServiceException, SdkClientException, QuickSightException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListUsersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListUsersRequest, ListUsersResponse>()
                .withOperationName("ListUsers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listUsersRequest)
                .withMarshaller(new ListUsersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates an Amazon QuickSight user, whose identity is associated with the AWS Identity and Access Management (IAM)
     * identity or role specified in the request.
     * </p>
     * <p>
     * The permission resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:user/default/<i>&lt;user-name&gt;</i> </code>.
     * </p>
     * <p>
     * The condition resource is the Amazon Resource Name (ARN) for the IAM user or role, and the session name.
     * </p>
     * <p>
     * The condition keys are <code>quicksight:IamArn</code> and <code>quicksight:SessionName</code>.
     * </p>
     * <p>
     * <b>CLI Sample:</b>
     * </p>
     * <p>
     * <code>aws quicksight register-user -\-aws-account-id=111122223333 -\-namespace=default -\-email=pat@example.com -\-identity-type=IAM -\-user-role=AUTHOR -\-iam-arn=arn:aws:iam::111122223333:user/Pat </code>
     * </p>
     *
     * @param registerUserRequest
     * @return Result of the RegisterUser operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have access to this. The provided credentials couldn't be validated. You might not be
     *         authorized to carry out the request. Ensure that your account is authorized to use the Amazon QuickSight
     *         service, that your policies have the correct permissions, and that you are using the correct access keys.
     * @throws InvalidParameterValueException
     *         One or more parameters don't have a valid value.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ThrottlingException
     *         Access is throttled.
     * @throws LimitExceededException
     *         A limit is exceeded.
     * @throws ResourceExistsException
     *         The resource specified doesn't exist.
     * @throws PreconditionNotMetException
     *         One or more preconditions aren't met.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceUnavailableException
     *         This resource is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QuickSightException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QuickSightClient.RegisterUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/RegisterUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RegisterUserResponse registerUser(RegisterUserRequest registerUserRequest) throws AccessDeniedException,
            InvalidParameterValueException, ResourceNotFoundException, ThrottlingException, LimitExceededException,
            ResourceExistsException, PreconditionNotMetException, InternalFailureException, ResourceUnavailableException,
            AwsServiceException, SdkClientException, QuickSightException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RegisterUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RegisterUserRequest, RegisterUserResponse>()
                .withOperationName("RegisterUser").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(registerUserRequest)
                .withMarshaller(new RegisterUserRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Changes a group description.
     * </p>
     * <p>
     * The permissions resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:group/default/<i>&lt;group-name&gt;</i> </code>.
     * </p>
     * <p>
     * The response is a group object.
     * </p>
     * <p>
     * <b>CLI Sample:</b>
     * </p>
     * <p>
     * <code>aws quicksight update-group --aws-account-id=111122223333 --namespace=default --group-name=Sales --description="Sales BI Dashboards" </code>
     * </p>
     *
     * @param updateGroupRequest
     * @return Result of the UpdateGroup operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have access to this. The provided credentials couldn't be validated. You might not be
     *         authorized to carry out the request. Ensure that your account is authorized to use the Amazon QuickSight
     *         service, that your policies have the correct permissions, and that you are using the correct access keys.
     * @throws InvalidParameterValueException
     *         One or more parameters don't have a valid value.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ThrottlingException
     *         Access is throttled.
     * @throws PreconditionNotMetException
     *         One or more preconditions aren't met.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceUnavailableException
     *         This resource is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QuickSightException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QuickSightClient.UpdateGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/UpdateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateGroupResponse updateGroup(UpdateGroupRequest updateGroupRequest) throws AccessDeniedException,
            InvalidParameterValueException, ResourceNotFoundException, ThrottlingException, PreconditionNotMetException,
            InternalFailureException, ResourceUnavailableException, AwsServiceException, SdkClientException, QuickSightException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateGroupRequest, UpdateGroupResponse>()
                .withOperationName("UpdateGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateGroupRequest)
                .withMarshaller(new UpdateGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates an Amazon QuickSight user.
     * </p>
     * <p>
     * The permission resource is
     * <code>arn:aws:quicksight:us-east-1:<i>&lt;aws-account-id&gt;</i>:user/default/<i>&lt;user-name&gt;</i> </code>.
     * </p>
     * <p>
     * The response is a user object that contains the user's Amazon QuickSight user name, email address, active or
     * inactive status in Amazon QuickSight, Amazon QuickSight role, and Amazon Resource Name (ARN).
     * </p>
     * <p>
     * <b>CLI Sample:</b>
     * </p>
     * <p>
     * <code>aws quicksight update-user --user-name=Pat --role=ADMIN --email=new_address@amazon.com --aws-account-id=111122223333 --namespace=default --region=us-east-1 </code>
     * </p>
     *
     * @param updateUserRequest
     * @return Result of the UpdateUser operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have access to this. The provided credentials couldn't be validated. You might not be
     *         authorized to carry out the request. Ensure that your account is authorized to use the Amazon QuickSight
     *         service, that your policies have the correct permissions, and that you are using the correct access keys.
     * @throws InvalidParameterValueException
     *         One or more parameters don't have a valid value.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ThrottlingException
     *         Access is throttled.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceUnavailableException
     *         This resource is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws QuickSightException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample QuickSightClient.UpdateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateUserResponse updateUser(UpdateUserRequest updateUserRequest) throws AccessDeniedException,
            InvalidParameterValueException, ResourceNotFoundException, ThrottlingException, InternalFailureException,
            ResourceUnavailableException, AwsServiceException, SdkClientException, QuickSightException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateUserRequest, UpdateUserResponse>()
                .withOperationName("UpdateUser").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateUserRequest)
                .withMarshaller(new UpdateUserRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(QuickSightException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DomainNotWhitelistedException")
                                .exceptionBuilderSupplier(DomainNotWhitelistedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("QuickSightUserNotFoundException")
                                .exceptionBuilderSupplier(QuickSightUserNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdentityTypeNotSupportedException")
                                .exceptionBuilderSupplier(IdentityTypeNotSupportedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceUnavailableException")
                                .exceptionBuilderSupplier(ResourceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceExistsException")
                                .exceptionBuilderSupplier(ResourceExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SessionLifetimeInMinutesInvalidException")
                                .exceptionBuilderSupplier(SessionLifetimeInMinutesInvalidException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedUserEditionException")
                                .exceptionBuilderSupplier(UnsupportedUserEditionException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PreconditionNotMetException")
                                .exceptionBuilderSupplier(PreconditionNotMetException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
