/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.quicksight.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure that grants Amazon QuickSight access to your cluster and make a call to the
 * <code>redshift:GetClusterCredentials</code> API. For more information on the
 * <code>redshift:GetClusterCredentials</code> API, see <a
 * href="https://docs.aws.amazon.com/redshift/latest/APIReference/API_GetClusterCredentials.html">
 * <code>GetClusterCredentials</code> </a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RedshiftIAMParameters implements SdkPojo, Serializable,
        ToCopyableBuilder<RedshiftIAMParameters.Builder, RedshiftIAMParameters> {
    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoleArn").getter(getter(RedshiftIAMParameters::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<String> DATABASE_USER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DatabaseUser").getter(getter(RedshiftIAMParameters::databaseUser)).setter(setter(Builder::databaseUser))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseUser").build()).build();

    private static final SdkField<List<String>> DATABASE_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DatabaseGroups")
            .getter(getter(RedshiftIAMParameters::databaseGroups))
            .setter(setter(Builder::databaseGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> AUTO_CREATE_DATABASE_USER_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AutoCreateDatabaseUser").getter(getter(RedshiftIAMParameters::autoCreateDatabaseUser))
            .setter(setter(Builder::autoCreateDatabaseUser))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoCreateDatabaseUser").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ROLE_ARN_FIELD,
            DATABASE_USER_FIELD, DATABASE_GROUPS_FIELD, AUTO_CREATE_DATABASE_USER_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String roleArn;

    private final String databaseUser;

    private final List<String> databaseGroups;

    private final Boolean autoCreateDatabaseUser;

    private RedshiftIAMParameters(BuilderImpl builder) {
        this.roleArn = builder.roleArn;
        this.databaseUser = builder.databaseUser;
        this.databaseGroups = builder.databaseGroups;
        this.autoCreateDatabaseUser = builder.autoCreateDatabaseUser;
    }

    /**
     * <p>
     * Use the <code>RoleArn</code> structure to allow Amazon QuickSight to call
     * <code>redshift:GetClusterCredentials</code> on your cluster. The calling principal must have
     * <code>iam:PassRole</code> access to pass the role to Amazon QuickSight. The role's trust policy must allow the
     * Amazon QuickSight service principal to assume the role.
     * </p>
     * 
     * @return Use the <code>RoleArn</code> structure to allow Amazon QuickSight to call
     *         <code>redshift:GetClusterCredentials</code> on your cluster. The calling principal must have
     *         <code>iam:PassRole</code> access to pass the role to Amazon QuickSight. The role's trust policy must
     *         allow the Amazon QuickSight service principal to assume the role.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The user whose permissions and group memberships will be used by Amazon QuickSight to access the cluster. If this
     * user already exists in your database, Amazon QuickSight is granted the same permissions that the user has. If the
     * user doesn't exist, set the value of <code>AutoCreateDatabaseUser</code> to <code>True</code> to create a new
     * user with PUBLIC permissions.
     * </p>
     * 
     * @return The user whose permissions and group memberships will be used by Amazon QuickSight to access the cluster.
     *         If this user already exists in your database, Amazon QuickSight is granted the same permissions that the
     *         user has. If the user doesn't exist, set the value of <code>AutoCreateDatabaseUser</code> to
     *         <code>True</code> to create a new user with PUBLIC permissions.
     */
    public final String databaseUser() {
        return databaseUser;
    }

    /**
     * For responses, this returns true if the service returned a value for the DatabaseGroups property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDatabaseGroups() {
        return databaseGroups != null && !(databaseGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of groups whose permissions will be granted to Amazon QuickSight to access the cluster. These permissions
     * are combined with the permissions granted to Amazon QuickSight by the <code>DatabaseUser</code>. If you choose to
     * include this parameter, the <code>RoleArn</code> must grant access to <code>redshift:JoinGroup</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDatabaseGroups} method.
     * </p>
     * 
     * @return A list of groups whose permissions will be granted to Amazon QuickSight to access the cluster. These
     *         permissions are combined with the permissions granted to Amazon QuickSight by the
     *         <code>DatabaseUser</code>. If you choose to include this parameter, the <code>RoleArn</code> must grant
     *         access to <code>redshift:JoinGroup</code>.
     */
    public final List<String> databaseGroups() {
        return databaseGroups;
    }

    /**
     * <p>
     * Automatically creates a database user. If your database doesn't have a <code>DatabaseUser</code>, set this
     * parameter to <code>True</code>. If there is no <code>DatabaseUser</code>, Amazon QuickSight can't connect to your
     * cluster. The <code>RoleArn</code> that you use for this operation must grant access to
     * <code>redshift:CreateClusterUser</code> to successfully create the user.
     * </p>
     * 
     * @return Automatically creates a database user. If your database doesn't have a <code>DatabaseUser</code>, set
     *         this parameter to <code>True</code>. If there is no <code>DatabaseUser</code>, Amazon QuickSight can't
     *         connect to your cluster. The <code>RoleArn</code> that you use for this operation must grant access to
     *         <code>redshift:CreateClusterUser</code> to successfully create the user.
     */
    public final Boolean autoCreateDatabaseUser() {
        return autoCreateDatabaseUser;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(databaseUser());
        hashCode = 31 * hashCode + Objects.hashCode(hasDatabaseGroups() ? databaseGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(autoCreateDatabaseUser());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RedshiftIAMParameters)) {
            return false;
        }
        RedshiftIAMParameters other = (RedshiftIAMParameters) obj;
        return Objects.equals(roleArn(), other.roleArn()) && Objects.equals(databaseUser(), other.databaseUser())
                && hasDatabaseGroups() == other.hasDatabaseGroups() && Objects.equals(databaseGroups(), other.databaseGroups())
                && Objects.equals(autoCreateDatabaseUser(), other.autoCreateDatabaseUser());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RedshiftIAMParameters").add("RoleArn", roleArn()).add("DatabaseUser", databaseUser())
                .add("DatabaseGroups", hasDatabaseGroups() ? databaseGroups() : null)
                .add("AutoCreateDatabaseUser", autoCreateDatabaseUser()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "DatabaseUser":
            return Optional.ofNullable(clazz.cast(databaseUser()));
        case "DatabaseGroups":
            return Optional.ofNullable(clazz.cast(databaseGroups()));
        case "AutoCreateDatabaseUser":
            return Optional.ofNullable(clazz.cast(autoCreateDatabaseUser()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RoleArn", ROLE_ARN_FIELD);
        map.put("DatabaseUser", DATABASE_USER_FIELD);
        map.put("DatabaseGroups", DATABASE_GROUPS_FIELD);
        map.put("AutoCreateDatabaseUser", AUTO_CREATE_DATABASE_USER_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RedshiftIAMParameters, T> g) {
        return obj -> g.apply((RedshiftIAMParameters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RedshiftIAMParameters> {
        /**
         * <p>
         * Use the <code>RoleArn</code> structure to allow Amazon QuickSight to call
         * <code>redshift:GetClusterCredentials</code> on your cluster. The calling principal must have
         * <code>iam:PassRole</code> access to pass the role to Amazon QuickSight. The role's trust policy must allow
         * the Amazon QuickSight service principal to assume the role.
         * </p>
         * 
         * @param roleArn
         *        Use the <code>RoleArn</code> structure to allow Amazon QuickSight to call
         *        <code>redshift:GetClusterCredentials</code> on your cluster. The calling principal must have
         *        <code>iam:PassRole</code> access to pass the role to Amazon QuickSight. The role's trust policy must
         *        allow the Amazon QuickSight service principal to assume the role.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The user whose permissions and group memberships will be used by Amazon QuickSight to access the cluster. If
         * this user already exists in your database, Amazon QuickSight is granted the same permissions that the user
         * has. If the user doesn't exist, set the value of <code>AutoCreateDatabaseUser</code> to <code>True</code> to
         * create a new user with PUBLIC permissions.
         * </p>
         * 
         * @param databaseUser
         *        The user whose permissions and group memberships will be used by Amazon QuickSight to access the
         *        cluster. If this user already exists in your database, Amazon QuickSight is granted the same
         *        permissions that the user has. If the user doesn't exist, set the value of
         *        <code>AutoCreateDatabaseUser</code> to <code>True</code> to create a new user with PUBLIC permissions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseUser(String databaseUser);

        /**
         * <p>
         * A list of groups whose permissions will be granted to Amazon QuickSight to access the cluster. These
         * permissions are combined with the permissions granted to Amazon QuickSight by the <code>DatabaseUser</code>.
         * If you choose to include this parameter, the <code>RoleArn</code> must grant access to
         * <code>redshift:JoinGroup</code>.
         * </p>
         * 
         * @param databaseGroups
         *        A list of groups whose permissions will be granted to Amazon QuickSight to access the cluster. These
         *        permissions are combined with the permissions granted to Amazon QuickSight by the
         *        <code>DatabaseUser</code>. If you choose to include this parameter, the <code>RoleArn</code> must
         *        grant access to <code>redshift:JoinGroup</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseGroups(Collection<String> databaseGroups);

        /**
         * <p>
         * A list of groups whose permissions will be granted to Amazon QuickSight to access the cluster. These
         * permissions are combined with the permissions granted to Amazon QuickSight by the <code>DatabaseUser</code>.
         * If you choose to include this parameter, the <code>RoleArn</code> must grant access to
         * <code>redshift:JoinGroup</code>.
         * </p>
         * 
         * @param databaseGroups
         *        A list of groups whose permissions will be granted to Amazon QuickSight to access the cluster. These
         *        permissions are combined with the permissions granted to Amazon QuickSight by the
         *        <code>DatabaseUser</code>. If you choose to include this parameter, the <code>RoleArn</code> must
         *        grant access to <code>redshift:JoinGroup</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseGroups(String... databaseGroups);

        /**
         * <p>
         * Automatically creates a database user. If your database doesn't have a <code>DatabaseUser</code>, set this
         * parameter to <code>True</code>. If there is no <code>DatabaseUser</code>, Amazon QuickSight can't connect to
         * your cluster. The <code>RoleArn</code> that you use for this operation must grant access to
         * <code>redshift:CreateClusterUser</code> to successfully create the user.
         * </p>
         * 
         * @param autoCreateDatabaseUser
         *        Automatically creates a database user. If your database doesn't have a <code>DatabaseUser</code>, set
         *        this parameter to <code>True</code>. If there is no <code>DatabaseUser</code>, Amazon QuickSight can't
         *        connect to your cluster. The <code>RoleArn</code> that you use for this operation must grant access to
         *        <code>redshift:CreateClusterUser</code> to successfully create the user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoCreateDatabaseUser(Boolean autoCreateDatabaseUser);
    }

    static final class BuilderImpl implements Builder {
        private String roleArn;

        private String databaseUser;

        private List<String> databaseGroups = DefaultSdkAutoConstructList.getInstance();

        private Boolean autoCreateDatabaseUser;

        private BuilderImpl() {
        }

        private BuilderImpl(RedshiftIAMParameters model) {
            roleArn(model.roleArn);
            databaseUser(model.databaseUser);
            databaseGroups(model.databaseGroups);
            autoCreateDatabaseUser(model.autoCreateDatabaseUser);
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final String getDatabaseUser() {
            return databaseUser;
        }

        public final void setDatabaseUser(String databaseUser) {
            this.databaseUser = databaseUser;
        }

        @Override
        public final Builder databaseUser(String databaseUser) {
            this.databaseUser = databaseUser;
            return this;
        }

        public final Collection<String> getDatabaseGroups() {
            if (databaseGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return databaseGroups;
        }

        public final void setDatabaseGroups(Collection<String> databaseGroups) {
            this.databaseGroups = DatabaseGroupListCopier.copy(databaseGroups);
        }

        @Override
        public final Builder databaseGroups(Collection<String> databaseGroups) {
            this.databaseGroups = DatabaseGroupListCopier.copy(databaseGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder databaseGroups(String... databaseGroups) {
            databaseGroups(Arrays.asList(databaseGroups));
            return this;
        }

        public final Boolean getAutoCreateDatabaseUser() {
            return autoCreateDatabaseUser;
        }

        public final void setAutoCreateDatabaseUser(Boolean autoCreateDatabaseUser) {
            this.autoCreateDatabaseUser = autoCreateDatabaseUser;
        }

        @Override
        public final Builder autoCreateDatabaseUser(Boolean autoCreateDatabaseUser) {
            this.autoCreateDatabaseUser = autoCreateDatabaseUser;
            return this;
        }

        @Override
        public RedshiftIAMParameters build() {
            return new RedshiftIAMParameters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
