/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.quicksight.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains detailed information about an action connector, including its configuration, status, and enabled actions.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ActionConnector implements SdkPojo, Serializable, ToCopyableBuilder<ActionConnector.Builder, ActionConnector> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(ActionConnector::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> ACTION_CONNECTOR_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ActionConnectorId").getter(getter(ActionConnector::actionConnectorId))
            .setter(setter(Builder::actionConnectorId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActionConnectorId").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(ActionConnector::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ActionConnector::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime").getter(getter(ActionConnector::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedTime").getter(getter(ActionConnector::lastUpdatedTime))
            .setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTime").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ActionConnector::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<ActionConnectorError> ERROR_FIELD = SdkField
            .<ActionConnectorError> builder(MarshallingType.SDK_POJO).memberName("Error").getter(getter(ActionConnector::error))
            .setter(setter(Builder::error)).constructor(ActionConnectorError::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Error").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(ActionConnector::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<ReadAuthConfig> AUTHENTICATION_CONFIG_FIELD = SdkField
            .<ReadAuthConfig> builder(MarshallingType.SDK_POJO).memberName("AuthenticationConfig")
            .getter(getter(ActionConnector::authenticationConfig)).setter(setter(Builder::authenticationConfig))
            .constructor(ReadAuthConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthenticationConfig").build())
            .build();

    private static final SdkField<List<String>> ENABLED_ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EnabledActions")
            .getter(getter(ActionConnector::enabledActions))
            .setter(setter(Builder::enabledActions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnabledActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> VPC_CONNECTION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VpcConnectionArn").getter(getter(ActionConnector::vpcConnectionArn))
            .setter(setter(Builder::vpcConnectionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConnectionArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD,
            ACTION_CONNECTOR_ID_FIELD, TYPE_FIELD, NAME_FIELD, CREATED_TIME_FIELD, LAST_UPDATED_TIME_FIELD, STATUS_FIELD,
            ERROR_FIELD, DESCRIPTION_FIELD, AUTHENTICATION_CONFIG_FIELD, ENABLED_ACTIONS_FIELD, VPC_CONNECTION_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String actionConnectorId;

    private final String type;

    private final String name;

    private final Instant createdTime;

    private final Instant lastUpdatedTime;

    private final String status;

    private final ActionConnectorError error;

    private final String description;

    private final ReadAuthConfig authenticationConfig;

    private final List<String> enabledActions;

    private final String vpcConnectionArn;

    private ActionConnector(BuilderImpl builder) {
        this.arn = builder.arn;
        this.actionConnectorId = builder.actionConnectorId;
        this.type = builder.type;
        this.name = builder.name;
        this.createdTime = builder.createdTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
        this.status = builder.status;
        this.error = builder.error;
        this.description = builder.description;
        this.authenticationConfig = builder.authenticationConfig;
        this.enabledActions = builder.enabledActions;
        this.vpcConnectionArn = builder.vpcConnectionArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the action connector.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the action connector.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The unique identifier of the action connector.
     * </p>
     * 
     * @return The unique identifier of the action connector.
     */
    public final String actionConnectorId() {
        return actionConnectorId;
    }

    /**
     * <p>
     * The type of action connector.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ActionConnectorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of action connector.
     * @see ActionConnectorType
     */
    public final ActionConnectorType type() {
        return ActionConnectorType.fromValue(type);
    }

    /**
     * <p>
     * The type of action connector.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ActionConnectorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of action connector.
     * @see ActionConnectorType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The name of the action connector.
     * </p>
     * 
     * @return The name of the action connector.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The timestamp when the action connector was created.
     * </p>
     * 
     * @return The timestamp when the action connector was created.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The date and time when the action connector was last updated.
     * </p>
     * 
     * @return The date and time when the action connector was last updated.
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    /**
     * <p>
     * The current status of the action connector.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the action connector.
     * @see ResourceStatus
     */
    public final ResourceStatus status() {
        return ResourceStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the action connector.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the action connector.
     * @see ResourceStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Error information if the action connector is in an error state.
     * </p>
     * 
     * @return Error information if the action connector is in an error state.
     */
    public final ActionConnectorError error() {
        return error;
    }

    /**
     * <p>
     * The description of the action connector.
     * </p>
     * 
     * @return The description of the action connector.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The authentication configuration used to connect to the external service.
     * </p>
     * 
     * @return The authentication configuration used to connect to the external service.
     */
    public final ReadAuthConfig authenticationConfig() {
        return authenticationConfig;
    }

    /**
     * For responses, this returns true if the service returned a value for the EnabledActions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnabledActions() {
        return enabledActions != null && !(enabledActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of actions that are enabled for this connector.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnabledActions} method.
     * </p>
     * 
     * @return The list of actions that are enabled for this connector.
     */
    public final List<String> enabledActions() {
        return enabledActions;
    }

    /**
     * <p>
     * The ARN of the VPC connection used for secure connectivity to the external service.
     * </p>
     * 
     * @return The ARN of the VPC connection used for secure connectivity to the external service.
     */
    public final String vpcConnectionArn() {
        return vpcConnectionArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(actionConnectorId());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(error());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(authenticationConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnabledActions() ? enabledActions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vpcConnectionArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ActionConnector)) {
            return false;
        }
        ActionConnector other = (ActionConnector) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(actionConnectorId(), other.actionConnectorId())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(name(), other.name())
                && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(error(), other.error())
                && Objects.equals(description(), other.description())
                && Objects.equals(authenticationConfig(), other.authenticationConfig())
                && hasEnabledActions() == other.hasEnabledActions() && Objects.equals(enabledActions(), other.enabledActions())
                && Objects.equals(vpcConnectionArn(), other.vpcConnectionArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ActionConnector").add("Arn", arn()).add("ActionConnectorId", actionConnectorId())
                .add("Type", typeAsString()).add("Name", name() == null ? null : "*** Sensitive Data Redacted ***")
                .add("CreatedTime", createdTime()).add("LastUpdatedTime", lastUpdatedTime()).add("Status", statusAsString())
                .add("Error", error()).add("Description", description() == null ? null : "*** Sensitive Data Redacted ***")
                .add("AuthenticationConfig", authenticationConfig())
                .add("EnabledActions", hasEnabledActions() ? enabledActions() : null).add("VpcConnectionArn", vpcConnectionArn())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "ActionConnectorId":
            return Optional.ofNullable(clazz.cast(actionConnectorId()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "LastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Error":
            return Optional.ofNullable(clazz.cast(error()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "AuthenticationConfig":
            return Optional.ofNullable(clazz.cast(authenticationConfig()));
        case "EnabledActions":
            return Optional.ofNullable(clazz.cast(enabledActions()));
        case "VpcConnectionArn":
            return Optional.ofNullable(clazz.cast(vpcConnectionArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Arn", ARN_FIELD);
        map.put("ActionConnectorId", ACTION_CONNECTOR_ID_FIELD);
        map.put("Type", TYPE_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("CreatedTime", CREATED_TIME_FIELD);
        map.put("LastUpdatedTime", LAST_UPDATED_TIME_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("Error", ERROR_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("AuthenticationConfig", AUTHENTICATION_CONFIG_FIELD);
        map.put("EnabledActions", ENABLED_ACTIONS_FIELD);
        map.put("VpcConnectionArn", VPC_CONNECTION_ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ActionConnector, T> g) {
        return obj -> g.apply((ActionConnector) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ActionConnector> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the action connector.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the action connector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The unique identifier of the action connector.
         * </p>
         * 
         * @param actionConnectorId
         *        The unique identifier of the action connector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionConnectorId(String actionConnectorId);

        /**
         * <p>
         * The type of action connector.
         * </p>
         * 
         * @param type
         *        The type of action connector.
         * @see ActionConnectorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActionConnectorType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of action connector.
         * </p>
         * 
         * @param type
         *        The type of action connector.
         * @see ActionConnectorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActionConnectorType
         */
        Builder type(ActionConnectorType type);

        /**
         * <p>
         * The name of the action connector.
         * </p>
         * 
         * @param name
         *        The name of the action connector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The timestamp when the action connector was created.
         * </p>
         * 
         * @param createdTime
         *        The timestamp when the action connector was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The date and time when the action connector was last updated.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The date and time when the action connector was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);

        /**
         * <p>
         * The current status of the action connector.
         * </p>
         * 
         * @param status
         *        The current status of the action connector.
         * @see ResourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the action connector.
         * </p>
         * 
         * @param status
         *        The current status of the action connector.
         * @see ResourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceStatus
         */
        Builder status(ResourceStatus status);

        /**
         * <p>
         * Error information if the action connector is in an error state.
         * </p>
         * 
         * @param error
         *        Error information if the action connector is in an error state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder error(ActionConnectorError error);

        /**
         * <p>
         * Error information if the action connector is in an error state.
         * </p>
         * This is a convenience method that creates an instance of the {@link ActionConnectorError.Builder} avoiding
         * the need to create one manually via {@link ActionConnectorError#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ActionConnectorError.Builder#build()} is called immediately and
         * its result is passed to {@link #error(ActionConnectorError)}.
         * 
         * @param error
         *        a consumer that will call methods on {@link ActionConnectorError.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #error(ActionConnectorError)
         */
        default Builder error(Consumer<ActionConnectorError.Builder> error) {
            return error(ActionConnectorError.builder().applyMutation(error).build());
        }

        /**
         * <p>
         * The description of the action connector.
         * </p>
         * 
         * @param description
         *        The description of the action connector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The authentication configuration used to connect to the external service.
         * </p>
         * 
         * @param authenticationConfig
         *        The authentication configuration used to connect to the external service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authenticationConfig(ReadAuthConfig authenticationConfig);

        /**
         * <p>
         * The authentication configuration used to connect to the external service.
         * </p>
         * This is a convenience method that creates an instance of the {@link ReadAuthConfig.Builder} avoiding the need
         * to create one manually via {@link ReadAuthConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ReadAuthConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #authenticationConfig(ReadAuthConfig)}.
         * 
         * @param authenticationConfig
         *        a consumer that will call methods on {@link ReadAuthConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #authenticationConfig(ReadAuthConfig)
         */
        default Builder authenticationConfig(Consumer<ReadAuthConfig.Builder> authenticationConfig) {
            return authenticationConfig(ReadAuthConfig.builder().applyMutation(authenticationConfig).build());
        }

        /**
         * <p>
         * The list of actions that are enabled for this connector.
         * </p>
         * 
         * @param enabledActions
         *        The list of actions that are enabled for this connector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabledActions(Collection<String> enabledActions);

        /**
         * <p>
         * The list of actions that are enabled for this connector.
         * </p>
         * 
         * @param enabledActions
         *        The list of actions that are enabled for this connector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabledActions(String... enabledActions);

        /**
         * <p>
         * The ARN of the VPC connection used for secure connectivity to the external service.
         * </p>
         * 
         * @param vpcConnectionArn
         *        The ARN of the VPC connection used for secure connectivity to the external service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConnectionArn(String vpcConnectionArn);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String actionConnectorId;

        private String type;

        private String name;

        private Instant createdTime;

        private Instant lastUpdatedTime;

        private String status;

        private ActionConnectorError error;

        private String description;

        private ReadAuthConfig authenticationConfig;

        private List<String> enabledActions = DefaultSdkAutoConstructList.getInstance();

        private String vpcConnectionArn;

        private BuilderImpl() {
        }

        private BuilderImpl(ActionConnector model) {
            arn(model.arn);
            actionConnectorId(model.actionConnectorId);
            type(model.type);
            name(model.name);
            createdTime(model.createdTime);
            lastUpdatedTime(model.lastUpdatedTime);
            status(model.status);
            error(model.error);
            description(model.description);
            authenticationConfig(model.authenticationConfig);
            enabledActions(model.enabledActions);
            vpcConnectionArn(model.vpcConnectionArn);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getActionConnectorId() {
            return actionConnectorId;
        }

        public final void setActionConnectorId(String actionConnectorId) {
            this.actionConnectorId = actionConnectorId;
        }

        @Override
        public final Builder actionConnectorId(String actionConnectorId) {
            this.actionConnectorId = actionConnectorId;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ActionConnectorType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ResourceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final ActionConnectorError.Builder getError() {
            return error != null ? error.toBuilder() : null;
        }

        public final void setError(ActionConnectorError.BuilderImpl error) {
            this.error = error != null ? error.build() : null;
        }

        @Override
        public final Builder error(ActionConnectorError error) {
            this.error = error;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final ReadAuthConfig.Builder getAuthenticationConfig() {
            return authenticationConfig != null ? authenticationConfig.toBuilder() : null;
        }

        public final void setAuthenticationConfig(ReadAuthConfig.BuilderImpl authenticationConfig) {
            this.authenticationConfig = authenticationConfig != null ? authenticationConfig.build() : null;
        }

        @Override
        public final Builder authenticationConfig(ReadAuthConfig authenticationConfig) {
            this.authenticationConfig = authenticationConfig;
            return this;
        }

        public final Collection<String> getEnabledActions() {
            if (enabledActions instanceof SdkAutoConstructList) {
                return null;
            }
            return enabledActions;
        }

        public final void setEnabledActions(Collection<String> enabledActions) {
            this.enabledActions = ActionIdListCopier.copy(enabledActions);
        }

        @Override
        public final Builder enabledActions(Collection<String> enabledActions) {
            this.enabledActions = ActionIdListCopier.copy(enabledActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder enabledActions(String... enabledActions) {
            enabledActions(Arrays.asList(enabledActions));
            return this;
        }

        public final String getVpcConnectionArn() {
            return vpcConnectionArn;
        }

        public final void setVpcConnectionArn(String vpcConnectionArn) {
            this.vpcConnectionArn = vpcConnectionArn;
        }

        @Override
        public final Builder vpcConnectionArn(String vpcConnectionArn) {
            this.vpcConnectionArn = vpcConnectionArn;
            return this;
        }

        @Override
        public ActionConnector build() {
            return new ActionConnector(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
