/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.quicksight.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The parameters that are required to connect to a S3 Knowledge Base data source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3KnowledgeBaseParameters implements SdkPojo, Serializable,
        ToCopyableBuilder<S3KnowledgeBaseParameters.Builder, S3KnowledgeBaseParameters> {
    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoleArn").getter(getter(S3KnowledgeBaseParameters::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<String> BUCKET_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BucketUrl").getter(getter(S3KnowledgeBaseParameters::bucketUrl)).setter(setter(Builder::bucketUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BucketUrl").build()).build();

    private static final SdkField<String> METADATA_FILES_LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetadataFilesLocation").getter(getter(S3KnowledgeBaseParameters::metadataFilesLocation))
            .setter(setter(Builder::metadataFilesLocation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetadataFilesLocation").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ROLE_ARN_FIELD,
            BUCKET_URL_FIELD, METADATA_FILES_LOCATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String roleArn;

    private final String bucketUrl;

    private final String metadataFilesLocation;

    private S3KnowledgeBaseParameters(BuilderImpl builder) {
        this.roleArn = builder.roleArn;
        this.bucketUrl = builder.bucketUrl;
        this.metadataFilesLocation = builder.metadataFilesLocation;
    }

    /**
     * <p>
     * Use the <code>RoleArn</code> structure to override an account-wide role for a specific S3 Knowledge Base data
     * source. For example, say an account administrator has turned off all S3 access with an account-wide role. The
     * administrator can then use <code>RoleArn</code> to bypass the account-wide role and allow S3 access for the
     * single S3 Knowledge Base data source that is specified in the structure, even if the account-wide role forbidding
     * S3 access is still active.
     * </p>
     * 
     * @return Use the <code>RoleArn</code> structure to override an account-wide role for a specific S3 Knowledge Base
     *         data source. For example, say an account administrator has turned off all S3 access with an account-wide
     *         role. The administrator can then use <code>RoleArn</code> to bypass the account-wide role and allow S3
     *         access for the single S3 Knowledge Base data source that is specified in the structure, even if the
     *         account-wide role forbidding S3 access is still active.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The URL of the S3 bucket that contains the knowledge base data.
     * </p>
     * 
     * @return The URL of the S3 bucket that contains the knowledge base data.
     */
    public final String bucketUrl() {
        return bucketUrl;
    }

    /**
     * <p>
     * The location of metadata files within the S3 bucket that describe the structure and content of the knowledge
     * base.
     * </p>
     * 
     * @return The location of metadata files within the S3 bucket that describe the structure and content of the
     *         knowledge base.
     */
    public final String metadataFilesLocation() {
        return metadataFilesLocation;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(bucketUrl());
        hashCode = 31 * hashCode + Objects.hashCode(metadataFilesLocation());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3KnowledgeBaseParameters)) {
            return false;
        }
        S3KnowledgeBaseParameters other = (S3KnowledgeBaseParameters) obj;
        return Objects.equals(roleArn(), other.roleArn()) && Objects.equals(bucketUrl(), other.bucketUrl())
                && Objects.equals(metadataFilesLocation(), other.metadataFilesLocation());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3KnowledgeBaseParameters").add("RoleArn", roleArn()).add("BucketUrl", bucketUrl())
                .add("MetadataFilesLocation", metadataFilesLocation()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "BucketUrl":
            return Optional.ofNullable(clazz.cast(bucketUrl()));
        case "MetadataFilesLocation":
            return Optional.ofNullable(clazz.cast(metadataFilesLocation()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RoleArn", ROLE_ARN_FIELD);
        map.put("BucketUrl", BUCKET_URL_FIELD);
        map.put("MetadataFilesLocation", METADATA_FILES_LOCATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<S3KnowledgeBaseParameters, T> g) {
        return obj -> g.apply((S3KnowledgeBaseParameters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3KnowledgeBaseParameters> {
        /**
         * <p>
         * Use the <code>RoleArn</code> structure to override an account-wide role for a specific S3 Knowledge Base data
         * source. For example, say an account administrator has turned off all S3 access with an account-wide role. The
         * administrator can then use <code>RoleArn</code> to bypass the account-wide role and allow S3 access for the
         * single S3 Knowledge Base data source that is specified in the structure, even if the account-wide role
         * forbidding S3 access is still active.
         * </p>
         * 
         * @param roleArn
         *        Use the <code>RoleArn</code> structure to override an account-wide role for a specific S3 Knowledge
         *        Base data source. For example, say an account administrator has turned off all S3 access with an
         *        account-wide role. The administrator can then use <code>RoleArn</code> to bypass the account-wide role
         *        and allow S3 access for the single S3 Knowledge Base data source that is specified in the structure,
         *        even if the account-wide role forbidding S3 access is still active.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The URL of the S3 bucket that contains the knowledge base data.
         * </p>
         * 
         * @param bucketUrl
         *        The URL of the S3 bucket that contains the knowledge base data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketUrl(String bucketUrl);

        /**
         * <p>
         * The location of metadata files within the S3 bucket that describe the structure and content of the knowledge
         * base.
         * </p>
         * 
         * @param metadataFilesLocation
         *        The location of metadata files within the S3 bucket that describe the structure and content of the
         *        knowledge base.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metadataFilesLocation(String metadataFilesLocation);
    }

    static final class BuilderImpl implements Builder {
        private String roleArn;

        private String bucketUrl;

        private String metadataFilesLocation;

        private BuilderImpl() {
        }

        private BuilderImpl(S3KnowledgeBaseParameters model) {
            roleArn(model.roleArn);
            bucketUrl(model.bucketUrl);
            metadataFilesLocation(model.metadataFilesLocation);
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final String getBucketUrl() {
            return bucketUrl;
        }

        public final void setBucketUrl(String bucketUrl) {
            this.bucketUrl = bucketUrl;
        }

        @Override
        public final Builder bucketUrl(String bucketUrl) {
            this.bucketUrl = bucketUrl;
            return this;
        }

        public final String getMetadataFilesLocation() {
            return metadataFilesLocation;
        }

        public final void setMetadataFilesLocation(String metadataFilesLocation) {
            this.metadataFilesLocation = metadataFilesLocation;
        }

        @Override
        public final Builder metadataFilesLocation(String metadataFilesLocation) {
            this.metadataFilesLocation = metadataFilesLocation;
            return this;
        }

        @Override
        public S3KnowledgeBaseParameters build() {
            return new S3KnowledgeBaseParameters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
