/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ram.paginators;

import java.util.Collections;
import java.util.Iterator;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.ram.RamClient;
import software.amazon.awssdk.services.ram.model.ListResourcesRequest;
import software.amazon.awssdk.services.ram.model.ListResourcesResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.ram.RamClient#listResourcesPaginator(software.amazon.awssdk.services.ram.model.ListResourcesRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.ram.model.ListResourcesResponse} that can be used to iterate through all the
 * response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.ram.paginators.ListResourcesIterable responses = client.listResourcesPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.ram.paginators.ListResourcesIterable responses = client.listResourcesPaginator(request);
 *     for (software.amazon.awssdk.services.ram.model.ListResourcesResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.ram.paginators.ListResourcesIterable responses = client.listResourcesPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listResources(software.amazon.awssdk.services.ram.model.ListResourcesRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListResourcesIterable implements SdkIterable<ListResourcesResponse> {
    private final RamClient client;

    private final ListResourcesRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListResourcesIterable(RamClient client, ListResourcesRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new ListResourcesResponseFetcher();
    }

    @Override
    public Iterator<ListResourcesResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link ListResourcesIterable} that can be used to retrieve the
     * consecutive pages that follows the input page.
     * </p>
     */
    private final ListResourcesIterable resume(ListResourcesResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new ListResourcesIterable(client, firstRequest.toBuilder().nextToken(lastSuccessfulPage.nextToken()).build());
        }
        return new ListResourcesIterable(client, firstRequest) {
            @Override
            public Iterator<ListResourcesResponse> iterator() {
                return Collections.emptyIterator();
            }
        };
    }

    private class ListResourcesResponseFetcher implements SyncPageFetcher<ListResourcesResponse> {
        @Override
        public boolean hasNextPage(ListResourcesResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public ListResourcesResponse nextPage(ListResourcesResponse previousPage) {
            if (previousPage == null) {
                return client.listResources(firstRequest);
            }
            return client.listResources(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
