/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ram;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ram.model.AcceptResourceShareInvitationRequest;
import software.amazon.awssdk.services.ram.model.AcceptResourceShareInvitationResponse;
import software.amazon.awssdk.services.ram.model.AssociateResourceSharePermissionRequest;
import software.amazon.awssdk.services.ram.model.AssociateResourceSharePermissionResponse;
import software.amazon.awssdk.services.ram.model.AssociateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.AssociateResourceShareResponse;
import software.amazon.awssdk.services.ram.model.CreateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.CreateResourceShareResponse;
import software.amazon.awssdk.services.ram.model.DeleteResourceShareRequest;
import software.amazon.awssdk.services.ram.model.DeleteResourceShareResponse;
import software.amazon.awssdk.services.ram.model.DisassociateResourceSharePermissionRequest;
import software.amazon.awssdk.services.ram.model.DisassociateResourceSharePermissionResponse;
import software.amazon.awssdk.services.ram.model.DisassociateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.DisassociateResourceShareResponse;
import software.amazon.awssdk.services.ram.model.EnableSharingWithAwsOrganizationRequest;
import software.amazon.awssdk.services.ram.model.EnableSharingWithAwsOrganizationResponse;
import software.amazon.awssdk.services.ram.model.GetPermissionRequest;
import software.amazon.awssdk.services.ram.model.GetPermissionResponse;
import software.amazon.awssdk.services.ram.model.GetResourcePoliciesRequest;
import software.amazon.awssdk.services.ram.model.GetResourcePoliciesResponse;
import software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsRequest;
import software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsResponse;
import software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsRequest;
import software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsResponse;
import software.amazon.awssdk.services.ram.model.GetResourceSharesRequest;
import software.amazon.awssdk.services.ram.model.GetResourceSharesResponse;
import software.amazon.awssdk.services.ram.model.IdempotentParameterMismatchException;
import software.amazon.awssdk.services.ram.model.InvalidClientTokenException;
import software.amazon.awssdk.services.ram.model.InvalidMaxResultsException;
import software.amazon.awssdk.services.ram.model.InvalidNextTokenException;
import software.amazon.awssdk.services.ram.model.InvalidParameterException;
import software.amazon.awssdk.services.ram.model.InvalidResourceTypeException;
import software.amazon.awssdk.services.ram.model.InvalidStateTransitionException;
import software.amazon.awssdk.services.ram.model.ListPendingInvitationResourcesRequest;
import software.amazon.awssdk.services.ram.model.ListPendingInvitationResourcesResponse;
import software.amazon.awssdk.services.ram.model.ListPermissionsRequest;
import software.amazon.awssdk.services.ram.model.ListPermissionsResponse;
import software.amazon.awssdk.services.ram.model.ListPrincipalsRequest;
import software.amazon.awssdk.services.ram.model.ListPrincipalsResponse;
import software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsRequest;
import software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsResponse;
import software.amazon.awssdk.services.ram.model.ListResourceTypesRequest;
import software.amazon.awssdk.services.ram.model.ListResourceTypesResponse;
import software.amazon.awssdk.services.ram.model.ListResourcesRequest;
import software.amazon.awssdk.services.ram.model.ListResourcesResponse;
import software.amazon.awssdk.services.ram.model.MalformedArnException;
import software.amazon.awssdk.services.ram.model.MissingRequiredParameterException;
import software.amazon.awssdk.services.ram.model.OperationNotPermittedException;
import software.amazon.awssdk.services.ram.model.PromoteResourceShareCreatedFromPolicyRequest;
import software.amazon.awssdk.services.ram.model.PromoteResourceShareCreatedFromPolicyResponse;
import software.amazon.awssdk.services.ram.model.RamException;
import software.amazon.awssdk.services.ram.model.RamRequest;
import software.amazon.awssdk.services.ram.model.RejectResourceShareInvitationRequest;
import software.amazon.awssdk.services.ram.model.RejectResourceShareInvitationResponse;
import software.amazon.awssdk.services.ram.model.ResourceArnNotFoundException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationAlreadyAcceptedException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationAlreadyRejectedException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationArnNotFoundException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationExpiredException;
import software.amazon.awssdk.services.ram.model.ResourceShareLimitExceededException;
import software.amazon.awssdk.services.ram.model.ServerInternalException;
import software.amazon.awssdk.services.ram.model.ServiceUnavailableException;
import software.amazon.awssdk.services.ram.model.TagLimitExceededException;
import software.amazon.awssdk.services.ram.model.TagPolicyViolationException;
import software.amazon.awssdk.services.ram.model.TagResourceRequest;
import software.amazon.awssdk.services.ram.model.TagResourceResponse;
import software.amazon.awssdk.services.ram.model.UnknownResourceException;
import software.amazon.awssdk.services.ram.model.UntagResourceRequest;
import software.amazon.awssdk.services.ram.model.UntagResourceResponse;
import software.amazon.awssdk.services.ram.model.UpdateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.UpdateResourceShareResponse;
import software.amazon.awssdk.services.ram.paginators.GetResourcePoliciesIterable;
import software.amazon.awssdk.services.ram.paginators.GetResourceShareAssociationsIterable;
import software.amazon.awssdk.services.ram.paginators.GetResourceShareInvitationsIterable;
import software.amazon.awssdk.services.ram.paginators.GetResourceSharesIterable;
import software.amazon.awssdk.services.ram.paginators.ListPendingInvitationResourcesIterable;
import software.amazon.awssdk.services.ram.paginators.ListPrincipalsIterable;
import software.amazon.awssdk.services.ram.paginators.ListResourcesIterable;
import software.amazon.awssdk.services.ram.transform.AcceptResourceShareInvitationRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.AssociateResourceSharePermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.AssociateResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.CreateResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DeleteResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DisassociateResourceSharePermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DisassociateResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.EnableSharingWithAwsOrganizationRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetPermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourcePoliciesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourceShareAssociationsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourceShareInvitationsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourceSharesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPendingInvitationResourcesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPermissionsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPrincipalsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListResourceSharePermissionsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListResourceTypesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListResourcesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.PromoteResourceShareCreatedFromPolicyRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.RejectResourceShareInvitationRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.UpdateResourceShareRequestMarshaller;

/**
 * Internal implementation of {@link RamClient}.
 *
 * @see RamClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRamClient implements RamClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRamClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Accepts an invitation to a resource share from another AWS account.
     * </p>
     *
     * @param acceptResourceShareInvitationRequest
     * @return Result of the AcceptResourceShareInvitation operation returned by the service.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws ResourceShareInvitationArnNotFoundException
     *         The Amazon Resource Name (ARN) for an invitation was not found.
     * @throws ResourceShareInvitationAlreadyAcceptedException
     *         The invitation was already accepted.
     * @throws ResourceShareInvitationAlreadyRejectedException
     *         The invitation was already rejected.
     * @throws ResourceShareInvitationExpiredException
     *         The invitation is expired.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws InvalidClientTokenException
     *         A client token is not valid.
     * @throws IdempotentParameterMismatchException
     *         A client token input parameter was reused with an operation, but at least one of the other input
     *         parameters is different from the previous call to the operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.AcceptResourceShareInvitation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/AcceptResourceShareInvitation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AcceptResourceShareInvitationResponse acceptResourceShareInvitation(
            AcceptResourceShareInvitationRequest acceptResourceShareInvitationRequest) throws MalformedArnException,
            OperationNotPermittedException, ResourceShareInvitationArnNotFoundException,
            ResourceShareInvitationAlreadyAcceptedException, ResourceShareInvitationAlreadyRejectedException,
            ResourceShareInvitationExpiredException, ServerInternalException, ServiceUnavailableException,
            InvalidClientTokenException, IdempotentParameterMismatchException, AwsServiceException, SdkClientException,
            RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AcceptResourceShareInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AcceptResourceShareInvitationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                acceptResourceShareInvitationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptResourceShareInvitation");

            return clientHandler
                    .execute(new ClientExecutionParams<AcceptResourceShareInvitationRequest, AcceptResourceShareInvitationResponse>()
                            .withOperationName("AcceptResourceShareInvitation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(acceptResourceShareInvitationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AcceptResourceShareInvitationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates the specified resource share with the specified principals and resources.
     * </p>
     *
     * @param associateResourceShareRequest
     * @return Result of the AssociateResourceShare operation returned by the service.
     * @throws IdempotentParameterMismatchException
     *         A client token input parameter was reused with an operation, but at least one of the other input
     *         parameters is different from the previous call to the operation.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws InvalidStateTransitionException
     *         The requested state transition is not valid.
     * @throws ResourceShareLimitExceededException
     *         The requested resource share exceeds the limit for your account.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws InvalidStateTransitionException
     *         The requested state transition is not valid.
     * @throws InvalidClientTokenException
     *         A client token is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.AssociateResourceShare
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/AssociateResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AssociateResourceShareResponse associateResourceShare(AssociateResourceShareRequest associateResourceShareRequest)
            throws IdempotentParameterMismatchException, UnknownResourceException, InvalidStateTransitionException,
            ResourceShareLimitExceededException, MalformedArnException, InvalidClientTokenException, InvalidParameterException,
            OperationNotPermittedException, ServerInternalException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateResourceShareResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResourceShare");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateResourceShareRequest, AssociateResourceShareResponse>()
                            .withOperationName("AssociateResourceShare").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(associateResourceShareRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateResourceShareRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates a permission with a resource share.
     * </p>
     *
     * @param associateResourceSharePermissionRequest
     * @return Result of the AssociateResourceSharePermission operation returned by the service.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws InvalidClientTokenException
     *         A client token is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.AssociateResourceSharePermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/AssociateResourceSharePermission"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateResourceSharePermissionResponse associateResourceSharePermission(
            AssociateResourceSharePermissionRequest associateResourceSharePermissionRequest) throws MalformedArnException,
            UnknownResourceException, InvalidParameterException, InvalidClientTokenException, ServerInternalException,
            ServiceUnavailableException, OperationNotPermittedException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateResourceSharePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateResourceSharePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateResourceSharePermissionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResourceSharePermission");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateResourceSharePermissionRequest, AssociateResourceSharePermissionResponse>()
                            .withOperationName("AssociateResourceSharePermission").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(associateResourceSharePermissionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateResourceSharePermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a resource share.
     * </p>
     *
     * @param createResourceShareRequest
     * @return Result of the CreateResourceShare operation returned by the service.
     * @throws IdempotentParameterMismatchException
     *         A client token input parameter was reused with an operation, but at least one of the other input
     *         parameters is different from the previous call to the operation.
     * @throws InvalidStateTransitionException
     *         The requested state transition is not valid.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws InvalidClientTokenException
     *         A client token is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws ResourceShareLimitExceededException
     *         The requested resource share exceeds the limit for your account.
     * @throws TagPolicyViolationException
     *         The specified tag is a reserved word and cannot be used.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.CreateResourceShare
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/CreateResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateResourceShareResponse createResourceShare(CreateResourceShareRequest createResourceShareRequest)
            throws IdempotentParameterMismatchException, InvalidStateTransitionException, UnknownResourceException,
            MalformedArnException, InvalidClientTokenException, InvalidParameterException, OperationNotPermittedException,
            ResourceShareLimitExceededException, TagPolicyViolationException, ServerInternalException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateResourceShareResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResourceShare");

            return clientHandler.execute(new ClientExecutionParams<CreateResourceShareRequest, CreateResourceShareResponse>()
                    .withOperationName("CreateResourceShare").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createResourceShareRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateResourceShareRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified resource share.
     * </p>
     *
     * @param deleteResourceShareRequest
     * @return Result of the DeleteResourceShare operation returned by the service.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws IdempotentParameterMismatchException
     *         A client token input parameter was reused with an operation, but at least one of the other input
     *         parameters is different from the previous call to the operation.
     * @throws InvalidStateTransitionException
     *         The requested state transition is not valid.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws InvalidClientTokenException
     *         A client token is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.DeleteResourceShare
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DeleteResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteResourceShareResponse deleteResourceShare(DeleteResourceShareRequest deleteResourceShareRequest)
            throws OperationNotPermittedException, IdempotentParameterMismatchException, InvalidStateTransitionException,
            UnknownResourceException, MalformedArnException, InvalidClientTokenException, InvalidParameterException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResourceShareResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourceShare");

            return clientHandler.execute(new ClientExecutionParams<DeleteResourceShareRequest, DeleteResourceShareResponse>()
                    .withOperationName("DeleteResourceShare").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteResourceShareRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteResourceShareRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates the specified principals or resources from the specified resource share.
     * </p>
     *
     * @param disassociateResourceShareRequest
     * @return Result of the DisassociateResourceShare operation returned by the service.
     * @throws IdempotentParameterMismatchException
     *         A client token input parameter was reused with an operation, but at least one of the other input
     *         parameters is different from the previous call to the operation.
     * @throws ResourceShareLimitExceededException
     *         The requested resource share exceeds the limit for your account.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws InvalidStateTransitionException
     *         The requested state transition is not valid.
     * @throws InvalidClientTokenException
     *         A client token is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.DisassociateResourceShare
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DisassociateResourceShare" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DisassociateResourceShareResponse disassociateResourceShare(
            DisassociateResourceShareRequest disassociateResourceShareRequest) throws IdempotentParameterMismatchException,
            ResourceShareLimitExceededException, MalformedArnException, InvalidStateTransitionException,
            InvalidClientTokenException, InvalidParameterException, OperationNotPermittedException, ServerInternalException,
            ServiceUnavailableException, UnknownResourceException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateResourceShareResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResourceShare");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateResourceShareRequest, DisassociateResourceShareResponse>()
                            .withOperationName("DisassociateResourceShare").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateResourceShareRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateResourceShareRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates an AWS RAM permission from a resource share.
     * </p>
     *
     * @param disassociateResourceSharePermissionRequest
     * @return Result of the DisassociateResourceSharePermission operation returned by the service.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws InvalidClientTokenException
     *         A client token is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.DisassociateResourceSharePermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DisassociateResourceSharePermission"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateResourceSharePermissionResponse disassociateResourceSharePermission(
            DisassociateResourceSharePermissionRequest disassociateResourceSharePermissionRequest) throws MalformedArnException,
            UnknownResourceException, InvalidParameterException, InvalidClientTokenException, ServerInternalException,
            ServiceUnavailableException, OperationNotPermittedException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateResourceSharePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateResourceSharePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateResourceSharePermissionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResourceSharePermission");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateResourceSharePermissionRequest, DisassociateResourceSharePermissionResponse>()
                            .withOperationName("DisassociateResourceSharePermission").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateResourceSharePermissionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateResourceSharePermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables resource sharing within your AWS Organization.
     * </p>
     * <p>
     * The caller must be the master account for the AWS Organization.
     * </p>
     *
     * @param enableSharingWithAwsOrganizationRequest
     * @return Result of the EnableSharingWithAwsOrganization operation returned by the service.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.EnableSharingWithAwsOrganization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/EnableSharingWithAwsOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EnableSharingWithAwsOrganizationResponse enableSharingWithAwsOrganization(
            EnableSharingWithAwsOrganizationRequest enableSharingWithAwsOrganizationRequest)
            throws OperationNotPermittedException, ServerInternalException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableSharingWithAwsOrganizationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, EnableSharingWithAwsOrganizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                enableSharingWithAwsOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableSharingWithAwsOrganization");

            return clientHandler
                    .execute(new ClientExecutionParams<EnableSharingWithAwsOrganizationRequest, EnableSharingWithAwsOrganizationResponse>()
                            .withOperationName("EnableSharingWithAwsOrganization").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(enableSharingWithAwsOrganizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new EnableSharingWithAwsOrganizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the contents of an AWS RAM permission in JSON format.
     * </p>
     *
     * @param getPermissionRequest
     * @return Result of the GetPermission operation returned by the service.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.GetPermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetPermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPermissionResponse getPermission(GetPermissionRequest getPermissionRequest) throws InvalidParameterException,
            MalformedArnException, UnknownResourceException, ServerInternalException, ServiceUnavailableException,
            OperationNotPermittedException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPermissionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetPermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPermission");

            return clientHandler.execute(new ClientExecutionParams<GetPermissionRequest, GetPermissionResponse>()
                    .withOperationName("GetPermission").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getPermissionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetPermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the policies for the specified resources that you own and have shared.
     * </p>
     *
     * @param getResourcePoliciesRequest
     * @return Result of the GetResourcePolicies operation returned by the service.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws ResourceArnNotFoundException
     *         An Amazon Resource Name (ARN) was not found.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.GetResourcePolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourcePolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetResourcePoliciesResponse getResourcePolicies(GetResourcePoliciesRequest getResourcePoliciesRequest)
            throws MalformedArnException, InvalidNextTokenException, InvalidParameterException, ResourceArnNotFoundException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourcePoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResourcePoliciesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicies");

            return clientHandler.execute(new ClientExecutionParams<GetResourcePoliciesRequest, GetResourcePoliciesResponse>()
                    .withOperationName("GetResourcePolicies").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getResourcePoliciesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourcePoliciesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the policies for the specified resources that you own and have shared.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getResourcePolicies(software.amazon.awssdk.services.ram.model.GetResourcePoliciesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourcePoliciesIterable responses = client.getResourcePoliciesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ram.paginators.GetResourcePoliciesIterable responses = client
     *             .getResourcePoliciesPaginator(request);
     *     for (software.amazon.awssdk.services.ram.model.GetResourcePoliciesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourcePoliciesIterable responses = client.getResourcePoliciesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourcePolicies(software.amazon.awssdk.services.ram.model.GetResourcePoliciesRequest)} operation.</b>
     * </p>
     *
     * @param getResourcePoliciesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws ResourceArnNotFoundException
     *         An Amazon Resource Name (ARN) was not found.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.GetResourcePolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourcePolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetResourcePoliciesIterable getResourcePoliciesPaginator(GetResourcePoliciesRequest getResourcePoliciesRequest)
            throws MalformedArnException, InvalidNextTokenException, InvalidParameterException, ResourceArnNotFoundException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        return new GetResourcePoliciesIterable(this, applyPaginatorUserAgent(getResourcePoliciesRequest));
    }

    /**
     * <p>
     * Gets the resources or principals for the resource shares that you own.
     * </p>
     *
     * @param getResourceShareAssociationsRequest
     * @return Result of the GetResourceShareAssociations operation returned by the service.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.GetResourceShareAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourceShareAssociationsResponse getResourceShareAssociations(
            GetResourceShareAssociationsRequest getResourceShareAssociationsRequest) throws UnknownResourceException,
            MalformedArnException, InvalidNextTokenException, InvalidParameterException, OperationNotPermittedException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceShareAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResourceShareAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceShareAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceShareAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<GetResourceShareAssociationsRequest, GetResourceShareAssociationsResponse>()
                            .withOperationName("GetResourceShareAssociations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getResourceShareAssociationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetResourceShareAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the resources or principals for the resource shares that you own.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getResourceShareAssociations(software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceShareAssociationsIterable responses = client.getResourceShareAssociationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ram.paginators.GetResourceShareAssociationsIterable responses = client
     *             .getResourceShareAssociationsPaginator(request);
     *     for (software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceShareAssociationsIterable responses = client.getResourceShareAssociationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourceShareAssociations(software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getResourceShareAssociationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.GetResourceShareAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourceShareAssociationsIterable getResourceShareAssociationsPaginator(
            GetResourceShareAssociationsRequest getResourceShareAssociationsRequest) throws UnknownResourceException,
            MalformedArnException, InvalidNextTokenException, InvalidParameterException, OperationNotPermittedException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        return new GetResourceShareAssociationsIterable(this, applyPaginatorUserAgent(getResourceShareAssociationsRequest));
    }

    /**
     * <p>
     * Gets the invitations for resource sharing that you've received.
     * </p>
     *
     * @param getResourceShareInvitationsRequest
     * @return Result of the GetResourceShareInvitations operation returned by the service.
     * @throws ResourceShareInvitationArnNotFoundException
     *         The Amazon Resource Name (ARN) for an invitation was not found.
     * @throws InvalidMaxResultsException
     *         The specified value for MaxResults is not valid.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.GetResourceShareInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourceShareInvitationsResponse getResourceShareInvitations(
            GetResourceShareInvitationsRequest getResourceShareInvitationsRequest)
            throws ResourceShareInvitationArnNotFoundException, InvalidMaxResultsException, MalformedArnException,
            UnknownResourceException, InvalidNextTokenException, InvalidParameterException, ServerInternalException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceShareInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResourceShareInvitationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceShareInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceShareInvitations");

            return clientHandler
                    .execute(new ClientExecutionParams<GetResourceShareInvitationsRequest, GetResourceShareInvitationsResponse>()
                            .withOperationName("GetResourceShareInvitations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getResourceShareInvitationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetResourceShareInvitationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the invitations for resource sharing that you've received.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getResourceShareInvitations(software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceShareInvitationsIterable responses = client.getResourceShareInvitationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ram.paginators.GetResourceShareInvitationsIterable responses = client
     *             .getResourceShareInvitationsPaginator(request);
     *     for (software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceShareInvitationsIterable responses = client.getResourceShareInvitationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourceShareInvitations(software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getResourceShareInvitationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceShareInvitationArnNotFoundException
     *         The Amazon Resource Name (ARN) for an invitation was not found.
     * @throws InvalidMaxResultsException
     *         The specified value for MaxResults is not valid.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.GetResourceShareInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourceShareInvitationsIterable getResourceShareInvitationsPaginator(
            GetResourceShareInvitationsRequest getResourceShareInvitationsRequest)
            throws ResourceShareInvitationArnNotFoundException, InvalidMaxResultsException, MalformedArnException,
            UnknownResourceException, InvalidNextTokenException, InvalidParameterException, ServerInternalException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        return new GetResourceShareInvitationsIterable(this, applyPaginatorUserAgent(getResourceShareInvitationsRequest));
    }

    /**
     * <p>
     * Gets the resource shares that you own or the resource shares that are shared with you.
     * </p>
     *
     * @param getResourceSharesRequest
     * @return Result of the GetResourceShares operation returned by the service.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.GetResourceShares
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShares" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetResourceSharesResponse getResourceShares(GetResourceSharesRequest getResourceSharesRequest)
            throws UnknownResourceException, MalformedArnException, InvalidNextTokenException, InvalidParameterException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceSharesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourceSharesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceSharesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceShares");

            return clientHandler.execute(new ClientExecutionParams<GetResourceSharesRequest, GetResourceSharesResponse>()
                    .withOperationName("GetResourceShares").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getResourceSharesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourceSharesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the resource shares that you own or the resource shares that are shared with you.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getResourceShares(software.amazon.awssdk.services.ram.model.GetResourceSharesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceSharesIterable responses = client.getResourceSharesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ram.paginators.GetResourceSharesIterable responses = client
     *             .getResourceSharesPaginator(request);
     *     for (software.amazon.awssdk.services.ram.model.GetResourceSharesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceSharesIterable responses = client.getResourceSharesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourceShares(software.amazon.awssdk.services.ram.model.GetResourceSharesRequest)} operation.</b>
     * </p>
     *
     * @param getResourceSharesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.GetResourceShares
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShares" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetResourceSharesIterable getResourceSharesPaginator(GetResourceSharesRequest getResourceSharesRequest)
            throws UnknownResourceException, MalformedArnException, InvalidNextTokenException, InvalidParameterException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        return new GetResourceSharesIterable(this, applyPaginatorUserAgent(getResourceSharesRequest));
    }

    /**
     * <p>
     * Lists the resources in a resource share that is shared with you but that the invitation is still pending for.
     * </p>
     *
     * @param listPendingInvitationResourcesRequest
     * @return Result of the ListPendingInvitationResources operation returned by the service.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws ResourceShareInvitationArnNotFoundException
     *         The Amazon Resource Name (ARN) for an invitation was not found.
     * @throws MissingRequiredParameterException
     *         A required input parameter is missing.
     * @throws ResourceShareInvitationAlreadyRejectedException
     *         The invitation was already rejected.
     * @throws ResourceShareInvitationExpiredException
     *         The invitation is expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListPendingInvitationResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPendingInvitationResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPendingInvitationResourcesResponse listPendingInvitationResources(
            ListPendingInvitationResourcesRequest listPendingInvitationResourcesRequest) throws MalformedArnException,
            InvalidNextTokenException, InvalidParameterException, ServerInternalException, ServiceUnavailableException,
            ResourceShareInvitationArnNotFoundException, MissingRequiredParameterException,
            ResourceShareInvitationAlreadyRejectedException, ResourceShareInvitationExpiredException, AwsServiceException,
            SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPendingInvitationResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPendingInvitationResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listPendingInvitationResourcesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPendingInvitationResources");

            return clientHandler
                    .execute(new ClientExecutionParams<ListPendingInvitationResourcesRequest, ListPendingInvitationResourcesResponse>()
                            .withOperationName("ListPendingInvitationResources").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listPendingInvitationResourcesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListPendingInvitationResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the resources in a resource share that is shared with you but that the invitation is still pending for.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPendingInvitationResources(software.amazon.awssdk.services.ram.model.ListPendingInvitationResourcesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListPendingInvitationResourcesIterable responses = client.listPendingInvitationResourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ram.paginators.ListPendingInvitationResourcesIterable responses = client
     *             .listPendingInvitationResourcesPaginator(request);
     *     for (software.amazon.awssdk.services.ram.model.ListPendingInvitationResourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListPendingInvitationResourcesIterable responses = client.listPendingInvitationResourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPendingInvitationResources(software.amazon.awssdk.services.ram.model.ListPendingInvitationResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPendingInvitationResourcesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws ResourceShareInvitationArnNotFoundException
     *         The Amazon Resource Name (ARN) for an invitation was not found.
     * @throws MissingRequiredParameterException
     *         A required input parameter is missing.
     * @throws ResourceShareInvitationAlreadyRejectedException
     *         The invitation was already rejected.
     * @throws ResourceShareInvitationExpiredException
     *         The invitation is expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListPendingInvitationResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPendingInvitationResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPendingInvitationResourcesIterable listPendingInvitationResourcesPaginator(
            ListPendingInvitationResourcesRequest listPendingInvitationResourcesRequest) throws MalformedArnException,
            InvalidNextTokenException, InvalidParameterException, ServerInternalException, ServiceUnavailableException,
            ResourceShareInvitationArnNotFoundException, MissingRequiredParameterException,
            ResourceShareInvitationAlreadyRejectedException, ResourceShareInvitationExpiredException, AwsServiceException,
            SdkClientException, RamException {
        return new ListPendingInvitationResourcesIterable(this, applyPaginatorUserAgent(listPendingInvitationResourcesRequest));
    }

    /**
     * <p>
     * Lists the AWS RAM permissions.
     * </p>
     *
     * @param listPermissionsRequest
     * @return Result of the ListPermissions operation returned by the service.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListPermissions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPermissions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPermissionsResponse listPermissions(ListPermissionsRequest listPermissionsRequest)
            throws InvalidParameterException, InvalidNextTokenException, ServerInternalException, ServiceUnavailableException,
            OperationNotPermittedException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPermissionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPermissions");

            return clientHandler.execute(new ClientExecutionParams<ListPermissionsRequest, ListPermissionsResponse>()
                    .withOperationName("ListPermissions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listPermissionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPermissionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the principals that you have shared resources with or that have shared resources with you.
     * </p>
     *
     * @param listPrincipalsRequest
     * @return Result of the ListPrincipals operation returned by the service.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListPrincipals
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPrincipals" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPrincipalsResponse listPrincipals(ListPrincipalsRequest listPrincipalsRequest) throws MalformedArnException,
            UnknownResourceException, InvalidNextTokenException, InvalidParameterException, ServerInternalException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPrincipalsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPrincipalsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPrincipalsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPrincipals");

            return clientHandler.execute(new ClientExecutionParams<ListPrincipalsRequest, ListPrincipalsResponse>()
                    .withOperationName("ListPrincipals").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listPrincipalsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPrincipalsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the principals that you have shared resources with or that have shared resources with you.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listPrincipals(software.amazon.awssdk.services.ram.model.ListPrincipalsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListPrincipalsIterable responses = client.listPrincipalsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ram.paginators.ListPrincipalsIterable responses = client.listPrincipalsPaginator(request);
     *     for (software.amazon.awssdk.services.ram.model.ListPrincipalsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListPrincipalsIterable responses = client.listPrincipalsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPrincipals(software.amazon.awssdk.services.ram.model.ListPrincipalsRequest)} operation.</b>
     * </p>
     *
     * @param listPrincipalsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListPrincipals
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPrincipals" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPrincipalsIterable listPrincipalsPaginator(ListPrincipalsRequest listPrincipalsRequest)
            throws MalformedArnException, UnknownResourceException, InvalidNextTokenException, InvalidParameterException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        return new ListPrincipalsIterable(this, applyPaginatorUserAgent(listPrincipalsRequest));
    }

    /**
     * <p>
     * Lists the AWS RAM permissions that are associated with a resource share.
     * </p>
     *
     * @param listResourceSharePermissionsRequest
     * @return Result of the ListResourceSharePermissions operation returned by the service.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListResourceSharePermissions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResourceSharePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResourceSharePermissionsResponse listResourceSharePermissions(
            ListResourceSharePermissionsRequest listResourceSharePermissionsRequest) throws InvalidParameterException,
            MalformedArnException, UnknownResourceException, InvalidNextTokenException, ServerInternalException,
            ServiceUnavailableException, OperationNotPermittedException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourceSharePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResourceSharePermissionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceSharePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceSharePermissions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListResourceSharePermissionsRequest, ListResourceSharePermissionsResponse>()
                            .withOperationName("ListResourceSharePermissions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listResourceSharePermissionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListResourceSharePermissionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the shareable resource types supported by AWS RAM.
     * </p>
     *
     * @param listResourceTypesRequest
     * @return Result of the ListResourceTypes operation returned by the service.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListResourceTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResourceTypes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListResourceTypesResponse listResourceTypes(ListResourceTypesRequest listResourceTypesRequest)
            throws InvalidNextTokenException, InvalidParameterException, ServerInternalException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourceTypesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListResourceTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceTypes");

            return clientHandler.execute(new ClientExecutionParams<ListResourceTypesRequest, ListResourceTypesResponse>()
                    .withOperationName("ListResourceTypes").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listResourceTypesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResourceTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the resources that you added to a resource shares or the resources that are shared with you.
     * </p>
     *
     * @param listResourcesRequest
     * @return Result of the ListResources operation returned by the service.
     * @throws InvalidResourceTypeException
     *         The specified resource type is not valid.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListResourcesResponse listResources(ListResourcesRequest listResourcesRequest) throws InvalidResourceTypeException,
            UnknownResourceException, MalformedArnException, InvalidNextTokenException, InvalidParameterException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResources");

            return clientHandler.execute(new ClientExecutionParams<ListResourcesRequest, ListResourcesResponse>()
                    .withOperationName("ListResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the resources that you added to a resource shares or the resources that are shared with you.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listResources(software.amazon.awssdk.services.ram.model.ListResourcesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListResourcesIterable responses = client.listResourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ram.paginators.ListResourcesIterable responses = client.listResourcesPaginator(request);
     *     for (software.amazon.awssdk.services.ram.model.ListResourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListResourcesIterable responses = client.listResourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResources(software.amazon.awssdk.services.ram.model.ListResourcesRequest)} operation.</b>
     * </p>
     *
     * @param listResourcesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidResourceTypeException
     *         The specified resource type is not valid.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws InvalidNextTokenException
     *         The specified value for NextToken is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListResourcesIterable listResourcesPaginator(ListResourcesRequest listResourcesRequest)
            throws InvalidResourceTypeException, UnknownResourceException, MalformedArnException, InvalidNextTokenException,
            InvalidParameterException, ServerInternalException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, RamException {
        return new ListResourcesIterable(this, applyPaginatorUserAgent(listResourcesRequest));
    }

    /**
     * <p>
     * Resource shares that were created by attaching a policy to a resource are visible only to the resource share
     * owner, and the resource share cannot be modified in AWS RAM.
     * </p>
     * <p>
     * Use this API action to promote the resource share. When you promote the resource share, it becomes:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Visible to all principals that it is shared with.
     * </p>
     * </li>
     * <li>
     * <p>
     * Modifiable in AWS RAM.
     * </p>
     * </li>
     * </ul>
     *
     * @param promoteResourceShareCreatedFromPolicyRequest
     * @return Result of the PromoteResourceShareCreatedFromPolicy operation returned by the service.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws MissingRequiredParameterException
     *         A required input parameter is missing.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.PromoteResourceShareCreatedFromPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/PromoteResourceShareCreatedFromPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PromoteResourceShareCreatedFromPolicyResponse promoteResourceShareCreatedFromPolicy(
            PromoteResourceShareCreatedFromPolicyRequest promoteResourceShareCreatedFromPolicyRequest)
            throws MalformedArnException, OperationNotPermittedException, InvalidParameterException,
            MissingRequiredParameterException, ServerInternalException, ServiceUnavailableException, UnknownResourceException,
            AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PromoteResourceShareCreatedFromPolicyResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, PromoteResourceShareCreatedFromPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                promoteResourceShareCreatedFromPolicyRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PromoteResourceShareCreatedFromPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<PromoteResourceShareCreatedFromPolicyRequest, PromoteResourceShareCreatedFromPolicyResponse>()
                            .withOperationName("PromoteResourceShareCreatedFromPolicy").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(promoteResourceShareCreatedFromPolicyRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PromoteResourceShareCreatedFromPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Rejects an invitation to a resource share from another AWS account.
     * </p>
     *
     * @param rejectResourceShareInvitationRequest
     * @return Result of the RejectResourceShareInvitation operation returned by the service.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws ResourceShareInvitationArnNotFoundException
     *         The Amazon Resource Name (ARN) for an invitation was not found.
     * @throws ResourceShareInvitationAlreadyAcceptedException
     *         The invitation was already accepted.
     * @throws ResourceShareInvitationAlreadyRejectedException
     *         The invitation was already rejected.
     * @throws ResourceShareInvitationExpiredException
     *         The invitation is expired.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws InvalidClientTokenException
     *         A client token is not valid.
     * @throws IdempotentParameterMismatchException
     *         A client token input parameter was reused with an operation, but at least one of the other input
     *         parameters is different from the previous call to the operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.RejectResourceShareInvitation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/RejectResourceShareInvitation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RejectResourceShareInvitationResponse rejectResourceShareInvitation(
            RejectResourceShareInvitationRequest rejectResourceShareInvitationRequest) throws MalformedArnException,
            OperationNotPermittedException, ResourceShareInvitationArnNotFoundException,
            ResourceShareInvitationAlreadyAcceptedException, ResourceShareInvitationAlreadyRejectedException,
            ResourceShareInvitationExpiredException, ServerInternalException, ServiceUnavailableException,
            InvalidClientTokenException, IdempotentParameterMismatchException, AwsServiceException, SdkClientException,
            RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RejectResourceShareInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RejectResourceShareInvitationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                rejectResourceShareInvitationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectResourceShareInvitation");

            return clientHandler
                    .execute(new ClientExecutionParams<RejectResourceShareInvitationRequest, RejectResourceShareInvitationResponse>()
                            .withOperationName("RejectResourceShareInvitation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(rejectResourceShareInvitationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RejectResourceShareInvitationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds the specified tags to the specified resource share that you own.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws TagLimitExceededException
     *         The requested tags exceed the limit for your account.
     * @throws ResourceArnNotFoundException
     *         An Amazon Resource Name (ARN) was not found.
     * @throws TagPolicyViolationException
     *         The specified tag is a reserved word and cannot be used.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidParameterException,
            MalformedArnException, TagLimitExceededException, ResourceArnNotFoundException, TagPolicyViolationException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified resource share that you own.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidParameterException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified resource share that you own.
     * </p>
     *
     * @param updateResourceShareRequest
     * @return Result of the UpdateResourceShare operation returned by the service.
     * @throws IdempotentParameterMismatchException
     *         A client token input parameter was reused with an operation, but at least one of the other input
     *         parameters is different from the previous call to the operation.
     * @throws MissingRequiredParameterException
     *         A required input parameter is missing.
     * @throws UnknownResourceException
     *         A specified resource was not found.
     * @throws MalformedArnException
     *         The format of an Amazon Resource Name (ARN) is not valid.
     * @throws InvalidClientTokenException
     *         A client token is not valid.
     * @throws InvalidParameterException
     *         A parameter is not valid.
     * @throws OperationNotPermittedException
     *         The requested operation is not permitted.
     * @throws ServerInternalException
     *         The service could not respond to the request due to an internal problem.
     * @throws ServiceUnavailableException
     *         The service is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.UpdateResourceShare
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/UpdateResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateResourceShareResponse updateResourceShare(UpdateResourceShareRequest updateResourceShareRequest)
            throws IdempotentParameterMismatchException, MissingRequiredParameterException, UnknownResourceException,
            MalformedArnException, InvalidClientTokenException, InvalidParameterException, OperationNotPermittedException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateResourceShareResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResourceShare");

            return clientHandler.execute(new ClientExecutionParams<UpdateResourceShareRequest, UpdateResourceShareResponse>()
                    .withOperationName("UpdateResourceShare").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateResourceShareRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateResourceShareRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RamException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidClientTokenException")
                                .exceptionBuilderSupplier(InvalidClientTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnknownResourceException")
                                .exceptionBuilderSupplier(UnknownResourceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerInternalException")
                                .exceptionBuilderSupplier(ServerInternalException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                                .exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidMaxResultsException")
                                .exceptionBuilderSupplier(InvalidMaxResultsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceTypeException")
                                .exceptionBuilderSupplier(InvalidResourceTypeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationExpiredException")
                                .exceptionBuilderSupplier(ResourceShareInvitationExpiredException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingRequiredParameterException")
                                .exceptionBuilderSupplier(MissingRequiredParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagLimitExceededException")
                                .exceptionBuilderSupplier(TagLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceArnNotFoundException")
                                .exceptionBuilderSupplier(ResourceArnNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationNotPermittedException")
                                .exceptionBuilderSupplier(OperationNotPermittedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareLimitExceededException")
                                .exceptionBuilderSupplier(ResourceShareLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationAlreadyRejectedException")
                                .exceptionBuilderSupplier(ResourceShareInvitationAlreadyRejectedException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MalformedArnException")
                                .exceptionBuilderSupplier(MalformedArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStateTransitionException")
                                .exceptionBuilderSupplier(InvalidStateTransitionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationAlreadyAcceptedException")
                                .exceptionBuilderSupplier(ResourceShareInvitationAlreadyAcceptedException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationArnNotFoundException")
                                .exceptionBuilderSupplier(ResourceShareInvitationArnNotFoundException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagPolicyViolationException")
                                .exceptionBuilderSupplier(TagPolicyViolationException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends RamRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
