/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ram;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ram.model.AcceptResourceShareInvitationRequest;
import software.amazon.awssdk.services.ram.model.AcceptResourceShareInvitationResponse;
import software.amazon.awssdk.services.ram.model.AssociateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.AssociateResourceShareResponse;
import software.amazon.awssdk.services.ram.model.CreateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.CreateResourceShareResponse;
import software.amazon.awssdk.services.ram.model.DeleteResourceShareRequest;
import software.amazon.awssdk.services.ram.model.DeleteResourceShareResponse;
import software.amazon.awssdk.services.ram.model.DisassociateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.DisassociateResourceShareResponse;
import software.amazon.awssdk.services.ram.model.EnableSharingWithAwsOrganizationRequest;
import software.amazon.awssdk.services.ram.model.EnableSharingWithAwsOrganizationResponse;
import software.amazon.awssdk.services.ram.model.GetResourcePoliciesRequest;
import software.amazon.awssdk.services.ram.model.GetResourcePoliciesResponse;
import software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsRequest;
import software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsResponse;
import software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsRequest;
import software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsResponse;
import software.amazon.awssdk.services.ram.model.GetResourceSharesRequest;
import software.amazon.awssdk.services.ram.model.GetResourceSharesResponse;
import software.amazon.awssdk.services.ram.model.IdempotentParameterMismatchException;
import software.amazon.awssdk.services.ram.model.InvalidClientTokenException;
import software.amazon.awssdk.services.ram.model.InvalidMaxResultsException;
import software.amazon.awssdk.services.ram.model.InvalidNextTokenException;
import software.amazon.awssdk.services.ram.model.InvalidParameterException;
import software.amazon.awssdk.services.ram.model.InvalidResourceTypeException;
import software.amazon.awssdk.services.ram.model.InvalidStateTransitionException;
import software.amazon.awssdk.services.ram.model.ListPrincipalsRequest;
import software.amazon.awssdk.services.ram.model.ListPrincipalsResponse;
import software.amazon.awssdk.services.ram.model.ListResourcesRequest;
import software.amazon.awssdk.services.ram.model.ListResourcesResponse;
import software.amazon.awssdk.services.ram.model.MalformedArnException;
import software.amazon.awssdk.services.ram.model.MissingRequiredParameterException;
import software.amazon.awssdk.services.ram.model.OperationNotPermittedException;
import software.amazon.awssdk.services.ram.model.RamException;
import software.amazon.awssdk.services.ram.model.RamRequest;
import software.amazon.awssdk.services.ram.model.RejectResourceShareInvitationRequest;
import software.amazon.awssdk.services.ram.model.RejectResourceShareInvitationResponse;
import software.amazon.awssdk.services.ram.model.ResourceArnNotFoundException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationAlreadyAcceptedException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationAlreadyRejectedException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationArnNotFoundException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationExpiredException;
import software.amazon.awssdk.services.ram.model.ResourceShareLimitExceededException;
import software.amazon.awssdk.services.ram.model.ServerInternalException;
import software.amazon.awssdk.services.ram.model.ServiceUnavailableException;
import software.amazon.awssdk.services.ram.model.TagLimitExceededException;
import software.amazon.awssdk.services.ram.model.TagResourceRequest;
import software.amazon.awssdk.services.ram.model.TagResourceResponse;
import software.amazon.awssdk.services.ram.model.UnknownResourceException;
import software.amazon.awssdk.services.ram.model.UntagResourceRequest;
import software.amazon.awssdk.services.ram.model.UntagResourceResponse;
import software.amazon.awssdk.services.ram.model.UpdateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.UpdateResourceShareResponse;
import software.amazon.awssdk.services.ram.paginators.GetResourcePoliciesPublisher;
import software.amazon.awssdk.services.ram.paginators.GetResourceShareAssociationsPublisher;
import software.amazon.awssdk.services.ram.paginators.GetResourceShareInvitationsPublisher;
import software.amazon.awssdk.services.ram.paginators.GetResourceSharesPublisher;
import software.amazon.awssdk.services.ram.paginators.ListPrincipalsPublisher;
import software.amazon.awssdk.services.ram.paginators.ListResourcesPublisher;
import software.amazon.awssdk.services.ram.transform.AcceptResourceShareInvitationRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.AssociateResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.CreateResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DeleteResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DisassociateResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.EnableSharingWithAwsOrganizationRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourcePoliciesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourceShareAssociationsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourceShareInvitationsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourceSharesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPrincipalsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListResourcesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.RejectResourceShareInvitationRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.UpdateResourceShareRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link RamAsyncClient}.
 *
 * @see RamAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRamAsyncClient implements RamAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRamAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRamAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Accepts an invitation to a resource share from another AWS account.
     * </p>
     *
     * @param acceptResourceShareInvitationRequest
     * @return A Java Future containing the result of the AcceptResourceShareInvitation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ResourceShareInvitationArnNotFoundException The Amazon Resource Name (ARN) for an invitation was not
     *         found.</li>
     *         <li>ResourceShareInvitationAlreadyAcceptedException The invitation was already accepted.</li>
     *         <li>ResourceShareInvitationAlreadyRejectedException The invitation was already rejected.</li>
     *         <li>ResourceShareInvitationExpiredException The invitation is expired.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.AcceptResourceShareInvitation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/AcceptResourceShareInvitation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptResourceShareInvitationResponse> acceptResourceShareInvitation(
            AcceptResourceShareInvitationRequest acceptResourceShareInvitationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptResourceShareInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcceptResourceShareInvitationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler
                    .execute(new ClientExecutionParams<AcceptResourceShareInvitationRequest, AcceptResourceShareInvitationResponse>()
                            .withOperationName("AcceptResourceShareInvitation")
                            .withMarshaller(new AcceptResourceShareInvitationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(acceptResourceShareInvitationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates the specified resource share with the specified principals and resources.
     * </p>
     *
     * @param associateResourceShareRequest
     * @return A Java Future containing the result of the AssociateResourceShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>IdempotentParameterMismatchException A client token input parameter was reused with an operation, but
     *         at least one of the other input parameters is different from the previous call to the operation.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>InvalidStateTransitionException The requested state transition is not valid.</li>
     *         <li>ResourceShareLimitExceededException The requested resource share exceeds the limit for your account.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidStateTransitionException The requested state transition is not valid.</li>
     *         <li>InvalidClientTokenException A client token is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.AssociateResourceShare
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/AssociateResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateResourceShareResponse> associateResourceShare(
            AssociateResourceShareRequest associateResourceShareRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateResourceShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateResourceShareRequest, AssociateResourceShareResponse>()
                            .withOperationName("AssociateResourceShare")
                            .withMarshaller(new AssociateResourceShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(associateResourceShareRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a resource share.
     * </p>
     *
     * @param createResourceShareRequest
     * @return A Java Future containing the result of the CreateResourceShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>IdempotentParameterMismatchException A client token input parameter was reused with an operation, but
     *         at least one of the other input parameters is different from the previous call to the operation.</li>
     *         <li>InvalidStateTransitionException The requested state transition is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidClientTokenException A client token is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ResourceShareLimitExceededException The requested resource share exceeds the limit for your account.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.CreateResourceShare
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/CreateResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateResourceShareResponse> createResourceShare(
            CreateResourceShareRequest createResourceShareRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateResourceShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<CreateResourceShareRequest, CreateResourceShareResponse>()
                    .withOperationName("CreateResourceShare")
                    .withMarshaller(new CreateResourceShareRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(createResourceShareRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified resource share.
     * </p>
     *
     * @param deleteResourceShareRequest
     * @return A Java Future containing the result of the DeleteResourceShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>IdempotentParameterMismatchException A client token input parameter was reused with an operation, but
     *         at least one of the other input parameters is different from the previous call to the operation.</li>
     *         <li>InvalidStateTransitionException The requested state transition is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidClientTokenException A client token is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.DeleteResourceShare
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DeleteResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResourceShareResponse> deleteResourceShare(
            DeleteResourceShareRequest deleteResourceShareRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResourceShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DeleteResourceShareRequest, DeleteResourceShareResponse>()
                    .withOperationName("DeleteResourceShare")
                    .withMarshaller(new DeleteResourceShareRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(deleteResourceShareRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates the specified principals or resources from the specified resource share.
     * </p>
     *
     * @param disassociateResourceShareRequest
     * @return A Java Future containing the result of the DisassociateResourceShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>IdempotentParameterMismatchException A client token input parameter was reused with an operation, but
     *         at least one of the other input parameters is different from the previous call to the operation.</li>
     *         <li>ResourceShareLimitExceededException The requested resource share exceeds the limit for your account.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidStateTransitionException The requested state transition is not valid.</li>
     *         <li>InvalidClientTokenException A client token is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.DisassociateResourceShare
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DisassociateResourceShare" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateResourceShareResponse> disassociateResourceShare(
            DisassociateResourceShareRequest disassociateResourceShareRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateResourceShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateResourceShareRequest, DisassociateResourceShareResponse>()
                            .withOperationName("DisassociateResourceShare")
                            .withMarshaller(new DisassociateResourceShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disassociateResourceShareRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables resource sharing within your organization.
     * </p>
     *
     * @param enableSharingWithAwsOrganizationRequest
     * @return A Java Future containing the result of the EnableSharingWithAwsOrganization operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.EnableSharingWithAwsOrganization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/EnableSharingWithAwsOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableSharingWithAwsOrganizationResponse> enableSharingWithAwsOrganization(
            EnableSharingWithAwsOrganizationRequest enableSharingWithAwsOrganizationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableSharingWithAwsOrganizationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, EnableSharingWithAwsOrganizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler
                    .execute(new ClientExecutionParams<EnableSharingWithAwsOrganizationRequest, EnableSharingWithAwsOrganizationResponse>()
                            .withOperationName("EnableSharingWithAwsOrganization")
                            .withMarshaller(new EnableSharingWithAwsOrganizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(enableSharingWithAwsOrganizationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the policies for the specifies resources.
     * </p>
     *
     * @param getResourcePoliciesRequest
     * @return A Java Future containing the result of the GetResourcePolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for NextToken is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourcePolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourcePolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourcePoliciesResponse> getResourcePolicies(
            GetResourcePoliciesRequest getResourcePoliciesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourcePoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourcePoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<GetResourcePoliciesRequest, GetResourcePoliciesResponse>()
                    .withOperationName("GetResourcePolicies")
                    .withMarshaller(new GetResourcePoliciesRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(getResourcePoliciesRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the policies for the specifies resources.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getResourcePolicies(software.amazon.awssdk.services.ram.model.GetResourcePoliciesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourcePoliciesPublisher publisher = client.getResourcePoliciesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourcePoliciesPublisher publisher = client.getResourcePoliciesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.GetResourcePoliciesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.GetResourcePoliciesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourcePolicies(software.amazon.awssdk.services.ram.model.GetResourcePoliciesRequest)} operation.</b>
     * </p>
     *
     * @param getResourcePoliciesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for NextToken is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourcePolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourcePolicies" target="_top">AWS API
     *      Documentation</a>
     */
    public GetResourcePoliciesPublisher getResourcePoliciesPaginator(GetResourcePoliciesRequest getResourcePoliciesRequest) {
        return new GetResourcePoliciesPublisher(this, applyPaginatorUserAgent(getResourcePoliciesRequest));
    }

    /**
     * <p>
     * Gets the associations for the specified resource share.
     * </p>
     *
     * @param getResourceShareAssociationsRequest
     * @return A Java Future containing the result of the GetResourceShareAssociations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for NextToken is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourceShareAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceShareAssociationsResponse> getResourceShareAssociations(
            GetResourceShareAssociationsRequest getResourceShareAssociationsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceShareAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceShareAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler
                    .execute(new ClientExecutionParams<GetResourceShareAssociationsRequest, GetResourceShareAssociationsResponse>()
                            .withOperationName("GetResourceShareAssociations")
                            .withMarshaller(new GetResourceShareAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getResourceShareAssociationsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the associations for the specified resource share.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getResourceShareAssociations(software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceShareAssociationsPublisher publisher = client.getResourceShareAssociationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceShareAssociationsPublisher publisher = client.getResourceShareAssociationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourceShareAssociations(software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getResourceShareAssociationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for NextToken is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourceShareAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    public GetResourceShareAssociationsPublisher getResourceShareAssociationsPaginator(
            GetResourceShareAssociationsRequest getResourceShareAssociationsRequest) {
        return new GetResourceShareAssociationsPublisher(this, applyPaginatorUserAgent(getResourceShareAssociationsRequest));
    }

    /**
     * <p>
     * Gets the specified invitations for resource sharing.
     * </p>
     *
     * @param getResourceShareInvitationsRequest
     * @return A Java Future containing the result of the GetResourceShareInvitations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceShareInvitationArnNotFoundException The Amazon Resource Name (ARN) for an invitation was not
     *         found.</li>
     *         <li>InvalidMaxResultsException The specified value for MaxResults is not valid.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for NextToken is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourceShareInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceShareInvitationsResponse> getResourceShareInvitations(
            GetResourceShareInvitationsRequest getResourceShareInvitationsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceShareInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceShareInvitationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler
                    .execute(new ClientExecutionParams<GetResourceShareInvitationsRequest, GetResourceShareInvitationsResponse>()
                            .withOperationName("GetResourceShareInvitations")
                            .withMarshaller(new GetResourceShareInvitationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getResourceShareInvitationsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the specified invitations for resource sharing.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getResourceShareInvitations(software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceShareInvitationsPublisher publisher = client.getResourceShareInvitationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceShareInvitationsPublisher publisher = client.getResourceShareInvitationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourceShareInvitations(software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getResourceShareInvitationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceShareInvitationArnNotFoundException The Amazon Resource Name (ARN) for an invitation was not
     *         found.</li>
     *         <li>InvalidMaxResultsException The specified value for MaxResults is not valid.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for NextToken is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourceShareInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    public GetResourceShareInvitationsPublisher getResourceShareInvitationsPaginator(
            GetResourceShareInvitationsRequest getResourceShareInvitationsRequest) {
        return new GetResourceShareInvitationsPublisher(this, applyPaginatorUserAgent(getResourceShareInvitationsRequest));
    }

    /**
     * <p>
     * Gets the specified resource shares or all of your resource shares.
     * </p>
     *
     * @param getResourceSharesRequest
     * @return A Java Future containing the result of the GetResourceShares operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for NextToken is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourceShares
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShares" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceSharesResponse> getResourceShares(GetResourceSharesRequest getResourceSharesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceSharesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceSharesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<GetResourceSharesRequest, GetResourceSharesResponse>()
                    .withOperationName("GetResourceShares")
                    .withMarshaller(new GetResourceSharesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getResourceSharesRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the specified resource shares or all of your resource shares.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getResourceShares(software.amazon.awssdk.services.ram.model.GetResourceSharesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceSharesPublisher publisher = client.getResourceSharesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceSharesPublisher publisher = client.getResourceSharesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.GetResourceSharesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.GetResourceSharesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourceShares(software.amazon.awssdk.services.ram.model.GetResourceSharesRequest)} operation.</b>
     * </p>
     *
     * @param getResourceSharesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for NextToken is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourceShares
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShares" target="_top">AWS API
     *      Documentation</a>
     */
    public GetResourceSharesPublisher getResourceSharesPaginator(GetResourceSharesRequest getResourceSharesRequest) {
        return new GetResourceSharesPublisher(this, applyPaginatorUserAgent(getResourceSharesRequest));
    }

    /**
     * <p>
     * Lists the principals with access to the specified resource.
     * </p>
     *
     * @param listPrincipalsRequest
     * @return A Java Future containing the result of the ListPrincipals operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>InvalidNextTokenException The specified value for NextToken is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListPrincipals
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPrincipals" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPrincipalsResponse> listPrincipals(ListPrincipalsRequest listPrincipalsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPrincipalsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPrincipalsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListPrincipalsRequest, ListPrincipalsResponse>()
                    .withOperationName("ListPrincipals").withMarshaller(new ListPrincipalsRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(listPrincipalsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the principals with access to the specified resource.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listPrincipals(software.amazon.awssdk.services.ram.model.ListPrincipalsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListPrincipalsPublisher publisher = client.listPrincipalsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListPrincipalsPublisher publisher = client.listPrincipalsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.ListPrincipalsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.ListPrincipalsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPrincipals(software.amazon.awssdk.services.ram.model.ListPrincipalsRequest)} operation.</b>
     * </p>
     *
     * @param listPrincipalsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>InvalidNextTokenException The specified value for NextToken is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListPrincipals
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPrincipals" target="_top">AWS API
     *      Documentation</a>
     */
    public ListPrincipalsPublisher listPrincipalsPaginator(ListPrincipalsRequest listPrincipalsRequest) {
        return new ListPrincipalsPublisher(this, applyPaginatorUserAgent(listPrincipalsRequest));
    }

    /**
     * <p>
     * Lists the resources that the specified principal can access.
     * </p>
     *
     * @param listResourcesRequest
     * @return A Java Future containing the result of the ListResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceTypeException The specified resource type is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for NextToken is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourcesResponse> listResources(ListResourcesRequest listResourcesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListResourcesRequest, ListResourcesResponse>()
                    .withOperationName("ListResources").withMarshaller(new ListResourcesRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(listResourcesRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resources that the specified principal can access.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listResources(software.amazon.awssdk.services.ram.model.ListResourcesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListResourcesPublisher publisher = client.listResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListResourcesPublisher publisher = client.listResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.ListResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.ListResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResources(software.amazon.awssdk.services.ram.model.ListResourcesRequest)} operation.</b>
     * </p>
     *
     * @param listResourcesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceTypeException The specified resource type is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for NextToken is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResources" target="_top">AWS API
     *      Documentation</a>
     */
    public ListResourcesPublisher listResourcesPaginator(ListResourcesRequest listResourcesRequest) {
        return new ListResourcesPublisher(this, applyPaginatorUserAgent(listResourcesRequest));
    }

    /**
     * <p>
     * Rejects an invitation to a resource share from another AWS account.
     * </p>
     *
     * @param rejectResourceShareInvitationRequest
     * @return A Java Future containing the result of the RejectResourceShareInvitation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ResourceShareInvitationArnNotFoundException The Amazon Resource Name (ARN) for an invitation was not
     *         found.</li>
     *         <li>ResourceShareInvitationAlreadyAcceptedException The invitation was already accepted.</li>
     *         <li>ResourceShareInvitationAlreadyRejectedException The invitation was already rejected.</li>
     *         <li>ResourceShareInvitationExpiredException The invitation is expired.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.RejectResourceShareInvitation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/RejectResourceShareInvitation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RejectResourceShareInvitationResponse> rejectResourceShareInvitation(
            RejectResourceShareInvitationRequest rejectResourceShareInvitationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RejectResourceShareInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RejectResourceShareInvitationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler
                    .execute(new ClientExecutionParams<RejectResourceShareInvitationRequest, RejectResourceShareInvitationResponse>()
                            .withOperationName("RejectResourceShareInvitation")
                            .withMarshaller(new RejectResourceShareInvitationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(rejectResourceShareInvitationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds the specified tags to the specified resource share.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>TagLimitExceededException The requested tags exceed the limit for your account.</li>
     *         <li>ResourceArnNotFoundException An Amazon Resource Name (ARN) was not found.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(tagResourceRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified resource share.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(untagResourceRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified resource share.
     * </p>
     *
     * @param updateResourceShareRequest
     * @return A Java Future containing the result of the UpdateResourceShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>IdempotentParameterMismatchException A client token input parameter was reused with an operation, but
     *         at least one of the other input parameters is different from the previous call to the operation.</li>
     *         <li>MissingRequiredParameterException A required input parameter is missing.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidClientTokenException A client token is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.UpdateResourceShare
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/UpdateResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateResourceShareResponse> updateResourceShare(
            UpdateResourceShareRequest updateResourceShareRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateResourceShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<UpdateResourceShareRequest, UpdateResourceShareResponse>()
                    .withOperationName("UpdateResourceShare")
                    .withMarshaller(new UpdateResourceShareRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(updateResourceShareRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RamException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidClientTokenException")
                                .exceptionBuilderSupplier(InvalidClientTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnknownResourceException")
                                .exceptionBuilderSupplier(UnknownResourceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerInternalException")
                                .exceptionBuilderSupplier(ServerInternalException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                                .exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidMaxResultsException")
                                .exceptionBuilderSupplier(InvalidMaxResultsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceTypeException")
                                .exceptionBuilderSupplier(InvalidResourceTypeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationExpiredException")
                                .exceptionBuilderSupplier(ResourceShareInvitationExpiredException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingRequiredParameterException")
                                .exceptionBuilderSupplier(MissingRequiredParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagLimitExceededException")
                                .exceptionBuilderSupplier(TagLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceArnNotFoundException")
                                .exceptionBuilderSupplier(ResourceArnNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationNotPermittedException")
                                .exceptionBuilderSupplier(OperationNotPermittedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareLimitExceededException")
                                .exceptionBuilderSupplier(ResourceShareLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationAlreadyRejectedException")
                                .exceptionBuilderSupplier(ResourceShareInvitationAlreadyRejectedException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MalformedArnException")
                                .exceptionBuilderSupplier(MalformedArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStateTransitionException")
                                .exceptionBuilderSupplier(InvalidStateTransitionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationAlreadyAcceptedException")
                                .exceptionBuilderSupplier(ResourceShareInvitationAlreadyAcceptedException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationArnNotFoundException")
                                .exceptionBuilderSupplier(ResourceShareInvitationArnNotFoundException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build());
    }

    private <T extends RamRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
