/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ram.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class ListPrincipalsRequest extends RamRequest implements
        ToCopyableBuilder<ListPrincipalsRequest.Builder, ListPrincipalsRequest> {
    private static final SdkField<String> RESOURCE_OWNER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceOwner").getter(getter(ListPrincipalsRequest::resourceOwnerAsString))
            .setter(setter(Builder::resourceOwner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceOwner").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceArn").getter(getter(ListPrincipalsRequest::resourceArn)).setter(setter(Builder::resourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceArn").build()).build();

    private static final SdkField<List<String>> PRINCIPALS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("principals")
            .getter(getter(ListPrincipalsRequest::principals))
            .setter(setter(Builder::principals))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("principals").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceType").getter(getter(ListPrincipalsRequest::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<List<String>> RESOURCE_SHARE_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("resourceShareArns")
            .getter(getter(ListPrincipalsRequest::resourceShareArns))
            .setter(setter(Builder::resourceShareArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceShareArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NEXT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("nextToken").getter(getter(ListPrincipalsRequest::nextToken)).setter(setter(Builder::nextToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nextToken").build()).build();

    private static final SdkField<Integer> MAX_RESULTS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxResults").getter(getter(ListPrincipalsRequest::maxResults)).setter(setter(Builder::maxResults))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxResults").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_OWNER_FIELD,
            RESOURCE_ARN_FIELD, PRINCIPALS_FIELD, RESOURCE_TYPE_FIELD, RESOURCE_SHARE_ARNS_FIELD, NEXT_TOKEN_FIELD,
            MAX_RESULTS_FIELD));

    private final String resourceOwner;

    private final String resourceArn;

    private final List<String> principals;

    private final String resourceType;

    private final List<String> resourceShareArns;

    private final String nextToken;

    private final Integer maxResults;

    private ListPrincipalsRequest(BuilderImpl builder) {
        super(builder);
        this.resourceOwner = builder.resourceOwner;
        this.resourceArn = builder.resourceArn;
        this.principals = builder.principals;
        this.resourceType = builder.resourceType;
        this.resourceShareArns = builder.resourceShareArns;
        this.nextToken = builder.nextToken;
        this.maxResults = builder.maxResults;
    }

    /**
     * <p>
     * Specifies that you want to list information for only resource shares that match the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>SELF</code> </b> – principals that your account is sharing resources with
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>OTHER-ACCOUNTS</code> </b> – principals that are sharing resources with your account
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceOwner}
     * will return {@link ResourceOwner#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceOwnerAsString}.
     * </p>
     * 
     * @return Specifies that you want to list information for only resource shares that match the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>SELF</code> </b> – principals that your account is sharing resources with
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>OTHER-ACCOUNTS</code> </b> – principals that are sharing resources with your account
     *         </p>
     *         </li>
     * @see ResourceOwner
     */
    public final ResourceOwner resourceOwner() {
        return ResourceOwner.fromValue(resourceOwner);
    }

    /**
     * <p>
     * Specifies that you want to list information for only resource shares that match the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>SELF</code> </b> – principals that your account is sharing resources with
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>OTHER-ACCOUNTS</code> </b> – principals that are sharing resources with your account
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceOwner}
     * will return {@link ResourceOwner#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceOwnerAsString}.
     * </p>
     * 
     * @return Specifies that you want to list information for only resource shares that match the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>SELF</code> </b> – principals that your account is sharing resources with
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>OTHER-ACCOUNTS</code> </b> – principals that are sharing resources with your account
     *         </p>
     *         </li>
     * @see ResourceOwner
     */
    public final String resourceOwnerAsString() {
        return resourceOwner;
    }

    /**
     * <p>
     * Specifies that you want to list principal information for the resource share with the specified <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resoure Name (ARN)</a>.
     * </p>
     * 
     * @return Specifies that you want to list principal information for the resource share with the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resoure Name
     *         (ARN)</a>.
     */
    public final String resourceArn() {
        return resourceArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Principals property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPrincipals() {
        return principals != null && !(principals instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies that you want to list information for only the listed principals.
     * </p>
     * <p>
     * You can include the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An Amazon Web Services account ID, for example: <code>123456789012</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * An <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resoure Name
     * (ARN)</a> of an organization in Organizations, for example:
     * <code>organizations::123456789012:organization/o-exampleorgid</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * An ARN of an organizational unit (OU) in Organizations, for example:
     * <code>organizations::123456789012:ou/o-exampleorgid/ou-examplerootid-exampleouid123</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * An ARN of an IAM role, for example: <code>iam::123456789012:role/rolename</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * An ARN of an IAM user, for example: <code>iam::123456789012user/username</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Not all resource types can be shared with IAM roles and users. For more information, see <a href=
     * "https://docs.aws.amazon.com/ram/latest/userguide/permissions.html#permissions-rbp-supported-resource-types"
     * >Sharing with IAM roles and users</a> in the <i>Resource Access Manager User Guide</i>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPrincipals} method.
     * </p>
     * 
     * @return Specifies that you want to list information for only the listed principals.</p>
     *         <p>
     *         You can include the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         An Amazon Web Services account ID, for example: <code>123456789012</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resoure
     *         Name (ARN)</a> of an organization in Organizations, for example:
     *         <code>organizations::123456789012:organization/o-exampleorgid</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An ARN of an organizational unit (OU) in Organizations, for example:
     *         <code>organizations::123456789012:ou/o-exampleorgid/ou-examplerootid-exampleouid123</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An ARN of an IAM role, for example: <code>iam::123456789012:role/rolename</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An ARN of an IAM user, for example: <code>iam::123456789012user/username</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         Not all resource types can be shared with IAM roles and users. For more information, see <a href=
     *         "https://docs.aws.amazon.com/ram/latest/userguide/permissions.html#permissions-rbp-supported-resource-types"
     *         >Sharing with IAM roles and users</a> in the <i>Resource Access Manager User Guide</i>.
     *         </p>
     */
    public final List<String> principals() {
        return principals;
    }

    /**
     * <p>
     * Specifies that you want to list information for only principals associated with resource shares that include the
     * specified resource type.
     * </p>
     * <p>
     * For a list of valid values, query the <a>ListResourceTypes</a> operation.
     * </p>
     * 
     * @return Specifies that you want to list information for only principals associated with resource shares that
     *         include the specified resource type.</p>
     *         <p>
     *         For a list of valid values, query the <a>ListResourceTypes</a> operation.
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceShareArns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceShareArns() {
        return resourceShareArns != null && !(resourceShareArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies that you want to list information for only principals associated with the resource shares specified by
     * a list the <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource
     * Names (ARNs)</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceShareArns} method.
     * </p>
     * 
     * @return Specifies that you want to list information for only principals associated with the resource shares
     *         specified by a list the <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
     *         (ARNs)</a>.
     */
    public final List<String> resourceShareArns() {
        return resourceShareArns;
    }

    /**
     * <p>
     * Specifies that you want to receive the next page of results. Valid only if you received a <code>NextToken</code>
     * response in the previous request. If you did, it indicates that more output is available. Set this parameter to
     * the value provided by the previous call's <code>NextToken</code> response to request the next page of results.
     * </p>
     * 
     * @return Specifies that you want to receive the next page of results. Valid only if you received a
     *         <code>NextToken</code> response in the previous request. If you did, it indicates that more output is
     *         available. Set this parameter to the value provided by the previous call's <code>NextToken</code>
     *         response to request the next page of results.
     */
    public final String nextToken() {
        return nextToken;
    }

    /**
     * <p>
     * Specifies the total number of results that you want included on each page of the response. If you do not include
     * this parameter, it defaults to a value that is specific to the operation. If additional items exist beyond the
     * number you specify, the <code>NextToken</code> response element is returned with a value (not null). Include the
     * specified value as the <code>NextToken</code> request parameter in the next call to the operation to get the next
     * part of the results. Note that the service might return fewer results than the maximum even when there are more
     * results available. You should check <code>NextToken</code> after every operation to ensure that you receive all
     * of the results.
     * </p>
     * 
     * @return Specifies the total number of results that you want included on each page of the response. If you do not
     *         include this parameter, it defaults to a value that is specific to the operation. If additional items
     *         exist beyond the number you specify, the <code>NextToken</code> response element is returned with a value
     *         (not null). Include the specified value as the <code>NextToken</code> request parameter in the next call
     *         to the operation to get the next part of the results. Note that the service might return fewer results
     *         than the maximum even when there are more results available. You should check <code>NextToken</code>
     *         after every operation to ensure that you receive all of the results.
     */
    public final Integer maxResults() {
        return maxResults;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(resourceOwnerAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasPrincipals() ? principals() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceShareArns() ? resourceShareArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        hashCode = 31 * hashCode + Objects.hashCode(maxResults());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ListPrincipalsRequest)) {
            return false;
        }
        ListPrincipalsRequest other = (ListPrincipalsRequest) obj;
        return Objects.equals(resourceOwnerAsString(), other.resourceOwnerAsString())
                && Objects.equals(resourceArn(), other.resourceArn()) && hasPrincipals() == other.hasPrincipals()
                && Objects.equals(principals(), other.principals()) && Objects.equals(resourceType(), other.resourceType())
                && hasResourceShareArns() == other.hasResourceShareArns()
                && Objects.equals(resourceShareArns(), other.resourceShareArns())
                && Objects.equals(nextToken(), other.nextToken()) && Objects.equals(maxResults(), other.maxResults());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ListPrincipalsRequest").add("ResourceOwner", resourceOwnerAsString())
                .add("ResourceArn", resourceArn()).add("Principals", hasPrincipals() ? principals() : null)
                .add("ResourceType", resourceType())
                .add("ResourceShareArns", hasResourceShareArns() ? resourceShareArns() : null).add("NextToken", nextToken())
                .add("MaxResults", maxResults()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "resourceOwner":
            return Optional.ofNullable(clazz.cast(resourceOwnerAsString()));
        case "resourceArn":
            return Optional.ofNullable(clazz.cast(resourceArn()));
        case "principals":
            return Optional.ofNullable(clazz.cast(principals()));
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "resourceShareArns":
            return Optional.ofNullable(clazz.cast(resourceShareArns()));
        case "nextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        case "maxResults":
            return Optional.ofNullable(clazz.cast(maxResults()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ListPrincipalsRequest, T> g) {
        return obj -> g.apply((ListPrincipalsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends RamRequest.Builder, SdkPojo, CopyableBuilder<Builder, ListPrincipalsRequest> {
        /**
         * <p>
         * Specifies that you want to list information for only resource shares that match the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>SELF</code> </b> – principals that your account is sharing resources with
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>OTHER-ACCOUNTS</code> </b> – principals that are sharing resources with your account
         * </p>
         * </li>
         * </ul>
         * 
         * @param resourceOwner
         *        Specifies that you want to list information for only resource shares that match the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>SELF</code> </b> – principals that your account is sharing resources with
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>OTHER-ACCOUNTS</code> </b> – principals that are sharing resources with your account
         *        </p>
         *        </li>
         * @see ResourceOwner
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceOwner
         */
        Builder resourceOwner(String resourceOwner);

        /**
         * <p>
         * Specifies that you want to list information for only resource shares that match the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>SELF</code> </b> – principals that your account is sharing resources with
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>OTHER-ACCOUNTS</code> </b> – principals that are sharing resources with your account
         * </p>
         * </li>
         * </ul>
         * 
         * @param resourceOwner
         *        Specifies that you want to list information for only resource shares that match the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>SELF</code> </b> – principals that your account is sharing resources with
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>OTHER-ACCOUNTS</code> </b> – principals that are sharing resources with your account
         *        </p>
         *        </li>
         * @see ResourceOwner
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceOwner
         */
        Builder resourceOwner(ResourceOwner resourceOwner);

        /**
         * <p>
         * Specifies that you want to list principal information for the resource share with the specified <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resoure Name
         * (ARN)</a>.
         * </p>
         * 
         * @param resourceArn
         *        Specifies that you want to list principal information for the resource share with the specified <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resoure Name
         *        (ARN)</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArn(String resourceArn);

        /**
         * <p>
         * Specifies that you want to list information for only the listed principals.
         * </p>
         * <p>
         * You can include the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * An Amazon Web Services account ID, for example: <code>123456789012</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * An <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resoure Name
         * (ARN)</a> of an organization in Organizations, for example:
         * <code>organizations::123456789012:organization/o-exampleorgid</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * An ARN of an organizational unit (OU) in Organizations, for example:
         * <code>organizations::123456789012:ou/o-exampleorgid/ou-examplerootid-exampleouid123</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * An ARN of an IAM role, for example: <code>iam::123456789012:role/rolename</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * An ARN of an IAM user, for example: <code>iam::123456789012user/username</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * Not all resource types can be shared with IAM roles and users. For more information, see <a href=
         * "https://docs.aws.amazon.com/ram/latest/userguide/permissions.html#permissions-rbp-supported-resource-types"
         * >Sharing with IAM roles and users</a> in the <i>Resource Access Manager User Guide</i>.
         * </p>
         * </note>
         * 
         * @param principals
         *        Specifies that you want to list information for only the listed principals.</p>
         *        <p>
         *        You can include the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        An Amazon Web Services account ID, for example: <code>123456789012</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resoure
         *        Name (ARN)</a> of an organization in Organizations, for example:
         *        <code>organizations::123456789012:organization/o-exampleorgid</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An ARN of an organizational unit (OU) in Organizations, for example:
         *        <code>organizations::123456789012:ou/o-exampleorgid/ou-examplerootid-exampleouid123</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An ARN of an IAM role, for example: <code>iam::123456789012:role/rolename</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An ARN of an IAM user, for example: <code>iam::123456789012user/username</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        Not all resource types can be shared with IAM roles and users. For more information, see <a href=
         *        "https://docs.aws.amazon.com/ram/latest/userguide/permissions.html#permissions-rbp-supported-resource-types"
         *        >Sharing with IAM roles and users</a> in the <i>Resource Access Manager User Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder principals(Collection<String> principals);

        /**
         * <p>
         * Specifies that you want to list information for only the listed principals.
         * </p>
         * <p>
         * You can include the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * An Amazon Web Services account ID, for example: <code>123456789012</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * An <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resoure Name
         * (ARN)</a> of an organization in Organizations, for example:
         * <code>organizations::123456789012:organization/o-exampleorgid</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * An ARN of an organizational unit (OU) in Organizations, for example:
         * <code>organizations::123456789012:ou/o-exampleorgid/ou-examplerootid-exampleouid123</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * An ARN of an IAM role, for example: <code>iam::123456789012:role/rolename</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * An ARN of an IAM user, for example: <code>iam::123456789012user/username</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * Not all resource types can be shared with IAM roles and users. For more information, see <a href=
         * "https://docs.aws.amazon.com/ram/latest/userguide/permissions.html#permissions-rbp-supported-resource-types"
         * >Sharing with IAM roles and users</a> in the <i>Resource Access Manager User Guide</i>.
         * </p>
         * </note>
         * 
         * @param principals
         *        Specifies that you want to list information for only the listed principals.</p>
         *        <p>
         *        You can include the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        An Amazon Web Services account ID, for example: <code>123456789012</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resoure
         *        Name (ARN)</a> of an organization in Organizations, for example:
         *        <code>organizations::123456789012:organization/o-exampleorgid</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An ARN of an organizational unit (OU) in Organizations, for example:
         *        <code>organizations::123456789012:ou/o-exampleorgid/ou-examplerootid-exampleouid123</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An ARN of an IAM role, for example: <code>iam::123456789012:role/rolename</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An ARN of an IAM user, for example: <code>iam::123456789012user/username</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        Not all resource types can be shared with IAM roles and users. For more information, see <a href=
         *        "https://docs.aws.amazon.com/ram/latest/userguide/permissions.html#permissions-rbp-supported-resource-types"
         *        >Sharing with IAM roles and users</a> in the <i>Resource Access Manager User Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder principals(String... principals);

        /**
         * <p>
         * Specifies that you want to list information for only principals associated with resource shares that include
         * the specified resource type.
         * </p>
         * <p>
         * For a list of valid values, query the <a>ListResourceTypes</a> operation.
         * </p>
         * 
         * @param resourceType
         *        Specifies that you want to list information for only principals associated with resource shares that
         *        include the specified resource type.</p>
         *        <p>
         *        For a list of valid values, query the <a>ListResourceTypes</a> operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * Specifies that you want to list information for only principals associated with the resource shares specified
         * by a list the <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
         * Resource Names (ARNs)</a>.
         * </p>
         * 
         * @param resourceShareArns
         *        Specifies that you want to list information for only principals associated with the resource shares
         *        specified by a list the <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource
         *        Names (ARNs)</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceShareArns(Collection<String> resourceShareArns);

        /**
         * <p>
         * Specifies that you want to list information for only principals associated with the resource shares specified
         * by a list the <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
         * Resource Names (ARNs)</a>.
         * </p>
         * 
         * @param resourceShareArns
         *        Specifies that you want to list information for only principals associated with the resource shares
         *        specified by a list the <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource
         *        Names (ARNs)</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceShareArns(String... resourceShareArns);

        /**
         * <p>
         * Specifies that you want to receive the next page of results. Valid only if you received a
         * <code>NextToken</code> response in the previous request. If you did, it indicates that more output is
         * available. Set this parameter to the value provided by the previous call's <code>NextToken</code> response to
         * request the next page of results.
         * </p>
         * 
         * @param nextToken
         *        Specifies that you want to receive the next page of results. Valid only if you received a
         *        <code>NextToken</code> response in the previous request. If you did, it indicates that more output is
         *        available. Set this parameter to the value provided by the previous call's <code>NextToken</code>
         *        response to request the next page of results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);

        /**
         * <p>
         * Specifies the total number of results that you want included on each page of the response. If you do not
         * include this parameter, it defaults to a value that is specific to the operation. If additional items exist
         * beyond the number you specify, the <code>NextToken</code> response element is returned with a value (not
         * null). Include the specified value as the <code>NextToken</code> request parameter in the next call to the
         * operation to get the next part of the results. Note that the service might return fewer results than the
         * maximum even when there are more results available. You should check <code>NextToken</code> after every
         * operation to ensure that you receive all of the results.
         * </p>
         * 
         * @param maxResults
         *        Specifies the total number of results that you want included on each page of the response. If you do
         *        not include this parameter, it defaults to a value that is specific to the operation. If additional
         *        items exist beyond the number you specify, the <code>NextToken</code> response element is returned
         *        with a value (not null). Include the specified value as the <code>NextToken</code> request parameter
         *        in the next call to the operation to get the next part of the results. Note that the service might
         *        return fewer results than the maximum even when there are more results available. You should check
         *        <code>NextToken</code> after every operation to ensure that you receive all of the results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxResults(Integer maxResults);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends RamRequest.BuilderImpl implements Builder {
        private String resourceOwner;

        private String resourceArn;

        private List<String> principals = DefaultSdkAutoConstructList.getInstance();

        private String resourceType;

        private List<String> resourceShareArns = DefaultSdkAutoConstructList.getInstance();

        private String nextToken;

        private Integer maxResults;

        private BuilderImpl() {
        }

        private BuilderImpl(ListPrincipalsRequest model) {
            super(model);
            resourceOwner(model.resourceOwner);
            resourceArn(model.resourceArn);
            principals(model.principals);
            resourceType(model.resourceType);
            resourceShareArns(model.resourceShareArns);
            nextToken(model.nextToken);
            maxResults(model.maxResults);
        }

        public final String getResourceOwner() {
            return resourceOwner;
        }

        public final void setResourceOwner(String resourceOwner) {
            this.resourceOwner = resourceOwner;
        }

        @Override
        public final Builder resourceOwner(String resourceOwner) {
            this.resourceOwner = resourceOwner;
            return this;
        }

        @Override
        public final Builder resourceOwner(ResourceOwner resourceOwner) {
            this.resourceOwner(resourceOwner == null ? null : resourceOwner.toString());
            return this;
        }

        public final String getResourceArn() {
            return resourceArn;
        }

        public final void setResourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
        }

        @Override
        public final Builder resourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
            return this;
        }

        public final Collection<String> getPrincipals() {
            if (principals instanceof SdkAutoConstructList) {
                return null;
            }
            return principals;
        }

        public final void setPrincipals(Collection<String> principals) {
            this.principals = PrincipalArnOrIdListCopier.copy(principals);
        }

        @Override
        public final Builder principals(Collection<String> principals) {
            this.principals = PrincipalArnOrIdListCopier.copy(principals);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder principals(String... principals) {
            principals(Arrays.asList(principals));
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final Collection<String> getResourceShareArns() {
            if (resourceShareArns instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceShareArns;
        }

        public final void setResourceShareArns(Collection<String> resourceShareArns) {
            this.resourceShareArns = ResourceShareArnListCopier.copy(resourceShareArns);
        }

        @Override
        public final Builder resourceShareArns(Collection<String> resourceShareArns) {
            this.resourceShareArns = ResourceShareArnListCopier.copy(resourceShareArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceShareArns(String... resourceShareArns) {
            resourceShareArns(Arrays.asList(resourceShareArns));
            return this;
        }

        public final String getNextToken() {
            return nextToken;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        public final Integer getMaxResults() {
            return maxResults;
        }

        public final void setMaxResults(Integer maxResults) {
            this.maxResults = maxResults;
        }

        @Override
        public final Builder maxResults(Integer maxResults) {
            this.maxResults = maxResults;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ListPrincipalsRequest build() {
            return new ListPrincipalsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
