/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ram;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ram.model.AcceptResourceShareInvitationRequest;
import software.amazon.awssdk.services.ram.model.AcceptResourceShareInvitationResponse;
import software.amazon.awssdk.services.ram.model.AssociateResourceSharePermissionRequest;
import software.amazon.awssdk.services.ram.model.AssociateResourceSharePermissionResponse;
import software.amazon.awssdk.services.ram.model.AssociateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.AssociateResourceShareResponse;
import software.amazon.awssdk.services.ram.model.CreateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.CreateResourceShareResponse;
import software.amazon.awssdk.services.ram.model.DeleteResourceShareRequest;
import software.amazon.awssdk.services.ram.model.DeleteResourceShareResponse;
import software.amazon.awssdk.services.ram.model.DisassociateResourceSharePermissionRequest;
import software.amazon.awssdk.services.ram.model.DisassociateResourceSharePermissionResponse;
import software.amazon.awssdk.services.ram.model.DisassociateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.DisassociateResourceShareResponse;
import software.amazon.awssdk.services.ram.model.EnableSharingWithAwsOrganizationRequest;
import software.amazon.awssdk.services.ram.model.EnableSharingWithAwsOrganizationResponse;
import software.amazon.awssdk.services.ram.model.GetPermissionRequest;
import software.amazon.awssdk.services.ram.model.GetPermissionResponse;
import software.amazon.awssdk.services.ram.model.GetResourcePoliciesRequest;
import software.amazon.awssdk.services.ram.model.GetResourcePoliciesResponse;
import software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsRequest;
import software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsResponse;
import software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsRequest;
import software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsResponse;
import software.amazon.awssdk.services.ram.model.GetResourceSharesRequest;
import software.amazon.awssdk.services.ram.model.GetResourceSharesResponse;
import software.amazon.awssdk.services.ram.model.IdempotentParameterMismatchException;
import software.amazon.awssdk.services.ram.model.InvalidClientTokenException;
import software.amazon.awssdk.services.ram.model.InvalidMaxResultsException;
import software.amazon.awssdk.services.ram.model.InvalidNextTokenException;
import software.amazon.awssdk.services.ram.model.InvalidParameterException;
import software.amazon.awssdk.services.ram.model.InvalidResourceTypeException;
import software.amazon.awssdk.services.ram.model.InvalidStateTransitionException;
import software.amazon.awssdk.services.ram.model.ListPendingInvitationResourcesRequest;
import software.amazon.awssdk.services.ram.model.ListPendingInvitationResourcesResponse;
import software.amazon.awssdk.services.ram.model.ListPermissionVersionsRequest;
import software.amazon.awssdk.services.ram.model.ListPermissionVersionsResponse;
import software.amazon.awssdk.services.ram.model.ListPermissionsRequest;
import software.amazon.awssdk.services.ram.model.ListPermissionsResponse;
import software.amazon.awssdk.services.ram.model.ListPrincipalsRequest;
import software.amazon.awssdk.services.ram.model.ListPrincipalsResponse;
import software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsRequest;
import software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsResponse;
import software.amazon.awssdk.services.ram.model.ListResourceTypesRequest;
import software.amazon.awssdk.services.ram.model.ListResourceTypesResponse;
import software.amazon.awssdk.services.ram.model.ListResourcesRequest;
import software.amazon.awssdk.services.ram.model.ListResourcesResponse;
import software.amazon.awssdk.services.ram.model.MalformedArnException;
import software.amazon.awssdk.services.ram.model.MissingRequiredParameterException;
import software.amazon.awssdk.services.ram.model.OperationNotPermittedException;
import software.amazon.awssdk.services.ram.model.PromoteResourceShareCreatedFromPolicyRequest;
import software.amazon.awssdk.services.ram.model.PromoteResourceShareCreatedFromPolicyResponse;
import software.amazon.awssdk.services.ram.model.RamException;
import software.amazon.awssdk.services.ram.model.RamRequest;
import software.amazon.awssdk.services.ram.model.RejectResourceShareInvitationRequest;
import software.amazon.awssdk.services.ram.model.RejectResourceShareInvitationResponse;
import software.amazon.awssdk.services.ram.model.ResourceArnNotFoundException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationAlreadyAcceptedException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationAlreadyRejectedException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationArnNotFoundException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationExpiredException;
import software.amazon.awssdk.services.ram.model.ResourceShareLimitExceededException;
import software.amazon.awssdk.services.ram.model.ServerInternalException;
import software.amazon.awssdk.services.ram.model.ServiceUnavailableException;
import software.amazon.awssdk.services.ram.model.TagLimitExceededException;
import software.amazon.awssdk.services.ram.model.TagPolicyViolationException;
import software.amazon.awssdk.services.ram.model.TagResourceRequest;
import software.amazon.awssdk.services.ram.model.TagResourceResponse;
import software.amazon.awssdk.services.ram.model.ThrottlingException;
import software.amazon.awssdk.services.ram.model.UnknownResourceException;
import software.amazon.awssdk.services.ram.model.UntagResourceRequest;
import software.amazon.awssdk.services.ram.model.UntagResourceResponse;
import software.amazon.awssdk.services.ram.model.UpdateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.UpdateResourceShareResponse;
import software.amazon.awssdk.services.ram.paginators.GetResourcePoliciesPublisher;
import software.amazon.awssdk.services.ram.paginators.GetResourceShareAssociationsPublisher;
import software.amazon.awssdk.services.ram.paginators.GetResourceShareInvitationsPublisher;
import software.amazon.awssdk.services.ram.paginators.GetResourceSharesPublisher;
import software.amazon.awssdk.services.ram.paginators.ListPendingInvitationResourcesPublisher;
import software.amazon.awssdk.services.ram.paginators.ListPermissionVersionsPublisher;
import software.amazon.awssdk.services.ram.paginators.ListPermissionsPublisher;
import software.amazon.awssdk.services.ram.paginators.ListPrincipalsPublisher;
import software.amazon.awssdk.services.ram.paginators.ListResourceSharePermissionsPublisher;
import software.amazon.awssdk.services.ram.paginators.ListResourceTypesPublisher;
import software.amazon.awssdk.services.ram.paginators.ListResourcesPublisher;
import software.amazon.awssdk.services.ram.transform.AcceptResourceShareInvitationRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.AssociateResourceSharePermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.AssociateResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.CreateResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DeleteResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DisassociateResourceSharePermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DisassociateResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.EnableSharingWithAwsOrganizationRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetPermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourcePoliciesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourceShareAssociationsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourceShareInvitationsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourceSharesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPendingInvitationResourcesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPermissionVersionsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPermissionsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPrincipalsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListResourceSharePermissionsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListResourceTypesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListResourcesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.PromoteResourceShareCreatedFromPolicyRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.RejectResourceShareInvitationRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.UpdateResourceShareRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link RamAsyncClient}.
 *
 * @see RamAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRamAsyncClient implements RamAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRamAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRamAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Accepts an invitation to a resource share from another Amazon Web Services account. After you accept the
     * invitation, the resources included in the resource share are available to interact with in the relevant Amazon
     * Web Services Management Consoles and tools.
     * </p>
     *
     * @param acceptResourceShareInvitationRequest
     * @return A Java Future containing the result of the AcceptResourceShareInvitation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ResourceShareInvitationArnNotFoundException The specified Amazon Resource Name (ARN) for an
     *         invitation was not found.</li>
     *         <li>ResourceShareInvitationAlreadyAcceptedException The specified invitation was already accepted.</li>
     *         <li>ResourceShareInvitationAlreadyRejectedException The specified invitation was already rejected.</li>
     *         <li>ResourceShareInvitationExpiredException The specified invitation is expired.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>InvalidClientTokenException The client token is not valid.</li>
     *         <li>IdempotentParameterMismatchException The client token input parameter was matched one used with a
     *         previous call to the operation, but at least one of the other input parameters is different from the
     *         previous call.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.AcceptResourceShareInvitation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/AcceptResourceShareInvitation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptResourceShareInvitationResponse> acceptResourceShareInvitation(
            AcceptResourceShareInvitationRequest acceptResourceShareInvitationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                acceptResourceShareInvitationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptResourceShareInvitation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptResourceShareInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcceptResourceShareInvitationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptResourceShareInvitationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptResourceShareInvitationRequest, AcceptResourceShareInvitationResponse>()
                            .withOperationName("AcceptResourceShareInvitation")
                            .withMarshaller(new AcceptResourceShareInvitationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(acceptResourceShareInvitationRequest));
            CompletableFuture<AcceptResourceShareInvitationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds the specified list of principals and list of resources to a resource share. Principals that already have
     * access to this resource share immediately receive access to the added resources. Newly added principals
     * immediately receive access to the resources shared in this resource share.
     * </p>
     *
     * @param associateResourceShareRequest
     * @return A Java Future containing the result of the AssociateResourceShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>IdempotentParameterMismatchException The client token input parameter was matched one used with a
     *         previous call to the operation, but at least one of the other input parameters is different from the
     *         previous call.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>InvalidStateTransitionException The requested state transition is not valid.</li>
     *         <li>ResourceShareLimitExceededException This request would exceed the limit for resource shares for your
     *         account.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidStateTransitionException The requested state transition is not valid.</li>
     *         <li>InvalidClientTokenException The client token is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>ThrottlingException You exceeded the rate at which you are allowed to perform this operation. Please
     *         try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.AssociateResourceShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/AssociateResourceShare" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateResourceShareResponse> associateResourceShare(
            AssociateResourceShareRequest associateResourceShareRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResourceShare");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateResourceShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateResourceShareResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateResourceShareRequest, AssociateResourceShareResponse>()
                            .withOperationName("AssociateResourceShare")
                            .withMarshaller(new AssociateResourceShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateResourceShareRequest));
            CompletableFuture<AssociateResourceShareResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or replaces the RAM permission for a resource type included in a resource share. You can have exactly one
     * permission associated with each resource type in the resource share. You can add a new RAM permission only if
     * there are currently no resources of that resource type currently in the resource share.
     * </p>
     *
     * @param associateResourceSharePermissionRequest
     * @return A Java Future containing the result of the AssociateResourceSharePermission operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>InvalidClientTokenException The client token is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.AssociateResourceSharePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/AssociateResourceSharePermission"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateResourceSharePermissionResponse> associateResourceSharePermission(
            AssociateResourceSharePermissionRequest associateResourceSharePermissionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateResourceSharePermissionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResourceSharePermission");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateResourceSharePermissionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AssociateResourceSharePermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateResourceSharePermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateResourceSharePermissionRequest, AssociateResourceSharePermissionResponse>()
                            .withOperationName("AssociateResourceSharePermission")
                            .withMarshaller(new AssociateResourceSharePermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateResourceSharePermissionRequest));
            CompletableFuture<AssociateResourceSharePermissionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a resource share. You can provide a list of the <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
     * (ARNs)</a> for the resources that you want to share, a list of principals you want to share the resources with,
     * and the permissions to grant those principals.
     * </p>
     * <note>
     * <p>
     * Sharing a resource makes it available for use by principals outside of the Amazon Web Services account that
     * created the resource. Sharing doesn't change any permissions or quotas that apply to the resource in the account
     * that created it.
     * </p>
     * </note>
     *
     * @param createResourceShareRequest
     * @return A Java Future containing the result of the CreateResourceShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>IdempotentParameterMismatchException The client token input parameter was matched one used with a
     *         previous call to the operation, but at least one of the other input parameters is different from the
     *         previous call.</li>
     *         <li>InvalidStateTransitionException The requested state transition is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidClientTokenException The client token is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ResourceShareLimitExceededException This request would exceed the limit for resource shares for your
     *         account.</li>
     *         <li>TagPolicyViolationException The specified tag key is a reserved word and can't be used.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.CreateResourceShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/CreateResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateResourceShareResponse> createResourceShare(
            CreateResourceShareRequest createResourceShareRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResourceShare");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateResourceShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateResourceShareResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateResourceShareRequest, CreateResourceShareResponse>()
                            .withOperationName("CreateResourceShare")
                            .withMarshaller(new CreateResourceShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createResourceShareRequest));
            CompletableFuture<CreateResourceShareResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified resource share. This doesn't delete any of the resources that were associated with the
     * resource share; it only stops the sharing of those resources outside of the Amazon Web Services account that
     * created them.
     * </p>
     *
     * @param deleteResourceShareRequest
     * @return A Java Future containing the result of the DeleteResourceShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>IdempotentParameterMismatchException The client token input parameter was matched one used with a
     *         previous call to the operation, but at least one of the other input parameters is different from the
     *         previous call.</li>
     *         <li>InvalidStateTransitionException The requested state transition is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidClientTokenException The client token is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.DeleteResourceShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DeleteResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResourceShareResponse> deleteResourceShare(
            DeleteResourceShareRequest deleteResourceShareRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourceShare");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResourceShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResourceShareResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResourceShareRequest, DeleteResourceShareResponse>()
                            .withOperationName("DeleteResourceShare")
                            .withMarshaller(new DeleteResourceShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteResourceShareRequest));
            CompletableFuture<DeleteResourceShareResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates the specified principals or resources from the specified resource share.
     * </p>
     *
     * @param disassociateResourceShareRequest
     * @return A Java Future containing the result of the DisassociateResourceShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>IdempotentParameterMismatchException The client token input parameter was matched one used with a
     *         previous call to the operation, but at least one of the other input parameters is different from the
     *         previous call.</li>
     *         <li>ResourceShareLimitExceededException This request would exceed the limit for resource shares for your
     *         account.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidStateTransitionException The requested state transition is not valid.</li>
     *         <li>InvalidClientTokenException The client token is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.DisassociateResourceShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DisassociateResourceShare" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateResourceShareResponse> disassociateResourceShare(
            DisassociateResourceShareRequest disassociateResourceShareRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResourceShare");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateResourceShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateResourceShareResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateResourceShareRequest, DisassociateResourceShareResponse>()
                            .withOperationName("DisassociateResourceShare")
                            .withMarshaller(new DisassociateResourceShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateResourceShareRequest));
            CompletableFuture<DisassociateResourceShareResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates an RAM permission from a resource share. Permission changes take effect immediately. You can remove
     * a RAM permission from a resource share only if there are currently no resources of the relevant resource type
     * currently attached to the resource share.
     * </p>
     *
     * @param disassociateResourceSharePermissionRequest
     * @return A Java Future containing the result of the DisassociateResourceSharePermission operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>InvalidClientTokenException The client token is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>InvalidStateTransitionException The requested state transition is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.DisassociateResourceSharePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DisassociateResourceSharePermission"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateResourceSharePermissionResponse> disassociateResourceSharePermission(
            DisassociateResourceSharePermissionRequest disassociateResourceSharePermissionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateResourceSharePermissionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResourceSharePermission");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateResourceSharePermissionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateResourceSharePermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateResourceSharePermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateResourceSharePermissionRequest, DisassociateResourceSharePermissionResponse>()
                            .withOperationName("DisassociateResourceSharePermission")
                            .withMarshaller(new DisassociateResourceSharePermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateResourceSharePermissionRequest));
            CompletableFuture<DisassociateResourceSharePermissionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables resource sharing within your organization in Organizations. Calling this operation enables RAM to
     * retrieve information about the organization and its structure. This lets you share resources with all of the
     * accounts in an organization by specifying the organization's ID, or all of the accounts in an organizational unit
     * (OU) by specifying the OU's ID. Until you enable sharing within the organization, you can specify only individual
     * Amazon Web Services accounts, or for supported resource types, IAM users and roles.
     * </p>
     * <p>
     * You must call this operation from an IAM user or role in the organization's management account.
     * </p>
     *
     * @param enableSharingWithAwsOrganizationRequest
     * @return A Java Future containing the result of the EnableSharingWithAwsOrganization operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.EnableSharingWithAwsOrganization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/EnableSharingWithAwsOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableSharingWithAwsOrganizationResponse> enableSharingWithAwsOrganization(
            EnableSharingWithAwsOrganizationRequest enableSharingWithAwsOrganizationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                enableSharingWithAwsOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableSharingWithAwsOrganization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableSharingWithAwsOrganizationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, EnableSharingWithAwsOrganizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableSharingWithAwsOrganizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableSharingWithAwsOrganizationRequest, EnableSharingWithAwsOrganizationResponse>()
                            .withOperationName("EnableSharingWithAwsOrganization")
                            .withMarshaller(new EnableSharingWithAwsOrganizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(enableSharingWithAwsOrganizationRequest));
            CompletableFuture<EnableSharingWithAwsOrganizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the contents of an RAM permission in JSON format.
     * </p>
     *
     * @param getPermissionRequest
     * @return A Java Future containing the result of the GetPermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetPermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetPermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPermissionResponse> getPermission(GetPermissionRequest getPermissionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPermission");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPermissionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetPermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPermissionRequest, GetPermissionResponse>()
                            .withOperationName("GetPermission")
                            .withMarshaller(new GetPermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPermissionRequest));
            CompletableFuture<GetPermissionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the resource policies for the specified resources that you own and have shared.
     * </p>
     *
     * @param getResourcePoliciesRequest
     * @return A Java Future containing the result of the GetResourcePolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ResourceArnNotFoundException The specified Amazon Resource Name (ARN) was not found.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourcePolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourcePolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourcePoliciesResponse> getResourcePolicies(
            GetResourcePoliciesRequest getResourcePoliciesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourcePoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourcePoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourcePoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourcePoliciesRequest, GetResourcePoliciesResponse>()
                            .withOperationName("GetResourcePolicies")
                            .withMarshaller(new GetResourcePoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResourcePoliciesRequest));
            CompletableFuture<GetResourcePoliciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the resource policies for the specified resources that you own and have shared.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getResourcePolicies(software.amazon.awssdk.services.ram.model.GetResourcePoliciesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourcePoliciesPublisher publisher = client.getResourcePoliciesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourcePoliciesPublisher publisher = client.getResourcePoliciesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.GetResourcePoliciesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.GetResourcePoliciesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourcePolicies(software.amazon.awssdk.services.ram.model.GetResourcePoliciesRequest)} operation.</b>
     * </p>
     *
     * @param getResourcePoliciesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ResourceArnNotFoundException The specified Amazon Resource Name (ARN) was not found.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourcePolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourcePolicies" target="_top">AWS API
     *      Documentation</a>
     */
    public GetResourcePoliciesPublisher getResourcePoliciesPaginator(GetResourcePoliciesRequest getResourcePoliciesRequest) {
        return new GetResourcePoliciesPublisher(this, applyPaginatorUserAgent(getResourcePoliciesRequest));
    }

    /**
     * <p>
     * Retrieves the resource and principal associations for resource shares that you own.
     * </p>
     *
     * @param getResourceShareAssociationsRequest
     * @return A Java Future containing the result of the GetResourceShareAssociations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourceShareAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceShareAssociationsResponse> getResourceShareAssociations(
            GetResourceShareAssociationsRequest getResourceShareAssociationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceShareAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceShareAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceShareAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceShareAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourceShareAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceShareAssociationsRequest, GetResourceShareAssociationsResponse>()
                            .withOperationName("GetResourceShareAssociations")
                            .withMarshaller(new GetResourceShareAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResourceShareAssociationsRequest));
            CompletableFuture<GetResourceShareAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the resource and principal associations for resource shares that you own.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getResourceShareAssociations(software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceShareAssociationsPublisher publisher = client.getResourceShareAssociationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceShareAssociationsPublisher publisher = client.getResourceShareAssociationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourceShareAssociations(software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getResourceShareAssociationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourceShareAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    public GetResourceShareAssociationsPublisher getResourceShareAssociationsPaginator(
            GetResourceShareAssociationsRequest getResourceShareAssociationsRequest) {
        return new GetResourceShareAssociationsPublisher(this, applyPaginatorUserAgent(getResourceShareAssociationsRequest));
    }

    /**
     * <p>
     * Retrieves details about invitations that you have received for resource shares.
     * </p>
     *
     * @param getResourceShareInvitationsRequest
     * @return A Java Future containing the result of the GetResourceShareInvitations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceShareInvitationArnNotFoundException The specified Amazon Resource Name (ARN) for an
     *         invitation was not found.</li>
     *         <li>InvalidMaxResultsException The specified value for <code>MaxResults</code> is not valid.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourceShareInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceShareInvitationsResponse> getResourceShareInvitations(
            GetResourceShareInvitationsRequest getResourceShareInvitationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceShareInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceShareInvitations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceShareInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceShareInvitationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourceShareInvitationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceShareInvitationsRequest, GetResourceShareInvitationsResponse>()
                            .withOperationName("GetResourceShareInvitations")
                            .withMarshaller(new GetResourceShareInvitationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResourceShareInvitationsRequest));
            CompletableFuture<GetResourceShareInvitationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about invitations that you have received for resource shares.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getResourceShareInvitations(software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceShareInvitationsPublisher publisher = client.getResourceShareInvitationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceShareInvitationsPublisher publisher = client.getResourceShareInvitationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourceShareInvitations(software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getResourceShareInvitationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceShareInvitationArnNotFoundException The specified Amazon Resource Name (ARN) for an
     *         invitation was not found.</li>
     *         <li>InvalidMaxResultsException The specified value for <code>MaxResults</code> is not valid.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourceShareInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    public GetResourceShareInvitationsPublisher getResourceShareInvitationsPaginator(
            GetResourceShareInvitationsRequest getResourceShareInvitationsRequest) {
        return new GetResourceShareInvitationsPublisher(this, applyPaginatorUserAgent(getResourceShareInvitationsRequest));
    }

    /**
     * <p>
     * Retrieves details about the resource shares that you own or that are shared with you.
     * </p>
     *
     * @param getResourceSharesRequest
     * @return A Java Future containing the result of the GetResourceShares operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourceShares
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShares" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceSharesResponse> getResourceShares(GetResourceSharesRequest getResourceSharesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceSharesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceShares");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceSharesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceSharesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourceSharesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceSharesRequest, GetResourceSharesResponse>()
                            .withOperationName("GetResourceShares")
                            .withMarshaller(new GetResourceSharesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResourceSharesRequest));
            CompletableFuture<GetResourceSharesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about the resource shares that you own or that are shared with you.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getResourceShares(software.amazon.awssdk.services.ram.model.GetResourceSharesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceSharesPublisher publisher = client.getResourceSharesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.GetResourceSharesPublisher publisher = client.getResourceSharesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.GetResourceSharesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.GetResourceSharesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourceShares(software.amazon.awssdk.services.ram.model.GetResourceSharesRequest)} operation.</b>
     * </p>
     *
     * @param getResourceSharesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourceShares
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShares" target="_top">AWS API
     *      Documentation</a>
     */
    public GetResourceSharesPublisher getResourceSharesPaginator(GetResourceSharesRequest getResourceSharesRequest) {
        return new GetResourceSharesPublisher(this, applyPaginatorUserAgent(getResourceSharesRequest));
    }

    /**
     * <p>
     * Lists the resources in a resource share that is shared with you but for which the invitation is still
     * <code>PENDING</code>. That means that you haven't accepted or rejected the invitation and the invitation hasn't
     * expired.
     * </p>
     *
     * @param listPendingInvitationResourcesRequest
     * @return A Java Future containing the result of the ListPendingInvitationResources operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>ResourceShareInvitationArnNotFoundException The specified Amazon Resource Name (ARN) for an
     *         invitation was not found.</li>
     *         <li>MissingRequiredParameterException A required input parameter is missing.</li>
     *         <li>ResourceShareInvitationAlreadyRejectedException The specified invitation was already rejected.</li>
     *         <li>ResourceShareInvitationExpiredException The specified invitation is expired.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListPendingInvitationResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPendingInvitationResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPendingInvitationResourcesResponse> listPendingInvitationResources(
            ListPendingInvitationResourcesRequest listPendingInvitationResourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listPendingInvitationResourcesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPendingInvitationResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPendingInvitationResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPendingInvitationResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPendingInvitationResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPendingInvitationResourcesRequest, ListPendingInvitationResourcesResponse>()
                            .withOperationName("ListPendingInvitationResources")
                            .withMarshaller(new ListPendingInvitationResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPendingInvitationResourcesRequest));
            CompletableFuture<ListPendingInvitationResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resources in a resource share that is shared with you but for which the invitation is still
     * <code>PENDING</code>. That means that you haven't accepted or rejected the invitation and the invitation hasn't
     * expired.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPendingInvitationResources(software.amazon.awssdk.services.ram.model.ListPendingInvitationResourcesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListPendingInvitationResourcesPublisher publisher = client.listPendingInvitationResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListPendingInvitationResourcesPublisher publisher = client.listPendingInvitationResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.ListPendingInvitationResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.ListPendingInvitationResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPendingInvitationResources(software.amazon.awssdk.services.ram.model.ListPendingInvitationResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPendingInvitationResourcesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>ResourceShareInvitationArnNotFoundException The specified Amazon Resource Name (ARN) for an
     *         invitation was not found.</li>
     *         <li>MissingRequiredParameterException A required input parameter is missing.</li>
     *         <li>ResourceShareInvitationAlreadyRejectedException The specified invitation was already rejected.</li>
     *         <li>ResourceShareInvitationExpiredException The specified invitation is expired.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListPendingInvitationResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPendingInvitationResources"
     *      target="_top">AWS API Documentation</a>
     */
    public ListPendingInvitationResourcesPublisher listPendingInvitationResourcesPaginator(
            ListPendingInvitationResourcesRequest listPendingInvitationResourcesRequest) {
        return new ListPendingInvitationResourcesPublisher(this, applyPaginatorUserAgent(listPendingInvitationResourcesRequest));
    }

    /**
     * <p>
     * Lists the available versions of the specified RAM permission.
     * </p>
     *
     * @param listPermissionVersionsRequest
     * @return A Java Future containing the result of the ListPermissionVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListPermissionVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPermissionVersions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPermissionVersionsResponse> listPermissionVersions(
            ListPermissionVersionsRequest listPermissionVersionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPermissionVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPermissionVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPermissionVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPermissionVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPermissionVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPermissionVersionsRequest, ListPermissionVersionsResponse>()
                            .withOperationName("ListPermissionVersions")
                            .withMarshaller(new ListPermissionVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPermissionVersionsRequest));
            CompletableFuture<ListPermissionVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the available versions of the specified RAM permission.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPermissionVersions(software.amazon.awssdk.services.ram.model.ListPermissionVersionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListPermissionVersionsPublisher publisher = client.listPermissionVersionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListPermissionVersionsPublisher publisher = client.listPermissionVersionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.ListPermissionVersionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.ListPermissionVersionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPermissionVersions(software.amazon.awssdk.services.ram.model.ListPermissionVersionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPermissionVersionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListPermissionVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPermissionVersions" target="_top">AWS
     *      API Documentation</a>
     */
    public ListPermissionVersionsPublisher listPermissionVersionsPaginator(
            ListPermissionVersionsRequest listPermissionVersionsRequest) {
        return new ListPermissionVersionsPublisher(this, applyPaginatorUserAgent(listPermissionVersionsRequest));
    }

    /**
     * <p>
     * Retrieves a list of available RAM permissions that you can use for the supported resource types.
     * </p>
     *
     * @param listPermissionsRequest
     * @return A Java Future containing the result of the ListPermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPermissions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPermissionsResponse> listPermissions(ListPermissionsRequest listPermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPermissionsRequest, ListPermissionsResponse>()
                            .withOperationName("ListPermissions")
                            .withMarshaller(new ListPermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPermissionsRequest));
            CompletableFuture<ListPermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of available RAM permissions that you can use for the supported resource types.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listPermissions(software.amazon.awssdk.services.ram.model.ListPermissionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListPermissionsPublisher publisher = client.listPermissionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListPermissionsPublisher publisher = client.listPermissionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.ListPermissionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.ListPermissionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPermissions(software.amazon.awssdk.services.ram.model.ListPermissionsRequest)} operation.</b>
     * </p>
     *
     * @param listPermissionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPermissions" target="_top">AWS API
     *      Documentation</a>
     */
    public ListPermissionsPublisher listPermissionsPaginator(ListPermissionsRequest listPermissionsRequest) {
        return new ListPermissionsPublisher(this, applyPaginatorUserAgent(listPermissionsRequest));
    }

    /**
     * <p>
     * Lists the principals that you are sharing resources with or that are sharing resources with you.
     * </p>
     *
     * @param listPrincipalsRequest
     * @return A Java Future containing the result of the ListPrincipals operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListPrincipals
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPrincipals" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPrincipalsResponse> listPrincipals(ListPrincipalsRequest listPrincipalsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPrincipalsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPrincipals");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPrincipalsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPrincipalsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPrincipalsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPrincipalsRequest, ListPrincipalsResponse>()
                            .withOperationName("ListPrincipals")
                            .withMarshaller(new ListPrincipalsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPrincipalsRequest));
            CompletableFuture<ListPrincipalsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the principals that you are sharing resources with or that are sharing resources with you.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listPrincipals(software.amazon.awssdk.services.ram.model.ListPrincipalsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListPrincipalsPublisher publisher = client.listPrincipalsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListPrincipalsPublisher publisher = client.listPrincipalsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.ListPrincipalsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.ListPrincipalsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPrincipals(software.amazon.awssdk.services.ram.model.ListPrincipalsRequest)} operation.</b>
     * </p>
     *
     * @param listPrincipalsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListPrincipals
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPrincipals" target="_top">AWS API
     *      Documentation</a>
     */
    public ListPrincipalsPublisher listPrincipalsPaginator(ListPrincipalsRequest listPrincipalsRequest) {
        return new ListPrincipalsPublisher(this, applyPaginatorUserAgent(listPrincipalsRequest));
    }

    /**
     * <p>
     * Lists the RAM permissions that are associated with a resource share.
     * </p>
     *
     * @param listResourceSharePermissionsRequest
     * @return A Java Future containing the result of the ListResourceSharePermissions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListResourceSharePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResourceSharePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourceSharePermissionsResponse> listResourceSharePermissions(
            ListResourceSharePermissionsRequest listResourceSharePermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceSharePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceSharePermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourceSharePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResourceSharePermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourceSharePermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourceSharePermissionsRequest, ListResourceSharePermissionsResponse>()
                            .withOperationName("ListResourceSharePermissions")
                            .withMarshaller(new ListResourceSharePermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResourceSharePermissionsRequest));
            CompletableFuture<ListResourceSharePermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the RAM permissions that are associated with a resource share.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResourceSharePermissions(software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListResourceSharePermissionsPublisher publisher = client.listResourceSharePermissionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListResourceSharePermissionsPublisher publisher = client.listResourceSharePermissionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResourceSharePermissions(software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResourceSharePermissionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListResourceSharePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResourceSharePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    public ListResourceSharePermissionsPublisher listResourceSharePermissionsPaginator(
            ListResourceSharePermissionsRequest listResourceSharePermissionsRequest) {
        return new ListResourceSharePermissionsPublisher(this, applyPaginatorUserAgent(listResourceSharePermissionsRequest));
    }

    /**
     * <p>
     * Lists the resource types that can be shared by RAM.
     * </p>
     *
     * @param listResourceTypesRequest
     * @return A Java Future containing the result of the ListResourceTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListResourceTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResourceTypes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourceTypesResponse> listResourceTypes(ListResourceTypesRequest listResourceTypesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceTypes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourceTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResourceTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourceTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourceTypesRequest, ListResourceTypesResponse>()
                            .withOperationName("ListResourceTypes")
                            .withMarshaller(new ListResourceTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResourceTypesRequest));
            CompletableFuture<ListResourceTypesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resource types that can be shared by RAM.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResourceTypes(software.amazon.awssdk.services.ram.model.ListResourceTypesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListResourceTypesPublisher publisher = client.listResourceTypesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListResourceTypesPublisher publisher = client.listResourceTypesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.ListResourceTypesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.ListResourceTypesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResourceTypes(software.amazon.awssdk.services.ram.model.ListResourceTypesRequest)} operation.</b>
     * </p>
     *
     * @param listResourceTypesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListResourceTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResourceTypes" target="_top">AWS API
     *      Documentation</a>
     */
    public ListResourceTypesPublisher listResourceTypesPaginator(ListResourceTypesRequest listResourceTypesRequest) {
        return new ListResourceTypesPublisher(this, applyPaginatorUserAgent(listResourceTypesRequest));
    }

    /**
     * <p>
     * Lists the resources that you added to a resource share or the resources that are shared with you.
     * </p>
     *
     * @param listResourcesRequest
     * @return A Java Future containing the result of the ListResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceTypeException The specified resource type is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourcesResponse> listResources(ListResourcesRequest listResourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourcesRequest, ListResourcesResponse>()
                            .withOperationName("ListResources")
                            .withMarshaller(new ListResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResourcesRequest));
            CompletableFuture<ListResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resources that you added to a resource share or the resources that are shared with you.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listResources(software.amazon.awssdk.services.ram.model.ListResourcesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListResourcesPublisher publisher = client.listResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ram.paginators.ListResourcesPublisher publisher = client.listResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ram.model.ListResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ram.model.ListResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResources(software.amazon.awssdk.services.ram.model.ListResourcesRequest)} operation.</b>
     * </p>
     *
     * @param listResourcesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceTypeException The specified resource type is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNextTokenException The specified value for <code>NextToken</code> is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResources" target="_top">AWS API
     *      Documentation</a>
     */
    public ListResourcesPublisher listResourcesPaginator(ListResourcesRequest listResourcesRequest) {
        return new ListResourcesPublisher(this, applyPaginatorUserAgent(listResourcesRequest));
    }

    /**
     * <p>
     * When you attach a resource-based permission policy to a resource, it automatically creates a resource share.
     * However, resource shares created this way are visible only to the resource share owner, and the resource share
     * can't be modified in RAM.
     * </p>
     * <p>
     * You can use this operation to promote the resource share to a full RAM resource share. When you promote a
     * resource share, you can then manage the resource share in RAM and it becomes visible to all of the principals you
     * shared it with.
     * </p>
     *
     * @param promoteResourceShareCreatedFromPolicyRequest
     * @return A Java Future containing the result of the PromoteResourceShareCreatedFromPolicy operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>ResourceShareLimitExceededException This request would exceed the limit for resource shares for your
     *         account.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required input parameter is missing.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.PromoteResourceShareCreatedFromPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/PromoteResourceShareCreatedFromPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PromoteResourceShareCreatedFromPolicyResponse> promoteResourceShareCreatedFromPolicy(
            PromoteResourceShareCreatedFromPolicyRequest promoteResourceShareCreatedFromPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                promoteResourceShareCreatedFromPolicyRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PromoteResourceShareCreatedFromPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PromoteResourceShareCreatedFromPolicyResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PromoteResourceShareCreatedFromPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PromoteResourceShareCreatedFromPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PromoteResourceShareCreatedFromPolicyRequest, PromoteResourceShareCreatedFromPolicyResponse>()
                            .withOperationName("PromoteResourceShareCreatedFromPolicy")
                            .withMarshaller(new PromoteResourceShareCreatedFromPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(promoteResourceShareCreatedFromPolicyRequest));
            CompletableFuture<PromoteResourceShareCreatedFromPolicyResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Rejects an invitation to a resource share from another Amazon Web Services account.
     * </p>
     *
     * @param rejectResourceShareInvitationRequest
     * @return A Java Future containing the result of the RejectResourceShareInvitation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ResourceShareInvitationArnNotFoundException The specified Amazon Resource Name (ARN) for an
     *         invitation was not found.</li>
     *         <li>ResourceShareInvitationAlreadyAcceptedException The specified invitation was already accepted.</li>
     *         <li>ResourceShareInvitationAlreadyRejectedException The specified invitation was already rejected.</li>
     *         <li>ResourceShareInvitationExpiredException The specified invitation is expired.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>InvalidClientTokenException The client token is not valid.</li>
     *         <li>IdempotentParameterMismatchException The client token input parameter was matched one used with a
     *         previous call to the operation, but at least one of the other input parameters is different from the
     *         previous call.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.RejectResourceShareInvitation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/RejectResourceShareInvitation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RejectResourceShareInvitationResponse> rejectResourceShareInvitation(
            RejectResourceShareInvitationRequest rejectResourceShareInvitationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                rejectResourceShareInvitationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectResourceShareInvitation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RejectResourceShareInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RejectResourceShareInvitationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RejectResourceShareInvitationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RejectResourceShareInvitationRequest, RejectResourceShareInvitationResponse>()
                            .withOperationName("RejectResourceShareInvitation")
                            .withMarshaller(new RejectResourceShareInvitationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(rejectResourceShareInvitationRequest));
            CompletableFuture<RejectResourceShareInvitationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds the specified tag keys and values to the specified resource share. The tags are attached only to the
     * resource share, not to the resources that are in the resource share.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>TagLimitExceededException This request would exceed the limit for tags for your account.</li>
     *         <li>ResourceArnNotFoundException The specified Amazon Resource Name (ARN) was not found.</li>
     *         <li>TagPolicyViolationException The specified tag key is a reserved word and can't be used.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tag key and value pairs from the specified resource share.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies some of the properties of the specified resource share.
     * </p>
     *
     * @param updateResourceShareRequest
     * @return A Java Future containing the result of the UpdateResourceShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>IdempotentParameterMismatchException The client token input parameter was matched one used with a
     *         previous call to the operation, but at least one of the other input parameters is different from the
     *         previous call.</li>
     *         <li>MissingRequiredParameterException A required input parameter is missing.</li>
     *         <li>UnknownResourceException A specified resource was not found.</li>
     *         <li>MalformedArnException The format of an Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidClientTokenException The client token is not valid.</li>
     *         <li>InvalidParameterException A parameter is not valid.</li>
     *         <li>OperationNotPermittedException The requested operation is not permitted.</li>
     *         <li>ServerInternalException The service could not respond to the request due to an internal problem.</li>
     *         <li>ServiceUnavailableException The service is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.UpdateResourceShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/UpdateResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateResourceShareResponse> updateResourceShare(
            UpdateResourceShareRequest updateResourceShareRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResourceShare");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateResourceShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateResourceShareResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateResourceShareRequest, UpdateResourceShareResponse>()
                            .withOperationName("UpdateResourceShare")
                            .withMarshaller(new UpdateResourceShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateResourceShareRequest));
            CompletableFuture<UpdateResourceShareResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RamException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidClientTokenException")
                                .exceptionBuilderSupplier(InvalidClientTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnknownResourceException")
                                .exceptionBuilderSupplier(UnknownResourceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerInternalException")
                                .exceptionBuilderSupplier(ServerInternalException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                                .exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidMaxResultsException")
                                .exceptionBuilderSupplier(InvalidMaxResultsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceTypeException")
                                .exceptionBuilderSupplier(InvalidResourceTypeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationExpiredException")
                                .exceptionBuilderSupplier(ResourceShareInvitationExpiredException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingRequiredParameterException")
                                .exceptionBuilderSupplier(MissingRequiredParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagLimitExceededException")
                                .exceptionBuilderSupplier(TagLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceArnNotFoundException")
                                .exceptionBuilderSupplier(ResourceArnNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationNotPermittedException")
                                .exceptionBuilderSupplier(OperationNotPermittedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareLimitExceededException")
                                .exceptionBuilderSupplier(ResourceShareLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationAlreadyRejectedException")
                                .exceptionBuilderSupplier(ResourceShareInvitationAlreadyRejectedException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MalformedArnException")
                                .exceptionBuilderSupplier(MalformedArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStateTransitionException")
                                .exceptionBuilderSupplier(InvalidStateTransitionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationAlreadyAcceptedException")
                                .exceptionBuilderSupplier(ResourceShareInvitationAlreadyAcceptedException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationArnNotFoundException")
                                .exceptionBuilderSupplier(ResourceShareInvitationArnNotFoundException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagPolicyViolationException")
                                .exceptionBuilderSupplier(TagPolicyViolationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends RamRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
