/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ram;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ram.internal.RamServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ram.model.AcceptResourceShareInvitationRequest;
import software.amazon.awssdk.services.ram.model.AcceptResourceShareInvitationResponse;
import software.amazon.awssdk.services.ram.model.AssociateResourceSharePermissionRequest;
import software.amazon.awssdk.services.ram.model.AssociateResourceSharePermissionResponse;
import software.amazon.awssdk.services.ram.model.AssociateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.AssociateResourceShareResponse;
import software.amazon.awssdk.services.ram.model.CreatePermissionRequest;
import software.amazon.awssdk.services.ram.model.CreatePermissionResponse;
import software.amazon.awssdk.services.ram.model.CreatePermissionVersionRequest;
import software.amazon.awssdk.services.ram.model.CreatePermissionVersionResponse;
import software.amazon.awssdk.services.ram.model.CreateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.CreateResourceShareResponse;
import software.amazon.awssdk.services.ram.model.DeletePermissionRequest;
import software.amazon.awssdk.services.ram.model.DeletePermissionResponse;
import software.amazon.awssdk.services.ram.model.DeletePermissionVersionRequest;
import software.amazon.awssdk.services.ram.model.DeletePermissionVersionResponse;
import software.amazon.awssdk.services.ram.model.DeleteResourceShareRequest;
import software.amazon.awssdk.services.ram.model.DeleteResourceShareResponse;
import software.amazon.awssdk.services.ram.model.DisassociateResourceSharePermissionRequest;
import software.amazon.awssdk.services.ram.model.DisassociateResourceSharePermissionResponse;
import software.amazon.awssdk.services.ram.model.DisassociateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.DisassociateResourceShareResponse;
import software.amazon.awssdk.services.ram.model.EnableSharingWithAwsOrganizationRequest;
import software.amazon.awssdk.services.ram.model.EnableSharingWithAwsOrganizationResponse;
import software.amazon.awssdk.services.ram.model.GetPermissionRequest;
import software.amazon.awssdk.services.ram.model.GetPermissionResponse;
import software.amazon.awssdk.services.ram.model.GetResourcePoliciesRequest;
import software.amazon.awssdk.services.ram.model.GetResourcePoliciesResponse;
import software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsRequest;
import software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsResponse;
import software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsRequest;
import software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsResponse;
import software.amazon.awssdk.services.ram.model.GetResourceSharesRequest;
import software.amazon.awssdk.services.ram.model.GetResourceSharesResponse;
import software.amazon.awssdk.services.ram.model.IdempotentParameterMismatchException;
import software.amazon.awssdk.services.ram.model.InvalidClientTokenException;
import software.amazon.awssdk.services.ram.model.InvalidMaxResultsException;
import software.amazon.awssdk.services.ram.model.InvalidNextTokenException;
import software.amazon.awssdk.services.ram.model.InvalidParameterException;
import software.amazon.awssdk.services.ram.model.InvalidPolicyException;
import software.amazon.awssdk.services.ram.model.InvalidResourceTypeException;
import software.amazon.awssdk.services.ram.model.InvalidStateTransitionException;
import software.amazon.awssdk.services.ram.model.ListPendingInvitationResourcesRequest;
import software.amazon.awssdk.services.ram.model.ListPendingInvitationResourcesResponse;
import software.amazon.awssdk.services.ram.model.ListPermissionAssociationsRequest;
import software.amazon.awssdk.services.ram.model.ListPermissionAssociationsResponse;
import software.amazon.awssdk.services.ram.model.ListPermissionVersionsRequest;
import software.amazon.awssdk.services.ram.model.ListPermissionVersionsResponse;
import software.amazon.awssdk.services.ram.model.ListPermissionsRequest;
import software.amazon.awssdk.services.ram.model.ListPermissionsResponse;
import software.amazon.awssdk.services.ram.model.ListPrincipalsRequest;
import software.amazon.awssdk.services.ram.model.ListPrincipalsResponse;
import software.amazon.awssdk.services.ram.model.ListReplacePermissionAssociationsWorkRequest;
import software.amazon.awssdk.services.ram.model.ListReplacePermissionAssociationsWorkResponse;
import software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsRequest;
import software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsResponse;
import software.amazon.awssdk.services.ram.model.ListResourceTypesRequest;
import software.amazon.awssdk.services.ram.model.ListResourceTypesResponse;
import software.amazon.awssdk.services.ram.model.ListResourcesRequest;
import software.amazon.awssdk.services.ram.model.ListResourcesResponse;
import software.amazon.awssdk.services.ram.model.MalformedArnException;
import software.amazon.awssdk.services.ram.model.MalformedPolicyTemplateException;
import software.amazon.awssdk.services.ram.model.MissingRequiredParameterException;
import software.amazon.awssdk.services.ram.model.OperationNotPermittedException;
import software.amazon.awssdk.services.ram.model.PermissionAlreadyExistsException;
import software.amazon.awssdk.services.ram.model.PermissionLimitExceededException;
import software.amazon.awssdk.services.ram.model.PermissionVersionsLimitExceededException;
import software.amazon.awssdk.services.ram.model.PromotePermissionCreatedFromPolicyRequest;
import software.amazon.awssdk.services.ram.model.PromotePermissionCreatedFromPolicyResponse;
import software.amazon.awssdk.services.ram.model.PromoteResourceShareCreatedFromPolicyRequest;
import software.amazon.awssdk.services.ram.model.PromoteResourceShareCreatedFromPolicyResponse;
import software.amazon.awssdk.services.ram.model.RamException;
import software.amazon.awssdk.services.ram.model.RejectResourceShareInvitationRequest;
import software.amazon.awssdk.services.ram.model.RejectResourceShareInvitationResponse;
import software.amazon.awssdk.services.ram.model.ReplacePermissionAssociationsRequest;
import software.amazon.awssdk.services.ram.model.ReplacePermissionAssociationsResponse;
import software.amazon.awssdk.services.ram.model.ResourceArnNotFoundException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationAlreadyAcceptedException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationAlreadyRejectedException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationArnNotFoundException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationExpiredException;
import software.amazon.awssdk.services.ram.model.ResourceShareLimitExceededException;
import software.amazon.awssdk.services.ram.model.ServerInternalException;
import software.amazon.awssdk.services.ram.model.ServiceUnavailableException;
import software.amazon.awssdk.services.ram.model.SetDefaultPermissionVersionRequest;
import software.amazon.awssdk.services.ram.model.SetDefaultPermissionVersionResponse;
import software.amazon.awssdk.services.ram.model.TagLimitExceededException;
import software.amazon.awssdk.services.ram.model.TagPolicyViolationException;
import software.amazon.awssdk.services.ram.model.TagResourceRequest;
import software.amazon.awssdk.services.ram.model.TagResourceResponse;
import software.amazon.awssdk.services.ram.model.ThrottlingException;
import software.amazon.awssdk.services.ram.model.UnknownResourceException;
import software.amazon.awssdk.services.ram.model.UnmatchedPolicyPermissionException;
import software.amazon.awssdk.services.ram.model.UntagResourceRequest;
import software.amazon.awssdk.services.ram.model.UntagResourceResponse;
import software.amazon.awssdk.services.ram.model.UpdateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.UpdateResourceShareResponse;
import software.amazon.awssdk.services.ram.transform.AcceptResourceShareInvitationRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.AssociateResourceSharePermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.AssociateResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.CreatePermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.CreatePermissionVersionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.CreateResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DeletePermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DeletePermissionVersionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DeleteResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DisassociateResourceSharePermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DisassociateResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.EnableSharingWithAwsOrganizationRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetPermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourcePoliciesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourceShareAssociationsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourceShareInvitationsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourceSharesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPendingInvitationResourcesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPermissionAssociationsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPermissionVersionsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPermissionsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPrincipalsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListReplacePermissionAssociationsWorkRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListResourceSharePermissionsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListResourceTypesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListResourcesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.PromotePermissionCreatedFromPolicyRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.PromoteResourceShareCreatedFromPolicyRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.RejectResourceShareInvitationRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ReplacePermissionAssociationsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.SetDefaultPermissionVersionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.UpdateResourceShareRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link RamClient}.
 *
 * @see RamClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRamClient implements RamClient {
    private static final Logger log = Logger.loggerFor(DefaultRamClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRamClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Accepts an invitation to a resource share from another Amazon Web Services account. After you accept the
     * invitation, the resources included in the resource share are available to interact with in the relevant Amazon
     * Web Services Management Consoles and tools.
     * </p>
     *
     * @param acceptResourceShareInvitationRequest
     * @return Result of the AcceptResourceShareInvitation operation returned by the service.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws ResourceShareInvitationArnNotFoundException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> for an invitation was not found.
     * @throws ResourceShareInvitationAlreadyAcceptedException
     *         The operation failed because the specified invitation was already accepted.
     * @throws ResourceShareInvitationAlreadyRejectedException
     *         The operation failed because the specified invitation was already rejected.
     * @throws ResourceShareInvitationExpiredException
     *         The operation failed because the specified invitation is past its expiration date and time.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws InvalidClientTokenException
     *         The operation failed because the specified client token isn't valid.
     * @throws IdempotentParameterMismatchException
     *         The operation failed because the client token input parameter matched one that was used with a previous
     *         call to the operation, but at least one of the other input parameters is different from the previous
     *         call.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.AcceptResourceShareInvitation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/AcceptResourceShareInvitation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AcceptResourceShareInvitationResponse acceptResourceShareInvitation(
            AcceptResourceShareInvitationRequest acceptResourceShareInvitationRequest) throws MalformedArnException,
            OperationNotPermittedException, ResourceShareInvitationArnNotFoundException,
            ResourceShareInvitationAlreadyAcceptedException, ResourceShareInvitationAlreadyRejectedException,
            ResourceShareInvitationExpiredException, ServerInternalException, ServiceUnavailableException,
            InvalidClientTokenException, IdempotentParameterMismatchException, AwsServiceException, SdkClientException,
            RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AcceptResourceShareInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AcceptResourceShareInvitationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(acceptResourceShareInvitationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                acceptResourceShareInvitationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptResourceShareInvitation");

            return clientHandler
                    .execute(new ClientExecutionParams<AcceptResourceShareInvitationRequest, AcceptResourceShareInvitationResponse>()
                            .withOperationName("AcceptResourceShareInvitation").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(acceptResourceShareInvitationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AcceptResourceShareInvitationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds the specified list of principals and list of resources to a resource share. Principals that already have
     * access to this resource share immediately receive access to the added resources. Newly added principals
     * immediately receive access to the resources shared in this resource share.
     * </p>
     *
     * @param associateResourceShareRequest
     * @return Result of the AssociateResourceShare operation returned by the service.
     * @throws IdempotentParameterMismatchException
     *         The operation failed because the client token input parameter matched one that was used with a previous
     *         call to the operation, but at least one of the other input parameters is different from the previous
     *         call.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws InvalidStateTransitionException
     *         The operation failed because the requested operation isn't valid for the resource share in its current
     *         state.
     * @throws ResourceShareLimitExceededException
     *         The operation failed because it would exceed the limit for resource shares for your account. To view the
     *         limits for your Amazon Web Services account, see the <a
     *         href="https://console.aws.amazon.com/servicequotas/home/services/ram/quotas">RAM page in the Service
     *         Quotas console</a>.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws InvalidStateTransitionException
     *         The operation failed because the requested operation isn't valid for the resource share in its current
     *         state.
     * @throws InvalidClientTokenException
     *         The operation failed because the specified client token isn't valid.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws ThrottlingException
     *         The operation failed because it exceeded the rate at which you are allowed to perform this operation.
     *         Please try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.AssociateResourceShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/AssociateResourceShare" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AssociateResourceShareResponse associateResourceShare(AssociateResourceShareRequest associateResourceShareRequest)
            throws IdempotentParameterMismatchException, UnknownResourceException, InvalidStateTransitionException,
            ResourceShareLimitExceededException, MalformedArnException, InvalidClientTokenException, InvalidParameterException,
            OperationNotPermittedException, ServerInternalException, ServiceUnavailableException, ThrottlingException,
            AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateResourceShareResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateResourceShareRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResourceShare");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateResourceShareRequest, AssociateResourceShareResponse>()
                            .withOperationName("AssociateResourceShare").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(associateResourceShareRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateResourceShareRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or replaces the RAM permission for a resource type included in a resource share. You can have exactly one
     * permission associated with each resource type in the resource share. You can add a new RAM permission only if
     * there are currently no resources of that resource type currently in the resource share.
     * </p>
     *
     * @param associateResourceSharePermissionRequest
     * @return Result of the AssociateResourceSharePermission operation returned by the service.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws InvalidClientTokenException
     *         The operation failed because the specified client token isn't valid.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.AssociateResourceSharePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/AssociateResourceSharePermission"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateResourceSharePermissionResponse associateResourceSharePermission(
            AssociateResourceSharePermissionRequest associateResourceSharePermissionRequest) throws MalformedArnException,
            UnknownResourceException, InvalidParameterException, InvalidClientTokenException, ServerInternalException,
            ServiceUnavailableException, OperationNotPermittedException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateResourceSharePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateResourceSharePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateResourceSharePermissionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateResourceSharePermissionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResourceSharePermission");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateResourceSharePermissionRequest, AssociateResourceSharePermissionResponse>()
                            .withOperationName("AssociateResourceSharePermission").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(associateResourceSharePermissionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateResourceSharePermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a customer managed permission for a specified resource type that you can attach to resource shares. It is
     * created in the Amazon Web Services Region in which you call the operation.
     * </p>
     *
     * @param createPermissionRequest
     * @return Result of the CreatePermission operation returned by the service.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws InvalidPolicyException
     *         The operation failed because a policy you specified isn't valid.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws PermissionAlreadyExistsException
     *         The operation failed because a permission with the specified name already exists in the requested Amazon
     *         Web Services Region. Choose a different name.
     * @throws MalformedPolicyTemplateException
     *         The operation failed because the policy template that you provided isn't valid.
     * @throws InvalidClientTokenException
     *         The operation failed because the specified client token isn't valid.
     * @throws PermissionLimitExceededException
     *         The operation failed because it would exceed the maximum number of permissions you can create in each
     *         Amazon Web Services Region. To view the limits for your Amazon Web Services account, see the <a
     *         href="https://console.aws.amazon.com/servicequotas/home/services/ram/quotas">RAM page in the Service
     *         Quotas console</a>.
     * @throws IdempotentParameterMismatchException
     *         The operation failed because the client token input parameter matched one that was used with a previous
     *         call to the operation, but at least one of the other input parameters is different from the previous
     *         call.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.CreatePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/CreatePermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreatePermissionResponse createPermission(CreatePermissionRequest createPermissionRequest)
            throws InvalidParameterException, InvalidPolicyException, OperationNotPermittedException, ServerInternalException,
            ServiceUnavailableException, PermissionAlreadyExistsException, MalformedPolicyTemplateException,
            InvalidClientTokenException, PermissionLimitExceededException, IdempotentParameterMismatchException,
            AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePermissionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreatePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPermissionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePermission");

            return clientHandler.execute(new ClientExecutionParams<CreatePermissionRequest, CreatePermissionResponse>()
                    .withOperationName("CreatePermission").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createPermissionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreatePermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new version of the specified customer managed permission. The new version is automatically set as the
     * default version of the customer managed permission. New resource shares automatically use the default permission.
     * Existing resource shares continue to use their original permission versions, but you can use
     * <a>ReplacePermissionAssociations</a> to update them.
     * </p>
     * <p>
     * If the specified customer managed permission already has the maximum of 5 versions, then you must delete one of
     * the existing versions before you can create a new one.
     * </p>
     *
     * @param createPermissionVersionRequest
     * @return Result of the CreatePermissionVersion operation returned by the service.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws InvalidPolicyException
     *         The operation failed because a policy you specified isn't valid.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws MalformedPolicyTemplateException
     *         The operation failed because the policy template that you provided isn't valid.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws InvalidClientTokenException
     *         The operation failed because the specified client token isn't valid.
     * @throws IdempotentParameterMismatchException
     *         The operation failed because the client token input parameter matched one that was used with a previous
     *         call to the operation, but at least one of the other input parameters is different from the previous
     *         call.
     * @throws PermissionVersionsLimitExceededException
     *         The operation failed because it would exceed the limit for the number of versions you can have for a
     *         permission. To view the limits for your Amazon Web Services account, see the <a
     *         href="https://console.aws.amazon.com/servicequotas/home/services/ram/quotas">RAM page in the Service
     *         Quotas console</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.CreatePermissionVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/CreatePermissionVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreatePermissionVersionResponse createPermissionVersion(CreatePermissionVersionRequest createPermissionVersionRequest)
            throws InvalidParameterException, InvalidPolicyException, ServerInternalException, ServiceUnavailableException,
            UnknownResourceException, MalformedPolicyTemplateException, MalformedArnException, InvalidClientTokenException,
            IdempotentParameterMismatchException, PermissionVersionsLimitExceededException, AwsServiceException,
            SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePermissionVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreatePermissionVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPermissionVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPermissionVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePermissionVersion");

            return clientHandler
                    .execute(new ClientExecutionParams<CreatePermissionVersionRequest, CreatePermissionVersionResponse>()
                            .withOperationName("CreatePermissionVersion").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createPermissionVersionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreatePermissionVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a resource share. You can provide a list of the <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
     * (ARNs)</a> for the resources that you want to share, a list of principals you want to share the resources with,
     * and the permissions to grant those principals.
     * </p>
     * <note>
     * <p>
     * Sharing a resource makes it available for use by principals outside of the Amazon Web Services account that
     * created the resource. Sharing doesn't change any permissions or quotas that apply to the resource in the account
     * that created it.
     * </p>
     * </note>
     *
     * @param createResourceShareRequest
     * @return Result of the CreateResourceShare operation returned by the service.
     * @throws IdempotentParameterMismatchException
     *         The operation failed because the client token input parameter matched one that was used with a previous
     *         call to the operation, but at least one of the other input parameters is different from the previous
     *         call.
     * @throws InvalidStateTransitionException
     *         The operation failed because the requested operation isn't valid for the resource share in its current
     *         state.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws InvalidClientTokenException
     *         The operation failed because the specified client token isn't valid.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws ResourceShareLimitExceededException
     *         The operation failed because it would exceed the limit for resource shares for your account. To view the
     *         limits for your Amazon Web Services account, see the <a
     *         href="https://console.aws.amazon.com/servicequotas/home/services/ram/quotas">RAM page in the Service
     *         Quotas console</a>.
     * @throws TagPolicyViolationException
     *         The operation failed because the specified tag key is a reserved word and can't be used.
     * @throws TagLimitExceededException
     *         The operation failed because it would exceed the limit for tags for your Amazon Web Services account.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.CreateResourceShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/CreateResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateResourceShareResponse createResourceShare(CreateResourceShareRequest createResourceShareRequest)
            throws IdempotentParameterMismatchException, InvalidStateTransitionException, UnknownResourceException,
            MalformedArnException, InvalidClientTokenException, InvalidParameterException, OperationNotPermittedException,
            ResourceShareLimitExceededException, TagPolicyViolationException, TagLimitExceededException, ServerInternalException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateResourceShareResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createResourceShareRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResourceShare");

            return clientHandler.execute(new ClientExecutionParams<CreateResourceShareRequest, CreateResourceShareResponse>()
                    .withOperationName("CreateResourceShare").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createResourceShareRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateResourceShareRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified customer managed permission in the Amazon Web Services Region in which you call this
     * operation. You can delete a customer managed permission only if it isn't attached to any resource share. The
     * operation deletes all versions associated with the customer managed permission.
     * </p>
     *
     * @param deletePermissionRequest
     * @return Result of the DeletePermission operation returned by the service.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws InvalidClientTokenException
     *         The operation failed because the specified client token isn't valid.
     * @throws IdempotentParameterMismatchException
     *         The operation failed because the client token input parameter matched one that was used with a previous
     *         call to the operation, but at least one of the other input parameters is different from the previous
     *         call.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.DeletePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DeletePermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeletePermissionResponse deletePermission(DeletePermissionRequest deletePermissionRequest)
            throws MalformedArnException, ServerInternalException, ServiceUnavailableException, OperationNotPermittedException,
            UnknownResourceException, InvalidClientTokenException, IdempotentParameterMismatchException, AwsServiceException,
            SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePermissionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeletePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePermissionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePermission");

            return clientHandler.execute(new ClientExecutionParams<DeletePermissionRequest, DeletePermissionResponse>()
                    .withOperationName("DeletePermission").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deletePermissionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes one version of a customer managed permission. The version you specify must not be attached to any
     * resource share and must not be the default version for the permission.
     * </p>
     * <p>
     * If a customer managed permission has the maximum of 5 versions, then you must delete at least one version before
     * you can create another.
     * </p>
     *
     * @param deletePermissionVersionRequest
     * @return Result of the DeletePermissionVersion operation returned by the service.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws InvalidClientTokenException
     *         The operation failed because the specified client token isn't valid.
     * @throws IdempotentParameterMismatchException
     *         The operation failed because the client token input parameter matched one that was used with a previous
     *         call to the operation, but at least one of the other input parameters is different from the previous
     *         call.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.DeletePermissionVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DeletePermissionVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeletePermissionVersionResponse deletePermissionVersion(DeletePermissionVersionRequest deletePermissionVersionRequest)
            throws MalformedArnException, InvalidParameterException, ServerInternalException, ServiceUnavailableException,
            OperationNotPermittedException, UnknownResourceException, InvalidClientTokenException,
            IdempotentParameterMismatchException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePermissionVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeletePermissionVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePermissionVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePermissionVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePermissionVersion");

            return clientHandler
                    .execute(new ClientExecutionParams<DeletePermissionVersionRequest, DeletePermissionVersionResponse>()
                            .withOperationName("DeletePermissionVersion").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deletePermissionVersionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeletePermissionVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified resource share.
     * </p>
     * <important>
     * <p>
     * This doesn't delete any of the resources that were associated with the resource share; it only stops the sharing
     * of those resources through this resource share.
     * </p>
     * </important>
     *
     * @param deleteResourceShareRequest
     * @return Result of the DeleteResourceShare operation returned by the service.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws IdempotentParameterMismatchException
     *         The operation failed because the client token input parameter matched one that was used with a previous
     *         call to the operation, but at least one of the other input parameters is different from the previous
     *         call.
     * @throws InvalidStateTransitionException
     *         The operation failed because the requested operation isn't valid for the resource share in its current
     *         state.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws InvalidClientTokenException
     *         The operation failed because the specified client token isn't valid.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.DeleteResourceShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DeleteResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteResourceShareResponse deleteResourceShare(DeleteResourceShareRequest deleteResourceShareRequest)
            throws OperationNotPermittedException, IdempotentParameterMismatchException, InvalidStateTransitionException,
            UnknownResourceException, MalformedArnException, InvalidClientTokenException, InvalidParameterException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResourceShareResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourceShareRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourceShare");

            return clientHandler.execute(new ClientExecutionParams<DeleteResourceShareRequest, DeleteResourceShareResponse>()
                    .withOperationName("DeleteResourceShare").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteResourceShareRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteResourceShareRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified principals or resources from participating in the specified resource share.
     * </p>
     *
     * @param disassociateResourceShareRequest
     * @return Result of the DisassociateResourceShare operation returned by the service.
     * @throws IdempotentParameterMismatchException
     *         The operation failed because the client token input parameter matched one that was used with a previous
     *         call to the operation, but at least one of the other input parameters is different from the previous
     *         call.
     * @throws ResourceShareLimitExceededException
     *         The operation failed because it would exceed the limit for resource shares for your account. To view the
     *         limits for your Amazon Web Services account, see the <a
     *         href="https://console.aws.amazon.com/servicequotas/home/services/ram/quotas">RAM page in the Service
     *         Quotas console</a>.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws InvalidStateTransitionException
     *         The operation failed because the requested operation isn't valid for the resource share in its current
     *         state.
     * @throws InvalidClientTokenException
     *         The operation failed because the specified client token isn't valid.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.DisassociateResourceShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DisassociateResourceShare" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DisassociateResourceShareResponse disassociateResourceShare(
            DisassociateResourceShareRequest disassociateResourceShareRequest) throws IdempotentParameterMismatchException,
            ResourceShareLimitExceededException, MalformedArnException, InvalidStateTransitionException,
            InvalidClientTokenException, InvalidParameterException, OperationNotPermittedException, ServerInternalException,
            ServiceUnavailableException, UnknownResourceException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateResourceShareResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateResourceShareRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResourceShare");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateResourceShareRequest, DisassociateResourceShareResponse>()
                            .withOperationName("DisassociateResourceShare").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(disassociateResourceShareRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateResourceShareRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a managed permission from a resource share. Permission changes take effect immediately. You can remove a
     * managed permission from a resource share only if there are currently no resources of the relevant resource type
     * currently attached to the resource share.
     * </p>
     *
     * @param disassociateResourceSharePermissionRequest
     * @return Result of the DisassociateResourceSharePermission operation returned by the service.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws InvalidClientTokenException
     *         The operation failed because the specified client token isn't valid.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws InvalidStateTransitionException
     *         The operation failed because the requested operation isn't valid for the resource share in its current
     *         state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.DisassociateResourceSharePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DisassociateResourceSharePermission"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateResourceSharePermissionResponse disassociateResourceSharePermission(
            DisassociateResourceSharePermissionRequest disassociateResourceSharePermissionRequest) throws MalformedArnException,
            UnknownResourceException, InvalidParameterException, InvalidClientTokenException, ServerInternalException,
            ServiceUnavailableException, OperationNotPermittedException, InvalidStateTransitionException, AwsServiceException,
            SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateResourceSharePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateResourceSharePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateResourceSharePermissionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateResourceSharePermissionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResourceSharePermission");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateResourceSharePermissionRequest, DisassociateResourceSharePermissionResponse>()
                            .withOperationName("DisassociateResourceSharePermission").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(disassociateResourceSharePermissionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateResourceSharePermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables resource sharing within your organization in Organizations. This operation creates a service-linked role
     * called <code>AWSServiceRoleForResourceAccessManager</code> that has the IAM managed policy named
     * AWSResourceAccessManagerServiceRolePolicy attached. This role permits RAM to retrieve information about the
     * organization and its structure. This lets you share resources with all of the accounts in the calling account's
     * organization by specifying the organization ID, or all of the accounts in an organizational unit (OU) by
     * specifying the OU ID. Until you enable sharing within the organization, you can specify only individual Amazon
     * Web Services accounts, or for supported resource types, IAM roles and users.
     * </p>
     * <p>
     * You must call this operation from an IAM role or user in the organization's management account.
     * </p>
     * <p/>
     *
     * @param enableSharingWithAwsOrganizationRequest
     * @return Result of the EnableSharingWithAwsOrganization operation returned by the service.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.EnableSharingWithAwsOrganization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/EnableSharingWithAwsOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EnableSharingWithAwsOrganizationResponse enableSharingWithAwsOrganization(
            EnableSharingWithAwsOrganizationRequest enableSharingWithAwsOrganizationRequest)
            throws OperationNotPermittedException, ServerInternalException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableSharingWithAwsOrganizationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, EnableSharingWithAwsOrganizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableSharingWithAwsOrganizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                enableSharingWithAwsOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableSharingWithAwsOrganization");

            return clientHandler
                    .execute(new ClientExecutionParams<EnableSharingWithAwsOrganizationRequest, EnableSharingWithAwsOrganizationResponse>()
                            .withOperationName("EnableSharingWithAwsOrganization").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(enableSharingWithAwsOrganizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new EnableSharingWithAwsOrganizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the contents of a managed permission in JSON format.
     * </p>
     *
     * @param getPermissionRequest
     * @return Result of the GetPermission operation returned by the service.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.GetPermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetPermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPermissionResponse getPermission(GetPermissionRequest getPermissionRequest) throws InvalidParameterException,
            MalformedArnException, UnknownResourceException, ServerInternalException, ServiceUnavailableException,
            OperationNotPermittedException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPermissionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetPermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPermissionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPermission");

            return clientHandler.execute(new ClientExecutionParams<GetPermissionRequest, GetPermissionResponse>()
                    .withOperationName("GetPermission").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getPermissionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetPermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the resource policies for the specified resources that you own and have shared.
     * </p>
     *
     * @param getResourcePoliciesRequest
     * @return Result of the GetResourcePolicies operation returned by the service.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws InvalidNextTokenException
     *         The operation failed because the specified value for <code>NextToken</code> isn't valid. You must specify
     *         a value you received in the <code>NextToken</code> response of a previous call to this operation.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws ResourceArnNotFoundException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> was not found.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.GetResourcePolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourcePolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetResourcePoliciesResponse getResourcePolicies(GetResourcePoliciesRequest getResourcePoliciesRequest)
            throws MalformedArnException, InvalidNextTokenException, InvalidParameterException, ResourceArnNotFoundException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourcePoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResourcePoliciesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcePoliciesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicies");

            return clientHandler.execute(new ClientExecutionParams<GetResourcePoliciesRequest, GetResourcePoliciesResponse>()
                    .withOperationName("GetResourcePolicies").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getResourcePoliciesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourcePoliciesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the lists of resources and principals that associated for resource shares that you own.
     * </p>
     *
     * @param getResourceShareAssociationsRequest
     * @return Result of the GetResourceShareAssociations operation returned by the service.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws InvalidNextTokenException
     *         The operation failed because the specified value for <code>NextToken</code> isn't valid. You must specify
     *         a value you received in the <code>NextToken</code> response of a previous call to this operation.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.GetResourceShareAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourceShareAssociationsResponse getResourceShareAssociations(
            GetResourceShareAssociationsRequest getResourceShareAssociationsRequest) throws UnknownResourceException,
            MalformedArnException, InvalidNextTokenException, InvalidParameterException, OperationNotPermittedException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceShareAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResourceShareAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceShareAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceShareAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceShareAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<GetResourceShareAssociationsRequest, GetResourceShareAssociationsResponse>()
                            .withOperationName("GetResourceShareAssociations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getResourceShareAssociationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetResourceShareAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves details about invitations that you have received for resource shares.
     * </p>
     *
     * @param getResourceShareInvitationsRequest
     * @return Result of the GetResourceShareInvitations operation returned by the service.
     * @throws ResourceShareInvitationArnNotFoundException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> for an invitation was not found.
     * @throws InvalidMaxResultsException
     *         The operation failed because the specified value for <code>MaxResults</code> isn't valid.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws InvalidNextTokenException
     *         The operation failed because the specified value for <code>NextToken</code> isn't valid. You must specify
     *         a value you received in the <code>NextToken</code> response of a previous call to this operation.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.GetResourceShareInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourceShareInvitationsResponse getResourceShareInvitations(
            GetResourceShareInvitationsRequest getResourceShareInvitationsRequest)
            throws ResourceShareInvitationArnNotFoundException, InvalidMaxResultsException, MalformedArnException,
            UnknownResourceException, InvalidNextTokenException, InvalidParameterException, ServerInternalException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceShareInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResourceShareInvitationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceShareInvitationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceShareInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceShareInvitations");

            return clientHandler
                    .execute(new ClientExecutionParams<GetResourceShareInvitationsRequest, GetResourceShareInvitationsResponse>()
                            .withOperationName("GetResourceShareInvitations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getResourceShareInvitationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetResourceShareInvitationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves details about the resource shares that you own or that are shared with you.
     * </p>
     *
     * @param getResourceSharesRequest
     * @return Result of the GetResourceShares operation returned by the service.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws InvalidNextTokenException
     *         The operation failed because the specified value for <code>NextToken</code> isn't valid. You must specify
     *         a value you received in the <code>NextToken</code> response of a previous call to this operation.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.GetResourceShares
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShares" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetResourceSharesResponse getResourceShares(GetResourceSharesRequest getResourceSharesRequest)
            throws UnknownResourceException, MalformedArnException, InvalidNextTokenException, InvalidParameterException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceSharesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourceSharesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceSharesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceSharesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceShares");

            return clientHandler.execute(new ClientExecutionParams<GetResourceSharesRequest, GetResourceSharesResponse>()
                    .withOperationName("GetResourceShares").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getResourceSharesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourceSharesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the resources in a resource share that is shared with you but for which the invitation is still
     * <code>PENDING</code>. That means that you haven't accepted or rejected the invitation and the invitation hasn't
     * expired.
     * </p>
     *
     * @param listPendingInvitationResourcesRequest
     * @return Result of the ListPendingInvitationResources operation returned by the service.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws InvalidNextTokenException
     *         The operation failed because the specified value for <code>NextToken</code> isn't valid. You must specify
     *         a value you received in the <code>NextToken</code> response of a previous call to this operation.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws ResourceShareInvitationArnNotFoundException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> for an invitation was not found.
     * @throws MissingRequiredParameterException
     *         The operation failed because a required input parameter is missing.
     * @throws ResourceShareInvitationAlreadyRejectedException
     *         The operation failed because the specified invitation was already rejected.
     * @throws ResourceShareInvitationExpiredException
     *         The operation failed because the specified invitation is past its expiration date and time.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListPendingInvitationResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPendingInvitationResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPendingInvitationResourcesResponse listPendingInvitationResources(
            ListPendingInvitationResourcesRequest listPendingInvitationResourcesRequest) throws MalformedArnException,
            InvalidNextTokenException, InvalidParameterException, ServerInternalException, ServiceUnavailableException,
            ResourceShareInvitationArnNotFoundException, MissingRequiredParameterException,
            ResourceShareInvitationAlreadyRejectedException, ResourceShareInvitationExpiredException, AwsServiceException,
            SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPendingInvitationResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPendingInvitationResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPendingInvitationResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listPendingInvitationResourcesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPendingInvitationResources");

            return clientHandler
                    .execute(new ClientExecutionParams<ListPendingInvitationResourcesRequest, ListPendingInvitationResourcesResponse>()
                            .withOperationName("ListPendingInvitationResources").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listPendingInvitationResourcesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListPendingInvitationResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists information about the managed permission and its associations to any resource shares that use this managed
     * permission. This lets you see which resource shares use which versions of the specified managed permission.
     * </p>
     *
     * @param listPermissionAssociationsRequest
     * @return Result of the ListPermissionAssociations operation returned by the service.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws InvalidNextTokenException
     *         The operation failed because the specified value for <code>NextToken</code> isn't valid. You must specify
     *         a value you received in the <code>NextToken</code> response of a previous call to this operation.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListPermissionAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPermissionAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPermissionAssociationsResponse listPermissionAssociations(
            ListPermissionAssociationsRequest listPermissionAssociationsRequest) throws InvalidParameterException,
            MalformedArnException, InvalidNextTokenException, ServerInternalException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPermissionAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPermissionAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPermissionAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPermissionAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPermissionAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListPermissionAssociationsRequest, ListPermissionAssociationsResponse>()
                            .withOperationName("ListPermissionAssociations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listPermissionAssociationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListPermissionAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the available versions of the specified RAM permission.
     * </p>
     *
     * @param listPermissionVersionsRequest
     * @return Result of the ListPermissionVersions operation returned by the service.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws InvalidNextTokenException
     *         The operation failed because the specified value for <code>NextToken</code> isn't valid. You must specify
     *         a value you received in the <code>NextToken</code> response of a previous call to this operation.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListPermissionVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPermissionVersions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListPermissionVersionsResponse listPermissionVersions(ListPermissionVersionsRequest listPermissionVersionsRequest)
            throws MalformedArnException, UnknownResourceException, InvalidNextTokenException, ServerInternalException,
            ServiceUnavailableException, OperationNotPermittedException, InvalidParameterException, AwsServiceException,
            SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPermissionVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPermissionVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPermissionVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPermissionVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPermissionVersions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListPermissionVersionsRequest, ListPermissionVersionsResponse>()
                            .withOperationName("ListPermissionVersions").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listPermissionVersionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListPermissionVersionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of available RAM permissions that you can use for the supported resource types.
     * </p>
     *
     * @param listPermissionsRequest
     * @return Result of the ListPermissions operation returned by the service.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws InvalidNextTokenException
     *         The operation failed because the specified value for <code>NextToken</code> isn't valid. You must specify
     *         a value you received in the <code>NextToken</code> response of a previous call to this operation.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPermissions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPermissionsResponse listPermissions(ListPermissionsRequest listPermissionsRequest)
            throws InvalidParameterException, InvalidNextTokenException, ServerInternalException, ServiceUnavailableException,
            OperationNotPermittedException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPermissionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPermissionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPermissions");

            return clientHandler.execute(new ClientExecutionParams<ListPermissionsRequest, ListPermissionsResponse>()
                    .withOperationName("ListPermissions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPermissionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPermissionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the principals that you are sharing resources with or that are sharing resources with you.
     * </p>
     *
     * @param listPrincipalsRequest
     * @return Result of the ListPrincipals operation returned by the service.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws InvalidNextTokenException
     *         The operation failed because the specified value for <code>NextToken</code> isn't valid. You must specify
     *         a value you received in the <code>NextToken</code> response of a previous call to this operation.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListPrincipals
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPrincipals" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPrincipalsResponse listPrincipals(ListPrincipalsRequest listPrincipalsRequest) throws MalformedArnException,
            UnknownResourceException, InvalidNextTokenException, InvalidParameterException, ServerInternalException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPrincipalsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPrincipalsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPrincipalsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPrincipalsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPrincipals");

            return clientHandler.execute(new ClientExecutionParams<ListPrincipalsRequest, ListPrincipalsResponse>()
                    .withOperationName("ListPrincipals").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPrincipalsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPrincipalsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the current status of the asynchronous tasks performed by RAM when you perform the
     * <a>ReplacePermissionAssociationsWork</a> operation.
     * </p>
     *
     * @param listReplacePermissionAssociationsWorkRequest
     * @return Result of the ListReplacePermissionAssociationsWork operation returned by the service.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws InvalidNextTokenException
     *         The operation failed because the specified value for <code>NextToken</code> isn't valid. You must specify
     *         a value you received in the <code>NextToken</code> response of a previous call to this operation.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListReplacePermissionAssociationsWork
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListReplacePermissionAssociationsWork"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListReplacePermissionAssociationsWorkResponse listReplacePermissionAssociationsWork(
            ListReplacePermissionAssociationsWorkRequest listReplacePermissionAssociationsWorkRequest)
            throws ServerInternalException, ServiceUnavailableException, InvalidNextTokenException, InvalidParameterException,
            AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListReplacePermissionAssociationsWorkResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, ListReplacePermissionAssociationsWorkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listReplacePermissionAssociationsWorkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listReplacePermissionAssociationsWorkRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReplacePermissionAssociationsWork");

            return clientHandler
                    .execute(new ClientExecutionParams<ListReplacePermissionAssociationsWorkRequest, ListReplacePermissionAssociationsWorkResponse>()
                            .withOperationName("ListReplacePermissionAssociationsWork").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(listReplacePermissionAssociationsWorkRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListReplacePermissionAssociationsWorkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the RAM permissions that are associated with a resource share.
     * </p>
     *
     * @param listResourceSharePermissionsRequest
     * @return Result of the ListResourceSharePermissions operation returned by the service.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws InvalidNextTokenException
     *         The operation failed because the specified value for <code>NextToken</code> isn't valid. You must specify
     *         a value you received in the <code>NextToken</code> response of a previous call to this operation.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListResourceSharePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResourceSharePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResourceSharePermissionsResponse listResourceSharePermissions(
            ListResourceSharePermissionsRequest listResourceSharePermissionsRequest) throws InvalidParameterException,
            MalformedArnException, UnknownResourceException, InvalidNextTokenException, ServerInternalException,
            ServiceUnavailableException, OperationNotPermittedException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourceSharePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResourceSharePermissionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourceSharePermissionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceSharePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceSharePermissions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListResourceSharePermissionsRequest, ListResourceSharePermissionsResponse>()
                            .withOperationName("ListResourceSharePermissions").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listResourceSharePermissionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListResourceSharePermissionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the resource types that can be shared by RAM.
     * </p>
     *
     * @param listResourceTypesRequest
     * @return Result of the ListResourceTypes operation returned by the service.
     * @throws InvalidNextTokenException
     *         The operation failed because the specified value for <code>NextToken</code> isn't valid. You must specify
     *         a value you received in the <code>NextToken</code> response of a previous call to this operation.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListResourceTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResourceTypes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListResourceTypesResponse listResourceTypes(ListResourceTypesRequest listResourceTypesRequest)
            throws InvalidNextTokenException, InvalidParameterException, ServerInternalException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourceTypesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListResourceTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourceTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceTypes");

            return clientHandler.execute(new ClientExecutionParams<ListResourceTypesRequest, ListResourceTypesResponse>()
                    .withOperationName("ListResourceTypes").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listResourceTypesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResourceTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the resources that you added to a resource share or the resources that are shared with you.
     * </p>
     *
     * @param listResourcesRequest
     * @return Result of the ListResources operation returned by the service.
     * @throws InvalidResourceTypeException
     *         The operation failed because the specified resource type isn't valid.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws InvalidNextTokenException
     *         The operation failed because the specified value for <code>NextToken</code> isn't valid. You must specify
     *         a value you received in the <code>NextToken</code> response of a previous call to this operation.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListResourcesResponse listResources(ListResourcesRequest listResourcesRequest) throws InvalidResourceTypeException,
            UnknownResourceException, MalformedArnException, InvalidNextTokenException, InvalidParameterException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourcesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResources");

            return clientHandler.execute(new ClientExecutionParams<ListResourcesRequest, ListResourcesResponse>()
                    .withOperationName("ListResources").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * When you attach a resource-based policy to a resource, RAM automatically creates a resource share of
     * <code>featureSet</code>=<code>CREATED_FROM_POLICY</code> with a managed permission that has the same IAM
     * permissions as the original resource-based policy. However, this type of managed permission is visible to only
     * the resource share owner, and the associated resource share can't be modified by using RAM.
     * </p>
     * <p>
     * This operation creates a separate, fully manageable customer managed permission that has the same IAM permissions
     * as the original resource-based policy. You can associate this customer managed permission to any resource shares.
     * </p>
     * <p>
     * Before you use <a>PromoteResourceShareCreatedFromPolicy</a>, you should first run this operation to ensure that
     * you have an appropriate customer managed permission that can be associated with the promoted resource share.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * The original <code>CREATED_FROM_POLICY</code> policy isn't deleted, and resource shares using that original
     * policy aren't automatically updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can't modify a <code>CREATED_FROM_POLICY</code> resource share so you can't associate the new customer
     * managed permission by using <code>ReplacePermsissionAssociations</code>. However, if you use
     * <a>PromoteResourceShareCreatedFromPolicy</a>, that operation automatically associates the fully manageable
     * customer managed permission to the newly promoted <code>STANDARD</code> resource share.
     * </p>
     * </li>
     * <li>
     * <p>
     * After you promote a resource share, if the original <code>CREATED_FROM_POLICY</code> managed permission has no
     * other associations to A resource share, then RAM automatically deletes it.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param promotePermissionCreatedFromPolicyRequest
     * @return Result of the PromotePermissionCreatedFromPolicy operation returned by the service.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws MissingRequiredParameterException
     *         The operation failed because a required input parameter is missing.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.PromotePermissionCreatedFromPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/PromotePermissionCreatedFromPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PromotePermissionCreatedFromPolicyResponse promotePermissionCreatedFromPolicy(
            PromotePermissionCreatedFromPolicyRequest promotePermissionCreatedFromPolicyRequest) throws MalformedArnException,
            OperationNotPermittedException, InvalidParameterException, MissingRequiredParameterException,
            ServerInternalException, ServiceUnavailableException, UnknownResourceException, AwsServiceException,
            SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PromotePermissionCreatedFromPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PromotePermissionCreatedFromPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(promotePermissionCreatedFromPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                promotePermissionCreatedFromPolicyRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PromotePermissionCreatedFromPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<PromotePermissionCreatedFromPolicyRequest, PromotePermissionCreatedFromPolicyResponse>()
                            .withOperationName("PromotePermissionCreatedFromPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(promotePermissionCreatedFromPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PromotePermissionCreatedFromPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * When you attach a resource-based policy to a resource, RAM automatically creates a resource share of
     * <code>featureSet</code>=<code>CREATED_FROM_POLICY</code> with a managed permission that has the same IAM
     * permissions as the original resource-based policy. However, this type of managed permission is visible to only
     * the resource share owner, and the associated resource share can't be modified by using RAM.
     * </p>
     * <p>
     * This operation promotes the resource share to a <code>STANDARD</code> resource share that is fully manageable in
     * RAM. When you promote a resource share, you can then manage the resource share in RAM and it becomes visible to
     * all of the principals you shared it with.
     * </p>
     * <important>
     * <p>
     * Before you perform this operation, you should first run <a>PromotePermissionCreatedFromPolicy</a>to ensure that
     * you have an appropriate customer managed permission that can be associated with this resource share after its is
     * promoted. If this operation can't find a managed permission that exactly matches the existing
     * <code>CREATED_FROM_POLICY</code> permission, then this operation fails.
     * </p>
     * </important>
     *
     * @param promoteResourceShareCreatedFromPolicyRequest
     * @return Result of the PromoteResourceShareCreatedFromPolicy operation returned by the service.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws ResourceShareLimitExceededException
     *         The operation failed because it would exceed the limit for resource shares for your account. To view the
     *         limits for your Amazon Web Services account, see the <a
     *         href="https://console.aws.amazon.com/servicequotas/home/services/ram/quotas">RAM page in the Service
     *         Quotas console</a>.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws MissingRequiredParameterException
     *         The operation failed because a required input parameter is missing.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws InvalidStateTransitionException
     *         The operation failed because the requested operation isn't valid for the resource share in its current
     *         state.
     * @throws UnmatchedPolicyPermissionException
     *         There isn't an existing managed permission defined in RAM that has the same IAM permissions as the
     *         resource-based policy attached to the resource. You should first run
     *         <a>PromotePermissionCreatedFromPolicy</a> to create that managed permission.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.PromoteResourceShareCreatedFromPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/PromoteResourceShareCreatedFromPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PromoteResourceShareCreatedFromPolicyResponse promoteResourceShareCreatedFromPolicy(
            PromoteResourceShareCreatedFromPolicyRequest promoteResourceShareCreatedFromPolicyRequest)
            throws MalformedArnException, ResourceShareLimitExceededException, OperationNotPermittedException,
            InvalidParameterException, MissingRequiredParameterException, ServerInternalException, ServiceUnavailableException,
            UnknownResourceException, InvalidStateTransitionException, UnmatchedPolicyPermissionException, AwsServiceException,
            SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PromoteResourceShareCreatedFromPolicyResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, PromoteResourceShareCreatedFromPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(promoteResourceShareCreatedFromPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                promoteResourceShareCreatedFromPolicyRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PromoteResourceShareCreatedFromPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<PromoteResourceShareCreatedFromPolicyRequest, PromoteResourceShareCreatedFromPolicyResponse>()
                            .withOperationName("PromoteResourceShareCreatedFromPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(promoteResourceShareCreatedFromPolicyRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PromoteResourceShareCreatedFromPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Rejects an invitation to a resource share from another Amazon Web Services account.
     * </p>
     *
     * @param rejectResourceShareInvitationRequest
     * @return Result of the RejectResourceShareInvitation operation returned by the service.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws ResourceShareInvitationArnNotFoundException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> for an invitation was not found.
     * @throws ResourceShareInvitationAlreadyAcceptedException
     *         The operation failed because the specified invitation was already accepted.
     * @throws ResourceShareInvitationAlreadyRejectedException
     *         The operation failed because the specified invitation was already rejected.
     * @throws ResourceShareInvitationExpiredException
     *         The operation failed because the specified invitation is past its expiration date and time.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws InvalidClientTokenException
     *         The operation failed because the specified client token isn't valid.
     * @throws IdempotentParameterMismatchException
     *         The operation failed because the client token input parameter matched one that was used with a previous
     *         call to the operation, but at least one of the other input parameters is different from the previous
     *         call.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.RejectResourceShareInvitation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/RejectResourceShareInvitation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RejectResourceShareInvitationResponse rejectResourceShareInvitation(
            RejectResourceShareInvitationRequest rejectResourceShareInvitationRequest) throws MalformedArnException,
            OperationNotPermittedException, ResourceShareInvitationArnNotFoundException,
            ResourceShareInvitationAlreadyAcceptedException, ResourceShareInvitationAlreadyRejectedException,
            ResourceShareInvitationExpiredException, ServerInternalException, ServiceUnavailableException,
            InvalidClientTokenException, IdempotentParameterMismatchException, AwsServiceException, SdkClientException,
            RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RejectResourceShareInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RejectResourceShareInvitationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rejectResourceShareInvitationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                rejectResourceShareInvitationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectResourceShareInvitation");

            return clientHandler
                    .execute(new ClientExecutionParams<RejectResourceShareInvitationRequest, RejectResourceShareInvitationResponse>()
                            .withOperationName("RejectResourceShareInvitation").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(rejectResourceShareInvitationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RejectResourceShareInvitationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates all resource shares that use a managed permission to a different managed permission. This operation
     * always applies the default version of the target managed permission. You can optionally specify that the update
     * applies to only resource shares that currently use a specified version. This enables you to update to the latest
     * version, without changing the which managed permission is used.
     * </p>
     * <p>
     * You can use this operation to update all of your resource shares to use the current default version of the
     * permission by specifying the same value for the <code>fromPermissionArn</code> and <code>toPermissionArn</code>
     * parameters.
     * </p>
     * <p>
     * You can use the optional <code>fromPermissionVersion</code> parameter to update only those resources that use a
     * specified version of the managed permission to the new managed permission.
     * </p>
     * <important>
     * <p>
     * To successfully perform this operation, you must have permission to update the resource-based policy on all
     * affected resource types.
     * </p>
     * </important>
     *
     * @param replacePermissionAssociationsRequest
     * @return Result of the ReplacePermissionAssociations operation returned by the service.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws InvalidClientTokenException
     *         The operation failed because the specified client token isn't valid.
     * @throws IdempotentParameterMismatchException
     *         The operation failed because the client token input parameter matched one that was used with a previous
     *         call to the operation, but at least one of the other input parameters is different from the previous
     *         call.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.ReplacePermissionAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ReplacePermissionAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ReplacePermissionAssociationsResponse replacePermissionAssociations(
            ReplacePermissionAssociationsRequest replacePermissionAssociationsRequest) throws MalformedArnException,
            InvalidParameterException, ServerInternalException, ServiceUnavailableException, OperationNotPermittedException,
            UnknownResourceException, InvalidClientTokenException, IdempotentParameterMismatchException, AwsServiceException,
            SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ReplacePermissionAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ReplacePermissionAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(replacePermissionAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                replacePermissionAssociationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReplacePermissionAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<ReplacePermissionAssociationsRequest, ReplacePermissionAssociationsResponse>()
                            .withOperationName("ReplacePermissionAssociations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(replacePermissionAssociationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ReplacePermissionAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Designates the specified version number as the default version for the specified customer managed permission. New
     * resource shares automatically use this new default permission. Existing resource shares continue to use their
     * original permission version, but you can use <a>ReplacePermissionAssociations</a> to update them.
     * </p>
     *
     * @param setDefaultPermissionVersionRequest
     * @return Result of the SetDefaultPermissionVersion operation returned by the service.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws InvalidClientTokenException
     *         The operation failed because the specified client token isn't valid.
     * @throws IdempotentParameterMismatchException
     *         The operation failed because the client token input parameter matched one that was used with a previous
     *         call to the operation, but at least one of the other input parameters is different from the previous
     *         call.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.SetDefaultPermissionVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/SetDefaultPermissionVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SetDefaultPermissionVersionResponse setDefaultPermissionVersion(
            SetDefaultPermissionVersionRequest setDefaultPermissionVersionRequest) throws InvalidParameterException,
            MalformedArnException, ServerInternalException, ServiceUnavailableException, UnknownResourceException,
            InvalidClientTokenException, IdempotentParameterMismatchException, AwsServiceException, SdkClientException,
            RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetDefaultPermissionVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SetDefaultPermissionVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setDefaultPermissionVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setDefaultPermissionVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetDefaultPermissionVersion");

            return clientHandler
                    .execute(new ClientExecutionParams<SetDefaultPermissionVersionRequest, SetDefaultPermissionVersionResponse>()
                            .withOperationName("SetDefaultPermissionVersion").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(setDefaultPermissionVersionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SetDefaultPermissionVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds the specified tag keys and values to a resource share or managed permission. If you choose a resource share,
     * the tags are attached to only the resource share, not to the resources that are in the resource share.
     * </p>
     * <p>
     * The tags on a managed permission are the same for all versions of the managed permission.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws TagLimitExceededException
     *         The operation failed because it would exceed the limit for tags for your Amazon Web Services account.
     * @throws ResourceArnNotFoundException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> was not found.
     * @throws TagPolicyViolationException
     *         The operation failed because the specified tag key is a reserved word and can't be used.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidParameterException,
            MalformedArnException, UnknownResourceException, TagLimitExceededException, ResourceArnNotFoundException,
            TagPolicyViolationException, ServerInternalException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tag key and value pairs from the specified resource share or managed permission.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws UnknownResourceException,
            InvalidParameterException, MalformedArnException, ServerInternalException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies some of the properties of the specified resource share.
     * </p>
     *
     * @param updateResourceShareRequest
     * @return Result of the UpdateResourceShare operation returned by the service.
     * @throws IdempotentParameterMismatchException
     *         The operation failed because the client token input parameter matched one that was used with a previous
     *         call to the operation, but at least one of the other input parameters is different from the previous
     *         call.
     * @throws MissingRequiredParameterException
     *         The operation failed because a required input parameter is missing.
     * @throws UnknownResourceException
     *         The operation failed because a specified resource couldn't be found.
     * @throws MalformedArnException
     *         The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.
     * @throws InvalidClientTokenException
     *         The operation failed because the specified client token isn't valid.
     * @throws InvalidParameterException
     *         The operation failed because a parameter you specified isn't valid.
     * @throws OperationNotPermittedException
     *         The operation failed because the requested operation isn't permitted.
     * @throws ServerInternalException
     *         The operation failed because the service could not respond to the request due to an internal problem. Try
     *         again later.
     * @throws ServiceUnavailableException
     *         The operation failed because the service isn't available. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RamClient.UpdateResourceShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/UpdateResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateResourceShareResponse updateResourceShare(UpdateResourceShareRequest updateResourceShareRequest)
            throws IdempotentParameterMismatchException, MissingRequiredParameterException, UnknownResourceException,
            MalformedArnException, InvalidClientTokenException, InvalidParameterException, OperationNotPermittedException,
            ServerInternalException, ServiceUnavailableException, AwsServiceException, SdkClientException, RamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateResourceShareResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateResourceShareRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResourceShare");

            return clientHandler.execute(new ClientExecutionParams<UpdateResourceShareRequest, UpdateResourceShareResponse>()
                    .withOperationName("UpdateResourceShare").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateResourceShareRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateResourceShareRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        RamServiceClientConfigurationBuilder serviceConfigBuilder = new RamServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RamException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidClientTokenException")
                                .exceptionBuilderSupplier(InvalidClientTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerInternalException")
                                .exceptionBuilderSupplier(ServerInternalException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceTypeException")
                                .exceptionBuilderSupplier(InvalidResourceTypeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationExpiredException")
                                .exceptionBuilderSupplier(ResourceShareInvitationExpiredException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingRequiredParameterException")
                                .exceptionBuilderSupplier(MissingRequiredParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationNotPermittedException")
                                .exceptionBuilderSupplier(OperationNotPermittedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareLimitExceededException")
                                .exceptionBuilderSupplier(ResourceShareLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PermissionAlreadyExistsException")
                                .exceptionBuilderSupplier(PermissionAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnknownResourceException")
                                .exceptionBuilderSupplier(UnknownResourceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PermissionLimitExceededException")
                                .exceptionBuilderSupplier(PermissionLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPolicyException")
                                .exceptionBuilderSupplier(InvalidPolicyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PermissionVersionsLimitExceededException")
                                .exceptionBuilderSupplier(PermissionVersionsLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                                .exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidMaxResultsException")
                                .exceptionBuilderSupplier(InvalidMaxResultsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagLimitExceededException")
                                .exceptionBuilderSupplier(TagLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceArnNotFoundException")
                                .exceptionBuilderSupplier(ResourceArnNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnmatchedPolicyPermissionException")
                                .exceptionBuilderSupplier(UnmatchedPolicyPermissionException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationAlreadyRejectedException")
                                .exceptionBuilderSupplier(ResourceShareInvitationAlreadyRejectedException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MalformedArnException")
                                .exceptionBuilderSupplier(MalformedArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStateTransitionException")
                                .exceptionBuilderSupplier(InvalidStateTransitionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationAlreadyAcceptedException")
                                .exceptionBuilderSupplier(ResourceShareInvitationAlreadyAcceptedException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationArnNotFoundException")
                                .exceptionBuilderSupplier(ResourceShareInvitationArnNotFoundException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MalformedPolicyTemplateException")
                                .exceptionBuilderSupplier(MalformedPolicyTemplateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagPolicyViolationException")
                                .exceptionBuilderSupplier(TagPolicyViolationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final RamServiceClientConfiguration serviceClientConfiguration() {
        return new RamServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
