/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ram;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ram.internal.RamServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ram.model.AcceptResourceShareInvitationRequest;
import software.amazon.awssdk.services.ram.model.AcceptResourceShareInvitationResponse;
import software.amazon.awssdk.services.ram.model.AssociateResourceSharePermissionRequest;
import software.amazon.awssdk.services.ram.model.AssociateResourceSharePermissionResponse;
import software.amazon.awssdk.services.ram.model.AssociateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.AssociateResourceShareResponse;
import software.amazon.awssdk.services.ram.model.CreatePermissionRequest;
import software.amazon.awssdk.services.ram.model.CreatePermissionResponse;
import software.amazon.awssdk.services.ram.model.CreatePermissionVersionRequest;
import software.amazon.awssdk.services.ram.model.CreatePermissionVersionResponse;
import software.amazon.awssdk.services.ram.model.CreateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.CreateResourceShareResponse;
import software.amazon.awssdk.services.ram.model.DeletePermissionRequest;
import software.amazon.awssdk.services.ram.model.DeletePermissionResponse;
import software.amazon.awssdk.services.ram.model.DeletePermissionVersionRequest;
import software.amazon.awssdk.services.ram.model.DeletePermissionVersionResponse;
import software.amazon.awssdk.services.ram.model.DeleteResourceShareRequest;
import software.amazon.awssdk.services.ram.model.DeleteResourceShareResponse;
import software.amazon.awssdk.services.ram.model.DisassociateResourceSharePermissionRequest;
import software.amazon.awssdk.services.ram.model.DisassociateResourceSharePermissionResponse;
import software.amazon.awssdk.services.ram.model.DisassociateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.DisassociateResourceShareResponse;
import software.amazon.awssdk.services.ram.model.EnableSharingWithAwsOrganizationRequest;
import software.amazon.awssdk.services.ram.model.EnableSharingWithAwsOrganizationResponse;
import software.amazon.awssdk.services.ram.model.GetPermissionRequest;
import software.amazon.awssdk.services.ram.model.GetPermissionResponse;
import software.amazon.awssdk.services.ram.model.GetResourcePoliciesRequest;
import software.amazon.awssdk.services.ram.model.GetResourcePoliciesResponse;
import software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsRequest;
import software.amazon.awssdk.services.ram.model.GetResourceShareAssociationsResponse;
import software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsRequest;
import software.amazon.awssdk.services.ram.model.GetResourceShareInvitationsResponse;
import software.amazon.awssdk.services.ram.model.GetResourceSharesRequest;
import software.amazon.awssdk.services.ram.model.GetResourceSharesResponse;
import software.amazon.awssdk.services.ram.model.IdempotentParameterMismatchException;
import software.amazon.awssdk.services.ram.model.InvalidClientTokenException;
import software.amazon.awssdk.services.ram.model.InvalidMaxResultsException;
import software.amazon.awssdk.services.ram.model.InvalidNextTokenException;
import software.amazon.awssdk.services.ram.model.InvalidParameterException;
import software.amazon.awssdk.services.ram.model.InvalidPolicyException;
import software.amazon.awssdk.services.ram.model.InvalidResourceTypeException;
import software.amazon.awssdk.services.ram.model.InvalidStateTransitionException;
import software.amazon.awssdk.services.ram.model.ListPendingInvitationResourcesRequest;
import software.amazon.awssdk.services.ram.model.ListPendingInvitationResourcesResponse;
import software.amazon.awssdk.services.ram.model.ListPermissionAssociationsRequest;
import software.amazon.awssdk.services.ram.model.ListPermissionAssociationsResponse;
import software.amazon.awssdk.services.ram.model.ListPermissionVersionsRequest;
import software.amazon.awssdk.services.ram.model.ListPermissionVersionsResponse;
import software.amazon.awssdk.services.ram.model.ListPermissionsRequest;
import software.amazon.awssdk.services.ram.model.ListPermissionsResponse;
import software.amazon.awssdk.services.ram.model.ListPrincipalsRequest;
import software.amazon.awssdk.services.ram.model.ListPrincipalsResponse;
import software.amazon.awssdk.services.ram.model.ListReplacePermissionAssociationsWorkRequest;
import software.amazon.awssdk.services.ram.model.ListReplacePermissionAssociationsWorkResponse;
import software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsRequest;
import software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsResponse;
import software.amazon.awssdk.services.ram.model.ListResourceTypesRequest;
import software.amazon.awssdk.services.ram.model.ListResourceTypesResponse;
import software.amazon.awssdk.services.ram.model.ListResourcesRequest;
import software.amazon.awssdk.services.ram.model.ListResourcesResponse;
import software.amazon.awssdk.services.ram.model.MalformedArnException;
import software.amazon.awssdk.services.ram.model.MalformedPolicyTemplateException;
import software.amazon.awssdk.services.ram.model.MissingRequiredParameterException;
import software.amazon.awssdk.services.ram.model.OperationNotPermittedException;
import software.amazon.awssdk.services.ram.model.PermissionAlreadyExistsException;
import software.amazon.awssdk.services.ram.model.PermissionLimitExceededException;
import software.amazon.awssdk.services.ram.model.PermissionVersionsLimitExceededException;
import software.amazon.awssdk.services.ram.model.PromotePermissionCreatedFromPolicyRequest;
import software.amazon.awssdk.services.ram.model.PromotePermissionCreatedFromPolicyResponse;
import software.amazon.awssdk.services.ram.model.PromoteResourceShareCreatedFromPolicyRequest;
import software.amazon.awssdk.services.ram.model.PromoteResourceShareCreatedFromPolicyResponse;
import software.amazon.awssdk.services.ram.model.RamException;
import software.amazon.awssdk.services.ram.model.RejectResourceShareInvitationRequest;
import software.amazon.awssdk.services.ram.model.RejectResourceShareInvitationResponse;
import software.amazon.awssdk.services.ram.model.ReplacePermissionAssociationsRequest;
import software.amazon.awssdk.services.ram.model.ReplacePermissionAssociationsResponse;
import software.amazon.awssdk.services.ram.model.ResourceArnNotFoundException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationAlreadyAcceptedException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationAlreadyRejectedException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationArnNotFoundException;
import software.amazon.awssdk.services.ram.model.ResourceShareInvitationExpiredException;
import software.amazon.awssdk.services.ram.model.ResourceShareLimitExceededException;
import software.amazon.awssdk.services.ram.model.ServerInternalException;
import software.amazon.awssdk.services.ram.model.ServiceUnavailableException;
import software.amazon.awssdk.services.ram.model.SetDefaultPermissionVersionRequest;
import software.amazon.awssdk.services.ram.model.SetDefaultPermissionVersionResponse;
import software.amazon.awssdk.services.ram.model.TagLimitExceededException;
import software.amazon.awssdk.services.ram.model.TagPolicyViolationException;
import software.amazon.awssdk.services.ram.model.TagResourceRequest;
import software.amazon.awssdk.services.ram.model.TagResourceResponse;
import software.amazon.awssdk.services.ram.model.ThrottlingException;
import software.amazon.awssdk.services.ram.model.UnknownResourceException;
import software.amazon.awssdk.services.ram.model.UnmatchedPolicyPermissionException;
import software.amazon.awssdk.services.ram.model.UntagResourceRequest;
import software.amazon.awssdk.services.ram.model.UntagResourceResponse;
import software.amazon.awssdk.services.ram.model.UpdateResourceShareRequest;
import software.amazon.awssdk.services.ram.model.UpdateResourceShareResponse;
import software.amazon.awssdk.services.ram.transform.AcceptResourceShareInvitationRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.AssociateResourceSharePermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.AssociateResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.CreatePermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.CreatePermissionVersionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.CreateResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DeletePermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DeletePermissionVersionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DeleteResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DisassociateResourceSharePermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.DisassociateResourceShareRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.EnableSharingWithAwsOrganizationRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetPermissionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourcePoliciesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourceShareAssociationsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourceShareInvitationsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.GetResourceSharesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPendingInvitationResourcesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPermissionAssociationsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPermissionVersionsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPermissionsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListPrincipalsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListReplacePermissionAssociationsWorkRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListResourceSharePermissionsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListResourceTypesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ListResourcesRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.PromotePermissionCreatedFromPolicyRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.PromoteResourceShareCreatedFromPolicyRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.RejectResourceShareInvitationRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.ReplacePermissionAssociationsRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.SetDefaultPermissionVersionRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ram.transform.UpdateResourceShareRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link RamAsyncClient}.
 *
 * @see RamAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRamAsyncClient implements RamAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRamAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRamAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Accepts an invitation to a resource share from another Amazon Web Services account. After you accept the
     * invitation, the resources included in the resource share are available to interact with in the relevant Amazon
     * Web Services Management Consoles and tools.
     * </p>
     *
     * @param acceptResourceShareInvitationRequest
     * @return A Java Future containing the result of the AcceptResourceShareInvitation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>ResourceShareInvitationArnNotFoundException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> for an invitation was not found.</li>
     *         <li>ResourceShareInvitationAlreadyAcceptedException The operation failed because the specified invitation
     *         was already accepted.</li>
     *         <li>ResourceShareInvitationAlreadyRejectedException The operation failed because the specified invitation
     *         was already rejected.</li>
     *         <li>ResourceShareInvitationExpiredException The operation failed because the specified invitation is past
     *         its expiration date and time.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>InvalidClientTokenException The operation failed because the specified client token isn't valid.</li>
     *         <li>IdempotentParameterMismatchException The operation failed because the client token input parameter
     *         matched one that was used with a previous call to the operation, but at least one of the other input
     *         parameters is different from the previous call.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.AcceptResourceShareInvitation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/AcceptResourceShareInvitation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptResourceShareInvitationResponse> acceptResourceShareInvitation(
            AcceptResourceShareInvitationRequest acceptResourceShareInvitationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(acceptResourceShareInvitationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                acceptResourceShareInvitationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptResourceShareInvitation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptResourceShareInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcceptResourceShareInvitationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptResourceShareInvitationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptResourceShareInvitationRequest, AcceptResourceShareInvitationResponse>()
                            .withOperationName("AcceptResourceShareInvitation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AcceptResourceShareInvitationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(acceptResourceShareInvitationRequest));
            CompletableFuture<AcceptResourceShareInvitationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds the specified list of principals and list of resources to a resource share. Principals that already have
     * access to this resource share immediately receive access to the added resources. Newly added principals
     * immediately receive access to the resources shared in this resource share.
     * </p>
     *
     * @param associateResourceShareRequest
     * @return A Java Future containing the result of the AssociateResourceShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>IdempotentParameterMismatchException The operation failed because the client token input parameter
     *         matched one that was used with a previous call to the operation, but at least one of the other input
     *         parameters is different from the previous call.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>InvalidStateTransitionException The operation failed because the requested operation isn't valid for
     *         the resource share in its current state.</li>
     *         <li>ResourceShareLimitExceededException The operation failed because it would exceed the limit for
     *         resource shares for your account. To view the limits for your Amazon Web Services account, see the <a
     *         href="https://console.aws.amazon.com/servicequotas/home/services/ram/quotas">RAM page in the Service
     *         Quotas console</a>.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>InvalidStateTransitionException The operation failed because the requested operation isn't valid for
     *         the resource share in its current state.</li>
     *         <li>InvalidClientTokenException The operation failed because the specified client token isn't valid.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>ThrottlingException The operation failed because it exceeded the rate at which you are allowed to
     *         perform this operation. Please try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.AssociateResourceShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/AssociateResourceShare" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateResourceShareResponse> associateResourceShare(
            AssociateResourceShareRequest associateResourceShareRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateResourceShareRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResourceShare");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateResourceShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateResourceShareResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateResourceShareRequest, AssociateResourceShareResponse>()
                            .withOperationName("AssociateResourceShare").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateResourceShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateResourceShareRequest));
            CompletableFuture<AssociateResourceShareResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or replaces the RAM permission for a resource type included in a resource share. You can have exactly one
     * permission associated with each resource type in the resource share. You can add a new RAM permission only if
     * there are currently no resources of that resource type currently in the resource share.
     * </p>
     *
     * @param associateResourceSharePermissionRequest
     * @return A Java Future containing the result of the AssociateResourceSharePermission operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>InvalidClientTokenException The operation failed because the specified client token isn't valid.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.AssociateResourceSharePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/AssociateResourceSharePermission"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateResourceSharePermissionResponse> associateResourceSharePermission(
            AssociateResourceSharePermissionRequest associateResourceSharePermissionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateResourceSharePermissionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateResourceSharePermissionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResourceSharePermission");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateResourceSharePermissionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AssociateResourceSharePermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateResourceSharePermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateResourceSharePermissionRequest, AssociateResourceSharePermissionResponse>()
                            .withOperationName("AssociateResourceSharePermission").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateResourceSharePermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateResourceSharePermissionRequest));
            CompletableFuture<AssociateResourceSharePermissionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a customer managed permission for a specified resource type that you can attach to resource shares. It is
     * created in the Amazon Web Services Region in which you call the operation.
     * </p>
     *
     * @param createPermissionRequest
     * @return A Java Future containing the result of the CreatePermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>InvalidPolicyException The operation failed because a policy you specified isn't valid.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>PermissionAlreadyExistsException The operation failed because a permission with the specified name
     *         already exists in the requested Amazon Web Services Region. Choose a different name.</li>
     *         <li>MalformedPolicyTemplateException The operation failed because the policy template that you provided
     *         isn't valid.</li>
     *         <li>InvalidClientTokenException The operation failed because the specified client token isn't valid.</li>
     *         <li>PermissionLimitExceededException The operation failed because it would exceed the maximum number of
     *         permissions you can create in each Amazon Web Services Region. To view the limits for your Amazon Web
     *         Services account, see the <a
     *         href="https://console.aws.amazon.com/servicequotas/home/services/ram/quotas">RAM page in the Service
     *         Quotas console</a>.</li>
     *         <li>IdempotentParameterMismatchException The operation failed because the client token input parameter
     *         matched one that was used with a previous call to the operation, but at least one of the other input
     *         parameters is different from the previous call.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.CreatePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/CreatePermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePermissionResponse> createPermission(CreatePermissionRequest createPermissionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPermissionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePermission");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePermissionRequest, CreatePermissionResponse>()
                            .withOperationName("CreatePermission").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPermissionRequest));
            CompletableFuture<CreatePermissionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new version of the specified customer managed permission. The new version is automatically set as the
     * default version of the customer managed permission. New resource shares automatically use the default permission.
     * Existing resource shares continue to use their original permission versions, but you can use
     * <a>ReplacePermissionAssociations</a> to update them.
     * </p>
     * <p>
     * If the specified customer managed permission already has the maximum of 5 versions, then you must delete one of
     * the existing versions before you can create a new one.
     * </p>
     *
     * @param createPermissionVersionRequest
     * @return A Java Future containing the result of the CreatePermissionVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>InvalidPolicyException The operation failed because a policy you specified isn't valid.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>MalformedPolicyTemplateException The operation failed because the policy template that you provided
     *         isn't valid.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>InvalidClientTokenException The operation failed because the specified client token isn't valid.</li>
     *         <li>IdempotentParameterMismatchException The operation failed because the client token input parameter
     *         matched one that was used with a previous call to the operation, but at least one of the other input
     *         parameters is different from the previous call.</li>
     *         <li>PermissionVersionsLimitExceededException The operation failed because it would exceed the limit for
     *         the number of versions you can have for a permission. To view the limits for your Amazon Web Services
     *         account, see the <a href="https://console.aws.amazon.com/servicequotas/home/services/ram/quotas">RAM page
     *         in the Service Quotas console</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.CreatePermissionVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/CreatePermissionVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePermissionVersionResponse> createPermissionVersion(
            CreatePermissionVersionRequest createPermissionVersionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPermissionVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPermissionVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePermissionVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePermissionVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePermissionVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePermissionVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePermissionVersionRequest, CreatePermissionVersionResponse>()
                            .withOperationName("CreatePermissionVersion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePermissionVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPermissionVersionRequest));
            CompletableFuture<CreatePermissionVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a resource share. You can provide a list of the <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
     * (ARNs)</a> for the resources that you want to share, a list of principals you want to share the resources with,
     * and the permissions to grant those principals.
     * </p>
     * <note>
     * <p>
     * Sharing a resource makes it available for use by principals outside of the Amazon Web Services account that
     * created the resource. Sharing doesn't change any permissions or quotas that apply to the resource in the account
     * that created it.
     * </p>
     * </note>
     *
     * @param createResourceShareRequest
     * @return A Java Future containing the result of the CreateResourceShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>IdempotentParameterMismatchException The operation failed because the client token input parameter
     *         matched one that was used with a previous call to the operation, but at least one of the other input
     *         parameters is different from the previous call.</li>
     *         <li>InvalidStateTransitionException The operation failed because the requested operation isn't valid for
     *         the resource share in its current state.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>InvalidClientTokenException The operation failed because the specified client token isn't valid.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>ResourceShareLimitExceededException The operation failed because it would exceed the limit for
     *         resource shares for your account. To view the limits for your Amazon Web Services account, see the <a
     *         href="https://console.aws.amazon.com/servicequotas/home/services/ram/quotas">RAM page in the Service
     *         Quotas console</a>.</li>
     *         <li>TagPolicyViolationException The operation failed because the specified tag key is a reserved word and
     *         can't be used.</li>
     *         <li>TagLimitExceededException The operation failed because it would exceed the limit for tags for your
     *         Amazon Web Services account.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.CreateResourceShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/CreateResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateResourceShareResponse> createResourceShare(
            CreateResourceShareRequest createResourceShareRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createResourceShareRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResourceShare");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateResourceShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateResourceShareResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateResourceShareRequest, CreateResourceShareResponse>()
                            .withOperationName("CreateResourceShare").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateResourceShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createResourceShareRequest));
            CompletableFuture<CreateResourceShareResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified customer managed permission in the Amazon Web Services Region in which you call this
     * operation. You can delete a customer managed permission only if it isn't attached to any resource share. The
     * operation deletes all versions associated with the customer managed permission.
     * </p>
     *
     * @param deletePermissionRequest
     * @return A Java Future containing the result of the DeletePermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>InvalidClientTokenException The operation failed because the specified client token isn't valid.</li>
     *         <li>IdempotentParameterMismatchException The operation failed because the client token input parameter
     *         matched one that was used with a previous call to the operation, but at least one of the other input
     *         parameters is different from the previous call.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.DeletePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DeletePermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePermissionResponse> deletePermission(DeletePermissionRequest deletePermissionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePermissionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePermission");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePermissionRequest, DeletePermissionResponse>()
                            .withOperationName("DeletePermission").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePermissionRequest));
            CompletableFuture<DeletePermissionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes one version of a customer managed permission. The version you specify must not be attached to any
     * resource share and must not be the default version for the permission.
     * </p>
     * <p>
     * If a customer managed permission has the maximum of 5 versions, then you must delete at least one version before
     * you can create another.
     * </p>
     *
     * @param deletePermissionVersionRequest
     * @return A Java Future containing the result of the DeletePermissionVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>InvalidClientTokenException The operation failed because the specified client token isn't valid.</li>
     *         <li>IdempotentParameterMismatchException The operation failed because the client token input parameter
     *         matched one that was used with a previous call to the operation, but at least one of the other input
     *         parameters is different from the previous call.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.DeletePermissionVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DeletePermissionVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePermissionVersionResponse> deletePermissionVersion(
            DeletePermissionVersionRequest deletePermissionVersionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePermissionVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePermissionVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePermissionVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePermissionVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePermissionVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePermissionVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePermissionVersionRequest, DeletePermissionVersionResponse>()
                            .withOperationName("DeletePermissionVersion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePermissionVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePermissionVersionRequest));
            CompletableFuture<DeletePermissionVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified resource share.
     * </p>
     * <important>
     * <p>
     * This doesn't delete any of the resources that were associated with the resource share; it only stops the sharing
     * of those resources through this resource share.
     * </p>
     * </important>
     *
     * @param deleteResourceShareRequest
     * @return A Java Future containing the result of the DeleteResourceShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>IdempotentParameterMismatchException The operation failed because the client token input parameter
     *         matched one that was used with a previous call to the operation, but at least one of the other input
     *         parameters is different from the previous call.</li>
     *         <li>InvalidStateTransitionException The operation failed because the requested operation isn't valid for
     *         the resource share in its current state.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>InvalidClientTokenException The operation failed because the specified client token isn't valid.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.DeleteResourceShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DeleteResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResourceShareResponse> deleteResourceShare(
            DeleteResourceShareRequest deleteResourceShareRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourceShareRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourceShare");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResourceShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResourceShareResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResourceShareRequest, DeleteResourceShareResponse>()
                            .withOperationName("DeleteResourceShare").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteResourceShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteResourceShareRequest));
            CompletableFuture<DeleteResourceShareResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified principals or resources from participating in the specified resource share.
     * </p>
     *
     * @param disassociateResourceShareRequest
     * @return A Java Future containing the result of the DisassociateResourceShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>IdempotentParameterMismatchException The operation failed because the client token input parameter
     *         matched one that was used with a previous call to the operation, but at least one of the other input
     *         parameters is different from the previous call.</li>
     *         <li>ResourceShareLimitExceededException The operation failed because it would exceed the limit for
     *         resource shares for your account. To view the limits for your Amazon Web Services account, see the <a
     *         href="https://console.aws.amazon.com/servicequotas/home/services/ram/quotas">RAM page in the Service
     *         Quotas console</a>.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>InvalidStateTransitionException The operation failed because the requested operation isn't valid for
     *         the resource share in its current state.</li>
     *         <li>InvalidClientTokenException The operation failed because the specified client token isn't valid.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.DisassociateResourceShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DisassociateResourceShare" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateResourceShareResponse> disassociateResourceShare(
            DisassociateResourceShareRequest disassociateResourceShareRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateResourceShareRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResourceShare");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateResourceShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateResourceShareResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateResourceShareRequest, DisassociateResourceShareResponse>()
                            .withOperationName("DisassociateResourceShare").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateResourceShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateResourceShareRequest));
            CompletableFuture<DisassociateResourceShareResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a managed permission from a resource share. Permission changes take effect immediately. You can remove a
     * managed permission from a resource share only if there are currently no resources of the relevant resource type
     * currently attached to the resource share.
     * </p>
     *
     * @param disassociateResourceSharePermissionRequest
     * @return A Java Future containing the result of the DisassociateResourceSharePermission operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>InvalidClientTokenException The operation failed because the specified client token isn't valid.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>InvalidStateTransitionException The operation failed because the requested operation isn't valid for
     *         the resource share in its current state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.DisassociateResourceSharePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/DisassociateResourceSharePermission"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateResourceSharePermissionResponse> disassociateResourceSharePermission(
            DisassociateResourceSharePermissionRequest disassociateResourceSharePermissionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateResourceSharePermissionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateResourceSharePermissionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResourceSharePermission");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateResourceSharePermissionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateResourceSharePermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateResourceSharePermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateResourceSharePermissionRequest, DisassociateResourceSharePermissionResponse>()
                            .withOperationName("DisassociateResourceSharePermission").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateResourceSharePermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateResourceSharePermissionRequest));
            CompletableFuture<DisassociateResourceSharePermissionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables resource sharing within your organization in Organizations. This operation creates a service-linked role
     * called <code>AWSServiceRoleForResourceAccessManager</code> that has the IAM managed policy named
     * AWSResourceAccessManagerServiceRolePolicy attached. This role permits RAM to retrieve information about the
     * organization and its structure. This lets you share resources with all of the accounts in the calling account's
     * organization by specifying the organization ID, or all of the accounts in an organizational unit (OU) by
     * specifying the OU ID. Until you enable sharing within the organization, you can specify only individual Amazon
     * Web Services accounts, or for supported resource types, IAM roles and users.
     * </p>
     * <p>
     * You must call this operation from an IAM role or user in the organization's management account.
     * </p>
     * <p/>
     *
     * @param enableSharingWithAwsOrganizationRequest
     * @return A Java Future containing the result of the EnableSharingWithAwsOrganization operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.
     *         </li> <li>ServerInternalException The operation failed because the service could not respond to the
     *         request due to an internal problem. Try again later.</li> <li>ServiceUnavailableException The operation
     *         failed because the service isn't available. Try again later.</li> <li>SdkException Base class for all
     *         exceptions that can be thrown by the SDK (both service and client). Can be used for catch all scenarios.
     *         </li> <li>SdkClientException If any client side error occurs such as an IO related failure, failure to
     *         get credentials, etc.</li> <li>RamException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.EnableSharingWithAwsOrganization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/EnableSharingWithAwsOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableSharingWithAwsOrganizationResponse> enableSharingWithAwsOrganization(
            EnableSharingWithAwsOrganizationRequest enableSharingWithAwsOrganizationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableSharingWithAwsOrganizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                enableSharingWithAwsOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableSharingWithAwsOrganization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableSharingWithAwsOrganizationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, EnableSharingWithAwsOrganizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableSharingWithAwsOrganizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableSharingWithAwsOrganizationRequest, EnableSharingWithAwsOrganizationResponse>()
                            .withOperationName("EnableSharingWithAwsOrganization").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new EnableSharingWithAwsOrganizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(enableSharingWithAwsOrganizationRequest));
            CompletableFuture<EnableSharingWithAwsOrganizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the contents of a managed permission in JSON format.
     * </p>
     *
     * @param getPermissionRequest
     * @return A Java Future containing the result of the GetPermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetPermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetPermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPermissionResponse> getPermission(GetPermissionRequest getPermissionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPermissionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPermission");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPermissionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetPermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPermissionRequest, GetPermissionResponse>()
                            .withOperationName("GetPermission").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPermissionRequest));
            CompletableFuture<GetPermissionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the resource policies for the specified resources that you own and have shared.
     * </p>
     *
     * @param getResourcePoliciesRequest
     * @return A Java Future containing the result of the GetResourcePolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>InvalidNextTokenException The operation failed because the specified value for <code>NextToken</code>
     *         isn't valid. You must specify a value you received in the <code>NextToken</code> response of a previous
     *         call to this operation.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>ResourceArnNotFoundException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> was not found.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourcePolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourcePolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourcePoliciesResponse> getResourcePolicies(
            GetResourcePoliciesRequest getResourcePoliciesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcePoliciesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourcePoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourcePoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourcePoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourcePoliciesRequest, GetResourcePoliciesResponse>()
                            .withOperationName("GetResourcePolicies").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourcePoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourcePoliciesRequest));
            CompletableFuture<GetResourcePoliciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the lists of resources and principals that associated for resource shares that you own.
     * </p>
     *
     * @param getResourceShareAssociationsRequest
     * @return A Java Future containing the result of the GetResourceShareAssociations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>InvalidNextTokenException The operation failed because the specified value for <code>NextToken</code>
     *         isn't valid. You must specify a value you received in the <code>NextToken</code> response of a previous
     *         call to this operation.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourceShareAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceShareAssociationsResponse> getResourceShareAssociations(
            GetResourceShareAssociationsRequest getResourceShareAssociationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceShareAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceShareAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceShareAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceShareAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceShareAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourceShareAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceShareAssociationsRequest, GetResourceShareAssociationsResponse>()
                            .withOperationName("GetResourceShareAssociations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourceShareAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourceShareAssociationsRequest));
            CompletableFuture<GetResourceShareAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about invitations that you have received for resource shares.
     * </p>
     *
     * @param getResourceShareInvitationsRequest
     * @return A Java Future containing the result of the GetResourceShareInvitations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceShareInvitationArnNotFoundException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> for an invitation was not found.</li>
     *         <li>InvalidMaxResultsException The operation failed because the specified value for
     *         <code>MaxResults</code> isn't valid.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>InvalidNextTokenException The operation failed because the specified value for <code>NextToken</code>
     *         isn't valid. You must specify a value you received in the <code>NextToken</code> response of a previous
     *         call to this operation.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourceShareInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShareInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceShareInvitationsResponse> getResourceShareInvitations(
            GetResourceShareInvitationsRequest getResourceShareInvitationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceShareInvitationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceShareInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceShareInvitations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceShareInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceShareInvitationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourceShareInvitationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceShareInvitationsRequest, GetResourceShareInvitationsResponse>()
                            .withOperationName("GetResourceShareInvitations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourceShareInvitationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourceShareInvitationsRequest));
            CompletableFuture<GetResourceShareInvitationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about the resource shares that you own or that are shared with you.
     * </p>
     *
     * @param getResourceSharesRequest
     * @return A Java Future containing the result of the GetResourceShares operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>InvalidNextTokenException The operation failed because the specified value for <code>NextToken</code>
     *         isn't valid. You must specify a value you received in the <code>NextToken</code> response of a previous
     *         call to this operation.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.GetResourceShares
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/GetResourceShares" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceSharesResponse> getResourceShares(GetResourceSharesRequest getResourceSharesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceSharesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceSharesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceShares");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceSharesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceSharesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourceSharesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceSharesRequest, GetResourceSharesResponse>()
                            .withOperationName("GetResourceShares").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourceSharesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourceSharesRequest));
            CompletableFuture<GetResourceSharesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resources in a resource share that is shared with you but for which the invitation is still
     * <code>PENDING</code>. That means that you haven't accepted or rejected the invitation and the invitation hasn't
     * expired.
     * </p>
     *
     * @param listPendingInvitationResourcesRequest
     * @return A Java Future containing the result of the ListPendingInvitationResources operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>InvalidNextTokenException The operation failed because the specified value for <code>NextToken</code>
     *         isn't valid. You must specify a value you received in the <code>NextToken</code> response of a previous
     *         call to this operation.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>ResourceShareInvitationArnNotFoundException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> for an invitation was not found.</li>
     *         <li>MissingRequiredParameterException The operation failed because a required input parameter is missing.
     *         </li>
     *         <li>ResourceShareInvitationAlreadyRejectedException The operation failed because the specified invitation
     *         was already rejected.</li>
     *         <li>ResourceShareInvitationExpiredException The operation failed because the specified invitation is past
     *         its expiration date and time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListPendingInvitationResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPendingInvitationResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPendingInvitationResourcesResponse> listPendingInvitationResources(
            ListPendingInvitationResourcesRequest listPendingInvitationResourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPendingInvitationResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listPendingInvitationResourcesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPendingInvitationResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPendingInvitationResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPendingInvitationResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPendingInvitationResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPendingInvitationResourcesRequest, ListPendingInvitationResourcesResponse>()
                            .withOperationName("ListPendingInvitationResources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPendingInvitationResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPendingInvitationResourcesRequest));
            CompletableFuture<ListPendingInvitationResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists information about the managed permission and its associations to any resource shares that use this managed
     * permission. This lets you see which resource shares use which versions of the specified managed permission.
     * </p>
     *
     * @param listPermissionAssociationsRequest
     * @return A Java Future containing the result of the ListPermissionAssociations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>InvalidNextTokenException The operation failed because the specified value for <code>NextToken</code>
     *         isn't valid. You must specify a value you received in the <code>NextToken</code> response of a previous
     *         call to this operation.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListPermissionAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPermissionAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPermissionAssociationsResponse> listPermissionAssociations(
            ListPermissionAssociationsRequest listPermissionAssociationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPermissionAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPermissionAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPermissionAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPermissionAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPermissionAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPermissionAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPermissionAssociationsRequest, ListPermissionAssociationsResponse>()
                            .withOperationName("ListPermissionAssociations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPermissionAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPermissionAssociationsRequest));
            CompletableFuture<ListPermissionAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the available versions of the specified RAM permission.
     * </p>
     *
     * @param listPermissionVersionsRequest
     * @return A Java Future containing the result of the ListPermissionVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>InvalidNextTokenException The operation failed because the specified value for <code>NextToken</code>
     *         isn't valid. You must specify a value you received in the <code>NextToken</code> response of a previous
     *         call to this operation.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListPermissionVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPermissionVersions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPermissionVersionsResponse> listPermissionVersions(
            ListPermissionVersionsRequest listPermissionVersionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPermissionVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPermissionVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPermissionVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPermissionVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPermissionVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPermissionVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPermissionVersionsRequest, ListPermissionVersionsResponse>()
                            .withOperationName("ListPermissionVersions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPermissionVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPermissionVersionsRequest));
            CompletableFuture<ListPermissionVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of available RAM permissions that you can use for the supported resource types.
     * </p>
     *
     * @param listPermissionsRequest
     * @return A Java Future containing the result of the ListPermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>InvalidNextTokenException The operation failed because the specified value for <code>NextToken</code>
     *         isn't valid. You must specify a value you received in the <code>NextToken</code> response of a previous
     *         call to this operation.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPermissions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPermissionsResponse> listPermissions(ListPermissionsRequest listPermissionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPermissionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPermissionsRequest, ListPermissionsResponse>()
                            .withOperationName("ListPermissions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPermissionsRequest));
            CompletableFuture<ListPermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the principals that you are sharing resources with or that are sharing resources with you.
     * </p>
     *
     * @param listPrincipalsRequest
     * @return A Java Future containing the result of the ListPrincipals operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>InvalidNextTokenException The operation failed because the specified value for <code>NextToken</code>
     *         isn't valid. You must specify a value you received in the <code>NextToken</code> response of a previous
     *         call to this operation.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListPrincipals
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListPrincipals" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPrincipalsResponse> listPrincipals(ListPrincipalsRequest listPrincipalsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPrincipalsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPrincipalsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPrincipals");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPrincipalsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPrincipalsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPrincipalsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPrincipalsRequest, ListPrincipalsResponse>()
                            .withOperationName("ListPrincipals").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPrincipalsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPrincipalsRequest));
            CompletableFuture<ListPrincipalsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the current status of the asynchronous tasks performed by RAM when you perform the
     * <a>ReplacePermissionAssociationsWork</a> operation.
     * </p>
     *
     * @param listReplacePermissionAssociationsWorkRequest
     * @return A Java Future containing the result of the ListReplacePermissionAssociationsWork operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>InvalidNextTokenException The operation failed because the specified value for <code>NextToken</code>
     *         isn't valid. You must specify a value you received in the <code>NextToken</code> response of a previous
     *         call to this operation.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListReplacePermissionAssociationsWork
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListReplacePermissionAssociationsWork"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListReplacePermissionAssociationsWorkResponse> listReplacePermissionAssociationsWork(
            ListReplacePermissionAssociationsWorkRequest listReplacePermissionAssociationsWorkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listReplacePermissionAssociationsWorkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listReplacePermissionAssociationsWorkRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReplacePermissionAssociationsWork");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReplacePermissionAssociationsWorkResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListReplacePermissionAssociationsWorkResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReplacePermissionAssociationsWorkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReplacePermissionAssociationsWorkRequest, ListReplacePermissionAssociationsWorkResponse>()
                            .withOperationName("ListReplacePermissionAssociationsWork").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListReplacePermissionAssociationsWorkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listReplacePermissionAssociationsWorkRequest));
            CompletableFuture<ListReplacePermissionAssociationsWorkResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the RAM permissions that are associated with a resource share.
     * </p>
     *
     * @param listResourceSharePermissionsRequest
     * @return A Java Future containing the result of the ListResourceSharePermissions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>InvalidNextTokenException The operation failed because the specified value for <code>NextToken</code>
     *         isn't valid. You must specify a value you received in the <code>NextToken</code> response of a previous
     *         call to this operation.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListResourceSharePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResourceSharePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourceSharePermissionsResponse> listResourceSharePermissions(
            ListResourceSharePermissionsRequest listResourceSharePermissionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourceSharePermissionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceSharePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceSharePermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourceSharePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResourceSharePermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourceSharePermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourceSharePermissionsRequest, ListResourceSharePermissionsResponse>()
                            .withOperationName("ListResourceSharePermissions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListResourceSharePermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listResourceSharePermissionsRequest));
            CompletableFuture<ListResourceSharePermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resource types that can be shared by RAM.
     * </p>
     *
     * @param listResourceTypesRequest
     * @return A Java Future containing the result of the ListResourceTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidNextTokenException The operation failed because the specified value for <code>NextToken</code>
     *         isn't valid. You must specify a value you received in the <code>NextToken</code> response of a previous
     *         call to this operation.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListResourceTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResourceTypes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourceTypesResponse> listResourceTypes(ListResourceTypesRequest listResourceTypesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourceTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceTypes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourceTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResourceTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourceTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourceTypesRequest, ListResourceTypesResponse>()
                            .withOperationName("ListResourceTypes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListResourceTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listResourceTypesRequest));
            CompletableFuture<ListResourceTypesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resources that you added to a resource share or the resources that are shared with you.
     * </p>
     *
     * @param listResourcesRequest
     * @return A Java Future containing the result of the ListResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidResourceTypeException The operation failed because the specified resource type isn't valid.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>InvalidNextTokenException The operation failed because the specified value for <code>NextToken</code>
     *         isn't valid. You must specify a value you received in the <code>NextToken</code> response of a previous
     *         call to this operation.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ListResources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourcesResponse> listResources(ListResourcesRequest listResourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourcesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourcesRequest, ListResourcesResponse>()
                            .withOperationName("ListResources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listResourcesRequest));
            CompletableFuture<ListResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * When you attach a resource-based policy to a resource, RAM automatically creates a resource share of
     * <code>featureSet</code>=<code>CREATED_FROM_POLICY</code> with a managed permission that has the same IAM
     * permissions as the original resource-based policy. However, this type of managed permission is visible to only
     * the resource share owner, and the associated resource share can't be modified by using RAM.
     * </p>
     * <p>
     * This operation creates a separate, fully manageable customer managed permission that has the same IAM permissions
     * as the original resource-based policy. You can associate this customer managed permission to any resource shares.
     * </p>
     * <p>
     * Before you use <a>PromoteResourceShareCreatedFromPolicy</a>, you should first run this operation to ensure that
     * you have an appropriate customer managed permission that can be associated with the promoted resource share.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * The original <code>CREATED_FROM_POLICY</code> policy isn't deleted, and resource shares using that original
     * policy aren't automatically updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can't modify a <code>CREATED_FROM_POLICY</code> resource share so you can't associate the new customer
     * managed permission by using <code>ReplacePermsissionAssociations</code>. However, if you use
     * <a>PromoteResourceShareCreatedFromPolicy</a>, that operation automatically associates the fully manageable
     * customer managed permission to the newly promoted <code>STANDARD</code> resource share.
     * </p>
     * </li>
     * <li>
     * <p>
     * After you promote a resource share, if the original <code>CREATED_FROM_POLICY</code> managed permission has no
     * other associations to A resource share, then RAM automatically deletes it.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param promotePermissionCreatedFromPolicyRequest
     * @return A Java Future containing the result of the PromotePermissionCreatedFromPolicy operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>MissingRequiredParameterException The operation failed because a required input parameter is missing.
     *         </li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.PromotePermissionCreatedFromPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/PromotePermissionCreatedFromPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PromotePermissionCreatedFromPolicyResponse> promotePermissionCreatedFromPolicy(
            PromotePermissionCreatedFromPolicyRequest promotePermissionCreatedFromPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(promotePermissionCreatedFromPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                promotePermissionCreatedFromPolicyRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PromotePermissionCreatedFromPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PromotePermissionCreatedFromPolicyResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PromotePermissionCreatedFromPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PromotePermissionCreatedFromPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PromotePermissionCreatedFromPolicyRequest, PromotePermissionCreatedFromPolicyResponse>()
                            .withOperationName("PromotePermissionCreatedFromPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PromotePermissionCreatedFromPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(promotePermissionCreatedFromPolicyRequest));
            CompletableFuture<PromotePermissionCreatedFromPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * When you attach a resource-based policy to a resource, RAM automatically creates a resource share of
     * <code>featureSet</code>=<code>CREATED_FROM_POLICY</code> with a managed permission that has the same IAM
     * permissions as the original resource-based policy. However, this type of managed permission is visible to only
     * the resource share owner, and the associated resource share can't be modified by using RAM.
     * </p>
     * <p>
     * This operation promotes the resource share to a <code>STANDARD</code> resource share that is fully manageable in
     * RAM. When you promote a resource share, you can then manage the resource share in RAM and it becomes visible to
     * all of the principals you shared it with.
     * </p>
     * <important>
     * <p>
     * Before you perform this operation, you should first run <a>PromotePermissionCreatedFromPolicy</a>to ensure that
     * you have an appropriate customer managed permission that can be associated with this resource share after its is
     * promoted. If this operation can't find a managed permission that exactly matches the existing
     * <code>CREATED_FROM_POLICY</code> permission, then this operation fails.
     * </p>
     * </important>
     *
     * @param promoteResourceShareCreatedFromPolicyRequest
     * @return A Java Future containing the result of the PromoteResourceShareCreatedFromPolicy operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>ResourceShareLimitExceededException The operation failed because it would exceed the limit for
     *         resource shares for your account. To view the limits for your Amazon Web Services account, see the <a
     *         href="https://console.aws.amazon.com/servicequotas/home/services/ram/quotas">RAM page in the Service
     *         Quotas console</a>.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>MissingRequiredParameterException The operation failed because a required input parameter is missing.
     *         </li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>InvalidStateTransitionException The operation failed because the requested operation isn't valid for
     *         the resource share in its current state.</li>
     *         <li>UnmatchedPolicyPermissionException There isn't an existing managed permission defined in RAM that has
     *         the same IAM permissions as the resource-based policy attached to the resource. You should first run
     *         <a>PromotePermissionCreatedFromPolicy</a> to create that managed permission.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.PromoteResourceShareCreatedFromPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/PromoteResourceShareCreatedFromPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PromoteResourceShareCreatedFromPolicyResponse> promoteResourceShareCreatedFromPolicy(
            PromoteResourceShareCreatedFromPolicyRequest promoteResourceShareCreatedFromPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(promoteResourceShareCreatedFromPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                promoteResourceShareCreatedFromPolicyRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PromoteResourceShareCreatedFromPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PromoteResourceShareCreatedFromPolicyResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PromoteResourceShareCreatedFromPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PromoteResourceShareCreatedFromPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PromoteResourceShareCreatedFromPolicyRequest, PromoteResourceShareCreatedFromPolicyResponse>()
                            .withOperationName("PromoteResourceShareCreatedFromPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PromoteResourceShareCreatedFromPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(promoteResourceShareCreatedFromPolicyRequest));
            CompletableFuture<PromoteResourceShareCreatedFromPolicyResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Rejects an invitation to a resource share from another Amazon Web Services account.
     * </p>
     *
     * @param rejectResourceShareInvitationRequest
     * @return A Java Future containing the result of the RejectResourceShareInvitation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>ResourceShareInvitationArnNotFoundException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> for an invitation was not found.</li>
     *         <li>ResourceShareInvitationAlreadyAcceptedException The operation failed because the specified invitation
     *         was already accepted.</li>
     *         <li>ResourceShareInvitationAlreadyRejectedException The operation failed because the specified invitation
     *         was already rejected.</li>
     *         <li>ResourceShareInvitationExpiredException The operation failed because the specified invitation is past
     *         its expiration date and time.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>InvalidClientTokenException The operation failed because the specified client token isn't valid.</li>
     *         <li>IdempotentParameterMismatchException The operation failed because the client token input parameter
     *         matched one that was used with a previous call to the operation, but at least one of the other input
     *         parameters is different from the previous call.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.RejectResourceShareInvitation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/RejectResourceShareInvitation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RejectResourceShareInvitationResponse> rejectResourceShareInvitation(
            RejectResourceShareInvitationRequest rejectResourceShareInvitationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rejectResourceShareInvitationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                rejectResourceShareInvitationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectResourceShareInvitation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RejectResourceShareInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RejectResourceShareInvitationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RejectResourceShareInvitationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RejectResourceShareInvitationRequest, RejectResourceShareInvitationResponse>()
                            .withOperationName("RejectResourceShareInvitation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RejectResourceShareInvitationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(rejectResourceShareInvitationRequest));
            CompletableFuture<RejectResourceShareInvitationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates all resource shares that use a managed permission to a different managed permission. This operation
     * always applies the default version of the target managed permission. You can optionally specify that the update
     * applies to only resource shares that currently use a specified version. This enables you to update to the latest
     * version, without changing the which managed permission is used.
     * </p>
     * <p>
     * You can use this operation to update all of your resource shares to use the current default version of the
     * permission by specifying the same value for the <code>fromPermissionArn</code> and <code>toPermissionArn</code>
     * parameters.
     * </p>
     * <p>
     * You can use the optional <code>fromPermissionVersion</code> parameter to update only those resources that use a
     * specified version of the managed permission to the new managed permission.
     * </p>
     * <important>
     * <p>
     * To successfully perform this operation, you must have permission to update the resource-based policy on all
     * affected resource types.
     * </p>
     * </important>
     *
     * @param replacePermissionAssociationsRequest
     * @return A Java Future containing the result of the ReplacePermissionAssociations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>InvalidClientTokenException The operation failed because the specified client token isn't valid.</li>
     *         <li>IdempotentParameterMismatchException The operation failed because the client token input parameter
     *         matched one that was used with a previous call to the operation, but at least one of the other input
     *         parameters is different from the previous call.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.ReplacePermissionAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/ReplacePermissionAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ReplacePermissionAssociationsResponse> replacePermissionAssociations(
            ReplacePermissionAssociationsRequest replacePermissionAssociationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(replacePermissionAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                replacePermissionAssociationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReplacePermissionAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ReplacePermissionAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ReplacePermissionAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReplacePermissionAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ReplacePermissionAssociationsRequest, ReplacePermissionAssociationsResponse>()
                            .withOperationName("ReplacePermissionAssociations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ReplacePermissionAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(replacePermissionAssociationsRequest));
            CompletableFuture<ReplacePermissionAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Designates the specified version number as the default version for the specified customer managed permission. New
     * resource shares automatically use this new default permission. Existing resource shares continue to use their
     * original permission version, but you can use <a>ReplacePermissionAssociations</a> to update them.
     * </p>
     *
     * @param setDefaultPermissionVersionRequest
     * @return A Java Future containing the result of the SetDefaultPermissionVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>InvalidClientTokenException The operation failed because the specified client token isn't valid.</li>
     *         <li>IdempotentParameterMismatchException The operation failed because the client token input parameter
     *         matched one that was used with a previous call to the operation, but at least one of the other input
     *         parameters is different from the previous call.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.SetDefaultPermissionVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/SetDefaultPermissionVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SetDefaultPermissionVersionResponse> setDefaultPermissionVersion(
            SetDefaultPermissionVersionRequest setDefaultPermissionVersionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setDefaultPermissionVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setDefaultPermissionVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetDefaultPermissionVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SetDefaultPermissionVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SetDefaultPermissionVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SetDefaultPermissionVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetDefaultPermissionVersionRequest, SetDefaultPermissionVersionResponse>()
                            .withOperationName("SetDefaultPermissionVersion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SetDefaultPermissionVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(setDefaultPermissionVersionRequest));
            CompletableFuture<SetDefaultPermissionVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds the specified tag keys and values to a resource share or managed permission. If you choose a resource share,
     * the tags are attached to only the resource share, not to the resources that are in the resource share.
     * </p>
     * <p>
     * The tags on a managed permission are the same for all versions of the managed permission.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>TagLimitExceededException The operation failed because it would exceed the limit for tags for your
     *         Amazon Web Services account.</li>
     *         <li>ResourceArnNotFoundException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> was not found.</li>
     *         <li>TagPolicyViolationException The operation failed because the specified tag key is a reserved word and
     *         can't be used.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tag key and value pairs from the specified resource share or managed permission.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies some of the properties of the specified resource share.
     * </p>
     *
     * @param updateResourceShareRequest
     * @return A Java Future containing the result of the UpdateResourceShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>IdempotentParameterMismatchException The operation failed because the client token input parameter
     *         matched one that was used with a previous call to the operation, but at least one of the other input
     *         parameters is different from the previous call.</li>
     *         <li>MissingRequiredParameterException The operation failed because a required input parameter is missing.
     *         </li>
     *         <li>UnknownResourceException The operation failed because a specified resource couldn't be found.</li>
     *         <li>MalformedArnException The operation failed because the specified <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> has a format that isn't valid.</li>
     *         <li>InvalidClientTokenException The operation failed because the specified client token isn't valid.</li>
     *         <li>InvalidParameterException The operation failed because a parameter you specified isn't valid.</li>
     *         <li>OperationNotPermittedException The operation failed because the requested operation isn't permitted.</li>
     *         <li>ServerInternalException The operation failed because the service could not respond to the request due
     *         to an internal problem. Try again later.</li>
     *         <li>ServiceUnavailableException The operation failed because the service isn't available. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RamException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RamAsyncClient.UpdateResourceShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ram-2018-01-04/UpdateResourceShare" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateResourceShareResponse> updateResourceShare(
            UpdateResourceShareRequest updateResourceShareRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateResourceShareRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResourceShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResourceShare");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateResourceShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateResourceShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateResourceShareResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateResourceShareRequest, UpdateResourceShareResponse>()
                            .withOperationName("UpdateResourceShare").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateResourceShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateResourceShareRequest));
            CompletableFuture<UpdateResourceShareResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final RamServiceClientConfiguration serviceClientConfiguration() {
        return new RamServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RamException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidClientTokenException")
                                .exceptionBuilderSupplier(InvalidClientTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerInternalException")
                                .exceptionBuilderSupplier(ServerInternalException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceTypeException")
                                .exceptionBuilderSupplier(InvalidResourceTypeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationExpiredException")
                                .exceptionBuilderSupplier(ResourceShareInvitationExpiredException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingRequiredParameterException")
                                .exceptionBuilderSupplier(MissingRequiredParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationNotPermittedException")
                                .exceptionBuilderSupplier(OperationNotPermittedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareLimitExceededException")
                                .exceptionBuilderSupplier(ResourceShareLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PermissionAlreadyExistsException")
                                .exceptionBuilderSupplier(PermissionAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnknownResourceException")
                                .exceptionBuilderSupplier(UnknownResourceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PermissionLimitExceededException")
                                .exceptionBuilderSupplier(PermissionLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPolicyException")
                                .exceptionBuilderSupplier(InvalidPolicyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PermissionVersionsLimitExceededException")
                                .exceptionBuilderSupplier(PermissionVersionsLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                                .exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidMaxResultsException")
                                .exceptionBuilderSupplier(InvalidMaxResultsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagLimitExceededException")
                                .exceptionBuilderSupplier(TagLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceArnNotFoundException")
                                .exceptionBuilderSupplier(ResourceArnNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnmatchedPolicyPermissionException")
                                .exceptionBuilderSupplier(UnmatchedPolicyPermissionException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationAlreadyRejectedException")
                                .exceptionBuilderSupplier(ResourceShareInvitationAlreadyRejectedException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MalformedArnException")
                                .exceptionBuilderSupplier(MalformedArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStateTransitionException")
                                .exceptionBuilderSupplier(InvalidStateTransitionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationAlreadyAcceptedException")
                                .exceptionBuilderSupplier(ResourceShareInvitationAlreadyAcceptedException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceShareInvitationArnNotFoundException")
                                .exceptionBuilderSupplier(ResourceShareInvitationArnNotFoundException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MalformedPolicyTemplateException")
                                .exceptionBuilderSupplier(MalformedPolicyTemplateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagPolicyViolationException")
                                .exceptionBuilderSupplier(TagPolicyViolationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        RamServiceClientConfigurationBuilder serviceConfigBuilder = new RamServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
