/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ram.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about an RAM permission.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceSharePermissionSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<ResourceSharePermissionSummary.Builder, ResourceSharePermissionSummary> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(ResourceSharePermissionSummary::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("version")
            .getter(getter(ResourceSharePermissionSummary::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<Boolean> DEFAULT_VERSION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("defaultVersion").getter(getter(ResourceSharePermissionSummary::defaultVersion))
            .setter(setter(Builder::defaultVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("defaultVersion").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(ResourceSharePermissionSummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceType").getter(getter(ResourceSharePermissionSummary::resourceType))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(ResourceSharePermissionSummary::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("creationTime").getter(getter(ResourceSharePermissionSummary::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdatedTime").getter(getter(ResourceSharePermissionSummary::lastUpdatedTime))
            .setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedTime").build()).build();

    private static final SdkField<Boolean> IS_RESOURCE_TYPE_DEFAULT_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("isResourceTypeDefault").getter(getter(ResourceSharePermissionSummary::isResourceTypeDefault))
            .setter(setter(Builder::isResourceTypeDefault))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("isResourceTypeDefault").build())
            .build();

    private static final SdkField<String> PERMISSION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("permissionType").getter(getter(ResourceSharePermissionSummary::permissionTypeAsString))
            .setter(setter(Builder::permissionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("permissionType").build()).build();

    private static final SdkField<String> FEATURE_SET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("featureSet").getter(getter(ResourceSharePermissionSummary::featureSetAsString))
            .setter(setter(Builder::featureSet))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("featureSet").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(ResourceSharePermissionSummary::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, VERSION_FIELD,
            DEFAULT_VERSION_FIELD, NAME_FIELD, RESOURCE_TYPE_FIELD, STATUS_FIELD, CREATION_TIME_FIELD, LAST_UPDATED_TIME_FIELD,
            IS_RESOURCE_TYPE_DEFAULT_FIELD, PERMISSION_TYPE_FIELD, FEATURE_SET_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String version;

    private final Boolean defaultVersion;

    private final String name;

    private final String resourceType;

    private final String status;

    private final Instant creationTime;

    private final Instant lastUpdatedTime;

    private final Boolean isResourceTypeDefault;

    private final String permissionType;

    private final String featureSet;

    private final List<Tag> tags;

    private ResourceSharePermissionSummary(BuilderImpl builder) {
        this.arn = builder.arn;
        this.version = builder.version;
        this.defaultVersion = builder.defaultVersion;
        this.name = builder.name;
        this.resourceType = builder.resourceType;
        this.status = builder.status;
        this.creationTime = builder.creationTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
        this.isResourceTypeDefault = builder.isResourceTypeDefault;
        this.permissionType = builder.permissionType;
        this.featureSet = builder.featureSet;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     * (ARN)</a> of the permission you want information about.
     * </p>
     * 
     * @return The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource
     *         Name (ARN)</a> of the permission you want information about.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The version of the permission associated with this resource share.
     * </p>
     * 
     * @return The version of the permission associated with this resource share.
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * Specifies whether the version of the managed permission used by this resource share is the default version for
     * this managed permission.
     * </p>
     * 
     * @return Specifies whether the version of the managed permission used by this resource share is the default
     *         version for this managed permission.
     */
    public final Boolean defaultVersion() {
        return defaultVersion;
    }

    /**
     * <p>
     * The name of this managed permission.
     * </p>
     * 
     * @return The name of this managed permission.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The type of resource to which this permission applies. This takes the form of: <code>service-code</code>:
     * <code>resource-code</code>, and is case-insensitive. For example, an Amazon EC2 Subnet would be represented by
     * the string <code>ec2:subnet</code>.
     * </p>
     * 
     * @return The type of resource to which this permission applies. This takes the form of: <code>service-code</code>:
     *         <code>resource-code</code>, and is case-insensitive. For example, an Amazon EC2 Subnet would be
     *         represented by the string <code>ec2:subnet</code>.
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * <p>
     * The current status of the permission.
     * </p>
     * 
     * @return The current status of the permission.
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * The date and time when the permission was created.
     * </p>
     * 
     * @return The date and time when the permission was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The date and time when the permission was last updated.
     * </p>
     * 
     * @return The date and time when the permission was last updated.
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    /**
     * <p>
     * Specifies whether the managed permission associated with this resource share is the default managed permission
     * for all resources of this resource type.
     * </p>
     * 
     * @return Specifies whether the managed permission associated with this resource share is the default managed
     *         permission for all resources of this resource type.
     */
    public final Boolean isResourceTypeDefault() {
        return isResourceTypeDefault;
    }

    /**
     * <p>
     * The type of managed permission. This can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS_MANAGED</code> – Amazon Web Services created and manages this managed permission. You can associate it
     * with your resource shares, but you can't modify it.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CUSTOMER_MANAGED</code> – You, or another principal in your account created this managed permission. You
     * can associate it with your resource shares and create new versions that have different permissions.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permissionType}
     * will return {@link PermissionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #permissionTypeAsString}.
     * </p>
     * 
     * @return The type of managed permission. This can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AWS_MANAGED</code> – Amazon Web Services created and manages this managed permission. You can
     *         associate it with your resource shares, but you can't modify it.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CUSTOMER_MANAGED</code> – You, or another principal in your account created this managed
     *         permission. You can associate it with your resource shares and create new versions that have different
     *         permissions.
     *         </p>
     *         </li>
     * @see PermissionType
     */
    public final PermissionType permissionType() {
        return PermissionType.fromValue(permissionType);
    }

    /**
     * <p>
     * The type of managed permission. This can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS_MANAGED</code> – Amazon Web Services created and manages this managed permission. You can associate it
     * with your resource shares, but you can't modify it.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CUSTOMER_MANAGED</code> – You, or another principal in your account created this managed permission. You
     * can associate it with your resource shares and create new versions that have different permissions.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permissionType}
     * will return {@link PermissionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #permissionTypeAsString}.
     * </p>
     * 
     * @return The type of managed permission. This can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AWS_MANAGED</code> – Amazon Web Services created and manages this managed permission. You can
     *         associate it with your resource shares, but you can't modify it.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CUSTOMER_MANAGED</code> – You, or another principal in your account created this managed
     *         permission. You can associate it with your resource shares and create new versions that have different
     *         permissions.
     *         </p>
     *         </li>
     * @see PermissionType
     */
    public final String permissionTypeAsString() {
        return permissionType;
    }

    /**
     * <p>
     * Indicates what features are available for this resource share. This parameter can have one of the following
     * values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible to all
     * principals you share the resource share with. You can modify these resource shares in RAM using the console or
     * APIs. This resource share might have been created by RAM, or it might have been <b>CREATED_FROM_POLICY</b> and
     * then promoted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based policy. That
     * policy did not match any existing managed permissions, so RAM created this customer managed permission
     * automatically on the customer's behalf based on the attached policy document. This type of resource share is
     * visible only to the Amazon Web Services account that created it. You can't modify it in RAM unless you promote
     * it. For more information, see <a>PromoteResourceShareCreatedFromPolicy</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>, but the
     * customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in progress. This value
     * changes to <code>STANDARD</code> when complete.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #featureSet} will
     * return {@link PermissionFeatureSet#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #featureSetAsString}.
     * </p>
     * 
     * @return Indicates what features are available for this resource share. This parameter can have one of the
     *         following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible to
     *         all principals you share the resource share with. You can modify these resource shares in RAM using the
     *         console or APIs. This resource share might have been created by RAM, or it might have been
     *         <b>CREATED_FROM_POLICY</b> and then promoted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based
     *         policy. That policy did not match any existing managed permissions, so RAM created this customer managed
     *         permission automatically on the customer's behalf based on the attached policy document. This type of
     *         resource share is visible only to the Amazon Web Services account that created it. You can't modify it in
     *         RAM unless you promote it. For more information, see <a>PromoteResourceShareCreatedFromPolicy</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>, but
     *         the customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in
     *         progress. This value changes to <code>STANDARD</code> when complete.
     *         </p>
     *         </li>
     * @see PermissionFeatureSet
     */
    public final PermissionFeatureSet featureSet() {
        return PermissionFeatureSet.fromValue(featureSet);
    }

    /**
     * <p>
     * Indicates what features are available for this resource share. This parameter can have one of the following
     * values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible to all
     * principals you share the resource share with. You can modify these resource shares in RAM using the console or
     * APIs. This resource share might have been created by RAM, or it might have been <b>CREATED_FROM_POLICY</b> and
     * then promoted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based policy. That
     * policy did not match any existing managed permissions, so RAM created this customer managed permission
     * automatically on the customer's behalf based on the attached policy document. This type of resource share is
     * visible only to the Amazon Web Services account that created it. You can't modify it in RAM unless you promote
     * it. For more information, see <a>PromoteResourceShareCreatedFromPolicy</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>, but the
     * customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in progress. This value
     * changes to <code>STANDARD</code> when complete.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #featureSet} will
     * return {@link PermissionFeatureSet#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #featureSetAsString}.
     * </p>
     * 
     * @return Indicates what features are available for this resource share. This parameter can have one of the
     *         following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible to
     *         all principals you share the resource share with. You can modify these resource shares in RAM using the
     *         console or APIs. This resource share might have been created by RAM, or it might have been
     *         <b>CREATED_FROM_POLICY</b> and then promoted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based
     *         policy. That policy did not match any existing managed permissions, so RAM created this customer managed
     *         permission automatically on the customer's behalf based on the attached policy document. This type of
     *         resource share is visible only to the Amazon Web Services account that created it. You can't modify it in
     *         RAM unless you promote it. For more information, see <a>PromoteResourceShareCreatedFromPolicy</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>, but
     *         the customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in
     *         progress. This value changes to <code>STANDARD</code> when complete.
     *         </p>
     *         </li>
     * @see PermissionFeatureSet
     */
    public final String featureSetAsString() {
        return featureSet;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the tag key value pairs currently attached to the permission.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of the tag key value pairs currently attached to the permission.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(defaultVersion());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(isResourceTypeDefault());
        hashCode = 31 * hashCode + Objects.hashCode(permissionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(featureSetAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceSharePermissionSummary)) {
            return false;
        }
        ResourceSharePermissionSummary other = (ResourceSharePermissionSummary) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(version(), other.version())
                && Objects.equals(defaultVersion(), other.defaultVersion()) && Objects.equals(name(), other.name())
                && Objects.equals(resourceType(), other.resourceType()) && Objects.equals(status(), other.status())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime())
                && Objects.equals(isResourceTypeDefault(), other.isResourceTypeDefault())
                && Objects.equals(permissionTypeAsString(), other.permissionTypeAsString())
                && Objects.equals(featureSetAsString(), other.featureSetAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceSharePermissionSummary").add("Arn", arn()).add("Version", version())
                .add("DefaultVersion", defaultVersion()).add("Name", name()).add("ResourceType", resourceType())
                .add("Status", status()).add("CreationTime", creationTime()).add("LastUpdatedTime", lastUpdatedTime())
                .add("IsResourceTypeDefault", isResourceTypeDefault()).add("PermissionType", permissionTypeAsString())
                .add("FeatureSet", featureSetAsString()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "defaultVersion":
            return Optional.ofNullable(clazz.cast(defaultVersion()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "status":
            return Optional.ofNullable(clazz.cast(status()));
        case "creationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "lastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        case "isResourceTypeDefault":
            return Optional.ofNullable(clazz.cast(isResourceTypeDefault()));
        case "permissionType":
            return Optional.ofNullable(clazz.cast(permissionTypeAsString()));
        case "featureSet":
            return Optional.ofNullable(clazz.cast(featureSetAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("arn", ARN_FIELD);
        map.put("version", VERSION_FIELD);
        map.put("defaultVersion", DEFAULT_VERSION_FIELD);
        map.put("name", NAME_FIELD);
        map.put("resourceType", RESOURCE_TYPE_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("creationTime", CREATION_TIME_FIELD);
        map.put("lastUpdatedTime", LAST_UPDATED_TIME_FIELD);
        map.put("isResourceTypeDefault", IS_RESOURCE_TYPE_DEFAULT_FIELD);
        map.put("permissionType", PERMISSION_TYPE_FIELD);
        map.put("featureSet", FEATURE_SET_FIELD);
        map.put("tags", TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ResourceSharePermissionSummary, T> g) {
        return obj -> g.apply((ResourceSharePermissionSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceSharePermissionSummary> {
        /**
         * <p>
         * The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
         * (ARN)</a> of the permission you want information about.
         * </p>
         * 
         * @param arn
         *        The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
         *        Resource Name (ARN)</a> of the permission you want information about.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The version of the permission associated with this resource share.
         * </p>
         * 
         * @param version
         *        The version of the permission associated with this resource share.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * Specifies whether the version of the managed permission used by this resource share is the default version
         * for this managed permission.
         * </p>
         * 
         * @param defaultVersion
         *        Specifies whether the version of the managed permission used by this resource share is the default
         *        version for this managed permission.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultVersion(Boolean defaultVersion);

        /**
         * <p>
         * The name of this managed permission.
         * </p>
         * 
         * @param name
         *        The name of this managed permission.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of resource to which this permission applies. This takes the form of: <code>service-code</code>:
         * <code>resource-code</code>, and is case-insensitive. For example, an Amazon EC2 Subnet would be represented
         * by the string <code>ec2:subnet</code>.
         * </p>
         * 
         * @param resourceType
         *        The type of resource to which this permission applies. This takes the form of:
         *        <code>service-code</code>:<code>resource-code</code>, and is case-insensitive. For example, an Amazon
         *        EC2 Subnet would be represented by the string <code>ec2:subnet</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The current status of the permission.
         * </p>
         * 
         * @param status
         *        The current status of the permission.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * The date and time when the permission was created.
         * </p>
         * 
         * @param creationTime
         *        The date and time when the permission was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The date and time when the permission was last updated.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The date and time when the permission was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);

        /**
         * <p>
         * Specifies whether the managed permission associated with this resource share is the default managed
         * permission for all resources of this resource type.
         * </p>
         * 
         * @param isResourceTypeDefault
         *        Specifies whether the managed permission associated with this resource share is the default managed
         *        permission for all resources of this resource type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isResourceTypeDefault(Boolean isResourceTypeDefault);

        /**
         * <p>
         * The type of managed permission. This can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AWS_MANAGED</code> – Amazon Web Services created and manages this managed permission. You can associate
         * it with your resource shares, but you can't modify it.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CUSTOMER_MANAGED</code> – You, or another principal in your account created this managed permission.
         * You can associate it with your resource shares and create new versions that have different permissions.
         * </p>
         * </li>
         * </ul>
         * 
         * @param permissionType
         *        The type of managed permission. This can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AWS_MANAGED</code> – Amazon Web Services created and manages this managed permission. You can
         *        associate it with your resource shares, but you can't modify it.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CUSTOMER_MANAGED</code> – You, or another principal in your account created this managed
         *        permission. You can associate it with your resource shares and create new versions that have different
         *        permissions.
         *        </p>
         *        </li>
         * @see PermissionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionType
         */
        Builder permissionType(String permissionType);

        /**
         * <p>
         * The type of managed permission. This can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AWS_MANAGED</code> – Amazon Web Services created and manages this managed permission. You can associate
         * it with your resource shares, but you can't modify it.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CUSTOMER_MANAGED</code> – You, or another principal in your account created this managed permission.
         * You can associate it with your resource shares and create new versions that have different permissions.
         * </p>
         * </li>
         * </ul>
         * 
         * @param permissionType
         *        The type of managed permission. This can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AWS_MANAGED</code> – Amazon Web Services created and manages this managed permission. You can
         *        associate it with your resource shares, but you can't modify it.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CUSTOMER_MANAGED</code> – You, or another principal in your account created this managed
         *        permission. You can associate it with your resource shares and create new versions that have different
         *        permissions.
         *        </p>
         *        </li>
         * @see PermissionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionType
         */
        Builder permissionType(PermissionType permissionType);

        /**
         * <p>
         * Indicates what features are available for this resource share. This parameter can have one of the following
         * values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible to all
         * principals you share the resource share with. You can modify these resource shares in RAM using the console
         * or APIs. This resource share might have been created by RAM, or it might have been <b>CREATED_FROM_POLICY</b>
         * and then promoted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based policy.
         * That policy did not match any existing managed permissions, so RAM created this customer managed permission
         * automatically on the customer's behalf based on the attached policy document. This type of resource share is
         * visible only to the Amazon Web Services account that created it. You can't modify it in RAM unless you
         * promote it. For more information, see <a>PromoteResourceShareCreatedFromPolicy</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>, but the
         * customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in progress. This
         * value changes to <code>STANDARD</code> when complete.
         * </p>
         * </li>
         * </ul>
         * 
         * @param featureSet
         *        Indicates what features are available for this resource share. This parameter can have one of the
         *        following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible
         *        to all principals you share the resource share with. You can modify these resource shares in RAM using
         *        the console or APIs. This resource share might have been created by RAM, or it might have been
         *        <b>CREATED_FROM_POLICY</b> and then promoted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based
         *        policy. That policy did not match any existing managed permissions, so RAM created this customer
         *        managed permission automatically on the customer's behalf based on the attached policy document. This
         *        type of resource share is visible only to the Amazon Web Services account that created it. You can't
         *        modify it in RAM unless you promote it. For more information, see
         *        <a>PromoteResourceShareCreatedFromPolicy</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>,
         *        but the customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in
         *        progress. This value changes to <code>STANDARD</code> when complete.
         *        </p>
         *        </li>
         * @see PermissionFeatureSet
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionFeatureSet
         */
        Builder featureSet(String featureSet);

        /**
         * <p>
         * Indicates what features are available for this resource share. This parameter can have one of the following
         * values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible to all
         * principals you share the resource share with. You can modify these resource shares in RAM using the console
         * or APIs. This resource share might have been created by RAM, or it might have been <b>CREATED_FROM_POLICY</b>
         * and then promoted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based policy.
         * That policy did not match any existing managed permissions, so RAM created this customer managed permission
         * automatically on the customer's behalf based on the attached policy document. This type of resource share is
         * visible only to the Amazon Web Services account that created it. You can't modify it in RAM unless you
         * promote it. For more information, see <a>PromoteResourceShareCreatedFromPolicy</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>, but the
         * customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in progress. This
         * value changes to <code>STANDARD</code> when complete.
         * </p>
         * </li>
         * </ul>
         * 
         * @param featureSet
         *        Indicates what features are available for this resource share. This parameter can have one of the
         *        following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible
         *        to all principals you share the resource share with. You can modify these resource shares in RAM using
         *        the console or APIs. This resource share might have been created by RAM, or it might have been
         *        <b>CREATED_FROM_POLICY</b> and then promoted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based
         *        policy. That policy did not match any existing managed permissions, so RAM created this customer
         *        managed permission automatically on the customer's behalf based on the attached policy document. This
         *        type of resource share is visible only to the Amazon Web Services account that created it. You can't
         *        modify it in RAM unless you promote it. For more information, see
         *        <a>PromoteResourceShareCreatedFromPolicy</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>,
         *        but the customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in
         *        progress. This value changes to <code>STANDARD</code> when complete.
         *        </p>
         *        </li>
         * @see PermissionFeatureSet
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionFeatureSet
         */
        Builder featureSet(PermissionFeatureSet featureSet);

        /**
         * <p>
         * A list of the tag key value pairs currently attached to the permission.
         * </p>
         * 
         * @param tags
         *        A list of the tag key value pairs currently attached to the permission.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of the tag key value pairs currently attached to the permission.
         * </p>
         * 
         * @param tags
         *        A list of the tag key value pairs currently attached to the permission.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of the tag key value pairs currently attached to the permission.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ram.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ram.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ram.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ram.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String version;

        private Boolean defaultVersion;

        private String name;

        private String resourceType;

        private String status;

        private Instant creationTime;

        private Instant lastUpdatedTime;

        private Boolean isResourceTypeDefault;

        private String permissionType;

        private String featureSet;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceSharePermissionSummary model) {
            arn(model.arn);
            version(model.version);
            defaultVersion(model.defaultVersion);
            name(model.name);
            resourceType(model.resourceType);
            status(model.status);
            creationTime(model.creationTime);
            lastUpdatedTime(model.lastUpdatedTime);
            isResourceTypeDefault(model.isResourceTypeDefault);
            permissionType(model.permissionType);
            featureSet(model.featureSet);
            tags(model.tags);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final Boolean getDefaultVersion() {
            return defaultVersion;
        }

        public final void setDefaultVersion(Boolean defaultVersion) {
            this.defaultVersion = defaultVersion;
        }

        @Override
        public final Builder defaultVersion(Boolean defaultVersion) {
            this.defaultVersion = defaultVersion;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final Boolean getIsResourceTypeDefault() {
            return isResourceTypeDefault;
        }

        public final void setIsResourceTypeDefault(Boolean isResourceTypeDefault) {
            this.isResourceTypeDefault = isResourceTypeDefault;
        }

        @Override
        public final Builder isResourceTypeDefault(Boolean isResourceTypeDefault) {
            this.isResourceTypeDefault = isResourceTypeDefault;
            return this;
        }

        public final String getPermissionType() {
            return permissionType;
        }

        public final void setPermissionType(String permissionType) {
            this.permissionType = permissionType;
        }

        @Override
        public final Builder permissionType(String permissionType) {
            this.permissionType = permissionType;
            return this;
        }

        @Override
        public final Builder permissionType(PermissionType permissionType) {
            this.permissionType(permissionType == null ? null : permissionType.toString());
            return this;
        }

        public final String getFeatureSet() {
            return featureSet;
        }

        public final void setFeatureSet(String featureSet) {
            this.featureSet = featureSet;
        }

        @Override
        public final Builder featureSet(String featureSet) {
            this.featureSet = featureSet;
            return this;
        }

        @Override
        public final Builder featureSet(PermissionFeatureSet featureSet) {
            this.featureSet(featureSet == null ? null : featureSet.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public ResourceSharePermissionSummary build() {
            return new ResourceSharePermissionSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
