/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rbin;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.rbin.internal.RbinServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.rbin.model.ConflictException;
import software.amazon.awssdk.services.rbin.model.CreateRuleRequest;
import software.amazon.awssdk.services.rbin.model.CreateRuleResponse;
import software.amazon.awssdk.services.rbin.model.DeleteRuleRequest;
import software.amazon.awssdk.services.rbin.model.DeleteRuleResponse;
import software.amazon.awssdk.services.rbin.model.GetRuleRequest;
import software.amazon.awssdk.services.rbin.model.GetRuleResponse;
import software.amazon.awssdk.services.rbin.model.InternalServerException;
import software.amazon.awssdk.services.rbin.model.ListRulesRequest;
import software.amazon.awssdk.services.rbin.model.ListRulesResponse;
import software.amazon.awssdk.services.rbin.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.rbin.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.rbin.model.LockRuleRequest;
import software.amazon.awssdk.services.rbin.model.LockRuleResponse;
import software.amazon.awssdk.services.rbin.model.RbinException;
import software.amazon.awssdk.services.rbin.model.ResourceNotFoundException;
import software.amazon.awssdk.services.rbin.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.rbin.model.TagResourceRequest;
import software.amazon.awssdk.services.rbin.model.TagResourceResponse;
import software.amazon.awssdk.services.rbin.model.UnlockRuleRequest;
import software.amazon.awssdk.services.rbin.model.UnlockRuleResponse;
import software.amazon.awssdk.services.rbin.model.UntagResourceRequest;
import software.amazon.awssdk.services.rbin.model.UntagResourceResponse;
import software.amazon.awssdk.services.rbin.model.UpdateRuleRequest;
import software.amazon.awssdk.services.rbin.model.UpdateRuleResponse;
import software.amazon.awssdk.services.rbin.model.ValidationException;
import software.amazon.awssdk.services.rbin.transform.CreateRuleRequestMarshaller;
import software.amazon.awssdk.services.rbin.transform.DeleteRuleRequestMarshaller;
import software.amazon.awssdk.services.rbin.transform.GetRuleRequestMarshaller;
import software.amazon.awssdk.services.rbin.transform.ListRulesRequestMarshaller;
import software.amazon.awssdk.services.rbin.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.rbin.transform.LockRuleRequestMarshaller;
import software.amazon.awssdk.services.rbin.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.rbin.transform.UnlockRuleRequestMarshaller;
import software.amazon.awssdk.services.rbin.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.rbin.transform.UpdateRuleRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link RbinClient}.
 *
 * @see RbinClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRbinClient implements RbinClient {
    private static final Logger log = Logger.loggerFor(DefaultRbinClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRbinClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a Recycle Bin retention rule. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/recycle-bin-working-with-rules.html#recycle-bin-create-rule"
     * > Create Recycle Bin retention rules</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     *
     * @param createRuleRequest
     * @return Result of the CreateRule operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota for the number of tags per resource to be exceeded.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.CreateRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/CreateRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateRuleResponse createRule(CreateRuleRequest createRuleRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, RbinException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRuleRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "rbin");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRule");

            return clientHandler.execute(new ClientExecutionParams<CreateRuleRequest, CreateRuleResponse>()
                    .withOperationName("CreateRule").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createRuleRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a Recycle Bin retention rule. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/recycle-bin-working-with-rules.html#recycle-bin-delete-rule"
     * > Delete Recycle Bin retention rules</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     *
     * @param deleteRuleRequest
     * @return Result of the DeleteRule operation returned by the service.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws ConflictException
     *         The specified retention rule lock request can't be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.DeleteRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/DeleteRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteRuleResponse deleteRule(DeleteRuleRequest deleteRuleRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ConflictException, AwsServiceException, SdkClientException,
            RbinException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRuleRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "rbin");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRule");

            return clientHandler.execute(new ClientExecutionParams<DeleteRuleRequest, DeleteRuleResponse>()
                    .withOperationName("DeleteRule").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteRuleRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a Recycle Bin retention rule.
     * </p>
     *
     * @param getRuleRequest
     * @return Result of the GetRule operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.GetRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/GetRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetRuleResponse getRule(GetRuleRequest getRuleRequest) throws ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, RbinException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRuleRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "rbin");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRule");

            return clientHandler.execute(new ClientExecutionParams<GetRuleRequest, GetRuleResponse>()
                    .withOperationName("GetRule").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getRuleRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the Recycle Bin retention rules in the Region.
     * </p>
     *
     * @param listRulesRequest
     * @return Result of the ListRules operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.ListRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/ListRules" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListRulesResponse listRules(ListRulesRequest listRulesRequest) throws ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, RbinException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRulesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRulesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "rbin");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRules");

            return clientHandler.execute(new ClientExecutionParams<ListRulesRequest, ListRulesResponse>()
                    .withOperationName("ListRules").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listRulesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags assigned to a retention rule.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, RbinException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "rbin");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Locks a retention rule. A locked retention rule can't be modified or deleted.
     * </p>
     *
     * @param lockRuleRequest
     * @return Result of the LockRule operation returned by the service.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws ConflictException
     *         The specified retention rule lock request can't be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.LockRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/LockRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public LockRuleResponse lockRule(LockRuleRequest lockRuleRequest) throws InternalServerException, ResourceNotFoundException,
            ValidationException, ConflictException, AwsServiceException, SdkClientException, RbinException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<LockRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                LockRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(lockRuleRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, lockRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "rbin");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "LockRule");

            return clientHandler.execute(new ClientExecutionParams<LockRuleRequest, LockRuleResponse>()
                    .withOperationName("LockRule").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(lockRuleRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new LockRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns tags to the specified retention rule.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota for the number of tags per resource to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, RbinException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "rbin");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Unlocks a retention rule. After a retention rule is unlocked, it can be modified or deleted only after the unlock
     * delay period expires.
     * </p>
     *
     * @param unlockRuleRequest
     * @return Result of the UnlockRule operation returned by the service.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws ConflictException
     *         The specified retention rule lock request can't be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.UnlockRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/UnlockRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UnlockRuleResponse unlockRule(UnlockRuleRequest unlockRuleRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ConflictException, AwsServiceException, SdkClientException,
            RbinException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UnlockRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UnlockRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(unlockRuleRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, unlockRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "rbin");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UnlockRule");

            return clientHandler.execute(new ClientExecutionParams<UnlockRuleRequest, UnlockRuleResponse>()
                    .withOperationName("UnlockRule").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(unlockRuleRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UnlockRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Unassigns a tag from a retention rule.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, RbinException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "rbin");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing Recycle Bin retention rule. You can update a retention rule's description, resource tags, and
     * retention period at any time after creation. You can't update a retention rule's resource type after creation.
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/recycle-bin-working-with-rules.html#recycle-bin-update-rule"
     * > Update Recycle Bin retention rules</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     *
     * @param updateRuleRequest
     * @return Result of the UpdateRule operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ConflictException
     *         The specified retention rule lock request can't be completed.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota for the number of tags per resource to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.UpdateRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/UpdateRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateRuleResponse updateRule(UpdateRuleRequest updateRuleRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ConflictException, ServiceQuotaExceededException,
            AwsServiceException, SdkClientException, RbinException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRuleRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "rbin");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRule");

            return clientHandler.execute(new ClientExecutionParams<UpdateRuleRequest, UpdateRuleResponse>()
                    .withOperationName("UpdateRule").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateRuleRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        RbinServiceClientConfigurationBuilder serviceConfigBuilder = new RbinServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RbinException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final RbinServiceClientConfiguration serviceClientConfiguration() {
        return new RbinServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
