/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rbin;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.rbin.model.ConflictException;
import software.amazon.awssdk.services.rbin.model.CreateRuleRequest;
import software.amazon.awssdk.services.rbin.model.CreateRuleResponse;
import software.amazon.awssdk.services.rbin.model.DeleteRuleRequest;
import software.amazon.awssdk.services.rbin.model.DeleteRuleResponse;
import software.amazon.awssdk.services.rbin.model.GetRuleRequest;
import software.amazon.awssdk.services.rbin.model.GetRuleResponse;
import software.amazon.awssdk.services.rbin.model.InternalServerException;
import software.amazon.awssdk.services.rbin.model.ListRulesRequest;
import software.amazon.awssdk.services.rbin.model.ListRulesResponse;
import software.amazon.awssdk.services.rbin.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.rbin.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.rbin.model.LockRuleRequest;
import software.amazon.awssdk.services.rbin.model.LockRuleResponse;
import software.amazon.awssdk.services.rbin.model.RbinException;
import software.amazon.awssdk.services.rbin.model.ResourceNotFoundException;
import software.amazon.awssdk.services.rbin.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.rbin.model.TagResourceRequest;
import software.amazon.awssdk.services.rbin.model.TagResourceResponse;
import software.amazon.awssdk.services.rbin.model.UnlockRuleRequest;
import software.amazon.awssdk.services.rbin.model.UnlockRuleResponse;
import software.amazon.awssdk.services.rbin.model.UntagResourceRequest;
import software.amazon.awssdk.services.rbin.model.UntagResourceResponse;
import software.amazon.awssdk.services.rbin.model.UpdateRuleRequest;
import software.amazon.awssdk.services.rbin.model.UpdateRuleResponse;
import software.amazon.awssdk.services.rbin.model.ValidationException;
import software.amazon.awssdk.services.rbin.paginators.ListRulesIterable;

/**
 * Service client for accessing Amazon Recycle Bin. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * This is the <i>Recycle Bin API Reference</i>. This documentation provides descriptions and syntax for each of the
 * actions and data types in Recycle Bin.
 * </p>
 * <p>
 * Recycle Bin is a resource recovery feature that enables you to restore accidentally deleted snapshots and EBS-backed
 * AMIs. When using Recycle Bin, if your resources are deleted, they are retained in the Recycle Bin for a time period
 * that you specify.
 * </p>
 * <p>
 * You can restore a resource from the Recycle Bin at any time before its retention period expires. After you restore a
 * resource from the Recycle Bin, the resource is removed from the Recycle Bin, and you can then use it in the same way
 * you use any other resource of that type in your account. If the retention period expires and the resource is not
 * restored, the resource is permanently deleted from the Recycle Bin and is no longer available for recovery. For more
 * information about Recycle Bin, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/recycle-bin.html">
 * Recycle Bin</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface RbinClient extends AwsClient {
    String SERVICE_NAME = "rbin";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "rbin";

    /**
     * <p>
     * Creates a Recycle Bin retention rule. You can create two types of retention rules:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Tag-level retention rules</b> - These retention rules use resource tags to identify the resources to protect.
     * For each retention rule, you specify one or more tag key and value pairs. Resources (of the specified type) that
     * have at least one of these tag key and value pairs are automatically retained in the Recycle Bin upon deletion.
     * Use this type of retention rule to protect specific resources in your account based on their tags.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Region-level retention rules</b> - These retention rules, by default, apply to all of the resources (of the
     * specified type) in the Region, even if the resources are not tagged. However, you can specify exclusion tags to
     * exclude resources that have specific tags. Use this type of retention rule to protect all resources of a specific
     * type in a Region.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/ebs/latest/userguide/recycle-bin.html"> Create
     * Recycle Bin retention rules</a> in the <i>Amazon EBS User Guide</i>.
     * </p>
     *
     * @param createRuleRequest
     * @return Result of the CreateRule operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota for the number of tags per resource to be exceeded.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.CreateRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/CreateRule" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateRuleResponse createRule(CreateRuleRequest createRuleRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, RbinException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a Recycle Bin retention rule. You can create two types of retention rules:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Tag-level retention rules</b> - These retention rules use resource tags to identify the resources to protect.
     * For each retention rule, you specify one or more tag key and value pairs. Resources (of the specified type) that
     * have at least one of these tag key and value pairs are automatically retained in the Recycle Bin upon deletion.
     * Use this type of retention rule to protect specific resources in your account based on their tags.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Region-level retention rules</b> - These retention rules, by default, apply to all of the resources (of the
     * specified type) in the Region, even if the resources are not tagged. However, you can specify exclusion tags to
     * exclude resources that have specific tags. Use this type of retention rule to protect all resources of a specific
     * type in a Region.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/ebs/latest/userguide/recycle-bin.html"> Create
     * Recycle Bin retention rules</a> in the <i>Amazon EBS User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateRuleRequest.Builder} avoiding the need to
     * create one manually via {@link CreateRuleRequest#builder()}
     * </p>
     *
     * @param createRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.CreateRuleRequest.Builder} to create a request.
     * @return Result of the CreateRule operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota for the number of tags per resource to be exceeded.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.CreateRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/CreateRule" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateRuleResponse createRule(Consumer<CreateRuleRequest.Builder> createRuleRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, RbinException {
        return createRule(CreateRuleRequest.builder().applyMutation(createRuleRequest).build());
    }

    /**
     * <p>
     * Deletes a Recycle Bin retention rule. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/recycle-bin-working-with-rules.html#recycle-bin-delete-rule"
     * > Delete Recycle Bin retention rules</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     *
     * @param deleteRuleRequest
     * @return Result of the DeleteRule operation returned by the service.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws ConflictException
     *         The specified retention rule lock request can't be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.DeleteRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/DeleteRule" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteRuleResponse deleteRule(DeleteRuleRequest deleteRuleRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ConflictException, AwsServiceException, SdkClientException,
            RbinException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a Recycle Bin retention rule. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/recycle-bin-working-with-rules.html#recycle-bin-delete-rule"
     * > Delete Recycle Bin retention rules</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteRuleRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteRuleRequest#builder()}
     * </p>
     *
     * @param deleteRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.DeleteRuleRequest.Builder} to create a request.
     * @return Result of the DeleteRule operation returned by the service.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws ConflictException
     *         The specified retention rule lock request can't be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.DeleteRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/DeleteRule" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteRuleResponse deleteRule(Consumer<DeleteRuleRequest.Builder> deleteRuleRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ConflictException, AwsServiceException, SdkClientException,
            RbinException {
        return deleteRule(DeleteRuleRequest.builder().applyMutation(deleteRuleRequest).build());
    }

    /**
     * <p>
     * Gets information about a Recycle Bin retention rule.
     * </p>
     *
     * @param getRuleRequest
     * @return Result of the GetRule operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.GetRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/GetRule" target="_top">AWS API
     *      Documentation</a>
     */
    default GetRuleResponse getRule(GetRuleRequest getRuleRequest) throws ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, RbinException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets information about a Recycle Bin retention rule.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetRuleRequest.Builder} avoiding the need to create
     * one manually via {@link GetRuleRequest#builder()}
     * </p>
     *
     * @param getRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.GetRuleRequest.Builder} to create a request.
     * @return Result of the GetRule operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.GetRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/GetRule" target="_top">AWS API
     *      Documentation</a>
     */
    default GetRuleResponse getRule(Consumer<GetRuleRequest.Builder> getRuleRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, RbinException {
        return getRule(GetRuleRequest.builder().applyMutation(getRuleRequest).build());
    }

    /**
     * <p>
     * Lists the Recycle Bin retention rules in the Region.
     * </p>
     *
     * @param listRulesRequest
     * @return Result of the ListRules operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.ListRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/ListRules" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRulesResponse listRules(ListRulesRequest listRulesRequest) throws ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, RbinException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the Recycle Bin retention rules in the Region.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRulesRequest.Builder} avoiding the need to
     * create one manually via {@link ListRulesRequest#builder()}
     * </p>
     *
     * @param listRulesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.ListRulesRequest.Builder} to create a request.
     * @return Result of the ListRules operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.ListRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/ListRules" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRulesResponse listRules(Consumer<ListRulesRequest.Builder> listRulesRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, RbinException {
        return listRules(ListRulesRequest.builder().applyMutation(listRulesRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #listRules(software.amazon.awssdk.services.rbin.model.ListRulesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rbin.paginators.ListRulesIterable responses = client.listRulesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rbin.paginators.ListRulesIterable responses = client.listRulesPaginator(request);
     *     for (software.amazon.awssdk.services.rbin.model.ListRulesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rbin.paginators.ListRulesIterable responses = client.listRulesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRules(software.amazon.awssdk.services.rbin.model.ListRulesRequest)} operation.</b>
     * </p>
     *
     * @param listRulesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.ListRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/ListRules" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRulesIterable listRulesPaginator(ListRulesRequest listRulesRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, RbinException {
        return new ListRulesIterable(this, listRulesRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listRules(software.amazon.awssdk.services.rbin.model.ListRulesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rbin.paginators.ListRulesIterable responses = client.listRulesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rbin.paginators.ListRulesIterable responses = client.listRulesPaginator(request);
     *     for (software.amazon.awssdk.services.rbin.model.ListRulesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rbin.paginators.ListRulesIterable responses = client.listRulesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRules(software.amazon.awssdk.services.rbin.model.ListRulesRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRulesRequest.Builder} avoiding the need to
     * create one manually via {@link ListRulesRequest#builder()}
     * </p>
     *
     * @param listRulesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.ListRulesRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.ListRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/ListRules" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRulesIterable listRulesPaginator(Consumer<ListRulesRequest.Builder> listRulesRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, RbinException {
        return listRulesPaginator(ListRulesRequest.builder().applyMutation(listRulesRequest).build());
    }

    /**
     * <p>
     * Lists the tags assigned to a retention rule.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, RbinException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the tags assigned to a retention rule.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.ListTagsForResourceRequest.Builder} to create a request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, RbinException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Locks a Region-level retention rule. A locked retention rule can't be modified or deleted.
     * </p>
     * <note>
     * <p>
     * You can't lock tag-level retention rules, or Region-level retention rules that have exclusion tags.
     * </p>
     * </note>
     *
     * @param lockRuleRequest
     * @return Result of the LockRule operation returned by the service.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws ConflictException
     *         The specified retention rule lock request can't be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.LockRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/LockRule" target="_top">AWS API
     *      Documentation</a>
     */
    default LockRuleResponse lockRule(LockRuleRequest lockRuleRequest) throws InternalServerException, ResourceNotFoundException,
            ValidationException, ConflictException, AwsServiceException, SdkClientException, RbinException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Locks a Region-level retention rule. A locked retention rule can't be modified or deleted.
     * </p>
     * <note>
     * <p>
     * You can't lock tag-level retention rules, or Region-level retention rules that have exclusion tags.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link LockRuleRequest.Builder} avoiding the need to
     * create one manually via {@link LockRuleRequest#builder()}
     * </p>
     *
     * @param lockRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.LockRuleRequest.Builder} to create a request.
     * @return Result of the LockRule operation returned by the service.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws ConflictException
     *         The specified retention rule lock request can't be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.LockRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/LockRule" target="_top">AWS API
     *      Documentation</a>
     */
    default LockRuleResponse lockRule(Consumer<LockRuleRequest.Builder> lockRuleRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ConflictException, AwsServiceException, SdkClientException,
            RbinException {
        return lockRule(LockRuleRequest.builder().applyMutation(lockRuleRequest).build());
    }

    /**
     * <p>
     * Assigns tags to the specified retention rule.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota for the number of tags per resource to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, RbinException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Assigns tags to the specified retention rule.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.TagResourceRequest.Builder} to create a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota for the number of tags per resource to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, RbinException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Unlocks a retention rule. After a retention rule is unlocked, it can be modified or deleted only after the unlock
     * delay period expires.
     * </p>
     *
     * @param unlockRuleRequest
     * @return Result of the UnlockRule operation returned by the service.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws ConflictException
     *         The specified retention rule lock request can't be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.UnlockRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/UnlockRule" target="_top">AWS API
     *      Documentation</a>
     */
    default UnlockRuleResponse unlockRule(UnlockRuleRequest unlockRuleRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ConflictException, AwsServiceException, SdkClientException,
            RbinException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Unlocks a retention rule. After a retention rule is unlocked, it can be modified or deleted only after the unlock
     * delay period expires.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UnlockRuleRequest.Builder} avoiding the need to
     * create one manually via {@link UnlockRuleRequest#builder()}
     * </p>
     *
     * @param unlockRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.UnlockRuleRequest.Builder} to create a request.
     * @return Result of the UnlockRule operation returned by the service.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws ConflictException
     *         The specified retention rule lock request can't be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.UnlockRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/UnlockRule" target="_top">AWS API
     *      Documentation</a>
     */
    default UnlockRuleResponse unlockRule(Consumer<UnlockRuleRequest.Builder> unlockRuleRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ConflictException, AwsServiceException, SdkClientException,
            RbinException {
        return unlockRule(UnlockRuleRequest.builder().applyMutation(unlockRuleRequest).build());
    }

    /**
     * <p>
     * Unassigns a tag from a retention rule.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, RbinException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Unassigns a tag from a retention rule.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.UntagResourceRequest.Builder} to create a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, RbinException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates an existing Recycle Bin retention rule. You can update a retention rule's description, resource tags, and
     * retention period at any time after creation. You can't update a retention rule's resource type after creation.
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/recycle-bin-working-with-rules.html#recycle-bin-update-rule"
     * > Update Recycle Bin retention rules</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     *
     * @param updateRuleRequest
     * @return Result of the UpdateRule operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ConflictException
     *         The specified retention rule lock request can't be completed.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota for the number of tags per resource to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.UpdateRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/UpdateRule" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateRuleResponse updateRule(UpdateRuleRequest updateRuleRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ConflictException, ServiceQuotaExceededException,
            AwsServiceException, SdkClientException, RbinException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates an existing Recycle Bin retention rule. You can update a retention rule's description, resource tags, and
     * retention period at any time after creation. You can't update a retention rule's resource type after creation.
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/recycle-bin-working-with-rules.html#recycle-bin-update-rule"
     * > Update Recycle Bin retention rules</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateRuleRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateRuleRequest#builder()}
     * </p>
     *
     * @param updateRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.UpdateRuleRequest.Builder} to create a request.
     * @return Result of the UpdateRule operation returned by the service.
     * @throws ValidationException
     *         One or more of the parameters in the request is not valid.
     * @throws InternalServerException
     *         The service could not respond to the request due to an internal problem.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ConflictException
     *         The specified retention rule lock request can't be completed.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota for the number of tags per resource to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RbinException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RbinClient.UpdateRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/UpdateRule" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateRuleResponse updateRule(Consumer<UpdateRuleRequest.Builder> updateRuleRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ConflictException, ServiceQuotaExceededException,
            AwsServiceException, SdkClientException, RbinException {
        return updateRule(UpdateRuleRequest.builder().applyMutation(updateRuleRequest).build());
    }

    /**
     * Create a {@link RbinClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static RbinClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link RbinClient}.
     */
    static RbinClientBuilder builder() {
        return new DefaultRbinClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default RbinServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
