/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rbin;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.rbin.model.CreateRuleRequest;
import software.amazon.awssdk.services.rbin.model.CreateRuleResponse;
import software.amazon.awssdk.services.rbin.model.DeleteRuleRequest;
import software.amazon.awssdk.services.rbin.model.DeleteRuleResponse;
import software.amazon.awssdk.services.rbin.model.GetRuleRequest;
import software.amazon.awssdk.services.rbin.model.GetRuleResponse;
import software.amazon.awssdk.services.rbin.model.ListRulesRequest;
import software.amazon.awssdk.services.rbin.model.ListRulesResponse;
import software.amazon.awssdk.services.rbin.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.rbin.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.rbin.model.LockRuleRequest;
import software.amazon.awssdk.services.rbin.model.LockRuleResponse;
import software.amazon.awssdk.services.rbin.model.TagResourceRequest;
import software.amazon.awssdk.services.rbin.model.TagResourceResponse;
import software.amazon.awssdk.services.rbin.model.UnlockRuleRequest;
import software.amazon.awssdk.services.rbin.model.UnlockRuleResponse;
import software.amazon.awssdk.services.rbin.model.UntagResourceRequest;
import software.amazon.awssdk.services.rbin.model.UntagResourceResponse;
import software.amazon.awssdk.services.rbin.model.UpdateRuleRequest;
import software.amazon.awssdk.services.rbin.model.UpdateRuleResponse;
import software.amazon.awssdk.services.rbin.paginators.ListRulesPublisher;

/**
 * Service client for accessing Amazon Recycle Bin asynchronously. This can be created using the static
 * {@link #builder()} method.The asynchronous client performs non-blocking I/O when configured with any
 * {@code SdkAsyncHttpClient} supported in the SDK. However, full non-blocking is not guaranteed as the async client may
 * perform blocking calls in some cases such as credentials retrieval and endpoint discovery as part of the async API
 * call.
 *
 * <p>
 * This is the <i>Recycle Bin API Reference</i>. This documentation provides descriptions and syntax for each of the
 * actions and data types in Recycle Bin.
 * </p>
 * <p>
 * Recycle Bin is a resource recovery feature that enables you to restore accidentally deleted snapshots and EBS-backed
 * AMIs. When using Recycle Bin, if your resources are deleted, they are retained in the Recycle Bin for a time period
 * that you specify.
 * </p>
 * <p>
 * You can restore a resource from the Recycle Bin at any time before its retention period expires. After you restore a
 * resource from the Recycle Bin, the resource is removed from the Recycle Bin, and you can then use it in the same way
 * you use any other resource of that type in your account. If the retention period expires and the resource is not
 * restored, the resource is permanently deleted from the Recycle Bin and is no longer available for recovery. For more
 * information about Recycle Bin, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/recycle-bin.html">
 * Recycle Bin</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface RbinAsyncClient extends AwsClient {
    String SERVICE_NAME = "rbin";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "rbin";

    /**
     * <p>
     * Creates a Recycle Bin retention rule. You can create two types of retention rules:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Tag-level retention rules</b> - These retention rules use resource tags to identify the resources to protect.
     * For each retention rule, you specify one or more tag key and value pairs. Resources (of the specified type) that
     * have at least one of these tag key and value pairs are automatically retained in the Recycle Bin upon deletion.
     * Use this type of retention rule to protect specific resources in your account based on their tags.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Region-level retention rules</b> - These retention rules, by default, apply to all of the resources (of the
     * specified type) in the Region, even if the resources are not tagged. However, you can specify exclusion tags to
     * exclude resources that have specific tags. Use this type of retention rule to protect all resources of a specific
     * type in a Region.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/ebs/latest/userguide/recycle-bin.html"> Create
     * Recycle Bin retention rules</a> in the <i>Amazon EBS User Guide</i>.
     * </p>
     *
     * @param createRuleRequest
     * @return A Java Future containing the result of the CreateRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota for the number of tags per
     *         resource to be exceeded.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.CreateRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/CreateRule" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateRuleResponse> createRule(CreateRuleRequest createRuleRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a Recycle Bin retention rule. You can create two types of retention rules:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Tag-level retention rules</b> - These retention rules use resource tags to identify the resources to protect.
     * For each retention rule, you specify one or more tag key and value pairs. Resources (of the specified type) that
     * have at least one of these tag key and value pairs are automatically retained in the Recycle Bin upon deletion.
     * Use this type of retention rule to protect specific resources in your account based on their tags.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Region-level retention rules</b> - These retention rules, by default, apply to all of the resources (of the
     * specified type) in the Region, even if the resources are not tagged. However, you can specify exclusion tags to
     * exclude resources that have specific tags. Use this type of retention rule to protect all resources of a specific
     * type in a Region.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/ebs/latest/userguide/recycle-bin.html"> Create
     * Recycle Bin retention rules</a> in the <i>Amazon EBS User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateRuleRequest.Builder} avoiding the need to
     * create one manually via {@link CreateRuleRequest#builder()}
     * </p>
     *
     * @param createRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.CreateRuleRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota for the number of tags per
     *         resource to be exceeded.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.CreateRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/CreateRule" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateRuleResponse> createRule(Consumer<CreateRuleRequest.Builder> createRuleRequest) {
        return createRule(CreateRuleRequest.builder().applyMutation(createRuleRequest).build());
    }

    /**
     * <p>
     * Deletes a Recycle Bin retention rule. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/recycle-bin-working-with-rules.html#recycle-bin-delete-rule"
     * > Delete Recycle Bin retention rules</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     *
     * @param deleteRuleRequest
     * @return A Java Future containing the result of the DeleteRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>ConflictException The specified retention rule lock request can't be completed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.DeleteRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/DeleteRule" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteRuleResponse> deleteRule(DeleteRuleRequest deleteRuleRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a Recycle Bin retention rule. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/recycle-bin-working-with-rules.html#recycle-bin-delete-rule"
     * > Delete Recycle Bin retention rules</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteRuleRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteRuleRequest#builder()}
     * </p>
     *
     * @param deleteRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.DeleteRuleRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>ConflictException The specified retention rule lock request can't be completed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.DeleteRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/DeleteRule" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteRuleResponse> deleteRule(Consumer<DeleteRuleRequest.Builder> deleteRuleRequest) {
        return deleteRule(DeleteRuleRequest.builder().applyMutation(deleteRuleRequest).build());
    }

    /**
     * <p>
     * Gets information about a Recycle Bin retention rule.
     * </p>
     *
     * @param getRuleRequest
     * @return A Java Future containing the result of the GetRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.GetRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/GetRule" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetRuleResponse> getRule(GetRuleRequest getRuleRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets information about a Recycle Bin retention rule.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetRuleRequest.Builder} avoiding the need to create
     * one manually via {@link GetRuleRequest#builder()}
     * </p>
     *
     * @param getRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.GetRuleRequest.Builder} to create a request.
     * @return A Java Future containing the result of the GetRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.GetRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/GetRule" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetRuleResponse> getRule(Consumer<GetRuleRequest.Builder> getRuleRequest) {
        return getRule(GetRuleRequest.builder().applyMutation(getRuleRequest).build());
    }

    /**
     * <p>
     * Lists the Recycle Bin retention rules in the Region.
     * </p>
     *
     * @param listRulesRequest
     * @return A Java Future containing the result of the ListRules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.ListRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/ListRules" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListRulesResponse> listRules(ListRulesRequest listRulesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the Recycle Bin retention rules in the Region.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRulesRequest.Builder} avoiding the need to
     * create one manually via {@link ListRulesRequest#builder()}
     * </p>
     *
     * @param listRulesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.ListRulesRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListRules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.ListRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/ListRules" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListRulesResponse> listRules(Consumer<ListRulesRequest.Builder> listRulesRequest) {
        return listRules(ListRulesRequest.builder().applyMutation(listRulesRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #listRules(software.amazon.awssdk.services.rbin.model.ListRulesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rbin.paginators.ListRulesPublisher publisher = client.listRulesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rbin.paginators.ListRulesPublisher publisher = client.listRulesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.rbin.model.ListRulesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.rbin.model.ListRulesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRules(software.amazon.awssdk.services.rbin.model.ListRulesRequest)} operation.</b>
     * </p>
     *
     * @param listRulesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.ListRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/ListRules" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRulesPublisher listRulesPaginator(ListRulesRequest listRulesRequest) {
        return new ListRulesPublisher(this, listRulesRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listRules(software.amazon.awssdk.services.rbin.model.ListRulesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rbin.paginators.ListRulesPublisher publisher = client.listRulesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rbin.paginators.ListRulesPublisher publisher = client.listRulesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.rbin.model.ListRulesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.rbin.model.ListRulesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRules(software.amazon.awssdk.services.rbin.model.ListRulesRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRulesRequest.Builder} avoiding the need to
     * create one manually via {@link ListRulesRequest#builder()}
     * </p>
     *
     * @param listRulesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.ListRulesRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.ListRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/ListRules" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRulesPublisher listRulesPaginator(Consumer<ListRulesRequest.Builder> listRulesRequest) {
        return listRulesPaginator(ListRulesRequest.builder().applyMutation(listRulesRequest).build());
    }

    /**
     * <p>
     * Lists the tags assigned to a retention rule.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the tags assigned to a retention rule.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.ListTagsForResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Locks a Region-level retention rule. A locked retention rule can't be modified or deleted.
     * </p>
     * <note>
     * <p>
     * You can't lock tag-level retention rules, or Region-level retention rules that have exclusion tags.
     * </p>
     * </note>
     *
     * @param lockRuleRequest
     * @return A Java Future containing the result of the LockRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>ConflictException The specified retention rule lock request can't be completed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.LockRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/LockRule" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<LockRuleResponse> lockRule(LockRuleRequest lockRuleRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Locks a Region-level retention rule. A locked retention rule can't be modified or deleted.
     * </p>
     * <note>
     * <p>
     * You can't lock tag-level retention rules, or Region-level retention rules that have exclusion tags.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link LockRuleRequest.Builder} avoiding the need to
     * create one manually via {@link LockRuleRequest#builder()}
     * </p>
     *
     * @param lockRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.LockRuleRequest.Builder} to create a request.
     * @return A Java Future containing the result of the LockRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>ConflictException The specified retention rule lock request can't be completed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.LockRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/LockRule" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<LockRuleResponse> lockRule(Consumer<LockRuleRequest.Builder> lockRuleRequest) {
        return lockRule(LockRuleRequest.builder().applyMutation(lockRuleRequest).build());
    }

    /**
     * <p>
     * Assigns tags to the specified retention rule.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota for the number of tags per
     *         resource to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Assigns tags to the specified retention rule.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.TagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota for the number of tags per
     *         resource to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Unlocks a retention rule. After a retention rule is unlocked, it can be modified or deleted only after the unlock
     * delay period expires.
     * </p>
     *
     * @param unlockRuleRequest
     * @return A Java Future containing the result of the UnlockRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>ConflictException The specified retention rule lock request can't be completed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.UnlockRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/UnlockRule" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UnlockRuleResponse> unlockRule(UnlockRuleRequest unlockRuleRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Unlocks a retention rule. After a retention rule is unlocked, it can be modified or deleted only after the unlock
     * delay period expires.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UnlockRuleRequest.Builder} avoiding the need to
     * create one manually via {@link UnlockRuleRequest#builder()}
     * </p>
     *
     * @param unlockRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.UnlockRuleRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UnlockRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>ConflictException The specified retention rule lock request can't be completed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.UnlockRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/UnlockRule" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UnlockRuleResponse> unlockRule(Consumer<UnlockRuleRequest.Builder> unlockRuleRequest) {
        return unlockRule(UnlockRuleRequest.builder().applyMutation(unlockRuleRequest).build());
    }

    /**
     * <p>
     * Unassigns a tag from a retention rule.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Unassigns a tag from a retention rule.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.UntagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest) {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates an existing Recycle Bin retention rule. You can update a retention rule's description, resource tags, and
     * retention period at any time after creation. You can't update a retention rule's resource type after creation.
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/recycle-bin-working-with-rules.html#recycle-bin-update-rule"
     * > Update Recycle Bin retention rules</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     *
     * @param updateRuleRequest
     * @return A Java Future containing the result of the UpdateRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>ConflictException The specified retention rule lock request can't be completed.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota for the number of tags per
     *         resource to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.UpdateRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/UpdateRule" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateRuleResponse> updateRule(UpdateRuleRequest updateRuleRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates an existing Recycle Bin retention rule. You can update a retention rule's description, resource tags, and
     * retention period at any time after creation. You can't update a retention rule's resource type after creation.
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/recycle-bin-working-with-rules.html#recycle-bin-update-rule"
     * > Update Recycle Bin retention rules</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateRuleRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateRuleRequest#builder()}
     * </p>
     *
     * @param updateRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.rbin.model.UpdateRuleRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UpdateRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException One or more of the parameters in the request is not valid.</li>
     *         <li>InternalServerException The service could not respond to the request due to an internal problem.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>ConflictException The specified retention rule lock request can't be completed.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota for the number of tags per
     *         resource to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RbinException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RbinAsyncClient.UpdateRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/UpdateRule" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateRuleResponse> updateRule(Consumer<UpdateRuleRequest.Builder> updateRuleRequest) {
        return updateRule(UpdateRuleRequest.builder().applyMutation(updateRuleRequest).build());
    }

    @Override
    default RbinServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link RbinAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static RbinAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link RbinAsyncClient}.
     */
    static RbinAsyncClientBuilder builder() {
        return new DefaultRbinAsyncClientBuilder();
    }
}
