/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the name and values of a manual DB cluster snapshot attribute.
 * </p>
 * <p>
 * Manual DB cluster snapshot attributes are used to authorize other Amazon Web Services accounts to restore a manual DB
 * cluster snapshot. For more information, see the <code>ModifyDBClusterSnapshotAttribute</code> API action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DBClusterSnapshotAttribute implements SdkPojo, Serializable,
        ToCopyableBuilder<DBClusterSnapshotAttribute.Builder, DBClusterSnapshotAttribute> {
    private static final SdkField<String> ATTRIBUTE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AttributeName").getter(getter(DBClusterSnapshotAttribute::attributeName))
            .setter(setter(Builder::attributeName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributeName").build()).build();

    private static final SdkField<List<String>> ATTRIBUTE_VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AttributeValues")
            .getter(getter(DBClusterSnapshotAttribute::attributeValues))
            .setter(setter(Builder::attributeValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributeValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("AttributeValue")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("AttributeValue").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ATTRIBUTE_NAME_FIELD,
            ATTRIBUTE_VALUES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String attributeName;

    private final List<String> attributeValues;

    private DBClusterSnapshotAttribute(BuilderImpl builder) {
        this.attributeName = builder.attributeName;
        this.attributeValues = builder.attributeValues;
    }

    /**
     * <p>
     * The name of the manual DB cluster snapshot attribute.
     * </p>
     * <p>
     * The attribute named <code>restore</code> refers to the list of Amazon Web Services accounts that have permission
     * to copy or restore the manual DB cluster snapshot. For more information, see the
     * <code>ModifyDBClusterSnapshotAttribute</code> API action.
     * </p>
     * 
     * @return The name of the manual DB cluster snapshot attribute.</p>
     *         <p>
     *         The attribute named <code>restore</code> refers to the list of Amazon Web Services accounts that have
     *         permission to copy or restore the manual DB cluster snapshot. For more information, see the
     *         <code>ModifyDBClusterSnapshotAttribute</code> API action.
     */
    public final String attributeName() {
        return attributeName;
    }

    /**
     * Returns true if the AttributeValues property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasAttributeValues() {
        return attributeValues != null && !(attributeValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The value(s) for the manual DB cluster snapshot attribute.
     * </p>
     * <p>
     * If the <code>AttributeName</code> field is set to <code>restore</code>, then this element returns a list of IDs
     * of the Amazon Web Services accounts that are authorized to copy or restore the manual DB cluster snapshot. If a
     * value of <code>all</code> is in the list, then the manual DB cluster snapshot is public and available for any
     * Amazon Web Services account to copy or restore.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAttributeValues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The value(s) for the manual DB cluster snapshot attribute.</p>
     *         <p>
     *         If the <code>AttributeName</code> field is set to <code>restore</code>, then this element returns a list
     *         of IDs of the Amazon Web Services accounts that are authorized to copy or restore the manual DB cluster
     *         snapshot. If a value of <code>all</code> is in the list, then the manual DB cluster snapshot is public
     *         and available for any Amazon Web Services account to copy or restore.
     */
    public final List<String> attributeValues() {
        return attributeValues;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(attributeName());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributeValues() ? attributeValues() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DBClusterSnapshotAttribute)) {
            return false;
        }
        DBClusterSnapshotAttribute other = (DBClusterSnapshotAttribute) obj;
        return Objects.equals(attributeName(), other.attributeName()) && hasAttributeValues() == other.hasAttributeValues()
                && Objects.equals(attributeValues(), other.attributeValues());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DBClusterSnapshotAttribute").add("AttributeName", attributeName())
                .add("AttributeValues", hasAttributeValues() ? attributeValues() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AttributeName":
            return Optional.ofNullable(clazz.cast(attributeName()));
        case "AttributeValues":
            return Optional.ofNullable(clazz.cast(attributeValues()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DBClusterSnapshotAttribute, T> g) {
        return obj -> g.apply((DBClusterSnapshotAttribute) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DBClusterSnapshotAttribute> {
        /**
         * <p>
         * The name of the manual DB cluster snapshot attribute.
         * </p>
         * <p>
         * The attribute named <code>restore</code> refers to the list of Amazon Web Services accounts that have
         * permission to copy or restore the manual DB cluster snapshot. For more information, see the
         * <code>ModifyDBClusterSnapshotAttribute</code> API action.
         * </p>
         * 
         * @param attributeName
         *        The name of the manual DB cluster snapshot attribute.</p>
         *        <p>
         *        The attribute named <code>restore</code> refers to the list of Amazon Web Services accounts that have
         *        permission to copy or restore the manual DB cluster snapshot. For more information, see the
         *        <code>ModifyDBClusterSnapshotAttribute</code> API action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeName(String attributeName);

        /**
         * <p>
         * The value(s) for the manual DB cluster snapshot attribute.
         * </p>
         * <p>
         * If the <code>AttributeName</code> field is set to <code>restore</code>, then this element returns a list of
         * IDs of the Amazon Web Services accounts that are authorized to copy or restore the manual DB cluster
         * snapshot. If a value of <code>all</code> is in the list, then the manual DB cluster snapshot is public and
         * available for any Amazon Web Services account to copy or restore.
         * </p>
         * 
         * @param attributeValues
         *        The value(s) for the manual DB cluster snapshot attribute.</p>
         *        <p>
         *        If the <code>AttributeName</code> field is set to <code>restore</code>, then this element returns a
         *        list of IDs of the Amazon Web Services accounts that are authorized to copy or restore the manual DB
         *        cluster snapshot. If a value of <code>all</code> is in the list, then the manual DB cluster snapshot
         *        is public and available for any Amazon Web Services account to copy or restore.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeValues(Collection<String> attributeValues);

        /**
         * <p>
         * The value(s) for the manual DB cluster snapshot attribute.
         * </p>
         * <p>
         * If the <code>AttributeName</code> field is set to <code>restore</code>, then this element returns a list of
         * IDs of the Amazon Web Services accounts that are authorized to copy or restore the manual DB cluster
         * snapshot. If a value of <code>all</code> is in the list, then the manual DB cluster snapshot is public and
         * available for any Amazon Web Services account to copy or restore.
         * </p>
         * 
         * @param attributeValues
         *        The value(s) for the manual DB cluster snapshot attribute.</p>
         *        <p>
         *        If the <code>AttributeName</code> field is set to <code>restore</code>, then this element returns a
         *        list of IDs of the Amazon Web Services accounts that are authorized to copy or restore the manual DB
         *        cluster snapshot. If a value of <code>all</code> is in the list, then the manual DB cluster snapshot
         *        is public and available for any Amazon Web Services account to copy or restore.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeValues(String... attributeValues);
    }

    static final class BuilderImpl implements Builder {
        private String attributeName;

        private List<String> attributeValues = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DBClusterSnapshotAttribute model) {
            attributeName(model.attributeName);
            attributeValues(model.attributeValues);
        }

        public final String getAttributeName() {
            return attributeName;
        }

        @Override
        public final Builder attributeName(String attributeName) {
            this.attributeName = attributeName;
            return this;
        }

        public final void setAttributeName(String attributeName) {
            this.attributeName = attributeName;
        }

        public final Collection<String> getAttributeValues() {
            if (attributeValues instanceof SdkAutoConstructList) {
                return null;
            }
            return attributeValues;
        }

        @Override
        public final Builder attributeValues(Collection<String> attributeValues) {
            this.attributeValues = AttributeValueListCopier.copy(attributeValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributeValues(String... attributeValues) {
            attributeValues(Arrays.asList(attributeValues));
            return this;
        }

        public final void setAttributeValues(Collection<String> attributeValues) {
            this.attributeValues = AttributeValueListCopier.copy(attributeValues);
        }

        @Override
        public DBClusterSnapshotAttribute build() {
            return new DBClusterSnapshotAttribute(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
