/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.rds.RdsAsyncClient;
import software.amazon.awssdk.services.rds.model.DescribeDbClusterSnapshotsRequest;
import software.amazon.awssdk.services.rds.model.DescribeDbClusterSnapshotsResponse;
import software.amazon.awssdk.services.rds.model.DescribeDbInstancesRequest;
import software.amazon.awssdk.services.rds.model.DescribeDbInstancesResponse;
import software.amazon.awssdk.services.rds.model.DescribeDbSnapshotsRequest;
import software.amazon.awssdk.services.rds.model.DescribeDbSnapshotsResponse;
import software.amazon.awssdk.services.rds.model.RdsRequest;
import software.amazon.awssdk.services.rds.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultRdsAsyncWaiter implements RdsAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final RdsAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeDbInstancesResponse> dBInstanceAvailableWaiter;

    private final AsyncWaiter<DescribeDbInstancesResponse> dBInstanceDeletedWaiter;

    private final AsyncWaiter<DescribeDbSnapshotsResponse> dBSnapshotAvailableWaiter;

    private final AsyncWaiter<DescribeDbSnapshotsResponse> dBSnapshotDeletedWaiter;

    private final AsyncWaiter<DescribeDbClusterSnapshotsResponse> dBClusterSnapshotAvailableWaiter;

    private final AsyncWaiter<DescribeDbClusterSnapshotsResponse> dBClusterSnapshotDeletedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultRdsAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = RdsAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.dBInstanceAvailableWaiter = AsyncWaiter.builder(DescribeDbInstancesResponse.class)
                .acceptors(dBInstanceAvailableWaiterAcceptors())
                .overrideConfiguration(dBInstanceAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.dBInstanceDeletedWaiter = AsyncWaiter.builder(DescribeDbInstancesResponse.class)
                .acceptors(dBInstanceDeletedWaiterAcceptors())
                .overrideConfiguration(dBInstanceDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.dBSnapshotAvailableWaiter = AsyncWaiter.builder(DescribeDbSnapshotsResponse.class)
                .acceptors(dBSnapshotAvailableWaiterAcceptors())
                .overrideConfiguration(dBSnapshotAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.dBSnapshotDeletedWaiter = AsyncWaiter.builder(DescribeDbSnapshotsResponse.class)
                .acceptors(dBSnapshotDeletedWaiterAcceptors())
                .overrideConfiguration(dBSnapshotDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.dBClusterSnapshotAvailableWaiter = AsyncWaiter.builder(DescribeDbClusterSnapshotsResponse.class)
                .acceptors(dBClusterSnapshotAvailableWaiterAcceptors())
                .overrideConfiguration(dBClusterSnapshotAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.dBClusterSnapshotDeletedWaiter = AsyncWaiter.builder(DescribeDbClusterSnapshotsResponse.class)
                .acceptors(dBClusterSnapshotDeletedWaiterAcceptors())
                .overrideConfiguration(dBClusterSnapshotDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDbClusterSnapshotsResponse>> waitUntilDBClusterSnapshotAvailable(
            DescribeDbClusterSnapshotsRequest describeDbClusterSnapshotsRequest) {
        return dBClusterSnapshotAvailableWaiter.runAsync(() -> client
                .describeDBClusterSnapshots(applyWaitersUserAgent(describeDbClusterSnapshotsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDbClusterSnapshotsResponse>> waitUntilDBClusterSnapshotAvailable(
            DescribeDbClusterSnapshotsRequest describeDbClusterSnapshotsRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBClusterSnapshotAvailableWaiter.runAsync(
                () -> client.describeDBClusterSnapshots(applyWaitersUserAgent(describeDbClusterSnapshotsRequest)),
                dBClusterSnapshotAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDbClusterSnapshotsResponse>> waitUntilDBClusterSnapshotDeleted(
            DescribeDbClusterSnapshotsRequest describeDbClusterSnapshotsRequest) {
        return dBClusterSnapshotDeletedWaiter.runAsync(() -> client
                .describeDBClusterSnapshots(applyWaitersUserAgent(describeDbClusterSnapshotsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDbClusterSnapshotsResponse>> waitUntilDBClusterSnapshotDeleted(
            DescribeDbClusterSnapshotsRequest describeDbClusterSnapshotsRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBClusterSnapshotDeletedWaiter.runAsync(
                () -> client.describeDBClusterSnapshots(applyWaitersUserAgent(describeDbClusterSnapshotsRequest)),
                dBClusterSnapshotDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDbInstancesResponse>> waitUntilDBInstanceAvailable(
            DescribeDbInstancesRequest describeDbInstancesRequest) {
        return dBInstanceAvailableWaiter.runAsync(() -> client
                .describeDBInstances(applyWaitersUserAgent(describeDbInstancesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDbInstancesResponse>> waitUntilDBInstanceAvailable(
            DescribeDbInstancesRequest describeDbInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBInstanceAvailableWaiter.runAsync(
                () -> client.describeDBInstances(applyWaitersUserAgent(describeDbInstancesRequest)),
                dBInstanceAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDbInstancesResponse>> waitUntilDBInstanceDeleted(
            DescribeDbInstancesRequest describeDbInstancesRequest) {
        return dBInstanceDeletedWaiter.runAsync(() -> client
                .describeDBInstances(applyWaitersUserAgent(describeDbInstancesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDbInstancesResponse>> waitUntilDBInstanceDeleted(
            DescribeDbInstancesRequest describeDbInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBInstanceDeletedWaiter.runAsync(
                () -> client.describeDBInstances(applyWaitersUserAgent(describeDbInstancesRequest)),
                dBInstanceDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDbSnapshotsResponse>> waitUntilDBSnapshotAvailable(
            DescribeDbSnapshotsRequest describeDbSnapshotsRequest) {
        return dBSnapshotAvailableWaiter.runAsync(() -> client
                .describeDBSnapshots(applyWaitersUserAgent(describeDbSnapshotsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDbSnapshotsResponse>> waitUntilDBSnapshotAvailable(
            DescribeDbSnapshotsRequest describeDbSnapshotsRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBSnapshotAvailableWaiter.runAsync(
                () -> client.describeDBSnapshots(applyWaitersUserAgent(describeDbSnapshotsRequest)),
                dBSnapshotAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDbSnapshotsResponse>> waitUntilDBSnapshotDeleted(
            DescribeDbSnapshotsRequest describeDbSnapshotsRequest) {
        return dBSnapshotDeletedWaiter.runAsync(() -> client
                .describeDBSnapshots(applyWaitersUserAgent(describeDbSnapshotsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDbSnapshotsResponse>> waitUntilDBSnapshotDeleted(
            DescribeDbSnapshotsRequest describeDbSnapshotsRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBSnapshotDeletedWaiter.runAsync(
                () -> client.describeDBSnapshots(applyWaitersUserAgent(describeDbSnapshotsRequest)),
                dBSnapshotDeletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeDbInstancesResponse>> dBInstanceAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-restore"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-parameters"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeDbInstancesResponse>> dBInstanceDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("DBInstances").length().compare("==", input.constant(0)).value(), true);
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "DBInstanceNotFound")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "creating"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "modifying"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "rebooting"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
            return !resultValues.isEmpty()
                    && resultValues.stream().anyMatch(v -> Objects.equals(v, "resetting-master-credentials"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeDbSnapshotsResponse>> dBSnapshotAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbSnapshotsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-restore"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-parameters"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeDbSnapshotsResponse>> dBSnapshotDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbSnapshotsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("DBSnapshots").length().compare("==", input.constant(0)).value(), true);
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "DBSnapshotNotFound")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "creating"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "modifying"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "rebooting"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty()
                    && resultValues.stream().anyMatch(v -> Objects.equals(v, "resetting-master-credentials"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeDbClusterSnapshotsResponse>> dBClusterSnapshotAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbClusterSnapshotsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-restore"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-parameters"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeDbClusterSnapshotsResponse>> dBClusterSnapshotDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbClusterSnapshotsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("DBClusterSnapshots").length().compare("==", input.constant(0)).value(), true);
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "DBClusterSnapshotNotFoundFault")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "creating"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "modifying"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "rebooting"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty()
                    && resultValues.stream().anyMatch(v -> Objects.equals(v, "resetting-master-credentials"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration dBInstanceAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration dBInstanceDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration dBSnapshotAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration dBSnapshotDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration dBClusterSnapshotAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration dBClusterSnapshotDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static RdsAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends RdsRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements RdsAsyncWaiter.Builder {
        private RdsAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public RdsAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public RdsAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public RdsAsyncWaiter.Builder client(RdsAsyncClient client) {
            this.client = client;
            return this;
        }

        public RdsAsyncWaiter build() {
            return new DefaultRdsAsyncWaiter(this);
        }
    }
}
