/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration setting for the log types to be enabled for export to CloudWatch Logs for a specific DB instance or
 * DB cluster.
 * </p>
 * <p>
 * The <code>EnableLogTypes</code> and <code>DisableLogTypes</code> arrays determine which logs will be exported (or not
 * exported) to CloudWatch Logs. The values within these arrays depend on the DB engine being used.
 * </p>
 * <p>
 * For more information about exporting CloudWatch Logs for Amazon RDS DB instances, see <a href=
 * "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch"
 * >Publishing Database Logs to Amazon CloudWatch Logs </a> in the <i>Amazon RDS User Guide</i>.
 * </p>
 * <p>
 * For more information about exporting CloudWatch Logs for Amazon Aurora DB clusters, see <a href=
 * "https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch"
 * >Publishing Database Logs to Amazon CloudWatch Logs</a> in the <i>Amazon Aurora User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CloudwatchLogsExportConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<CloudwatchLogsExportConfiguration.Builder, CloudwatchLogsExportConfiguration> {
    private static final SdkField<List<String>> ENABLE_LOG_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EnableLogTypes")
            .getter(getter(CloudwatchLogsExportConfiguration::enableLogTypes))
            .setter(setter(Builder::enableLogTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableLogTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> DISABLE_LOG_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DisableLogTypes")
            .getter(getter(CloudwatchLogsExportConfiguration::disableLogTypes))
            .setter(setter(Builder::disableLogTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisableLogTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENABLE_LOG_TYPES_FIELD,
            DISABLE_LOG_TYPES_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> enableLogTypes;

    private final List<String> disableLogTypes;

    private CloudwatchLogsExportConfiguration(BuilderImpl builder) {
        this.enableLogTypes = builder.enableLogTypes;
        this.disableLogTypes = builder.disableLogTypes;
    }

    /**
     * For responses, this returns true if the service returned a value for the EnableLogTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnableLogTypes() {
        return enableLogTypes != null && !(enableLogTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of log types to enable.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnableLogTypes} method.
     * </p>
     * 
     * @return The list of log types to enable.
     */
    public final List<String> enableLogTypes() {
        return enableLogTypes;
    }

    /**
     * For responses, this returns true if the service returned a value for the DisableLogTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDisableLogTypes() {
        return disableLogTypes != null && !(disableLogTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of log types to disable.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDisableLogTypes} method.
     * </p>
     * 
     * @return The list of log types to disable.
     */
    public final List<String> disableLogTypes() {
        return disableLogTypes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasEnableLogTypes() ? enableLogTypes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDisableLogTypes() ? disableLogTypes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CloudwatchLogsExportConfiguration)) {
            return false;
        }
        CloudwatchLogsExportConfiguration other = (CloudwatchLogsExportConfiguration) obj;
        return hasEnableLogTypes() == other.hasEnableLogTypes() && Objects.equals(enableLogTypes(), other.enableLogTypes())
                && hasDisableLogTypes() == other.hasDisableLogTypes()
                && Objects.equals(disableLogTypes(), other.disableLogTypes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CloudwatchLogsExportConfiguration")
                .add("EnableLogTypes", hasEnableLogTypes() ? enableLogTypes() : null)
                .add("DisableLogTypes", hasDisableLogTypes() ? disableLogTypes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EnableLogTypes":
            return Optional.ofNullable(clazz.cast(enableLogTypes()));
        case "DisableLogTypes":
            return Optional.ofNullable(clazz.cast(disableLogTypes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CloudwatchLogsExportConfiguration, T> g) {
        return obj -> g.apply((CloudwatchLogsExportConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CloudwatchLogsExportConfiguration> {
        /**
         * <p>
         * The list of log types to enable.
         * </p>
         * 
         * @param enableLogTypes
         *        The list of log types to enable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableLogTypes(Collection<String> enableLogTypes);

        /**
         * <p>
         * The list of log types to enable.
         * </p>
         * 
         * @param enableLogTypes
         *        The list of log types to enable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableLogTypes(String... enableLogTypes);

        /**
         * <p>
         * The list of log types to disable.
         * </p>
         * 
         * @param disableLogTypes
         *        The list of log types to disable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disableLogTypes(Collection<String> disableLogTypes);

        /**
         * <p>
         * The list of log types to disable.
         * </p>
         * 
         * @param disableLogTypes
         *        The list of log types to disable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disableLogTypes(String... disableLogTypes);
    }

    static final class BuilderImpl implements Builder {
        private List<String> enableLogTypes = DefaultSdkAutoConstructList.getInstance();

        private List<String> disableLogTypes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CloudwatchLogsExportConfiguration model) {
            enableLogTypes(model.enableLogTypes);
            disableLogTypes(model.disableLogTypes);
        }

        public final Collection<String> getEnableLogTypes() {
            if (enableLogTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return enableLogTypes;
        }

        public final void setEnableLogTypes(Collection<String> enableLogTypes) {
            this.enableLogTypes = LogTypeListCopier.copy(enableLogTypes);
        }

        @Override
        @Transient
        public final Builder enableLogTypes(Collection<String> enableLogTypes) {
            this.enableLogTypes = LogTypeListCopier.copy(enableLogTypes);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder enableLogTypes(String... enableLogTypes) {
            enableLogTypes(Arrays.asList(enableLogTypes));
            return this;
        }

        public final Collection<String> getDisableLogTypes() {
            if (disableLogTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return disableLogTypes;
        }

        public final void setDisableLogTypes(Collection<String> disableLogTypes) {
            this.disableLogTypes = LogTypeListCopier.copy(disableLogTypes);
        }

        @Override
        @Transient
        public final Builder disableLogTypes(Collection<String> disableLogTypes) {
            this.disableLogTypes = LogTypeListCopier.copy(disableLogTypes);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder disableLogTypes(String... disableLogTypes) {
            disableLogTypes(Arrays.asList(disableLogTypes));
            return this;
        }

        @Override
        public CloudwatchLogsExportConfiguration build() {
            return new CloudwatchLogsExportConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
