/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The data structure representing an endpoint associated with a DB proxy. RDS automatically creates one endpoint for
 * each DB proxy. For Aurora DB clusters, you can associate additional endpoints with the same DB proxy. These endpoints
 * can be read/write or read-only. They can also reside in different VPCs than the associated DB proxy.
 * </p>
 * <p>
 * This data type is used as a response element in the <code>DescribeDBProxyEndpoints</code> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DBProxyEndpoint implements SdkPojo, Serializable, ToCopyableBuilder<DBProxyEndpoint.Builder, DBProxyEndpoint> {
    private static final SdkField<String> DB_PROXY_ENDPOINT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBProxyEndpointName").getter(getter(DBProxyEndpoint::dbProxyEndpointName))
            .setter(setter(Builder::dbProxyEndpointName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBProxyEndpointName").build())
            .build();

    private static final SdkField<String> DB_PROXY_ENDPOINT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBProxyEndpointArn").getter(getter(DBProxyEndpoint::dbProxyEndpointArn))
            .setter(setter(Builder::dbProxyEndpointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBProxyEndpointArn").build())
            .build();

    private static final SdkField<String> DB_PROXY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBProxyName").getter(getter(DBProxyEndpoint::dbProxyName)).setter(setter(Builder::dbProxyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBProxyName").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(DBProxyEndpoint::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(DBProxyEndpoint::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<List<String>> VPC_SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("VpcSecurityGroupIds")
            .getter(getter(DBProxyEndpoint::vpcSecurityGroupIds))
            .setter(setter(Builder::vpcSecurityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcSecurityGroupIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> VPC_SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("VpcSubnetIds")
            .getter(getter(DBProxyEndpoint::vpcSubnetIds))
            .setter(setter(Builder::vpcSubnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcSubnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Endpoint").getter(getter(DBProxyEndpoint::endpoint)).setter(setter(Builder::endpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoint").build()).build();

    private static final SdkField<Instant> CREATED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedDate").getter(getter(DBProxyEndpoint::createdDate)).setter(setter(Builder::createdDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedDate").build()).build();

    private static final SdkField<String> TARGET_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetRole").getter(getter(DBProxyEndpoint::targetRoleAsString)).setter(setter(Builder::targetRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetRole").build()).build();

    private static final SdkField<Boolean> IS_DEFAULT_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsDefault").getter(getter(DBProxyEndpoint::isDefault)).setter(setter(Builder::isDefault))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsDefault").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DB_PROXY_ENDPOINT_NAME_FIELD,
            DB_PROXY_ENDPOINT_ARN_FIELD, DB_PROXY_NAME_FIELD, STATUS_FIELD, VPC_ID_FIELD, VPC_SECURITY_GROUP_IDS_FIELD,
            VPC_SUBNET_IDS_FIELD, ENDPOINT_FIELD, CREATED_DATE_FIELD, TARGET_ROLE_FIELD, IS_DEFAULT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String dbProxyEndpointName;

    private final String dbProxyEndpointArn;

    private final String dbProxyName;

    private final String status;

    private final String vpcId;

    private final List<String> vpcSecurityGroupIds;

    private final List<String> vpcSubnetIds;

    private final String endpoint;

    private final Instant createdDate;

    private final String targetRole;

    private final Boolean isDefault;

    private DBProxyEndpoint(BuilderImpl builder) {
        this.dbProxyEndpointName = builder.dbProxyEndpointName;
        this.dbProxyEndpointArn = builder.dbProxyEndpointArn;
        this.dbProxyName = builder.dbProxyName;
        this.status = builder.status;
        this.vpcId = builder.vpcId;
        this.vpcSecurityGroupIds = builder.vpcSecurityGroupIds;
        this.vpcSubnetIds = builder.vpcSubnetIds;
        this.endpoint = builder.endpoint;
        this.createdDate = builder.createdDate;
        this.targetRole = builder.targetRole;
        this.isDefault = builder.isDefault;
    }

    /**
     * <p>
     * The name for the DB proxy endpoint. An identifier must begin with a letter and must contain only ASCII letters,
     * digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens.
     * </p>
     * 
     * @return The name for the DB proxy endpoint. An identifier must begin with a letter and must contain only ASCII
     *         letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens.
     */
    public final String dbProxyEndpointName() {
        return dbProxyEndpointName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the DB proxy endpoint.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the DB proxy endpoint.
     */
    public final String dbProxyEndpointArn() {
        return dbProxyEndpointArn;
    }

    /**
     * <p>
     * The identifier for the DB proxy that is associated with this DB proxy endpoint.
     * </p>
     * 
     * @return The identifier for the DB proxy that is associated with this DB proxy endpoint.
     */
    public final String dbProxyName() {
        return dbProxyName;
    }

    /**
     * <p>
     * The current status of this DB proxy endpoint. A status of <code>available</code> means the endpoint is ready to
     * handle requests. Other values indicate that you must wait for the endpoint to be ready, or take some action to
     * resolve an issue.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DBProxyEndpointStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of this DB proxy endpoint. A status of <code>available</code> means the endpoint is
     *         ready to handle requests. Other values indicate that you must wait for the endpoint to be ready, or take
     *         some action to resolve an issue.
     * @see DBProxyEndpointStatus
     */
    public final DBProxyEndpointStatus status() {
        return DBProxyEndpointStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of this DB proxy endpoint. A status of <code>available</code> means the endpoint is ready to
     * handle requests. Other values indicate that you must wait for the endpoint to be ready, or take some action to
     * resolve an issue.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DBProxyEndpointStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of this DB proxy endpoint. A status of <code>available</code> means the endpoint is
     *         ready to handle requests. Other values indicate that you must wait for the endpoint to be ready, or take
     *         some action to resolve an issue.
     * @see DBProxyEndpointStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Provides the VPC ID of the DB proxy endpoint.
     * </p>
     * 
     * @return Provides the VPC ID of the DB proxy endpoint.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * For responses, this returns true if the service returned a value for the VpcSecurityGroupIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVpcSecurityGroupIds() {
        return vpcSecurityGroupIds != null && !(vpcSecurityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Provides a list of VPC security groups that the DB proxy endpoint belongs to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVpcSecurityGroupIds} method.
     * </p>
     * 
     * @return Provides a list of VPC security groups that the DB proxy endpoint belongs to.
     */
    public final List<String> vpcSecurityGroupIds() {
        return vpcSecurityGroupIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the VpcSubnetIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVpcSubnetIds() {
        return vpcSubnetIds != null && !(vpcSubnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The EC2 subnet IDs for the DB proxy endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVpcSubnetIds} method.
     * </p>
     * 
     * @return The EC2 subnet IDs for the DB proxy endpoint.
     */
    public final List<String> vpcSubnetIds() {
        return vpcSubnetIds;
    }

    /**
     * <p>
     * The endpoint that you can use to connect to the DB proxy. You include the endpoint value in the connection string
     * for a database client application.
     * </p>
     * 
     * @return The endpoint that you can use to connect to the DB proxy. You include the endpoint value in the
     *         connection string for a database client application.
     */
    public final String endpoint() {
        return endpoint;
    }

    /**
     * <p>
     * The date and time when the DB proxy endpoint was first created.
     * </p>
     * 
     * @return The date and time when the DB proxy endpoint was first created.
     */
    public final Instant createdDate() {
        return createdDate;
    }

    /**
     * <p>
     * A value that indicates whether the DB proxy endpoint can be used for read/write or read-only operations.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #targetRole} will
     * return {@link DBProxyEndpointTargetRole#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #targetRoleAsString}.
     * </p>
     * 
     * @return A value that indicates whether the DB proxy endpoint can be used for read/write or read-only operations.
     * @see DBProxyEndpointTargetRole
     */
    public final DBProxyEndpointTargetRole targetRole() {
        return DBProxyEndpointTargetRole.fromValue(targetRole);
    }

    /**
     * <p>
     * A value that indicates whether the DB proxy endpoint can be used for read/write or read-only operations.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #targetRole} will
     * return {@link DBProxyEndpointTargetRole#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #targetRoleAsString}.
     * </p>
     * 
     * @return A value that indicates whether the DB proxy endpoint can be used for read/write or read-only operations.
     * @see DBProxyEndpointTargetRole
     */
    public final String targetRoleAsString() {
        return targetRole;
    }

    /**
     * <p>
     * A value that indicates whether this endpoint is the default endpoint for the associated DB proxy. Default DB
     * proxy endpoints always have read/write capability. Other endpoints that you associate with the DB proxy can be
     * either read/write or read-only.
     * </p>
     * 
     * @return A value that indicates whether this endpoint is the default endpoint for the associated DB proxy. Default
     *         DB proxy endpoints always have read/write capability. Other endpoints that you associate with the DB
     *         proxy can be either read/write or read-only.
     */
    public final Boolean isDefault() {
        return isDefault;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dbProxyEndpointName());
        hashCode = 31 * hashCode + Objects.hashCode(dbProxyEndpointArn());
        hashCode = 31 * hashCode + Objects.hashCode(dbProxyName());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(hasVpcSecurityGroupIds() ? vpcSecurityGroupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasVpcSubnetIds() ? vpcSubnetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(endpoint());
        hashCode = 31 * hashCode + Objects.hashCode(createdDate());
        hashCode = 31 * hashCode + Objects.hashCode(targetRoleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(isDefault());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DBProxyEndpoint)) {
            return false;
        }
        DBProxyEndpoint other = (DBProxyEndpoint) obj;
        return Objects.equals(dbProxyEndpointName(), other.dbProxyEndpointName())
                && Objects.equals(dbProxyEndpointArn(), other.dbProxyEndpointArn())
                && Objects.equals(dbProxyName(), other.dbProxyName()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(vpcId(), other.vpcId()) && hasVpcSecurityGroupIds() == other.hasVpcSecurityGroupIds()
                && Objects.equals(vpcSecurityGroupIds(), other.vpcSecurityGroupIds())
                && hasVpcSubnetIds() == other.hasVpcSubnetIds() && Objects.equals(vpcSubnetIds(), other.vpcSubnetIds())
                && Objects.equals(endpoint(), other.endpoint()) && Objects.equals(createdDate(), other.createdDate())
                && Objects.equals(targetRoleAsString(), other.targetRoleAsString())
                && Objects.equals(isDefault(), other.isDefault());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DBProxyEndpoint").add("DBProxyEndpointName", dbProxyEndpointName())
                .add("DBProxyEndpointArn", dbProxyEndpointArn()).add("DBProxyName", dbProxyName())
                .add("Status", statusAsString()).add("VpcId", vpcId())
                .add("VpcSecurityGroupIds", hasVpcSecurityGroupIds() ? vpcSecurityGroupIds() : null)
                .add("VpcSubnetIds", hasVpcSubnetIds() ? vpcSubnetIds() : null).add("Endpoint", endpoint())
                .add("CreatedDate", createdDate()).add("TargetRole", targetRoleAsString()).add("IsDefault", isDefault()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DBProxyEndpointName":
            return Optional.ofNullable(clazz.cast(dbProxyEndpointName()));
        case "DBProxyEndpointArn":
            return Optional.ofNullable(clazz.cast(dbProxyEndpointArn()));
        case "DBProxyName":
            return Optional.ofNullable(clazz.cast(dbProxyName()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "VpcSecurityGroupIds":
            return Optional.ofNullable(clazz.cast(vpcSecurityGroupIds()));
        case "VpcSubnetIds":
            return Optional.ofNullable(clazz.cast(vpcSubnetIds()));
        case "Endpoint":
            return Optional.ofNullable(clazz.cast(endpoint()));
        case "CreatedDate":
            return Optional.ofNullable(clazz.cast(createdDate()));
        case "TargetRole":
            return Optional.ofNullable(clazz.cast(targetRoleAsString()));
        case "IsDefault":
            return Optional.ofNullable(clazz.cast(isDefault()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DBProxyEndpoint, T> g) {
        return obj -> g.apply((DBProxyEndpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DBProxyEndpoint> {
        /**
         * <p>
         * The name for the DB proxy endpoint. An identifier must begin with a letter and must contain only ASCII
         * letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens.
         * </p>
         * 
         * @param dbProxyEndpointName
         *        The name for the DB proxy endpoint. An identifier must begin with a letter and must contain only ASCII
         *        letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbProxyEndpointName(String dbProxyEndpointName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the DB proxy endpoint.
         * </p>
         * 
         * @param dbProxyEndpointArn
         *        The Amazon Resource Name (ARN) for the DB proxy endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbProxyEndpointArn(String dbProxyEndpointArn);

        /**
         * <p>
         * The identifier for the DB proxy that is associated with this DB proxy endpoint.
         * </p>
         * 
         * @param dbProxyName
         *        The identifier for the DB proxy that is associated with this DB proxy endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbProxyName(String dbProxyName);

        /**
         * <p>
         * The current status of this DB proxy endpoint. A status of <code>available</code> means the endpoint is ready
         * to handle requests. Other values indicate that you must wait for the endpoint to be ready, or take some
         * action to resolve an issue.
         * </p>
         * 
         * @param status
         *        The current status of this DB proxy endpoint. A status of <code>available</code> means the endpoint is
         *        ready to handle requests. Other values indicate that you must wait for the endpoint to be ready, or
         *        take some action to resolve an issue.
         * @see DBProxyEndpointStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DBProxyEndpointStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of this DB proxy endpoint. A status of <code>available</code> means the endpoint is ready
         * to handle requests. Other values indicate that you must wait for the endpoint to be ready, or take some
         * action to resolve an issue.
         * </p>
         * 
         * @param status
         *        The current status of this DB proxy endpoint. A status of <code>available</code> means the endpoint is
         *        ready to handle requests. Other values indicate that you must wait for the endpoint to be ready, or
         *        take some action to resolve an issue.
         * @see DBProxyEndpointStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DBProxyEndpointStatus
         */
        Builder status(DBProxyEndpointStatus status);

        /**
         * <p>
         * Provides the VPC ID of the DB proxy endpoint.
         * </p>
         * 
         * @param vpcId
         *        Provides the VPC ID of the DB proxy endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * Provides a list of VPC security groups that the DB proxy endpoint belongs to.
         * </p>
         * 
         * @param vpcSecurityGroupIds
         *        Provides a list of VPC security groups that the DB proxy endpoint belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSecurityGroupIds(Collection<String> vpcSecurityGroupIds);

        /**
         * <p>
         * Provides a list of VPC security groups that the DB proxy endpoint belongs to.
         * </p>
         * 
         * @param vpcSecurityGroupIds
         *        Provides a list of VPC security groups that the DB proxy endpoint belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSecurityGroupIds(String... vpcSecurityGroupIds);

        /**
         * <p>
         * The EC2 subnet IDs for the DB proxy endpoint.
         * </p>
         * 
         * @param vpcSubnetIds
         *        The EC2 subnet IDs for the DB proxy endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSubnetIds(Collection<String> vpcSubnetIds);

        /**
         * <p>
         * The EC2 subnet IDs for the DB proxy endpoint.
         * </p>
         * 
         * @param vpcSubnetIds
         *        The EC2 subnet IDs for the DB proxy endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSubnetIds(String... vpcSubnetIds);

        /**
         * <p>
         * The endpoint that you can use to connect to the DB proxy. You include the endpoint value in the connection
         * string for a database client application.
         * </p>
         * 
         * @param endpoint
         *        The endpoint that you can use to connect to the DB proxy. You include the endpoint value in the
         *        connection string for a database client application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoint(String endpoint);

        /**
         * <p>
         * The date and time when the DB proxy endpoint was first created.
         * </p>
         * 
         * @param createdDate
         *        The date and time when the DB proxy endpoint was first created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdDate(Instant createdDate);

        /**
         * <p>
         * A value that indicates whether the DB proxy endpoint can be used for read/write or read-only operations.
         * </p>
         * 
         * @param targetRole
         *        A value that indicates whether the DB proxy endpoint can be used for read/write or read-only
         *        operations.
         * @see DBProxyEndpointTargetRole
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DBProxyEndpointTargetRole
         */
        Builder targetRole(String targetRole);

        /**
         * <p>
         * A value that indicates whether the DB proxy endpoint can be used for read/write or read-only operations.
         * </p>
         * 
         * @param targetRole
         *        A value that indicates whether the DB proxy endpoint can be used for read/write or read-only
         *        operations.
         * @see DBProxyEndpointTargetRole
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DBProxyEndpointTargetRole
         */
        Builder targetRole(DBProxyEndpointTargetRole targetRole);

        /**
         * <p>
         * A value that indicates whether this endpoint is the default endpoint for the associated DB proxy. Default DB
         * proxy endpoints always have read/write capability. Other endpoints that you associate with the DB proxy can
         * be either read/write or read-only.
         * </p>
         * 
         * @param isDefault
         *        A value that indicates whether this endpoint is the default endpoint for the associated DB proxy.
         *        Default DB proxy endpoints always have read/write capability. Other endpoints that you associate with
         *        the DB proxy can be either read/write or read-only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isDefault(Boolean isDefault);
    }

    static final class BuilderImpl implements Builder {
        private String dbProxyEndpointName;

        private String dbProxyEndpointArn;

        private String dbProxyName;

        private String status;

        private String vpcId;

        private List<String> vpcSecurityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> vpcSubnetIds = DefaultSdkAutoConstructList.getInstance();

        private String endpoint;

        private Instant createdDate;

        private String targetRole;

        private Boolean isDefault;

        private BuilderImpl() {
        }

        private BuilderImpl(DBProxyEndpoint model) {
            dbProxyEndpointName(model.dbProxyEndpointName);
            dbProxyEndpointArn(model.dbProxyEndpointArn);
            dbProxyName(model.dbProxyName);
            status(model.status);
            vpcId(model.vpcId);
            vpcSecurityGroupIds(model.vpcSecurityGroupIds);
            vpcSubnetIds(model.vpcSubnetIds);
            endpoint(model.endpoint);
            createdDate(model.createdDate);
            targetRole(model.targetRole);
            isDefault(model.isDefault);
        }

        public final String getDbProxyEndpointName() {
            return dbProxyEndpointName;
        }

        public final void setDbProxyEndpointName(String dbProxyEndpointName) {
            this.dbProxyEndpointName = dbProxyEndpointName;
        }

        @Override
        @Transient
        public final Builder dbProxyEndpointName(String dbProxyEndpointName) {
            this.dbProxyEndpointName = dbProxyEndpointName;
            return this;
        }

        public final String getDbProxyEndpointArn() {
            return dbProxyEndpointArn;
        }

        public final void setDbProxyEndpointArn(String dbProxyEndpointArn) {
            this.dbProxyEndpointArn = dbProxyEndpointArn;
        }

        @Override
        @Transient
        public final Builder dbProxyEndpointArn(String dbProxyEndpointArn) {
            this.dbProxyEndpointArn = dbProxyEndpointArn;
            return this;
        }

        public final String getDbProxyName() {
            return dbProxyName;
        }

        public final void setDbProxyName(String dbProxyName) {
            this.dbProxyName = dbProxyName;
        }

        @Override
        @Transient
        public final Builder dbProxyName(String dbProxyName) {
            this.dbProxyName = dbProxyName;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        @Transient
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        @Transient
        public final Builder status(DBProxyEndpointStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        @Transient
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final Collection<String> getVpcSecurityGroupIds() {
            if (vpcSecurityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return vpcSecurityGroupIds;
        }

        public final void setVpcSecurityGroupIds(Collection<String> vpcSecurityGroupIds) {
            this.vpcSecurityGroupIds = StringListCopier.copy(vpcSecurityGroupIds);
        }

        @Override
        @Transient
        public final Builder vpcSecurityGroupIds(Collection<String> vpcSecurityGroupIds) {
            this.vpcSecurityGroupIds = StringListCopier.copy(vpcSecurityGroupIds);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder vpcSecurityGroupIds(String... vpcSecurityGroupIds) {
            vpcSecurityGroupIds(Arrays.asList(vpcSecurityGroupIds));
            return this;
        }

        public final Collection<String> getVpcSubnetIds() {
            if (vpcSubnetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return vpcSubnetIds;
        }

        public final void setVpcSubnetIds(Collection<String> vpcSubnetIds) {
            this.vpcSubnetIds = StringListCopier.copy(vpcSubnetIds);
        }

        @Override
        @Transient
        public final Builder vpcSubnetIds(Collection<String> vpcSubnetIds) {
            this.vpcSubnetIds = StringListCopier.copy(vpcSubnetIds);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder vpcSubnetIds(String... vpcSubnetIds) {
            vpcSubnetIds(Arrays.asList(vpcSubnetIds));
            return this;
        }

        public final String getEndpoint() {
            return endpoint;
        }

        public final void setEndpoint(String endpoint) {
            this.endpoint = endpoint;
        }

        @Override
        @Transient
        public final Builder endpoint(String endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        public final Instant getCreatedDate() {
            return createdDate;
        }

        public final void setCreatedDate(Instant createdDate) {
            this.createdDate = createdDate;
        }

        @Override
        @Transient
        public final Builder createdDate(Instant createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public final String getTargetRole() {
            return targetRole;
        }

        public final void setTargetRole(String targetRole) {
            this.targetRole = targetRole;
        }

        @Override
        @Transient
        public final Builder targetRole(String targetRole) {
            this.targetRole = targetRole;
            return this;
        }

        @Override
        @Transient
        public final Builder targetRole(DBProxyEndpointTargetRole targetRole) {
            this.targetRole(targetRole == null ? null : targetRole.toString());
            return this;
        }

        public final Boolean getIsDefault() {
            return isDefault;
        }

        public final void setIsDefault(Boolean isDefault) {
            this.isDefault = isDefault;
        }

        @Override
        @Transient
        public final Builder isDefault(Boolean isDefault) {
            this.isDefault = isDefault;
            return this;
        }

        @Override
        public DBProxyEndpoint build() {
            return new DBProxyEndpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
