/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the details of an Amazon Aurora DB cluster or Multi-AZ DB cluster.
 * </p>
 * <p>
 * For an Amazon Aurora DB cluster, this data type is used as a response element in the operations
 * <code>CreateDBCluster</code>, <code>DeleteDBCluster</code>, <code>DescribeDBClusters</code>,
 * <code>FailoverDBCluster</code>, <code>ModifyDBCluster</code>, <code>PromoteReadReplicaDBCluster</code>,
 * <code>RestoreDBClusterFromS3</code>, <code>RestoreDBClusterFromSnapshot</code>,
 * <code>RestoreDBClusterToPointInTime</code>, <code>StartDBCluster</code>, and <code>StopDBCluster</code>.
 * </p>
 * <p>
 * For a Multi-AZ DB cluster, this data type is used as a response element in the operations
 * <code>CreateDBCluster</code>, <code>DeleteDBCluster</code>, <code>DescribeDBClusters</code>,
 * <code>FailoverDBCluster</code>, <code>ModifyDBCluster</code>, <code>RebootDBCluster</code>,
 * <code>RestoreDBClusterFromSnapshot</code>, and <code>RestoreDBClusterToPointInTime</code>.
 * </p>
 * <p>
 * For more information on Amazon Aurora DB clusters, see <a
 * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/CHAP_AuroraOverview.html"> What is Amazon
 * Aurora?</a> in the <i>Amazon Aurora User Guide.</i>
 * </p>
 * <p>
 * For more information on Multi-AZ DB clusters, see <a
 * href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/multi-az-db-clusters-concepts.html"> Multi-AZ
 * deployments with two readable standby DB instances</a> in the <i>Amazon RDS User Guide.</i>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DBCluster implements SdkPojo, Serializable, ToCopyableBuilder<DBCluster.Builder, DBCluster> {
    private static final SdkField<Integer> ALLOCATED_STORAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("AllocatedStorage").getter(getter(DBCluster::allocatedStorage)).setter(setter(Builder::allocatedStorage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocatedStorage").build()).build();

    private static final SdkField<List<String>> AVAILABILITY_ZONES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AvailabilityZones")
            .getter(getter(DBCluster::availabilityZones))
            .setter(setter(Builder::availabilityZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("AvailabilityZone")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("AvailabilityZone").build()).build()).build()).build();

    private static final SdkField<Integer> BACKUP_RETENTION_PERIOD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("BackupRetentionPeriod").getter(getter(DBCluster::backupRetentionPeriod))
            .setter(setter(Builder::backupRetentionPeriod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupRetentionPeriod").build())
            .build();

    private static final SdkField<String> CHARACTER_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CharacterSetName").getter(getter(DBCluster::characterSetName)).setter(setter(Builder::characterSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CharacterSetName").build()).build();

    private static final SdkField<String> DATABASE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DatabaseName").getter(getter(DBCluster::databaseName)).setter(setter(Builder::databaseName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseName").build()).build();

    private static final SdkField<String> DB_CLUSTER_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBClusterIdentifier").getter(getter(DBCluster::dbClusterIdentifier))
            .setter(setter(Builder::dbClusterIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterIdentifier").build())
            .build();

    private static final SdkField<String> DB_CLUSTER_PARAMETER_GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBClusterParameterGroup").getter(getter(DBCluster::dbClusterParameterGroup))
            .setter(setter(Builder::dbClusterParameterGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterParameterGroup").build())
            .build();

    private static final SdkField<String> DB_SUBNET_GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBSubnetGroup").getter(getter(DBCluster::dbSubnetGroup)).setter(setter(Builder::dbSubnetGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBSubnetGroup").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(DBCluster::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> AUTOMATIC_RESTART_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("AutomaticRestartTime").getter(getter(DBCluster::automaticRestartTime))
            .setter(setter(Builder::automaticRestartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutomaticRestartTime").build())
            .build();

    private static final SdkField<String> PERCENT_PROGRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PercentProgress").getter(getter(DBCluster::percentProgress)).setter(setter(Builder::percentProgress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PercentProgress").build()).build();

    private static final SdkField<Instant> EARLIEST_RESTORABLE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EarliestRestorableTime").getter(getter(DBCluster::earliestRestorableTime))
            .setter(setter(Builder::earliestRestorableTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EarliestRestorableTime").build())
            .build();

    private static final SdkField<String> ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Endpoint").getter(getter(DBCluster::endpoint)).setter(setter(Builder::endpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoint").build()).build();

    private static final SdkField<String> READER_ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReaderEndpoint").getter(getter(DBCluster::readerEndpoint)).setter(setter(Builder::readerEndpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReaderEndpoint").build()).build();

    private static final SdkField<List<String>> CUSTOM_ENDPOINTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("CustomEndpoints")
            .getter(getter(DBCluster::customEndpoints))
            .setter(setter(Builder::customEndpoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomEndpoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> MULTI_AZ_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("MultiAZ").getter(getter(DBCluster::multiAZ)).setter(setter(Builder::multiAZ))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MultiAZ").build()).build();

    private static final SdkField<String> ENGINE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Engine")
            .getter(getter(DBCluster::engine)).setter(setter(Builder::engine))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Engine").build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EngineVersion").getter(getter(DBCluster::engineVersion)).setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<Instant> LATEST_RESTORABLE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LatestRestorableTime").getter(getter(DBCluster::latestRestorableTime))
            .setter(setter(Builder::latestRestorableTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LatestRestorableTime").build())
            .build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Port")
            .getter(getter(DBCluster::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<String> MASTER_USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MasterUsername").getter(getter(DBCluster::masterUsername)).setter(setter(Builder::masterUsername))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MasterUsername").build()).build();

    private static final SdkField<List<DBClusterOptionGroupStatus>> DB_CLUSTER_OPTION_GROUP_MEMBERSHIPS_FIELD = SdkField
            .<List<DBClusterOptionGroupStatus>> builder(MarshallingType.LIST)
            .memberName("DBClusterOptionGroupMemberships")
            .getter(getter(DBCluster::dbClusterOptionGroupMemberships))
            .setter(setter(Builder::dbClusterOptionGroupMemberships))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterOptionGroupMemberships")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName("DBClusterOptionGroup")
                            .memberFieldInfo(
                                    SdkField.<DBClusterOptionGroupStatus> builder(MarshallingType.SDK_POJO)
                                            .constructor(DBClusterOptionGroupStatus::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("DBClusterOptionGroup").build()).build()).build()).build();

    private static final SdkField<String> PREFERRED_BACKUP_WINDOW_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreferredBackupWindow").getter(getter(DBCluster::preferredBackupWindow))
            .setter(setter(Builder::preferredBackupWindow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredBackupWindow").build())
            .build();

    private static final SdkField<String> PREFERRED_MAINTENANCE_WINDOW_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PreferredMaintenanceWindow")
            .getter(getter(DBCluster::preferredMaintenanceWindow))
            .setter(setter(Builder::preferredMaintenanceWindow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredMaintenanceWindow").build())
            .build();

    private static final SdkField<String> REPLICATION_SOURCE_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ReplicationSourceIdentifier")
            .getter(getter(DBCluster::replicationSourceIdentifier))
            .setter(setter(Builder::replicationSourceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationSourceIdentifier")
                    .build()).build();

    private static final SdkField<List<String>> READ_REPLICA_IDENTIFIERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ReadReplicaIdentifiers")
            .getter(getter(DBCluster::readReplicaIdentifiers))
            .setter(setter(Builder::readReplicaIdentifiers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReadReplicaIdentifiers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("ReadReplicaIdentifier")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("ReadReplicaIdentifier").build()).build()).build()).build();

    private static final SdkField<List<DBClusterStatusInfo>> STATUS_INFOS_FIELD = SdkField
            .<List<DBClusterStatusInfo>> builder(MarshallingType.LIST)
            .memberName("StatusInfos")
            .getter(getter(DBCluster::statusInfos))
            .setter(setter(Builder::statusInfos))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusInfos").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("DBClusterStatusInfo")
                            .memberFieldInfo(
                                    SdkField.<DBClusterStatusInfo> builder(MarshallingType.SDK_POJO)
                                            .constructor(DBClusterStatusInfo::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("DBClusterStatusInfo").build()).build()).build()).build();

    private static final SdkField<List<DBClusterMember>> DB_CLUSTER_MEMBERS_FIELD = SdkField
            .<List<DBClusterMember>> builder(MarshallingType.LIST)
            .memberName("DBClusterMembers")
            .getter(getter(DBCluster::dbClusterMembers))
            .setter(setter(Builder::dbClusterMembers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterMembers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("DBClusterMember")
                            .memberFieldInfo(
                                    SdkField.<DBClusterMember> builder(MarshallingType.SDK_POJO)
                                            .constructor(DBClusterMember::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("DBClusterMember").build()).build()).build()).build();

    private static final SdkField<List<VpcSecurityGroupMembership>> VPC_SECURITY_GROUPS_FIELD = SdkField
            .<List<VpcSecurityGroupMembership>> builder(MarshallingType.LIST)
            .memberName("VpcSecurityGroups")
            .getter(getter(DBCluster::vpcSecurityGroups))
            .setter(setter(Builder::vpcSecurityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcSecurityGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("VpcSecurityGroupMembership")
                            .memberFieldInfo(
                                    SdkField.<VpcSecurityGroupMembership> builder(MarshallingType.SDK_POJO)
                                            .constructor(VpcSecurityGroupMembership::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("VpcSecurityGroupMembership").build()).build()).build())
            .build();

    private static final SdkField<String> HOSTED_ZONE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HostedZoneId").getter(getter(DBCluster::hostedZoneId)).setter(setter(Builder::hostedZoneId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HostedZoneId").build()).build();

    private static final SdkField<Boolean> STORAGE_ENCRYPTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("StorageEncrypted").getter(getter(DBCluster::storageEncrypted)).setter(setter(Builder::storageEncrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageEncrypted").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(DBCluster::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<String> DB_CLUSTER_RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DbClusterResourceId").getter(getter(DBCluster::dbClusterResourceId))
            .setter(setter(Builder::dbClusterResourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DbClusterResourceId").build())
            .build();

    private static final SdkField<String> DB_CLUSTER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBClusterArn").getter(getter(DBCluster::dbClusterArn)).setter(setter(Builder::dbClusterArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterArn").build()).build();

    private static final SdkField<List<DBClusterRole>> ASSOCIATED_ROLES_FIELD = SdkField
            .<List<DBClusterRole>> builder(MarshallingType.LIST)
            .memberName("AssociatedRoles")
            .getter(getter(DBCluster::associatedRoles))
            .setter(setter(Builder::associatedRoles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociatedRoles").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("DBClusterRole")
                            .memberFieldInfo(
                                    SdkField.<DBClusterRole> builder(MarshallingType.SDK_POJO)
                                            .constructor(DBClusterRole::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("DBClusterRole").build()).build()).build()).build();

    private static final SdkField<Boolean> IAM_DATABASE_AUTHENTICATION_ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IAMDatabaseAuthenticationEnabled")
            .getter(getter(DBCluster::iamDatabaseAuthenticationEnabled))
            .setter(setter(Builder::iamDatabaseAuthenticationEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IAMDatabaseAuthenticationEnabled")
                    .build()).build();

    private static final SdkField<String> CLONE_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CloneGroupId").getter(getter(DBCluster::cloneGroupId)).setter(setter(Builder::cloneGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloneGroupId").build()).build();

    private static final SdkField<Instant> CLUSTER_CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ClusterCreateTime").getter(getter(DBCluster::clusterCreateTime))
            .setter(setter(Builder::clusterCreateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterCreateTime").build()).build();

    private static final SdkField<Instant> EARLIEST_BACKTRACK_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EarliestBacktrackTime").getter(getter(DBCluster::earliestBacktrackTime))
            .setter(setter(Builder::earliestBacktrackTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EarliestBacktrackTime").build())
            .build();

    private static final SdkField<Long> BACKTRACK_WINDOW_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("BacktrackWindow").getter(getter(DBCluster::backtrackWindow)).setter(setter(Builder::backtrackWindow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BacktrackWindow").build()).build();

    private static final SdkField<Long> BACKTRACK_CONSUMED_CHANGE_RECORDS_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("BacktrackConsumedChangeRecords")
            .getter(getter(DBCluster::backtrackConsumedChangeRecords))
            .setter(setter(Builder::backtrackConsumedChangeRecords))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BacktrackConsumedChangeRecords")
                    .build()).build();

    private static final SdkField<List<String>> ENABLED_CLOUDWATCH_LOGS_EXPORTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EnabledCloudwatchLogsExports")
            .getter(getter(DBCluster::enabledCloudwatchLogsExports))
            .setter(setter(Builder::enabledCloudwatchLogsExports))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnabledCloudwatchLogsExports")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Capacity").getter(getter(DBCluster::capacity)).setter(setter(Builder::capacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Capacity").build()).build();

    private static final SdkField<String> ENGINE_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EngineMode").getter(getter(DBCluster::engineMode)).setter(setter(Builder::engineMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineMode").build()).build();

    private static final SdkField<ScalingConfigurationInfo> SCALING_CONFIGURATION_INFO_FIELD = SdkField
            .<ScalingConfigurationInfo> builder(MarshallingType.SDK_POJO).memberName("ScalingConfigurationInfo")
            .getter(getter(DBCluster::scalingConfigurationInfo)).setter(setter(Builder::scalingConfigurationInfo))
            .constructor(ScalingConfigurationInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingConfigurationInfo").build())
            .build();

    private static final SdkField<RdsCustomClusterConfiguration> RDS_CUSTOM_CLUSTER_CONFIGURATION_FIELD = SdkField
            .<RdsCustomClusterConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("RdsCustomClusterConfiguration")
            .getter(getter(DBCluster::rdsCustomClusterConfiguration))
            .setter(setter(Builder::rdsCustomClusterConfiguration))
            .constructor(RdsCustomClusterConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RdsCustomClusterConfiguration")
                    .build()).build();

    private static final SdkField<Boolean> DELETION_PROTECTION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DeletionProtection").getter(getter(DBCluster::deletionProtection))
            .setter(setter(Builder::deletionProtection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeletionProtection").build())
            .build();

    private static final SdkField<Boolean> HTTP_ENDPOINT_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("HttpEndpointEnabled").getter(getter(DBCluster::httpEndpointEnabled))
            .setter(setter(Builder::httpEndpointEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HttpEndpointEnabled").build())
            .build();

    private static final SdkField<String> ACTIVITY_STREAM_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ActivityStreamMode").getter(getter(DBCluster::activityStreamModeAsString))
            .setter(setter(Builder::activityStreamMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActivityStreamMode").build())
            .build();

    private static final SdkField<String> ACTIVITY_STREAM_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ActivityStreamStatus").getter(getter(DBCluster::activityStreamStatusAsString))
            .setter(setter(Builder::activityStreamStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActivityStreamStatus").build())
            .build();

    private static final SdkField<String> ACTIVITY_STREAM_KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ActivityStreamKmsKeyId").getter(getter(DBCluster::activityStreamKmsKeyId))
            .setter(setter(Builder::activityStreamKmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActivityStreamKmsKeyId").build())
            .build();

    private static final SdkField<String> ACTIVITY_STREAM_KINESIS_STREAM_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ActivityStreamKinesisStreamName")
            .getter(getter(DBCluster::activityStreamKinesisStreamName))
            .setter(setter(Builder::activityStreamKinesisStreamName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActivityStreamKinesisStreamName")
                    .build()).build();

    private static final SdkField<Boolean> COPY_TAGS_TO_SNAPSHOT_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CopyTagsToSnapshot").getter(getter(DBCluster::copyTagsToSnapshot))
            .setter(setter(Builder::copyTagsToSnapshot))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyTagsToSnapshot").build())
            .build();

    private static final SdkField<Boolean> CROSS_ACCOUNT_CLONE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CrossAccountClone").getter(getter(DBCluster::crossAccountClone))
            .setter(setter(Builder::crossAccountClone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CrossAccountClone").build()).build();

    private static final SdkField<List<DomainMembership>> DOMAIN_MEMBERSHIPS_FIELD = SdkField
            .<List<DomainMembership>> builder(MarshallingType.LIST)
            .memberName("DomainMemberships")
            .getter(getter(DBCluster::domainMemberships))
            .setter(setter(Builder::domainMemberships))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainMemberships").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("DomainMembership")
                            .memberFieldInfo(
                                    SdkField.<DomainMembership> builder(MarshallingType.SDK_POJO)
                                            .constructor(DomainMembership::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("DomainMembership").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAG_LIST_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("TagList")
            .getter(getter(DBCluster::tagList))
            .setter(setter(Builder::tagList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Tag")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Tag").build()).build()).build()).build();

    private static final SdkField<String> GLOBAL_WRITE_FORWARDING_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("GlobalWriteForwardingStatus")
            .getter(getter(DBCluster::globalWriteForwardingStatusAsString))
            .setter(setter(Builder::globalWriteForwardingStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GlobalWriteForwardingStatus")
                    .build()).build();

    private static final SdkField<Boolean> GLOBAL_WRITE_FORWARDING_REQUESTED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("GlobalWriteForwardingRequested")
            .getter(getter(DBCluster::globalWriteForwardingRequested))
            .setter(setter(Builder::globalWriteForwardingRequested))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GlobalWriteForwardingRequested")
                    .build()).build();

    private static final SdkField<ClusterPendingModifiedValues> PENDING_MODIFIED_VALUES_FIELD = SdkField
            .<ClusterPendingModifiedValues> builder(MarshallingType.SDK_POJO).memberName("PendingModifiedValues")
            .getter(getter(DBCluster::pendingModifiedValues)).setter(setter(Builder::pendingModifiedValues))
            .constructor(ClusterPendingModifiedValues::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PendingModifiedValues").build())
            .build();

    private static final SdkField<String> DB_CLUSTER_INSTANCE_CLASS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBClusterInstanceClass").getter(getter(DBCluster::dbClusterInstanceClass))
            .setter(setter(Builder::dbClusterInstanceClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterInstanceClass").build())
            .build();

    private static final SdkField<String> STORAGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StorageType").getter(getter(DBCluster::storageType)).setter(setter(Builder::storageType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageType").build()).build();

    private static final SdkField<Integer> IOPS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Iops")
            .getter(getter(DBCluster::iops)).setter(setter(Builder::iops))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Iops").build()).build();

    private static final SdkField<Boolean> PUBLICLY_ACCESSIBLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("PubliclyAccessible").getter(getter(DBCluster::publiclyAccessible))
            .setter(setter(Builder::publiclyAccessible))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PubliclyAccessible").build())
            .build();

    private static final SdkField<Boolean> AUTO_MINOR_VERSION_UPGRADE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AutoMinorVersionUpgrade").getter(getter(DBCluster::autoMinorVersionUpgrade))
            .setter(setter(Builder::autoMinorVersionUpgrade))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoMinorVersionUpgrade").build())
            .build();

    private static final SdkField<Integer> MONITORING_INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MonitoringInterval").getter(getter(DBCluster::monitoringInterval))
            .setter(setter(Builder::monitoringInterval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MonitoringInterval").build())
            .build();

    private static final SdkField<String> MONITORING_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MonitoringRoleArn").getter(getter(DBCluster::monitoringRoleArn))
            .setter(setter(Builder::monitoringRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MonitoringRoleArn").build()).build();

    private static final SdkField<Boolean> PERFORMANCE_INSIGHTS_ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("PerformanceInsightsEnabled")
            .getter(getter(DBCluster::performanceInsightsEnabled))
            .setter(setter(Builder::performanceInsightsEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PerformanceInsightsEnabled").build())
            .build();

    private static final SdkField<String> PERFORMANCE_INSIGHTS_KMS_KEY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PerformanceInsightsKMSKeyId")
            .getter(getter(DBCluster::performanceInsightsKMSKeyId))
            .setter(setter(Builder::performanceInsightsKMSKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PerformanceInsightsKMSKeyId")
                    .build()).build();

    private static final SdkField<Integer> PERFORMANCE_INSIGHTS_RETENTION_PERIOD_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("PerformanceInsightsRetentionPeriod")
            .getter(getter(DBCluster::performanceInsightsRetentionPeriod))
            .setter(setter(Builder::performanceInsightsRetentionPeriod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PerformanceInsightsRetentionPeriod")
                    .build()).build();

    private static final SdkField<ServerlessV2ScalingConfigurationInfo> SERVERLESS_V2_SCALING_CONFIGURATION_FIELD = SdkField
            .<ServerlessV2ScalingConfigurationInfo> builder(MarshallingType.SDK_POJO)
            .memberName("ServerlessV2ScalingConfiguration")
            .getter(getter(DBCluster::serverlessV2ScalingConfiguration))
            .setter(setter(Builder::serverlessV2ScalingConfiguration))
            .constructor(ServerlessV2ScalingConfigurationInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerlessV2ScalingConfiguration")
                    .build()).build();

    private static final SdkField<String> NETWORK_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NetworkType").getter(getter(DBCluster::networkType)).setter(setter(Builder::networkType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkType").build()).build();

    private static final SdkField<String> DB_SYSTEM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBSystemId").getter(getter(DBCluster::dbSystemId)).setter(setter(Builder::dbSystemId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBSystemId").build()).build();

    private static final SdkField<MasterUserSecret> MASTER_USER_SECRET_FIELD = SdkField
            .<MasterUserSecret> builder(MarshallingType.SDK_POJO).memberName("MasterUserSecret")
            .getter(getter(DBCluster::masterUserSecret)).setter(setter(Builder::masterUserSecret))
            .constructor(MasterUserSecret::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MasterUserSecret").build()).build();

    private static final SdkField<Instant> IO_OPTIMIZED_NEXT_ALLOWED_MODIFICATION_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("IOOptimizedNextAllowedModificationTime")
            .getter(getter(DBCluster::ioOptimizedNextAllowedModificationTime))
            .setter(setter(Builder::ioOptimizedNextAllowedModificationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("IOOptimizedNextAllowedModificationTime").build()).build();

    private static final SdkField<String> LOCAL_WRITE_FORWARDING_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LocalWriteForwardingStatus")
            .getter(getter(DBCluster::localWriteForwardingStatusAsString))
            .setter(setter(Builder::localWriteForwardingStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocalWriteForwardingStatus").build())
            .build();

    private static final SdkField<String> AWS_BACKUP_RECOVERY_POINT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AwsBackupRecoveryPointArn").getter(getter(DBCluster::awsBackupRecoveryPointArn))
            .setter(setter(Builder::awsBackupRecoveryPointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AwsBackupRecoveryPointArn").build())
            .build();

    private static final SdkField<LimitlessDatabase> LIMITLESS_DATABASE_FIELD = SdkField
            .<LimitlessDatabase> builder(MarshallingType.SDK_POJO).memberName("LimitlessDatabase")
            .getter(getter(DBCluster::limitlessDatabase)).setter(setter(Builder::limitlessDatabase))
            .constructor(LimitlessDatabase::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LimitlessDatabase").build()).build();

    private static final SdkField<Integer> STORAGE_THROUGHPUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("StorageThroughput").getter(getter(DBCluster::storageThroughput))
            .setter(setter(Builder::storageThroughput))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageThroughput").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALLOCATED_STORAGE_FIELD,
            AVAILABILITY_ZONES_FIELD, BACKUP_RETENTION_PERIOD_FIELD, CHARACTER_SET_NAME_FIELD, DATABASE_NAME_FIELD,
            DB_CLUSTER_IDENTIFIER_FIELD, DB_CLUSTER_PARAMETER_GROUP_FIELD, DB_SUBNET_GROUP_FIELD, STATUS_FIELD,
            AUTOMATIC_RESTART_TIME_FIELD, PERCENT_PROGRESS_FIELD, EARLIEST_RESTORABLE_TIME_FIELD, ENDPOINT_FIELD,
            READER_ENDPOINT_FIELD, CUSTOM_ENDPOINTS_FIELD, MULTI_AZ_FIELD, ENGINE_FIELD, ENGINE_VERSION_FIELD,
            LATEST_RESTORABLE_TIME_FIELD, PORT_FIELD, MASTER_USERNAME_FIELD, DB_CLUSTER_OPTION_GROUP_MEMBERSHIPS_FIELD,
            PREFERRED_BACKUP_WINDOW_FIELD, PREFERRED_MAINTENANCE_WINDOW_FIELD, REPLICATION_SOURCE_IDENTIFIER_FIELD,
            READ_REPLICA_IDENTIFIERS_FIELD, STATUS_INFOS_FIELD, DB_CLUSTER_MEMBERS_FIELD, VPC_SECURITY_GROUPS_FIELD,
            HOSTED_ZONE_ID_FIELD, STORAGE_ENCRYPTED_FIELD, KMS_KEY_ID_FIELD, DB_CLUSTER_RESOURCE_ID_FIELD, DB_CLUSTER_ARN_FIELD,
            ASSOCIATED_ROLES_FIELD, IAM_DATABASE_AUTHENTICATION_ENABLED_FIELD, CLONE_GROUP_ID_FIELD, CLUSTER_CREATE_TIME_FIELD,
            EARLIEST_BACKTRACK_TIME_FIELD, BACKTRACK_WINDOW_FIELD, BACKTRACK_CONSUMED_CHANGE_RECORDS_FIELD,
            ENABLED_CLOUDWATCH_LOGS_EXPORTS_FIELD, CAPACITY_FIELD, ENGINE_MODE_FIELD, SCALING_CONFIGURATION_INFO_FIELD,
            RDS_CUSTOM_CLUSTER_CONFIGURATION_FIELD, DELETION_PROTECTION_FIELD, HTTP_ENDPOINT_ENABLED_FIELD,
            ACTIVITY_STREAM_MODE_FIELD, ACTIVITY_STREAM_STATUS_FIELD, ACTIVITY_STREAM_KMS_KEY_ID_FIELD,
            ACTIVITY_STREAM_KINESIS_STREAM_NAME_FIELD, COPY_TAGS_TO_SNAPSHOT_FIELD, CROSS_ACCOUNT_CLONE_FIELD,
            DOMAIN_MEMBERSHIPS_FIELD, TAG_LIST_FIELD, GLOBAL_WRITE_FORWARDING_STATUS_FIELD,
            GLOBAL_WRITE_FORWARDING_REQUESTED_FIELD, PENDING_MODIFIED_VALUES_FIELD, DB_CLUSTER_INSTANCE_CLASS_FIELD,
            STORAGE_TYPE_FIELD, IOPS_FIELD, PUBLICLY_ACCESSIBLE_FIELD, AUTO_MINOR_VERSION_UPGRADE_FIELD,
            MONITORING_INTERVAL_FIELD, MONITORING_ROLE_ARN_FIELD, PERFORMANCE_INSIGHTS_ENABLED_FIELD,
            PERFORMANCE_INSIGHTS_KMS_KEY_ID_FIELD, PERFORMANCE_INSIGHTS_RETENTION_PERIOD_FIELD,
            SERVERLESS_V2_SCALING_CONFIGURATION_FIELD, NETWORK_TYPE_FIELD, DB_SYSTEM_ID_FIELD, MASTER_USER_SECRET_FIELD,
            IO_OPTIMIZED_NEXT_ALLOWED_MODIFICATION_TIME_FIELD, LOCAL_WRITE_FORWARDING_STATUS_FIELD,
            AWS_BACKUP_RECOVERY_POINT_ARN_FIELD, LIMITLESS_DATABASE_FIELD, STORAGE_THROUGHPUT_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer allocatedStorage;

    private final List<String> availabilityZones;

    private final Integer backupRetentionPeriod;

    private final String characterSetName;

    private final String databaseName;

    private final String dbClusterIdentifier;

    private final String dbClusterParameterGroup;

    private final String dbSubnetGroup;

    private final String status;

    private final Instant automaticRestartTime;

    private final String percentProgress;

    private final Instant earliestRestorableTime;

    private final String endpoint;

    private final String readerEndpoint;

    private final List<String> customEndpoints;

    private final Boolean multiAZ;

    private final String engine;

    private final String engineVersion;

    private final Instant latestRestorableTime;

    private final Integer port;

    private final String masterUsername;

    private final List<DBClusterOptionGroupStatus> dbClusterOptionGroupMemberships;

    private final String preferredBackupWindow;

    private final String preferredMaintenanceWindow;

    private final String replicationSourceIdentifier;

    private final List<String> readReplicaIdentifiers;

    private final List<DBClusterStatusInfo> statusInfos;

    private final List<DBClusterMember> dbClusterMembers;

    private final List<VpcSecurityGroupMembership> vpcSecurityGroups;

    private final String hostedZoneId;

    private final Boolean storageEncrypted;

    private final String kmsKeyId;

    private final String dbClusterResourceId;

    private final String dbClusterArn;

    private final List<DBClusterRole> associatedRoles;

    private final Boolean iamDatabaseAuthenticationEnabled;

    private final String cloneGroupId;

    private final Instant clusterCreateTime;

    private final Instant earliestBacktrackTime;

    private final Long backtrackWindow;

    private final Long backtrackConsumedChangeRecords;

    private final List<String> enabledCloudwatchLogsExports;

    private final Integer capacity;

    private final String engineMode;

    private final ScalingConfigurationInfo scalingConfigurationInfo;

    private final RdsCustomClusterConfiguration rdsCustomClusterConfiguration;

    private final Boolean deletionProtection;

    private final Boolean httpEndpointEnabled;

    private final String activityStreamMode;

    private final String activityStreamStatus;

    private final String activityStreamKmsKeyId;

    private final String activityStreamKinesisStreamName;

    private final Boolean copyTagsToSnapshot;

    private final Boolean crossAccountClone;

    private final List<DomainMembership> domainMemberships;

    private final List<Tag> tagList;

    private final String globalWriteForwardingStatus;

    private final Boolean globalWriteForwardingRequested;

    private final ClusterPendingModifiedValues pendingModifiedValues;

    private final String dbClusterInstanceClass;

    private final String storageType;

    private final Integer iops;

    private final Boolean publiclyAccessible;

    private final Boolean autoMinorVersionUpgrade;

    private final Integer monitoringInterval;

    private final String monitoringRoleArn;

    private final Boolean performanceInsightsEnabled;

    private final String performanceInsightsKMSKeyId;

    private final Integer performanceInsightsRetentionPeriod;

    private final ServerlessV2ScalingConfigurationInfo serverlessV2ScalingConfiguration;

    private final String networkType;

    private final String dbSystemId;

    private final MasterUserSecret masterUserSecret;

    private final Instant ioOptimizedNextAllowedModificationTime;

    private final String localWriteForwardingStatus;

    private final String awsBackupRecoveryPointArn;

    private final LimitlessDatabase limitlessDatabase;

    private final Integer storageThroughput;

    private DBCluster(BuilderImpl builder) {
        this.allocatedStorage = builder.allocatedStorage;
        this.availabilityZones = builder.availabilityZones;
        this.backupRetentionPeriod = builder.backupRetentionPeriod;
        this.characterSetName = builder.characterSetName;
        this.databaseName = builder.databaseName;
        this.dbClusterIdentifier = builder.dbClusterIdentifier;
        this.dbClusterParameterGroup = builder.dbClusterParameterGroup;
        this.dbSubnetGroup = builder.dbSubnetGroup;
        this.status = builder.status;
        this.automaticRestartTime = builder.automaticRestartTime;
        this.percentProgress = builder.percentProgress;
        this.earliestRestorableTime = builder.earliestRestorableTime;
        this.endpoint = builder.endpoint;
        this.readerEndpoint = builder.readerEndpoint;
        this.customEndpoints = builder.customEndpoints;
        this.multiAZ = builder.multiAZ;
        this.engine = builder.engine;
        this.engineVersion = builder.engineVersion;
        this.latestRestorableTime = builder.latestRestorableTime;
        this.port = builder.port;
        this.masterUsername = builder.masterUsername;
        this.dbClusterOptionGroupMemberships = builder.dbClusterOptionGroupMemberships;
        this.preferredBackupWindow = builder.preferredBackupWindow;
        this.preferredMaintenanceWindow = builder.preferredMaintenanceWindow;
        this.replicationSourceIdentifier = builder.replicationSourceIdentifier;
        this.readReplicaIdentifiers = builder.readReplicaIdentifiers;
        this.statusInfos = builder.statusInfos;
        this.dbClusterMembers = builder.dbClusterMembers;
        this.vpcSecurityGroups = builder.vpcSecurityGroups;
        this.hostedZoneId = builder.hostedZoneId;
        this.storageEncrypted = builder.storageEncrypted;
        this.kmsKeyId = builder.kmsKeyId;
        this.dbClusterResourceId = builder.dbClusterResourceId;
        this.dbClusterArn = builder.dbClusterArn;
        this.associatedRoles = builder.associatedRoles;
        this.iamDatabaseAuthenticationEnabled = builder.iamDatabaseAuthenticationEnabled;
        this.cloneGroupId = builder.cloneGroupId;
        this.clusterCreateTime = builder.clusterCreateTime;
        this.earliestBacktrackTime = builder.earliestBacktrackTime;
        this.backtrackWindow = builder.backtrackWindow;
        this.backtrackConsumedChangeRecords = builder.backtrackConsumedChangeRecords;
        this.enabledCloudwatchLogsExports = builder.enabledCloudwatchLogsExports;
        this.capacity = builder.capacity;
        this.engineMode = builder.engineMode;
        this.scalingConfigurationInfo = builder.scalingConfigurationInfo;
        this.rdsCustomClusterConfiguration = builder.rdsCustomClusterConfiguration;
        this.deletionProtection = builder.deletionProtection;
        this.httpEndpointEnabled = builder.httpEndpointEnabled;
        this.activityStreamMode = builder.activityStreamMode;
        this.activityStreamStatus = builder.activityStreamStatus;
        this.activityStreamKmsKeyId = builder.activityStreamKmsKeyId;
        this.activityStreamKinesisStreamName = builder.activityStreamKinesisStreamName;
        this.copyTagsToSnapshot = builder.copyTagsToSnapshot;
        this.crossAccountClone = builder.crossAccountClone;
        this.domainMemberships = builder.domainMemberships;
        this.tagList = builder.tagList;
        this.globalWriteForwardingStatus = builder.globalWriteForwardingStatus;
        this.globalWriteForwardingRequested = builder.globalWriteForwardingRequested;
        this.pendingModifiedValues = builder.pendingModifiedValues;
        this.dbClusterInstanceClass = builder.dbClusterInstanceClass;
        this.storageType = builder.storageType;
        this.iops = builder.iops;
        this.publiclyAccessible = builder.publiclyAccessible;
        this.autoMinorVersionUpgrade = builder.autoMinorVersionUpgrade;
        this.monitoringInterval = builder.monitoringInterval;
        this.monitoringRoleArn = builder.monitoringRoleArn;
        this.performanceInsightsEnabled = builder.performanceInsightsEnabled;
        this.performanceInsightsKMSKeyId = builder.performanceInsightsKMSKeyId;
        this.performanceInsightsRetentionPeriod = builder.performanceInsightsRetentionPeriod;
        this.serverlessV2ScalingConfiguration = builder.serverlessV2ScalingConfiguration;
        this.networkType = builder.networkType;
        this.dbSystemId = builder.dbSystemId;
        this.masterUserSecret = builder.masterUserSecret;
        this.ioOptimizedNextAllowedModificationTime = builder.ioOptimizedNextAllowedModificationTime;
        this.localWriteForwardingStatus = builder.localWriteForwardingStatus;
        this.awsBackupRecoveryPointArn = builder.awsBackupRecoveryPointArn;
        this.limitlessDatabase = builder.limitlessDatabase;
        this.storageThroughput = builder.storageThroughput;
    }

    /**
     * <p>
     * For all database engines except Amazon Aurora, <code>AllocatedStorage</code> specifies the allocated storage size
     * in gibibytes (GiB). For Aurora, <code>AllocatedStorage</code> always returns 1, because Aurora DB cluster storage
     * size isn't fixed, but instead automatically adjusts as needed.
     * </p>
     * 
     * @return For all database engines except Amazon Aurora, <code>AllocatedStorage</code> specifies the allocated
     *         storage size in gibibytes (GiB). For Aurora, <code>AllocatedStorage</code> always returns 1, because
     *         Aurora DB cluster storage size isn't fixed, but instead automatically adjusts as needed.
     */
    public final Integer allocatedStorage() {
        return allocatedStorage;
    }

    /**
     * For responses, this returns true if the service returned a value for the AvailabilityZones property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAvailabilityZones() {
        return availabilityZones != null && !(availabilityZones instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of Availability Zones (AZs) where instances in the DB cluster can be created.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAvailabilityZones} method.
     * </p>
     * 
     * @return The list of Availability Zones (AZs) where instances in the DB cluster can be created.
     */
    public final List<String> availabilityZones() {
        return availabilityZones;
    }

    /**
     * <p>
     * The number of days for which automatic DB snapshots are retained.
     * </p>
     * 
     * @return The number of days for which automatic DB snapshots are retained.
     */
    public final Integer backupRetentionPeriod() {
        return backupRetentionPeriod;
    }

    /**
     * <p>
     * If present, specifies the name of the character set that this cluster is associated with.
     * </p>
     * 
     * @return If present, specifies the name of the character set that this cluster is associated with.
     */
    public final String characterSetName() {
        return characterSetName;
    }

    /**
     * <p>
     * The name of the initial database that was specified for the DB cluster when it was created, if one was provided.
     * This same name is returned for the life of the DB cluster.
     * </p>
     * 
     * @return The name of the initial database that was specified for the DB cluster when it was created, if one was
     *         provided. This same name is returned for the life of the DB cluster.
     */
    public final String databaseName() {
        return databaseName;
    }

    /**
     * <p>
     * The user-supplied identifier for the DB cluster. This identifier is the unique key that identifies a DB cluster.
     * </p>
     * 
     * @return The user-supplied identifier for the DB cluster. This identifier is the unique key that identifies a DB
     *         cluster.
     */
    public final String dbClusterIdentifier() {
        return dbClusterIdentifier;
    }

    /**
     * <p>
     * The name of the DB cluster parameter group for the DB cluster.
     * </p>
     * 
     * @return The name of the DB cluster parameter group for the DB cluster.
     */
    public final String dbClusterParameterGroup() {
        return dbClusterParameterGroup;
    }

    /**
     * <p>
     * Information about the subnet group associated with the DB cluster, including the name, description, and subnets
     * in the subnet group.
     * </p>
     * 
     * @return Information about the subnet group associated with the DB cluster, including the name, description, and
     *         subnets in the subnet group.
     */
    public final String dbSubnetGroup() {
        return dbSubnetGroup;
    }

    /**
     * <p>
     * The current state of this DB cluster.
     * </p>
     * 
     * @return The current state of this DB cluster.
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * The time when a stopped DB cluster is restarted automatically.
     * </p>
     * 
     * @return The time when a stopped DB cluster is restarted automatically.
     */
    public final Instant automaticRestartTime() {
        return automaticRestartTime;
    }

    /**
     * <p>
     * The progress of the operation as a percentage.
     * </p>
     * 
     * @return The progress of the operation as a percentage.
     */
    public final String percentProgress() {
        return percentProgress;
    }

    /**
     * <p>
     * The earliest time to which a database can be restored with point-in-time restore.
     * </p>
     * 
     * @return The earliest time to which a database can be restored with point-in-time restore.
     */
    public final Instant earliestRestorableTime() {
        return earliestRestorableTime;
    }

    /**
     * <p>
     * The connection endpoint for the primary instance of the DB cluster.
     * </p>
     * 
     * @return The connection endpoint for the primary instance of the DB cluster.
     */
    public final String endpoint() {
        return endpoint;
    }

    /**
     * <p>
     * The reader endpoint for the DB cluster. The reader endpoint for a DB cluster load-balances connections across the
     * Aurora Replicas that are available in a DB cluster. As clients request new connections to the reader endpoint,
     * Aurora distributes the connection requests among the Aurora Replicas in the DB cluster. This functionality can
     * help balance your read workload across multiple Aurora Replicas in your DB cluster.
     * </p>
     * <p>
     * If a failover occurs, and the Aurora Replica that you are connected to is promoted to be the primary instance,
     * your connection is dropped. To continue sending your read workload to other Aurora Replicas in the cluster, you
     * can then reconnect to the reader endpoint.
     * </p>
     * 
     * @return The reader endpoint for the DB cluster. The reader endpoint for a DB cluster load-balances connections
     *         across the Aurora Replicas that are available in a DB cluster. As clients request new connections to the
     *         reader endpoint, Aurora distributes the connection requests among the Aurora Replicas in the DB cluster.
     *         This functionality can help balance your read workload across multiple Aurora Replicas in your DB
     *         cluster.</p>
     *         <p>
     *         If a failover occurs, and the Aurora Replica that you are connected to is promoted to be the primary
     *         instance, your connection is dropped. To continue sending your read workload to other Aurora Replicas in
     *         the cluster, you can then reconnect to the reader endpoint.
     */
    public final String readerEndpoint() {
        return readerEndpoint;
    }

    /**
     * For responses, this returns true if the service returned a value for the CustomEndpoints property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCustomEndpoints() {
        return customEndpoints != null && !(customEndpoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The custom endpoints associated with the DB cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCustomEndpoints} method.
     * </p>
     * 
     * @return The custom endpoints associated with the DB cluster.
     */
    public final List<String> customEndpoints() {
        return customEndpoints;
    }

    /**
     * <p>
     * Indicates whether the DB cluster has instances in multiple Availability Zones.
     * </p>
     * 
     * @return Indicates whether the DB cluster has instances in multiple Availability Zones.
     */
    public final Boolean multiAZ() {
        return multiAZ;
    }

    /**
     * <p>
     * The database engine used for this DB cluster.
     * </p>
     * 
     * @return The database engine used for this DB cluster.
     */
    public final String engine() {
        return engine;
    }

    /**
     * <p>
     * The version of the database engine.
     * </p>
     * 
     * @return The version of the database engine.
     */
    public final String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * The latest time to which a database can be restored with point-in-time restore.
     * </p>
     * 
     * @return The latest time to which a database can be restored with point-in-time restore.
     */
    public final Instant latestRestorableTime() {
        return latestRestorableTime;
    }

    /**
     * <p>
     * The port that the database engine is listening on.
     * </p>
     * 
     * @return The port that the database engine is listening on.
     */
    public final Integer port() {
        return port;
    }

    /**
     * <p>
     * The master username for the DB cluster.
     * </p>
     * 
     * @return The master username for the DB cluster.
     */
    public final String masterUsername() {
        return masterUsername;
    }

    /**
     * For responses, this returns true if the service returned a value for the DBClusterOptionGroupMemberships
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasDbClusterOptionGroupMemberships() {
        return dbClusterOptionGroupMemberships != null && !(dbClusterOptionGroupMemberships instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of option group memberships for this DB cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDbClusterOptionGroupMemberships}
     * method.
     * </p>
     * 
     * @return The list of option group memberships for this DB cluster.
     */
    public final List<DBClusterOptionGroupStatus> dbClusterOptionGroupMemberships() {
        return dbClusterOptionGroupMemberships;
    }

    /**
     * <p>
     * The daily time range during which automated backups are created if automated backups are enabled, as determined
     * by the <code>BackupRetentionPeriod</code>.
     * </p>
     * 
     * @return The daily time range during which automated backups are created if automated backups are enabled, as
     *         determined by the <code>BackupRetentionPeriod</code>.
     */
    public final String preferredBackupWindow() {
        return preferredBackupWindow;
    }

    /**
     * <p>
     * The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).
     * </p>
     * 
     * @return The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).
     */
    public final String preferredMaintenanceWindow() {
        return preferredMaintenanceWindow;
    }

    /**
     * <p>
     * The identifier of the source DB cluster if this DB cluster is a read replica.
     * </p>
     * 
     * @return The identifier of the source DB cluster if this DB cluster is a read replica.
     */
    public final String replicationSourceIdentifier() {
        return replicationSourceIdentifier;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReadReplicaIdentifiers property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasReadReplicaIdentifiers() {
        return readReplicaIdentifiers != null && !(readReplicaIdentifiers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains one or more identifiers of the read replicas associated with this DB cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReadReplicaIdentifiers} method.
     * </p>
     * 
     * @return Contains one or more identifiers of the read replicas associated with this DB cluster.
     */
    public final List<String> readReplicaIdentifiers() {
        return readReplicaIdentifiers;
    }

    /**
     * For responses, this returns true if the service returned a value for the StatusInfos property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStatusInfos() {
        return statusInfos != null && !(statusInfos instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Reserved for future use.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStatusInfos} method.
     * </p>
     * 
     * @return Reserved for future use.
     */
    public final List<DBClusterStatusInfo> statusInfos() {
        return statusInfos;
    }

    /**
     * For responses, this returns true if the service returned a value for the DBClusterMembers property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDbClusterMembers() {
        return dbClusterMembers != null && !(dbClusterMembers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of DB instances that make up the DB cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDbClusterMembers} method.
     * </p>
     * 
     * @return The list of DB instances that make up the DB cluster.
     */
    public final List<DBClusterMember> dbClusterMembers() {
        return dbClusterMembers;
    }

    /**
     * For responses, this returns true if the service returned a value for the VpcSecurityGroups property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVpcSecurityGroups() {
        return vpcSecurityGroups != null && !(vpcSecurityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of VPC security groups that the DB cluster belongs to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVpcSecurityGroups} method.
     * </p>
     * 
     * @return The list of VPC security groups that the DB cluster belongs to.
     */
    public final List<VpcSecurityGroupMembership> vpcSecurityGroups() {
        return vpcSecurityGroups;
    }

    /**
     * <p>
     * The ID that Amazon Route 53 assigns when you create a hosted zone.
     * </p>
     * 
     * @return The ID that Amazon Route 53 assigns when you create a hosted zone.
     */
    public final String hostedZoneId() {
        return hostedZoneId;
    }

    /**
     * <p>
     * Indicates whether the DB cluster is encrypted.
     * </p>
     * 
     * @return Indicates whether the DB cluster is encrypted.
     */
    public final Boolean storageEncrypted() {
        return storageEncrypted;
    }

    /**
     * <p>
     * If <code>StorageEncrypted</code> is enabled, the Amazon Web Services KMS key identifier for the encrypted DB
     * cluster.
     * </p>
     * <p>
     * The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.
     * </p>
     * 
     * @return If <code>StorageEncrypted</code> is enabled, the Amazon Web Services KMS key identifier for the encrypted
     *         DB cluster.</p>
     *         <p>
     *         The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS
     *         key.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The Amazon Web Services Region-unique, immutable identifier for the DB cluster. This identifier is found in
     * Amazon Web Services CloudTrail log entries whenever the KMS key for the DB cluster is accessed.
     * </p>
     * 
     * @return The Amazon Web Services Region-unique, immutable identifier for the DB cluster. This identifier is found
     *         in Amazon Web Services CloudTrail log entries whenever the KMS key for the DB cluster is accessed.
     */
    public final String dbClusterResourceId() {
        return dbClusterResourceId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the DB cluster.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the DB cluster.
     */
    public final String dbClusterArn() {
        return dbClusterArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the AssociatedRoles property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAssociatedRoles() {
        return associatedRoles != null && !(associatedRoles instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the Amazon Web Services Identity and Access Management (IAM) roles that are associated with the DB
     * cluster. IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other
     * Amazon Web Services on your behalf.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAssociatedRoles} method.
     * </p>
     * 
     * @return A list of the Amazon Web Services Identity and Access Management (IAM) roles that are associated with the
     *         DB cluster. IAM roles that are associated with a DB cluster grant permission for the DB cluster to access
     *         other Amazon Web Services on your behalf.
     */
    public final List<DBClusterRole> associatedRoles() {
        return associatedRoles;
    }

    /**
     * <p>
     * Indicates whether the mapping of Amazon Web Services Identity and Access Management (IAM) accounts to database
     * accounts is enabled.
     * </p>
     * 
     * @return Indicates whether the mapping of Amazon Web Services Identity and Access Management (IAM) accounts to
     *         database accounts is enabled.
     */
    public final Boolean iamDatabaseAuthenticationEnabled() {
        return iamDatabaseAuthenticationEnabled;
    }

    /**
     * <p>
     * The ID of the clone group with which the DB cluster is associated.
     * </p>
     * 
     * @return The ID of the clone group with which the DB cluster is associated.
     */
    public final String cloneGroupId() {
        return cloneGroupId;
    }

    /**
     * <p>
     * The time when the DB cluster was created, in Universal Coordinated Time (UTC).
     * </p>
     * 
     * @return The time when the DB cluster was created, in Universal Coordinated Time (UTC).
     */
    public final Instant clusterCreateTime() {
        return clusterCreateTime;
    }

    /**
     * <p>
     * The earliest time to which a DB cluster can be backtracked.
     * </p>
     * 
     * @return The earliest time to which a DB cluster can be backtracked.
     */
    public final Instant earliestBacktrackTime() {
        return earliestBacktrackTime;
    }

    /**
     * <p>
     * The target backtrack window, in seconds. If this value is set to <code>0</code>, backtracking is disabled for the
     * DB cluster. Otherwise, backtracking is enabled.
     * </p>
     * 
     * @return The target backtrack window, in seconds. If this value is set to <code>0</code>, backtracking is disabled
     *         for the DB cluster. Otherwise, backtracking is enabled.
     */
    public final Long backtrackWindow() {
        return backtrackWindow;
    }

    /**
     * <p>
     * The number of change records stored for Backtrack.
     * </p>
     * 
     * @return The number of change records stored for Backtrack.
     */
    public final Long backtrackConsumedChangeRecords() {
        return backtrackConsumedChangeRecords;
    }

    /**
     * For responses, this returns true if the service returned a value for the EnabledCloudwatchLogsExports property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasEnabledCloudwatchLogsExports() {
        return enabledCloudwatchLogsExports != null && !(enabledCloudwatchLogsExports instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of log types that this DB cluster is configured to export to CloudWatch Logs.
     * </p>
     * <p>
     * Log types vary by DB engine. For information about the log types for each DB engine, see <a
     * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_LogAccess.html">Amazon RDS Database Log
     * Files</a> in the <i>Amazon Aurora User Guide.</i>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnabledCloudwatchLogsExports} method.
     * </p>
     * 
     * @return A list of log types that this DB cluster is configured to export to CloudWatch Logs.</p>
     *         <p>
     *         Log types vary by DB engine. For information about the log types for each DB engine, see <a
     *         href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_LogAccess.html">Amazon RDS
     *         Database Log Files</a> in the <i>Amazon Aurora User Guide.</i>
     */
    public final List<String> enabledCloudwatchLogsExports() {
        return enabledCloudwatchLogsExports;
    }

    /**
     * <p>
     * The current capacity of an Aurora Serverless v1 DB cluster. The capacity is <code>0</code> (zero) when the
     * cluster is paused.
     * </p>
     * <p>
     * For more information about Aurora Serverless v1, see <a
     * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html">Using Amazon Aurora
     * Serverless v1</a> in the <i>Amazon Aurora User Guide</i>.
     * </p>
     * 
     * @return The current capacity of an Aurora Serverless v1 DB cluster. The capacity is <code>0</code> (zero) when
     *         the cluster is paused.</p>
     *         <p>
     *         For more information about Aurora Serverless v1, see <a
     *         href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html">Using Amazon
     *         Aurora Serverless v1</a> in the <i>Amazon Aurora User Guide</i>.
     */
    public final Integer capacity() {
        return capacity;
    }

    /**
     * <p>
     * The DB engine mode of the DB cluster, either <code>provisioned</code> or <code>serverless</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBCluster.html"> CreateDBCluster</a>.
     * </p>
     * 
     * @return The DB engine mode of the DB cluster, either <code>provisioned</code> or <code>serverless</code>.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBCluster.html">
     *         CreateDBCluster</a>.
     */
    public final String engineMode() {
        return engineMode;
    }

    /**
     * Returns the value of the ScalingConfigurationInfo property for this object.
     * 
     * @return The value of the ScalingConfigurationInfo property for this object.
     */
    public final ScalingConfigurationInfo scalingConfigurationInfo() {
        return scalingConfigurationInfo;
    }

    /**
     * <p>
     * Reserved for future use.
     * </p>
     * 
     * @return Reserved for future use.
     */
    public final RdsCustomClusterConfiguration rdsCustomClusterConfiguration() {
        return rdsCustomClusterConfiguration;
    }

    /**
     * <p>
     * Indicates whether the DB cluster has deletion protection enabled. The database can't be deleted when deletion
     * protection is enabled.
     * </p>
     * 
     * @return Indicates whether the DB cluster has deletion protection enabled. The database can't be deleted when
     *         deletion protection is enabled.
     */
    public final Boolean deletionProtection() {
        return deletionProtection;
    }

    /**
     * <p>
     * Indicates whether the HTTP endpoint is enabled for an Aurora DB cluster.
     * </p>
     * <p>
     * When enabled, the HTTP endpoint provides a connectionless web service API (RDS Data API) for running SQL queries
     * on the DB cluster. You can also query your database from inside the RDS console with the RDS query editor.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html">Using RDS Data API</a> in the
     * <i>Amazon Aurora User Guide</i>.
     * </p>
     * 
     * @return Indicates whether the HTTP endpoint is enabled for an Aurora DB cluster.</p>
     *         <p>
     *         When enabled, the HTTP endpoint provides a connectionless web service API (RDS Data API) for running SQL
     *         queries on the DB cluster. You can also query your database from inside the RDS console with the RDS
     *         query editor.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html">Using RDS Data API</a>
     *         in the <i>Amazon Aurora User Guide</i>.
     */
    public final Boolean httpEndpointEnabled() {
        return httpEndpointEnabled;
    }

    /**
     * <p>
     * The mode of the database activity stream. Database events such as a change or access generate an activity stream
     * event. The database session can handle these events either synchronously or asynchronously.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #activityStreamMode} will return {@link ActivityStreamMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #activityStreamModeAsString}.
     * </p>
     * 
     * @return The mode of the database activity stream. Database events such as a change or access generate an activity
     *         stream event. The database session can handle these events either synchronously or asynchronously.
     * @see ActivityStreamMode
     */
    public final ActivityStreamMode activityStreamMode() {
        return ActivityStreamMode.fromValue(activityStreamMode);
    }

    /**
     * <p>
     * The mode of the database activity stream. Database events such as a change or access generate an activity stream
     * event. The database session can handle these events either synchronously or asynchronously.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #activityStreamMode} will return {@link ActivityStreamMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #activityStreamModeAsString}.
     * </p>
     * 
     * @return The mode of the database activity stream. Database events such as a change or access generate an activity
     *         stream event. The database session can handle these events either synchronously or asynchronously.
     * @see ActivityStreamMode
     */
    public final String activityStreamModeAsString() {
        return activityStreamMode;
    }

    /**
     * <p>
     * The status of the database activity stream.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #activityStreamStatus} will return {@link ActivityStreamStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #activityStreamStatusAsString}.
     * </p>
     * 
     * @return The status of the database activity stream.
     * @see ActivityStreamStatus
     */
    public final ActivityStreamStatus activityStreamStatus() {
        return ActivityStreamStatus.fromValue(activityStreamStatus);
    }

    /**
     * <p>
     * The status of the database activity stream.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #activityStreamStatus} will return {@link ActivityStreamStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #activityStreamStatusAsString}.
     * </p>
     * 
     * @return The status of the database activity stream.
     * @see ActivityStreamStatus
     */
    public final String activityStreamStatusAsString() {
        return activityStreamStatus;
    }

    /**
     * <p>
     * The Amazon Web Services KMS key identifier used for encrypting messages in the database activity stream.
     * </p>
     * <p>
     * The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.
     * </p>
     * 
     * @return The Amazon Web Services KMS key identifier used for encrypting messages in the database activity
     *         stream.</p>
     *         <p>
     *         The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS
     *         key.
     */
    public final String activityStreamKmsKeyId() {
        return activityStreamKmsKeyId;
    }

    /**
     * <p>
     * The name of the Amazon Kinesis data stream used for the database activity stream.
     * </p>
     * 
     * @return The name of the Amazon Kinesis data stream used for the database activity stream.
     */
    public final String activityStreamKinesisStreamName() {
        return activityStreamKinesisStreamName;
    }

    /**
     * <p>
     * Indicates whether tags are copied from the DB cluster to snapshots of the DB cluster.
     * </p>
     * 
     * @return Indicates whether tags are copied from the DB cluster to snapshots of the DB cluster.
     */
    public final Boolean copyTagsToSnapshot() {
        return copyTagsToSnapshot;
    }

    /**
     * <p>
     * Indicates whether the DB cluster is a clone of a DB cluster owned by a different Amazon Web Services account.
     * </p>
     * 
     * @return Indicates whether the DB cluster is a clone of a DB cluster owned by a different Amazon Web Services
     *         account.
     */
    public final Boolean crossAccountClone() {
        return crossAccountClone;
    }

    /**
     * For responses, this returns true if the service returned a value for the DomainMemberships property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDomainMemberships() {
        return domainMemberships != null && !(domainMemberships instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Active Directory Domain membership records associated with the DB cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDomainMemberships} method.
     * </p>
     * 
     * @return The Active Directory Domain membership records associated with the DB cluster.
     */
    public final List<DomainMembership> domainMemberships() {
        return domainMemberships;
    }

    /**
     * For responses, this returns true if the service returned a value for the TagList property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTagList() {
        return tagList != null && !(tagList instanceof SdkAutoConstructList);
    }

    /**
     * Returns the value of the TagList property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTagList} method.
     * </p>
     * 
     * @return The value of the TagList property for this object.
     */
    public final List<Tag> tagList() {
        return tagList;
    }

    /**
     * <p>
     * The status of write forwarding for a secondary cluster in an Aurora global database.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #globalWriteForwardingStatus} will return {@link WriteForwardingStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #globalWriteForwardingStatusAsString}.
     * </p>
     * 
     * @return The status of write forwarding for a secondary cluster in an Aurora global database.
     * @see WriteForwardingStatus
     */
    public final WriteForwardingStatus globalWriteForwardingStatus() {
        return WriteForwardingStatus.fromValue(globalWriteForwardingStatus);
    }

    /**
     * <p>
     * The status of write forwarding for a secondary cluster in an Aurora global database.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #globalWriteForwardingStatus} will return {@link WriteForwardingStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #globalWriteForwardingStatusAsString}.
     * </p>
     * 
     * @return The status of write forwarding for a secondary cluster in an Aurora global database.
     * @see WriteForwardingStatus
     */
    public final String globalWriteForwardingStatusAsString() {
        return globalWriteForwardingStatus;
    }

    /**
     * <p>
     * Indicates whether write forwarding is enabled for a secondary cluster in an Aurora global database. Because write
     * forwarding takes time to enable, check the value of <code>GlobalWriteForwardingStatus</code> to confirm that the
     * request has completed before using the write forwarding feature for this cluster.
     * </p>
     * 
     * @return Indicates whether write forwarding is enabled for a secondary cluster in an Aurora global database.
     *         Because write forwarding takes time to enable, check the value of
     *         <code>GlobalWriteForwardingStatus</code> to confirm that the request has completed before using the write
     *         forwarding feature for this cluster.
     */
    public final Boolean globalWriteForwardingRequested() {
        return globalWriteForwardingRequested;
    }

    /**
     * <p>
     * Information about pending changes to the DB cluster. This information is returned only when there are pending
     * changes. Specific changes are identified by subelements.
     * </p>
     * 
     * @return Information about pending changes to the DB cluster. This information is returned only when there are
     *         pending changes. Specific changes are identified by subelements.
     */
    public final ClusterPendingModifiedValues pendingModifiedValues() {
        return pendingModifiedValues;
    }

    /**
     * <p>
     * The name of the compute and memory capacity class of the DB instance.
     * </p>
     * <p>
     * This setting is only for non-Aurora Multi-AZ DB clusters.
     * </p>
     * 
     * @return The name of the compute and memory capacity class of the DB instance.</p>
     *         <p>
     *         This setting is only for non-Aurora Multi-AZ DB clusters.
     */
    public final String dbClusterInstanceClass() {
        return dbClusterInstanceClass;
    }

    /**
     * <p>
     * The storage type associated with the DB cluster.
     * </p>
     * 
     * @return The storage type associated with the DB cluster.
     */
    public final String storageType() {
        return storageType;
    }

    /**
     * <p>
     * The Provisioned IOPS (I/O operations per second) value.
     * </p>
     * <p>
     * This setting is only for non-Aurora Multi-AZ DB clusters.
     * </p>
     * 
     * @return The Provisioned IOPS (I/O operations per second) value.</p>
     *         <p>
     *         This setting is only for non-Aurora Multi-AZ DB clusters.
     */
    public final Integer iops() {
        return iops;
    }

    /**
     * <p>
     * Indicates whether the DB cluster is publicly accessible.
     * </p>
     * <p>
     * When the DB cluster is publicly accessible, its Domain Name System (DNS) endpoint resolves to the private IP
     * address from within the DB cluster's virtual private cloud (VPC). It resolves to the public IP address from
     * outside of the DB cluster's VPC. Access to the DB cluster is ultimately controlled by the security group it uses.
     * That public access isn't permitted if the security group assigned to the DB cluster doesn't permit it.
     * </p>
     * <p>
     * When the DB cluster isn't publicly accessible, it is an internal DB cluster with a DNS name that resolves to a
     * private IP address.
     * </p>
     * <p>
     * For more information, see <a>CreateDBCluster</a>.
     * </p>
     * <p>
     * This setting is only for non-Aurora Multi-AZ DB clusters.
     * </p>
     * 
     * @return Indicates whether the DB cluster is publicly accessible.</p>
     *         <p>
     *         When the DB cluster is publicly accessible, its Domain Name System (DNS) endpoint resolves to the private
     *         IP address from within the DB cluster's virtual private cloud (VPC). It resolves to the public IP address
     *         from outside of the DB cluster's VPC. Access to the DB cluster is ultimately controlled by the security
     *         group it uses. That public access isn't permitted if the security group assigned to the DB cluster
     *         doesn't permit it.
     *         </p>
     *         <p>
     *         When the DB cluster isn't publicly accessible, it is an internal DB cluster with a DNS name that resolves
     *         to a private IP address.
     *         </p>
     *         <p>
     *         For more information, see <a>CreateDBCluster</a>.
     *         </p>
     *         <p>
     *         This setting is only for non-Aurora Multi-AZ DB clusters.
     */
    public final Boolean publiclyAccessible() {
        return publiclyAccessible;
    }

    /**
     * <p>
     * Indicates whether minor version patches are applied automatically.
     * </p>
     * <p>
     * This setting is only for non-Aurora Multi-AZ DB clusters.
     * </p>
     * 
     * @return Indicates whether minor version patches are applied automatically.</p>
     *         <p>
     *         This setting is only for non-Aurora Multi-AZ DB clusters.
     */
    public final Boolean autoMinorVersionUpgrade() {
        return autoMinorVersionUpgrade;
    }

    /**
     * <p>
     * The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB cluster.
     * </p>
     * <p>
     * This setting is only for non-Aurora Multi-AZ DB clusters.
     * </p>
     * 
     * @return The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB
     *         cluster.</p>
     *         <p>
     *         This setting is only for non-Aurora Multi-AZ DB clusters.
     */
    public final Integer monitoringInterval() {
        return monitoringInterval;
    }

    /**
     * <p>
     * The ARN for the IAM role that permits RDS to send Enhanced Monitoring metrics to Amazon CloudWatch Logs.
     * </p>
     * <p>
     * This setting is only for non-Aurora Multi-AZ DB clusters.
     * </p>
     * 
     * @return The ARN for the IAM role that permits RDS to send Enhanced Monitoring metrics to Amazon CloudWatch
     *         Logs.</p>
     *         <p>
     *         This setting is only for non-Aurora Multi-AZ DB clusters.
     */
    public final String monitoringRoleArn() {
        return monitoringRoleArn;
    }

    /**
     * <p>
     * Indicates whether Performance Insights is enabled for the DB cluster.
     * </p>
     * <p>
     * This setting is only for non-Aurora Multi-AZ DB clusters.
     * </p>
     * 
     * @return Indicates whether Performance Insights is enabled for the DB cluster.</p>
     *         <p>
     *         This setting is only for non-Aurora Multi-AZ DB clusters.
     */
    public final Boolean performanceInsightsEnabled() {
        return performanceInsightsEnabled;
    }

    /**
     * <p>
     * The Amazon Web Services KMS key identifier for encryption of Performance Insights data.
     * </p>
     * <p>
     * The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.
     * </p>
     * <p>
     * This setting is only for non-Aurora Multi-AZ DB clusters.
     * </p>
     * 
     * @return The Amazon Web Services KMS key identifier for encryption of Performance Insights data.</p>
     *         <p>
     *         The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS
     *         key.
     *         </p>
     *         <p>
     *         This setting is only for non-Aurora Multi-AZ DB clusters.
     */
    public final String performanceInsightsKMSKeyId() {
        return performanceInsightsKMSKeyId;
    }

    /**
     * <p>
     * The number of days to retain Performance Insights data.
     * </p>
     * <p>
     * This setting is only for non-Aurora Multi-AZ DB clusters.
     * </p>
     * <p>
     * Valid Values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>7</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>month</i> * 31, where <i>month</i> is a number of months from 1-23. Examples: <code>93</code> (3 months * 31),
     * <code>341</code> (11 months * 31), <code>589</code> (19 months * 31)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>731</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>7</code> days
     * </p>
     * 
     * @return The number of days to retain Performance Insights data.</p>
     *         <p>
     *         This setting is only for non-Aurora Multi-AZ DB clusters.
     *         </p>
     *         <p>
     *         Valid Values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>7</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>month</i> * 31, where <i>month</i> is a number of months from 1-23. Examples: <code>93</code> (3
     *         months * 31), <code>341</code> (11 months * 31), <code>589</code> (19 months * 31)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>731</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>7</code> days
     */
    public final Integer performanceInsightsRetentionPeriod() {
        return performanceInsightsRetentionPeriod;
    }

    /**
     * Returns the value of the ServerlessV2ScalingConfiguration property for this object.
     * 
     * @return The value of the ServerlessV2ScalingConfiguration property for this object.
     */
    public final ServerlessV2ScalingConfigurationInfo serverlessV2ScalingConfiguration() {
        return serverlessV2ScalingConfiguration;
    }

    /**
     * <p>
     * The network type of the DB instance.
     * </p>
     * <p>
     * The network type is determined by the <code>DBSubnetGroup</code> specified for the DB cluster. A
     * <code>DBSubnetGroup</code> can support only the IPv4 protocol or the IPv4 and the IPv6 protocols (
     * <code>DUAL</code>).
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html">
     * Working with a DB instance in a VPC</a> in the <i>Amazon Aurora User Guide.</i>
     * </p>
     * <p>
     * This setting is only for Aurora DB clusters.
     * </p>
     * <p>
     * Valid Values: <code>IPV4 | DUAL</code>
     * </p>
     * 
     * @return The network type of the DB instance.</p>
     *         <p>
     *         The network type is determined by the <code>DBSubnetGroup</code> specified for the DB cluster. A
     *         <code>DBSubnetGroup</code> can support only the IPv4 protocol or the IPv4 and the IPv6 protocols (
     *         <code>DUAL</code>).
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html"
     *         > Working with a DB instance in a VPC</a> in the <i>Amazon Aurora User Guide.</i>
     *         </p>
     *         <p>
     *         This setting is only for Aurora DB clusters.
     *         </p>
     *         <p>
     *         Valid Values: <code>IPV4 | DUAL</code>
     */
    public final String networkType() {
        return networkType;
    }

    /**
     * <p>
     * Reserved for future use.
     * </p>
     * 
     * @return Reserved for future use.
     */
    public final String dbSystemId() {
        return dbSystemId;
    }

    /**
     * <p>
     * The secret managed by RDS in Amazon Web Services Secrets Manager for the master user password.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html">Password management with
     * Amazon Web Services Secrets Manager</a> in the <i>Amazon RDS User Guide</i> and <a
     * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html">Password management
     * with Amazon Web Services Secrets Manager</a> in the <i>Amazon Aurora User Guide.</i>
     * </p>
     * 
     * @return The secret managed by RDS in Amazon Web Services Secrets Manager for the master user password.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html">Password
     *         management with Amazon Web Services Secrets Manager</a> in the <i>Amazon RDS User Guide</i> and <a
     *         href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html">Password
     *         management with Amazon Web Services Secrets Manager</a> in the <i>Amazon Aurora User Guide.</i>
     */
    public final MasterUserSecret masterUserSecret() {
        return masterUserSecret;
    }

    /**
     * <p>
     * The next time you can modify the DB cluster to use the <code>aurora-iopt1</code> storage type.
     * </p>
     * <p>
     * This setting is only for Aurora DB clusters.
     * </p>
     * 
     * @return The next time you can modify the DB cluster to use the <code>aurora-iopt1</code> storage type.</p>
     *         <p>
     *         This setting is only for Aurora DB clusters.
     */
    public final Instant ioOptimizedNextAllowedModificationTime() {
        return ioOptimizedNextAllowedModificationTime;
    }

    /**
     * <p>
     * Indicates whether an Aurora DB cluster has in-cluster write forwarding enabled, not enabled, requested, or is in
     * the process of enabling it.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #localWriteForwardingStatus} will return {@link LocalWriteForwardingStatus#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #localWriteForwardingStatusAsString}.
     * </p>
     * 
     * @return Indicates whether an Aurora DB cluster has in-cluster write forwarding enabled, not enabled, requested,
     *         or is in the process of enabling it.
     * @see LocalWriteForwardingStatus
     */
    public final LocalWriteForwardingStatus localWriteForwardingStatus() {
        return LocalWriteForwardingStatus.fromValue(localWriteForwardingStatus);
    }

    /**
     * <p>
     * Indicates whether an Aurora DB cluster has in-cluster write forwarding enabled, not enabled, requested, or is in
     * the process of enabling it.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #localWriteForwardingStatus} will return {@link LocalWriteForwardingStatus#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #localWriteForwardingStatusAsString}.
     * </p>
     * 
     * @return Indicates whether an Aurora DB cluster has in-cluster write forwarding enabled, not enabled, requested,
     *         or is in the process of enabling it.
     * @see LocalWriteForwardingStatus
     */
    public final String localWriteForwardingStatusAsString() {
        return localWriteForwardingStatus;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the recovery point in Amazon Web Services Backup.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the recovery point in Amazon Web Services Backup.
     */
    public final String awsBackupRecoveryPointArn() {
        return awsBackupRecoveryPointArn;
    }

    /**
     * <p>
     * The details for Aurora Limitless Database.
     * </p>
     * 
     * @return The details for Aurora Limitless Database.
     */
    public final LimitlessDatabase limitlessDatabase() {
        return limitlessDatabase;
    }

    /**
     * <p>
     * The storage throughput for the DB cluster. The throughput is automatically set based on the IOPS that you
     * provision, and is not configurable.
     * </p>
     * <p>
     * This setting is only for non-Aurora Multi-AZ DB clusters.
     * </p>
     * 
     * @return The storage throughput for the DB cluster. The throughput is automatically set based on the IOPS that you
     *         provision, and is not configurable.</p>
     *         <p>
     *         This setting is only for non-Aurora Multi-AZ DB clusters.
     */
    public final Integer storageThroughput() {
        return storageThroughput;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(allocatedStorage());
        hashCode = 31 * hashCode + Objects.hashCode(hasAvailabilityZones() ? availabilityZones() : null);
        hashCode = 31 * hashCode + Objects.hashCode(backupRetentionPeriod());
        hashCode = 31 * hashCode + Objects.hashCode(characterSetName());
        hashCode = 31 * hashCode + Objects.hashCode(databaseName());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterParameterGroup());
        hashCode = 31 * hashCode + Objects.hashCode(dbSubnetGroup());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(automaticRestartTime());
        hashCode = 31 * hashCode + Objects.hashCode(percentProgress());
        hashCode = 31 * hashCode + Objects.hashCode(earliestRestorableTime());
        hashCode = 31 * hashCode + Objects.hashCode(endpoint());
        hashCode = 31 * hashCode + Objects.hashCode(readerEndpoint());
        hashCode = 31 * hashCode + Objects.hashCode(hasCustomEndpoints() ? customEndpoints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(multiAZ());
        hashCode = 31 * hashCode + Objects.hashCode(engine());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(latestRestorableTime());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(masterUsername());
        hashCode = 31 * hashCode
                + Objects.hashCode(hasDbClusterOptionGroupMemberships() ? dbClusterOptionGroupMemberships() : null);
        hashCode = 31 * hashCode + Objects.hashCode(preferredBackupWindow());
        hashCode = 31 * hashCode + Objects.hashCode(preferredMaintenanceWindow());
        hashCode = 31 * hashCode + Objects.hashCode(replicationSourceIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(hasReadReplicaIdentifiers() ? readReplicaIdentifiers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasStatusInfos() ? statusInfos() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDbClusterMembers() ? dbClusterMembers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasVpcSecurityGroups() ? vpcSecurityGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hostedZoneId());
        hashCode = 31 * hashCode + Objects.hashCode(storageEncrypted());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterResourceId());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasAssociatedRoles() ? associatedRoles() : null);
        hashCode = 31 * hashCode + Objects.hashCode(iamDatabaseAuthenticationEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(cloneGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(clusterCreateTime());
        hashCode = 31 * hashCode + Objects.hashCode(earliestBacktrackTime());
        hashCode = 31 * hashCode + Objects.hashCode(backtrackWindow());
        hashCode = 31 * hashCode + Objects.hashCode(backtrackConsumedChangeRecords());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnabledCloudwatchLogsExports() ? enabledCloudwatchLogsExports() : null);
        hashCode = 31 * hashCode + Objects.hashCode(capacity());
        hashCode = 31 * hashCode + Objects.hashCode(engineMode());
        hashCode = 31 * hashCode + Objects.hashCode(scalingConfigurationInfo());
        hashCode = 31 * hashCode + Objects.hashCode(rdsCustomClusterConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(deletionProtection());
        hashCode = 31 * hashCode + Objects.hashCode(httpEndpointEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(activityStreamModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(activityStreamStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(activityStreamKmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(activityStreamKinesisStreamName());
        hashCode = 31 * hashCode + Objects.hashCode(copyTagsToSnapshot());
        hashCode = 31 * hashCode + Objects.hashCode(crossAccountClone());
        hashCode = 31 * hashCode + Objects.hashCode(hasDomainMemberships() ? domainMemberships() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTagList() ? tagList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(globalWriteForwardingStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(globalWriteForwardingRequested());
        hashCode = 31 * hashCode + Objects.hashCode(pendingModifiedValues());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterInstanceClass());
        hashCode = 31 * hashCode + Objects.hashCode(storageType());
        hashCode = 31 * hashCode + Objects.hashCode(iops());
        hashCode = 31 * hashCode + Objects.hashCode(publiclyAccessible());
        hashCode = 31 * hashCode + Objects.hashCode(autoMinorVersionUpgrade());
        hashCode = 31 * hashCode + Objects.hashCode(monitoringInterval());
        hashCode = 31 * hashCode + Objects.hashCode(monitoringRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(performanceInsightsEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(performanceInsightsKMSKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(performanceInsightsRetentionPeriod());
        hashCode = 31 * hashCode + Objects.hashCode(serverlessV2ScalingConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(networkType());
        hashCode = 31 * hashCode + Objects.hashCode(dbSystemId());
        hashCode = 31 * hashCode + Objects.hashCode(masterUserSecret());
        hashCode = 31 * hashCode + Objects.hashCode(ioOptimizedNextAllowedModificationTime());
        hashCode = 31 * hashCode + Objects.hashCode(localWriteForwardingStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(awsBackupRecoveryPointArn());
        hashCode = 31 * hashCode + Objects.hashCode(limitlessDatabase());
        hashCode = 31 * hashCode + Objects.hashCode(storageThroughput());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DBCluster)) {
            return false;
        }
        DBCluster other = (DBCluster) obj;
        return Objects.equals(allocatedStorage(), other.allocatedStorage())
                && hasAvailabilityZones() == other.hasAvailabilityZones()
                && Objects.equals(availabilityZones(), other.availabilityZones())
                && Objects.equals(backupRetentionPeriod(), other.backupRetentionPeriod())
                && Objects.equals(characterSetName(), other.characterSetName())
                && Objects.equals(databaseName(), other.databaseName())
                && Objects.equals(dbClusterIdentifier(), other.dbClusterIdentifier())
                && Objects.equals(dbClusterParameterGroup(), other.dbClusterParameterGroup())
                && Objects.equals(dbSubnetGroup(), other.dbSubnetGroup()) && Objects.equals(status(), other.status())
                && Objects.equals(automaticRestartTime(), other.automaticRestartTime())
                && Objects.equals(percentProgress(), other.percentProgress())
                && Objects.equals(earliestRestorableTime(), other.earliestRestorableTime())
                && Objects.equals(endpoint(), other.endpoint()) && Objects.equals(readerEndpoint(), other.readerEndpoint())
                && hasCustomEndpoints() == other.hasCustomEndpoints()
                && Objects.equals(customEndpoints(), other.customEndpoints()) && Objects.equals(multiAZ(), other.multiAZ())
                && Objects.equals(engine(), other.engine()) && Objects.equals(engineVersion(), other.engineVersion())
                && Objects.equals(latestRestorableTime(), other.latestRestorableTime()) && Objects.equals(port(), other.port())
                && Objects.equals(masterUsername(), other.masterUsername())
                && hasDbClusterOptionGroupMemberships() == other.hasDbClusterOptionGroupMemberships()
                && Objects.equals(dbClusterOptionGroupMemberships(), other.dbClusterOptionGroupMemberships())
                && Objects.equals(preferredBackupWindow(), other.preferredBackupWindow())
                && Objects.equals(preferredMaintenanceWindow(), other.preferredMaintenanceWindow())
                && Objects.equals(replicationSourceIdentifier(), other.replicationSourceIdentifier())
                && hasReadReplicaIdentifiers() == other.hasReadReplicaIdentifiers()
                && Objects.equals(readReplicaIdentifiers(), other.readReplicaIdentifiers())
                && hasStatusInfos() == other.hasStatusInfos() && Objects.equals(statusInfos(), other.statusInfos())
                && hasDbClusterMembers() == other.hasDbClusterMembers()
                && Objects.equals(dbClusterMembers(), other.dbClusterMembers())
                && hasVpcSecurityGroups() == other.hasVpcSecurityGroups()
                && Objects.equals(vpcSecurityGroups(), other.vpcSecurityGroups())
                && Objects.equals(hostedZoneId(), other.hostedZoneId())
                && Objects.equals(storageEncrypted(), other.storageEncrypted()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(dbClusterResourceId(), other.dbClusterResourceId())
                && Objects.equals(dbClusterArn(), other.dbClusterArn()) && hasAssociatedRoles() == other.hasAssociatedRoles()
                && Objects.equals(associatedRoles(), other.associatedRoles())
                && Objects.equals(iamDatabaseAuthenticationEnabled(), other.iamDatabaseAuthenticationEnabled())
                && Objects.equals(cloneGroupId(), other.cloneGroupId())
                && Objects.equals(clusterCreateTime(), other.clusterCreateTime())
                && Objects.equals(earliestBacktrackTime(), other.earliestBacktrackTime())
                && Objects.equals(backtrackWindow(), other.backtrackWindow())
                && Objects.equals(backtrackConsumedChangeRecords(), other.backtrackConsumedChangeRecords())
                && hasEnabledCloudwatchLogsExports() == other.hasEnabledCloudwatchLogsExports()
                && Objects.equals(enabledCloudwatchLogsExports(), other.enabledCloudwatchLogsExports())
                && Objects.equals(capacity(), other.capacity()) && Objects.equals(engineMode(), other.engineMode())
                && Objects.equals(scalingConfigurationInfo(), other.scalingConfigurationInfo())
                && Objects.equals(rdsCustomClusterConfiguration(), other.rdsCustomClusterConfiguration())
                && Objects.equals(deletionProtection(), other.deletionProtection())
                && Objects.equals(httpEndpointEnabled(), other.httpEndpointEnabled())
                && Objects.equals(activityStreamModeAsString(), other.activityStreamModeAsString())
                && Objects.equals(activityStreamStatusAsString(), other.activityStreamStatusAsString())
                && Objects.equals(activityStreamKmsKeyId(), other.activityStreamKmsKeyId())
                && Objects.equals(activityStreamKinesisStreamName(), other.activityStreamKinesisStreamName())
                && Objects.equals(copyTagsToSnapshot(), other.copyTagsToSnapshot())
                && Objects.equals(crossAccountClone(), other.crossAccountClone())
                && hasDomainMemberships() == other.hasDomainMemberships()
                && Objects.equals(domainMemberships(), other.domainMemberships()) && hasTagList() == other.hasTagList()
                && Objects.equals(tagList(), other.tagList())
                && Objects.equals(globalWriteForwardingStatusAsString(), other.globalWriteForwardingStatusAsString())
                && Objects.equals(globalWriteForwardingRequested(), other.globalWriteForwardingRequested())
                && Objects.equals(pendingModifiedValues(), other.pendingModifiedValues())
                && Objects.equals(dbClusterInstanceClass(), other.dbClusterInstanceClass())
                && Objects.equals(storageType(), other.storageType()) && Objects.equals(iops(), other.iops())
                && Objects.equals(publiclyAccessible(), other.publiclyAccessible())
                && Objects.equals(autoMinorVersionUpgrade(), other.autoMinorVersionUpgrade())
                && Objects.equals(monitoringInterval(), other.monitoringInterval())
                && Objects.equals(monitoringRoleArn(), other.monitoringRoleArn())
                && Objects.equals(performanceInsightsEnabled(), other.performanceInsightsEnabled())
                && Objects.equals(performanceInsightsKMSKeyId(), other.performanceInsightsKMSKeyId())
                && Objects.equals(performanceInsightsRetentionPeriod(), other.performanceInsightsRetentionPeriod())
                && Objects.equals(serverlessV2ScalingConfiguration(), other.serverlessV2ScalingConfiguration())
                && Objects.equals(networkType(), other.networkType()) && Objects.equals(dbSystemId(), other.dbSystemId())
                && Objects.equals(masterUserSecret(), other.masterUserSecret())
                && Objects.equals(ioOptimizedNextAllowedModificationTime(), other.ioOptimizedNextAllowedModificationTime())
                && Objects.equals(localWriteForwardingStatusAsString(), other.localWriteForwardingStatusAsString())
                && Objects.equals(awsBackupRecoveryPointArn(), other.awsBackupRecoveryPointArn())
                && Objects.equals(limitlessDatabase(), other.limitlessDatabase())
                && Objects.equals(storageThroughput(), other.storageThroughput());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("DBCluster")
                .add("AllocatedStorage", allocatedStorage())
                .add("AvailabilityZones", hasAvailabilityZones() ? availabilityZones() : null)
                .add("BackupRetentionPeriod", backupRetentionPeriod())
                .add("CharacterSetName", characterSetName())
                .add("DatabaseName", databaseName())
                .add("DBClusterIdentifier", dbClusterIdentifier())
                .add("DBClusterParameterGroup", dbClusterParameterGroup())
                .add("DBSubnetGroup", dbSubnetGroup())
                .add("Status", status())
                .add("AutomaticRestartTime", automaticRestartTime())
                .add("PercentProgress", percentProgress())
                .add("EarliestRestorableTime", earliestRestorableTime())
                .add("Endpoint", endpoint())
                .add("ReaderEndpoint", readerEndpoint())
                .add("CustomEndpoints", hasCustomEndpoints() ? customEndpoints() : null)
                .add("MultiAZ", multiAZ())
                .add("Engine", engine())
                .add("EngineVersion", engineVersion())
                .add("LatestRestorableTime", latestRestorableTime())
                .add("Port", port())
                .add("MasterUsername", masterUsername())
                .add("DBClusterOptionGroupMemberships",
                        hasDbClusterOptionGroupMemberships() ? dbClusterOptionGroupMemberships() : null)
                .add("PreferredBackupWindow", preferredBackupWindow())
                .add("PreferredMaintenanceWindow", preferredMaintenanceWindow())
                .add("ReplicationSourceIdentifier", replicationSourceIdentifier())
                .add("ReadReplicaIdentifiers", hasReadReplicaIdentifiers() ? readReplicaIdentifiers() : null)
                .add("StatusInfos", hasStatusInfos() ? statusInfos() : null)
                .add("DBClusterMembers", hasDbClusterMembers() ? dbClusterMembers() : null)
                .add("VpcSecurityGroups", hasVpcSecurityGroups() ? vpcSecurityGroups() : null)
                .add("HostedZoneId", hostedZoneId()).add("StorageEncrypted", storageEncrypted()).add("KmsKeyId", kmsKeyId())
                .add("DbClusterResourceId", dbClusterResourceId()).add("DBClusterArn", dbClusterArn())
                .add("AssociatedRoles", hasAssociatedRoles() ? associatedRoles() : null)
                .add("IAMDatabaseAuthenticationEnabled", iamDatabaseAuthenticationEnabled()).add("CloneGroupId", cloneGroupId())
                .add("ClusterCreateTime", clusterCreateTime()).add("EarliestBacktrackTime", earliestBacktrackTime())
                .add("BacktrackWindow", backtrackWindow())
                .add("BacktrackConsumedChangeRecords", backtrackConsumedChangeRecords())
                .add("EnabledCloudwatchLogsExports", hasEnabledCloudwatchLogsExports() ? enabledCloudwatchLogsExports() : null)
                .add("Capacity", capacity()).add("EngineMode", engineMode())
                .add("ScalingConfigurationInfo", scalingConfigurationInfo())
                .add("RdsCustomClusterConfiguration", rdsCustomClusterConfiguration())
                .add("DeletionProtection", deletionProtection()).add("HttpEndpointEnabled", httpEndpointEnabled())
                .add("ActivityStreamMode", activityStreamModeAsString())
                .add("ActivityStreamStatus", activityStreamStatusAsString())
                .add("ActivityStreamKmsKeyId", activityStreamKmsKeyId())
                .add("ActivityStreamKinesisStreamName", activityStreamKinesisStreamName())
                .add("CopyTagsToSnapshot", copyTagsToSnapshot()).add("CrossAccountClone", crossAccountClone())
                .add("DomainMemberships", hasDomainMemberships() ? domainMemberships() : null)
                .add("TagList", hasTagList() ? tagList() : null)
                .add("GlobalWriteForwardingStatus", globalWriteForwardingStatusAsString())
                .add("GlobalWriteForwardingRequested", globalWriteForwardingRequested())
                .add("PendingModifiedValues", pendingModifiedValues()).add("DBClusterInstanceClass", dbClusterInstanceClass())
                .add("StorageType", storageType()).add("Iops", iops()).add("PubliclyAccessible", publiclyAccessible())
                .add("AutoMinorVersionUpgrade", autoMinorVersionUpgrade()).add("MonitoringInterval", monitoringInterval())
                .add("MonitoringRoleArn", monitoringRoleArn()).add("PerformanceInsightsEnabled", performanceInsightsEnabled())
                .add("PerformanceInsightsKMSKeyId", performanceInsightsKMSKeyId())
                .add("PerformanceInsightsRetentionPeriod", performanceInsightsRetentionPeriod())
                .add("ServerlessV2ScalingConfiguration", serverlessV2ScalingConfiguration()).add("NetworkType", networkType())
                .add("DBSystemId", dbSystemId()).add("MasterUserSecret", masterUserSecret())
                .add("IOOptimizedNextAllowedModificationTime", ioOptimizedNextAllowedModificationTime())
                .add("LocalWriteForwardingStatus", localWriteForwardingStatusAsString())
                .add("AwsBackupRecoveryPointArn", awsBackupRecoveryPointArn()).add("LimitlessDatabase", limitlessDatabase())
                .add("StorageThroughput", storageThroughput()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AllocatedStorage":
            return Optional.ofNullable(clazz.cast(allocatedStorage()));
        case "AvailabilityZones":
            return Optional.ofNullable(clazz.cast(availabilityZones()));
        case "BackupRetentionPeriod":
            return Optional.ofNullable(clazz.cast(backupRetentionPeriod()));
        case "CharacterSetName":
            return Optional.ofNullable(clazz.cast(characterSetName()));
        case "DatabaseName":
            return Optional.ofNullable(clazz.cast(databaseName()));
        case "DBClusterIdentifier":
            return Optional.ofNullable(clazz.cast(dbClusterIdentifier()));
        case "DBClusterParameterGroup":
            return Optional.ofNullable(clazz.cast(dbClusterParameterGroup()));
        case "DBSubnetGroup":
            return Optional.ofNullable(clazz.cast(dbSubnetGroup()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "AutomaticRestartTime":
            return Optional.ofNullable(clazz.cast(automaticRestartTime()));
        case "PercentProgress":
            return Optional.ofNullable(clazz.cast(percentProgress()));
        case "EarliestRestorableTime":
            return Optional.ofNullable(clazz.cast(earliestRestorableTime()));
        case "Endpoint":
            return Optional.ofNullable(clazz.cast(endpoint()));
        case "ReaderEndpoint":
            return Optional.ofNullable(clazz.cast(readerEndpoint()));
        case "CustomEndpoints":
            return Optional.ofNullable(clazz.cast(customEndpoints()));
        case "MultiAZ":
            return Optional.ofNullable(clazz.cast(multiAZ()));
        case "Engine":
            return Optional.ofNullable(clazz.cast(engine()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "LatestRestorableTime":
            return Optional.ofNullable(clazz.cast(latestRestorableTime()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "MasterUsername":
            return Optional.ofNullable(clazz.cast(masterUsername()));
        case "DBClusterOptionGroupMemberships":
            return Optional.ofNullable(clazz.cast(dbClusterOptionGroupMemberships()));
        case "PreferredBackupWindow":
            return Optional.ofNullable(clazz.cast(preferredBackupWindow()));
        case "PreferredMaintenanceWindow":
            return Optional.ofNullable(clazz.cast(preferredMaintenanceWindow()));
        case "ReplicationSourceIdentifier":
            return Optional.ofNullable(clazz.cast(replicationSourceIdentifier()));
        case "ReadReplicaIdentifiers":
            return Optional.ofNullable(clazz.cast(readReplicaIdentifiers()));
        case "StatusInfos":
            return Optional.ofNullable(clazz.cast(statusInfos()));
        case "DBClusterMembers":
            return Optional.ofNullable(clazz.cast(dbClusterMembers()));
        case "VpcSecurityGroups":
            return Optional.ofNullable(clazz.cast(vpcSecurityGroups()));
        case "HostedZoneId":
            return Optional.ofNullable(clazz.cast(hostedZoneId()));
        case "StorageEncrypted":
            return Optional.ofNullable(clazz.cast(storageEncrypted()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "DbClusterResourceId":
            return Optional.ofNullable(clazz.cast(dbClusterResourceId()));
        case "DBClusterArn":
            return Optional.ofNullable(clazz.cast(dbClusterArn()));
        case "AssociatedRoles":
            return Optional.ofNullable(clazz.cast(associatedRoles()));
        case "IAMDatabaseAuthenticationEnabled":
            return Optional.ofNullable(clazz.cast(iamDatabaseAuthenticationEnabled()));
        case "CloneGroupId":
            return Optional.ofNullable(clazz.cast(cloneGroupId()));
        case "ClusterCreateTime":
            return Optional.ofNullable(clazz.cast(clusterCreateTime()));
        case "EarliestBacktrackTime":
            return Optional.ofNullable(clazz.cast(earliestBacktrackTime()));
        case "BacktrackWindow":
            return Optional.ofNullable(clazz.cast(backtrackWindow()));
        case "BacktrackConsumedChangeRecords":
            return Optional.ofNullable(clazz.cast(backtrackConsumedChangeRecords()));
        case "EnabledCloudwatchLogsExports":
            return Optional.ofNullable(clazz.cast(enabledCloudwatchLogsExports()));
        case "Capacity":
            return Optional.ofNullable(clazz.cast(capacity()));
        case "EngineMode":
            return Optional.ofNullable(clazz.cast(engineMode()));
        case "ScalingConfigurationInfo":
            return Optional.ofNullable(clazz.cast(scalingConfigurationInfo()));
        case "RdsCustomClusterConfiguration":
            return Optional.ofNullable(clazz.cast(rdsCustomClusterConfiguration()));
        case "DeletionProtection":
            return Optional.ofNullable(clazz.cast(deletionProtection()));
        case "HttpEndpointEnabled":
            return Optional.ofNullable(clazz.cast(httpEndpointEnabled()));
        case "ActivityStreamMode":
            return Optional.ofNullable(clazz.cast(activityStreamModeAsString()));
        case "ActivityStreamStatus":
            return Optional.ofNullable(clazz.cast(activityStreamStatusAsString()));
        case "ActivityStreamKmsKeyId":
            return Optional.ofNullable(clazz.cast(activityStreamKmsKeyId()));
        case "ActivityStreamKinesisStreamName":
            return Optional.ofNullable(clazz.cast(activityStreamKinesisStreamName()));
        case "CopyTagsToSnapshot":
            return Optional.ofNullable(clazz.cast(copyTagsToSnapshot()));
        case "CrossAccountClone":
            return Optional.ofNullable(clazz.cast(crossAccountClone()));
        case "DomainMemberships":
            return Optional.ofNullable(clazz.cast(domainMemberships()));
        case "TagList":
            return Optional.ofNullable(clazz.cast(tagList()));
        case "GlobalWriteForwardingStatus":
            return Optional.ofNullable(clazz.cast(globalWriteForwardingStatusAsString()));
        case "GlobalWriteForwardingRequested":
            return Optional.ofNullable(clazz.cast(globalWriteForwardingRequested()));
        case "PendingModifiedValues":
            return Optional.ofNullable(clazz.cast(pendingModifiedValues()));
        case "DBClusterInstanceClass":
            return Optional.ofNullable(clazz.cast(dbClusterInstanceClass()));
        case "StorageType":
            return Optional.ofNullable(clazz.cast(storageType()));
        case "Iops":
            return Optional.ofNullable(clazz.cast(iops()));
        case "PubliclyAccessible":
            return Optional.ofNullable(clazz.cast(publiclyAccessible()));
        case "AutoMinorVersionUpgrade":
            return Optional.ofNullable(clazz.cast(autoMinorVersionUpgrade()));
        case "MonitoringInterval":
            return Optional.ofNullable(clazz.cast(monitoringInterval()));
        case "MonitoringRoleArn":
            return Optional.ofNullable(clazz.cast(monitoringRoleArn()));
        case "PerformanceInsightsEnabled":
            return Optional.ofNullable(clazz.cast(performanceInsightsEnabled()));
        case "PerformanceInsightsKMSKeyId":
            return Optional.ofNullable(clazz.cast(performanceInsightsKMSKeyId()));
        case "PerformanceInsightsRetentionPeriod":
            return Optional.ofNullable(clazz.cast(performanceInsightsRetentionPeriod()));
        case "ServerlessV2ScalingConfiguration":
            return Optional.ofNullable(clazz.cast(serverlessV2ScalingConfiguration()));
        case "NetworkType":
            return Optional.ofNullable(clazz.cast(networkType()));
        case "DBSystemId":
            return Optional.ofNullable(clazz.cast(dbSystemId()));
        case "MasterUserSecret":
            return Optional.ofNullable(clazz.cast(masterUserSecret()));
        case "IOOptimizedNextAllowedModificationTime":
            return Optional.ofNullable(clazz.cast(ioOptimizedNextAllowedModificationTime()));
        case "LocalWriteForwardingStatus":
            return Optional.ofNullable(clazz.cast(localWriteForwardingStatusAsString()));
        case "AwsBackupRecoveryPointArn":
            return Optional.ofNullable(clazz.cast(awsBackupRecoveryPointArn()));
        case "LimitlessDatabase":
            return Optional.ofNullable(clazz.cast(limitlessDatabase()));
        case "StorageThroughput":
            return Optional.ofNullable(clazz.cast(storageThroughput()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DBCluster, T> g) {
        return obj -> g.apply((DBCluster) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DBCluster> {
        /**
         * <p>
         * For all database engines except Amazon Aurora, <code>AllocatedStorage</code> specifies the allocated storage
         * size in gibibytes (GiB). For Aurora, <code>AllocatedStorage</code> always returns 1, because Aurora DB
         * cluster storage size isn't fixed, but instead automatically adjusts as needed.
         * </p>
         * 
         * @param allocatedStorage
         *        For all database engines except Amazon Aurora, <code>AllocatedStorage</code> specifies the allocated
         *        storage size in gibibytes (GiB). For Aurora, <code>AllocatedStorage</code> always returns 1, because
         *        Aurora DB cluster storage size isn't fixed, but instead automatically adjusts as needed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocatedStorage(Integer allocatedStorage);

        /**
         * <p>
         * The list of Availability Zones (AZs) where instances in the DB cluster can be created.
         * </p>
         * 
         * @param availabilityZones
         *        The list of Availability Zones (AZs) where instances in the DB cluster can be created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(Collection<String> availabilityZones);

        /**
         * <p>
         * The list of Availability Zones (AZs) where instances in the DB cluster can be created.
         * </p>
         * 
         * @param availabilityZones
         *        The list of Availability Zones (AZs) where instances in the DB cluster can be created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(String... availabilityZones);

        /**
         * <p>
         * The number of days for which automatic DB snapshots are retained.
         * </p>
         * 
         * @param backupRetentionPeriod
         *        The number of days for which automatic DB snapshots are retained.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupRetentionPeriod(Integer backupRetentionPeriod);

        /**
         * <p>
         * If present, specifies the name of the character set that this cluster is associated with.
         * </p>
         * 
         * @param characterSetName
         *        If present, specifies the name of the character set that this cluster is associated with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder characterSetName(String characterSetName);

        /**
         * <p>
         * The name of the initial database that was specified for the DB cluster when it was created, if one was
         * provided. This same name is returned for the life of the DB cluster.
         * </p>
         * 
         * @param databaseName
         *        The name of the initial database that was specified for the DB cluster when it was created, if one was
         *        provided. This same name is returned for the life of the DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseName(String databaseName);

        /**
         * <p>
         * The user-supplied identifier for the DB cluster. This identifier is the unique key that identifies a DB
         * cluster.
         * </p>
         * 
         * @param dbClusterIdentifier
         *        The user-supplied identifier for the DB cluster. This identifier is the unique key that identifies a
         *        DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterIdentifier(String dbClusterIdentifier);

        /**
         * <p>
         * The name of the DB cluster parameter group for the DB cluster.
         * </p>
         * 
         * @param dbClusterParameterGroup
         *        The name of the DB cluster parameter group for the DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterParameterGroup(String dbClusterParameterGroup);

        /**
         * <p>
         * Information about the subnet group associated with the DB cluster, including the name, description, and
         * subnets in the subnet group.
         * </p>
         * 
         * @param dbSubnetGroup
         *        Information about the subnet group associated with the DB cluster, including the name, description,
         *        and subnets in the subnet group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbSubnetGroup(String dbSubnetGroup);

        /**
         * <p>
         * The current state of this DB cluster.
         * </p>
         * 
         * @param status
         *        The current state of this DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * The time when a stopped DB cluster is restarted automatically.
         * </p>
         * 
         * @param automaticRestartTime
         *        The time when a stopped DB cluster is restarted automatically.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder automaticRestartTime(Instant automaticRestartTime);

        /**
         * <p>
         * The progress of the operation as a percentage.
         * </p>
         * 
         * @param percentProgress
         *        The progress of the operation as a percentage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentProgress(String percentProgress);

        /**
         * <p>
         * The earliest time to which a database can be restored with point-in-time restore.
         * </p>
         * 
         * @param earliestRestorableTime
         *        The earliest time to which a database can be restored with point-in-time restore.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder earliestRestorableTime(Instant earliestRestorableTime);

        /**
         * <p>
         * The connection endpoint for the primary instance of the DB cluster.
         * </p>
         * 
         * @param endpoint
         *        The connection endpoint for the primary instance of the DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoint(String endpoint);

        /**
         * <p>
         * The reader endpoint for the DB cluster. The reader endpoint for a DB cluster load-balances connections across
         * the Aurora Replicas that are available in a DB cluster. As clients request new connections to the reader
         * endpoint, Aurora distributes the connection requests among the Aurora Replicas in the DB cluster. This
         * functionality can help balance your read workload across multiple Aurora Replicas in your DB cluster.
         * </p>
         * <p>
         * If a failover occurs, and the Aurora Replica that you are connected to is promoted to be the primary
         * instance, your connection is dropped. To continue sending your read workload to other Aurora Replicas in the
         * cluster, you can then reconnect to the reader endpoint.
         * </p>
         * 
         * @param readerEndpoint
         *        The reader endpoint for the DB cluster. The reader endpoint for a DB cluster load-balances connections
         *        across the Aurora Replicas that are available in a DB cluster. As clients request new connections to
         *        the reader endpoint, Aurora distributes the connection requests among the Aurora Replicas in the DB
         *        cluster. This functionality can help balance your read workload across multiple Aurora Replicas in
         *        your DB cluster.</p>
         *        <p>
         *        If a failover occurs, and the Aurora Replica that you are connected to is promoted to be the primary
         *        instance, your connection is dropped. To continue sending your read workload to other Aurora Replicas
         *        in the cluster, you can then reconnect to the reader endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readerEndpoint(String readerEndpoint);

        /**
         * <p>
         * The custom endpoints associated with the DB cluster.
         * </p>
         * 
         * @param customEndpoints
         *        The custom endpoints associated with the DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customEndpoints(Collection<String> customEndpoints);

        /**
         * <p>
         * The custom endpoints associated with the DB cluster.
         * </p>
         * 
         * @param customEndpoints
         *        The custom endpoints associated with the DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customEndpoints(String... customEndpoints);

        /**
         * <p>
         * Indicates whether the DB cluster has instances in multiple Availability Zones.
         * </p>
         * 
         * @param multiAZ
         *        Indicates whether the DB cluster has instances in multiple Availability Zones.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder multiAZ(Boolean multiAZ);

        /**
         * <p>
         * The database engine used for this DB cluster.
         * </p>
         * 
         * @param engine
         *        The database engine used for this DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engine(String engine);

        /**
         * <p>
         * The version of the database engine.
         * </p>
         * 
         * @param engineVersion
         *        The version of the database engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * The latest time to which a database can be restored with point-in-time restore.
         * </p>
         * 
         * @param latestRestorableTime
         *        The latest time to which a database can be restored with point-in-time restore.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latestRestorableTime(Instant latestRestorableTime);

        /**
         * <p>
         * The port that the database engine is listening on.
         * </p>
         * 
         * @param port
         *        The port that the database engine is listening on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * The master username for the DB cluster.
         * </p>
         * 
         * @param masterUsername
         *        The master username for the DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterUsername(String masterUsername);

        /**
         * <p>
         * The list of option group memberships for this DB cluster.
         * </p>
         * 
         * @param dbClusterOptionGroupMemberships
         *        The list of option group memberships for this DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterOptionGroupMemberships(Collection<DBClusterOptionGroupStatus> dbClusterOptionGroupMemberships);

        /**
         * <p>
         * The list of option group memberships for this DB cluster.
         * </p>
         * 
         * @param dbClusterOptionGroupMemberships
         *        The list of option group memberships for this DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterOptionGroupMemberships(DBClusterOptionGroupStatus... dbClusterOptionGroupMemberships);

        /**
         * <p>
         * The list of option group memberships for this DB cluster.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.rds.model.DBClusterOptionGroupStatus.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.rds.model.DBClusterOptionGroupStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.rds.model.DBClusterOptionGroupStatus.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #dbClusterOptionGroupMemberships(List<DBClusterOptionGroupStatus>)}.
         * 
         * @param dbClusterOptionGroupMemberships
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.rds.model.DBClusterOptionGroupStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dbClusterOptionGroupMemberships(java.util.Collection<DBClusterOptionGroupStatus>)
         */
        Builder dbClusterOptionGroupMemberships(Consumer<DBClusterOptionGroupStatus.Builder>... dbClusterOptionGroupMemberships);

        /**
         * <p>
         * The daily time range during which automated backups are created if automated backups are enabled, as
         * determined by the <code>BackupRetentionPeriod</code>.
         * </p>
         * 
         * @param preferredBackupWindow
         *        The daily time range during which automated backups are created if automated backups are enabled, as
         *        determined by the <code>BackupRetentionPeriod</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredBackupWindow(String preferredBackupWindow);

        /**
         * <p>
         * The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).
         * </p>
         * 
         * @param preferredMaintenanceWindow
         *        The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredMaintenanceWindow(String preferredMaintenanceWindow);

        /**
         * <p>
         * The identifier of the source DB cluster if this DB cluster is a read replica.
         * </p>
         * 
         * @param replicationSourceIdentifier
         *        The identifier of the source DB cluster if this DB cluster is a read replica.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationSourceIdentifier(String replicationSourceIdentifier);

        /**
         * <p>
         * Contains one or more identifiers of the read replicas associated with this DB cluster.
         * </p>
         * 
         * @param readReplicaIdentifiers
         *        Contains one or more identifiers of the read replicas associated with this DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readReplicaIdentifiers(Collection<String> readReplicaIdentifiers);

        /**
         * <p>
         * Contains one or more identifiers of the read replicas associated with this DB cluster.
         * </p>
         * 
         * @param readReplicaIdentifiers
         *        Contains one or more identifiers of the read replicas associated with this DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readReplicaIdentifiers(String... readReplicaIdentifiers);

        /**
         * <p>
         * Reserved for future use.
         * </p>
         * 
         * @param statusInfos
         *        Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusInfos(Collection<DBClusterStatusInfo> statusInfos);

        /**
         * <p>
         * Reserved for future use.
         * </p>
         * 
         * @param statusInfos
         *        Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusInfos(DBClusterStatusInfo... statusInfos);

        /**
         * <p>
         * Reserved for future use.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.rds.model.DBClusterStatusInfo.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.rds.model.DBClusterStatusInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.rds.model.DBClusterStatusInfo.Builder#build()} is called immediately
         * and its result is passed to {@link #statusInfos(List<DBClusterStatusInfo>)}.
         * 
         * @param statusInfos
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.rds.model.DBClusterStatusInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statusInfos(java.util.Collection<DBClusterStatusInfo>)
         */
        Builder statusInfos(Consumer<DBClusterStatusInfo.Builder>... statusInfos);

        /**
         * <p>
         * The list of DB instances that make up the DB cluster.
         * </p>
         * 
         * @param dbClusterMembers
         *        The list of DB instances that make up the DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterMembers(Collection<DBClusterMember> dbClusterMembers);

        /**
         * <p>
         * The list of DB instances that make up the DB cluster.
         * </p>
         * 
         * @param dbClusterMembers
         *        The list of DB instances that make up the DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterMembers(DBClusterMember... dbClusterMembers);

        /**
         * <p>
         * The list of DB instances that make up the DB cluster.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.rds.model.DBClusterMember.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.rds.model.DBClusterMember#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.rds.model.DBClusterMember.Builder#build()} is called immediately and
         * its result is passed to {@link #dbClusterMembers(List<DBClusterMember>)}.
         * 
         * @param dbClusterMembers
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.rds.model.DBClusterMember.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dbClusterMembers(java.util.Collection<DBClusterMember>)
         */
        Builder dbClusterMembers(Consumer<DBClusterMember.Builder>... dbClusterMembers);

        /**
         * <p>
         * The list of VPC security groups that the DB cluster belongs to.
         * </p>
         * 
         * @param vpcSecurityGroups
         *        The list of VPC security groups that the DB cluster belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSecurityGroups(Collection<VpcSecurityGroupMembership> vpcSecurityGroups);

        /**
         * <p>
         * The list of VPC security groups that the DB cluster belongs to.
         * </p>
         * 
         * @param vpcSecurityGroups
         *        The list of VPC security groups that the DB cluster belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSecurityGroups(VpcSecurityGroupMembership... vpcSecurityGroups);

        /**
         * <p>
         * The list of VPC security groups that the DB cluster belongs to.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.rds.model.VpcSecurityGroupMembership.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.rds.model.VpcSecurityGroupMembership#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.rds.model.VpcSecurityGroupMembership.Builder#build()} is called
         * immediately and its result is passed to {@link #vpcSecurityGroups(List<VpcSecurityGroupMembership>)}.
         * 
         * @param vpcSecurityGroups
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.rds.model.VpcSecurityGroupMembership.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcSecurityGroups(java.util.Collection<VpcSecurityGroupMembership>)
         */
        Builder vpcSecurityGroups(Consumer<VpcSecurityGroupMembership.Builder>... vpcSecurityGroups);

        /**
         * <p>
         * The ID that Amazon Route 53 assigns when you create a hosted zone.
         * </p>
         * 
         * @param hostedZoneId
         *        The ID that Amazon Route 53 assigns when you create a hosted zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostedZoneId(String hostedZoneId);

        /**
         * <p>
         * Indicates whether the DB cluster is encrypted.
         * </p>
         * 
         * @param storageEncrypted
         *        Indicates whether the DB cluster is encrypted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageEncrypted(Boolean storageEncrypted);

        /**
         * <p>
         * If <code>StorageEncrypted</code> is enabled, the Amazon Web Services KMS key identifier for the encrypted DB
         * cluster.
         * </p>
         * <p>
         * The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.
         * </p>
         * 
         * @param kmsKeyId
         *        If <code>StorageEncrypted</code> is enabled, the Amazon Web Services KMS key identifier for the
         *        encrypted DB cluster.</p>
         *        <p>
         *        The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the
         *        KMS key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The Amazon Web Services Region-unique, immutable identifier for the DB cluster. This identifier is found in
         * Amazon Web Services CloudTrail log entries whenever the KMS key for the DB cluster is accessed.
         * </p>
         * 
         * @param dbClusterResourceId
         *        The Amazon Web Services Region-unique, immutable identifier for the DB cluster. This identifier is
         *        found in Amazon Web Services CloudTrail log entries whenever the KMS key for the DB cluster is
         *        accessed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterResourceId(String dbClusterResourceId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the DB cluster.
         * </p>
         * 
         * @param dbClusterArn
         *        The Amazon Resource Name (ARN) for the DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterArn(String dbClusterArn);

        /**
         * <p>
         * A list of the Amazon Web Services Identity and Access Management (IAM) roles that are associated with the DB
         * cluster. IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other
         * Amazon Web Services on your behalf.
         * </p>
         * 
         * @param associatedRoles
         *        A list of the Amazon Web Services Identity and Access Management (IAM) roles that are associated with
         *        the DB cluster. IAM roles that are associated with a DB cluster grant permission for the DB cluster to
         *        access other Amazon Web Services on your behalf.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatedRoles(Collection<DBClusterRole> associatedRoles);

        /**
         * <p>
         * A list of the Amazon Web Services Identity and Access Management (IAM) roles that are associated with the DB
         * cluster. IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other
         * Amazon Web Services on your behalf.
         * </p>
         * 
         * @param associatedRoles
         *        A list of the Amazon Web Services Identity and Access Management (IAM) roles that are associated with
         *        the DB cluster. IAM roles that are associated with a DB cluster grant permission for the DB cluster to
         *        access other Amazon Web Services on your behalf.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatedRoles(DBClusterRole... associatedRoles);

        /**
         * <p>
         * A list of the Amazon Web Services Identity and Access Management (IAM) roles that are associated with the DB
         * cluster. IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other
         * Amazon Web Services on your behalf.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.rds.model.DBClusterRole.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.rds.model.DBClusterRole#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.rds.model.DBClusterRole.Builder#build()} is called immediately and its
         * result is passed to {@link #associatedRoles(List<DBClusterRole>)}.
         * 
         * @param associatedRoles
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.rds.model.DBClusterRole.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #associatedRoles(java.util.Collection<DBClusterRole>)
         */
        Builder associatedRoles(Consumer<DBClusterRole.Builder>... associatedRoles);

        /**
         * <p>
         * Indicates whether the mapping of Amazon Web Services Identity and Access Management (IAM) accounts to
         * database accounts is enabled.
         * </p>
         * 
         * @param iamDatabaseAuthenticationEnabled
         *        Indicates whether the mapping of Amazon Web Services Identity and Access Management (IAM) accounts to
         *        database accounts is enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamDatabaseAuthenticationEnabled(Boolean iamDatabaseAuthenticationEnabled);

        /**
         * <p>
         * The ID of the clone group with which the DB cluster is associated.
         * </p>
         * 
         * @param cloneGroupId
         *        The ID of the clone group with which the DB cluster is associated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloneGroupId(String cloneGroupId);

        /**
         * <p>
         * The time when the DB cluster was created, in Universal Coordinated Time (UTC).
         * </p>
         * 
         * @param clusterCreateTime
         *        The time when the DB cluster was created, in Universal Coordinated Time (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterCreateTime(Instant clusterCreateTime);

        /**
         * <p>
         * The earliest time to which a DB cluster can be backtracked.
         * </p>
         * 
         * @param earliestBacktrackTime
         *        The earliest time to which a DB cluster can be backtracked.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder earliestBacktrackTime(Instant earliestBacktrackTime);

        /**
         * <p>
         * The target backtrack window, in seconds. If this value is set to <code>0</code>, backtracking is disabled for
         * the DB cluster. Otherwise, backtracking is enabled.
         * </p>
         * 
         * @param backtrackWindow
         *        The target backtrack window, in seconds. If this value is set to <code>0</code>, backtracking is
         *        disabled for the DB cluster. Otherwise, backtracking is enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backtrackWindow(Long backtrackWindow);

        /**
         * <p>
         * The number of change records stored for Backtrack.
         * </p>
         * 
         * @param backtrackConsumedChangeRecords
         *        The number of change records stored for Backtrack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backtrackConsumedChangeRecords(Long backtrackConsumedChangeRecords);

        /**
         * <p>
         * A list of log types that this DB cluster is configured to export to CloudWatch Logs.
         * </p>
         * <p>
         * Log types vary by DB engine. For information about the log types for each DB engine, see <a
         * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_LogAccess.html">Amazon RDS Database
         * Log Files</a> in the <i>Amazon Aurora User Guide.</i>
         * </p>
         * 
         * @param enabledCloudwatchLogsExports
         *        A list of log types that this DB cluster is configured to export to CloudWatch Logs.</p>
         *        <p>
         *        Log types vary by DB engine. For information about the log types for each DB engine, see <a
         *        href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_LogAccess.html">Amazon RDS
         *        Database Log Files</a> in the <i>Amazon Aurora User Guide.</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabledCloudwatchLogsExports(Collection<String> enabledCloudwatchLogsExports);

        /**
         * <p>
         * A list of log types that this DB cluster is configured to export to CloudWatch Logs.
         * </p>
         * <p>
         * Log types vary by DB engine. For information about the log types for each DB engine, see <a
         * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_LogAccess.html">Amazon RDS Database
         * Log Files</a> in the <i>Amazon Aurora User Guide.</i>
         * </p>
         * 
         * @param enabledCloudwatchLogsExports
         *        A list of log types that this DB cluster is configured to export to CloudWatch Logs.</p>
         *        <p>
         *        Log types vary by DB engine. For information about the log types for each DB engine, see <a
         *        href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_LogAccess.html">Amazon RDS
         *        Database Log Files</a> in the <i>Amazon Aurora User Guide.</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabledCloudwatchLogsExports(String... enabledCloudwatchLogsExports);

        /**
         * <p>
         * The current capacity of an Aurora Serverless v1 DB cluster. The capacity is <code>0</code> (zero) when the
         * cluster is paused.
         * </p>
         * <p>
         * For more information about Aurora Serverless v1, see <a
         * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html">Using Amazon
         * Aurora Serverless v1</a> in the <i>Amazon Aurora User Guide</i>.
         * </p>
         * 
         * @param capacity
         *        The current capacity of an Aurora Serverless v1 DB cluster. The capacity is <code>0</code> (zero) when
         *        the cluster is paused.</p>
         *        <p>
         *        For more information about Aurora Serverless v1, see <a
         *        href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html">Using
         *        Amazon Aurora Serverless v1</a> in the <i>Amazon Aurora User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacity(Integer capacity);

        /**
         * <p>
         * The DB engine mode of the DB cluster, either <code>provisioned</code> or <code>serverless</code>.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBCluster.html">
         * CreateDBCluster</a>.
         * </p>
         * 
         * @param engineMode
         *        The DB engine mode of the DB cluster, either <code>provisioned</code> or <code>serverless</code>.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBCluster.html">
         *        CreateDBCluster</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineMode(String engineMode);

        /**
         * Sets the value of the ScalingConfigurationInfo property for this object.
         *
         * @param scalingConfigurationInfo
         *        The new value for the ScalingConfigurationInfo property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingConfigurationInfo(ScalingConfigurationInfo scalingConfigurationInfo);

        /**
         * Sets the value of the ScalingConfigurationInfo property for this object.
         *
         * This is a convenience method that creates an instance of the {@link ScalingConfigurationInfo.Builder}
         * avoiding the need to create one manually via {@link ScalingConfigurationInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ScalingConfigurationInfo.Builder#build()} is called immediately
         * and its result is passed to {@link #scalingConfigurationInfo(ScalingConfigurationInfo)}.
         * 
         * @param scalingConfigurationInfo
         *        a consumer that will call methods on {@link ScalingConfigurationInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scalingConfigurationInfo(ScalingConfigurationInfo)
         */
        default Builder scalingConfigurationInfo(Consumer<ScalingConfigurationInfo.Builder> scalingConfigurationInfo) {
            return scalingConfigurationInfo(ScalingConfigurationInfo.builder().applyMutation(scalingConfigurationInfo).build());
        }

        /**
         * <p>
         * Reserved for future use.
         * </p>
         * 
         * @param rdsCustomClusterConfiguration
         *        Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rdsCustomClusterConfiguration(RdsCustomClusterConfiguration rdsCustomClusterConfiguration);

        /**
         * <p>
         * Reserved for future use.
         * </p>
         * This is a convenience method that creates an instance of the {@link RdsCustomClusterConfiguration.Builder}
         * avoiding the need to create one manually via {@link RdsCustomClusterConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RdsCustomClusterConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #rdsCustomClusterConfiguration(RdsCustomClusterConfiguration)}.
         * 
         * @param rdsCustomClusterConfiguration
         *        a consumer that will call methods on {@link RdsCustomClusterConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rdsCustomClusterConfiguration(RdsCustomClusterConfiguration)
         */
        default Builder rdsCustomClusterConfiguration(
                Consumer<RdsCustomClusterConfiguration.Builder> rdsCustomClusterConfiguration) {
            return rdsCustomClusterConfiguration(RdsCustomClusterConfiguration.builder()
                    .applyMutation(rdsCustomClusterConfiguration).build());
        }

        /**
         * <p>
         * Indicates whether the DB cluster has deletion protection enabled. The database can't be deleted when deletion
         * protection is enabled.
         * </p>
         * 
         * @param deletionProtection
         *        Indicates whether the DB cluster has deletion protection enabled. The database can't be deleted when
         *        deletion protection is enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletionProtection(Boolean deletionProtection);

        /**
         * <p>
         * Indicates whether the HTTP endpoint is enabled for an Aurora DB cluster.
         * </p>
         * <p>
         * When enabled, the HTTP endpoint provides a connectionless web service API (RDS Data API) for running SQL
         * queries on the DB cluster. You can also query your database from inside the RDS console with the RDS query
         * editor.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html">Using RDS Data API</a> in
         * the <i>Amazon Aurora User Guide</i>.
         * </p>
         * 
         * @param httpEndpointEnabled
         *        Indicates whether the HTTP endpoint is enabled for an Aurora DB cluster.</p>
         *        <p>
         *        When enabled, the HTTP endpoint provides a connectionless web service API (RDS Data API) for running
         *        SQL queries on the DB cluster. You can also query your database from inside the RDS console with the
         *        RDS query editor.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html">Using RDS Data
         *        API</a> in the <i>Amazon Aurora User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder httpEndpointEnabled(Boolean httpEndpointEnabled);

        /**
         * <p>
         * The mode of the database activity stream. Database events such as a change or access generate an activity
         * stream event. The database session can handle these events either synchronously or asynchronously.
         * </p>
         * 
         * @param activityStreamMode
         *        The mode of the database activity stream. Database events such as a change or access generate an
         *        activity stream event. The database session can handle these events either synchronously or
         *        asynchronously.
         * @see ActivityStreamMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActivityStreamMode
         */
        Builder activityStreamMode(String activityStreamMode);

        /**
         * <p>
         * The mode of the database activity stream. Database events such as a change or access generate an activity
         * stream event. The database session can handle these events either synchronously or asynchronously.
         * </p>
         * 
         * @param activityStreamMode
         *        The mode of the database activity stream. Database events such as a change or access generate an
         *        activity stream event. The database session can handle these events either synchronously or
         *        asynchronously.
         * @see ActivityStreamMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActivityStreamMode
         */
        Builder activityStreamMode(ActivityStreamMode activityStreamMode);

        /**
         * <p>
         * The status of the database activity stream.
         * </p>
         * 
         * @param activityStreamStatus
         *        The status of the database activity stream.
         * @see ActivityStreamStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActivityStreamStatus
         */
        Builder activityStreamStatus(String activityStreamStatus);

        /**
         * <p>
         * The status of the database activity stream.
         * </p>
         * 
         * @param activityStreamStatus
         *        The status of the database activity stream.
         * @see ActivityStreamStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActivityStreamStatus
         */
        Builder activityStreamStatus(ActivityStreamStatus activityStreamStatus);

        /**
         * <p>
         * The Amazon Web Services KMS key identifier used for encrypting messages in the database activity stream.
         * </p>
         * <p>
         * The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.
         * </p>
         * 
         * @param activityStreamKmsKeyId
         *        The Amazon Web Services KMS key identifier used for encrypting messages in the database activity
         *        stream.</p>
         *        <p>
         *        The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the
         *        KMS key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder activityStreamKmsKeyId(String activityStreamKmsKeyId);

        /**
         * <p>
         * The name of the Amazon Kinesis data stream used for the database activity stream.
         * </p>
         * 
         * @param activityStreamKinesisStreamName
         *        The name of the Amazon Kinesis data stream used for the database activity stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder activityStreamKinesisStreamName(String activityStreamKinesisStreamName);

        /**
         * <p>
         * Indicates whether tags are copied from the DB cluster to snapshots of the DB cluster.
         * </p>
         * 
         * @param copyTagsToSnapshot
         *        Indicates whether tags are copied from the DB cluster to snapshots of the DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyTagsToSnapshot(Boolean copyTagsToSnapshot);

        /**
         * <p>
         * Indicates whether the DB cluster is a clone of a DB cluster owned by a different Amazon Web Services account.
         * </p>
         * 
         * @param crossAccountClone
         *        Indicates whether the DB cluster is a clone of a DB cluster owned by a different Amazon Web Services
         *        account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crossAccountClone(Boolean crossAccountClone);

        /**
         * <p>
         * The Active Directory Domain membership records associated with the DB cluster.
         * </p>
         * 
         * @param domainMemberships
         *        The Active Directory Domain membership records associated with the DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainMemberships(Collection<DomainMembership> domainMemberships);

        /**
         * <p>
         * The Active Directory Domain membership records associated with the DB cluster.
         * </p>
         * 
         * @param domainMemberships
         *        The Active Directory Domain membership records associated with the DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainMemberships(DomainMembership... domainMemberships);

        /**
         * <p>
         * The Active Directory Domain membership records associated with the DB cluster.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.rds.model.DomainMembership.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.rds.model.DomainMembership#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.rds.model.DomainMembership.Builder#build()} is called immediately and
         * its result is passed to {@link #domainMemberships(List<DomainMembership>)}.
         * 
         * @param domainMemberships
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.rds.model.DomainMembership.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #domainMemberships(java.util.Collection<DomainMembership>)
         */
        Builder domainMemberships(Consumer<DomainMembership.Builder>... domainMemberships);

        /**
         * Sets the value of the TagList property for this object.
         *
         * @param tagList
         *        The new value for the TagList property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagList(Collection<Tag> tagList);

        /**
         * Sets the value of the TagList property for this object.
         *
         * @param tagList
         *        The new value for the TagList property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagList(Tag... tagList);

        /**
         * Sets the value of the TagList property for this object.
         *
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.rds.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.rds.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.rds.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tagList(List<Tag>)}.
         * 
         * @param tagList
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.rds.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagList(java.util.Collection<Tag>)
         */
        Builder tagList(Consumer<Tag.Builder>... tagList);

        /**
         * <p>
         * The status of write forwarding for a secondary cluster in an Aurora global database.
         * </p>
         * 
         * @param globalWriteForwardingStatus
         *        The status of write forwarding for a secondary cluster in an Aurora global database.
         * @see WriteForwardingStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WriteForwardingStatus
         */
        Builder globalWriteForwardingStatus(String globalWriteForwardingStatus);

        /**
         * <p>
         * The status of write forwarding for a secondary cluster in an Aurora global database.
         * </p>
         * 
         * @param globalWriteForwardingStatus
         *        The status of write forwarding for a secondary cluster in an Aurora global database.
         * @see WriteForwardingStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WriteForwardingStatus
         */
        Builder globalWriteForwardingStatus(WriteForwardingStatus globalWriteForwardingStatus);

        /**
         * <p>
         * Indicates whether write forwarding is enabled for a secondary cluster in an Aurora global database. Because
         * write forwarding takes time to enable, check the value of <code>GlobalWriteForwardingStatus</code> to confirm
         * that the request has completed before using the write forwarding feature for this cluster.
         * </p>
         * 
         * @param globalWriteForwardingRequested
         *        Indicates whether write forwarding is enabled for a secondary cluster in an Aurora global database.
         *        Because write forwarding takes time to enable, check the value of
         *        <code>GlobalWriteForwardingStatus</code> to confirm that the request has completed before using the
         *        write forwarding feature for this cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder globalWriteForwardingRequested(Boolean globalWriteForwardingRequested);

        /**
         * <p>
         * Information about pending changes to the DB cluster. This information is returned only when there are pending
         * changes. Specific changes are identified by subelements.
         * </p>
         * 
         * @param pendingModifiedValues
         *        Information about pending changes to the DB cluster. This information is returned only when there are
         *        pending changes. Specific changes are identified by subelements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pendingModifiedValues(ClusterPendingModifiedValues pendingModifiedValues);

        /**
         * <p>
         * Information about pending changes to the DB cluster. This information is returned only when there are pending
         * changes. Specific changes are identified by subelements.
         * </p>
         * This is a convenience method that creates an instance of the {@link ClusterPendingModifiedValues.Builder}
         * avoiding the need to create one manually via {@link ClusterPendingModifiedValues#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ClusterPendingModifiedValues.Builder#build()} is called
         * immediately and its result is passed to {@link #pendingModifiedValues(ClusterPendingModifiedValues)}.
         * 
         * @param pendingModifiedValues
         *        a consumer that will call methods on {@link ClusterPendingModifiedValues.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #pendingModifiedValues(ClusterPendingModifiedValues)
         */
        default Builder pendingModifiedValues(Consumer<ClusterPendingModifiedValues.Builder> pendingModifiedValues) {
            return pendingModifiedValues(ClusterPendingModifiedValues.builder().applyMutation(pendingModifiedValues).build());
        }

        /**
         * <p>
         * The name of the compute and memory capacity class of the DB instance.
         * </p>
         * <p>
         * This setting is only for non-Aurora Multi-AZ DB clusters.
         * </p>
         * 
         * @param dbClusterInstanceClass
         *        The name of the compute and memory capacity class of the DB instance.</p>
         *        <p>
         *        This setting is only for non-Aurora Multi-AZ DB clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterInstanceClass(String dbClusterInstanceClass);

        /**
         * <p>
         * The storage type associated with the DB cluster.
         * </p>
         * 
         * @param storageType
         *        The storage type associated with the DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageType(String storageType);

        /**
         * <p>
         * The Provisioned IOPS (I/O operations per second) value.
         * </p>
         * <p>
         * This setting is only for non-Aurora Multi-AZ DB clusters.
         * </p>
         * 
         * @param iops
         *        The Provisioned IOPS (I/O operations per second) value.</p>
         *        <p>
         *        This setting is only for non-Aurora Multi-AZ DB clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iops(Integer iops);

        /**
         * <p>
         * Indicates whether the DB cluster is publicly accessible.
         * </p>
         * <p>
         * When the DB cluster is publicly accessible, its Domain Name System (DNS) endpoint resolves to the private IP
         * address from within the DB cluster's virtual private cloud (VPC). It resolves to the public IP address from
         * outside of the DB cluster's VPC. Access to the DB cluster is ultimately controlled by the security group it
         * uses. That public access isn't permitted if the security group assigned to the DB cluster doesn't permit it.
         * </p>
         * <p>
         * When the DB cluster isn't publicly accessible, it is an internal DB cluster with a DNS name that resolves to
         * a private IP address.
         * </p>
         * <p>
         * For more information, see <a>CreateDBCluster</a>.
         * </p>
         * <p>
         * This setting is only for non-Aurora Multi-AZ DB clusters.
         * </p>
         * 
         * @param publiclyAccessible
         *        Indicates whether the DB cluster is publicly accessible.</p>
         *        <p>
         *        When the DB cluster is publicly accessible, its Domain Name System (DNS) endpoint resolves to the
         *        private IP address from within the DB cluster's virtual private cloud (VPC). It resolves to the public
         *        IP address from outside of the DB cluster's VPC. Access to the DB cluster is ultimately controlled by
         *        the security group it uses. That public access isn't permitted if the security group assigned to the
         *        DB cluster doesn't permit it.
         *        </p>
         *        <p>
         *        When the DB cluster isn't publicly accessible, it is an internal DB cluster with a DNS name that
         *        resolves to a private IP address.
         *        </p>
         *        <p>
         *        For more information, see <a>CreateDBCluster</a>.
         *        </p>
         *        <p>
         *        This setting is only for non-Aurora Multi-AZ DB clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publiclyAccessible(Boolean publiclyAccessible);

        /**
         * <p>
         * Indicates whether minor version patches are applied automatically.
         * </p>
         * <p>
         * This setting is only for non-Aurora Multi-AZ DB clusters.
         * </p>
         * 
         * @param autoMinorVersionUpgrade
         *        Indicates whether minor version patches are applied automatically.</p>
         *        <p>
         *        This setting is only for non-Aurora Multi-AZ DB clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoMinorVersionUpgrade(Boolean autoMinorVersionUpgrade);

        /**
         * <p>
         * The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB cluster.
         * </p>
         * <p>
         * This setting is only for non-Aurora Multi-AZ DB clusters.
         * </p>
         * 
         * @param monitoringInterval
         *        The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB
         *        cluster.</p>
         *        <p>
         *        This setting is only for non-Aurora Multi-AZ DB clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder monitoringInterval(Integer monitoringInterval);

        /**
         * <p>
         * The ARN for the IAM role that permits RDS to send Enhanced Monitoring metrics to Amazon CloudWatch Logs.
         * </p>
         * <p>
         * This setting is only for non-Aurora Multi-AZ DB clusters.
         * </p>
         * 
         * @param monitoringRoleArn
         *        The ARN for the IAM role that permits RDS to send Enhanced Monitoring metrics to Amazon CloudWatch
         *        Logs.</p>
         *        <p>
         *        This setting is only for non-Aurora Multi-AZ DB clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder monitoringRoleArn(String monitoringRoleArn);

        /**
         * <p>
         * Indicates whether Performance Insights is enabled for the DB cluster.
         * </p>
         * <p>
         * This setting is only for non-Aurora Multi-AZ DB clusters.
         * </p>
         * 
         * @param performanceInsightsEnabled
         *        Indicates whether Performance Insights is enabled for the DB cluster.</p>
         *        <p>
         *        This setting is only for non-Aurora Multi-AZ DB clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performanceInsightsEnabled(Boolean performanceInsightsEnabled);

        /**
         * <p>
         * The Amazon Web Services KMS key identifier for encryption of Performance Insights data.
         * </p>
         * <p>
         * The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.
         * </p>
         * <p>
         * This setting is only for non-Aurora Multi-AZ DB clusters.
         * </p>
         * 
         * @param performanceInsightsKMSKeyId
         *        The Amazon Web Services KMS key identifier for encryption of Performance Insights data.</p>
         *        <p>
         *        The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the
         *        KMS key.
         *        </p>
         *        <p>
         *        This setting is only for non-Aurora Multi-AZ DB clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performanceInsightsKMSKeyId(String performanceInsightsKMSKeyId);

        /**
         * <p>
         * The number of days to retain Performance Insights data.
         * </p>
         * <p>
         * This setting is only for non-Aurora Multi-AZ DB clusters.
         * </p>
         * <p>
         * Valid Values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>7</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>month</i> * 31, where <i>month</i> is a number of months from 1-23. Examples: <code>93</code> (3 months *
         * 31), <code>341</code> (11 months * 31), <code>589</code> (19 months * 31)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>731</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>7</code> days
         * </p>
         * 
         * @param performanceInsightsRetentionPeriod
         *        The number of days to retain Performance Insights data.</p>
         *        <p>
         *        This setting is only for non-Aurora Multi-AZ DB clusters.
         *        </p>
         *        <p>
         *        Valid Values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>7</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>month</i> * 31, where <i>month</i> is a number of months from 1-23. Examples: <code>93</code> (3
         *        months * 31), <code>341</code> (11 months * 31), <code>589</code> (19 months * 31)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>731</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>7</code> days
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performanceInsightsRetentionPeriod(Integer performanceInsightsRetentionPeriod);

        /**
         * Sets the value of the ServerlessV2ScalingConfiguration property for this object.
         *
         * @param serverlessV2ScalingConfiguration
         *        The new value for the ServerlessV2ScalingConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverlessV2ScalingConfiguration(ServerlessV2ScalingConfigurationInfo serverlessV2ScalingConfiguration);

        /**
         * Sets the value of the ServerlessV2ScalingConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the
         * {@link ServerlessV2ScalingConfigurationInfo.Builder} avoiding the need to create one manually via
         * {@link ServerlessV2ScalingConfigurationInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ServerlessV2ScalingConfigurationInfo.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #serverlessV2ScalingConfiguration(ServerlessV2ScalingConfigurationInfo)}.
         * 
         * @param serverlessV2ScalingConfiguration
         *        a consumer that will call methods on {@link ServerlessV2ScalingConfigurationInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serverlessV2ScalingConfiguration(ServerlessV2ScalingConfigurationInfo)
         */
        default Builder serverlessV2ScalingConfiguration(
                Consumer<ServerlessV2ScalingConfigurationInfo.Builder> serverlessV2ScalingConfiguration) {
            return serverlessV2ScalingConfiguration(ServerlessV2ScalingConfigurationInfo.builder()
                    .applyMutation(serverlessV2ScalingConfiguration).build());
        }

        /**
         * <p>
         * The network type of the DB instance.
         * </p>
         * <p>
         * The network type is determined by the <code>DBSubnetGroup</code> specified for the DB cluster. A
         * <code>DBSubnetGroup</code> can support only the IPv4 protocol or the IPv4 and the IPv6 protocols (
         * <code>DUAL</code>).
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html">
         * Working with a DB instance in a VPC</a> in the <i>Amazon Aurora User Guide.</i>
         * </p>
         * <p>
         * This setting is only for Aurora DB clusters.
         * </p>
         * <p>
         * Valid Values: <code>IPV4 | DUAL</code>
         * </p>
         * 
         * @param networkType
         *        The network type of the DB instance.</p>
         *        <p>
         *        The network type is determined by the <code>DBSubnetGroup</code> specified for the DB cluster. A
         *        <code>DBSubnetGroup</code> can support only the IPv4 protocol or the IPv4 and the IPv6 protocols (
         *        <code>DUAL</code>).
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html"
         *        > Working with a DB instance in a VPC</a> in the <i>Amazon Aurora User Guide.</i>
         *        </p>
         *        <p>
         *        This setting is only for Aurora DB clusters.
         *        </p>
         *        <p>
         *        Valid Values: <code>IPV4 | DUAL</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkType(String networkType);

        /**
         * <p>
         * Reserved for future use.
         * </p>
         * 
         * @param dbSystemId
         *        Reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbSystemId(String dbSystemId);

        /**
         * <p>
         * The secret managed by RDS in Amazon Web Services Secrets Manager for the master user password.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html">Password management
         * with Amazon Web Services Secrets Manager</a> in the <i>Amazon RDS User Guide</i> and <a
         * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html">Password
         * management with Amazon Web Services Secrets Manager</a> in the <i>Amazon Aurora User Guide.</i>
         * </p>
         * 
         * @param masterUserSecret
         *        The secret managed by RDS in Amazon Web Services Secrets Manager for the master user password.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html">Password
         *        management with Amazon Web Services Secrets Manager</a> in the <i>Amazon RDS User Guide</i> and <a
         *        href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html">Password
         *        management with Amazon Web Services Secrets Manager</a> in the <i>Amazon Aurora User Guide.</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterUserSecret(MasterUserSecret masterUserSecret);

        /**
         * <p>
         * The secret managed by RDS in Amazon Web Services Secrets Manager for the master user password.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html">Password management
         * with Amazon Web Services Secrets Manager</a> in the <i>Amazon RDS User Guide</i> and <a
         * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html">Password
         * management with Amazon Web Services Secrets Manager</a> in the <i>Amazon Aurora User Guide.</i>
         * </p>
         * This is a convenience method that creates an instance of the {@link MasterUserSecret.Builder} avoiding the
         * need to create one manually via {@link MasterUserSecret#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MasterUserSecret.Builder#build()} is called immediately and its
         * result is passed to {@link #masterUserSecret(MasterUserSecret)}.
         * 
         * @param masterUserSecret
         *        a consumer that will call methods on {@link MasterUserSecret.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #masterUserSecret(MasterUserSecret)
         */
        default Builder masterUserSecret(Consumer<MasterUserSecret.Builder> masterUserSecret) {
            return masterUserSecret(MasterUserSecret.builder().applyMutation(masterUserSecret).build());
        }

        /**
         * <p>
         * The next time you can modify the DB cluster to use the <code>aurora-iopt1</code> storage type.
         * </p>
         * <p>
         * This setting is only for Aurora DB clusters.
         * </p>
         * 
         * @param ioOptimizedNextAllowedModificationTime
         *        The next time you can modify the DB cluster to use the <code>aurora-iopt1</code> storage type.</p>
         *        <p>
         *        This setting is only for Aurora DB clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ioOptimizedNextAllowedModificationTime(Instant ioOptimizedNextAllowedModificationTime);

        /**
         * <p>
         * Indicates whether an Aurora DB cluster has in-cluster write forwarding enabled, not enabled, requested, or is
         * in the process of enabling it.
         * </p>
         * 
         * @param localWriteForwardingStatus
         *        Indicates whether an Aurora DB cluster has in-cluster write forwarding enabled, not enabled,
         *        requested, or is in the process of enabling it.
         * @see LocalWriteForwardingStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LocalWriteForwardingStatus
         */
        Builder localWriteForwardingStatus(String localWriteForwardingStatus);

        /**
         * <p>
         * Indicates whether an Aurora DB cluster has in-cluster write forwarding enabled, not enabled, requested, or is
         * in the process of enabling it.
         * </p>
         * 
         * @param localWriteForwardingStatus
         *        Indicates whether an Aurora DB cluster has in-cluster write forwarding enabled, not enabled,
         *        requested, or is in the process of enabling it.
         * @see LocalWriteForwardingStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LocalWriteForwardingStatus
         */
        Builder localWriteForwardingStatus(LocalWriteForwardingStatus localWriteForwardingStatus);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the recovery point in Amazon Web Services Backup.
         * </p>
         * 
         * @param awsBackupRecoveryPointArn
         *        The Amazon Resource Name (ARN) of the recovery point in Amazon Web Services Backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsBackupRecoveryPointArn(String awsBackupRecoveryPointArn);

        /**
         * <p>
         * The details for Aurora Limitless Database.
         * </p>
         * 
         * @param limitlessDatabase
         *        The details for Aurora Limitless Database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder limitlessDatabase(LimitlessDatabase limitlessDatabase);

        /**
         * <p>
         * The details for Aurora Limitless Database.
         * </p>
         * This is a convenience method that creates an instance of the {@link LimitlessDatabase.Builder} avoiding the
         * need to create one manually via {@link LimitlessDatabase#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LimitlessDatabase.Builder#build()} is called immediately and its
         * result is passed to {@link #limitlessDatabase(LimitlessDatabase)}.
         * 
         * @param limitlessDatabase
         *        a consumer that will call methods on {@link LimitlessDatabase.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #limitlessDatabase(LimitlessDatabase)
         */
        default Builder limitlessDatabase(Consumer<LimitlessDatabase.Builder> limitlessDatabase) {
            return limitlessDatabase(LimitlessDatabase.builder().applyMutation(limitlessDatabase).build());
        }

        /**
         * <p>
         * The storage throughput for the DB cluster. The throughput is automatically set based on the IOPS that you
         * provision, and is not configurable.
         * </p>
         * <p>
         * This setting is only for non-Aurora Multi-AZ DB clusters.
         * </p>
         * 
         * @param storageThroughput
         *        The storage throughput for the DB cluster. The throughput is automatically set based on the IOPS that
         *        you provision, and is not configurable.</p>
         *        <p>
         *        This setting is only for non-Aurora Multi-AZ DB clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageThroughput(Integer storageThroughput);
    }

    static final class BuilderImpl implements Builder {
        private Integer allocatedStorage;

        private List<String> availabilityZones = DefaultSdkAutoConstructList.getInstance();

        private Integer backupRetentionPeriod;

        private String characterSetName;

        private String databaseName;

        private String dbClusterIdentifier;

        private String dbClusterParameterGroup;

        private String dbSubnetGroup;

        private String status;

        private Instant automaticRestartTime;

        private String percentProgress;

        private Instant earliestRestorableTime;

        private String endpoint;

        private String readerEndpoint;

        private List<String> customEndpoints = DefaultSdkAutoConstructList.getInstance();

        private Boolean multiAZ;

        private String engine;

        private String engineVersion;

        private Instant latestRestorableTime;

        private Integer port;

        private String masterUsername;

        private List<DBClusterOptionGroupStatus> dbClusterOptionGroupMemberships = DefaultSdkAutoConstructList.getInstance();

        private String preferredBackupWindow;

        private String preferredMaintenanceWindow;

        private String replicationSourceIdentifier;

        private List<String> readReplicaIdentifiers = DefaultSdkAutoConstructList.getInstance();

        private List<DBClusterStatusInfo> statusInfos = DefaultSdkAutoConstructList.getInstance();

        private List<DBClusterMember> dbClusterMembers = DefaultSdkAutoConstructList.getInstance();

        private List<VpcSecurityGroupMembership> vpcSecurityGroups = DefaultSdkAutoConstructList.getInstance();

        private String hostedZoneId;

        private Boolean storageEncrypted;

        private String kmsKeyId;

        private String dbClusterResourceId;

        private String dbClusterArn;

        private List<DBClusterRole> associatedRoles = DefaultSdkAutoConstructList.getInstance();

        private Boolean iamDatabaseAuthenticationEnabled;

        private String cloneGroupId;

        private Instant clusterCreateTime;

        private Instant earliestBacktrackTime;

        private Long backtrackWindow;

        private Long backtrackConsumedChangeRecords;

        private List<String> enabledCloudwatchLogsExports = DefaultSdkAutoConstructList.getInstance();

        private Integer capacity;

        private String engineMode;

        private ScalingConfigurationInfo scalingConfigurationInfo;

        private RdsCustomClusterConfiguration rdsCustomClusterConfiguration;

        private Boolean deletionProtection;

        private Boolean httpEndpointEnabled;

        private String activityStreamMode;

        private String activityStreamStatus;

        private String activityStreamKmsKeyId;

        private String activityStreamKinesisStreamName;

        private Boolean copyTagsToSnapshot;

        private Boolean crossAccountClone;

        private List<DomainMembership> domainMemberships = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tagList = DefaultSdkAutoConstructList.getInstance();

        private String globalWriteForwardingStatus;

        private Boolean globalWriteForwardingRequested;

        private ClusterPendingModifiedValues pendingModifiedValues;

        private String dbClusterInstanceClass;

        private String storageType;

        private Integer iops;

        private Boolean publiclyAccessible;

        private Boolean autoMinorVersionUpgrade;

        private Integer monitoringInterval;

        private String monitoringRoleArn;

        private Boolean performanceInsightsEnabled;

        private String performanceInsightsKMSKeyId;

        private Integer performanceInsightsRetentionPeriod;

        private ServerlessV2ScalingConfigurationInfo serverlessV2ScalingConfiguration;

        private String networkType;

        private String dbSystemId;

        private MasterUserSecret masterUserSecret;

        private Instant ioOptimizedNextAllowedModificationTime;

        private String localWriteForwardingStatus;

        private String awsBackupRecoveryPointArn;

        private LimitlessDatabase limitlessDatabase;

        private Integer storageThroughput;

        private BuilderImpl() {
        }

        private BuilderImpl(DBCluster model) {
            allocatedStorage(model.allocatedStorage);
            availabilityZones(model.availabilityZones);
            backupRetentionPeriod(model.backupRetentionPeriod);
            characterSetName(model.characterSetName);
            databaseName(model.databaseName);
            dbClusterIdentifier(model.dbClusterIdentifier);
            dbClusterParameterGroup(model.dbClusterParameterGroup);
            dbSubnetGroup(model.dbSubnetGroup);
            status(model.status);
            automaticRestartTime(model.automaticRestartTime);
            percentProgress(model.percentProgress);
            earliestRestorableTime(model.earliestRestorableTime);
            endpoint(model.endpoint);
            readerEndpoint(model.readerEndpoint);
            customEndpoints(model.customEndpoints);
            multiAZ(model.multiAZ);
            engine(model.engine);
            engineVersion(model.engineVersion);
            latestRestorableTime(model.latestRestorableTime);
            port(model.port);
            masterUsername(model.masterUsername);
            dbClusterOptionGroupMemberships(model.dbClusterOptionGroupMemberships);
            preferredBackupWindow(model.preferredBackupWindow);
            preferredMaintenanceWindow(model.preferredMaintenanceWindow);
            replicationSourceIdentifier(model.replicationSourceIdentifier);
            readReplicaIdentifiers(model.readReplicaIdentifiers);
            statusInfos(model.statusInfos);
            dbClusterMembers(model.dbClusterMembers);
            vpcSecurityGroups(model.vpcSecurityGroups);
            hostedZoneId(model.hostedZoneId);
            storageEncrypted(model.storageEncrypted);
            kmsKeyId(model.kmsKeyId);
            dbClusterResourceId(model.dbClusterResourceId);
            dbClusterArn(model.dbClusterArn);
            associatedRoles(model.associatedRoles);
            iamDatabaseAuthenticationEnabled(model.iamDatabaseAuthenticationEnabled);
            cloneGroupId(model.cloneGroupId);
            clusterCreateTime(model.clusterCreateTime);
            earliestBacktrackTime(model.earliestBacktrackTime);
            backtrackWindow(model.backtrackWindow);
            backtrackConsumedChangeRecords(model.backtrackConsumedChangeRecords);
            enabledCloudwatchLogsExports(model.enabledCloudwatchLogsExports);
            capacity(model.capacity);
            engineMode(model.engineMode);
            scalingConfigurationInfo(model.scalingConfigurationInfo);
            rdsCustomClusterConfiguration(model.rdsCustomClusterConfiguration);
            deletionProtection(model.deletionProtection);
            httpEndpointEnabled(model.httpEndpointEnabled);
            activityStreamMode(model.activityStreamMode);
            activityStreamStatus(model.activityStreamStatus);
            activityStreamKmsKeyId(model.activityStreamKmsKeyId);
            activityStreamKinesisStreamName(model.activityStreamKinesisStreamName);
            copyTagsToSnapshot(model.copyTagsToSnapshot);
            crossAccountClone(model.crossAccountClone);
            domainMemberships(model.domainMemberships);
            tagList(model.tagList);
            globalWriteForwardingStatus(model.globalWriteForwardingStatus);
            globalWriteForwardingRequested(model.globalWriteForwardingRequested);
            pendingModifiedValues(model.pendingModifiedValues);
            dbClusterInstanceClass(model.dbClusterInstanceClass);
            storageType(model.storageType);
            iops(model.iops);
            publiclyAccessible(model.publiclyAccessible);
            autoMinorVersionUpgrade(model.autoMinorVersionUpgrade);
            monitoringInterval(model.monitoringInterval);
            monitoringRoleArn(model.monitoringRoleArn);
            performanceInsightsEnabled(model.performanceInsightsEnabled);
            performanceInsightsKMSKeyId(model.performanceInsightsKMSKeyId);
            performanceInsightsRetentionPeriod(model.performanceInsightsRetentionPeriod);
            serverlessV2ScalingConfiguration(model.serverlessV2ScalingConfiguration);
            networkType(model.networkType);
            dbSystemId(model.dbSystemId);
            masterUserSecret(model.masterUserSecret);
            ioOptimizedNextAllowedModificationTime(model.ioOptimizedNextAllowedModificationTime);
            localWriteForwardingStatus(model.localWriteForwardingStatus);
            awsBackupRecoveryPointArn(model.awsBackupRecoveryPointArn);
            limitlessDatabase(model.limitlessDatabase);
            storageThroughput(model.storageThroughput);
        }

        public final Integer getAllocatedStorage() {
            return allocatedStorage;
        }

        public final void setAllocatedStorage(Integer allocatedStorage) {
            this.allocatedStorage = allocatedStorage;
        }

        @Override
        public final Builder allocatedStorage(Integer allocatedStorage) {
            this.allocatedStorage = allocatedStorage;
            return this;
        }

        public final Collection<String> getAvailabilityZones() {
            if (availabilityZones instanceof SdkAutoConstructList) {
                return null;
            }
            return availabilityZones;
        }

        public final void setAvailabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZonesCopier.copy(availabilityZones);
        }

        @Override
        public final Builder availabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZonesCopier.copy(availabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availabilityZones(String... availabilityZones) {
            availabilityZones(Arrays.asList(availabilityZones));
            return this;
        }

        public final Integer getBackupRetentionPeriod() {
            return backupRetentionPeriod;
        }

        public final void setBackupRetentionPeriod(Integer backupRetentionPeriod) {
            this.backupRetentionPeriod = backupRetentionPeriod;
        }

        @Override
        public final Builder backupRetentionPeriod(Integer backupRetentionPeriod) {
            this.backupRetentionPeriod = backupRetentionPeriod;
            return this;
        }

        public final String getCharacterSetName() {
            return characterSetName;
        }

        public final void setCharacterSetName(String characterSetName) {
            this.characterSetName = characterSetName;
        }

        @Override
        public final Builder characterSetName(String characterSetName) {
            this.characterSetName = characterSetName;
            return this;
        }

        public final String getDatabaseName() {
            return databaseName;
        }

        public final void setDatabaseName(String databaseName) {
            this.databaseName = databaseName;
        }

        @Override
        public final Builder databaseName(String databaseName) {
            this.databaseName = databaseName;
            return this;
        }

        public final String getDbClusterIdentifier() {
            return dbClusterIdentifier;
        }

        public final void setDbClusterIdentifier(String dbClusterIdentifier) {
            this.dbClusterIdentifier = dbClusterIdentifier;
        }

        @Override
        public final Builder dbClusterIdentifier(String dbClusterIdentifier) {
            this.dbClusterIdentifier = dbClusterIdentifier;
            return this;
        }

        public final String getDbClusterParameterGroup() {
            return dbClusterParameterGroup;
        }

        public final void setDbClusterParameterGroup(String dbClusterParameterGroup) {
            this.dbClusterParameterGroup = dbClusterParameterGroup;
        }

        @Override
        public final Builder dbClusterParameterGroup(String dbClusterParameterGroup) {
            this.dbClusterParameterGroup = dbClusterParameterGroup;
            return this;
        }

        public final String getDbSubnetGroup() {
            return dbSubnetGroup;
        }

        public final void setDbSubnetGroup(String dbSubnetGroup) {
            this.dbSubnetGroup = dbSubnetGroup;
        }

        @Override
        public final Builder dbSubnetGroup(String dbSubnetGroup) {
            this.dbSubnetGroup = dbSubnetGroup;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final Instant getAutomaticRestartTime() {
            return automaticRestartTime;
        }

        public final void setAutomaticRestartTime(Instant automaticRestartTime) {
            this.automaticRestartTime = automaticRestartTime;
        }

        @Override
        public final Builder automaticRestartTime(Instant automaticRestartTime) {
            this.automaticRestartTime = automaticRestartTime;
            return this;
        }

        public final String getPercentProgress() {
            return percentProgress;
        }

        public final void setPercentProgress(String percentProgress) {
            this.percentProgress = percentProgress;
        }

        @Override
        public final Builder percentProgress(String percentProgress) {
            this.percentProgress = percentProgress;
            return this;
        }

        public final Instant getEarliestRestorableTime() {
            return earliestRestorableTime;
        }

        public final void setEarliestRestorableTime(Instant earliestRestorableTime) {
            this.earliestRestorableTime = earliestRestorableTime;
        }

        @Override
        public final Builder earliestRestorableTime(Instant earliestRestorableTime) {
            this.earliestRestorableTime = earliestRestorableTime;
            return this;
        }

        public final String getEndpoint() {
            return endpoint;
        }

        public final void setEndpoint(String endpoint) {
            this.endpoint = endpoint;
        }

        @Override
        public final Builder endpoint(String endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        public final String getReaderEndpoint() {
            return readerEndpoint;
        }

        public final void setReaderEndpoint(String readerEndpoint) {
            this.readerEndpoint = readerEndpoint;
        }

        @Override
        public final Builder readerEndpoint(String readerEndpoint) {
            this.readerEndpoint = readerEndpoint;
            return this;
        }

        public final Collection<String> getCustomEndpoints() {
            if (customEndpoints instanceof SdkAutoConstructList) {
                return null;
            }
            return customEndpoints;
        }

        public final void setCustomEndpoints(Collection<String> customEndpoints) {
            this.customEndpoints = StringListCopier.copy(customEndpoints);
        }

        @Override
        public final Builder customEndpoints(Collection<String> customEndpoints) {
            this.customEndpoints = StringListCopier.copy(customEndpoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder customEndpoints(String... customEndpoints) {
            customEndpoints(Arrays.asList(customEndpoints));
            return this;
        }

        public final Boolean getMultiAZ() {
            return multiAZ;
        }

        public final void setMultiAZ(Boolean multiAZ) {
            this.multiAZ = multiAZ;
        }

        @Override
        public final Builder multiAZ(Boolean multiAZ) {
            this.multiAZ = multiAZ;
            return this;
        }

        public final String getEngine() {
            return engine;
        }

        public final void setEngine(String engine) {
            this.engine = engine;
        }

        @Override
        public final Builder engine(String engine) {
            this.engine = engine;
            return this;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final Instant getLatestRestorableTime() {
            return latestRestorableTime;
        }

        public final void setLatestRestorableTime(Instant latestRestorableTime) {
            this.latestRestorableTime = latestRestorableTime;
        }

        @Override
        public final Builder latestRestorableTime(Instant latestRestorableTime) {
            this.latestRestorableTime = latestRestorableTime;
            return this;
        }

        public final Integer getPort() {
            return port;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final String getMasterUsername() {
            return masterUsername;
        }

        public final void setMasterUsername(String masterUsername) {
            this.masterUsername = masterUsername;
        }

        @Override
        public final Builder masterUsername(String masterUsername) {
            this.masterUsername = masterUsername;
            return this;
        }

        public final List<DBClusterOptionGroupStatus.Builder> getDbClusterOptionGroupMemberships() {
            List<DBClusterOptionGroupStatus.Builder> result = DBClusterOptionGroupMembershipsCopier
                    .copyToBuilder(this.dbClusterOptionGroupMemberships);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDbClusterOptionGroupMemberships(
                Collection<DBClusterOptionGroupStatus.BuilderImpl> dbClusterOptionGroupMemberships) {
            this.dbClusterOptionGroupMemberships = DBClusterOptionGroupMembershipsCopier
                    .copyFromBuilder(dbClusterOptionGroupMemberships);
        }

        @Override
        public final Builder dbClusterOptionGroupMemberships(
                Collection<DBClusterOptionGroupStatus> dbClusterOptionGroupMemberships) {
            this.dbClusterOptionGroupMemberships = DBClusterOptionGroupMembershipsCopier.copy(dbClusterOptionGroupMemberships);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dbClusterOptionGroupMemberships(DBClusterOptionGroupStatus... dbClusterOptionGroupMemberships) {
            dbClusterOptionGroupMemberships(Arrays.asList(dbClusterOptionGroupMemberships));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dbClusterOptionGroupMemberships(
                Consumer<DBClusterOptionGroupStatus.Builder>... dbClusterOptionGroupMemberships) {
            dbClusterOptionGroupMemberships(Stream.of(dbClusterOptionGroupMemberships)
                    .map(c -> DBClusterOptionGroupStatus.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getPreferredBackupWindow() {
            return preferredBackupWindow;
        }

        public final void setPreferredBackupWindow(String preferredBackupWindow) {
            this.preferredBackupWindow = preferredBackupWindow;
        }

        @Override
        public final Builder preferredBackupWindow(String preferredBackupWindow) {
            this.preferredBackupWindow = preferredBackupWindow;
            return this;
        }

        public final String getPreferredMaintenanceWindow() {
            return preferredMaintenanceWindow;
        }

        public final void setPreferredMaintenanceWindow(String preferredMaintenanceWindow) {
            this.preferredMaintenanceWindow = preferredMaintenanceWindow;
        }

        @Override
        public final Builder preferredMaintenanceWindow(String preferredMaintenanceWindow) {
            this.preferredMaintenanceWindow = preferredMaintenanceWindow;
            return this;
        }

        public final String getReplicationSourceIdentifier() {
            return replicationSourceIdentifier;
        }

        public final void setReplicationSourceIdentifier(String replicationSourceIdentifier) {
            this.replicationSourceIdentifier = replicationSourceIdentifier;
        }

        @Override
        public final Builder replicationSourceIdentifier(String replicationSourceIdentifier) {
            this.replicationSourceIdentifier = replicationSourceIdentifier;
            return this;
        }

        public final Collection<String> getReadReplicaIdentifiers() {
            if (readReplicaIdentifiers instanceof SdkAutoConstructList) {
                return null;
            }
            return readReplicaIdentifiers;
        }

        public final void setReadReplicaIdentifiers(Collection<String> readReplicaIdentifiers) {
            this.readReplicaIdentifiers = ReadReplicaIdentifierListCopier.copy(readReplicaIdentifiers);
        }

        @Override
        public final Builder readReplicaIdentifiers(Collection<String> readReplicaIdentifiers) {
            this.readReplicaIdentifiers = ReadReplicaIdentifierListCopier.copy(readReplicaIdentifiers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder readReplicaIdentifiers(String... readReplicaIdentifiers) {
            readReplicaIdentifiers(Arrays.asList(readReplicaIdentifiers));
            return this;
        }

        public final List<DBClusterStatusInfo.Builder> getStatusInfos() {
            List<DBClusterStatusInfo.Builder> result = DBClusterStatusInfoListCopier.copyToBuilder(this.statusInfos);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setStatusInfos(Collection<DBClusterStatusInfo.BuilderImpl> statusInfos) {
            this.statusInfos = DBClusterStatusInfoListCopier.copyFromBuilder(statusInfos);
        }

        @Override
        public final Builder statusInfos(Collection<DBClusterStatusInfo> statusInfos) {
            this.statusInfos = DBClusterStatusInfoListCopier.copy(statusInfos);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statusInfos(DBClusterStatusInfo... statusInfos) {
            statusInfos(Arrays.asList(statusInfos));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statusInfos(Consumer<DBClusterStatusInfo.Builder>... statusInfos) {
            statusInfos(Stream.of(statusInfos).map(c -> DBClusterStatusInfo.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<DBClusterMember.Builder> getDbClusterMembers() {
            List<DBClusterMember.Builder> result = DBClusterMemberListCopier.copyToBuilder(this.dbClusterMembers);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDbClusterMembers(Collection<DBClusterMember.BuilderImpl> dbClusterMembers) {
            this.dbClusterMembers = DBClusterMemberListCopier.copyFromBuilder(dbClusterMembers);
        }

        @Override
        public final Builder dbClusterMembers(Collection<DBClusterMember> dbClusterMembers) {
            this.dbClusterMembers = DBClusterMemberListCopier.copy(dbClusterMembers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dbClusterMembers(DBClusterMember... dbClusterMembers) {
            dbClusterMembers(Arrays.asList(dbClusterMembers));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dbClusterMembers(Consumer<DBClusterMember.Builder>... dbClusterMembers) {
            dbClusterMembers(Stream.of(dbClusterMembers).map(c -> DBClusterMember.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<VpcSecurityGroupMembership.Builder> getVpcSecurityGroups() {
            List<VpcSecurityGroupMembership.Builder> result = VpcSecurityGroupMembershipListCopier
                    .copyToBuilder(this.vpcSecurityGroups);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVpcSecurityGroups(Collection<VpcSecurityGroupMembership.BuilderImpl> vpcSecurityGroups) {
            this.vpcSecurityGroups = VpcSecurityGroupMembershipListCopier.copyFromBuilder(vpcSecurityGroups);
        }

        @Override
        public final Builder vpcSecurityGroups(Collection<VpcSecurityGroupMembership> vpcSecurityGroups) {
            this.vpcSecurityGroups = VpcSecurityGroupMembershipListCopier.copy(vpcSecurityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vpcSecurityGroups(VpcSecurityGroupMembership... vpcSecurityGroups) {
            vpcSecurityGroups(Arrays.asList(vpcSecurityGroups));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vpcSecurityGroups(Consumer<VpcSecurityGroupMembership.Builder>... vpcSecurityGroups) {
            vpcSecurityGroups(Stream.of(vpcSecurityGroups)
                    .map(c -> VpcSecurityGroupMembership.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getHostedZoneId() {
            return hostedZoneId;
        }

        public final void setHostedZoneId(String hostedZoneId) {
            this.hostedZoneId = hostedZoneId;
        }

        @Override
        public final Builder hostedZoneId(String hostedZoneId) {
            this.hostedZoneId = hostedZoneId;
            return this;
        }

        public final Boolean getStorageEncrypted() {
            return storageEncrypted;
        }

        public final void setStorageEncrypted(Boolean storageEncrypted) {
            this.storageEncrypted = storageEncrypted;
        }

        @Override
        public final Builder storageEncrypted(Boolean storageEncrypted) {
            this.storageEncrypted = storageEncrypted;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final String getDbClusterResourceId() {
            return dbClusterResourceId;
        }

        public final void setDbClusterResourceId(String dbClusterResourceId) {
            this.dbClusterResourceId = dbClusterResourceId;
        }

        @Override
        public final Builder dbClusterResourceId(String dbClusterResourceId) {
            this.dbClusterResourceId = dbClusterResourceId;
            return this;
        }

        public final String getDbClusterArn() {
            return dbClusterArn;
        }

        public final void setDbClusterArn(String dbClusterArn) {
            this.dbClusterArn = dbClusterArn;
        }

        @Override
        public final Builder dbClusterArn(String dbClusterArn) {
            this.dbClusterArn = dbClusterArn;
            return this;
        }

        public final List<DBClusterRole.Builder> getAssociatedRoles() {
            List<DBClusterRole.Builder> result = DBClusterRolesCopier.copyToBuilder(this.associatedRoles);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAssociatedRoles(Collection<DBClusterRole.BuilderImpl> associatedRoles) {
            this.associatedRoles = DBClusterRolesCopier.copyFromBuilder(associatedRoles);
        }

        @Override
        public final Builder associatedRoles(Collection<DBClusterRole> associatedRoles) {
            this.associatedRoles = DBClusterRolesCopier.copy(associatedRoles);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder associatedRoles(DBClusterRole... associatedRoles) {
            associatedRoles(Arrays.asList(associatedRoles));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder associatedRoles(Consumer<DBClusterRole.Builder>... associatedRoles) {
            associatedRoles(Stream.of(associatedRoles).map(c -> DBClusterRole.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getIamDatabaseAuthenticationEnabled() {
            return iamDatabaseAuthenticationEnabled;
        }

        public final void setIamDatabaseAuthenticationEnabled(Boolean iamDatabaseAuthenticationEnabled) {
            this.iamDatabaseAuthenticationEnabled = iamDatabaseAuthenticationEnabled;
        }

        @Override
        public final Builder iamDatabaseAuthenticationEnabled(Boolean iamDatabaseAuthenticationEnabled) {
            this.iamDatabaseAuthenticationEnabled = iamDatabaseAuthenticationEnabled;
            return this;
        }

        public final String getCloneGroupId() {
            return cloneGroupId;
        }

        public final void setCloneGroupId(String cloneGroupId) {
            this.cloneGroupId = cloneGroupId;
        }

        @Override
        public final Builder cloneGroupId(String cloneGroupId) {
            this.cloneGroupId = cloneGroupId;
            return this;
        }

        public final Instant getClusterCreateTime() {
            return clusterCreateTime;
        }

        public final void setClusterCreateTime(Instant clusterCreateTime) {
            this.clusterCreateTime = clusterCreateTime;
        }

        @Override
        public final Builder clusterCreateTime(Instant clusterCreateTime) {
            this.clusterCreateTime = clusterCreateTime;
            return this;
        }

        public final Instant getEarliestBacktrackTime() {
            return earliestBacktrackTime;
        }

        public final void setEarliestBacktrackTime(Instant earliestBacktrackTime) {
            this.earliestBacktrackTime = earliestBacktrackTime;
        }

        @Override
        public final Builder earliestBacktrackTime(Instant earliestBacktrackTime) {
            this.earliestBacktrackTime = earliestBacktrackTime;
            return this;
        }

        public final Long getBacktrackWindow() {
            return backtrackWindow;
        }

        public final void setBacktrackWindow(Long backtrackWindow) {
            this.backtrackWindow = backtrackWindow;
        }

        @Override
        public final Builder backtrackWindow(Long backtrackWindow) {
            this.backtrackWindow = backtrackWindow;
            return this;
        }

        public final Long getBacktrackConsumedChangeRecords() {
            return backtrackConsumedChangeRecords;
        }

        public final void setBacktrackConsumedChangeRecords(Long backtrackConsumedChangeRecords) {
            this.backtrackConsumedChangeRecords = backtrackConsumedChangeRecords;
        }

        @Override
        public final Builder backtrackConsumedChangeRecords(Long backtrackConsumedChangeRecords) {
            this.backtrackConsumedChangeRecords = backtrackConsumedChangeRecords;
            return this;
        }

        public final Collection<String> getEnabledCloudwatchLogsExports() {
            if (enabledCloudwatchLogsExports instanceof SdkAutoConstructList) {
                return null;
            }
            return enabledCloudwatchLogsExports;
        }

        public final void setEnabledCloudwatchLogsExports(Collection<String> enabledCloudwatchLogsExports) {
            this.enabledCloudwatchLogsExports = LogTypeListCopier.copy(enabledCloudwatchLogsExports);
        }

        @Override
        public final Builder enabledCloudwatchLogsExports(Collection<String> enabledCloudwatchLogsExports) {
            this.enabledCloudwatchLogsExports = LogTypeListCopier.copy(enabledCloudwatchLogsExports);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder enabledCloudwatchLogsExports(String... enabledCloudwatchLogsExports) {
            enabledCloudwatchLogsExports(Arrays.asList(enabledCloudwatchLogsExports));
            return this;
        }

        public final Integer getCapacity() {
            return capacity;
        }

        public final void setCapacity(Integer capacity) {
            this.capacity = capacity;
        }

        @Override
        public final Builder capacity(Integer capacity) {
            this.capacity = capacity;
            return this;
        }

        public final String getEngineMode() {
            return engineMode;
        }

        public final void setEngineMode(String engineMode) {
            this.engineMode = engineMode;
        }

        @Override
        public final Builder engineMode(String engineMode) {
            this.engineMode = engineMode;
            return this;
        }

        public final ScalingConfigurationInfo.Builder getScalingConfigurationInfo() {
            return scalingConfigurationInfo != null ? scalingConfigurationInfo.toBuilder() : null;
        }

        public final void setScalingConfigurationInfo(ScalingConfigurationInfo.BuilderImpl scalingConfigurationInfo) {
            this.scalingConfigurationInfo = scalingConfigurationInfo != null ? scalingConfigurationInfo.build() : null;
        }

        @Override
        public final Builder scalingConfigurationInfo(ScalingConfigurationInfo scalingConfigurationInfo) {
            this.scalingConfigurationInfo = scalingConfigurationInfo;
            return this;
        }

        public final RdsCustomClusterConfiguration.Builder getRdsCustomClusterConfiguration() {
            return rdsCustomClusterConfiguration != null ? rdsCustomClusterConfiguration.toBuilder() : null;
        }

        public final void setRdsCustomClusterConfiguration(RdsCustomClusterConfiguration.BuilderImpl rdsCustomClusterConfiguration) {
            this.rdsCustomClusterConfiguration = rdsCustomClusterConfiguration != null ? rdsCustomClusterConfiguration.build()
                    : null;
        }

        @Override
        public final Builder rdsCustomClusterConfiguration(RdsCustomClusterConfiguration rdsCustomClusterConfiguration) {
            this.rdsCustomClusterConfiguration = rdsCustomClusterConfiguration;
            return this;
        }

        public final Boolean getDeletionProtection() {
            return deletionProtection;
        }

        public final void setDeletionProtection(Boolean deletionProtection) {
            this.deletionProtection = deletionProtection;
        }

        @Override
        public final Builder deletionProtection(Boolean deletionProtection) {
            this.deletionProtection = deletionProtection;
            return this;
        }

        public final Boolean getHttpEndpointEnabled() {
            return httpEndpointEnabled;
        }

        public final void setHttpEndpointEnabled(Boolean httpEndpointEnabled) {
            this.httpEndpointEnabled = httpEndpointEnabled;
        }

        @Override
        public final Builder httpEndpointEnabled(Boolean httpEndpointEnabled) {
            this.httpEndpointEnabled = httpEndpointEnabled;
            return this;
        }

        public final String getActivityStreamMode() {
            return activityStreamMode;
        }

        public final void setActivityStreamMode(String activityStreamMode) {
            this.activityStreamMode = activityStreamMode;
        }

        @Override
        public final Builder activityStreamMode(String activityStreamMode) {
            this.activityStreamMode = activityStreamMode;
            return this;
        }

        @Override
        public final Builder activityStreamMode(ActivityStreamMode activityStreamMode) {
            this.activityStreamMode(activityStreamMode == null ? null : activityStreamMode.toString());
            return this;
        }

        public final String getActivityStreamStatus() {
            return activityStreamStatus;
        }

        public final void setActivityStreamStatus(String activityStreamStatus) {
            this.activityStreamStatus = activityStreamStatus;
        }

        @Override
        public final Builder activityStreamStatus(String activityStreamStatus) {
            this.activityStreamStatus = activityStreamStatus;
            return this;
        }

        @Override
        public final Builder activityStreamStatus(ActivityStreamStatus activityStreamStatus) {
            this.activityStreamStatus(activityStreamStatus == null ? null : activityStreamStatus.toString());
            return this;
        }

        public final String getActivityStreamKmsKeyId() {
            return activityStreamKmsKeyId;
        }

        public final void setActivityStreamKmsKeyId(String activityStreamKmsKeyId) {
            this.activityStreamKmsKeyId = activityStreamKmsKeyId;
        }

        @Override
        public final Builder activityStreamKmsKeyId(String activityStreamKmsKeyId) {
            this.activityStreamKmsKeyId = activityStreamKmsKeyId;
            return this;
        }

        public final String getActivityStreamKinesisStreamName() {
            return activityStreamKinesisStreamName;
        }

        public final void setActivityStreamKinesisStreamName(String activityStreamKinesisStreamName) {
            this.activityStreamKinesisStreamName = activityStreamKinesisStreamName;
        }

        @Override
        public final Builder activityStreamKinesisStreamName(String activityStreamKinesisStreamName) {
            this.activityStreamKinesisStreamName = activityStreamKinesisStreamName;
            return this;
        }

        public final Boolean getCopyTagsToSnapshot() {
            return copyTagsToSnapshot;
        }

        public final void setCopyTagsToSnapshot(Boolean copyTagsToSnapshot) {
            this.copyTagsToSnapshot = copyTagsToSnapshot;
        }

        @Override
        public final Builder copyTagsToSnapshot(Boolean copyTagsToSnapshot) {
            this.copyTagsToSnapshot = copyTagsToSnapshot;
            return this;
        }

        public final Boolean getCrossAccountClone() {
            return crossAccountClone;
        }

        public final void setCrossAccountClone(Boolean crossAccountClone) {
            this.crossAccountClone = crossAccountClone;
        }

        @Override
        public final Builder crossAccountClone(Boolean crossAccountClone) {
            this.crossAccountClone = crossAccountClone;
            return this;
        }

        public final List<DomainMembership.Builder> getDomainMemberships() {
            List<DomainMembership.Builder> result = DomainMembershipListCopier.copyToBuilder(this.domainMemberships);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDomainMemberships(Collection<DomainMembership.BuilderImpl> domainMemberships) {
            this.domainMemberships = DomainMembershipListCopier.copyFromBuilder(domainMemberships);
        }

        @Override
        public final Builder domainMemberships(Collection<DomainMembership> domainMemberships) {
            this.domainMemberships = DomainMembershipListCopier.copy(domainMemberships);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder domainMemberships(DomainMembership... domainMemberships) {
            domainMemberships(Arrays.asList(domainMemberships));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder domainMemberships(Consumer<DomainMembership.Builder>... domainMemberships) {
            domainMemberships(Stream.of(domainMemberships).map(c -> DomainMembership.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<Tag.Builder> getTagList() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tagList);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTagList(Collection<Tag.BuilderImpl> tagList) {
            this.tagList = TagListCopier.copyFromBuilder(tagList);
        }

        @Override
        public final Builder tagList(Collection<Tag> tagList) {
            this.tagList = TagListCopier.copy(tagList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagList(Tag... tagList) {
            tagList(Arrays.asList(tagList));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagList(Consumer<Tag.Builder>... tagList) {
            tagList(Stream.of(tagList).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getGlobalWriteForwardingStatus() {
            return globalWriteForwardingStatus;
        }

        public final void setGlobalWriteForwardingStatus(String globalWriteForwardingStatus) {
            this.globalWriteForwardingStatus = globalWriteForwardingStatus;
        }

        @Override
        public final Builder globalWriteForwardingStatus(String globalWriteForwardingStatus) {
            this.globalWriteForwardingStatus = globalWriteForwardingStatus;
            return this;
        }

        @Override
        public final Builder globalWriteForwardingStatus(WriteForwardingStatus globalWriteForwardingStatus) {
            this.globalWriteForwardingStatus(globalWriteForwardingStatus == null ? null : globalWriteForwardingStatus.toString());
            return this;
        }

        public final Boolean getGlobalWriteForwardingRequested() {
            return globalWriteForwardingRequested;
        }

        public final void setGlobalWriteForwardingRequested(Boolean globalWriteForwardingRequested) {
            this.globalWriteForwardingRequested = globalWriteForwardingRequested;
        }

        @Override
        public final Builder globalWriteForwardingRequested(Boolean globalWriteForwardingRequested) {
            this.globalWriteForwardingRequested = globalWriteForwardingRequested;
            return this;
        }

        public final ClusterPendingModifiedValues.Builder getPendingModifiedValues() {
            return pendingModifiedValues != null ? pendingModifiedValues.toBuilder() : null;
        }

        public final void setPendingModifiedValues(ClusterPendingModifiedValues.BuilderImpl pendingModifiedValues) {
            this.pendingModifiedValues = pendingModifiedValues != null ? pendingModifiedValues.build() : null;
        }

        @Override
        public final Builder pendingModifiedValues(ClusterPendingModifiedValues pendingModifiedValues) {
            this.pendingModifiedValues = pendingModifiedValues;
            return this;
        }

        public final String getDbClusterInstanceClass() {
            return dbClusterInstanceClass;
        }

        public final void setDbClusterInstanceClass(String dbClusterInstanceClass) {
            this.dbClusterInstanceClass = dbClusterInstanceClass;
        }

        @Override
        public final Builder dbClusterInstanceClass(String dbClusterInstanceClass) {
            this.dbClusterInstanceClass = dbClusterInstanceClass;
            return this;
        }

        public final String getStorageType() {
            return storageType;
        }

        public final void setStorageType(String storageType) {
            this.storageType = storageType;
        }

        @Override
        public final Builder storageType(String storageType) {
            this.storageType = storageType;
            return this;
        }

        public final Integer getIops() {
            return iops;
        }

        public final void setIops(Integer iops) {
            this.iops = iops;
        }

        @Override
        public final Builder iops(Integer iops) {
            this.iops = iops;
            return this;
        }

        public final Boolean getPubliclyAccessible() {
            return publiclyAccessible;
        }

        public final void setPubliclyAccessible(Boolean publiclyAccessible) {
            this.publiclyAccessible = publiclyAccessible;
        }

        @Override
        public final Builder publiclyAccessible(Boolean publiclyAccessible) {
            this.publiclyAccessible = publiclyAccessible;
            return this;
        }

        public final Boolean getAutoMinorVersionUpgrade() {
            return autoMinorVersionUpgrade;
        }

        public final void setAutoMinorVersionUpgrade(Boolean autoMinorVersionUpgrade) {
            this.autoMinorVersionUpgrade = autoMinorVersionUpgrade;
        }

        @Override
        public final Builder autoMinorVersionUpgrade(Boolean autoMinorVersionUpgrade) {
            this.autoMinorVersionUpgrade = autoMinorVersionUpgrade;
            return this;
        }

        public final Integer getMonitoringInterval() {
            return monitoringInterval;
        }

        public final void setMonitoringInterval(Integer monitoringInterval) {
            this.monitoringInterval = monitoringInterval;
        }

        @Override
        public final Builder monitoringInterval(Integer monitoringInterval) {
            this.monitoringInterval = monitoringInterval;
            return this;
        }

        public final String getMonitoringRoleArn() {
            return monitoringRoleArn;
        }

        public final void setMonitoringRoleArn(String monitoringRoleArn) {
            this.monitoringRoleArn = monitoringRoleArn;
        }

        @Override
        public final Builder monitoringRoleArn(String monitoringRoleArn) {
            this.monitoringRoleArn = monitoringRoleArn;
            return this;
        }

        public final Boolean getPerformanceInsightsEnabled() {
            return performanceInsightsEnabled;
        }

        public final void setPerformanceInsightsEnabled(Boolean performanceInsightsEnabled) {
            this.performanceInsightsEnabled = performanceInsightsEnabled;
        }

        @Override
        public final Builder performanceInsightsEnabled(Boolean performanceInsightsEnabled) {
            this.performanceInsightsEnabled = performanceInsightsEnabled;
            return this;
        }

        public final String getPerformanceInsightsKMSKeyId() {
            return performanceInsightsKMSKeyId;
        }

        public final void setPerformanceInsightsKMSKeyId(String performanceInsightsKMSKeyId) {
            this.performanceInsightsKMSKeyId = performanceInsightsKMSKeyId;
        }

        @Override
        public final Builder performanceInsightsKMSKeyId(String performanceInsightsKMSKeyId) {
            this.performanceInsightsKMSKeyId = performanceInsightsKMSKeyId;
            return this;
        }

        public final Integer getPerformanceInsightsRetentionPeriod() {
            return performanceInsightsRetentionPeriod;
        }

        public final void setPerformanceInsightsRetentionPeriod(Integer performanceInsightsRetentionPeriod) {
            this.performanceInsightsRetentionPeriod = performanceInsightsRetentionPeriod;
        }

        @Override
        public final Builder performanceInsightsRetentionPeriod(Integer performanceInsightsRetentionPeriod) {
            this.performanceInsightsRetentionPeriod = performanceInsightsRetentionPeriod;
            return this;
        }

        public final ServerlessV2ScalingConfigurationInfo.Builder getServerlessV2ScalingConfiguration() {
            return serverlessV2ScalingConfiguration != null ? serverlessV2ScalingConfiguration.toBuilder() : null;
        }

        public final void setServerlessV2ScalingConfiguration(
                ServerlessV2ScalingConfigurationInfo.BuilderImpl serverlessV2ScalingConfiguration) {
            this.serverlessV2ScalingConfiguration = serverlessV2ScalingConfiguration != null ? serverlessV2ScalingConfiguration
                    .build() : null;
        }

        @Override
        public final Builder serverlessV2ScalingConfiguration(
                ServerlessV2ScalingConfigurationInfo serverlessV2ScalingConfiguration) {
            this.serverlessV2ScalingConfiguration = serverlessV2ScalingConfiguration;
            return this;
        }

        public final String getNetworkType() {
            return networkType;
        }

        public final void setNetworkType(String networkType) {
            this.networkType = networkType;
        }

        @Override
        public final Builder networkType(String networkType) {
            this.networkType = networkType;
            return this;
        }

        public final String getDbSystemId() {
            return dbSystemId;
        }

        public final void setDbSystemId(String dbSystemId) {
            this.dbSystemId = dbSystemId;
        }

        @Override
        public final Builder dbSystemId(String dbSystemId) {
            this.dbSystemId = dbSystemId;
            return this;
        }

        public final MasterUserSecret.Builder getMasterUserSecret() {
            return masterUserSecret != null ? masterUserSecret.toBuilder() : null;
        }

        public final void setMasterUserSecret(MasterUserSecret.BuilderImpl masterUserSecret) {
            this.masterUserSecret = masterUserSecret != null ? masterUserSecret.build() : null;
        }

        @Override
        public final Builder masterUserSecret(MasterUserSecret masterUserSecret) {
            this.masterUserSecret = masterUserSecret;
            return this;
        }

        public final Instant getIoOptimizedNextAllowedModificationTime() {
            return ioOptimizedNextAllowedModificationTime;
        }

        public final void setIoOptimizedNextAllowedModificationTime(Instant ioOptimizedNextAllowedModificationTime) {
            this.ioOptimizedNextAllowedModificationTime = ioOptimizedNextAllowedModificationTime;
        }

        @Override
        public final Builder ioOptimizedNextAllowedModificationTime(Instant ioOptimizedNextAllowedModificationTime) {
            this.ioOptimizedNextAllowedModificationTime = ioOptimizedNextAllowedModificationTime;
            return this;
        }

        public final String getLocalWriteForwardingStatus() {
            return localWriteForwardingStatus;
        }

        public final void setLocalWriteForwardingStatus(String localWriteForwardingStatus) {
            this.localWriteForwardingStatus = localWriteForwardingStatus;
        }

        @Override
        public final Builder localWriteForwardingStatus(String localWriteForwardingStatus) {
            this.localWriteForwardingStatus = localWriteForwardingStatus;
            return this;
        }

        @Override
        public final Builder localWriteForwardingStatus(LocalWriteForwardingStatus localWriteForwardingStatus) {
            this.localWriteForwardingStatus(localWriteForwardingStatus == null ? null : localWriteForwardingStatus.toString());
            return this;
        }

        public final String getAwsBackupRecoveryPointArn() {
            return awsBackupRecoveryPointArn;
        }

        public final void setAwsBackupRecoveryPointArn(String awsBackupRecoveryPointArn) {
            this.awsBackupRecoveryPointArn = awsBackupRecoveryPointArn;
        }

        @Override
        public final Builder awsBackupRecoveryPointArn(String awsBackupRecoveryPointArn) {
            this.awsBackupRecoveryPointArn = awsBackupRecoveryPointArn;
            return this;
        }

        public final LimitlessDatabase.Builder getLimitlessDatabase() {
            return limitlessDatabase != null ? limitlessDatabase.toBuilder() : null;
        }

        public final void setLimitlessDatabase(LimitlessDatabase.BuilderImpl limitlessDatabase) {
            this.limitlessDatabase = limitlessDatabase != null ? limitlessDatabase.build() : null;
        }

        @Override
        public final Builder limitlessDatabase(LimitlessDatabase limitlessDatabase) {
            this.limitlessDatabase = limitlessDatabase;
            return this;
        }

        public final Integer getStorageThroughput() {
            return storageThroughput;
        }

        public final void setStorageThroughput(Integer storageThroughput) {
            this.storageThroughput = storageThroughput;
        }

        @Override
        public final Builder storageThroughput(Integer storageThroughput) {
            this.storageThroughput = storageThroughput;
            return this;
        }

        @Override
        public DBCluster build() {
            return new DBCluster(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
