/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rdsdata;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.rdsdata.model.BadRequestException;
import software.amazon.awssdk.services.rdsdata.model.ExecuteSqlRequest;
import software.amazon.awssdk.services.rdsdata.model.ExecuteSqlResponse;
import software.amazon.awssdk.services.rdsdata.model.ForbiddenException;
import software.amazon.awssdk.services.rdsdata.model.InternalServerErrorException;
import software.amazon.awssdk.services.rdsdata.model.RdsDataException;
import software.amazon.awssdk.services.rdsdata.model.ServiceUnavailableErrorException;

/**
 * Service client for accessing AWS RDS DataService. This can be created using the static {@link #builder()} method.
 *
 * AWS RDS DataService provides Http Endpoint to query RDS databases.
 */
@Generated("software.amazon.awssdk:codegen")
public interface RdsDataClient extends SdkClient {
    String SERVICE_NAME = "rds-data";

    /**
     * Create a {@link RdsDataClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static RdsDataClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link RdsDataClient}.
     */
    static RdsDataClientBuilder builder() {
        return new DefaultRdsDataClientBuilder();
    }

    /**
     * Executes any SQL statement on the target database synchronously
     *
     * @param executeSqlRequest
     *        Execute SQL Request
     * @return Result of the ExecuteSql operation returned by the service.
     * @throws BadRequestException
     *         Invalid Request exception
     * @throws ForbiddenException
     *         Access denied exception
     * @throws InternalServerErrorException
     *         Internal service error
     * @throws ServiceUnavailableErrorException
     *         Internal service unavailable error
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RdsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RdsDataClient.ExecuteSql
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/ExecuteSql" target="_top">AWS API
     *      Documentation</a>
     */
    default ExecuteSqlResponse executeSql(ExecuteSqlRequest executeSqlRequest) throws BadRequestException, ForbiddenException,
            InternalServerErrorException, ServiceUnavailableErrorException, AwsServiceException, SdkClientException,
            RdsDataException {
        throw new UnsupportedOperationException();
    }

    /**
     * Executes any SQL statement on the target database synchronously<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ExecuteSqlRequest.Builder} avoiding the need to
     * create one manually via {@link ExecuteSqlRequest#builder()}
     * </p>
     *
     * @param executeSqlRequest
     *        A {@link Consumer} that will call methods on {@link ExecuteSqlRequest.Builder} to create a request.
     *        Execute SQL Request
     * @return Result of the ExecuteSql operation returned by the service.
     * @throws BadRequestException
     *         Invalid Request exception
     * @throws ForbiddenException
     *         Access denied exception
     * @throws InternalServerErrorException
     *         Internal service error
     * @throws ServiceUnavailableErrorException
     *         Internal service unavailable error
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RdsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RdsDataClient.ExecuteSql
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/ExecuteSql" target="_top">AWS API
     *      Documentation</a>
     */
    default ExecuteSqlResponse executeSql(Consumer<ExecuteSqlRequest.Builder> executeSqlRequest) throws BadRequestException,
            ForbiddenException, InternalServerErrorException, ServiceUnavailableErrorException, AwsServiceException,
            SdkClientException, RdsDataException {
        return executeSql(ExecuteSqlRequest.builder().applyMutation(executeSqlRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of("rds-data");
    }
}
