/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rdsdata;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.rdsdata.model.BadRequestException;
import software.amazon.awssdk.services.rdsdata.model.BatchExecuteStatementRequest;
import software.amazon.awssdk.services.rdsdata.model.BatchExecuteStatementResponse;
import software.amazon.awssdk.services.rdsdata.model.BeginTransactionRequest;
import software.amazon.awssdk.services.rdsdata.model.BeginTransactionResponse;
import software.amazon.awssdk.services.rdsdata.model.CommitTransactionRequest;
import software.amazon.awssdk.services.rdsdata.model.CommitTransactionResponse;
import software.amazon.awssdk.services.rdsdata.model.ExecuteSqlRequest;
import software.amazon.awssdk.services.rdsdata.model.ExecuteSqlResponse;
import software.amazon.awssdk.services.rdsdata.model.ExecuteStatementRequest;
import software.amazon.awssdk.services.rdsdata.model.ExecuteStatementResponse;
import software.amazon.awssdk.services.rdsdata.model.ForbiddenException;
import software.amazon.awssdk.services.rdsdata.model.InternalServerErrorException;
import software.amazon.awssdk.services.rdsdata.model.NotFoundException;
import software.amazon.awssdk.services.rdsdata.model.RdsDataException;
import software.amazon.awssdk.services.rdsdata.model.RollbackTransactionRequest;
import software.amazon.awssdk.services.rdsdata.model.RollbackTransactionResponse;
import software.amazon.awssdk.services.rdsdata.model.ServiceUnavailableErrorException;
import software.amazon.awssdk.services.rdsdata.model.StatementTimeoutException;
import software.amazon.awssdk.services.rdsdata.transform.BatchExecuteStatementRequestMarshaller;
import software.amazon.awssdk.services.rdsdata.transform.BeginTransactionRequestMarshaller;
import software.amazon.awssdk.services.rdsdata.transform.CommitTransactionRequestMarshaller;
import software.amazon.awssdk.services.rdsdata.transform.ExecuteSqlRequestMarshaller;
import software.amazon.awssdk.services.rdsdata.transform.ExecuteStatementRequestMarshaller;
import software.amazon.awssdk.services.rdsdata.transform.RollbackTransactionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link RdsDataAsyncClient}.
 *
 * @see RdsDataAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRdsDataAsyncClient implements RdsDataAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRdsDataAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRdsDataAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Runs a batch SQL statement over an array of data.
     * </p>
     * <p>
     * You can run bulk update and insert operations for multiple records using a DML statement with different parameter
     * sets. Bulk operations can provide a significant performance improvement over individual insert and update
     * operations.
     * </p>
     * <important>
     * <p>
     * If a call isn't part of a transaction because it doesn't include the <code>transactionID</code> parameter,
     * changes that result from the call are committed automatically.
     * </p>
     * </important>
     *
     * @param batchExecuteStatementRequest
     *        The request parameters represent the input of a SQL statement over an array of data.
     * @return A Java Future containing the result of the BatchExecuteStatement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException There is an error in the call or in a SQL statement.</li>
     *         <li>StatementTimeoutException The execution of the SQL statement timed out.</li>
     *         <li>InternalServerErrorException An internal error occurred.</li>
     *         <li>ForbiddenException There are insufficient privileges to make the call.</li>
     *         <li>ServiceUnavailableErrorException The service specified by the <code>resourceArn</code> parameter is
     *         not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RdsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RdsDataAsyncClient.BatchExecuteStatement
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/BatchExecuteStatement" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchExecuteStatementResponse> batchExecuteStatement(
            BatchExecuteStatementRequest batchExecuteStatementRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchExecuteStatementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchExecuteStatementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchExecuteStatementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchExecuteStatementRequest, BatchExecuteStatementResponse>()
                            .withOperationName("BatchExecuteStatement")
                            .withMarshaller(new BatchExecuteStatementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(batchExecuteStatementRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a SQL transaction.
     * </p>
     * 
     * <pre>
     * <code> &lt;important&gt; &lt;p&gt;A transaction can run for a maximum of 24 hours. A transaction is terminated and rolled back automatically after 24 hours.&lt;/p&gt; &lt;p&gt;A transaction times out if no calls use its transaction ID in three minutes. If a transaction times out before it's committed, it's rolled back automatically.&lt;/p&gt; &lt;p&gt;DDL statements inside a transaction cause an implicit commit. We recommend that you run each DDL statement in a separate &lt;code&gt;ExecuteStatement&lt;/code&gt; call with &lt;code&gt;continueAfterTimeout&lt;/code&gt; enabled.&lt;/p&gt; &lt;/important&gt; </code>
     * </pre>
     *
     * @param beginTransactionRequest
     *        The request parameters represent the input of a request to start a SQL transaction.
     * @return A Java Future containing the result of the BeginTransaction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException There is an error in the call or in a SQL statement.</li>
     *         <li>StatementTimeoutException The execution of the SQL statement timed out.</li>
     *         <li>InternalServerErrorException An internal error occurred.</li>
     *         <li>ForbiddenException There are insufficient privileges to make the call.</li>
     *         <li>ServiceUnavailableErrorException The service specified by the <code>resourceArn</code> parameter is
     *         not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RdsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RdsDataAsyncClient.BeginTransaction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/BeginTransaction" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BeginTransactionResponse> beginTransaction(BeginTransactionRequest beginTransactionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BeginTransactionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BeginTransactionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BeginTransactionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BeginTransactionRequest, BeginTransactionResponse>()
                            .withOperationName("BeginTransaction")
                            .withMarshaller(new BeginTransactionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(beginTransactionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Ends a SQL transaction started with the <code>BeginTransaction</code> operation and commits the changes.
     * </p>
     *
     * @param commitTransactionRequest
     *        The request parameters represent the input of a commit transaction request.
     * @return A Java Future containing the result of the CommitTransaction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException There is an error in the call or in a SQL statement.</li>
     *         <li>StatementTimeoutException The execution of the SQL statement timed out.</li>
     *         <li>InternalServerErrorException An internal error occurred.</li>
     *         <li>ForbiddenException There are insufficient privileges to make the call.</li>
     *         <li>ServiceUnavailableErrorException The service specified by the <code>resourceArn</code> parameter is
     *         not available.</li>
     *         <li>NotFoundException The <code>resourceArn</code>, <code>secretArn</code>, or <code>transactionId</code>
     *         value can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RdsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RdsDataAsyncClient.CommitTransaction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/CommitTransaction" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CommitTransactionResponse> commitTransaction(CommitTransactionRequest commitTransactionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CommitTransactionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CommitTransactionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CommitTransactionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CommitTransactionRequest, CommitTransactionResponse>()
                            .withOperationName("CommitTransaction")
                            .withMarshaller(new CommitTransactionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(commitTransactionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Runs one or more SQL statements.
     * </p>
     * <important>
     * <p>
     * This operation is deprecated. Use the <code>BatchExecuteStatement</code> or <code>ExecuteStatement</code>
     * operation.
     * </p>
     * </important>
     *
     * @param executeSqlRequest
     *        The request parameters represent the input of a request to run one or more SQL statements.
     * @return A Java Future containing the result of the ExecuteSql operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException There is an error in the call or in a SQL statement.</li>
     *         <li>InternalServerErrorException An internal error occurred.</li>
     *         <li>ForbiddenException There are insufficient privileges to make the call.</li>
     *         <li>ServiceUnavailableErrorException The service specified by the <code>resourceArn</code> parameter is
     *         not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RdsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RdsDataAsyncClient.ExecuteSql
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/ExecuteSql" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ExecuteSqlResponse> executeSql(ExecuteSqlRequest executeSqlRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExecuteSqlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ExecuteSqlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExecuteSqlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExecuteSqlRequest, ExecuteSqlResponse>().withOperationName("ExecuteSql")
                            .withMarshaller(new ExecuteSqlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(executeSqlRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Runs a SQL statement against a database.
     * </p>
     * <important>
     * <p>
     * If a call isn't part of a transaction because it doesn't include the <code>transactionID</code> parameter,
     * changes that result from the call are committed automatically.
     * </p>
     * </important>
     * <p>
     * The response size limit is 1 MB or 1,000 records. If the call returns more than 1 MB of response data or over
     * 1,000 records, the call is terminated.
     * </p>
     *
     * @param executeStatementRequest
     *        The request parameters represent the input of a request to run a SQL statement against a database.
     * @return A Java Future containing the result of the ExecuteStatement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException There is an error in the call or in a SQL statement.</li>
     *         <li>StatementTimeoutException The execution of the SQL statement timed out.</li>
     *         <li>InternalServerErrorException An internal error occurred.</li>
     *         <li>ForbiddenException There are insufficient privileges to make the call.</li>
     *         <li>ServiceUnavailableErrorException The service specified by the <code>resourceArn</code> parameter is
     *         not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RdsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RdsDataAsyncClient.ExecuteStatement
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/ExecuteStatement" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ExecuteStatementResponse> executeStatement(ExecuteStatementRequest executeStatementRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExecuteStatementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExecuteStatementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExecuteStatementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExecuteStatementRequest, ExecuteStatementResponse>()
                            .withOperationName("ExecuteStatement")
                            .withMarshaller(new ExecuteStatementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(executeStatementRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Performs a rollback of a transaction. Rolling back a transaction cancels its changes.
     * </p>
     *
     * @param rollbackTransactionRequest
     *        The request parameters represent the input of a request to perform a rollback of a transaction.
     * @return A Java Future containing the result of the RollbackTransaction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException There is an error in the call or in a SQL statement.</li>
     *         <li>StatementTimeoutException The execution of the SQL statement timed out.</li>
     *         <li>InternalServerErrorException An internal error occurred.</li>
     *         <li>ForbiddenException There are insufficient privileges to make the call.</li>
     *         <li>ServiceUnavailableErrorException The service specified by the <code>resourceArn</code> parameter is
     *         not available.</li>
     *         <li>NotFoundException The <code>resourceArn</code>, <code>secretArn</code>, or <code>transactionId</code>
     *         value can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RdsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RdsDataAsyncClient.RollbackTransaction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/RollbackTransaction" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RollbackTransactionResponse> rollbackTransaction(
            RollbackTransactionRequest rollbackTransactionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RollbackTransactionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RollbackTransactionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RollbackTransactionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RollbackTransactionRequest, RollbackTransactionResponse>()
                            .withOperationName("RollbackTransaction")
                            .withMarshaller(new RollbackTransactionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(rollbackTransactionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RdsDataException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableError")
                                .exceptionBuilderSupplier(ServiceUnavailableErrorException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StatementTimeoutException")
                                .exceptionBuilderSupplier(StatementTimeoutException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
