/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rdsdata;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.rdsdata.model.AccessDeniedException;
import software.amazon.awssdk.services.rdsdata.model.BadRequestException;
import software.amazon.awssdk.services.rdsdata.model.BatchExecuteStatementRequest;
import software.amazon.awssdk.services.rdsdata.model.BatchExecuteStatementResponse;
import software.amazon.awssdk.services.rdsdata.model.BeginTransactionRequest;
import software.amazon.awssdk.services.rdsdata.model.BeginTransactionResponse;
import software.amazon.awssdk.services.rdsdata.model.CommitTransactionRequest;
import software.amazon.awssdk.services.rdsdata.model.CommitTransactionResponse;
import software.amazon.awssdk.services.rdsdata.model.ExecuteSqlRequest;
import software.amazon.awssdk.services.rdsdata.model.ExecuteSqlResponse;
import software.amazon.awssdk.services.rdsdata.model.ExecuteStatementRequest;
import software.amazon.awssdk.services.rdsdata.model.ExecuteStatementResponse;
import software.amazon.awssdk.services.rdsdata.model.ForbiddenException;
import software.amazon.awssdk.services.rdsdata.model.InternalServerErrorException;
import software.amazon.awssdk.services.rdsdata.model.NotFoundException;
import software.amazon.awssdk.services.rdsdata.model.RdsDataException;
import software.amazon.awssdk.services.rdsdata.model.RollbackTransactionRequest;
import software.amazon.awssdk.services.rdsdata.model.RollbackTransactionResponse;
import software.amazon.awssdk.services.rdsdata.model.ServiceUnavailableErrorException;
import software.amazon.awssdk.services.rdsdata.model.StatementTimeoutException;
import software.amazon.awssdk.services.rdsdata.transform.BatchExecuteStatementRequestMarshaller;
import software.amazon.awssdk.services.rdsdata.transform.BeginTransactionRequestMarshaller;
import software.amazon.awssdk.services.rdsdata.transform.CommitTransactionRequestMarshaller;
import software.amazon.awssdk.services.rdsdata.transform.ExecuteSqlRequestMarshaller;
import software.amazon.awssdk.services.rdsdata.transform.ExecuteStatementRequestMarshaller;
import software.amazon.awssdk.services.rdsdata.transform.RollbackTransactionRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link RdsDataClient}.
 *
 * @see RdsDataClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRdsDataClient implements RdsDataClient {
    private static final Logger log = Logger.loggerFor(DefaultRdsDataClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final RdsDataServiceClientConfiguration serviceClientConfiguration;

    protected DefaultRdsDataClient(RdsDataServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Runs a batch SQL statement over an array of data.
     * </p>
     * <p>
     * You can run bulk update and insert operations for multiple records using a DML statement with different parameter
     * sets. Bulk operations can provide a significant performance improvement over individual insert and update
     * operations.
     * </p>
     * <note>
     * <p>
     * If a call isn't part of a transaction because it doesn't include the <code>transactionID</code> parameter,
     * changes that result from the call are committed automatically.
     * </p>
     * <p>
     * There isn't a fixed upper limit on the number of parameter sets. However, the maximum size of the HTTP request
     * submitted through the Data API is 4 MiB. If the request exceeds this limit, the Data API returns an error and
     * doesn't process the request. This 4-MiB limit includes the size of the HTTP headers and the JSON notation in the
     * request. Thus, the number of parameter sets that you can include depends on a combination of factors, such as the
     * size of the SQL statement and the size of each parameter set.
     * </p>
     * <p>
     * The response size limit is 1 MiB. If the call returns more than 1 MiB of response data, the call is terminated.
     * </p>
     * </note>
     *
     * @param batchExecuteStatementRequest
     *        The request parameters represent the input of a SQL statement over an array of data.
     * @return Result of the BatchExecuteStatement operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws BadRequestException
     *         There is an error in the call or in a SQL statement.
     * @throws StatementTimeoutException
     *         The execution of the SQL statement timed out.
     * @throws InternalServerErrorException
     *         An internal error occurred.
     * @throws ForbiddenException
     *         There are insufficient privileges to make the call.
     * @throws ServiceUnavailableErrorException
     *         The service specified by the <code>resourceArn</code> parameter is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RdsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RdsDataClient.BatchExecuteStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/BatchExecuteStatement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchExecuteStatementResponse batchExecuteStatement(BatchExecuteStatementRequest batchExecuteStatementRequest)
            throws AccessDeniedException, BadRequestException, StatementTimeoutException, InternalServerErrorException,
            ForbiddenException, ServiceUnavailableErrorException, AwsServiceException, SdkClientException, RdsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchExecuteStatementResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchExecuteStatementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchExecuteStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RDS Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchExecuteStatement");

            return clientHandler.execute(new ClientExecutionParams<BatchExecuteStatementRequest, BatchExecuteStatementResponse>()
                    .withOperationName("BatchExecuteStatement").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(batchExecuteStatementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchExecuteStatementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a SQL transaction.
     * </p>
     * <note>
     * <p>
     * A transaction can run for a maximum of 24 hours. A transaction is terminated and rolled back automatically after
     * 24 hours.
     * </p>
     * <p>
     * A transaction times out if no calls use its transaction ID in three minutes. If a transaction times out before
     * it's committed, it's rolled back automatically.
     * </p>
     * <p>
     * DDL statements inside a transaction cause an implicit commit. We recommend that you run each DDL statement in a
     * separate <code>ExecuteStatement</code> call with <code>continueAfterTimeout</code> enabled.
     * </p>
     * </note>
     *
     * @param beginTransactionRequest
     *        The request parameters represent the input of a request to start a SQL transaction.
     * @return Result of the BeginTransaction operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws BadRequestException
     *         There is an error in the call or in a SQL statement.
     * @throws StatementTimeoutException
     *         The execution of the SQL statement timed out.
     * @throws InternalServerErrorException
     *         An internal error occurred.
     * @throws ForbiddenException
     *         There are insufficient privileges to make the call.
     * @throws ServiceUnavailableErrorException
     *         The service specified by the <code>resourceArn</code> parameter is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RdsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RdsDataClient.BeginTransaction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/BeginTransaction" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BeginTransactionResponse beginTransaction(BeginTransactionRequest beginTransactionRequest)
            throws AccessDeniedException, BadRequestException, StatementTimeoutException, InternalServerErrorException,
            ForbiddenException, ServiceUnavailableErrorException, AwsServiceException, SdkClientException, RdsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BeginTransactionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BeginTransactionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, beginTransactionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RDS Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BeginTransaction");

            return clientHandler.execute(new ClientExecutionParams<BeginTransactionRequest, BeginTransactionResponse>()
                    .withOperationName("BeginTransaction").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(beginTransactionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BeginTransactionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Ends a SQL transaction started with the <code>BeginTransaction</code> operation and commits the changes.
     * </p>
     *
     * @param commitTransactionRequest
     *        The request parameters represent the input of a commit transaction request.
     * @return Result of the CommitTransaction operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws BadRequestException
     *         There is an error in the call or in a SQL statement.
     * @throws StatementTimeoutException
     *         The execution of the SQL statement timed out.
     * @throws InternalServerErrorException
     *         An internal error occurred.
     * @throws ForbiddenException
     *         There are insufficient privileges to make the call.
     * @throws ServiceUnavailableErrorException
     *         The service specified by the <code>resourceArn</code> parameter is not available.
     * @throws NotFoundException
     *         The <code>resourceArn</code>, <code>secretArn</code>, or <code>transactionId</code> value can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RdsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RdsDataClient.CommitTransaction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/CommitTransaction" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CommitTransactionResponse commitTransaction(CommitTransactionRequest commitTransactionRequest)
            throws AccessDeniedException, BadRequestException, StatementTimeoutException, InternalServerErrorException,
            ForbiddenException, ServiceUnavailableErrorException, NotFoundException, AwsServiceException, SdkClientException,
            RdsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CommitTransactionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CommitTransactionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, commitTransactionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RDS Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CommitTransaction");

            return clientHandler.execute(new ClientExecutionParams<CommitTransactionRequest, CommitTransactionResponse>()
                    .withOperationName("CommitTransaction").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(commitTransactionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CommitTransactionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Runs one or more SQL statements.
     * </p>
     * <note>
     * <p>
     * This operation is deprecated. Use the <code>BatchExecuteStatement</code> or <code>ExecuteStatement</code>
     * operation.
     * </p>
     * </note>
     *
     * @param executeSqlRequest
     *        The request parameters represent the input of a request to run one or more SQL statements.
     * @return Result of the ExecuteSql operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws BadRequestException
     *         There is an error in the call or in a SQL statement.
     * @throws InternalServerErrorException
     *         An internal error occurred.
     * @throws ForbiddenException
     *         There are insufficient privileges to make the call.
     * @throws ServiceUnavailableErrorException
     *         The service specified by the <code>resourceArn</code> parameter is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RdsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RdsDataClient.ExecuteSql
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/ExecuteSql" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ExecuteSqlResponse executeSql(ExecuteSqlRequest executeSqlRequest) throws AccessDeniedException, BadRequestException,
            InternalServerErrorException, ForbiddenException, ServiceUnavailableErrorException, AwsServiceException,
            SdkClientException, RdsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExecuteSqlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ExecuteSqlResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeSqlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RDS Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteSql");

            return clientHandler
                    .execute(new ClientExecutionParams<ExecuteSqlRequest, ExecuteSqlResponse>().withOperationName("ExecuteSql")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(executeSqlRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ExecuteSqlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Runs a SQL statement against a database.
     * </p>
     * <note>
     * <p>
     * If a call isn't part of a transaction because it doesn't include the <code>transactionID</code> parameter,
     * changes that result from the call are committed automatically.
     * </p>
     * <p>
     * If the binary response data from the database is more than 1 MB, the call is terminated.
     * </p>
     * </note>
     *
     * @param executeStatementRequest
     *        The request parameters represent the input of a request to run a SQL statement against a database.
     * @return Result of the ExecuteStatement operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws BadRequestException
     *         There is an error in the call or in a SQL statement.
     * @throws StatementTimeoutException
     *         The execution of the SQL statement timed out.
     * @throws InternalServerErrorException
     *         An internal error occurred.
     * @throws ForbiddenException
     *         There are insufficient privileges to make the call.
     * @throws ServiceUnavailableErrorException
     *         The service specified by the <code>resourceArn</code> parameter is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RdsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RdsDataClient.ExecuteStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/ExecuteStatement" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ExecuteStatementResponse executeStatement(ExecuteStatementRequest executeStatementRequest)
            throws AccessDeniedException, BadRequestException, StatementTimeoutException, InternalServerErrorException,
            ForbiddenException, ServiceUnavailableErrorException, AwsServiceException, SdkClientException, RdsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExecuteStatementResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ExecuteStatementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RDS Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteStatement");

            return clientHandler.execute(new ClientExecutionParams<ExecuteStatementRequest, ExecuteStatementResponse>()
                    .withOperationName("ExecuteStatement").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(executeStatementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExecuteStatementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Performs a rollback of a transaction. Rolling back a transaction cancels its changes.
     * </p>
     *
     * @param rollbackTransactionRequest
     *        The request parameters represent the input of a request to perform a rollback of a transaction.
     * @return Result of the RollbackTransaction operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws BadRequestException
     *         There is an error in the call or in a SQL statement.
     * @throws StatementTimeoutException
     *         The execution of the SQL statement timed out.
     * @throws InternalServerErrorException
     *         An internal error occurred.
     * @throws ForbiddenException
     *         There are insufficient privileges to make the call.
     * @throws ServiceUnavailableErrorException
     *         The service specified by the <code>resourceArn</code> parameter is not available.
     * @throws NotFoundException
     *         The <code>resourceArn</code>, <code>secretArn</code>, or <code>transactionId</code> value can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RdsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RdsDataClient.RollbackTransaction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/RollbackTransaction" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RollbackTransactionResponse rollbackTransaction(RollbackTransactionRequest rollbackTransactionRequest)
            throws AccessDeniedException, BadRequestException, StatementTimeoutException, InternalServerErrorException,
            ForbiddenException, ServiceUnavailableErrorException, NotFoundException, AwsServiceException, SdkClientException,
            RdsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RollbackTransactionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RollbackTransactionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rollbackTransactionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RDS Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RollbackTransaction");

            return clientHandler.execute(new ClientExecutionParams<RollbackTransactionRequest, RollbackTransactionResponse>()
                    .withOperationName("RollbackTransaction").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(rollbackTransactionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RollbackTransactionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RdsDataException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableError")
                                .exceptionBuilderSupplier(ServiceUnavailableErrorException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StatementTimeoutException")
                                .exceptionBuilderSupplier(StatementTimeoutException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    @Override
    public final RdsDataServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
