/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rdsdata;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.rdsdata.internal.RdsDataServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.rdsdata.model.AccessDeniedException;
import software.amazon.awssdk.services.rdsdata.model.BadRequestException;
import software.amazon.awssdk.services.rdsdata.model.BatchExecuteStatementRequest;
import software.amazon.awssdk.services.rdsdata.model.BatchExecuteStatementResponse;
import software.amazon.awssdk.services.rdsdata.model.BeginTransactionRequest;
import software.amazon.awssdk.services.rdsdata.model.BeginTransactionResponse;
import software.amazon.awssdk.services.rdsdata.model.CommitTransactionRequest;
import software.amazon.awssdk.services.rdsdata.model.CommitTransactionResponse;
import software.amazon.awssdk.services.rdsdata.model.DatabaseErrorException;
import software.amazon.awssdk.services.rdsdata.model.DatabaseNotFoundException;
import software.amazon.awssdk.services.rdsdata.model.DatabaseResumingException;
import software.amazon.awssdk.services.rdsdata.model.DatabaseUnavailableException;
import software.amazon.awssdk.services.rdsdata.model.ExecuteSqlRequest;
import software.amazon.awssdk.services.rdsdata.model.ExecuteSqlResponse;
import software.amazon.awssdk.services.rdsdata.model.ExecuteStatementRequest;
import software.amazon.awssdk.services.rdsdata.model.ExecuteStatementResponse;
import software.amazon.awssdk.services.rdsdata.model.ForbiddenException;
import software.amazon.awssdk.services.rdsdata.model.HttpEndpointNotEnabledException;
import software.amazon.awssdk.services.rdsdata.model.InternalServerErrorException;
import software.amazon.awssdk.services.rdsdata.model.InvalidSecretException;
import software.amazon.awssdk.services.rdsdata.model.NotFoundException;
import software.amazon.awssdk.services.rdsdata.model.RdsDataException;
import software.amazon.awssdk.services.rdsdata.model.RollbackTransactionRequest;
import software.amazon.awssdk.services.rdsdata.model.RollbackTransactionResponse;
import software.amazon.awssdk.services.rdsdata.model.SecretsErrorException;
import software.amazon.awssdk.services.rdsdata.model.ServiceUnavailableErrorException;
import software.amazon.awssdk.services.rdsdata.model.StatementTimeoutException;
import software.amazon.awssdk.services.rdsdata.model.TransactionNotFoundException;
import software.amazon.awssdk.services.rdsdata.model.UnsupportedResultException;
import software.amazon.awssdk.services.rdsdata.transform.BatchExecuteStatementRequestMarshaller;
import software.amazon.awssdk.services.rdsdata.transform.BeginTransactionRequestMarshaller;
import software.amazon.awssdk.services.rdsdata.transform.CommitTransactionRequestMarshaller;
import software.amazon.awssdk.services.rdsdata.transform.ExecuteSqlRequestMarshaller;
import software.amazon.awssdk.services.rdsdata.transform.ExecuteStatementRequestMarshaller;
import software.amazon.awssdk.services.rdsdata.transform.RollbackTransactionRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link RdsDataClient}.
 *
 * @see RdsDataClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRdsDataClient implements RdsDataClient {
    private static final Logger log = Logger.loggerFor(DefaultRdsDataClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRdsDataClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Runs a batch SQL statement over an array of data.
     * </p>
     * <p>
     * You can run bulk update and insert operations for multiple records using a DML statement with different parameter
     * sets. Bulk operations can provide a significant performance improvement over individual insert and update
     * operations.
     * </p>
     * <note>
     * <p>
     * If a call isn't part of a transaction because it doesn't include the <code>transactionID</code> parameter,
     * changes that result from the call are committed automatically.
     * </p>
     * <p>
     * There isn't a fixed upper limit on the number of parameter sets. However, the maximum size of the HTTP request
     * submitted through the Data API is 4 MiB. If the request exceeds this limit, the Data API returns an error and
     * doesn't process the request. This 4-MiB limit includes the size of the HTTP headers and the JSON notation in the
     * request. Thus, the number of parameter sets that you can include depends on a combination of factors, such as the
     * size of the SQL statement and the size of each parameter set.
     * </p>
     * <p>
     * The response size limit is 1 MiB. If the call returns more than 1 MiB of response data, the call is terminated.
     * </p>
     * </note>
     *
     * @param batchExecuteStatementRequest
     *        The request parameters represent the input of a SQL statement over an array of data.
     * @return Result of the BatchExecuteStatement operation returned by the service.
     * @throws SecretsErrorException
     *         There was a problem with the Secrets Manager secret used with the request, caused by one of the following
     *         conditions:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         RDS Data API timed out retrieving the secret.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The secret provided wasn't found.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The secret couldn't be decrypted.
     *         </p>
     *         </li>
     * @throws HttpEndpointNotEnabledException
     *         The HTTP endpoint for using RDS Data API isn't enabled for the DB cluster.
     * @throws DatabaseErrorException
     *         There was an error in processing the SQL statement.
     * @throws DatabaseResumingException
     *         A request was canceled because the Aurora Serverless v2 DB instance was in a paused state. The Data API
     *         request automatically causes the DB instance to begin resuming. Wait a few seconds and try again.
     * @throws DatabaseUnavailableException
     *         The writer instance in the DB cluster isn't available.
     * @throws TransactionNotFoundException
     *         The transaction ID wasn't found.
     * @throws InvalidSecretException
     *         The Secrets Manager secret used with the request isn't valid.
     * @throws ServiceUnavailableErrorException
     *         The service specified by the <code>resourceArn</code> parameter isn't available.
     * @throws ForbiddenException
     *         There are insufficient privileges to make the call.
     * @throws DatabaseNotFoundException
     *         The DB cluster doesn't have a DB instance.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws BadRequestException
     *         There is an error in the call or in a SQL statement. (This error only appears in calls from Aurora
     *         Serverless v1 databases.)
     * @throws StatementTimeoutException
     *         The execution of the SQL statement timed out.
     * @throws InternalServerErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RdsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RdsDataClient.BatchExecuteStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/BatchExecuteStatement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchExecuteStatementResponse batchExecuteStatement(BatchExecuteStatementRequest batchExecuteStatementRequest)
            throws SecretsErrorException, HttpEndpointNotEnabledException, DatabaseErrorException, DatabaseResumingException,
            DatabaseUnavailableException, TransactionNotFoundException, InvalidSecretException, ServiceUnavailableErrorException,
            ForbiddenException, DatabaseNotFoundException, AccessDeniedException, BadRequestException, StatementTimeoutException,
            InternalServerErrorException, AwsServiceException, SdkClientException, RdsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchExecuteStatementResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchExecuteStatementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchExecuteStatementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchExecuteStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RDS Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchExecuteStatement");

            return clientHandler.execute(new ClientExecutionParams<BatchExecuteStatementRequest, BatchExecuteStatementResponse>()
                    .withOperationName("BatchExecuteStatement").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(batchExecuteStatementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchExecuteStatementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a SQL transaction.
     * </p>
     * <note>
     * <p>
     * A transaction can run for a maximum of 24 hours. A transaction is terminated and rolled back automatically after
     * 24 hours.
     * </p>
     * <p>
     * A transaction times out if no calls use its transaction ID in three minutes. If a transaction times out before
     * it's committed, it's rolled back automatically.
     * </p>
     * <p>
     * For Aurora MySQL, DDL statements inside a transaction cause an implicit commit. We recommend that you run each
     * MySQL DDL statement in a separate <code>ExecuteStatement</code> call with <code>continueAfterTimeout</code>
     * enabled.
     * </p>
     * </note>
     *
     * @param beginTransactionRequest
     *        The request parameters represent the input of a request to start a SQL transaction.
     * @return Result of the BeginTransaction operation returned by the service.
     * @throws SecretsErrorException
     *         There was a problem with the Secrets Manager secret used with the request, caused by one of the following
     *         conditions:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         RDS Data API timed out retrieving the secret.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The secret provided wasn't found.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The secret couldn't be decrypted.
     *         </p>
     *         </li>
     * @throws HttpEndpointNotEnabledException
     *         The HTTP endpoint for using RDS Data API isn't enabled for the DB cluster.
     * @throws DatabaseErrorException
     *         There was an error in processing the SQL statement.
     * @throws DatabaseResumingException
     *         A request was canceled because the Aurora Serverless v2 DB instance was in a paused state. The Data API
     *         request automatically causes the DB instance to begin resuming. Wait a few seconds and try again.
     * @throws DatabaseUnavailableException
     *         The writer instance in the DB cluster isn't available.
     * @throws TransactionNotFoundException
     *         The transaction ID wasn't found.
     * @throws InvalidSecretException
     *         The Secrets Manager secret used with the request isn't valid.
     * @throws ServiceUnavailableErrorException
     *         The service specified by the <code>resourceArn</code> parameter isn't available.
     * @throws ForbiddenException
     *         There are insufficient privileges to make the call.
     * @throws DatabaseNotFoundException
     *         The DB cluster doesn't have a DB instance.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws BadRequestException
     *         There is an error in the call or in a SQL statement. (This error only appears in calls from Aurora
     *         Serverless v1 databases.)
     * @throws StatementTimeoutException
     *         The execution of the SQL statement timed out.
     * @throws InternalServerErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RdsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RdsDataClient.BeginTransaction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/BeginTransaction" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BeginTransactionResponse beginTransaction(BeginTransactionRequest beginTransactionRequest)
            throws SecretsErrorException, HttpEndpointNotEnabledException, DatabaseErrorException, DatabaseResumingException,
            DatabaseUnavailableException, TransactionNotFoundException, InvalidSecretException, ServiceUnavailableErrorException,
            ForbiddenException, DatabaseNotFoundException, AccessDeniedException, BadRequestException, StatementTimeoutException,
            InternalServerErrorException, AwsServiceException, SdkClientException, RdsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BeginTransactionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BeginTransactionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(beginTransactionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, beginTransactionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RDS Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BeginTransaction");

            return clientHandler.execute(new ClientExecutionParams<BeginTransactionRequest, BeginTransactionResponse>()
                    .withOperationName("BeginTransaction").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(beginTransactionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BeginTransactionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Ends a SQL transaction started with the <code>BeginTransaction</code> operation and commits the changes.
     * </p>
     *
     * @param commitTransactionRequest
     *        The request parameters represent the input of a commit transaction request.
     * @return Result of the CommitTransaction operation returned by the service.
     * @throws SecretsErrorException
     *         There was a problem with the Secrets Manager secret used with the request, caused by one of the following
     *         conditions:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         RDS Data API timed out retrieving the secret.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The secret provided wasn't found.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The secret couldn't be decrypted.
     *         </p>
     *         </li>
     * @throws HttpEndpointNotEnabledException
     *         The HTTP endpoint for using RDS Data API isn't enabled for the DB cluster.
     * @throws DatabaseErrorException
     *         There was an error in processing the SQL statement.
     * @throws DatabaseUnavailableException
     *         The writer instance in the DB cluster isn't available.
     * @throws TransactionNotFoundException
     *         The transaction ID wasn't found.
     * @throws InvalidSecretException
     *         The Secrets Manager secret used with the request isn't valid.
     * @throws ServiceUnavailableErrorException
     *         The service specified by the <code>resourceArn</code> parameter isn't available.
     * @throws ForbiddenException
     *         There are insufficient privileges to make the call.
     * @throws DatabaseNotFoundException
     *         The DB cluster doesn't have a DB instance.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws BadRequestException
     *         There is an error in the call or in a SQL statement. (This error only appears in calls from Aurora
     *         Serverless v1 databases.)
     * @throws StatementTimeoutException
     *         The execution of the SQL statement timed out.
     * @throws InternalServerErrorException
     *         An internal error occurred.
     * @throws NotFoundException
     *         The <code>resourceArn</code>, <code>secretArn</code>, or <code>transactionId</code> value can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RdsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RdsDataClient.CommitTransaction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/CommitTransaction" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CommitTransactionResponse commitTransaction(CommitTransactionRequest commitTransactionRequest)
            throws SecretsErrorException, HttpEndpointNotEnabledException, DatabaseErrorException, DatabaseUnavailableException,
            TransactionNotFoundException, InvalidSecretException, ServiceUnavailableErrorException, ForbiddenException,
            DatabaseNotFoundException, AccessDeniedException, BadRequestException, StatementTimeoutException,
            InternalServerErrorException, NotFoundException, AwsServiceException, SdkClientException, RdsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CommitTransactionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CommitTransactionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(commitTransactionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, commitTransactionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RDS Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CommitTransaction");

            return clientHandler.execute(new ClientExecutionParams<CommitTransactionRequest, CommitTransactionResponse>()
                    .withOperationName("CommitTransaction").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(commitTransactionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CommitTransactionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Runs one or more SQL statements.
     * </p>
     * <note>
     * <p>
     * This operation isn't supported for Aurora Serverless v2 and provisioned DB clusters. For Aurora Serverless v1 DB
     * clusters, the operation is deprecated. Use the <code>BatchExecuteStatement</code> or
     * <code>ExecuteStatement</code> operation.
     * </p>
     * </note>
     *
     * @param executeSqlRequest
     *        The request parameters represent the input of a request to run one or more SQL statements.
     * @return Result of the ExecuteSql operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws BadRequestException
     *         There is an error in the call or in a SQL statement. (This error only appears in calls from Aurora
     *         Serverless v1 databases.)
     * @throws InternalServerErrorException
     *         An internal error occurred.
     * @throws ForbiddenException
     *         There are insufficient privileges to make the call.
     * @throws ServiceUnavailableErrorException
     *         The service specified by the <code>resourceArn</code> parameter isn't available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RdsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RdsDataClient.ExecuteSql
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/ExecuteSql" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ExecuteSqlResponse executeSql(ExecuteSqlRequest executeSqlRequest) throws AccessDeniedException, BadRequestException,
            InternalServerErrorException, ForbiddenException, ServiceUnavailableErrorException, AwsServiceException,
            SdkClientException, RdsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExecuteSqlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ExecuteSqlResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(executeSqlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeSqlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RDS Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteSql");

            return clientHandler.execute(new ClientExecutionParams<ExecuteSqlRequest, ExecuteSqlResponse>()
                    .withOperationName("ExecuteSql").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(executeSqlRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExecuteSqlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Runs a SQL statement against a database.
     * </p>
     * <note>
     * <p>
     * If a call isn't part of a transaction because it doesn't include the <code>transactionID</code> parameter,
     * changes that result from the call are committed automatically.
     * </p>
     * <p>
     * If the binary response data from the database is more than 1 MB, the call is terminated.
     * </p>
     * </note>
     *
     * @param executeStatementRequest
     *        The request parameters represent the input of a request to run a SQL statement against a database.
     * @return Result of the ExecuteStatement operation returned by the service.
     * @throws SecretsErrorException
     *         There was a problem with the Secrets Manager secret used with the request, caused by one of the following
     *         conditions:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         RDS Data API timed out retrieving the secret.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The secret provided wasn't found.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The secret couldn't be decrypted.
     *         </p>
     *         </li>
     * @throws HttpEndpointNotEnabledException
     *         The HTTP endpoint for using RDS Data API isn't enabled for the DB cluster.
     * @throws DatabaseErrorException
     *         There was an error in processing the SQL statement.
     * @throws DatabaseResumingException
     *         A request was canceled because the Aurora Serverless v2 DB instance was in a paused state. The Data API
     *         request automatically causes the DB instance to begin resuming. Wait a few seconds and try again.
     * @throws DatabaseUnavailableException
     *         The writer instance in the DB cluster isn't available.
     * @throws TransactionNotFoundException
     *         The transaction ID wasn't found.
     * @throws InvalidSecretException
     *         The Secrets Manager secret used with the request isn't valid.
     * @throws ServiceUnavailableErrorException
     *         The service specified by the <code>resourceArn</code> parameter isn't available.
     * @throws ForbiddenException
     *         There are insufficient privileges to make the call.
     * @throws DatabaseNotFoundException
     *         The DB cluster doesn't have a DB instance.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws BadRequestException
     *         There is an error in the call or in a SQL statement. (This error only appears in calls from Aurora
     *         Serverless v1 databases.)
     * @throws StatementTimeoutException
     *         The execution of the SQL statement timed out.
     * @throws InternalServerErrorException
     *         An internal error occurred.
     * @throws UnsupportedResultException
     *         There was a problem with the result because of one of the following conditions:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         It contained an unsupported data type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         It contained a multidimensional array.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The size was too large.
     *         </p>
     *         </li>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RdsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RdsDataClient.ExecuteStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/ExecuteStatement" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ExecuteStatementResponse executeStatement(ExecuteStatementRequest executeStatementRequest)
            throws SecretsErrorException, HttpEndpointNotEnabledException, DatabaseErrorException, DatabaseResumingException,
            DatabaseUnavailableException, TransactionNotFoundException, InvalidSecretException, ServiceUnavailableErrorException,
            ForbiddenException, DatabaseNotFoundException, AccessDeniedException, BadRequestException, StatementTimeoutException,
            InternalServerErrorException, UnsupportedResultException, AwsServiceException, SdkClientException, RdsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExecuteStatementResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ExecuteStatementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(executeStatementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RDS Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteStatement");

            return clientHandler.execute(new ClientExecutionParams<ExecuteStatementRequest, ExecuteStatementResponse>()
                    .withOperationName("ExecuteStatement").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(executeStatementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExecuteStatementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Performs a rollback of a transaction. Rolling back a transaction cancels its changes.
     * </p>
     *
     * @param rollbackTransactionRequest
     *        The request parameters represent the input of a request to perform a rollback of a transaction.
     * @return Result of the RollbackTransaction operation returned by the service.
     * @throws SecretsErrorException
     *         There was a problem with the Secrets Manager secret used with the request, caused by one of the following
     *         conditions:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         RDS Data API timed out retrieving the secret.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The secret provided wasn't found.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The secret couldn't be decrypted.
     *         </p>
     *         </li>
     * @throws HttpEndpointNotEnabledException
     *         The HTTP endpoint for using RDS Data API isn't enabled for the DB cluster.
     * @throws DatabaseErrorException
     *         There was an error in processing the SQL statement.
     * @throws DatabaseUnavailableException
     *         The writer instance in the DB cluster isn't available.
     * @throws TransactionNotFoundException
     *         The transaction ID wasn't found.
     * @throws InvalidSecretException
     *         The Secrets Manager secret used with the request isn't valid.
     * @throws ServiceUnavailableErrorException
     *         The service specified by the <code>resourceArn</code> parameter isn't available.
     * @throws ForbiddenException
     *         There are insufficient privileges to make the call.
     * @throws DatabaseNotFoundException
     *         The DB cluster doesn't have a DB instance.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws BadRequestException
     *         There is an error in the call or in a SQL statement. (This error only appears in calls from Aurora
     *         Serverless v1 databases.)
     * @throws StatementTimeoutException
     *         The execution of the SQL statement timed out.
     * @throws InternalServerErrorException
     *         An internal error occurred.
     * @throws NotFoundException
     *         The <code>resourceArn</code>, <code>secretArn</code>, or <code>transactionId</code> value can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RdsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RdsDataClient.RollbackTransaction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rds-data-2018-08-01/RollbackTransaction" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RollbackTransactionResponse rollbackTransaction(RollbackTransactionRequest rollbackTransactionRequest)
            throws SecretsErrorException, HttpEndpointNotEnabledException, DatabaseErrorException, DatabaseUnavailableException,
            TransactionNotFoundException, InvalidSecretException, ServiceUnavailableErrorException, ForbiddenException,
            DatabaseNotFoundException, AccessDeniedException, BadRequestException, StatementTimeoutException,
            InternalServerErrorException, NotFoundException, AwsServiceException, SdkClientException, RdsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RollbackTransactionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RollbackTransactionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rollbackTransactionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rollbackTransactionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RDS Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RollbackTransaction");

            return clientHandler.execute(new ClientExecutionParams<RollbackTransactionRequest, RollbackTransactionResponse>()
                    .withOperationName("RollbackTransaction").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(rollbackTransactionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RollbackTransactionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        RdsDataServiceClientConfigurationBuilder serviceConfigBuilder = new RdsDataServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RdsDataException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TransactionNotFoundException")
                                .exceptionBuilderSupplier(TransactionNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DatabaseResumingException")
                                .exceptionBuilderSupplier(DatabaseResumingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSecretException")
                                .exceptionBuilderSupplier(InvalidSecretException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DatabaseUnavailableException")
                                .exceptionBuilderSupplier(DatabaseUnavailableException::builder).httpStatusCode(504).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DatabaseNotFoundException")
                                .exceptionBuilderSupplier(DatabaseNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DatabaseErrorException")
                                .exceptionBuilderSupplier(DatabaseErrorException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableError")
                                .exceptionBuilderSupplier(ServiceUnavailableErrorException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StatementTimeoutException")
                                .exceptionBuilderSupplier(StatementTimeoutException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedResultException")
                                .exceptionBuilderSupplier(UnsupportedResultException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("HttpEndpointNotEnabledException")
                                .exceptionBuilderSupplier(HttpEndpointNotEnabledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SecretsErrorException")
                                .exceptionBuilderSupplier(SecretsErrorException::builder).httpStatusCode(400).build());
    }

    @Override
    public final RdsDataServiceClientConfiguration serviceClientConfiguration() {
        return new RdsDataServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
