/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshiftdata;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.redshiftdata.model.ActiveStatementsExceededException;
import software.amazon.awssdk.services.redshiftdata.model.BatchExecuteStatementException;
import software.amazon.awssdk.services.redshiftdata.model.BatchExecuteStatementRequest;
import software.amazon.awssdk.services.redshiftdata.model.BatchExecuteStatementResponse;
import software.amazon.awssdk.services.redshiftdata.model.CancelStatementRequest;
import software.amazon.awssdk.services.redshiftdata.model.CancelStatementResponse;
import software.amazon.awssdk.services.redshiftdata.model.DescribeStatementRequest;
import software.amazon.awssdk.services.redshiftdata.model.DescribeStatementResponse;
import software.amazon.awssdk.services.redshiftdata.model.DescribeTableRequest;
import software.amazon.awssdk.services.redshiftdata.model.DescribeTableResponse;
import software.amazon.awssdk.services.redshiftdata.model.ExecuteStatementException;
import software.amazon.awssdk.services.redshiftdata.model.ExecuteStatementRequest;
import software.amazon.awssdk.services.redshiftdata.model.ExecuteStatementResponse;
import software.amazon.awssdk.services.redshiftdata.model.GetStatementResultRequest;
import software.amazon.awssdk.services.redshiftdata.model.GetStatementResultResponse;
import software.amazon.awssdk.services.redshiftdata.model.InternalServerException;
import software.amazon.awssdk.services.redshiftdata.model.ListDatabasesRequest;
import software.amazon.awssdk.services.redshiftdata.model.ListDatabasesResponse;
import software.amazon.awssdk.services.redshiftdata.model.ListSchemasRequest;
import software.amazon.awssdk.services.redshiftdata.model.ListSchemasResponse;
import software.amazon.awssdk.services.redshiftdata.model.ListStatementsRequest;
import software.amazon.awssdk.services.redshiftdata.model.ListStatementsResponse;
import software.amazon.awssdk.services.redshiftdata.model.ListTablesRequest;
import software.amazon.awssdk.services.redshiftdata.model.ListTablesResponse;
import software.amazon.awssdk.services.redshiftdata.model.RedshiftDataException;
import software.amazon.awssdk.services.redshiftdata.model.RedshiftDataRequest;
import software.amazon.awssdk.services.redshiftdata.model.ResourceNotFoundException;
import software.amazon.awssdk.services.redshiftdata.model.ValidationException;
import software.amazon.awssdk.services.redshiftdata.paginators.DescribeTablePublisher;
import software.amazon.awssdk.services.redshiftdata.paginators.GetStatementResultPublisher;
import software.amazon.awssdk.services.redshiftdata.paginators.ListDatabasesPublisher;
import software.amazon.awssdk.services.redshiftdata.paginators.ListSchemasPublisher;
import software.amazon.awssdk.services.redshiftdata.paginators.ListStatementsPublisher;
import software.amazon.awssdk.services.redshiftdata.paginators.ListTablesPublisher;
import software.amazon.awssdk.services.redshiftdata.transform.BatchExecuteStatementRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.CancelStatementRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.DescribeStatementRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.DescribeTableRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.ExecuteStatementRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.GetStatementResultRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.ListDatabasesRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.ListSchemasRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.ListStatementsRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.ListTablesRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link RedshiftDataAsyncClient}.
 *
 * @see RedshiftDataAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRedshiftDataAsyncClient implements RedshiftDataAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRedshiftDataAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRedshiftDataAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Runs one or more SQL statements, which can be data manipulation language (DML) or data definition language (DDL).
     * Depending on the authorization method, use one of the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - specify the Amazon Resource Name (ARN) of the secret, the database name, and the cluster
     * identifier that matches the cluster in the secret.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - specify the cluster identifier, the database name, and the database user name. Permission
     * to call the <code>redshift:GetClusterCredentials</code> operation is required to use this method.
     * </p>
     * </li>
     * </ul>
     *
     * @param batchExecuteStatementRequest
     * @return A Java Future containing the result of the BatchExecuteStatement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>ActiveStatementsExceededException The number of active statements exceeds the limit.</li>
     *         <li>BatchExecuteStatementException An SQL statement encountered an environmental error while running.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.BatchExecuteStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/BatchExecuteStatement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchExecuteStatementResponse> batchExecuteStatement(
            BatchExecuteStatementRequest batchExecuteStatementRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchExecuteStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchExecuteStatement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchExecuteStatementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchExecuteStatementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchExecuteStatementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchExecuteStatementRequest, BatchExecuteStatementResponse>()
                            .withOperationName("BatchExecuteStatement")
                            .withMarshaller(new BatchExecuteStatementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchExecuteStatementRequest));
            CompletableFuture<BatchExecuteStatementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels a running query. To be canceled, a query must be running.
     * </p>
     *
     * @param cancelStatementRequest
     * @return A Java Future containing the result of the CancelStatement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>ResourceNotFoundException The Amazon Redshift Data API operation failed due to a missing resource.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.CancelStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/CancelStatement" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelStatementResponse> cancelStatement(CancelStatementRequest cancelStatementRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelStatement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelStatementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelStatementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelStatementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelStatementRequest, CancelStatementResponse>()
                            .withOperationName("CancelStatement")
                            .withMarshaller(new CancelStatementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelStatementRequest));
            CompletableFuture<CancelStatementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the details about a specific instance when a query was run by the Amazon Redshift Data API. The
     * information includes when the query started, when it finished, the query status, the number of rows returned, and
     * the SQL statement.
     * </p>
     *
     * @param describeStatementRequest
     * @return A Java Future containing the result of the DescribeStatement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>ResourceNotFoundException The Amazon Redshift Data API operation failed due to a missing resource.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.DescribeStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/DescribeStatement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeStatementResponse> describeStatement(DescribeStatementRequest describeStatementRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStatement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeStatementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeStatementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeStatementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeStatementRequest, DescribeStatementResponse>()
                            .withOperationName("DescribeStatement")
                            .withMarshaller(new DescribeStatementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeStatementRequest));
            CompletableFuture<DescribeStatementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the detailed information about a table from metadata in the cluster. The information includes its
     * columns. A token is returned to page through the column list. Depending on the authorization method, use one of
     * the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - specify the Amazon Resource Name (ARN) of the secret, the database name, and the cluster
     * identifier that matches the cluster in the secret.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - specify the cluster identifier, the database name, and the database user name. Permission
     * to call the <code>redshift:GetClusterCredentials</code> operation is required to use this method.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeTableRequest
     * @return A Java Future containing the result of the DescribeTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.DescribeTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/DescribeTable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTableResponse> describeTable(DescribeTableRequest describeTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTableRequest, DescribeTableResponse>()
                            .withOperationName("DescribeTable")
                            .withMarshaller(new DescribeTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeTableRequest));
            CompletableFuture<DescribeTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the detailed information about a table from metadata in the cluster. The information includes its
     * columns. A token is returned to page through the column list. Depending on the authorization method, use one of
     * the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - specify the Amazon Resource Name (ARN) of the secret, the database name, and the cluster
     * identifier that matches the cluster in the secret.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - specify the cluster identifier, the database name, and the database user name. Permission
     * to call the <code>redshift:GetClusterCredentials</code> operation is required to use this method.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeTable(software.amazon.awssdk.services.redshiftdata.model.DescribeTableRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.redshiftdata.paginators.DescribeTablePublisher publisher = client.describeTablePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.redshiftdata.paginators.DescribeTablePublisher publisher = client.describeTablePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.redshiftdata.model.DescribeTableResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.redshiftdata.model.DescribeTableResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeTable(software.amazon.awssdk.services.redshiftdata.model.DescribeTableRequest)} operation.</b>
     * </p>
     *
     * @param describeTableRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.DescribeTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/DescribeTable" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeTablePublisher describeTablePaginator(DescribeTableRequest describeTableRequest) {
        return new DescribeTablePublisher(this, applyPaginatorUserAgent(describeTableRequest));
    }

    /**
     * <p>
     * Runs an SQL statement, which can be data manipulation language (DML) or data definition language (DDL). This
     * statement must be a single SQL statement. Depending on the authorization method, use one of the following
     * combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - specify the Amazon Resource Name (ARN) of the secret, the database name, and the cluster
     * identifier that matches the cluster in the secret.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - specify the cluster identifier, the database name, and the database user name. Permission
     * to call the <code>redshift:GetClusterCredentials</code> operation is required to use this method.
     * </p>
     * </li>
     * </ul>
     *
     * @param executeStatementRequest
     * @return A Java Future containing the result of the ExecuteStatement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>ExecuteStatementException The SQL statement encountered an environmental error while running.</li>
     *         <li>ActiveStatementsExceededException The number of active statements exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.ExecuteStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ExecuteStatement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExecuteStatementResponse> executeStatement(ExecuteStatementRequest executeStatementRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteStatement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExecuteStatementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExecuteStatementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExecuteStatementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExecuteStatementRequest, ExecuteStatementResponse>()
                            .withOperationName("ExecuteStatement")
                            .withMarshaller(new ExecuteStatementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(executeStatementRequest));
            CompletableFuture<ExecuteStatementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Fetches the temporarily cached result of an SQL statement. A token is returned to page through the statement
     * results.
     * </p>
     *
     * @param getStatementResultRequest
     * @return A Java Future containing the result of the GetStatementResult operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>ResourceNotFoundException The Amazon Redshift Data API operation failed due to a missing resource.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.GetStatementResult
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/GetStatementResult"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetStatementResultResponse> getStatementResult(GetStatementResultRequest getStatementResultRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStatementResultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStatementResult");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetStatementResultResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetStatementResultResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetStatementResultResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetStatementResultRequest, GetStatementResultResponse>()
                            .withOperationName("GetStatementResult")
                            .withMarshaller(new GetStatementResultRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getStatementResultRequest));
            CompletableFuture<GetStatementResultResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Fetches the temporarily cached result of an SQL statement. A token is returned to page through the statement
     * results.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getStatementResult(software.amazon.awssdk.services.redshiftdata.model.GetStatementResultRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.redshiftdata.paginators.GetStatementResultPublisher publisher = client.getStatementResultPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.redshiftdata.paginators.GetStatementResultPublisher publisher = client.getStatementResultPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.redshiftdata.model.GetStatementResultResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.redshiftdata.model.GetStatementResultResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getStatementResult(software.amazon.awssdk.services.redshiftdata.model.GetStatementResultRequest)}
     * operation.</b>
     * </p>
     *
     * @param getStatementResultRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>ResourceNotFoundException The Amazon Redshift Data API operation failed due to a missing resource.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.GetStatementResult
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/GetStatementResult"
     *      target="_top">AWS API Documentation</a>
     */
    public GetStatementResultPublisher getStatementResultPaginator(GetStatementResultRequest getStatementResultRequest) {
        return new GetStatementResultPublisher(this, applyPaginatorUserAgent(getStatementResultRequest));
    }

    /**
     * <p>
     * List the databases in a cluster. A token is returned to page through the database list. Depending on the
     * authorization method, use one of the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - specify the Amazon Resource Name (ARN) of the secret, the database name, and the cluster
     * identifier that matches the cluster in the secret.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - specify the cluster identifier, the database name, and the database user name. Permission
     * to call the <code>redshift:GetClusterCredentials</code> operation is required to use this method.
     * </p>
     * </li>
     * </ul>
     *
     * @param listDatabasesRequest
     * @return A Java Future containing the result of the ListDatabases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.ListDatabases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListDatabases" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatabasesResponse> listDatabases(ListDatabasesRequest listDatabasesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatabasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatabases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatabasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDatabasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatabasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatabasesRequest, ListDatabasesResponse>()
                            .withOperationName("ListDatabases")
                            .withMarshaller(new ListDatabasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDatabasesRequest));
            CompletableFuture<ListDatabasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the databases in a cluster. A token is returned to page through the database list. Depending on the
     * authorization method, use one of the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - specify the Amazon Resource Name (ARN) of the secret, the database name, and the cluster
     * identifier that matches the cluster in the secret.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - specify the cluster identifier, the database name, and the database user name. Permission
     * to call the <code>redshift:GetClusterCredentials</code> operation is required to use this method.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDatabases(software.amazon.awssdk.services.redshiftdata.model.ListDatabasesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.redshiftdata.paginators.ListDatabasesPublisher publisher = client.listDatabasesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.redshiftdata.paginators.ListDatabasesPublisher publisher = client.listDatabasesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.redshiftdata.model.ListDatabasesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.redshiftdata.model.ListDatabasesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatabases(software.amazon.awssdk.services.redshiftdata.model.ListDatabasesRequest)} operation.</b>
     * </p>
     *
     * @param listDatabasesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.ListDatabases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListDatabases" target="_top">AWS
     *      API Documentation</a>
     */
    public ListDatabasesPublisher listDatabasesPaginator(ListDatabasesRequest listDatabasesRequest) {
        return new ListDatabasesPublisher(this, applyPaginatorUserAgent(listDatabasesRequest));
    }

    /**
     * <p>
     * Lists the schemas in a database. A token is returned to page through the schema list. Depending on the
     * authorization method, use one of the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - specify the Amazon Resource Name (ARN) of the secret, the database name, and the cluster
     * identifier that matches the cluster in the secret.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - specify the cluster identifier, the database name, and the database user name. Permission
     * to call the <code>redshift:GetClusterCredentials</code> operation is required to use this method.
     * </p>
     * </li>
     * </ul>
     *
     * @param listSchemasRequest
     * @return A Java Future containing the result of the ListSchemas operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.ListSchemas
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListSchemas" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSchemasResponse> listSchemas(ListSchemasRequest listSchemasRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSchemasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSchemas");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSchemasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSchemasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSchemasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSchemasRequest, ListSchemasResponse>()
                            .withOperationName("ListSchemas").withMarshaller(new ListSchemasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSchemasRequest));
            CompletableFuture<ListSchemasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the schemas in a database. A token is returned to page through the schema list. Depending on the
     * authorization method, use one of the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - specify the Amazon Resource Name (ARN) of the secret, the database name, and the cluster
     * identifier that matches the cluster in the secret.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - specify the cluster identifier, the database name, and the database user name. Permission
     * to call the <code>redshift:GetClusterCredentials</code> operation is required to use this method.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of {@link #listSchemas(software.amazon.awssdk.services.redshiftdata.model.ListSchemasRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.redshiftdata.paginators.ListSchemasPublisher publisher = client.listSchemasPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.redshiftdata.paginators.ListSchemasPublisher publisher = client.listSchemasPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.redshiftdata.model.ListSchemasResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.redshiftdata.model.ListSchemasResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSchemas(software.amazon.awssdk.services.redshiftdata.model.ListSchemasRequest)} operation.</b>
     * </p>
     *
     * @param listSchemasRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.ListSchemas
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListSchemas" target="_top">AWS API
     *      Documentation</a>
     */
    public ListSchemasPublisher listSchemasPaginator(ListSchemasRequest listSchemasRequest) {
        return new ListSchemasPublisher(this, applyPaginatorUserAgent(listSchemasRequest));
    }

    /**
     * <p>
     * List of SQL statements. By default, only finished statements are shown. A token is returned to page through the
     * statement list.
     * </p>
     *
     * @param listStatementsRequest
     * @return A Java Future containing the result of the ListStatements operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.ListStatements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListStatements" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListStatementsResponse> listStatements(ListStatementsRequest listStatementsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStatementsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStatements");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStatementsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListStatementsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListStatementsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStatementsRequest, ListStatementsResponse>()
                            .withOperationName("ListStatements")
                            .withMarshaller(new ListStatementsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listStatementsRequest));
            CompletableFuture<ListStatementsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List of SQL statements. By default, only finished statements are shown. A token is returned to page through the
     * statement list.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listStatements(software.amazon.awssdk.services.redshiftdata.model.ListStatementsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.redshiftdata.paginators.ListStatementsPublisher publisher = client.listStatementsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.redshiftdata.paginators.ListStatementsPublisher publisher = client.listStatementsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.redshiftdata.model.ListStatementsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.redshiftdata.model.ListStatementsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStatements(software.amazon.awssdk.services.redshiftdata.model.ListStatementsRequest)} operation.</b>
     * </p>
     *
     * @param listStatementsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.ListStatements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListStatements" target="_top">AWS
     *      API Documentation</a>
     */
    public ListStatementsPublisher listStatementsPaginator(ListStatementsRequest listStatementsRequest) {
        return new ListStatementsPublisher(this, applyPaginatorUserAgent(listStatementsRequest));
    }

    /**
     * <p>
     * List the tables in a database. If neither <code>SchemaPattern</code> nor <code>TablePattern</code> are specified,
     * then all tables in the database are returned. A token is returned to page through the table list. Depending on
     * the authorization method, use one of the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - specify the Amazon Resource Name (ARN) of the secret, the database name, and the cluster
     * identifier that matches the cluster in the secret.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - specify the cluster identifier, the database name, and the database user name. Permission
     * to call the <code>redshift:GetClusterCredentials</code> operation is required to use this method.
     * </p>
     * </li>
     * </ul>
     *
     * @param listTablesRequest
     * @return A Java Future containing the result of the ListTables operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTablesResponse> listTables(ListTablesRequest listTablesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTablesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTables");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTablesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTablesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTablesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTablesRequest, ListTablesResponse>().withOperationName("ListTables")
                            .withMarshaller(new ListTablesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTablesRequest));
            CompletableFuture<ListTablesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the tables in a database. If neither <code>SchemaPattern</code> nor <code>TablePattern</code> are specified,
     * then all tables in the database are returned. A token is returned to page through the table list. Depending on
     * the authorization method, use one of the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - specify the Amazon Resource Name (ARN) of the secret, the database name, and the cluster
     * identifier that matches the cluster in the secret.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - specify the cluster identifier, the database name, and the database user name. Permission
     * to call the <code>redshift:GetClusterCredentials</code> operation is required to use this method.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of {@link #listTables(software.amazon.awssdk.services.redshiftdata.model.ListTablesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.redshiftdata.paginators.ListTablesPublisher publisher = client.listTablesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.redshiftdata.paginators.ListTablesPublisher publisher = client.listTablesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.redshiftdata.model.ListTablesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.redshiftdata.model.ListTablesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTables(software.amazon.awssdk.services.redshiftdata.model.ListTablesRequest)} operation.</b>
     * </p>
     *
     * @param listTablesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    public ListTablesPublisher listTablesPaginator(ListTablesRequest listTablesRequest) {
        return new ListTablesPublisher(this, applyPaginatorUserAgent(listTablesRequest));
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RedshiftDataException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActiveStatementsExceededException")
                                .exceptionBuilderSupplier(ActiveStatementsExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExecuteStatementException")
                                .exceptionBuilderSupplier(ExecuteStatementException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BatchExecuteStatementException")
                                .exceptionBuilderSupplier(BatchExecuteStatementException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends RedshiftDataRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
