/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshiftdata;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.redshiftdata.internal.RedshiftDataServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.redshiftdata.model.ActiveSessionsExceededException;
import software.amazon.awssdk.services.redshiftdata.model.ActiveStatementsExceededException;
import software.amazon.awssdk.services.redshiftdata.model.BatchExecuteStatementException;
import software.amazon.awssdk.services.redshiftdata.model.BatchExecuteStatementRequest;
import software.amazon.awssdk.services.redshiftdata.model.BatchExecuteStatementResponse;
import software.amazon.awssdk.services.redshiftdata.model.CancelStatementRequest;
import software.amazon.awssdk.services.redshiftdata.model.CancelStatementResponse;
import software.amazon.awssdk.services.redshiftdata.model.DatabaseConnectionException;
import software.amazon.awssdk.services.redshiftdata.model.DescribeStatementRequest;
import software.amazon.awssdk.services.redshiftdata.model.DescribeStatementResponse;
import software.amazon.awssdk.services.redshiftdata.model.DescribeTableRequest;
import software.amazon.awssdk.services.redshiftdata.model.DescribeTableResponse;
import software.amazon.awssdk.services.redshiftdata.model.ExecuteStatementException;
import software.amazon.awssdk.services.redshiftdata.model.ExecuteStatementRequest;
import software.amazon.awssdk.services.redshiftdata.model.ExecuteStatementResponse;
import software.amazon.awssdk.services.redshiftdata.model.GetStatementResultRequest;
import software.amazon.awssdk.services.redshiftdata.model.GetStatementResultResponse;
import software.amazon.awssdk.services.redshiftdata.model.GetStatementResultV2Request;
import software.amazon.awssdk.services.redshiftdata.model.GetStatementResultV2Response;
import software.amazon.awssdk.services.redshiftdata.model.InternalServerException;
import software.amazon.awssdk.services.redshiftdata.model.ListDatabasesRequest;
import software.amazon.awssdk.services.redshiftdata.model.ListDatabasesResponse;
import software.amazon.awssdk.services.redshiftdata.model.ListSchemasRequest;
import software.amazon.awssdk.services.redshiftdata.model.ListSchemasResponse;
import software.amazon.awssdk.services.redshiftdata.model.ListStatementsRequest;
import software.amazon.awssdk.services.redshiftdata.model.ListStatementsResponse;
import software.amazon.awssdk.services.redshiftdata.model.ListTablesRequest;
import software.amazon.awssdk.services.redshiftdata.model.ListTablesResponse;
import software.amazon.awssdk.services.redshiftdata.model.QueryTimeoutException;
import software.amazon.awssdk.services.redshiftdata.model.RedshiftDataException;
import software.amazon.awssdk.services.redshiftdata.model.ResourceNotFoundException;
import software.amazon.awssdk.services.redshiftdata.model.ValidationException;
import software.amazon.awssdk.services.redshiftdata.transform.BatchExecuteStatementRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.CancelStatementRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.DescribeStatementRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.DescribeTableRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.ExecuteStatementRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.GetStatementResultRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.GetStatementResultV2RequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.ListDatabasesRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.ListSchemasRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.ListStatementsRequestMarshaller;
import software.amazon.awssdk.services.redshiftdata.transform.ListTablesRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link RedshiftDataAsyncClient}.
 *
 * @see RedshiftDataAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRedshiftDataAsyncClient implements RedshiftDataAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRedshiftDataAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRedshiftDataAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Runs one or more SQL statements, which can be data manipulation language (DML) or data definition language (DDL).
     * Depending on the authorization method, use one of the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - when connecting to a cluster, provide the <code>secret-arn</code> of a secret stored in Secrets
     * Manager which has <code>username</code> and <code>password</code>. The specified secret contains credentials to
     * connect to the <code>database</code> you specify. When you are connecting to a cluster, you also supply the
     * database name, If you provide a cluster identifier (<code>dbClusterIdentifier</code>), it must match the cluster
     * identifier stored in the secret. When you are connecting to a serverless workgroup, you also supply the database
     * name.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - when connecting to your data warehouse, choose one of the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When connecting to a serverless workgroup, specify the workgroup name and database name. The database user name
     * is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has the database user
     * name <code>IAM:foo</code>. Also, permission to call the <code>redshift-serverless:GetCredentials</code> operation
     * is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as an IAM identity, specify the cluster identifier and the database name. The
     * database user name is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has
     * the database user name <code>IAM:foo</code>. Also, permission to call the
     * <code>redshift:GetClusterCredentialsWithIAM</code> operation is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as a database user, specify the cluster identifier, the database name, and the
     * database user name. Also, permission to call the <code>redshift:GetClusterCredentials</code> operation is
     * required.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param batchExecuteStatementRequest
     * @return A Java Future containing the result of the BatchExecuteStatement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>ActiveSessionsExceededException The Amazon Redshift Data API operation failed because the maximum
     *         number of active sessions exceeded.</li>
     *         <li>ActiveStatementsExceededException The number of active statements exceeds the limit.</li>
     *         <li>BatchExecuteStatementException An SQL statement encountered an environmental error while running.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.BatchExecuteStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/BatchExecuteStatement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchExecuteStatementResponse> batchExecuteStatement(
            BatchExecuteStatementRequest batchExecuteStatementRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchExecuteStatementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchExecuteStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchExecuteStatement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchExecuteStatementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchExecuteStatementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchExecuteStatementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchExecuteStatementRequest, BatchExecuteStatementResponse>()
                            .withOperationName("BatchExecuteStatement").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchExecuteStatementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchExecuteStatementRequest));
            CompletableFuture<BatchExecuteStatementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels a running query. To be canceled, a query must be running.
     * </p>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param cancelStatementRequest
     * @return A Java Future containing the result of the CancelStatement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>ResourceNotFoundException The Amazon Redshift Data API operation failed due to a missing resource.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>DatabaseConnectionException Connection to a database failed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.CancelStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/CancelStatement" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelStatementResponse> cancelStatement(CancelStatementRequest cancelStatementRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelStatementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelStatement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelStatementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelStatementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelStatementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelStatementRequest, CancelStatementResponse>()
                            .withOperationName("CancelStatement").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelStatementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelStatementRequest));
            CompletableFuture<CancelStatementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the details about a specific instance when a query was run by the Amazon Redshift Data API. The
     * information includes when the query started, when it finished, the query status, the number of rows returned, and
     * the SQL statement.
     * </p>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param describeStatementRequest
     * @return A Java Future containing the result of the DescribeStatement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>ResourceNotFoundException The Amazon Redshift Data API operation failed due to a missing resource.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.DescribeStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/DescribeStatement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeStatementResponse> describeStatement(DescribeStatementRequest describeStatementRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeStatementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStatement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeStatementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeStatementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeStatementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeStatementRequest, DescribeStatementResponse>()
                            .withOperationName("DescribeStatement").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeStatementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeStatementRequest));
            CompletableFuture<DescribeStatementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the detailed information about a table from metadata in the cluster. The information includes its
     * columns. A token is returned to page through the column list. Depending on the authorization method, use one of
     * the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - when connecting to a cluster, provide the <code>secret-arn</code> of a secret stored in Secrets
     * Manager which has <code>username</code> and <code>password</code>. The specified secret contains credentials to
     * connect to the <code>database</code> you specify. When you are connecting to a cluster, you also supply the
     * database name, If you provide a cluster identifier (<code>dbClusterIdentifier</code>), it must match the cluster
     * identifier stored in the secret. When you are connecting to a serverless workgroup, you also supply the database
     * name.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - when connecting to your data warehouse, choose one of the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When connecting to a serverless workgroup, specify the workgroup name and database name. The database user name
     * is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has the database user
     * name <code>IAM:foo</code>. Also, permission to call the <code>redshift-serverless:GetCredentials</code> operation
     * is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as an IAM identity, specify the cluster identifier and the database name. The
     * database user name is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has
     * the database user name <code>IAM:foo</code>. Also, permission to call the
     * <code>redshift:GetClusterCredentialsWithIAM</code> operation is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as a database user, specify the cluster identifier, the database name, and the
     * database user name. Also, permission to call the <code>redshift:GetClusterCredentials</code> operation is
     * required.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param describeTableRequest
     * @return A Java Future containing the result of the DescribeTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>QueryTimeoutException The Amazon Redshift Data API operation failed due to timeout.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>DatabaseConnectionException Connection to a database failed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.DescribeTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/DescribeTable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTableResponse> describeTable(DescribeTableRequest describeTableRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTableRequest, DescribeTableResponse>()
                            .withOperationName("DescribeTable").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeTableRequest));
            CompletableFuture<DescribeTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Runs an SQL statement, which can be data manipulation language (DML) or data definition language (DDL). This
     * statement must be a single SQL statement. Depending on the authorization method, use one of the following
     * combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - when connecting to a cluster, provide the <code>secret-arn</code> of a secret stored in Secrets
     * Manager which has <code>username</code> and <code>password</code>. The specified secret contains credentials to
     * connect to the <code>database</code> you specify. When you are connecting to a cluster, you also supply the
     * database name, If you provide a cluster identifier (<code>dbClusterIdentifier</code>), it must match the cluster
     * identifier stored in the secret. When you are connecting to a serverless workgroup, you also supply the database
     * name.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - when connecting to your data warehouse, choose one of the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When connecting to a serverless workgroup, specify the workgroup name and database name. The database user name
     * is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has the database user
     * name <code>IAM:foo</code>. Also, permission to call the <code>redshift-serverless:GetCredentials</code> operation
     * is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as an IAM identity, specify the cluster identifier and the database name. The
     * database user name is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has
     * the database user name <code>IAM:foo</code>. Also, permission to call the
     * <code>redshift:GetClusterCredentialsWithIAM</code> operation is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as a database user, specify the cluster identifier, the database name, and the
     * database user name. Also, permission to call the <code>redshift:GetClusterCredentials</code> operation is
     * required.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param executeStatementRequest
     * @return A Java Future containing the result of the ExecuteStatement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>ActiveSessionsExceededException The Amazon Redshift Data API operation failed because the maximum
     *         number of active sessions exceeded.</li>
     *         <li>ExecuteStatementException The SQL statement encountered an environmental error while running.</li>
     *         <li>ActiveStatementsExceededException The number of active statements exceeds the limit.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.ExecuteStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ExecuteStatement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExecuteStatementResponse> executeStatement(ExecuteStatementRequest executeStatementRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(executeStatementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteStatement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExecuteStatementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExecuteStatementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExecuteStatementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExecuteStatementRequest, ExecuteStatementResponse>()
                            .withOperationName("ExecuteStatement").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ExecuteStatementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(executeStatementRequest));
            CompletableFuture<ExecuteStatementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Fetches the temporarily cached result of an SQL statement in JSON format. The <code>ExecuteStatement</code> or
     * <code>BatchExecuteStatement</code> operation that ran the SQL statement must have specified
     * <code>ResultFormat</code> as <code>JSON</code> , or let the format default to JSON. A token is returned to page
     * through the statement results.
     * </p>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param getStatementResultRequest
     * @return A Java Future containing the result of the GetStatementResult operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>ResourceNotFoundException The Amazon Redshift Data API operation failed due to a missing resource.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.GetStatementResult
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/GetStatementResult"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetStatementResultResponse> getStatementResult(GetStatementResultRequest getStatementResultRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStatementResultRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStatementResultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStatementResult");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetStatementResultResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetStatementResultResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetStatementResultResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetStatementResultRequest, GetStatementResultResponse>()
                            .withOperationName("GetStatementResult").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetStatementResultRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getStatementResultRequest));
            CompletableFuture<GetStatementResultResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Fetches the temporarily cached result of an SQL statement in CSV format. The <code>ExecuteStatement</code> or
     * <code>BatchExecuteStatement</code> operation that ran the SQL statement must have specified
     * <code>ResultFormat</code> as <code>CSV</code>. A token is returned to page through the statement results.
     * </p>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param getStatementResultV2Request
     * @return A Java Future containing the result of the GetStatementResultV2 operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>ResourceNotFoundException The Amazon Redshift Data API operation failed due to a missing resource.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.GetStatementResultV2
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/GetStatementResultV2"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetStatementResultV2Response> getStatementResultV2(
            GetStatementResultV2Request getStatementResultV2Request) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStatementResultV2Request,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStatementResultV2Request
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStatementResultV2");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetStatementResultV2Response> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetStatementResultV2Response::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetStatementResultV2Response> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetStatementResultV2Request, GetStatementResultV2Response>()
                            .withOperationName("GetStatementResultV2").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetStatementResultV2RequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getStatementResultV2Request));
            CompletableFuture<GetStatementResultV2Response> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the databases in a cluster. A token is returned to page through the database list. Depending on the
     * authorization method, use one of the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - when connecting to a cluster, provide the <code>secret-arn</code> of a secret stored in Secrets
     * Manager which has <code>username</code> and <code>password</code>. The specified secret contains credentials to
     * connect to the <code>database</code> you specify. When you are connecting to a cluster, you also supply the
     * database name, If you provide a cluster identifier (<code>dbClusterIdentifier</code>), it must match the cluster
     * identifier stored in the secret. When you are connecting to a serverless workgroup, you also supply the database
     * name.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - when connecting to your data warehouse, choose one of the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When connecting to a serverless workgroup, specify the workgroup name and database name. The database user name
     * is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has the database user
     * name <code>IAM:foo</code>. Also, permission to call the <code>redshift-serverless:GetCredentials</code> operation
     * is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as an IAM identity, specify the cluster identifier and the database name. The
     * database user name is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has
     * the database user name <code>IAM:foo</code>. Also, permission to call the
     * <code>redshift:GetClusterCredentialsWithIAM</code> operation is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as a database user, specify the cluster identifier, the database name, and the
     * database user name. Also, permission to call the <code>redshift:GetClusterCredentials</code> operation is
     * required.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param listDatabasesRequest
     * @return A Java Future containing the result of the ListDatabases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>QueryTimeoutException The Amazon Redshift Data API operation failed due to timeout.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>DatabaseConnectionException Connection to a database failed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.ListDatabases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListDatabases" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatabasesResponse> listDatabases(ListDatabasesRequest listDatabasesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDatabasesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatabasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatabases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatabasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDatabasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatabasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatabasesRequest, ListDatabasesResponse>()
                            .withOperationName("ListDatabases").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDatabasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDatabasesRequest));
            CompletableFuture<ListDatabasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the schemas in a database. A token is returned to page through the schema list. Depending on the
     * authorization method, use one of the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - when connecting to a cluster, provide the <code>secret-arn</code> of a secret stored in Secrets
     * Manager which has <code>username</code> and <code>password</code>. The specified secret contains credentials to
     * connect to the <code>database</code> you specify. When you are connecting to a cluster, you also supply the
     * database name, If you provide a cluster identifier (<code>dbClusterIdentifier</code>), it must match the cluster
     * identifier stored in the secret. When you are connecting to a serverless workgroup, you also supply the database
     * name.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - when connecting to your data warehouse, choose one of the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When connecting to a serverless workgroup, specify the workgroup name and database name. The database user name
     * is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has the database user
     * name <code>IAM:foo</code>. Also, permission to call the <code>redshift-serverless:GetCredentials</code> operation
     * is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as an IAM identity, specify the cluster identifier and the database name. The
     * database user name is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has
     * the database user name <code>IAM:foo</code>. Also, permission to call the
     * <code>redshift:GetClusterCredentialsWithIAM</code> operation is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as a database user, specify the cluster identifier, the database name, and the
     * database user name. Also, permission to call the <code>redshift:GetClusterCredentials</code> operation is
     * required.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param listSchemasRequest
     * @return A Java Future containing the result of the ListSchemas operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>QueryTimeoutException The Amazon Redshift Data API operation failed due to timeout.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>DatabaseConnectionException Connection to a database failed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.ListSchemas
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListSchemas" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSchemasResponse> listSchemas(ListSchemasRequest listSchemasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSchemasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSchemasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSchemas");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSchemasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSchemasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSchemasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSchemasRequest, ListSchemasResponse>()
                            .withOperationName("ListSchemas").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSchemasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSchemasRequest));
            CompletableFuture<ListSchemasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List of SQL statements. By default, only finished statements are shown. A token is returned to page through the
     * statement list.
     * </p>
     * <p>
     * When you use identity-enhanced role sessions to list statements, you must provide either the
     * <code>cluster-identifier</code> or <code>workgroup-name</code> parameter. This ensures that the IdC user can only
     * access the Amazon Redshift IdC applications they are assigned. For more information, see <a
     * href="https://docs.aws.amazon.com/singlesignon/latest/userguide/trustedidentitypropagation-overview.html">
     * Trusted identity propagation overview</a>.
     * </p>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param listStatementsRequest
     * @return A Java Future containing the result of the ListStatements operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.ListStatements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListStatements" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListStatementsResponse> listStatements(ListStatementsRequest listStatementsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStatementsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStatementsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStatements");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStatementsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListStatementsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListStatementsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStatementsRequest, ListStatementsResponse>()
                            .withOperationName("ListStatements").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListStatementsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listStatementsRequest));
            CompletableFuture<ListStatementsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the tables in a database. If neither <code>SchemaPattern</code> nor <code>TablePattern</code> are specified,
     * then all tables in the database are returned. A token is returned to page through the table list. Depending on
     * the authorization method, use one of the following combinations of request parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager - when connecting to a cluster, provide the <code>secret-arn</code> of a secret stored in Secrets
     * Manager which has <code>username</code> and <code>password</code>. The specified secret contains credentials to
     * connect to the <code>database</code> you specify. When you are connecting to a cluster, you also supply the
     * database name, If you provide a cluster identifier (<code>dbClusterIdentifier</code>), it must match the cluster
     * identifier stored in the secret. When you are connecting to a serverless workgroup, you also supply the database
     * name.
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary credentials - when connecting to your data warehouse, choose one of the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When connecting to a serverless workgroup, specify the workgroup name and database name. The database user name
     * is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has the database user
     * name <code>IAM:foo</code>. Also, permission to call the <code>redshift-serverless:GetCredentials</code> operation
     * is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as an IAM identity, specify the cluster identifier and the database name. The
     * database user name is derived from the IAM identity. For example, <code>arn:iam::123456789012:user:foo</code> has
     * the database user name <code>IAM:foo</code>. Also, permission to call the
     * <code>redshift:GetClusterCredentialsWithIAM</code> operation is required.
     * </p>
     * </li>
     * <li>
     * <p>
     * When connecting to a cluster as a database user, specify the cluster identifier, the database name, and the
     * database user name. Also, permission to call the <code>redshift:GetClusterCredentials</code> operation is
     * required.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * For more information about the Amazon Redshift Data API and CLI usage examples, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html">Using the Amazon Redshift Data API</a> in
     * the <i>Amazon Redshift Management Guide</i>.
     * </p>
     *
     * @param listTablesRequest
     * @return A Java Future containing the result of the ListTables operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>QueryTimeoutException The Amazon Redshift Data API operation failed due to timeout.</li>
     *         <li>InternalServerException The Amazon Redshift Data API operation failed due to invalid input.</li>
     *         <li>DatabaseConnectionException Connection to a database failed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RedshiftDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RedshiftDataAsyncClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/redshift-data-2019-12-20/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTablesResponse> listTables(ListTablesRequest listTablesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTablesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTablesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Redshift Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTables");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTablesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTablesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTablesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTablesRequest, ListTablesResponse>().withOperationName("ListTables")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTablesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTablesRequest));
            CompletableFuture<ListTablesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final RedshiftDataServiceClientConfiguration serviceClientConfiguration() {
        return new RedshiftDataServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RedshiftDataException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActiveStatementsExceededException")
                                .exceptionBuilderSupplier(ActiveStatementsExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("QueryTimeoutException")
                                .exceptionBuilderSupplier(QueryTimeoutException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DatabaseConnectionException")
                                .exceptionBuilderSupplier(DatabaseConnectionException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActiveSessionsExceededException")
                                .exceptionBuilderSupplier(ActiveSessionsExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExecuteStatementException")
                                .exceptionBuilderSupplier(ExecuteStatementException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BatchExecuteStatementException")
                                .exceptionBuilderSupplier(BatchExecuteStatementException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        RedshiftDataServiceClientConfigurationBuilder serviceConfigBuilder = new RedshiftDataServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
