/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition;

import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.config.AwsSyncClientConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocol;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolMetadata;
import software.amazon.awssdk.core.client.ClientExecutionParams;
import software.amazon.awssdk.core.client.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.protocol.json.JsonClientMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorResponseMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorShapeMetadata;
import software.amazon.awssdk.core.protocol.json.JsonOperationMetadata;
import software.amazon.awssdk.services.rekognition.model.AccessDeniedException;
import software.amazon.awssdk.services.rekognition.model.CompareFacesRequest;
import software.amazon.awssdk.services.rekognition.model.CompareFacesResponse;
import software.amazon.awssdk.services.rekognition.model.CreateCollectionRequest;
import software.amazon.awssdk.services.rekognition.model.CreateCollectionResponse;
import software.amazon.awssdk.services.rekognition.model.DeleteCollectionRequest;
import software.amazon.awssdk.services.rekognition.model.DeleteCollectionResponse;
import software.amazon.awssdk.services.rekognition.model.DeleteFacesRequest;
import software.amazon.awssdk.services.rekognition.model.DeleteFacesResponse;
import software.amazon.awssdk.services.rekognition.model.DetectFacesRequest;
import software.amazon.awssdk.services.rekognition.model.DetectFacesResponse;
import software.amazon.awssdk.services.rekognition.model.DetectLabelsRequest;
import software.amazon.awssdk.services.rekognition.model.DetectLabelsResponse;
import software.amazon.awssdk.services.rekognition.model.DetectModerationLabelsRequest;
import software.amazon.awssdk.services.rekognition.model.DetectModerationLabelsResponse;
import software.amazon.awssdk.services.rekognition.model.GetCelebrityInfoRequest;
import software.amazon.awssdk.services.rekognition.model.GetCelebrityInfoResponse;
import software.amazon.awssdk.services.rekognition.model.ImageTooLargeException;
import software.amazon.awssdk.services.rekognition.model.IndexFacesRequest;
import software.amazon.awssdk.services.rekognition.model.IndexFacesResponse;
import software.amazon.awssdk.services.rekognition.model.InternalServerErrorException;
import software.amazon.awssdk.services.rekognition.model.InvalidImageFormatException;
import software.amazon.awssdk.services.rekognition.model.InvalidPaginationTokenException;
import software.amazon.awssdk.services.rekognition.model.InvalidParameterException;
import software.amazon.awssdk.services.rekognition.model.InvalidS3ObjectException;
import software.amazon.awssdk.services.rekognition.model.ListCollectionsRequest;
import software.amazon.awssdk.services.rekognition.model.ListCollectionsResponse;
import software.amazon.awssdk.services.rekognition.model.ListFacesRequest;
import software.amazon.awssdk.services.rekognition.model.ListFacesResponse;
import software.amazon.awssdk.services.rekognition.model.ProvisionedThroughputExceededException;
import software.amazon.awssdk.services.rekognition.model.RecognizeCelebritiesRequest;
import software.amazon.awssdk.services.rekognition.model.RecognizeCelebritiesResponse;
import software.amazon.awssdk.services.rekognition.model.RekognitionException;
import software.amazon.awssdk.services.rekognition.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.rekognition.model.ResourceNotFoundException;
import software.amazon.awssdk.services.rekognition.model.SearchFacesByImageRequest;
import software.amazon.awssdk.services.rekognition.model.SearchFacesByImageResponse;
import software.amazon.awssdk.services.rekognition.model.SearchFacesRequest;
import software.amazon.awssdk.services.rekognition.model.SearchFacesResponse;
import software.amazon.awssdk.services.rekognition.model.ThrottlingException;
import software.amazon.awssdk.services.rekognition.paginators.ListCollectionsIterable;
import software.amazon.awssdk.services.rekognition.paginators.ListFacesIterable;
import software.amazon.awssdk.services.rekognition.transform.CompareFacesRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.CompareFacesResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.CreateCollectionRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.CreateCollectionResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.DeleteCollectionRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.DeleteCollectionResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.DeleteFacesRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.DeleteFacesResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.DetectFacesRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.DetectFacesResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.DetectLabelsRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.DetectLabelsResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.DetectModerationLabelsRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.DetectModerationLabelsResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetCelebrityInfoRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetCelebrityInfoResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.IndexFacesRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.IndexFacesResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.ListCollectionsRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.ListCollectionsResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.ListFacesRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.ListFacesResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.RecognizeCelebritiesRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.RecognizeCelebritiesResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.SearchFacesByImageRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.SearchFacesByImageResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.SearchFacesRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.SearchFacesResponseUnmarshaller;

/**
 * Internal implementation of {@link RekognitionClient}.
 *
 * @see RekognitionClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRekognitionClient implements RekognitionClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final AwsSyncClientConfiguration clientConfiguration;

    protected DefaultRekognitionClient(AwsSyncClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration, null);
        this.protocolFactory = init();
        this.clientConfiguration = clientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Compares a face in the <i>source</i> input image with each face detected in the <i>target</i> input image.
     * </p>
     * <note>
     * <p>
     * If the source image contains multiple faces, the service detects the largest face and compares it with each face
     * detected in the target image.
     * </p>
     * </note>
     * <p>
     * In response, the operation returns an array of face matches ordered by similarity score in descending order. For
     * each face match, the response provides a bounding box of the face, facial landmarks, pose details (pitch, role,
     * and yaw), quality (brightness and sharpness), and confidence value (indicating the level of confidence that the
     * bounding box contains a face). The response also provides a similarity score, which indicates how closely the
     * faces match.
     * </p>
     * <note>
     * <p>
     * By default, only faces with a similarity score of greater than or equal to 80% are returned in the response. You
     * can change this value by specifying the <code>SimilarityThreshold</code> parameter.
     * </p>
     * </note>
     * <p>
     * <code>CompareFaces</code> also returns an array of faces that don't match the source image. For each face, it
     * returns a bounding box, confidence value, landmarks, pose details, and quality. The response also returns
     * information about the face in the source image, including the bounding box of the face and confidence value.
     * </p>
     * <p>
     * If the image doesn't contain Exif metadata, <code>CompareFaces</code> returns orientation information for the
     * source and target images. Use these values to display the images with the correct image orientation.
     * </p>
     * <note>
     * <p>
     * This is a stateless API operation. That is, data returned by this operation doesn't persist.
     * </p>
     * </note>
     * <p>
     * For an example, see <a>get-started-exercise-compare-faces</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:CompareFaces</code> action.
     * </p>
     *
     * @param compareFacesRequest
     * @return Result of the CompareFaces operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws ImageTooLargeException
     *         The input image size exceeds the allowed limit. For more information, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.CompareFaces
     */
    @Override
    public CompareFacesResponse compareFaces(CompareFacesRequest compareFacesRequest) throws InvalidParameterException,
            InvalidS3ObjectException, ImageTooLargeException, AccessDeniedException, InternalServerErrorException,
            ThrottlingException, ProvisionedThroughputExceededException, InvalidImageFormatException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<CompareFacesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CompareFacesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CompareFacesRequest, CompareFacesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(compareFacesRequest).withMarshaller(new CompareFacesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a collection in an AWS Region. You can add faces to the collection using the operation.
     * </p>
     * <p>
     * For example, you might create collections, one for each of your application users. A user can then index faces
     * using the <code>IndexFaces</code> operation and persist results in a specific collection. Then, a user can search
     * the collection for faces in the user-specific container.
     * </p>
     * <note>
     * <p>
     * Collection names are case-sensitive.
     * </p>
     * </note>
     * <p>
     * For an example, see <a>example1</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:CreateCollection</code> action.
     * </p>
     *
     * @param createCollectionRequest
     * @return Result of the CreateCollection operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceAlreadyExistsException
     *         A collection with the specified ID already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.CreateCollection
     */
    @Override
    public CreateCollectionResponse createCollection(CreateCollectionRequest createCollectionRequest)
            throws InvalidParameterException, AccessDeniedException, InternalServerErrorException, ThrottlingException,
            ProvisionedThroughputExceededException, ResourceAlreadyExistsException, AwsServiceException, SdkClientException,
            RekognitionException {

        HttpResponseHandler<CreateCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateCollectionResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CreateCollectionRequest, CreateCollectionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(createCollectionRequest).withMarshaller(new CreateCollectionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the specified collection. Note that this operation removes all faces in the collection. For an example,
     * see <a>example1</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:DeleteCollection</code> action.
     * </p>
     *
     * @param deleteCollectionRequest
     * @return Result of the DeleteCollection operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         Collection specified in the request is not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.DeleteCollection
     */
    @Override
    public DeleteCollectionResponse deleteCollection(DeleteCollectionRequest deleteCollectionRequest)
            throws InvalidParameterException, AccessDeniedException, InternalServerErrorException, ThrottlingException,
            ProvisionedThroughputExceededException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            RekognitionException {

        HttpResponseHandler<DeleteCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DeleteCollectionResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DeleteCollectionRequest, DeleteCollectionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(deleteCollectionRequest).withMarshaller(new DeleteCollectionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes faces from a collection. You specify a collection ID and an array of face IDs to remove from the
     * collection.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:DeleteFaces</code> action.
     * </p>
     *
     * @param deleteFacesRequest
     * @return Result of the DeleteFaces operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         Collection specified in the request is not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.DeleteFaces
     */
    @Override
    public DeleteFacesResponse deleteFaces(DeleteFacesRequest deleteFacesRequest) throws InvalidParameterException,
            AccessDeniedException, InternalServerErrorException, ThrottlingException, ProvisionedThroughputExceededException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<DeleteFacesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DeleteFacesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DeleteFacesRequest, DeleteFacesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(deleteFacesRequest).withMarshaller(new DeleteFacesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Detects faces within an image (JPEG or PNG) that is provided as input.
     * </p>
     * <p>
     * For each face detected, the operation returns face details including a bounding box of the face, a confidence
     * value (that the bounding box contains a face), and a fixed set of attributes such as facial landmarks (for
     * example, coordinates of eye and mouth), gender, presence of beard, sunglasses, etc.
     * </p>
     * <p>
     * The face-detection algorithm is most effective on frontal faces. For non-frontal or obscured faces, the algorithm
     * may not detect the faces or might detect faces with lower confidence.
     * </p>
     * <note>
     * <p>
     * This is a stateless API operation. That is, the operation does not persist any data.
     * </p>
     * </note>
     * <p>
     * For an example, see <a>get-started-exercise-detect-faces</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:DetectFaces</code> action.
     * </p>
     *
     * @param detectFacesRequest
     * @return Result of the DetectFaces operation returned by the service.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws ImageTooLargeException
     *         The input image size exceeds the allowed limit. For more information, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.DetectFaces
     */
    @Override
    public DetectFacesResponse detectFaces(DetectFacesRequest detectFacesRequest) throws InvalidS3ObjectException,
            InvalidParameterException, ImageTooLargeException, AccessDeniedException, InternalServerErrorException,
            ThrottlingException, ProvisionedThroughputExceededException, InvalidImageFormatException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<DetectFacesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DetectFacesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DetectFacesRequest, DetectFacesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(detectFacesRequest).withMarshaller(new DetectFacesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Detects instances of real-world labels within an image (JPEG or PNG) provided as input. This includes objects
     * like flower, tree, and table; events like wedding, graduation, and birthday party; and concepts like landscape,
     * evening, and nature. For an example, see <a>get-started-exercise-detect-labels</a>.
     * </p>
     * <p>
     * For each object, scene, and concept the API returns one or more labels. Each label provides the object name, and
     * the level of confidence that the image contains the object. For example, suppose the input image has a
     * lighthouse, the sea, and a rock. The response will include all three labels, one for each object.
     * </p>
     * <p>
     * <code>{Name: lighthouse, Confidence: 98.4629}</code>
     * </p>
     * <p>
     * <code>{Name: rock,Confidence: 79.2097}</code>
     * </p>
     * <p>
     * <code> {Name: sea,Confidence: 75.061}</code>
     * </p>
     * <p>
     * In the preceding example, the operation returns one label for each of the three objects. The operation can also
     * return multiple labels for the same object in the image. For example, if the input image shows a flower (for
     * example, a tulip), the operation might return the following three labels.
     * </p>
     * <p>
     * <code>{Name: flower,Confidence: 99.0562}</code>
     * </p>
     * <p>
     * <code>{Name: plant,Confidence: 99.0562}</code>
     * </p>
     * <p>
     * <code>{Name: tulip,Confidence: 99.0562}</code>
     * </p>
     * <p>
     * In this example, the detection algorithm more precisely identifies the flower as a tulip.
     * </p>
     * <p>
     * You can provide the input image as an S3 object or as base64-encoded bytes. In response, the API returns an array
     * of labels. In addition, the response also includes the orientation correction. Optionally, you can specify
     * <code>MinConfidence</code> to control the confidence threshold for the labels returned. The default is 50%. You
     * can also add the <code>MaxLabels</code> parameter to limit the number of labels returned.
     * </p>
     * <note>
     * <p>
     * If the object detected is a person, the operation doesn't provide the same facial details that the
     * <a>DetectFaces</a> operation provides.
     * </p>
     * </note>
     * <p>
     * This is a stateless API operation. That is, the operation does not persist any data.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:DetectLabels</code> action.
     * </p>
     *
     * @param detectLabelsRequest
     * @return Result of the DetectLabels operation returned by the service.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws ImageTooLargeException
     *         The input image size exceeds the allowed limit. For more information, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.DetectLabels
     */
    @Override
    public DetectLabelsResponse detectLabels(DetectLabelsRequest detectLabelsRequest) throws InvalidS3ObjectException,
            InvalidParameterException, ImageTooLargeException, AccessDeniedException, InternalServerErrorException,
            ThrottlingException, ProvisionedThroughputExceededException, InvalidImageFormatException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<DetectLabelsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DetectLabelsResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DetectLabelsRequest, DetectLabelsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(detectLabelsRequest).withMarshaller(new DetectLabelsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Detects explicit or suggestive adult content in a specified JPEG or PNG format image. Use
     * <code>DetectModerationLabels</code> to moderate images depending on your requirements. For example, you might
     * want to filter images that contain nudity, but not images containing suggestive content.
     * </p>
     * <p>
     * To filter images, use the labels returned by <code>DetectModerationLabels</code> to determine which types of
     * content are appropriate. For information about moderation labels, see <a>image-moderation</a>.
     * </p>
     *
     * @param detectModerationLabelsRequest
     * @return Result of the DetectModerationLabels operation returned by the service.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws ImageTooLargeException
     *         The input image size exceeds the allowed limit. For more information, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.DetectModerationLabels
     */
    @Override
    public DetectModerationLabelsResponse detectModerationLabels(DetectModerationLabelsRequest detectModerationLabelsRequest)
            throws InvalidS3ObjectException, InvalidParameterException, ImageTooLargeException, AccessDeniedException,
            InternalServerErrorException, ThrottlingException, ProvisionedThroughputExceededException,
            InvalidImageFormatException, AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<DetectModerationLabelsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DetectModerationLabelsResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DetectModerationLabelsRequest, DetectModerationLabelsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(detectModerationLabelsRequest)
                .withMarshaller(new DetectModerationLabelsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the name and additional information about a celebrity based on his or her Rekognition ID. The additional
     * information is returned as an array of URLs. If there is no additional information about the celebrity, this list
     * is empty. For more information, see <a>celebrity-recognition</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:GetCelebrityInfo</code> action.
     * </p>
     *
     * @param getCelebrityInfoRequest
     * @return Result of the GetCelebrityInfo operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         Collection specified in the request is not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.GetCelebrityInfo
     */
    @Override
    public GetCelebrityInfoResponse getCelebrityInfo(GetCelebrityInfoRequest getCelebrityInfoRequest)
            throws InvalidParameterException, AccessDeniedException, InternalServerErrorException, ThrottlingException,
            ProvisionedThroughputExceededException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            RekognitionException {

        HttpResponseHandler<GetCelebrityInfoResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetCelebrityInfoResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetCelebrityInfoRequest, GetCelebrityInfoResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getCelebrityInfoRequest).withMarshaller(new GetCelebrityInfoRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Detects faces in the input image and adds them to the specified collection.
     * </p>
     * <p>
     * Amazon Rekognition does not save the actual faces detected. Instead, the underlying detection algorithm first
     * detects the faces in the input image, and for each face extracts facial features into a feature vector, and
     * stores it in the back-end database. Amazon Rekognition uses feature vectors when performing face match and search
     * operations using the and operations.
     * </p>
     * <p>
     * If you provide the optional <code>externalImageID</code> for the input image you provided, Amazon Rekognition
     * associates this ID with all faces that it detects. When you call the operation, the response returns the external
     * ID. You can use this external image ID to create a client-side index to associate the faces with each image. You
     * can then use the index to find all faces in an image.
     * </p>
     * <p>
     * In response, the operation returns an array of metadata for all detected faces. This includes, the bounding box
     * of the detected face, confidence value (indicating the bounding box contains a face), a face ID assigned by the
     * service for each face that is detected and stored, and an image ID assigned by the service for the input image.
     * If you request all facial attributes (using the <code>detectionAttributes</code> parameter, Amazon Rekognition
     * returns detailed facial attributes such as facial landmarks (for example, location of eye and mount) and other
     * facial attributes such gender. If you provide the same image, specify the same collection, and use the same
     * external ID in the <code>IndexFaces</code> operation, Amazon Rekognition doesn't save duplicate face metadata.
     * </p>
     * <p>
     * For an example, see <a>example2</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:IndexFaces</code> action.
     * </p>
     *
     * @param indexFacesRequest
     * @return Result of the IndexFaces operation returned by the service.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws ImageTooLargeException
     *         The input image size exceeds the allowed limit. For more information, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         Collection specified in the request is not found.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.IndexFaces
     */
    @Override
    public IndexFacesResponse indexFaces(IndexFacesRequest indexFacesRequest) throws InvalidS3ObjectException,
            InvalidParameterException, ImageTooLargeException, AccessDeniedException, InternalServerErrorException,
            ThrottlingException, ProvisionedThroughputExceededException, ResourceNotFoundException, InvalidImageFormatException,
            AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<IndexFacesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new IndexFacesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<IndexFacesRequest, IndexFacesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(indexFacesRequest)
                .withMarshaller(new IndexFacesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns list of collection IDs in your account. If the result is truncated, the response also provides a
     * <code>NextToken</code> that you can use in the subsequent request to fetch the next set of collection IDs.
     * </p>
     * <p>
     * For an example, see <a>example1</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:ListCollections</code> action.
     * </p>
     *
     * @param listCollectionsRequest
     * @return Result of the ListCollections operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ResourceNotFoundException
     *         Collection specified in the request is not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.ListCollections
     */
    @Override
    public ListCollectionsResponse listCollections(ListCollectionsRequest listCollectionsRequest)
            throws InvalidParameterException, AccessDeniedException, InternalServerErrorException, ThrottlingException,
            ProvisionedThroughputExceededException, InvalidPaginationTokenException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<ListCollectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListCollectionsResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListCollectionsRequest, ListCollectionsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listCollectionsRequest).withMarshaller(new ListCollectionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns list of collection IDs in your account. If the result is truncated, the response also provides a
     * <code>NextToken</code> that you can use in the subsequent request to fetch the next set of collection IDs.
     * </p>
     * <p>
     * For an example, see <a>example1</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:ListCollections</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listCollections(software.amazon.awssdk.services.rekognition.model.ListCollectionsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.ListCollectionsIterable responses = client.listCollectionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rekognition.paginators.ListCollectionsIterable responses = client
     *             .listCollectionsPaginator(request);
     *     for (software.amazon.awssdk.services.rekognition.model.ListCollectionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.ListCollectionsIterable responses = client.listCollectionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCollections(software.amazon.awssdk.services.rekognition.model.ListCollectionsRequest)} operation.</b>
     * </p>
     *
     * @param listCollectionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ResourceNotFoundException
     *         Collection specified in the request is not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.ListCollections
     */
    @Override
    public ListCollectionsIterable listCollectionsPaginator(ListCollectionsRequest listCollectionsRequest)
            throws InvalidParameterException, AccessDeniedException, InternalServerErrorException, ThrottlingException,
            ProvisionedThroughputExceededException, InvalidPaginationTokenException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, RekognitionException {
        return new ListCollectionsIterable(this, listCollectionsRequest);
    }

    /**
     * <p>
     * Returns metadata for faces in the specified collection. This metadata includes information such as the bounding
     * box coordinates, the confidence (that the bounding box contains a face), and face ID. For an example, see
     * <a>example3</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:ListFaces</code> action.
     * </p>
     *
     * @param listFacesRequest
     * @return Result of the ListFaces operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ResourceNotFoundException
     *         Collection specified in the request is not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.ListFaces
     */
    @Override
    public ListFacesResponse listFaces(ListFacesRequest listFacesRequest) throws InvalidParameterException,
            AccessDeniedException, InternalServerErrorException, ThrottlingException, ProvisionedThroughputExceededException,
            InvalidPaginationTokenException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            RekognitionException {

        HttpResponseHandler<ListFacesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListFacesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListFacesRequest, ListFacesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(listFacesRequest)
                .withMarshaller(new ListFacesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns metadata for faces in the specified collection. This metadata includes information such as the bounding
     * box coordinates, the confidence (that the bounding box contains a face), and face ID. For an example, see
     * <a>example3</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:ListFaces</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listFaces(software.amazon.awssdk.services.rekognition.model.ListFacesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.ListFacesIterable responses = client.listFacesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rekognition.paginators.ListFacesIterable responses = client.listFacesPaginator(request);
     *     for (software.amazon.awssdk.services.rekognition.model.ListFacesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.ListFacesIterable responses = client.listFacesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFaces(software.amazon.awssdk.services.rekognition.model.ListFacesRequest)} operation.</b>
     * </p>
     *
     * @param listFacesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ResourceNotFoundException
     *         Collection specified in the request is not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.ListFaces
     */
    @Override
    public ListFacesIterable listFacesPaginator(ListFacesRequest listFacesRequest) throws InvalidParameterException,
            AccessDeniedException, InternalServerErrorException, ThrottlingException, ProvisionedThroughputExceededException,
            InvalidPaginationTokenException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            RekognitionException {
        return new ListFacesIterable(this, listFacesRequest);
    }

    /**
     * <p>
     * Returns an array of celebrities recognized in the input image. The image is passed either as base64-encoded image
     * bytes or as a reference to an image in an Amazon S3 bucket. The image must be either a PNG or JPEG formatted
     * file. For more information, see <a>celebrity-recognition</a>.
     * </p>
     * <p>
     * <code>RecognizeCelebrities</code> returns the 15 largest faces in the image. It lists recognized celebrities in
     * the <code>CelebrityFaces</code> list and unrecognized faces in the <code>UnrecognizedFaces</code> list. The
     * operation doesn't return celebrities whose face sizes are smaller than the largest 15 faces in the image.
     * </p>
     * <p>
     * For each celebrity recognized, the API returns a <code>Celebrity</code> object. The <code>Celebrity</code> object
     * contains the celebrity name, ID, URL links to additional information, match confidence, and a
     * <code>ComparedFace</code> object that you can use to locate the celebrity's face on the image.
     * </p>
     * <p>
     * Rekognition does not retain information about which images a celebrity has been recognized in. Your application
     * must store this information and use the <code>Celebrity</code> ID property as a unique identifier for the
     * celebrity. If you don't store the celebrity name or additional information URLs returned by
     * <code>RecognizeCelebrities</code>, you will need the ID to identify the celebrity in a call to the operation.
     * </p>
     * <p>
     * For an example, see <a>recognize-celebrities-tutorial</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:RecognizeCelebrities</code> operation.
     * </p>
     *
     * @param recognizeCelebritiesRequest
     * @return Result of the RecognizeCelebrities operation returned by the service.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws ImageTooLargeException
     *         The input image size exceeds the allowed limit. For more information, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.RecognizeCelebrities
     */
    @Override
    public RecognizeCelebritiesResponse recognizeCelebrities(RecognizeCelebritiesRequest recognizeCelebritiesRequest)
            throws InvalidS3ObjectException, InvalidParameterException, InvalidImageFormatException, ImageTooLargeException,
            AccessDeniedException, InternalServerErrorException, ThrottlingException, ProvisionedThroughputExceededException,
            AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<RecognizeCelebritiesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new RecognizeCelebritiesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<RecognizeCelebritiesRequest, RecognizeCelebritiesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(recognizeCelebritiesRequest)
                .withMarshaller(new RecognizeCelebritiesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * For a given input face ID, searches for matching faces in the collection the face belongs to. You get a face ID
     * when you add a face to the collection using the <a>IndexFaces</a> operation. The operation compares the features
     * of the input face with faces in the specified collection.
     * </p>
     * <note>
     * <p>
     * You can also search faces without indexing faces by using the <code>SearchFacesByImage</code> operation.
     * </p>
     * </note>
     * <p>
     * The operation response returns an array of faces that match, ordered by similarity score with the highest
     * similarity first. More specifically, it is an array of metadata for each face match that is found. Along with the
     * metadata, the response also includes a <code>confidence</code> value for each face match, indicating the
     * confidence that the specific face matches the input face.
     * </p>
     * <p>
     * For an example, see <a>example3</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:SearchFaces</code> action.
     * </p>
     *
     * @param searchFacesRequest
     * @return Result of the SearchFaces operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         Collection specified in the request is not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.SearchFaces
     */
    @Override
    public SearchFacesResponse searchFaces(SearchFacesRequest searchFacesRequest) throws InvalidParameterException,
            AccessDeniedException, InternalServerErrorException, ThrottlingException, ProvisionedThroughputExceededException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<SearchFacesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new SearchFacesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<SearchFacesRequest, SearchFacesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(searchFacesRequest).withMarshaller(new SearchFacesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * For a given input image, first detects the largest face in the image, and then searches the specified collection
     * for matching faces. The operation compares the features of the input face with faces in the specified collection.
     * </p>
     * <note>
     * <p>
     * To search for all faces in an input image, you might first call the operation, and then use the face IDs returned
     * in subsequent calls to the operation.
     * </p>
     * <p>
     * You can also call the <code>DetectFaces</code> operation and use the bounding boxes in the response to make face
     * crops, which then you can pass in to the <code>SearchFacesByImage</code> operation.
     * </p>
     * </note>
     * <p>
     * The response returns an array of faces that match, ordered by similarity score with the highest similarity first.
     * More specifically, it is an array of metadata for each face match found. Along with the metadata, the response
     * also includes a <code>similarity</code> indicating how similar the face is to the input face. In the response,
     * the operation also returns the bounding box (and a confidence level that the bounding box contains a face) of the
     * face that Amazon Rekognition used for the input image.
     * </p>
     * <p>
     * For an example, see <a>example3</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:SearchFacesByImage</code> action.
     * </p>
     *
     * @param searchFacesByImageRequest
     * @return Result of the SearchFacesByImage operation returned by the service.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws ImageTooLargeException
     *         The input image size exceeds the allowed limit. For more information, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         Collection specified in the request is not found.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.SearchFacesByImage
     */
    @Override
    public SearchFacesByImageResponse searchFacesByImage(SearchFacesByImageRequest searchFacesByImageRequest)
            throws InvalidS3ObjectException, InvalidParameterException, ImageTooLargeException, AccessDeniedException,
            InternalServerErrorException, ThrottlingException, ProvisionedThroughputExceededException, ResourceNotFoundException,
            InvalidImageFormatException, AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<SearchFacesByImageResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new SearchFacesByImageResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<SearchFacesByImageRequest, SearchFacesByImageResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(searchFacesByImageRequest).withMarshaller(new SearchFacesByImageRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler() {
        return protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());
    }

    private software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory init() {
        return new AwsJsonProtocolFactory(new JsonClientMetadata()
                .withSupportsCbor(false)
                .withSupportsIon(false)
                .withBaseServiceExceptionClass(software.amazon.awssdk.services.rekognition.model.RekognitionException.class)
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("AccessDeniedException").withModeledClass(
                                AccessDeniedException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidParameterException").withModeledClass(
                                InvalidParameterException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ResourceNotFoundException").withModeledClass(
                                ResourceNotFoundException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidImageFormatException").withModeledClass(
                                InvalidImageFormatException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ThrottlingException").withModeledClass(
                                ThrottlingException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ProvisionedThroughputExceededException").withModeledClass(
                                ProvisionedThroughputExceededException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ImageTooLargeException").withModeledClass(
                                ImageTooLargeException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ResourceAlreadyExistsException").withModeledClass(
                                ResourceAlreadyExistsException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidPaginationTokenException").withModeledClass(
                                InvalidPaginationTokenException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidS3ObjectException").withModeledClass(
                                InvalidS3ObjectException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InternalServerError").withModeledClass(
                                InternalServerErrorException.class)), AwsJsonProtocolMetadata.builder().protocolVersion("1.1")
                .protocol(AwsJsonProtocol.AWS_JSON).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
