/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.rekognition.transform.ComparedFaceMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides face metadata for target image faces that are analysed by <code>CompareFaces</code> and
 * <code>RecognizeCelebrities</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ComparedFace implements StructuredPojo, ToCopyableBuilder<ComparedFace.Builder, ComparedFace> {
    private final BoundingBox boundingBox;

    private final Float confidence;

    private final List<Landmark> landmarks;

    private final Pose pose;

    private final ImageQuality quality;

    private ComparedFace(BuilderImpl builder) {
        this.boundingBox = builder.boundingBox;
        this.confidence = builder.confidence;
        this.landmarks = builder.landmarks;
        this.pose = builder.pose;
        this.quality = builder.quality;
    }

    /**
     * <p>
     * Bounding box of the face.
     * </p>
     * 
     * @return Bounding box of the face.
     */
    public BoundingBox boundingBox() {
        return boundingBox;
    }

    /**
     * <p>
     * Level of confidence that what the bounding box contains is a face.
     * </p>
     * 
     * @return Level of confidence that what the bounding box contains is a face.
     */
    public Float confidence() {
        return confidence;
    }

    /**
     * <p>
     * An array of facial landmarks.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of facial landmarks.
     */
    public List<Landmark> landmarks() {
        return landmarks;
    }

    /**
     * <p>
     * Indicates the pose of the face as determined by its pitch, roll, and yaw.
     * </p>
     * 
     * @return Indicates the pose of the face as determined by its pitch, roll, and yaw.
     */
    public Pose pose() {
        return pose;
    }

    /**
     * <p>
     * Identifies face image brightness and sharpness.
     * </p>
     * 
     * @return Identifies face image brightness and sharpness.
     */
    public ImageQuality quality() {
        return quality;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(boundingBox());
        hashCode = 31 * hashCode + Objects.hashCode(confidence());
        hashCode = 31 * hashCode + Objects.hashCode(landmarks());
        hashCode = 31 * hashCode + Objects.hashCode(pose());
        hashCode = 31 * hashCode + Objects.hashCode(quality());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComparedFace)) {
            return false;
        }
        ComparedFace other = (ComparedFace) obj;
        return Objects.equals(boundingBox(), other.boundingBox()) && Objects.equals(confidence(), other.confidence())
                && Objects.equals(landmarks(), other.landmarks()) && Objects.equals(pose(), other.pose())
                && Objects.equals(quality(), other.quality());
    }

    @Override
    public String toString() {
        return ToString.builder("ComparedFace").add("BoundingBox", boundingBox()).add("Confidence", confidence())
                .add("Landmarks", landmarks()).add("Pose", pose()).add("Quality", quality()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BoundingBox":
            return Optional.of(clazz.cast(boundingBox()));
        case "Confidence":
            return Optional.of(clazz.cast(confidence()));
        case "Landmarks":
            return Optional.of(clazz.cast(landmarks()));
        case "Pose":
            return Optional.of(clazz.cast(pose()));
        case "Quality":
            return Optional.of(clazz.cast(quality()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ComparedFaceMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ComparedFace> {
        /**
         * <p>
         * Bounding box of the face.
         * </p>
         * 
         * @param boundingBox
         *        Bounding box of the face.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder boundingBox(BoundingBox boundingBox);

        /**
         * <p>
         * Bounding box of the face.
         * </p>
         * This is a convenience that creates an instance of the {@link BoundingBox.Builder} avoiding the need to create
         * one manually via {@link BoundingBox#builder()}.
         *
         * When the {@link Consumer} completes, {@link BoundingBox.Builder#build()} is called immediately and its result
         * is passed to {@link #boundingBox(BoundingBox)}.
         * 
         * @param boundingBox
         *        a consumer that will call methods on {@link BoundingBox.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #boundingBox(BoundingBox)
         */
        default Builder boundingBox(Consumer<BoundingBox.Builder> boundingBox) {
            return boundingBox(BoundingBox.builder().apply(boundingBox).build());
        }

        /**
         * <p>
         * Level of confidence that what the bounding box contains is a face.
         * </p>
         * 
         * @param confidence
         *        Level of confidence that what the bounding box contains is a face.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confidence(Float confidence);

        /**
         * <p>
         * An array of facial landmarks.
         * </p>
         * 
         * @param landmarks
         *        An array of facial landmarks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder landmarks(Collection<Landmark> landmarks);

        /**
         * <p>
         * An array of facial landmarks.
         * </p>
         * 
         * @param landmarks
         *        An array of facial landmarks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder landmarks(Landmark... landmarks);

        /**
         * <p>
         * An array of facial landmarks.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Landmark>.Builder} avoiding the need to
         * create one manually via {@link List<Landmark>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Landmark>.Builder#build()} is called immediately and its
         * result is passed to {@link #landmarks(List<Landmark>)}.
         * 
         * @param landmarks
         *        a consumer that will call methods on {@link List<Landmark>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #landmarks(List<Landmark>)
         */
        Builder landmarks(Consumer<Landmark.Builder>... landmarks);

        /**
         * <p>
         * Indicates the pose of the face as determined by its pitch, roll, and yaw.
         * </p>
         * 
         * @param pose
         *        Indicates the pose of the face as determined by its pitch, roll, and yaw.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pose(Pose pose);

        /**
         * <p>
         * Indicates the pose of the face as determined by its pitch, roll, and yaw.
         * </p>
         * This is a convenience that creates an instance of the {@link Pose.Builder} avoiding the need to create one
         * manually via {@link Pose#builder()}.
         *
         * When the {@link Consumer} completes, {@link Pose.Builder#build()} is called immediately and its result is
         * passed to {@link #pose(Pose)}.
         * 
         * @param pose
         *        a consumer that will call methods on {@link Pose.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #pose(Pose)
         */
        default Builder pose(Consumer<Pose.Builder> pose) {
            return pose(Pose.builder().apply(pose).build());
        }

        /**
         * <p>
         * Identifies face image brightness and sharpness.
         * </p>
         * 
         * @param quality
         *        Identifies face image brightness and sharpness.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quality(ImageQuality quality);

        /**
         * <p>
         * Identifies face image brightness and sharpness.
         * </p>
         * This is a convenience that creates an instance of the {@link ImageQuality.Builder} avoiding the need to
         * create one manually via {@link ImageQuality#builder()}.
         *
         * When the {@link Consumer} completes, {@link ImageQuality.Builder#build()} is called immediately and its
         * result is passed to {@link #quality(ImageQuality)}.
         * 
         * @param quality
         *        a consumer that will call methods on {@link ImageQuality.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #quality(ImageQuality)
         */
        default Builder quality(Consumer<ImageQuality.Builder> quality) {
            return quality(ImageQuality.builder().apply(quality).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private BoundingBox boundingBox;

        private Float confidence;

        private List<Landmark> landmarks;

        private Pose pose;

        private ImageQuality quality;

        private BuilderImpl() {
        }

        private BuilderImpl(ComparedFace model) {
            boundingBox(model.boundingBox);
            confidence(model.confidence);
            landmarks(model.landmarks);
            pose(model.pose);
            quality(model.quality);
        }

        public final BoundingBox.Builder getBoundingBox() {
            return boundingBox != null ? boundingBox.toBuilder() : null;
        }

        @Override
        public final Builder boundingBox(BoundingBox boundingBox) {
            this.boundingBox = boundingBox;
            return this;
        }

        public final void setBoundingBox(BoundingBox.BuilderImpl boundingBox) {
            this.boundingBox = boundingBox != null ? boundingBox.build() : null;
        }

        public final Float getConfidence() {
            return confidence;
        }

        @Override
        public final Builder confidence(Float confidence) {
            this.confidence = confidence;
            return this;
        }

        public final void setConfidence(Float confidence) {
            this.confidence = confidence;
        }

        public final Collection<Landmark.Builder> getLandmarks() {
            return landmarks != null ? landmarks.stream().map(Landmark::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder landmarks(Collection<Landmark> landmarks) {
            this.landmarks = LandmarksCopier.copy(landmarks);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder landmarks(Landmark... landmarks) {
            landmarks(Arrays.asList(landmarks));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder landmarks(Consumer<Landmark.Builder>... landmarks) {
            landmarks(Stream.of(landmarks).map(c -> Landmark.builder().apply(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setLandmarks(Collection<Landmark.BuilderImpl> landmarks) {
            this.landmarks = LandmarksCopier.copyFromBuilder(landmarks);
        }

        public final Pose.Builder getPose() {
            return pose != null ? pose.toBuilder() : null;
        }

        @Override
        public final Builder pose(Pose pose) {
            this.pose = pose;
            return this;
        }

        public final void setPose(Pose.BuilderImpl pose) {
            this.pose = pose != null ? pose.build() : null;
        }

        public final ImageQuality.Builder getQuality() {
            return quality != null ? quality.toBuilder() : null;
        }

        @Override
        public final Builder quality(ImageQuality quality) {
            this.quality = quality;
            return this;
        }

        public final void setQuality(ImageQuality.BuilderImpl quality) {
            this.quality = quality != null ? quality.build() : null;
        }

        @Override
        public ComparedFace build() {
            return new ComparedFace(this);
        }
    }
}
