/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.protocol.json.AwsJsonProtocol;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolMetadata;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.internal.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.protocol.json.JsonClientMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorResponseMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorShapeMetadata;
import software.amazon.awssdk.core.protocol.json.JsonOperationMetadata;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.services.rekognition.model.AccessDeniedException;
import software.amazon.awssdk.services.rekognition.model.CompareFacesRequest;
import software.amazon.awssdk.services.rekognition.model.CompareFacesResponse;
import software.amazon.awssdk.services.rekognition.model.CreateCollectionRequest;
import software.amazon.awssdk.services.rekognition.model.CreateCollectionResponse;
import software.amazon.awssdk.services.rekognition.model.CreateStreamProcessorRequest;
import software.amazon.awssdk.services.rekognition.model.CreateStreamProcessorResponse;
import software.amazon.awssdk.services.rekognition.model.DeleteCollectionRequest;
import software.amazon.awssdk.services.rekognition.model.DeleteCollectionResponse;
import software.amazon.awssdk.services.rekognition.model.DeleteFacesRequest;
import software.amazon.awssdk.services.rekognition.model.DeleteFacesResponse;
import software.amazon.awssdk.services.rekognition.model.DeleteStreamProcessorRequest;
import software.amazon.awssdk.services.rekognition.model.DeleteStreamProcessorResponse;
import software.amazon.awssdk.services.rekognition.model.DescribeCollectionRequest;
import software.amazon.awssdk.services.rekognition.model.DescribeCollectionResponse;
import software.amazon.awssdk.services.rekognition.model.DescribeStreamProcessorRequest;
import software.amazon.awssdk.services.rekognition.model.DescribeStreamProcessorResponse;
import software.amazon.awssdk.services.rekognition.model.DetectFacesRequest;
import software.amazon.awssdk.services.rekognition.model.DetectFacesResponse;
import software.amazon.awssdk.services.rekognition.model.DetectLabelsRequest;
import software.amazon.awssdk.services.rekognition.model.DetectLabelsResponse;
import software.amazon.awssdk.services.rekognition.model.DetectModerationLabelsRequest;
import software.amazon.awssdk.services.rekognition.model.DetectModerationLabelsResponse;
import software.amazon.awssdk.services.rekognition.model.DetectTextRequest;
import software.amazon.awssdk.services.rekognition.model.DetectTextResponse;
import software.amazon.awssdk.services.rekognition.model.GetCelebrityInfoRequest;
import software.amazon.awssdk.services.rekognition.model.GetCelebrityInfoResponse;
import software.amazon.awssdk.services.rekognition.model.GetCelebrityRecognitionRequest;
import software.amazon.awssdk.services.rekognition.model.GetCelebrityRecognitionResponse;
import software.amazon.awssdk.services.rekognition.model.GetContentModerationRequest;
import software.amazon.awssdk.services.rekognition.model.GetContentModerationResponse;
import software.amazon.awssdk.services.rekognition.model.GetFaceDetectionRequest;
import software.amazon.awssdk.services.rekognition.model.GetFaceDetectionResponse;
import software.amazon.awssdk.services.rekognition.model.GetFaceSearchRequest;
import software.amazon.awssdk.services.rekognition.model.GetFaceSearchResponse;
import software.amazon.awssdk.services.rekognition.model.GetLabelDetectionRequest;
import software.amazon.awssdk.services.rekognition.model.GetLabelDetectionResponse;
import software.amazon.awssdk.services.rekognition.model.GetPersonTrackingRequest;
import software.amazon.awssdk.services.rekognition.model.GetPersonTrackingResponse;
import software.amazon.awssdk.services.rekognition.model.IdempotentParameterMismatchException;
import software.amazon.awssdk.services.rekognition.model.ImageTooLargeException;
import software.amazon.awssdk.services.rekognition.model.IndexFacesRequest;
import software.amazon.awssdk.services.rekognition.model.IndexFacesResponse;
import software.amazon.awssdk.services.rekognition.model.InternalServerErrorException;
import software.amazon.awssdk.services.rekognition.model.InvalidImageFormatException;
import software.amazon.awssdk.services.rekognition.model.InvalidPaginationTokenException;
import software.amazon.awssdk.services.rekognition.model.InvalidParameterException;
import software.amazon.awssdk.services.rekognition.model.InvalidS3ObjectException;
import software.amazon.awssdk.services.rekognition.model.LimitExceededException;
import software.amazon.awssdk.services.rekognition.model.ListCollectionsRequest;
import software.amazon.awssdk.services.rekognition.model.ListCollectionsResponse;
import software.amazon.awssdk.services.rekognition.model.ListFacesRequest;
import software.amazon.awssdk.services.rekognition.model.ListFacesResponse;
import software.amazon.awssdk.services.rekognition.model.ListStreamProcessorsRequest;
import software.amazon.awssdk.services.rekognition.model.ListStreamProcessorsResponse;
import software.amazon.awssdk.services.rekognition.model.ProvisionedThroughputExceededException;
import software.amazon.awssdk.services.rekognition.model.RecognizeCelebritiesRequest;
import software.amazon.awssdk.services.rekognition.model.RecognizeCelebritiesResponse;
import software.amazon.awssdk.services.rekognition.model.RekognitionException;
import software.amazon.awssdk.services.rekognition.model.RekognitionRequest;
import software.amazon.awssdk.services.rekognition.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.rekognition.model.ResourceInUseException;
import software.amazon.awssdk.services.rekognition.model.ResourceNotFoundException;
import software.amazon.awssdk.services.rekognition.model.SearchFacesByImageRequest;
import software.amazon.awssdk.services.rekognition.model.SearchFacesByImageResponse;
import software.amazon.awssdk.services.rekognition.model.SearchFacesRequest;
import software.amazon.awssdk.services.rekognition.model.SearchFacesResponse;
import software.amazon.awssdk.services.rekognition.model.StartCelebrityRecognitionRequest;
import software.amazon.awssdk.services.rekognition.model.StartCelebrityRecognitionResponse;
import software.amazon.awssdk.services.rekognition.model.StartContentModerationRequest;
import software.amazon.awssdk.services.rekognition.model.StartContentModerationResponse;
import software.amazon.awssdk.services.rekognition.model.StartFaceDetectionRequest;
import software.amazon.awssdk.services.rekognition.model.StartFaceDetectionResponse;
import software.amazon.awssdk.services.rekognition.model.StartFaceSearchRequest;
import software.amazon.awssdk.services.rekognition.model.StartFaceSearchResponse;
import software.amazon.awssdk.services.rekognition.model.StartLabelDetectionRequest;
import software.amazon.awssdk.services.rekognition.model.StartLabelDetectionResponse;
import software.amazon.awssdk.services.rekognition.model.StartPersonTrackingRequest;
import software.amazon.awssdk.services.rekognition.model.StartPersonTrackingResponse;
import software.amazon.awssdk.services.rekognition.model.StartStreamProcessorRequest;
import software.amazon.awssdk.services.rekognition.model.StartStreamProcessorResponse;
import software.amazon.awssdk.services.rekognition.model.StopStreamProcessorRequest;
import software.amazon.awssdk.services.rekognition.model.StopStreamProcessorResponse;
import software.amazon.awssdk.services.rekognition.model.ThrottlingException;
import software.amazon.awssdk.services.rekognition.model.VideoTooLargeException;
import software.amazon.awssdk.services.rekognition.paginators.GetCelebrityRecognitionIterable;
import software.amazon.awssdk.services.rekognition.paginators.GetContentModerationIterable;
import software.amazon.awssdk.services.rekognition.paginators.GetFaceDetectionIterable;
import software.amazon.awssdk.services.rekognition.paginators.GetFaceSearchIterable;
import software.amazon.awssdk.services.rekognition.paginators.GetLabelDetectionIterable;
import software.amazon.awssdk.services.rekognition.paginators.GetPersonTrackingIterable;
import software.amazon.awssdk.services.rekognition.paginators.ListCollectionsIterable;
import software.amazon.awssdk.services.rekognition.paginators.ListFacesIterable;
import software.amazon.awssdk.services.rekognition.paginators.ListStreamProcessorsIterable;
import software.amazon.awssdk.services.rekognition.transform.CompareFacesRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.CompareFacesResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.CreateCollectionRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.CreateCollectionResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.CreateStreamProcessorRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.CreateStreamProcessorResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.DeleteCollectionRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.DeleteCollectionResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.DeleteFacesRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.DeleteFacesResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.DeleteStreamProcessorRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.DeleteStreamProcessorResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.DescribeCollectionRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.DescribeCollectionResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.DescribeStreamProcessorRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.DescribeStreamProcessorResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.DetectFacesRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.DetectFacesResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.DetectLabelsRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.DetectLabelsResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.DetectModerationLabelsRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.DetectModerationLabelsResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.DetectTextRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.DetectTextResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetCelebrityInfoRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetCelebrityInfoResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetCelebrityRecognitionRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetCelebrityRecognitionResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetContentModerationRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetContentModerationResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetFaceDetectionRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetFaceDetectionResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetFaceSearchRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetFaceSearchResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetLabelDetectionRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetLabelDetectionResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetPersonTrackingRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.GetPersonTrackingResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.IndexFacesRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.IndexFacesResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.ListCollectionsRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.ListCollectionsResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.ListFacesRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.ListFacesResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.ListStreamProcessorsRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.ListStreamProcessorsResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.RecognizeCelebritiesRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.RecognizeCelebritiesResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.SearchFacesByImageRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.SearchFacesByImageResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.SearchFacesRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.SearchFacesResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.StartCelebrityRecognitionRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.StartCelebrityRecognitionResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.StartContentModerationRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.StartContentModerationResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.StartFaceDetectionRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.StartFaceDetectionResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.StartFaceSearchRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.StartFaceSearchResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.StartLabelDetectionRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.StartLabelDetectionResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.StartPersonTrackingRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.StartPersonTrackingResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.StartStreamProcessorRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.StartStreamProcessorResponseUnmarshaller;
import software.amazon.awssdk.services.rekognition.transform.StopStreamProcessorRequestMarshaller;
import software.amazon.awssdk.services.rekognition.transform.StopStreamProcessorResponseUnmarshaller;

/**
 * Internal implementation of {@link RekognitionClient}.
 *
 * @see RekognitionClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRekognitionClient implements RekognitionClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRekognitionClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(false);
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Compares a face in the <i>source</i> input image with each of the 100 largest faces detected in the <i>target</i>
     * input image.
     * </p>
     * <note>
     * <p>
     * If the source image contains multiple faces, the service detects the largest face and compares it with each face
     * detected in the target image.
     * </p>
     * </note>
     * <p>
     * You pass the input and target images either as base64-encoded image bytes or as a references to images in an
     * Amazon S3 bucket. If you use the Amazon CLI to call Amazon Rekognition operations, passing image bytes is not
     * supported. The image must be either a PNG or JPEG formatted file.
     * </p>
     * <p>
     * In response, the operation returns an array of face matches ordered by similarity score in descending order. For
     * each face match, the response provides a bounding box of the face, facial landmarks, pose details (pitch, role,
     * and yaw), quality (brightness and sharpness), and confidence value (indicating the level of confidence that the
     * bounding box contains a face). The response also provides a similarity score, which indicates how closely the
     * faces match.
     * </p>
     * <note>
     * <p>
     * By default, only faces with a similarity score of greater than or equal to 80% are returned in the response. You
     * can change this value by specifying the <code>SimilarityThreshold</code> parameter.
     * </p>
     * </note>
     * <p>
     * <code>CompareFaces</code> also returns an array of faces that don't match the source image. For each face, it
     * returns a bounding box, confidence value, landmarks, pose details, and quality. The response also returns
     * information about the face in the source image, including the bounding box of the face and confidence value.
     * </p>
     * <p>
     * If the image doesn't contain Exif metadata, <code>CompareFaces</code> returns orientation information for the
     * source and target images. Use these values to display the images with the correct image orientation.
     * </p>
     * <p>
     * If no faces are detected in the source or target images, <code>CompareFaces</code> returns an
     * <code>InvalidParameterException</code> error.
     * </p>
     * <note>
     * <p>
     * This is a stateless API operation. That is, data returned by this operation doesn't persist.
     * </p>
     * </note>
     * <p>
     * For an example, see Comparing Faces in Images in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:CompareFaces</code> action.
     * </p>
     *
     * @param compareFacesRequest
     * @return Result of the CompareFaces operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws ImageTooLargeException
     *         The input image size exceeds the allowed limit. For more information, see Limits in Amazon Rekognition in
     *         the Amazon Rekognition Developer Guide.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.CompareFaces
     */
    @Override
    public CompareFacesResponse compareFaces(CompareFacesRequest compareFacesRequest) throws InvalidParameterException,
            InvalidS3ObjectException, ImageTooLargeException, AccessDeniedException, InternalServerErrorException,
            ThrottlingException, ProvisionedThroughputExceededException, InvalidImageFormatException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<CompareFacesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CompareFacesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<CompareFacesRequest, CompareFacesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(compareFacesRequest).withMarshaller(new CompareFacesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a collection in an AWS Region. You can add faces to the collection using the operation.
     * </p>
     * <p>
     * For example, you might create collections, one for each of your application users. A user can then index faces
     * using the <code>IndexFaces</code> operation and persist results in a specific collection. Then, a user can search
     * the collection for faces in the user-specific container.
     * </p>
     * <note>
     * <p>
     * Collection names are case-sensitive.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:CreateCollection</code> action.
     * </p>
     *
     * @param createCollectionRequest
     * @return Result of the CreateCollection operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceAlreadyExistsException
     *         A collection with the specified ID already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.CreateCollection
     */
    @Override
    public CreateCollectionResponse createCollection(CreateCollectionRequest createCollectionRequest)
            throws InvalidParameterException, AccessDeniedException, InternalServerErrorException, ThrottlingException,
            ProvisionedThroughputExceededException, ResourceAlreadyExistsException, AwsServiceException, SdkClientException,
            RekognitionException {

        HttpResponseHandler<CreateCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateCollectionResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<CreateCollectionRequest, CreateCollectionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(createCollectionRequest).withMarshaller(new CreateCollectionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates an Amazon Rekognition stream processor that you can use to detect and recognize faces in a streaming
     * video.
     * </p>
     * <p>
     * Amazon Rekognition Video is a consumer of live video from Amazon Kinesis Video Streams. Amazon Rekognition Video
     * sends analysis results to Amazon Kinesis Data Streams.
     * </p>
     * <p>
     * You provide as input a Kinesis video stream (<code>Input</code>) and a Kinesis data stream (<code>Output</code>)
     * stream. You also specify the face recognition criteria in <code>Settings</code>. For example, the collection
     * containing faces that you want to recognize. Use <code>Name</code> to assign an identifier for the stream
     * processor. You use <code>Name</code> to manage the stream processor. For example, you can start processing the
     * source video by calling with the <code>Name</code> field.
     * </p>
     * <p>
     * After you have finished analyzing a streaming video, use to stop processing. You can delete the stream processor
     * by calling .
     * </p>
     *
     * @param createStreamProcessorRequest
     * @return Result of the CreateStreamProcessor operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws LimitExceededException
     *         An Amazon Rekognition service limit was exceeded. For example, if you start too many Amazon Rekognition
     *         Video jobs concurrently, calls to start operations (<code>StartLabelDetection</code>, for example) will
     *         raise a <code>LimitExceededException</code> exception (HTTP status code: 400) until the number of
     *         concurrently running jobs is below the Amazon Rekognition service limit.
     * @throws ResourceInUseException
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.CreateStreamProcessor
     */
    @Override
    public CreateStreamProcessorResponse createStreamProcessor(CreateStreamProcessorRequest createStreamProcessorRequest)
            throws AccessDeniedException, InternalServerErrorException, ThrottlingException, InvalidParameterException,
            LimitExceededException, ResourceInUseException, ProvisionedThroughputExceededException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<CreateStreamProcessorResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateStreamProcessorResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<CreateStreamProcessorRequest, CreateStreamProcessorResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(createStreamProcessorRequest)
                .withMarshaller(new CreateStreamProcessorRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the specified collection. Note that this operation removes all faces in the collection. For an example,
     * see <a>delete-collection-procedure</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:DeleteCollection</code> action.
     * </p>
     *
     * @param deleteCollectionRequest
     * @return Result of the DeleteCollection operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.DeleteCollection
     */
    @Override
    public DeleteCollectionResponse deleteCollection(DeleteCollectionRequest deleteCollectionRequest)
            throws InvalidParameterException, AccessDeniedException, InternalServerErrorException, ThrottlingException,
            ProvisionedThroughputExceededException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            RekognitionException {

        HttpResponseHandler<DeleteCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DeleteCollectionResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<DeleteCollectionRequest, DeleteCollectionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(deleteCollectionRequest).withMarshaller(new DeleteCollectionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes faces from a collection. You specify a collection ID and an array of face IDs to remove from the
     * collection.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:DeleteFaces</code> action.
     * </p>
     *
     * @param deleteFacesRequest
     * @return Result of the DeleteFaces operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.DeleteFaces
     */
    @Override
    public DeleteFacesResponse deleteFaces(DeleteFacesRequest deleteFacesRequest) throws InvalidParameterException,
            AccessDeniedException, InternalServerErrorException, ThrottlingException, ProvisionedThroughputExceededException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<DeleteFacesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DeleteFacesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<DeleteFacesRequest, DeleteFacesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(deleteFacesRequest).withMarshaller(new DeleteFacesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the stream processor identified by <code>Name</code>. You assign the value for <code>Name</code> when you
     * create the stream processor with . You might not be able to use the same name for a stream processor for a few
     * seconds after calling <code>DeleteStreamProcessor</code>.
     * </p>
     *
     * @param deleteStreamProcessorRequest
     * @return Result of the DeleteStreamProcessor operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ResourceInUseException
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.DeleteStreamProcessor
     */
    @Override
    public DeleteStreamProcessorResponse deleteStreamProcessor(DeleteStreamProcessorRequest deleteStreamProcessorRequest)
            throws AccessDeniedException, InternalServerErrorException, ThrottlingException, InvalidParameterException,
            ResourceNotFoundException, ResourceInUseException, ProvisionedThroughputExceededException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<DeleteStreamProcessorResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DeleteStreamProcessorResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<DeleteStreamProcessorRequest, DeleteStreamProcessorResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(deleteStreamProcessorRequest)
                .withMarshaller(new DeleteStreamProcessorRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes the specified collection. You can use <code>DescribeCollection</code> to get information, such as the
     * number of faces indexed into a collection and the version of the model used by the collection for face detection.
     * </p>
     * <p>
     * For more information, see Describing a Collection in the Amazon Rekognition Developer Guide.
     * </p>
     *
     * @param describeCollectionRequest
     * @return Result of the DescribeCollection operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.DescribeCollection
     */
    @Override
    public DescribeCollectionResponse describeCollection(DescribeCollectionRequest describeCollectionRequest)
            throws InvalidParameterException, AccessDeniedException, InternalServerErrorException, ThrottlingException,
            ProvisionedThroughputExceededException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            RekognitionException {

        HttpResponseHandler<DescribeCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeCollectionResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<DescribeCollectionRequest, DescribeCollectionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(describeCollectionRequest).withMarshaller(new DescribeCollectionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides information about a stream processor created by . You can get information about the input and output
     * streams, the input parameters for the face recognition being performed, and the current status of the stream
     * processor.
     * </p>
     *
     * @param describeStreamProcessorRequest
     * @return Result of the DescribeStreamProcessor operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.DescribeStreamProcessor
     */
    @Override
    public DescribeStreamProcessorResponse describeStreamProcessor(DescribeStreamProcessorRequest describeStreamProcessorRequest)
            throws AccessDeniedException, InternalServerErrorException, ThrottlingException, InvalidParameterException,
            ResourceNotFoundException, ProvisionedThroughputExceededException, AwsServiceException, SdkClientException,
            RekognitionException {

        HttpResponseHandler<DescribeStreamProcessorResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeStreamProcessorResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<DescribeStreamProcessorRequest, DescribeStreamProcessorResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(describeStreamProcessorRequest)
                .withMarshaller(new DescribeStreamProcessorRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Detects faces within an image that is provided as input.
     * </p>
     * <p>
     * <code>DetectFaces</code> detects the 100 largest faces in the image. For each face detected, the operation
     * returns face details including a bounding box of the face, a confidence value (that the bounding box contains a
     * face), and a fixed set of attributes such as facial landmarks (for example, coordinates of eye and mouth),
     * gender, presence of beard, sunglasses, etc.
     * </p>
     * <p>
     * The face-detection algorithm is most effective on frontal faces. For non-frontal or obscured faces, the algorithm
     * may not detect the faces or might detect faces with lower confidence.
     * </p>
     * <p>
     * You pass the input image either as base64-encoded image bytes or as a reference to an image in an Amazon S3
     * bucket. If you use the Amazon CLI to call Amazon Rekognition operations, passing image bytes is not supported.
     * The image must be either a PNG or JPEG formatted file.
     * </p>
     * <note>
     * <p>
     * This is a stateless API operation. That is, the operation does not persist any data.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:DetectFaces</code> action.
     * </p>
     *
     * @param detectFacesRequest
     * @return Result of the DetectFaces operation returned by the service.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws ImageTooLargeException
     *         The input image size exceeds the allowed limit. For more information, see Limits in Amazon Rekognition in
     *         the Amazon Rekognition Developer Guide.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.DetectFaces
     */
    @Override
    public DetectFacesResponse detectFaces(DetectFacesRequest detectFacesRequest) throws InvalidS3ObjectException,
            InvalidParameterException, ImageTooLargeException, AccessDeniedException, InternalServerErrorException,
            ThrottlingException, ProvisionedThroughputExceededException, InvalidImageFormatException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<DetectFacesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DetectFacesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<DetectFacesRequest, DetectFacesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(detectFacesRequest).withMarshaller(new DetectFacesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Detects instances of real-world entities within an image (JPEG or PNG) provided as input. This includes objects
     * like flower, tree, and table; events like wedding, graduation, and birthday party; and concepts like landscape,
     * evening, and nature.
     * </p>
     * <p>
     * For an example, see Analyzing Images Stored in an Amazon S3 Bucket in the Amazon Rekognition Developer Guide.
     * </p>
     * <note>
     * <p>
     * <code>DetectLabels</code> does not support the detection of activities. However, activity detection is supported
     * for label detection in videos. For more information, see StartLabelDetection in the Amazon Rekognition Developer
     * Guide.
     * </p>
     * </note>
     * <p>
     * You pass the input image as base64-encoded image bytes or as a reference to an image in an Amazon S3 bucket. If
     * you use the Amazon CLI to call Amazon Rekognition operations, passing image bytes is not supported. The image
     * must be either a PNG or JPEG formatted file.
     * </p>
     * <p>
     * For each object, scene, and concept the API returns one or more labels. Each label provides the object name, and
     * the level of confidence that the image contains the object. For example, suppose the input image has a
     * lighthouse, the sea, and a rock. The response will include all three labels, one for each object.
     * </p>
     * <p>
     * <code>{Name: lighthouse, Confidence: 98.4629}</code>
     * </p>
     * <p>
     * <code>{Name: rock,Confidence: 79.2097}</code>
     * </p>
     * <p>
     * <code> {Name: sea,Confidence: 75.061}</code>
     * </p>
     * <p>
     * In the preceding example, the operation returns one label for each of the three objects. The operation can also
     * return multiple labels for the same object in the image. For example, if the input image shows a flower (for
     * example, a tulip), the operation might return the following three labels.
     * </p>
     * <p>
     * <code>{Name: flower,Confidence: 99.0562}</code>
     * </p>
     * <p>
     * <code>{Name: plant,Confidence: 99.0562}</code>
     * </p>
     * <p>
     * <code>{Name: tulip,Confidence: 99.0562}</code>
     * </p>
     * <p>
     * In this example, the detection algorithm more precisely identifies the flower as a tulip.
     * </p>
     * <p>
     * In response, the API returns an array of labels. In addition, the response also includes the orientation
     * correction. Optionally, you can specify <code>MinConfidence</code> to control the confidence threshold for the
     * labels returned. The default is 50%. You can also add the <code>MaxLabels</code> parameter to limit the number of
     * labels returned.
     * </p>
     * <note>
     * <p>
     * If the object detected is a person, the operation doesn't provide the same facial details that the
     * <a>DetectFaces</a> operation provides.
     * </p>
     * </note>
     * <p>
     * This is a stateless API operation. That is, the operation does not persist any data.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:DetectLabels</code> action.
     * </p>
     *
     * @param detectLabelsRequest
     * @return Result of the DetectLabels operation returned by the service.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws ImageTooLargeException
     *         The input image size exceeds the allowed limit. For more information, see Limits in Amazon Rekognition in
     *         the Amazon Rekognition Developer Guide.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.DetectLabels
     */
    @Override
    public DetectLabelsResponse detectLabels(DetectLabelsRequest detectLabelsRequest) throws InvalidS3ObjectException,
            InvalidParameterException, ImageTooLargeException, AccessDeniedException, InternalServerErrorException,
            ThrottlingException, ProvisionedThroughputExceededException, InvalidImageFormatException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<DetectLabelsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DetectLabelsResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<DetectLabelsRequest, DetectLabelsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(detectLabelsRequest).withMarshaller(new DetectLabelsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Detects explicit or suggestive adult content in a specified JPEG or PNG format image. Use
     * <code>DetectModerationLabels</code> to moderate images depending on your requirements. For example, you might
     * want to filter images that contain nudity, but not images containing suggestive content.
     * </p>
     * <p>
     * To filter images, use the labels returned by <code>DetectModerationLabels</code> to determine which types of
     * content are appropriate.
     * </p>
     * <p>
     * For information about moderation labels, see Detecting Unsafe Content in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * You pass the input image either as base64-encoded image bytes or as a reference to an image in an Amazon S3
     * bucket. If you use the Amazon CLI to call Amazon Rekognition operations, passing image bytes is not supported.
     * The image must be either a PNG or JPEG formatted file.
     * </p>
     *
     * @param detectModerationLabelsRequest
     * @return Result of the DetectModerationLabels operation returned by the service.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws ImageTooLargeException
     *         The input image size exceeds the allowed limit. For more information, see Limits in Amazon Rekognition in
     *         the Amazon Rekognition Developer Guide.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.DetectModerationLabels
     */
    @Override
    public DetectModerationLabelsResponse detectModerationLabels(DetectModerationLabelsRequest detectModerationLabelsRequest)
            throws InvalidS3ObjectException, InvalidParameterException, ImageTooLargeException, AccessDeniedException,
            InternalServerErrorException, ThrottlingException, ProvisionedThroughputExceededException,
            InvalidImageFormatException, AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<DetectModerationLabelsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DetectModerationLabelsResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<DetectModerationLabelsRequest, DetectModerationLabelsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(detectModerationLabelsRequest)
                .withMarshaller(new DetectModerationLabelsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Detects text in the input image and converts it into machine-readable text.
     * </p>
     * <p>
     * Pass the input image as base64-encoded image bytes or as a reference to an image in an Amazon S3 bucket. If you
     * use the AWS CLI to call Amazon Rekognition operations, you must pass it as a reference to an image in an Amazon
     * S3 bucket. For the AWS CLI, passing image bytes is not supported. The image must be either a .png or .jpeg
     * formatted file.
     * </p>
     * <p>
     * The <code>DetectText</code> operation returns text in an array of elements, <code>TextDetections</code>. Each
     * <code>TextDetection</code> element provides information about a single word or line of text that was detected in
     * the image.
     * </p>
     * <p>
     * A word is one or more ISO basic latin script characters that are not separated by spaces. <code>DetectText</code>
     * can detect up to 50 words in an image.
     * </p>
     * <p>
     * A line is a string of equally spaced words. A line isn't necessarily a complete sentence. For example, a driver's
     * license number is detected as a line. A line ends when there is no aligned text after it. Also, a line ends when
     * there is a large gap between words, relative to the length of the words. This means, depending on the gap between
     * words, Amazon Rekognition may detect multiple lines in text aligned in the same direction. Periods don't
     * represent the end of a line. If a sentence spans multiple lines, the <code>DetectText</code> operation returns
     * multiple lines.
     * </p>
     * <p>
     * To determine whether a <code>TextDetection</code> element is a line of text or a word, use the
     * <code>TextDetection</code> object <code>Type</code> field.
     * </p>
     * <p>
     * To be detected, text must be within +/- 90 degrees orientation of the horizontal axis.
     * </p>
     * <p>
     * For more information, see DetectText in the Amazon Rekognition Developer Guide.
     * </p>
     *
     * @param detectTextRequest
     * @return Result of the DetectText operation returned by the service.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws ImageTooLargeException
     *         The input image size exceeds the allowed limit. For more information, see Limits in Amazon Rekognition in
     *         the Amazon Rekognition Developer Guide.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.DetectText
     */
    @Override
    public DetectTextResponse detectText(DetectTextRequest detectTextRequest) throws InvalidS3ObjectException,
            InvalidParameterException, ImageTooLargeException, AccessDeniedException, InternalServerErrorException,
            ThrottlingException, ProvisionedThroughputExceededException, InvalidImageFormatException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<DetectTextResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DetectTextResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<DetectTextRequest, DetectTextResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(detectTextRequest)
                .withMarshaller(new DetectTextRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the name and additional information about a celebrity based on his or her Rekognition ID. The additional
     * information is returned as an array of URLs. If there is no additional information about the celebrity, this list
     * is empty.
     * </p>
     * <p>
     * For more information, see Recognizing Celebrities in an Image in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:GetCelebrityInfo</code> action.
     * </p>
     *
     * @param getCelebrityInfoRequest
     * @return Result of the GetCelebrityInfo operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.GetCelebrityInfo
     */
    @Override
    public GetCelebrityInfoResponse getCelebrityInfo(GetCelebrityInfoRequest getCelebrityInfoRequest)
            throws InvalidParameterException, AccessDeniedException, InternalServerErrorException, ThrottlingException,
            ProvisionedThroughputExceededException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            RekognitionException {

        HttpResponseHandler<GetCelebrityInfoResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetCelebrityInfoResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<GetCelebrityInfoRequest, GetCelebrityInfoResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getCelebrityInfoRequest).withMarshaller(new GetCelebrityInfoRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the celebrity recognition results for a Amazon Rekognition Video analysis started by .
     * </p>
     * <p>
     * Celebrity recognition in a video is an asynchronous operation. Analysis is started by a call to which returns a
     * job identifier (<code>JobId</code>). When the celebrity recognition operation finishes, Amazon Rekognition Video
     * publishes a completion status to the Amazon Simple Notification Service topic registered in the initial call to
     * <code>StartCelebrityRecognition</code>. To get the results of the celebrity recognition analysis, first check
     * that the status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetCelebrityDetection</code> and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartCelebrityDetection</code>.
     * </p>
     * <p>
     * For more information, see Working With Stored Videos in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * <code>GetCelebrityRecognition</code> returns detected celebrities and the time(s) they are detected in an array (
     * <code>Celebrities</code>) of objects. Each <code>CelebrityRecognition</code> contains information about the
     * celebrity in a object and the time, <code>Timestamp</code>, the celebrity was detected.
     * </p>
     * <note>
     * <p>
     * <code>GetCelebrityRecognition</code> only returns the default facial attributes (<code>BoundingBox</code>,
     * <code>Confidence</code>, <code>Landmarks</code>, <code>Pose</code>, and <code>Quality</code>). The other facial
     * attributes listed in the <code>Face</code> object of the following response syntax are not returned. For more
     * information, see FaceDetail in the Amazon Rekognition Developer Guide.
     * </p>
     * </note>
     * <p>
     * By default, the <code>Celebrities</code> array is sorted by time (milliseconds from the start of the video). You
     * can also sort the array by celebrity by specifying the value <code>ID</code> in the <code>SortBy</code> input
     * parameter.
     * </p>
     * <p>
     * The <code>CelebrityDetail</code> object includes the celebrity identifer and additional information urls. If you
     * don't store the additional information urls, you can get them later by calling with the celebrity identifer.
     * </p>
     * <p>
     * No information is returned for faces not recognized as celebrities.
     * </p>
     * <p>
     * Use MaxResults parameter to limit the number of labels returned. If there are more results than specified in
     * <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a pagination
     * token for getting the next set of results. To get the next page of results, call
     * <code>GetCelebrityDetection</code> and populate the <code>NextToken</code> request parameter with the token value
     * returned from the previous call to <code>GetCelebrityRecognition</code>.
     * </p>
     *
     * @param getCelebrityRecognitionRequest
     * @return Result of the GetCelebrityRecognition operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.GetCelebrityRecognition
     */
    @Override
    public GetCelebrityRecognitionResponse getCelebrityRecognition(GetCelebrityRecognitionRequest getCelebrityRecognitionRequest)
            throws AccessDeniedException, InternalServerErrorException, InvalidParameterException,
            InvalidPaginationTokenException, ProvisionedThroughputExceededException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<GetCelebrityRecognitionResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetCelebrityRecognitionResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<GetCelebrityRecognitionRequest, GetCelebrityRecognitionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getCelebrityRecognitionRequest)
                .withMarshaller(new GetCelebrityRecognitionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the celebrity recognition results for a Amazon Rekognition Video analysis started by .
     * </p>
     * <p>
     * Celebrity recognition in a video is an asynchronous operation. Analysis is started by a call to which returns a
     * job identifier (<code>JobId</code>). When the celebrity recognition operation finishes, Amazon Rekognition Video
     * publishes a completion status to the Amazon Simple Notification Service topic registered in the initial call to
     * <code>StartCelebrityRecognition</code>. To get the results of the celebrity recognition analysis, first check
     * that the status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetCelebrityDetection</code> and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartCelebrityDetection</code>.
     * </p>
     * <p>
     * For more information, see Working With Stored Videos in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * <code>GetCelebrityRecognition</code> returns detected celebrities and the time(s) they are detected in an array (
     * <code>Celebrities</code>) of objects. Each <code>CelebrityRecognition</code> contains information about the
     * celebrity in a object and the time, <code>Timestamp</code>, the celebrity was detected.
     * </p>
     * <note>
     * <p>
     * <code>GetCelebrityRecognition</code> only returns the default facial attributes (<code>BoundingBox</code>,
     * <code>Confidence</code>, <code>Landmarks</code>, <code>Pose</code>, and <code>Quality</code>). The other facial
     * attributes listed in the <code>Face</code> object of the following response syntax are not returned. For more
     * information, see FaceDetail in the Amazon Rekognition Developer Guide.
     * </p>
     * </note>
     * <p>
     * By default, the <code>Celebrities</code> array is sorted by time (milliseconds from the start of the video). You
     * can also sort the array by celebrity by specifying the value <code>ID</code> in the <code>SortBy</code> input
     * parameter.
     * </p>
     * <p>
     * The <code>CelebrityDetail</code> object includes the celebrity identifer and additional information urls. If you
     * don't store the additional information urls, you can get them later by calling with the celebrity identifer.
     * </p>
     * <p>
     * No information is returned for faces not recognized as celebrities.
     * </p>
     * <p>
     * Use MaxResults parameter to limit the number of labels returned. If there are more results than specified in
     * <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a pagination
     * token for getting the next set of results. To get the next page of results, call
     * <code>GetCelebrityDetection</code> and populate the <code>NextToken</code> request parameter with the token value
     * returned from the previous call to <code>GetCelebrityRecognition</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getCelebrityRecognition(software.amazon.awssdk.services.rekognition.model.GetCelebrityRecognitionRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.GetCelebrityRecognitionIterable responses = client.getCelebrityRecognitionPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rekognition.paginators.GetCelebrityRecognitionIterable responses = client
     *             .getCelebrityRecognitionPaginator(request);
     *     for (software.amazon.awssdk.services.rekognition.model.GetCelebrityRecognitionResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.GetCelebrityRecognitionIterable responses = client.getCelebrityRecognitionPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getCelebrityRecognition(software.amazon.awssdk.services.rekognition.model.GetCelebrityRecognitionRequest)}
     * operation.</b>
     * </p>
     *
     * @param getCelebrityRecognitionRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.GetCelebrityRecognition
     */
    @Override
    public GetCelebrityRecognitionIterable getCelebrityRecognitionPaginator(
            GetCelebrityRecognitionRequest getCelebrityRecognitionRequest) throws AccessDeniedException,
            InternalServerErrorException, InvalidParameterException, InvalidPaginationTokenException,
            ProvisionedThroughputExceededException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, RekognitionException {
        return new GetCelebrityRecognitionIterable(this, applyPaginatorUserAgent(getCelebrityRecognitionRequest));
    }

    /**
     * <p>
     * Gets the content moderation analysis results for a Amazon Rekognition Video analysis started by .
     * </p>
     * <p>
     * Content moderation analysis of a video is an asynchronous operation. You start analysis by calling . which
     * returns a job identifier (<code>JobId</code>). When analysis finishes, Amazon Rekognition Video publishes a
     * completion status to the Amazon Simple Notification Service topic registered in the initial call to
     * <code>StartContentModeration</code>. To get the results of the content moderation analysis, first check that the
     * status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetCelebrityDetection</code> and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartCelebrityDetection</code>.
     * </p>
     * <p>
     * For more information, see Working with Stored Videos in the Amazon Rekognition Devlopers Guide.
     * </p>
     * <p>
     * <code>GetContentModeration</code> returns detected content moderation labels, and the time they are detected, in
     * an array, <code>ModerationLabels</code>, of objects.
     * </p>
     * <p>
     * By default, the moderated labels are returned sorted by time, in milliseconds from the start of the video. You
     * can also sort them by moderated label by specifying <code>NAME</code> for the <code>SortBy</code> input
     * parameter.
     * </p>
     * <p>
     * Since video analysis can return a large number of results, use the <code>MaxResults</code> parameter to limit the
     * number of labels returned in a single call to <code>GetContentModeration</code>. If there are more results than
     * specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a
     * pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetContentModeration</code> and populate the <code>NextToken</code> request parameter with the value of
     * <code>NextToken</code> returned from the previous call to <code>GetContentModeration</code>.
     * </p>
     * <p>
     * For more information, see Detecting Unsafe Content in the Amazon Rekognition Developer Guide.
     * </p>
     *
     * @param getContentModerationRequest
     * @return Result of the GetContentModeration operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.GetContentModeration
     */
    @Override
    public GetContentModerationResponse getContentModeration(GetContentModerationRequest getContentModerationRequest)
            throws AccessDeniedException, InternalServerErrorException, InvalidParameterException,
            InvalidPaginationTokenException, ProvisionedThroughputExceededException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<GetContentModerationResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetContentModerationResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<GetContentModerationRequest, GetContentModerationResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getContentModerationRequest)
                .withMarshaller(new GetContentModerationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the content moderation analysis results for a Amazon Rekognition Video analysis started by .
     * </p>
     * <p>
     * Content moderation analysis of a video is an asynchronous operation. You start analysis by calling . which
     * returns a job identifier (<code>JobId</code>). When analysis finishes, Amazon Rekognition Video publishes a
     * completion status to the Amazon Simple Notification Service topic registered in the initial call to
     * <code>StartContentModeration</code>. To get the results of the content moderation analysis, first check that the
     * status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetCelebrityDetection</code> and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartCelebrityDetection</code>.
     * </p>
     * <p>
     * For more information, see Working with Stored Videos in the Amazon Rekognition Devlopers Guide.
     * </p>
     * <p>
     * <code>GetContentModeration</code> returns detected content moderation labels, and the time they are detected, in
     * an array, <code>ModerationLabels</code>, of objects.
     * </p>
     * <p>
     * By default, the moderated labels are returned sorted by time, in milliseconds from the start of the video. You
     * can also sort them by moderated label by specifying <code>NAME</code> for the <code>SortBy</code> input
     * parameter.
     * </p>
     * <p>
     * Since video analysis can return a large number of results, use the <code>MaxResults</code> parameter to limit the
     * number of labels returned in a single call to <code>GetContentModeration</code>. If there are more results than
     * specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a
     * pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetContentModeration</code> and populate the <code>NextToken</code> request parameter with the value of
     * <code>NextToken</code> returned from the previous call to <code>GetContentModeration</code>.
     * </p>
     * <p>
     * For more information, see Detecting Unsafe Content in the Amazon Rekognition Developer Guide.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getContentModeration(software.amazon.awssdk.services.rekognition.model.GetContentModerationRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.GetContentModerationIterable responses = client.getContentModerationPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rekognition.paginators.GetContentModerationIterable responses = client
     *             .getContentModerationPaginator(request);
     *     for (software.amazon.awssdk.services.rekognition.model.GetContentModerationResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.GetContentModerationIterable responses = client.getContentModerationPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getContentModeration(software.amazon.awssdk.services.rekognition.model.GetContentModerationRequest)}
     * operation.</b>
     * </p>
     *
     * @param getContentModerationRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.GetContentModeration
     */
    @Override
    public GetContentModerationIterable getContentModerationPaginator(GetContentModerationRequest getContentModerationRequest)
            throws AccessDeniedException, InternalServerErrorException, InvalidParameterException,
            InvalidPaginationTokenException, ProvisionedThroughputExceededException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, RekognitionException {
        return new GetContentModerationIterable(this, applyPaginatorUserAgent(getContentModerationRequest));
    }

    /**
     * <p>
     * Gets face detection results for a Amazon Rekognition Video analysis started by .
     * </p>
     * <p>
     * Face detection with Amazon Rekognition Video is an asynchronous operation. You start face detection by calling
     * which returns a job identifier (<code>JobId</code>). When the face detection operation finishes, Amazon
     * Rekognition Video publishes a completion status to the Amazon Simple Notification Service topic registered in the
     * initial call to <code>StartFaceDetection</code>. To get the results of the face detection operation, first check
     * that the status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call and pass the job
     * identifier (<code>JobId</code>) from the initial call to <code>StartFaceDetection</code>.
     * </p>
     * <p>
     * <code>GetFaceDetection</code> returns an array of detected faces (<code>Faces</code>) sorted by the time the
     * faces were detected.
     * </p>
     * <p>
     * Use MaxResults parameter to limit the number of labels returned. If there are more results than specified in
     * <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a pagination
     * token for getting the next set of results. To get the next page of results, call <code>GetFaceDetection</code>
     * and populate the <code>NextToken</code> request parameter with the token value returned from the previous call to
     * <code>GetFaceDetection</code>.
     * </p>
     *
     * @param getFaceDetectionRequest
     * @return Result of the GetFaceDetection operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.GetFaceDetection
     */
    @Override
    public GetFaceDetectionResponse getFaceDetection(GetFaceDetectionRequest getFaceDetectionRequest)
            throws AccessDeniedException, InternalServerErrorException, InvalidParameterException,
            InvalidPaginationTokenException, ProvisionedThroughputExceededException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<GetFaceDetectionResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetFaceDetectionResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<GetFaceDetectionRequest, GetFaceDetectionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getFaceDetectionRequest).withMarshaller(new GetFaceDetectionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets face detection results for a Amazon Rekognition Video analysis started by .
     * </p>
     * <p>
     * Face detection with Amazon Rekognition Video is an asynchronous operation. You start face detection by calling
     * which returns a job identifier (<code>JobId</code>). When the face detection operation finishes, Amazon
     * Rekognition Video publishes a completion status to the Amazon Simple Notification Service topic registered in the
     * initial call to <code>StartFaceDetection</code>. To get the results of the face detection operation, first check
     * that the status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call and pass the job
     * identifier (<code>JobId</code>) from the initial call to <code>StartFaceDetection</code>.
     * </p>
     * <p>
     * <code>GetFaceDetection</code> returns an array of detected faces (<code>Faces</code>) sorted by the time the
     * faces were detected.
     * </p>
     * <p>
     * Use MaxResults parameter to limit the number of labels returned. If there are more results than specified in
     * <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a pagination
     * token for getting the next set of results. To get the next page of results, call <code>GetFaceDetection</code>
     * and populate the <code>NextToken</code> request parameter with the token value returned from the previous call to
     * <code>GetFaceDetection</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getFaceDetection(software.amazon.awssdk.services.rekognition.model.GetFaceDetectionRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.GetFaceDetectionIterable responses = client.getFaceDetectionPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rekognition.paginators.GetFaceDetectionIterable responses = client
     *             .getFaceDetectionPaginator(request);
     *     for (software.amazon.awssdk.services.rekognition.model.GetFaceDetectionResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.GetFaceDetectionIterable responses = client.getFaceDetectionPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getFaceDetection(software.amazon.awssdk.services.rekognition.model.GetFaceDetectionRequest)}
     * operation.</b>
     * </p>
     *
     * @param getFaceDetectionRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.GetFaceDetection
     */
    @Override
    public GetFaceDetectionIterable getFaceDetectionPaginator(GetFaceDetectionRequest getFaceDetectionRequest)
            throws AccessDeniedException, InternalServerErrorException, InvalidParameterException,
            InvalidPaginationTokenException, ProvisionedThroughputExceededException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, RekognitionException {
        return new GetFaceDetectionIterable(this, applyPaginatorUserAgent(getFaceDetectionRequest));
    }

    /**
     * <p>
     * Gets the face search results for Amazon Rekognition Video face search started by . The search returns faces in a
     * collection that match the faces of persons detected in a video. It also includes the time(s) that faces are
     * matched in the video.
     * </p>
     * <p>
     * Face search in a video is an asynchronous operation. You start face search by calling to which returns a job
     * identifier (<code>JobId</code>). When the search operation finishes, Amazon Rekognition Video publishes a
     * completion status to the Amazon Simple Notification Service topic registered in the initial call to
     * <code>StartFaceSearch</code>. To get the search results, first check that the status value published to the
     * Amazon SNS topic is <code>SUCCEEDED</code>. If so, call <code>GetFaceSearch</code> and pass the job identifier (
     * <code>JobId</code>) from the initial call to <code>StartFaceSearch</code>.
     * </p>
     * <p>
     * For more information, see Searching Faces in a Collection in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * The search results are retured in an array, <code>Persons</code>, of objects. Each<code>PersonMatch</code>
     * element contains details about the matching faces in the input collection, person information (facial attributes,
     * bounding boxes, and person identifer) for the matched person, and the time the person was matched in the video.
     * </p>
     * <note>
     * <p>
     * <code>GetFaceSearch</code> only returns the default facial attributes (<code>BoundingBox</code>,
     * <code>Confidence</code>, <code>Landmarks</code>, <code>Pose</code>, and <code>Quality</code>). The other facial
     * attributes listed in the <code>Face</code> object of the following response syntax are not returned. For more
     * information, see FaceDetail in the Amazon Rekognition Developer Guide.
     * </p>
     * </note>
     * <p>
     * By default, the <code>Persons</code> array is sorted by the time, in milliseconds from the start of the video,
     * persons are matched. You can also sort by persons by specifying <code>INDEX</code> for the <code>SORTBY</code>
     * input parameter.
     * </p>
     *
     * @param getFaceSearchRequest
     * @return Result of the GetFaceSearch operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.GetFaceSearch
     */
    @Override
    public GetFaceSearchResponse getFaceSearch(GetFaceSearchRequest getFaceSearchRequest) throws AccessDeniedException,
            InternalServerErrorException, InvalidParameterException, InvalidPaginationTokenException,
            ProvisionedThroughputExceededException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<GetFaceSearchResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetFaceSearchResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<GetFaceSearchRequest, GetFaceSearchResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getFaceSearchRequest).withMarshaller(new GetFaceSearchRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the face search results for Amazon Rekognition Video face search started by . The search returns faces in a
     * collection that match the faces of persons detected in a video. It also includes the time(s) that faces are
     * matched in the video.
     * </p>
     * <p>
     * Face search in a video is an asynchronous operation. You start face search by calling to which returns a job
     * identifier (<code>JobId</code>). When the search operation finishes, Amazon Rekognition Video publishes a
     * completion status to the Amazon Simple Notification Service topic registered in the initial call to
     * <code>StartFaceSearch</code>. To get the search results, first check that the status value published to the
     * Amazon SNS topic is <code>SUCCEEDED</code>. If so, call <code>GetFaceSearch</code> and pass the job identifier (
     * <code>JobId</code>) from the initial call to <code>StartFaceSearch</code>.
     * </p>
     * <p>
     * For more information, see Searching Faces in a Collection in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * The search results are retured in an array, <code>Persons</code>, of objects. Each<code>PersonMatch</code>
     * element contains details about the matching faces in the input collection, person information (facial attributes,
     * bounding boxes, and person identifer) for the matched person, and the time the person was matched in the video.
     * </p>
     * <note>
     * <p>
     * <code>GetFaceSearch</code> only returns the default facial attributes (<code>BoundingBox</code>,
     * <code>Confidence</code>, <code>Landmarks</code>, <code>Pose</code>, and <code>Quality</code>). The other facial
     * attributes listed in the <code>Face</code> object of the following response syntax are not returned. For more
     * information, see FaceDetail in the Amazon Rekognition Developer Guide.
     * </p>
     * </note>
     * <p>
     * By default, the <code>Persons</code> array is sorted by the time, in milliseconds from the start of the video,
     * persons are matched. You can also sort by persons by specifying <code>INDEX</code> for the <code>SORTBY</code>
     * input parameter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getFaceSearch(software.amazon.awssdk.services.rekognition.model.GetFaceSearchRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.GetFaceSearchIterable responses = client.getFaceSearchPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rekognition.paginators.GetFaceSearchIterable responses = client
     *             .getFaceSearchPaginator(request);
     *     for (software.amazon.awssdk.services.rekognition.model.GetFaceSearchResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.GetFaceSearchIterable responses = client.getFaceSearchPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getFaceSearch(software.amazon.awssdk.services.rekognition.model.GetFaceSearchRequest)} operation.</b>
     * </p>
     *
     * @param getFaceSearchRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.GetFaceSearch
     */
    @Override
    public GetFaceSearchIterable getFaceSearchPaginator(GetFaceSearchRequest getFaceSearchRequest) throws AccessDeniedException,
            InternalServerErrorException, InvalidParameterException, InvalidPaginationTokenException,
            ProvisionedThroughputExceededException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, RekognitionException {
        return new GetFaceSearchIterable(this, applyPaginatorUserAgent(getFaceSearchRequest));
    }

    /**
     * <p>
     * Gets the label detection results of a Amazon Rekognition Video analysis started by .
     * </p>
     * <p>
     * The label detection operation is started by a call to which returns a job identifier (<code>JobId</code>). When
     * the label detection operation finishes, Amazon Rekognition publishes a completion status to the Amazon Simple
     * Notification Service topic registered in the initial call to <code>StartlabelDetection</code>. To get the results
     * of the label detection operation, first check that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartLabelDetection</code>.
     * </p>
     * <p>
     * <code>GetLabelDetection</code> returns an array of detected labels (<code>Labels</code>) sorted by the time the
     * labels were detected. You can also sort by the label name by specifying <code>NAME</code> for the
     * <code>SortBy</code> input parameter.
     * </p>
     * <p>
     * The labels returned include the label name, the percentage confidence in the accuracy of the detected label, and
     * the time the label was detected in the video.
     * </p>
     * <p>
     * Use MaxResults parameter to limit the number of labels returned. If there are more results than specified in
     * <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a pagination
     * token for getting the next set of results. To get the next page of results, call <code>GetlabelDetection</code>
     * and populate the <code>NextToken</code> request parameter with the token value returned from the previous call to
     * <code>GetLabelDetection</code>.
     * </p>
     *
     * @param getLabelDetectionRequest
     * @return Result of the GetLabelDetection operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.GetLabelDetection
     */
    @Override
    public GetLabelDetectionResponse getLabelDetection(GetLabelDetectionRequest getLabelDetectionRequest)
            throws AccessDeniedException, InternalServerErrorException, InvalidParameterException,
            InvalidPaginationTokenException, ProvisionedThroughputExceededException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<GetLabelDetectionResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetLabelDetectionResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<GetLabelDetectionRequest, GetLabelDetectionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getLabelDetectionRequest).withMarshaller(new GetLabelDetectionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the label detection results of a Amazon Rekognition Video analysis started by .
     * </p>
     * <p>
     * The label detection operation is started by a call to which returns a job identifier (<code>JobId</code>). When
     * the label detection operation finishes, Amazon Rekognition publishes a completion status to the Amazon Simple
     * Notification Service topic registered in the initial call to <code>StartlabelDetection</code>. To get the results
     * of the label detection operation, first check that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartLabelDetection</code>.
     * </p>
     * <p>
     * <code>GetLabelDetection</code> returns an array of detected labels (<code>Labels</code>) sorted by the time the
     * labels were detected. You can also sort by the label name by specifying <code>NAME</code> for the
     * <code>SortBy</code> input parameter.
     * </p>
     * <p>
     * The labels returned include the label name, the percentage confidence in the accuracy of the detected label, and
     * the time the label was detected in the video.
     * </p>
     * <p>
     * Use MaxResults parameter to limit the number of labels returned. If there are more results than specified in
     * <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a pagination
     * token for getting the next set of results. To get the next page of results, call <code>GetlabelDetection</code>
     * and populate the <code>NextToken</code> request parameter with the token value returned from the previous call to
     * <code>GetLabelDetection</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getLabelDetection(software.amazon.awssdk.services.rekognition.model.GetLabelDetectionRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.GetLabelDetectionIterable responses = client.getLabelDetectionPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rekognition.paginators.GetLabelDetectionIterable responses = client
     *             .getLabelDetectionPaginator(request);
     *     for (software.amazon.awssdk.services.rekognition.model.GetLabelDetectionResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.GetLabelDetectionIterable responses = client.getLabelDetectionPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getLabelDetection(software.amazon.awssdk.services.rekognition.model.GetLabelDetectionRequest)}
     * operation.</b>
     * </p>
     *
     * @param getLabelDetectionRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.GetLabelDetection
     */
    @Override
    public GetLabelDetectionIterable getLabelDetectionPaginator(GetLabelDetectionRequest getLabelDetectionRequest)
            throws AccessDeniedException, InternalServerErrorException, InvalidParameterException,
            InvalidPaginationTokenException, ProvisionedThroughputExceededException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, RekognitionException {
        return new GetLabelDetectionIterable(this, applyPaginatorUserAgent(getLabelDetectionRequest));
    }

    /**
     * <p>
     * Gets the person tracking results of a Amazon Rekognition Video analysis started by .
     * </p>
     * <p>
     * The person detection operation is started by a call to <code>StartPersonTracking</code> which returns a job
     * identifier (<code>JobId</code>). When the person detection operation finishes, Amazon Rekognition Video publishes
     * a completion status to the Amazon Simple Notification Service topic registered in the initial call to
     * <code>StartPersonTracking</code>.
     * </p>
     * <p>
     * To get the results of the person tracking operation, first check that the status value published to the Amazon
     * SNS topic is <code>SUCCEEDED</code>. If so, call and pass the job identifier (<code>JobId</code>) from the
     * initial call to <code>StartPersonTracking</code>.
     * </p>
     * <p>
     * <code>GetPersonTracking</code> returns an array, <code>Persons</code>, of tracked persons and the time(s) they
     * were tracked in the video.
     * </p>
     * <note>
     * <p>
     * <code>GetPersonTracking</code> only returns the default facial attributes (<code>BoundingBox</code>,
     * <code>Confidence</code>, <code>Landmarks</code>, <code>Pose</code>, and <code>Quality</code>). The other facial
     * attributes listed in the <code>Face</code> object of the following response syntax are not returned.
     * </p>
     * <p>
     * For more information, see FaceDetail in the Amazon Rekognition Developer Guide.
     * </p>
     * </note>
     * <p>
     * By default, the array is sorted by the time(s) a person is tracked in the video. You can sort by tracked persons
     * by specifying <code>INDEX</code> for the <code>SortBy</code> input parameter.
     * </p>
     * <p>
     * Use the <code>MaxResults</code> parameter to limit the number of items returned. If there are more results than
     * specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a
     * pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetPersonTracking</code> and populate the <code>NextToken</code> request parameter with the token value
     * returned from the previous call to <code>GetPersonTracking</code>.
     * </p>
     *
     * @param getPersonTrackingRequest
     * @return Result of the GetPersonTracking operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.GetPersonTracking
     */
    @Override
    public GetPersonTrackingResponse getPersonTracking(GetPersonTrackingRequest getPersonTrackingRequest)
            throws AccessDeniedException, InternalServerErrorException, InvalidParameterException,
            InvalidPaginationTokenException, ProvisionedThroughputExceededException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<GetPersonTrackingResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetPersonTrackingResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<GetPersonTrackingRequest, GetPersonTrackingResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getPersonTrackingRequest).withMarshaller(new GetPersonTrackingRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the person tracking results of a Amazon Rekognition Video analysis started by .
     * </p>
     * <p>
     * The person detection operation is started by a call to <code>StartPersonTracking</code> which returns a job
     * identifier (<code>JobId</code>). When the person detection operation finishes, Amazon Rekognition Video publishes
     * a completion status to the Amazon Simple Notification Service topic registered in the initial call to
     * <code>StartPersonTracking</code>.
     * </p>
     * <p>
     * To get the results of the person tracking operation, first check that the status value published to the Amazon
     * SNS topic is <code>SUCCEEDED</code>. If so, call and pass the job identifier (<code>JobId</code>) from the
     * initial call to <code>StartPersonTracking</code>.
     * </p>
     * <p>
     * <code>GetPersonTracking</code> returns an array, <code>Persons</code>, of tracked persons and the time(s) they
     * were tracked in the video.
     * </p>
     * <note>
     * <p>
     * <code>GetPersonTracking</code> only returns the default facial attributes (<code>BoundingBox</code>,
     * <code>Confidence</code>, <code>Landmarks</code>, <code>Pose</code>, and <code>Quality</code>). The other facial
     * attributes listed in the <code>Face</code> object of the following response syntax are not returned.
     * </p>
     * <p>
     * For more information, see FaceDetail in the Amazon Rekognition Developer Guide.
     * </p>
     * </note>
     * <p>
     * By default, the array is sorted by the time(s) a person is tracked in the video. You can sort by tracked persons
     * by specifying <code>INDEX</code> for the <code>SortBy</code> input parameter.
     * </p>
     * <p>
     * Use the <code>MaxResults</code> parameter to limit the number of items returned. If there are more results than
     * specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a
     * pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetPersonTracking</code> and populate the <code>NextToken</code> request parameter with the token value
     * returned from the previous call to <code>GetPersonTracking</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getPersonTracking(software.amazon.awssdk.services.rekognition.model.GetPersonTrackingRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.GetPersonTrackingIterable responses = client.getPersonTrackingPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rekognition.paginators.GetPersonTrackingIterable responses = client
     *             .getPersonTrackingPaginator(request);
     *     for (software.amazon.awssdk.services.rekognition.model.GetPersonTrackingResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.GetPersonTrackingIterable responses = client.getPersonTrackingPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getPersonTracking(software.amazon.awssdk.services.rekognition.model.GetPersonTrackingRequest)}
     * operation.</b>
     * </p>
     *
     * @param getPersonTrackingRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.GetPersonTracking
     */
    @Override
    public GetPersonTrackingIterable getPersonTrackingPaginator(GetPersonTrackingRequest getPersonTrackingRequest)
            throws AccessDeniedException, InternalServerErrorException, InvalidParameterException,
            InvalidPaginationTokenException, ProvisionedThroughputExceededException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, RekognitionException {
        return new GetPersonTrackingIterable(this, applyPaginatorUserAgent(getPersonTrackingRequest));
    }

    /**
     * <p>
     * Detects faces in the input image and adds them to the specified collection.
     * </p>
     * <p>
     * Amazon Rekognition does not save the actual faces detected. Instead, the underlying detection algorithm first
     * detects the faces in the input image, and for each face extracts facial features into a feature vector, and
     * stores it in the back-end database. Amazon Rekognition uses feature vectors when performing face match and search
     * operations using the and operations.
     * </p>
     * <p>
     * To get the number of faces in a collection, call .
     * </p>
     * <p>
     * If you are using version 1.0 of the face detection model, <code>IndexFaces</code> indexes the 15 largest faces in
     * the input image. Later versions of the face detection model index the 100 largest faces in the input image. To
     * determine which version of the model you are using, call and supply the collection ID. You also get the model
     * version from the value of <code>FaceModelVersion</code> in the response from <code>IndexFaces</code>.
     * </p>
     * <p>
     * For more information, see Model Versioning in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * If you provide the optional <code>ExternalImageID</code> for the input image you provided, Amazon Rekognition
     * associates this ID with all faces that it detects. When you call the operation, the response returns the external
     * ID. You can use this external image ID to create a client-side index to associate the faces with each image. You
     * can then use the index to find all faces in an image.
     * </p>
     * <p>
     * In response, the operation returns an array of metadata for all detected faces. This includes, the bounding box
     * of the detected face, confidence value (indicating the bounding box contains a face), a face ID assigned by the
     * service for each face that is detected and stored, and an image ID assigned by the service for the input image.
     * If you request all facial attributes (using the <code>detectionAttributes</code> parameter, Amazon Rekognition
     * returns detailed facial attributes such as facial landmarks (for example, location of eye and mouth) and other
     * facial attributes such gender. If you provide the same image, specify the same collection, and use the same
     * external ID in the <code>IndexFaces</code> operation, Amazon Rekognition doesn't save duplicate face metadata.
     * </p>
     * <p>
     * For more information, see Adding Faces to a Collection in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * The input image is passed either as base64-encoded image bytes or as a reference to an image in an Amazon S3
     * bucket. If you use the Amazon CLI to call Amazon Rekognition operations, passing image bytes is not supported.
     * The image must be either a PNG or JPEG formatted file.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:IndexFaces</code> action.
     * </p>
     *
     * @param indexFacesRequest
     * @return Result of the IndexFaces operation returned by the service.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws ImageTooLargeException
     *         The input image size exceeds the allowed limit. For more information, see Limits in Amazon Rekognition in
     *         the Amazon Rekognition Developer Guide.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.IndexFaces
     */
    @Override
    public IndexFacesResponse indexFaces(IndexFacesRequest indexFacesRequest) throws InvalidS3ObjectException,
            InvalidParameterException, ImageTooLargeException, AccessDeniedException, InternalServerErrorException,
            ThrottlingException, ProvisionedThroughputExceededException, ResourceNotFoundException, InvalidImageFormatException,
            AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<IndexFacesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new IndexFacesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<IndexFacesRequest, IndexFacesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(indexFacesRequest)
                .withMarshaller(new IndexFacesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns list of collection IDs in your account. If the result is truncated, the response also provides a
     * <code>NextToken</code> that you can use in the subsequent request to fetch the next set of collection IDs.
     * </p>
     * <p>
     * For an example, see Listing Collections in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:ListCollections</code> action.
     * </p>
     *
     * @param listCollectionsRequest
     * @return Result of the ListCollections operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.ListCollections
     */
    @Override
    public ListCollectionsResponse listCollections(ListCollectionsRequest listCollectionsRequest)
            throws InvalidParameterException, AccessDeniedException, InternalServerErrorException, ThrottlingException,
            ProvisionedThroughputExceededException, InvalidPaginationTokenException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<ListCollectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListCollectionsResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<ListCollectionsRequest, ListCollectionsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listCollectionsRequest).withMarshaller(new ListCollectionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns list of collection IDs in your account. If the result is truncated, the response also provides a
     * <code>NextToken</code> that you can use in the subsequent request to fetch the next set of collection IDs.
     * </p>
     * <p>
     * For an example, see Listing Collections in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:ListCollections</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listCollections(software.amazon.awssdk.services.rekognition.model.ListCollectionsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.ListCollectionsIterable responses = client.listCollectionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rekognition.paginators.ListCollectionsIterable responses = client
     *             .listCollectionsPaginator(request);
     *     for (software.amazon.awssdk.services.rekognition.model.ListCollectionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.ListCollectionsIterable responses = client.listCollectionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCollections(software.amazon.awssdk.services.rekognition.model.ListCollectionsRequest)} operation.</b>
     * </p>
     *
     * @param listCollectionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.ListCollections
     */
    @Override
    public ListCollectionsIterable listCollectionsPaginator(ListCollectionsRequest listCollectionsRequest)
            throws InvalidParameterException, AccessDeniedException, InternalServerErrorException, ThrottlingException,
            ProvisionedThroughputExceededException, InvalidPaginationTokenException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, RekognitionException {
        return new ListCollectionsIterable(this, applyPaginatorUserAgent(listCollectionsRequest));
    }

    /**
     * <p>
     * Returns metadata for faces in the specified collection. This metadata includes information such as the bounding
     * box coordinates, the confidence (that the bounding box contains a face), and face ID. For an example, see Listing
     * Faces in a Collection in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:ListFaces</code> action.
     * </p>
     *
     * @param listFacesRequest
     * @return Result of the ListFaces operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.ListFaces
     */
    @Override
    public ListFacesResponse listFaces(ListFacesRequest listFacesRequest) throws InvalidParameterException,
            AccessDeniedException, InternalServerErrorException, ThrottlingException, ProvisionedThroughputExceededException,
            InvalidPaginationTokenException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            RekognitionException {

        HttpResponseHandler<ListFacesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListFacesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<ListFacesRequest, ListFacesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(listFacesRequest)
                .withMarshaller(new ListFacesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns metadata for faces in the specified collection. This metadata includes information such as the bounding
     * box coordinates, the confidence (that the bounding box contains a face), and face ID. For an example, see Listing
     * Faces in a Collection in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:ListFaces</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listFaces(software.amazon.awssdk.services.rekognition.model.ListFacesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.ListFacesIterable responses = client.listFacesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rekognition.paginators.ListFacesIterable responses = client.listFacesPaginator(request);
     *     for (software.amazon.awssdk.services.rekognition.model.ListFacesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.ListFacesIterable responses = client.listFacesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFaces(software.amazon.awssdk.services.rekognition.model.ListFacesRequest)} operation.</b>
     * </p>
     *
     * @param listFacesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.ListFaces
     */
    @Override
    public ListFacesIterable listFacesPaginator(ListFacesRequest listFacesRequest) throws InvalidParameterException,
            AccessDeniedException, InternalServerErrorException, ThrottlingException, ProvisionedThroughputExceededException,
            InvalidPaginationTokenException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            RekognitionException {
        return new ListFacesIterable(this, applyPaginatorUserAgent(listFacesRequest));
    }

    /**
     * <p>
     * Gets a list of stream processors that you have created with .
     * </p>
     *
     * @param listStreamProcessorsRequest
     * @return Result of the ListStreamProcessors operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.ListStreamProcessors
     */
    @Override
    public ListStreamProcessorsResponse listStreamProcessors(ListStreamProcessorsRequest listStreamProcessorsRequest)
            throws AccessDeniedException, InternalServerErrorException, ThrottlingException, InvalidParameterException,
            InvalidPaginationTokenException, ProvisionedThroughputExceededException, AwsServiceException, SdkClientException,
            RekognitionException {

        HttpResponseHandler<ListStreamProcessorsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListStreamProcessorsResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<ListStreamProcessorsRequest, ListStreamProcessorsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listStreamProcessorsRequest)
                .withMarshaller(new ListStreamProcessorsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list of stream processors that you have created with .
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listStreamProcessors(software.amazon.awssdk.services.rekognition.model.ListStreamProcessorsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.ListStreamProcessorsIterable responses = client.listStreamProcessorsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rekognition.paginators.ListStreamProcessorsIterable responses = client
     *             .listStreamProcessorsPaginator(request);
     *     for (software.amazon.awssdk.services.rekognition.model.ListStreamProcessorsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rekognition.paginators.ListStreamProcessorsIterable responses = client.listStreamProcessorsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStreamProcessors(software.amazon.awssdk.services.rekognition.model.ListStreamProcessorsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listStreamProcessorsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidPaginationTokenException
     *         Pagination token in the request is not valid.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.ListStreamProcessors
     */
    @Override
    public ListStreamProcessorsIterable listStreamProcessorsPaginator(ListStreamProcessorsRequest listStreamProcessorsRequest)
            throws AccessDeniedException, InternalServerErrorException, ThrottlingException, InvalidParameterException,
            InvalidPaginationTokenException, ProvisionedThroughputExceededException, AwsServiceException, SdkClientException,
            RekognitionException {
        return new ListStreamProcessorsIterable(this, applyPaginatorUserAgent(listStreamProcessorsRequest));
    }

    /**
     * <p>
     * Returns an array of celebrities recognized in the input image. For more information, see Recognizing Celebrities
     * in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * <code>RecognizeCelebrities</code> returns the 100 largest faces in the image. It lists recognized celebrities in
     * the <code>CelebrityFaces</code> array and unrecognized faces in the <code>UnrecognizedFaces</code> array.
     * <code>RecognizeCelebrities</code> doesn't return celebrities whose faces are not amongst the largest 100 faces in
     * the image.
     * </p>
     * <p>
     * For each celebrity recognized, the <code>RecognizeCelebrities</code> returns a <code>Celebrity</code> object. The
     * <code>Celebrity</code> object contains the celebrity name, ID, URL links to additional information, match
     * confidence, and a <code>ComparedFace</code> object that you can use to locate the celebrity's face on the image.
     * </p>
     * <p>
     * Rekognition does not retain information about which images a celebrity has been recognized in. Your application
     * must store this information and use the <code>Celebrity</code> ID property as a unique identifier for the
     * celebrity. If you don't store the celebrity name or additional information URLs returned by
     * <code>RecognizeCelebrities</code>, you will need the ID to identify the celebrity in a call to the operation.
     * </p>
     * <p>
     * You pass the imput image either as base64-encoded image bytes or as a reference to an image in an Amazon S3
     * bucket. If you use the Amazon CLI to call Amazon Rekognition operations, passing image bytes is not supported.
     * The image must be either a PNG or JPEG formatted file.
     * </p>
     * <p>
     * For an example, see Recognizing Celebrities in an Image in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:RecognizeCelebrities</code> operation.
     * </p>
     *
     * @param recognizeCelebritiesRequest
     * @return Result of the RecognizeCelebrities operation returned by the service.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws ImageTooLargeException
     *         The input image size exceeds the allowed limit. For more information, see Limits in Amazon Rekognition in
     *         the Amazon Rekognition Developer Guide.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.RecognizeCelebrities
     */
    @Override
    public RecognizeCelebritiesResponse recognizeCelebrities(RecognizeCelebritiesRequest recognizeCelebritiesRequest)
            throws InvalidS3ObjectException, InvalidParameterException, InvalidImageFormatException, ImageTooLargeException,
            AccessDeniedException, InternalServerErrorException, ThrottlingException, ProvisionedThroughputExceededException,
            AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<RecognizeCelebritiesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new RecognizeCelebritiesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<RecognizeCelebritiesRequest, RecognizeCelebritiesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(recognizeCelebritiesRequest)
                .withMarshaller(new RecognizeCelebritiesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * For a given input face ID, searches for matching faces in the collection the face belongs to. You get a face ID
     * when you add a face to the collection using the <a>IndexFaces</a> operation. The operation compares the features
     * of the input face with faces in the specified collection.
     * </p>
     * <note>
     * <p>
     * You can also search faces without indexing faces by using the <code>SearchFacesByImage</code> operation.
     * </p>
     * </note>
     * <p>
     * The operation response returns an array of faces that match, ordered by similarity score with the highest
     * similarity first. More specifically, it is an array of metadata for each face match that is found. Along with the
     * metadata, the response also includes a <code>confidence</code> value for each face match, indicating the
     * confidence that the specific face matches the input face.
     * </p>
     * <p>
     * For an example, see Searching for a Face Using Its Face ID in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:SearchFaces</code> action.
     * </p>
     *
     * @param searchFacesRequest
     * @return Result of the SearchFaces operation returned by the service.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.SearchFaces
     */
    @Override
    public SearchFacesResponse searchFaces(SearchFacesRequest searchFacesRequest) throws InvalidParameterException,
            AccessDeniedException, InternalServerErrorException, ThrottlingException, ProvisionedThroughputExceededException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<SearchFacesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new SearchFacesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<SearchFacesRequest, SearchFacesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(searchFacesRequest).withMarshaller(new SearchFacesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * For a given input image, first detects the largest face in the image, and then searches the specified collection
     * for matching faces. The operation compares the features of the input face with faces in the specified collection.
     * </p>
     * <note>
     * <p>
     * To search for all faces in an input image, you might first call the operation, and then use the face IDs returned
     * in subsequent calls to the operation.
     * </p>
     * <p>
     * You can also call the <code>DetectFaces</code> operation and use the bounding boxes in the response to make face
     * crops, which then you can pass in to the <code>SearchFacesByImage</code> operation.
     * </p>
     * </note>
     * <p>
     * You pass the input image either as base64-encoded image bytes or as a reference to an image in an Amazon S3
     * bucket. If you use the Amazon CLI to call Amazon Rekognition operations, passing image bytes is not supported.
     * The image must be either a PNG or JPEG formatted file.
     * </p>
     * <p>
     * The response returns an array of faces that match, ordered by similarity score with the highest similarity first.
     * More specifically, it is an array of metadata for each face match found. Along with the metadata, the response
     * also includes a <code>similarity</code> indicating how similar the face is to the input face. In the response,
     * the operation also returns the bounding box (and a confidence level that the bounding box contains a face) of the
     * face that Amazon Rekognition used for the input image.
     * </p>
     * <p>
     * For an example, Searching for a Face Using an Image in the Amazon Rekognition Developer Guide.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>rekognition:SearchFacesByImage</code> action.
     * </p>
     *
     * @param searchFacesByImageRequest
     * @return Result of the SearchFacesByImage operation returned by the service.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws ImageTooLargeException
     *         The input image size exceeds the allowed limit. For more information, see Limits in Amazon Rekognition in
     *         the Amazon Rekognition Developer Guide.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws InvalidImageFormatException
     *         The provided image format is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.SearchFacesByImage
     */
    @Override
    public SearchFacesByImageResponse searchFacesByImage(SearchFacesByImageRequest searchFacesByImageRequest)
            throws InvalidS3ObjectException, InvalidParameterException, ImageTooLargeException, AccessDeniedException,
            InternalServerErrorException, ThrottlingException, ProvisionedThroughputExceededException, ResourceNotFoundException,
            InvalidImageFormatException, AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<SearchFacesByImageResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new SearchFacesByImageResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<SearchFacesByImageRequest, SearchFacesByImageResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(searchFacesByImageRequest).withMarshaller(new SearchFacesByImageRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Starts asynchronous recognition of celebrities in a stored video.
     * </p>
     * <p>
     * Amazon Rekognition Video can detect celebrities in a video must be stored in an Amazon S3 bucket. Use
     * <a>Video</a> to specify the bucket name and the filename of the video. <code>StartCelebrityRecognition</code>
     * returns a job identifier (<code>JobId</code>) which you use to get the results of the analysis. When celebrity
     * recognition analysis is finished, Amazon Rekognition Video publishes a completion status to the Amazon Simple
     * Notification Service topic that you specify in <code>NotificationChannel</code>. To get the results of the
     * celebrity recognition analysis, first check that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartCelebrityRecognition</code>.
     * </p>
     * <p>
     * For more information, see Recognizing Celebrities in the Amazon Rekognition Developer Guide.
     * </p>
     *
     * @param startCelebrityRecognitionRequest
     * @return Result of the StartCelebrityRecognition operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws VideoTooLargeException
     *         The file size or duration of the supplied media is too large. The maximum file size is 8GB. The maximum
     *         duration is 2 hours.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws LimitExceededException
     *         An Amazon Rekognition service limit was exceeded. For example, if you start too many Amazon Rekognition
     *         Video jobs concurrently, calls to start operations (<code>StartLabelDetection</code>, for example) will
     *         raise a <code>LimitExceededException</code> exception (HTTP status code: 400) until the number of
     *         concurrently running jobs is below the Amazon Rekognition service limit.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.StartCelebrityRecognition
     */
    @Override
    public StartCelebrityRecognitionResponse startCelebrityRecognition(
            StartCelebrityRecognitionRequest startCelebrityRecognitionRequest) throws AccessDeniedException,
            IdempotentParameterMismatchException, InvalidParameterException, InvalidS3ObjectException,
            InternalServerErrorException, VideoTooLargeException, ProvisionedThroughputExceededException, LimitExceededException,
            ThrottlingException, AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<StartCelebrityRecognitionResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new StartCelebrityRecognitionResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler
                .execute(new ClientExecutionParams<StartCelebrityRecognitionRequest, StartCelebrityRecognitionResponse>()
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(startCelebrityRecognitionRequest)
                        .withMarshaller(new StartCelebrityRecognitionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Starts asynchronous detection of explicit or suggestive adult content in a stored video.
     * </p>
     * <p>
     * Amazon Rekognition Video can moderate content in a video stored in an Amazon S3 bucket. Use <a>Video</a> to
     * specify the bucket name and the filename of the video. <code>StartContentModeration</code> returns a job
     * identifier (<code>JobId</code>) which you use to get the results of the analysis. When content moderation
     * analysis is finished, Amazon Rekognition Video publishes a completion status to the Amazon Simple Notification
     * Service topic that you specify in <code>NotificationChannel</code>.
     * </p>
     * <p>
     * To get the results of the content moderation analysis, first check that the status value published to the Amazon
     * SNS topic is <code>SUCCEEDED</code>. If so, call and pass the job identifier (<code>JobId</code>) from the
     * initial call to <code>StartContentModeration</code>.
     * </p>
     * <p>
     * For more information, see Detecting Unsafe Content in the Amazon Rekognition Developer Guide.
     * </p>
     *
     * @param startContentModerationRequest
     * @return Result of the StartContentModeration operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws VideoTooLargeException
     *         The file size or duration of the supplied media is too large. The maximum file size is 8GB. The maximum
     *         duration is 2 hours.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws LimitExceededException
     *         An Amazon Rekognition service limit was exceeded. For example, if you start too many Amazon Rekognition
     *         Video jobs concurrently, calls to start operations (<code>StartLabelDetection</code>, for example) will
     *         raise a <code>LimitExceededException</code> exception (HTTP status code: 400) until the number of
     *         concurrently running jobs is below the Amazon Rekognition service limit.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.StartContentModeration
     */
    @Override
    public StartContentModerationResponse startContentModeration(StartContentModerationRequest startContentModerationRequest)
            throws AccessDeniedException, IdempotentParameterMismatchException, InvalidParameterException,
            InvalidS3ObjectException, InternalServerErrorException, VideoTooLargeException,
            ProvisionedThroughputExceededException, LimitExceededException, ThrottlingException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<StartContentModerationResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new StartContentModerationResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<StartContentModerationRequest, StartContentModerationResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(startContentModerationRequest)
                .withMarshaller(new StartContentModerationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Starts asynchronous detection of faces in a stored video.
     * </p>
     * <p>
     * Amazon Rekognition Video can detect faces in a video stored in an Amazon S3 bucket. Use <a>Video</a> to specify
     * the bucket name and the filename of the video. <code>StartFaceDetection</code> returns a job identifier (
     * <code>JobId</code>) that you use to get the results of the operation. When face detection is finished, Amazon
     * Rekognition Video publishes a completion status to the Amazon Simple Notification Service topic that you specify
     * in <code>NotificationChannel</code>. To get the results of the label detection operation, first check that the
     * status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call and pass the job identifier
     * (<code>JobId</code>) from the initial call to <code>StartFaceDetection</code>.
     * </p>
     * <p>
     * For more information, see Detecting Faces in a Stored Video in the Amazon Rekognition Developer Guide.
     * </p>
     *
     * @param startFaceDetectionRequest
     * @return Result of the StartFaceDetection operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws VideoTooLargeException
     *         The file size or duration of the supplied media is too large. The maximum file size is 8GB. The maximum
     *         duration is 2 hours.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws LimitExceededException
     *         An Amazon Rekognition service limit was exceeded. For example, if you start too many Amazon Rekognition
     *         Video jobs concurrently, calls to start operations (<code>StartLabelDetection</code>, for example) will
     *         raise a <code>LimitExceededException</code> exception (HTTP status code: 400) until the number of
     *         concurrently running jobs is below the Amazon Rekognition service limit.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.StartFaceDetection
     */
    @Override
    public StartFaceDetectionResponse startFaceDetection(StartFaceDetectionRequest startFaceDetectionRequest)
            throws AccessDeniedException, IdempotentParameterMismatchException, InvalidParameterException,
            InvalidS3ObjectException, InternalServerErrorException, VideoTooLargeException,
            ProvisionedThroughputExceededException, LimitExceededException, ThrottlingException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<StartFaceDetectionResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new StartFaceDetectionResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<StartFaceDetectionRequest, StartFaceDetectionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(startFaceDetectionRequest).withMarshaller(new StartFaceDetectionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Starts the asynchronous search for faces in a collection that match the faces of persons detected in a stored
     * video.
     * </p>
     * <p>
     * The video must be stored in an Amazon S3 bucket. Use <a>Video</a> to specify the bucket name and the filename of
     * the video. <code>StartFaceSearch</code> returns a job identifier (<code>JobId</code>) which you use to get the
     * search results once the search has completed. When searching is finished, Amazon Rekognition Video publishes a
     * completion status to the Amazon Simple Notification Service topic that you specify in
     * <code>NotificationChannel</code>. To get the search results, first check that the status value published to the
     * Amazon SNS topic is <code>SUCCEEDED</code>. If so, call and pass the job identifier (<code>JobId</code>) from the
     * initial call to <code>StartFaceSearch</code>. For more information, see <a>procedure-person-search-videos</a>.
     * </p>
     *
     * @param startFaceSearchRequest
     * @return Result of the StartFaceSearch operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws VideoTooLargeException
     *         The file size or duration of the supplied media is too large. The maximum file size is 8GB. The maximum
     *         duration is 2 hours.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws LimitExceededException
     *         An Amazon Rekognition service limit was exceeded. For example, if you start too many Amazon Rekognition
     *         Video jobs concurrently, calls to start operations (<code>StartLabelDetection</code>, for example) will
     *         raise a <code>LimitExceededException</code> exception (HTTP status code: 400) until the number of
     *         concurrently running jobs is below the Amazon Rekognition service limit.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.StartFaceSearch
     */
    @Override
    public StartFaceSearchResponse startFaceSearch(StartFaceSearchRequest startFaceSearchRequest) throws AccessDeniedException,
            IdempotentParameterMismatchException, InvalidParameterException, InvalidS3ObjectException,
            InternalServerErrorException, VideoTooLargeException, ProvisionedThroughputExceededException, LimitExceededException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, RekognitionException {

        HttpResponseHandler<StartFaceSearchResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new StartFaceSearchResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<StartFaceSearchRequest, StartFaceSearchResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(startFaceSearchRequest).withMarshaller(new StartFaceSearchRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Starts asynchronous detection of labels in a stored video.
     * </p>
     * <p>
     * Amazon Rekognition Video can detect labels in a video. Labels are instances of real-world entities. This includes
     * objects like flower, tree, and table; events like wedding, graduation, and birthday party; concepts like
     * landscape, evening, and nature; and activities like a person getting out of a car or a person skiing.
     * </p>
     * <p>
     * The video must be stored in an Amazon S3 bucket. Use <a>Video</a> to specify the bucket name and the filename of
     * the video. <code>StartLabelDetection</code> returns a job identifier (<code>JobId</code>) which you use to get
     * the results of the operation. When label detection is finished, Amazon Rekognition Video publishes a completion
     * status to the Amazon Simple Notification Service topic that you specify in <code>NotificationChannel</code>.
     * </p>
     * <p>
     * To get the results of the label detection operation, first check that the status value published to the Amazon
     * SNS topic is <code>SUCCEEDED</code>. If so, call and pass the job identifier (<code>JobId</code>) from the
     * initial call to <code>StartLabelDetection</code>.
     * </p>
     * <p/>
     *
     * @param startLabelDetectionRequest
     * @return Result of the StartLabelDetection operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws VideoTooLargeException
     *         The file size or duration of the supplied media is too large. The maximum file size is 8GB. The maximum
     *         duration is 2 hours.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws LimitExceededException
     *         An Amazon Rekognition service limit was exceeded. For example, if you start too many Amazon Rekognition
     *         Video jobs concurrently, calls to start operations (<code>StartLabelDetection</code>, for example) will
     *         raise a <code>LimitExceededException</code> exception (HTTP status code: 400) until the number of
     *         concurrently running jobs is below the Amazon Rekognition service limit.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.StartLabelDetection
     */
    @Override
    public StartLabelDetectionResponse startLabelDetection(StartLabelDetectionRequest startLabelDetectionRequest)
            throws AccessDeniedException, IdempotentParameterMismatchException, InvalidParameterException,
            InvalidS3ObjectException, InternalServerErrorException, VideoTooLargeException,
            ProvisionedThroughputExceededException, LimitExceededException, ThrottlingException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<StartLabelDetectionResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new StartLabelDetectionResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<StartLabelDetectionRequest, StartLabelDetectionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(startLabelDetectionRequest).withMarshaller(new StartLabelDetectionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Starts the asynchronous tracking of persons in a stored video.
     * </p>
     * <p>
     * Amazon Rekognition Video can track persons in a video stored in an Amazon S3 bucket. Use <a>Video</a> to specify
     * the bucket name and the filename of the video. <code>StartPersonTracking</code> returns a job identifier (
     * <code>JobId</code>) which you use to get the results of the operation. When label detection is finished, Amazon
     * Rekognition publishes a completion status to the Amazon Simple Notification Service topic that you specify in
     * <code>NotificationChannel</code>.
     * </p>
     * <p>
     * To get the results of the person detection operation, first check that the status value published to the Amazon
     * SNS topic is <code>SUCCEEDED</code>. If so, call and pass the job identifier (<code>JobId</code>) from the
     * initial call to <code>StartPersonTracking</code>.
     * </p>
     *
     * @param startPersonTrackingRequest
     * @return Result of the StartPersonTracking operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Rekognition is unable to access the S3 object specified in the request.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws VideoTooLargeException
     *         The file size or duration of the supplied media is too large. The maximum file size is 8GB. The maximum
     *         duration is 2 hours.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws LimitExceededException
     *         An Amazon Rekognition service limit was exceeded. For example, if you start too many Amazon Rekognition
     *         Video jobs concurrently, calls to start operations (<code>StartLabelDetection</code>, for example) will
     *         raise a <code>LimitExceededException</code> exception (HTTP status code: 400) until the number of
     *         concurrently running jobs is below the Amazon Rekognition service limit.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.StartPersonTracking
     */
    @Override
    public StartPersonTrackingResponse startPersonTracking(StartPersonTrackingRequest startPersonTrackingRequest)
            throws AccessDeniedException, IdempotentParameterMismatchException, InvalidParameterException,
            InvalidS3ObjectException, InternalServerErrorException, VideoTooLargeException,
            ProvisionedThroughputExceededException, LimitExceededException, ThrottlingException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<StartPersonTrackingResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new StartPersonTrackingResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<StartPersonTrackingRequest, StartPersonTrackingResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(startPersonTrackingRequest).withMarshaller(new StartPersonTrackingRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Starts processing a stream processor. You create a stream processor by calling . To tell
     * <code>StartStreamProcessor</code> which stream processor to start, use the value of the <code>Name</code> field
     * specified in the call to <code>CreateStreamProcessor</code>.
     * </p>
     *
     * @param startStreamProcessorRequest
     * @return Result of the StartStreamProcessor operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ResourceInUseException
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.StartStreamProcessor
     */
    @Override
    public StartStreamProcessorResponse startStreamProcessor(StartStreamProcessorRequest startStreamProcessorRequest)
            throws AccessDeniedException, InternalServerErrorException, ThrottlingException, InvalidParameterException,
            ResourceNotFoundException, ResourceInUseException, ProvisionedThroughputExceededException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<StartStreamProcessorResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new StartStreamProcessorResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<StartStreamProcessorRequest, StartStreamProcessorResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(startStreamProcessorRequest)
                .withMarshaller(new StartStreamProcessorRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Stops a running stream processor that was created by .
     * </p>
     *
     * @param stopStreamProcessorRequest
     * @return Result of the StopStreamProcessor operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerErrorException
     *         Amazon Rekognition experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Rekognition is temporarily unable to process the request. Try your call again.
     * @throws InvalidParameterException
     *         Input parameter violated a constraint. Validate your parameter before calling the API operation again.
     * @throws ResourceNotFoundException
     *         The collection specified in the request cannot be found.
     * @throws ResourceInUseException
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Rekognition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RekognitionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RekognitionClient.StopStreamProcessor
     */
    @Override
    public StopStreamProcessorResponse stopStreamProcessor(StopStreamProcessorRequest stopStreamProcessorRequest)
            throws AccessDeniedException, InternalServerErrorException, ThrottlingException, InvalidParameterException,
            ResourceNotFoundException, ResourceInUseException, ProvisionedThroughputExceededException, AwsServiceException,
            SdkClientException, RekognitionException {

        HttpResponseHandler<StopStreamProcessorResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new StopStreamProcessorResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<StopStreamProcessorRequest, StopStreamProcessorResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(stopStreamProcessorRequest).withMarshaller(new StopStreamProcessorRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(AwsJsonProtocolFactory protocolFactory) {
        return protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());
    }

    private software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory init(boolean supportsCbor) {
        return new AwsJsonProtocolFactory(new JsonClientMetadata()
                .withSupportsCbor(supportsCbor)
                .withSupportsIon(false)
                .withBaseServiceExceptionClass(software.amazon.awssdk.services.rekognition.model.RekognitionException.class)
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidParameterException").withModeledClass(
                                InvalidParameterException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ResourceInUseException").withModeledClass(
                                ResourceInUseException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidImageFormatException").withModeledClass(
                                InvalidImageFormatException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ThrottlingException").withModeledClass(
                                ThrottlingException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidS3ObjectException").withModeledClass(
                                InvalidS3ObjectException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("LimitExceededException").withModeledClass(
                                LimitExceededException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("VideoTooLargeException").withModeledClass(
                                VideoTooLargeException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("AccessDeniedException").withModeledClass(
                                AccessDeniedException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ResourceNotFoundException").withModeledClass(
                                ResourceNotFoundException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("IdempotentParameterMismatchException").withModeledClass(
                                IdempotentParameterMismatchException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ProvisionedThroughputExceededException").withModeledClass(
                                ProvisionedThroughputExceededException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ImageTooLargeException").withModeledClass(
                                ImageTooLargeException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ResourceAlreadyExistsException").withModeledClass(
                                ResourceAlreadyExistsException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidPaginationTokenException").withModeledClass(
                                InvalidPaginationTokenException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InternalServerError").withModeledClass(
                                InternalServerErrorException.class)), AwsJsonProtocolMetadata.builder().protocolVersion("1.1")
                .protocol(AwsJsonProtocol.AWS_JSON).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends RekognitionRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
