/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.rekognition.transform.PersonMatchMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a person whose face matches a face(s) in a Amazon Rekognition collection. Includes information
 * about the faces in the Amazon Rekognition collection (), information about the person (<a>PersonDetail</a>) and the
 * timestamp for when the person was detected in a video. An array of <code>PersonMatch</code> objects is returned by .
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PersonMatch implements StructuredPojo, ToCopyableBuilder<PersonMatch.Builder, PersonMatch> {
    private final Long timestamp;

    private final PersonDetail person;

    private final List<FaceMatch> faceMatches;

    private PersonMatch(BuilderImpl builder) {
        this.timestamp = builder.timestamp;
        this.person = builder.person;
        this.faceMatches = builder.faceMatches;
    }

    /**
     * <p>
     * The time, in milliseconds from the beginning of the video, that the person was matched in the video.
     * </p>
     * 
     * @return The time, in milliseconds from the beginning of the video, that the person was matched in the video.
     */
    public Long timestamp() {
        return timestamp;
    }

    /**
     * <p>
     * Information about the matched person.
     * </p>
     * 
     * @return Information about the matched person.
     */
    public PersonDetail person() {
        return person;
    }

    /**
     * <p>
     * Information about the faces in the input collection that match the face of a person in the video.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Information about the faces in the input collection that match the face of a person in the video.
     */
    public List<FaceMatch> faceMatches() {
        return faceMatches;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        hashCode = 31 * hashCode + Objects.hashCode(person());
        hashCode = 31 * hashCode + Objects.hashCode(faceMatches());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PersonMatch)) {
            return false;
        }
        PersonMatch other = (PersonMatch) obj;
        return Objects.equals(timestamp(), other.timestamp()) && Objects.equals(person(), other.person())
                && Objects.equals(faceMatches(), other.faceMatches());
    }

    @Override
    public String toString() {
        return ToString.builder("PersonMatch").add("Timestamp", timestamp()).add("Person", person())
                .add("FaceMatches", faceMatches()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        case "Person":
            return Optional.ofNullable(clazz.cast(person()));
        case "FaceMatches":
            return Optional.ofNullable(clazz.cast(faceMatches()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        PersonMatchMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, PersonMatch> {
        /**
         * <p>
         * The time, in milliseconds from the beginning of the video, that the person was matched in the video.
         * </p>
         * 
         * @param timestamp
         *        The time, in milliseconds from the beginning of the video, that the person was matched in the video.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Long timestamp);

        /**
         * <p>
         * Information about the matched person.
         * </p>
         * 
         * @param person
         *        Information about the matched person.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder person(PersonDetail person);

        /**
         * <p>
         * Information about the matched person.
         * </p>
         * This is a convenience that creates an instance of the {@link PersonDetail.Builder} avoiding the need to
         * create one manually via {@link PersonDetail#builder()}.
         *
         * When the {@link Consumer} completes, {@link PersonDetail.Builder#build()} is called immediately and its
         * result is passed to {@link #person(PersonDetail)}.
         * 
         * @param person
         *        a consumer that will call methods on {@link PersonDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #person(PersonDetail)
         */
        default Builder person(Consumer<PersonDetail.Builder> person) {
            return person(PersonDetail.builder().applyMutation(person).build());
        }

        /**
         * <p>
         * Information about the faces in the input collection that match the face of a person in the video.
         * </p>
         * 
         * @param faceMatches
         *        Information about the faces in the input collection that match the face of a person in the video.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder faceMatches(Collection<FaceMatch> faceMatches);

        /**
         * <p>
         * Information about the faces in the input collection that match the face of a person in the video.
         * </p>
         * 
         * @param faceMatches
         *        Information about the faces in the input collection that match the face of a person in the video.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder faceMatches(FaceMatch... faceMatches);

        /**
         * <p>
         * Information about the faces in the input collection that match the face of a person in the video.
         * </p>
         * This is a convenience that creates an instance of the {@link List<FaceMatch>.Builder} avoiding the need to
         * create one manually via {@link List<FaceMatch>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<FaceMatch>.Builder#build()} is called immediately and its
         * result is passed to {@link #faceMatches(List<FaceMatch>)}.
         * 
         * @param faceMatches
         *        a consumer that will call methods on {@link List<FaceMatch>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #faceMatches(List<FaceMatch>)
         */
        Builder faceMatches(Consumer<FaceMatch.Builder>... faceMatches);
    }

    static final class BuilderImpl implements Builder {
        private Long timestamp;

        private PersonDetail person;

        private List<FaceMatch> faceMatches = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PersonMatch model) {
            timestamp(model.timestamp);
            person(model.person);
            faceMatches(model.faceMatches);
        }

        public final Long getTimestamp() {
            return timestamp;
        }

        @Override
        public final Builder timestamp(Long timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final void setTimestamp(Long timestamp) {
            this.timestamp = timestamp;
        }

        public final PersonDetail.Builder getPerson() {
            return person != null ? person.toBuilder() : null;
        }

        @Override
        public final Builder person(PersonDetail person) {
            this.person = person;
            return this;
        }

        public final void setPerson(PersonDetail.BuilderImpl person) {
            this.person = person != null ? person.build() : null;
        }

        public final Collection<FaceMatch.Builder> getFaceMatches() {
            return faceMatches != null ? faceMatches.stream().map(FaceMatch::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder faceMatches(Collection<FaceMatch> faceMatches) {
            this.faceMatches = FaceMatchListCopier.copy(faceMatches);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder faceMatches(FaceMatch... faceMatches) {
            faceMatches(Arrays.asList(faceMatches));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder faceMatches(Consumer<FaceMatch.Builder>... faceMatches) {
            faceMatches(Stream.of(faceMatches).map(c -> FaceMatch.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setFaceMatches(Collection<FaceMatch.BuilderImpl> faceMatches) {
            this.faceMatches = FaceMatchListCopier.copyFromBuilder(faceMatches);
        }

        @Override
        public PersonMatch build() {
            return new PersonMatch(this);
        }
    }
}
