/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.rekognition.transform.VideoMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Video file stored in an Amazon S3 bucket. Amazon Rekognition video start operations such as use <code>Video</code> to
 * specify a video for analysis. The supported file formats are .mp4, .mov and .avi.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Video implements StructuredPojo, ToCopyableBuilder<Video.Builder, Video> {
    private final S3Object s3Object;

    private Video(BuilderImpl builder) {
        this.s3Object = builder.s3Object;
    }

    /**
     * <p>
     * The Amazon S3 bucket name and file name for the video.
     * </p>
     * 
     * @return The Amazon S3 bucket name and file name for the video.
     */
    public S3Object s3Object() {
        return s3Object;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3Object());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Video)) {
            return false;
        }
        Video other = (Video) obj;
        return Objects.equals(s3Object(), other.s3Object());
    }

    @Override
    public String toString() {
        return ToString.builder("Video").add("S3Object", s3Object()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3Object":
            return Optional.ofNullable(clazz.cast(s3Object()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        VideoMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Video> {
        /**
         * <p>
         * The Amazon S3 bucket name and file name for the video.
         * </p>
         * 
         * @param s3Object
         *        The Amazon S3 bucket name and file name for the video.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Object(S3Object s3Object);

        /**
         * <p>
         * The Amazon S3 bucket name and file name for the video.
         * </p>
         * This is a convenience that creates an instance of the {@link S3Object.Builder} avoiding the need to create
         * one manually via {@link S3Object#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3Object.Builder#build()} is called immediately and its result is
         * passed to {@link #s3Object(S3Object)}.
         * 
         * @param s3Object
         *        a consumer that will call methods on {@link S3Object.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Object(S3Object)
         */
        default Builder s3Object(Consumer<S3Object.Builder> s3Object) {
            return s3Object(S3Object.builder().applyMutation(s3Object).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private S3Object s3Object;

        private BuilderImpl() {
        }

        private BuilderImpl(Video model) {
            s3Object(model.s3Object);
        }

        public final S3Object.Builder getS3Object() {
            return s3Object != null ? s3Object.toBuilder() : null;
        }

        @Override
        public final Builder s3Object(S3Object s3Object) {
            this.s3Object = s3Object;
            return this;
        }

        public final void setS3Object(S3Object.BuilderImpl s3Object) {
            this.s3Object = s3Object != null ? s3Object.build() : null;
        }

        @Override
        public Video build() {
            return new Video(this);
        }
    }
}
