/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.nio.ByteBuffer;
import javax.annotation.Generated;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.protocol.ProtocolMarshaller;
import software.amazon.awssdk.protocol.StructuredPojo;
import software.amazon.awssdk.runtime.StandardMemberCopier;
import software.amazon.awssdk.services.rekognition.transform.ImageMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the input image either as bytes or an S3 object.
 * </p>
 * <p>
 * You pass image bytes to a Rekognition API operation by using the <code>Bytes</code> property. For example, you would
 * use the <code>Bytes</code> property to pass an image loaded from a local file system. Image bytes passed by using the
 * <code>Bytes</code> property must be base64-encoded. Your code may not need to encode image bytes if you are using an
 * AWS SDK to call Rekognition API operations. For more information, see <a>example4</a>.
 * </p>
 * <p>
 * You pass images stored in an S3 bucket to a Rekognition API operation by using the <code>S3Object</code> property.
 * Images stored in an S3 bucket do not need to be base64-encoded.
 * </p>
 * <p>
 * The region for the S3 bucket containing the S3 object must match the region you use for Amazon Rekognition
 * operations.
 * </p>
 * <p>
 * If you use the Amazon CLI to call Amazon Rekognition operations, passing image bytes using the Bytes property is not
 * supported. You must first upload the image to an Amazon S3 bucket and then call the operation using the S3Object
 * property.
 * </p>
 * <p>
 * For Amazon Rekognition to process an S3 object, the user must have permission to access the S3 object. For more
 * information, see <a>manage-access-resource-policies</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Image implements StructuredPojo, ToCopyableBuilder<Image.Builder, Image> {
    private final ByteBuffer bytes;

    private final S3Object s3Object;

    private Image(BuilderImpl builder) {
        this.bytes = builder.bytes;
        this.s3Object = builder.s3Object;
    }

    /**
     * <p>
     * Blob of image bytes up to 5 MBs.
     * </p>
     * <p>
     * This method will return a new read-only {@code ByteBuffer} each time it is invoked.
     * </p>
     * 
     * @return Blob of image bytes up to 5 MBs.
     */
    public ByteBuffer bytes() {
        return bytes == null ? null : bytes.asReadOnlyBuffer();
    }

    /**
     * <p>
     * Identifies an S3 object as the image source.
     * </p>
     * 
     * @return Identifies an S3 object as the image source.
     */
    public S3Object s3Object() {
        return s3Object;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((bytes() == null) ? 0 : bytes().hashCode());
        hashCode = 31 * hashCode + ((s3Object() == null) ? 0 : s3Object().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Image)) {
            return false;
        }
        Image other = (Image) obj;
        if (other.bytes() == null ^ this.bytes() == null) {
            return false;
        }
        if (other.bytes() != null && !other.bytes().equals(this.bytes())) {
            return false;
        }
        if (other.s3Object() == null ^ this.s3Object() == null) {
            return false;
        }
        if (other.s3Object() != null && !other.s3Object().equals(this.s3Object())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (bytes() != null) {
            sb.append("Bytes: ").append(bytes()).append(",");
        }
        if (s3Object() != null) {
            sb.append("S3Object: ").append(s3Object()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ImageMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Image> {
        /**
         * <p>
         * Blob of image bytes up to 5 MBs.
         * </p>
         * <p>
         * To preserve immutability, the remaining bytes in the provided buffer will be copied into a new buffer when
         * set.
         * </p>
         *
         * @param bytes
         *        Blob of image bytes up to 5 MBs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bytes(ByteBuffer bytes);

        /**
         * <p>
         * Identifies an S3 object as the image source.
         * </p>
         * 
         * @param s3Object
         *        Identifies an S3 object as the image source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Object(S3Object s3Object);
    }

    private static final class BuilderImpl implements Builder {
        private ByteBuffer bytes;

        private S3Object s3Object;

        private BuilderImpl() {
        }

        private BuilderImpl(Image model) {
            setBytes(model.bytes);
            setS3Object(model.s3Object);
        }

        public final ByteBuffer getBytes() {
            return bytes;
        }

        @Override
        public final Builder bytes(ByteBuffer bytes) {
            this.bytes = StandardMemberCopier.copy(bytes);
            return this;
        }

        public final void setBytes(ByteBuffer bytes) {
            this.bytes = StandardMemberCopier.copy(bytes);
        }

        public final S3Object getS3Object() {
            return s3Object;
        }

        @Override
        public final Builder s3Object(S3Object s3Object) {
            this.s3Object = s3Object;
            return this;
        }

        public final void setS3Object(S3Object s3Object) {
            this.s3Object = s3Object;
        }

        @Override
        public Image build() {
            return new Image(this);
        }
    }
}
