/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.rekognition.transform.BoundingBoxMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Identifies the bounding box around the object or face. The <code>left</code> (x-coordinate) and <code>top</code>
 * (y-coordinate) are coordinates representing the top and left sides of the bounding box. Note that the upper-left
 * corner of the image is the origin (0,0).
 * </p>
 * <p>
 * The <code>top</code> and <code>left</code> values returned are ratios of the overall image size. For example, if the
 * input image is 700x200 pixels, and the top-left coordinate of the bounding box is 350x50 pixels, the API returns a
 * <code>left</code> value of 0.5 (350/700) and a <code>top</code> value of 0.25 (50/200).
 * </p>
 * <p>
 * The <code>width</code> and <code>height</code> values represent the dimensions of the bounding box as a ratio of the
 * overall image dimension. For example, if the input image is 700x200 pixels, and the bounding box width is 70 pixels,
 * the width returned is 0.1.
 * </p>
 * <note>
 * <p>
 * The bounding box coordinates can have negative values. For example, if Amazon Rekognition is able to detect a face
 * that is at the image edge and is only partially visible, the service can return coordinates that are outside the
 * image bounds and, depending on the image edge, you might get negative values or values greater than 1 for the
 * <code>left</code> or <code>top</code> values.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public class BoundingBox implements StructuredPojo, ToCopyableBuilder<BoundingBox.Builder, BoundingBox> {
    private final Float width;

    private final Float height;

    private final Float left;

    private final Float top;

    private BoundingBox(BuilderImpl builder) {
        this.width = builder.width;
        this.height = builder.height;
        this.left = builder.left;
        this.top = builder.top;
    }

    /**
     * <p>
     * Width of the bounding box as a ratio of the overall image width.
     * </p>
     * 
     * @return Width of the bounding box as a ratio of the overall image width.
     */
    public Float width() {
        return width;
    }

    /**
     * <p>
     * Height of the bounding box as a ratio of the overall image height.
     * </p>
     * 
     * @return Height of the bounding box as a ratio of the overall image height.
     */
    public Float height() {
        return height;
    }

    /**
     * <p>
     * Left coordinate of the bounding box as a ratio of overall image width.
     * </p>
     * 
     * @return Left coordinate of the bounding box as a ratio of overall image width.
     */
    public Float left() {
        return left;
    }

    /**
     * <p>
     * Top coordinate of the bounding box as a ratio of overall image height.
     * </p>
     * 
     * @return Top coordinate of the bounding box as a ratio of overall image height.
     */
    public Float top() {
        return top;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((width() == null) ? 0 : width().hashCode());
        hashCode = 31 * hashCode + ((height() == null) ? 0 : height().hashCode());
        hashCode = 31 * hashCode + ((left() == null) ? 0 : left().hashCode());
        hashCode = 31 * hashCode + ((top() == null) ? 0 : top().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BoundingBox)) {
            return false;
        }
        BoundingBox other = (BoundingBox) obj;
        if (other.width() == null ^ this.width() == null) {
            return false;
        }
        if (other.width() != null && !other.width().equals(this.width())) {
            return false;
        }
        if (other.height() == null ^ this.height() == null) {
            return false;
        }
        if (other.height() != null && !other.height().equals(this.height())) {
            return false;
        }
        if (other.left() == null ^ this.left() == null) {
            return false;
        }
        if (other.left() != null && !other.left().equals(this.left())) {
            return false;
        }
        if (other.top() == null ^ this.top() == null) {
            return false;
        }
        if (other.top() != null && !other.top().equals(this.top())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (width() != null) {
            sb.append("Width: ").append(width()).append(",");
        }
        if (height() != null) {
            sb.append("Height: ").append(height()).append(",");
        }
        if (left() != null) {
            sb.append("Left: ").append(left()).append(",");
        }
        if (top() != null) {
            sb.append("Top: ").append(top()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Width":
            return Optional.of(clazz.cast(width()));
        case "Height":
            return Optional.of(clazz.cast(height()));
        case "Left":
            return Optional.of(clazz.cast(left()));
        case "Top":
            return Optional.of(clazz.cast(top()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        BoundingBoxMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, BoundingBox> {
        /**
         * <p>
         * Width of the bounding box as a ratio of the overall image width.
         * </p>
         * 
         * @param width
         *        Width of the bounding box as a ratio of the overall image width.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder width(Float width);

        /**
         * <p>
         * Height of the bounding box as a ratio of the overall image height.
         * </p>
         * 
         * @param height
         *        Height of the bounding box as a ratio of the overall image height.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder height(Float height);

        /**
         * <p>
         * Left coordinate of the bounding box as a ratio of overall image width.
         * </p>
         * 
         * @param left
         *        Left coordinate of the bounding box as a ratio of overall image width.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder left(Float left);

        /**
         * <p>
         * Top coordinate of the bounding box as a ratio of overall image height.
         * </p>
         * 
         * @param top
         *        Top coordinate of the bounding box as a ratio of overall image height.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder top(Float top);
    }

    static final class BuilderImpl implements Builder {
        private Float width;

        private Float height;

        private Float left;

        private Float top;

        private BuilderImpl() {
        }

        private BuilderImpl(BoundingBox model) {
            width(model.width);
            height(model.height);
            left(model.left);
            top(model.top);
        }

        public final Float getWidth() {
            return width;
        }

        @Override
        public final Builder width(Float width) {
            this.width = width;
            return this;
        }

        public final void setWidth(Float width) {
            this.width = width;
        }

        public final Float getHeight() {
            return height;
        }

        @Override
        public final Builder height(Float height) {
            this.height = height;
            return this;
        }

        public final void setHeight(Float height) {
            this.height = height;
        }

        public final Float getLeft() {
            return left;
        }

        @Override
        public final Builder left(Float left) {
            this.left = left;
            return this;
        }

        public final void setLeft(Float left) {
            this.left = left;
        }

        public final Float getTop() {
            return top;
        }

        @Override
        public final Builder top(Float top) {
            this.top = top;
            return this;
        }

        public final void setTop(Float top) {
            this.top = top;
        }

        @Override
        public BoundingBox build() {
            return new BoundingBox(this);
        }
    }
}
