/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.rekognition.transform.FaceDetailMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Structure containing attributes of the face that the algorithm detected.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class FaceDetail implements StructuredPojo, ToCopyableBuilder<FaceDetail.Builder, FaceDetail> {
    private final BoundingBox boundingBox;

    private final AgeRange ageRange;

    private final Smile smile;

    private final Eyeglasses eyeglasses;

    private final Sunglasses sunglasses;

    private final Gender gender;

    private final Beard beard;

    private final Mustache mustache;

    private final EyeOpen eyesOpen;

    private final MouthOpen mouthOpen;

    private final List<Emotion> emotions;

    private final List<Landmark> landmarks;

    private final Pose pose;

    private final ImageQuality quality;

    private final Float confidence;

    private FaceDetail(BuilderImpl builder) {
        this.boundingBox = builder.boundingBox;
        this.ageRange = builder.ageRange;
        this.smile = builder.smile;
        this.eyeglasses = builder.eyeglasses;
        this.sunglasses = builder.sunglasses;
        this.gender = builder.gender;
        this.beard = builder.beard;
        this.mustache = builder.mustache;
        this.eyesOpen = builder.eyesOpen;
        this.mouthOpen = builder.mouthOpen;
        this.emotions = builder.emotions;
        this.landmarks = builder.landmarks;
        this.pose = builder.pose;
        this.quality = builder.quality;
        this.confidence = builder.confidence;
    }

    /**
     * <p>
     * Bounding box of the face.
     * </p>
     * 
     * @return Bounding box of the face.
     */
    public BoundingBox boundingBox() {
        return boundingBox;
    }

    /**
     * <p>
     * The estimated age range, in years, for the face. Low represents the lowest estimated age and High represents the
     * highest estimated age.
     * </p>
     * 
     * @return The estimated age range, in years, for the face. Low represents the lowest estimated age and High
     *         represents the highest estimated age.
     */
    public AgeRange ageRange() {
        return ageRange;
    }

    /**
     * <p>
     * Indicates whether or not the face is smiling, and the confidence level in the determination.
     * </p>
     * 
     * @return Indicates whether or not the face is smiling, and the confidence level in the determination.
     */
    public Smile smile() {
        return smile;
    }

    /**
     * <p>
     * Indicates whether or not the face is wearing eye glasses, and the confidence level in the determination.
     * </p>
     * 
     * @return Indicates whether or not the face is wearing eye glasses, and the confidence level in the determination.
     */
    public Eyeglasses eyeglasses() {
        return eyeglasses;
    }

    /**
     * <p>
     * Indicates whether or not the face is wearing sunglasses, and the confidence level in the determination.
     * </p>
     * 
     * @return Indicates whether or not the face is wearing sunglasses, and the confidence level in the determination.
     */
    public Sunglasses sunglasses() {
        return sunglasses;
    }

    /**
     * <p>
     * Gender of the face and the confidence level in the determination.
     * </p>
     * 
     * @return Gender of the face and the confidence level in the determination.
     */
    public Gender gender() {
        return gender;
    }

    /**
     * <p>
     * Indicates whether or not the face has a beard, and the confidence level in the determination.
     * </p>
     * 
     * @return Indicates whether or not the face has a beard, and the confidence level in the determination.
     */
    public Beard beard() {
        return beard;
    }

    /**
     * <p>
     * Indicates whether or not the face has a mustache, and the confidence level in the determination.
     * </p>
     * 
     * @return Indicates whether or not the face has a mustache, and the confidence level in the determination.
     */
    public Mustache mustache() {
        return mustache;
    }

    /**
     * <p>
     * Indicates whether or not the eyes on the face are open, and the confidence level in the determination.
     * </p>
     * 
     * @return Indicates whether or not the eyes on the face are open, and the confidence level in the determination.
     */
    public EyeOpen eyesOpen() {
        return eyesOpen;
    }

    /**
     * <p>
     * Indicates whether or not the mouth on the face is open, and the confidence level in the determination.
     * </p>
     * 
     * @return Indicates whether or not the mouth on the face is open, and the confidence level in the determination.
     */
    public MouthOpen mouthOpen() {
        return mouthOpen;
    }

    /**
     * <p>
     * The emotions detected on the face, and the confidence level in the determination. For example, HAPPY, SAD, and
     * ANGRY.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The emotions detected on the face, and the confidence level in the determination. For example, HAPPY,
     *         SAD, and ANGRY.
     */
    public List<Emotion> emotions() {
        return emotions;
    }

    /**
     * <p>
     * Indicates the location of landmarks on the face.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Indicates the location of landmarks on the face.
     */
    public List<Landmark> landmarks() {
        return landmarks;
    }

    /**
     * <p>
     * Indicates the pose of the face as determined by its pitch, roll, and yaw.
     * </p>
     * 
     * @return Indicates the pose of the face as determined by its pitch, roll, and yaw.
     */
    public Pose pose() {
        return pose;
    }

    /**
     * <p>
     * Identifies image brightness and sharpness.
     * </p>
     * 
     * @return Identifies image brightness and sharpness.
     */
    public ImageQuality quality() {
        return quality;
    }

    /**
     * <p>
     * Confidence level that the bounding box contains a face (and not a different object such as a tree).
     * </p>
     * 
     * @return Confidence level that the bounding box contains a face (and not a different object such as a tree).
     */
    public Float confidence() {
        return confidence;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((boundingBox() == null) ? 0 : boundingBox().hashCode());
        hashCode = 31 * hashCode + ((ageRange() == null) ? 0 : ageRange().hashCode());
        hashCode = 31 * hashCode + ((smile() == null) ? 0 : smile().hashCode());
        hashCode = 31 * hashCode + ((eyeglasses() == null) ? 0 : eyeglasses().hashCode());
        hashCode = 31 * hashCode + ((sunglasses() == null) ? 0 : sunglasses().hashCode());
        hashCode = 31 * hashCode + ((gender() == null) ? 0 : gender().hashCode());
        hashCode = 31 * hashCode + ((beard() == null) ? 0 : beard().hashCode());
        hashCode = 31 * hashCode + ((mustache() == null) ? 0 : mustache().hashCode());
        hashCode = 31 * hashCode + ((eyesOpen() == null) ? 0 : eyesOpen().hashCode());
        hashCode = 31 * hashCode + ((mouthOpen() == null) ? 0 : mouthOpen().hashCode());
        hashCode = 31 * hashCode + ((emotions() == null) ? 0 : emotions().hashCode());
        hashCode = 31 * hashCode + ((landmarks() == null) ? 0 : landmarks().hashCode());
        hashCode = 31 * hashCode + ((pose() == null) ? 0 : pose().hashCode());
        hashCode = 31 * hashCode + ((quality() == null) ? 0 : quality().hashCode());
        hashCode = 31 * hashCode + ((confidence() == null) ? 0 : confidence().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FaceDetail)) {
            return false;
        }
        FaceDetail other = (FaceDetail) obj;
        if (other.boundingBox() == null ^ this.boundingBox() == null) {
            return false;
        }
        if (other.boundingBox() != null && !other.boundingBox().equals(this.boundingBox())) {
            return false;
        }
        if (other.ageRange() == null ^ this.ageRange() == null) {
            return false;
        }
        if (other.ageRange() != null && !other.ageRange().equals(this.ageRange())) {
            return false;
        }
        if (other.smile() == null ^ this.smile() == null) {
            return false;
        }
        if (other.smile() != null && !other.smile().equals(this.smile())) {
            return false;
        }
        if (other.eyeglasses() == null ^ this.eyeglasses() == null) {
            return false;
        }
        if (other.eyeglasses() != null && !other.eyeglasses().equals(this.eyeglasses())) {
            return false;
        }
        if (other.sunglasses() == null ^ this.sunglasses() == null) {
            return false;
        }
        if (other.sunglasses() != null && !other.sunglasses().equals(this.sunglasses())) {
            return false;
        }
        if (other.gender() == null ^ this.gender() == null) {
            return false;
        }
        if (other.gender() != null && !other.gender().equals(this.gender())) {
            return false;
        }
        if (other.beard() == null ^ this.beard() == null) {
            return false;
        }
        if (other.beard() != null && !other.beard().equals(this.beard())) {
            return false;
        }
        if (other.mustache() == null ^ this.mustache() == null) {
            return false;
        }
        if (other.mustache() != null && !other.mustache().equals(this.mustache())) {
            return false;
        }
        if (other.eyesOpen() == null ^ this.eyesOpen() == null) {
            return false;
        }
        if (other.eyesOpen() != null && !other.eyesOpen().equals(this.eyesOpen())) {
            return false;
        }
        if (other.mouthOpen() == null ^ this.mouthOpen() == null) {
            return false;
        }
        if (other.mouthOpen() != null && !other.mouthOpen().equals(this.mouthOpen())) {
            return false;
        }
        if (other.emotions() == null ^ this.emotions() == null) {
            return false;
        }
        if (other.emotions() != null && !other.emotions().equals(this.emotions())) {
            return false;
        }
        if (other.landmarks() == null ^ this.landmarks() == null) {
            return false;
        }
        if (other.landmarks() != null && !other.landmarks().equals(this.landmarks())) {
            return false;
        }
        if (other.pose() == null ^ this.pose() == null) {
            return false;
        }
        if (other.pose() != null && !other.pose().equals(this.pose())) {
            return false;
        }
        if (other.quality() == null ^ this.quality() == null) {
            return false;
        }
        if (other.quality() != null && !other.quality().equals(this.quality())) {
            return false;
        }
        if (other.confidence() == null ^ this.confidence() == null) {
            return false;
        }
        if (other.confidence() != null && !other.confidence().equals(this.confidence())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (boundingBox() != null) {
            sb.append("BoundingBox: ").append(boundingBox()).append(",");
        }
        if (ageRange() != null) {
            sb.append("AgeRange: ").append(ageRange()).append(",");
        }
        if (smile() != null) {
            sb.append("Smile: ").append(smile()).append(",");
        }
        if (eyeglasses() != null) {
            sb.append("Eyeglasses: ").append(eyeglasses()).append(",");
        }
        if (sunglasses() != null) {
            sb.append("Sunglasses: ").append(sunglasses()).append(",");
        }
        if (gender() != null) {
            sb.append("Gender: ").append(gender()).append(",");
        }
        if (beard() != null) {
            sb.append("Beard: ").append(beard()).append(",");
        }
        if (mustache() != null) {
            sb.append("Mustache: ").append(mustache()).append(",");
        }
        if (eyesOpen() != null) {
            sb.append("EyesOpen: ").append(eyesOpen()).append(",");
        }
        if (mouthOpen() != null) {
            sb.append("MouthOpen: ").append(mouthOpen()).append(",");
        }
        if (emotions() != null) {
            sb.append("Emotions: ").append(emotions()).append(",");
        }
        if (landmarks() != null) {
            sb.append("Landmarks: ").append(landmarks()).append(",");
        }
        if (pose() != null) {
            sb.append("Pose: ").append(pose()).append(",");
        }
        if (quality() != null) {
            sb.append("Quality: ").append(quality()).append(",");
        }
        if (confidence() != null) {
            sb.append("Confidence: ").append(confidence()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BoundingBox":
            return Optional.of(clazz.cast(boundingBox()));
        case "AgeRange":
            return Optional.of(clazz.cast(ageRange()));
        case "Smile":
            return Optional.of(clazz.cast(smile()));
        case "Eyeglasses":
            return Optional.of(clazz.cast(eyeglasses()));
        case "Sunglasses":
            return Optional.of(clazz.cast(sunglasses()));
        case "Gender":
            return Optional.of(clazz.cast(gender()));
        case "Beard":
            return Optional.of(clazz.cast(beard()));
        case "Mustache":
            return Optional.of(clazz.cast(mustache()));
        case "EyesOpen":
            return Optional.of(clazz.cast(eyesOpen()));
        case "MouthOpen":
            return Optional.of(clazz.cast(mouthOpen()));
        case "Emotions":
            return Optional.of(clazz.cast(emotions()));
        case "Landmarks":
            return Optional.of(clazz.cast(landmarks()));
        case "Pose":
            return Optional.of(clazz.cast(pose()));
        case "Quality":
            return Optional.of(clazz.cast(quality()));
        case "Confidence":
            return Optional.of(clazz.cast(confidence()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        FaceDetailMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, FaceDetail> {
        /**
         * <p>
         * Bounding box of the face.
         * </p>
         * 
         * @param boundingBox
         *        Bounding box of the face.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder boundingBox(BoundingBox boundingBox);

        /**
         * <p>
         * The estimated age range, in years, for the face. Low represents the lowest estimated age and High represents
         * the highest estimated age.
         * </p>
         * 
         * @param ageRange
         *        The estimated age range, in years, for the face. Low represents the lowest estimated age and High
         *        represents the highest estimated age.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ageRange(AgeRange ageRange);

        /**
         * <p>
         * Indicates whether or not the face is smiling, and the confidence level in the determination.
         * </p>
         * 
         * @param smile
         *        Indicates whether or not the face is smiling, and the confidence level in the determination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder smile(Smile smile);

        /**
         * <p>
         * Indicates whether or not the face is wearing eye glasses, and the confidence level in the determination.
         * </p>
         * 
         * @param eyeglasses
         *        Indicates whether or not the face is wearing eye glasses, and the confidence level in the
         *        determination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eyeglasses(Eyeglasses eyeglasses);

        /**
         * <p>
         * Indicates whether or not the face is wearing sunglasses, and the confidence level in the determination.
         * </p>
         * 
         * @param sunglasses
         *        Indicates whether or not the face is wearing sunglasses, and the confidence level in the
         *        determination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sunglasses(Sunglasses sunglasses);

        /**
         * <p>
         * Gender of the face and the confidence level in the determination.
         * </p>
         * 
         * @param gender
         *        Gender of the face and the confidence level in the determination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gender(Gender gender);

        /**
         * <p>
         * Indicates whether or not the face has a beard, and the confidence level in the determination.
         * </p>
         * 
         * @param beard
         *        Indicates whether or not the face has a beard, and the confidence level in the determination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder beard(Beard beard);

        /**
         * <p>
         * Indicates whether or not the face has a mustache, and the confidence level in the determination.
         * </p>
         * 
         * @param mustache
         *        Indicates whether or not the face has a mustache, and the confidence level in the determination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mustache(Mustache mustache);

        /**
         * <p>
         * Indicates whether or not the eyes on the face are open, and the confidence level in the determination.
         * </p>
         * 
         * @param eyesOpen
         *        Indicates whether or not the eyes on the face are open, and the confidence level in the determination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eyesOpen(EyeOpen eyesOpen);

        /**
         * <p>
         * Indicates whether or not the mouth on the face is open, and the confidence level in the determination.
         * </p>
         * 
         * @param mouthOpen
         *        Indicates whether or not the mouth on the face is open, and the confidence level in the determination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mouthOpen(MouthOpen mouthOpen);

        /**
         * <p>
         * The emotions detected on the face, and the confidence level in the determination. For example, HAPPY, SAD,
         * and ANGRY.
         * </p>
         * 
         * @param emotions
         *        The emotions detected on the face, and the confidence level in the determination. For example, HAPPY,
         *        SAD, and ANGRY.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emotions(Collection<Emotion> emotions);

        /**
         * <p>
         * The emotions detected on the face, and the confidence level in the determination. For example, HAPPY, SAD,
         * and ANGRY.
         * </p>
         * 
         * @param emotions
         *        The emotions detected on the face, and the confidence level in the determination. For example, HAPPY,
         *        SAD, and ANGRY.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emotions(Emotion... emotions);

        /**
         * <p>
         * Indicates the location of landmarks on the face.
         * </p>
         * 
         * @param landmarks
         *        Indicates the location of landmarks on the face.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder landmarks(Collection<Landmark> landmarks);

        /**
         * <p>
         * Indicates the location of landmarks on the face.
         * </p>
         * 
         * @param landmarks
         *        Indicates the location of landmarks on the face.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder landmarks(Landmark... landmarks);

        /**
         * <p>
         * Indicates the pose of the face as determined by its pitch, roll, and yaw.
         * </p>
         * 
         * @param pose
         *        Indicates the pose of the face as determined by its pitch, roll, and yaw.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pose(Pose pose);

        /**
         * <p>
         * Identifies image brightness and sharpness.
         * </p>
         * 
         * @param quality
         *        Identifies image brightness and sharpness.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quality(ImageQuality quality);

        /**
         * <p>
         * Confidence level that the bounding box contains a face (and not a different object such as a tree).
         * </p>
         * 
         * @param confidence
         *        Confidence level that the bounding box contains a face (and not a different object such as a tree).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confidence(Float confidence);
    }

    static final class BuilderImpl implements Builder {
        private BoundingBox boundingBox;

        private AgeRange ageRange;

        private Smile smile;

        private Eyeglasses eyeglasses;

        private Sunglasses sunglasses;

        private Gender gender;

        private Beard beard;

        private Mustache mustache;

        private EyeOpen eyesOpen;

        private MouthOpen mouthOpen;

        private List<Emotion> emotions;

        private List<Landmark> landmarks;

        private Pose pose;

        private ImageQuality quality;

        private Float confidence;

        private BuilderImpl() {
        }

        private BuilderImpl(FaceDetail model) {
            boundingBox(model.boundingBox);
            ageRange(model.ageRange);
            smile(model.smile);
            eyeglasses(model.eyeglasses);
            sunglasses(model.sunglasses);
            gender(model.gender);
            beard(model.beard);
            mustache(model.mustache);
            eyesOpen(model.eyesOpen);
            mouthOpen(model.mouthOpen);
            emotions(model.emotions);
            landmarks(model.landmarks);
            pose(model.pose);
            quality(model.quality);
            confidence(model.confidence);
        }

        public final BoundingBox.Builder getBoundingBox() {
            return boundingBox != null ? boundingBox.toBuilder() : null;
        }

        @Override
        public final Builder boundingBox(BoundingBox boundingBox) {
            this.boundingBox = boundingBox;
            return this;
        }

        public final void setBoundingBox(BoundingBox.BuilderImpl boundingBox) {
            this.boundingBox = boundingBox != null ? boundingBox.build() : null;
        }

        public final AgeRange.Builder getAgeRange() {
            return ageRange != null ? ageRange.toBuilder() : null;
        }

        @Override
        public final Builder ageRange(AgeRange ageRange) {
            this.ageRange = ageRange;
            return this;
        }

        public final void setAgeRange(AgeRange.BuilderImpl ageRange) {
            this.ageRange = ageRange != null ? ageRange.build() : null;
        }

        public final Smile.Builder getSmile() {
            return smile != null ? smile.toBuilder() : null;
        }

        @Override
        public final Builder smile(Smile smile) {
            this.smile = smile;
            return this;
        }

        public final void setSmile(Smile.BuilderImpl smile) {
            this.smile = smile != null ? smile.build() : null;
        }

        public final Eyeglasses.Builder getEyeglasses() {
            return eyeglasses != null ? eyeglasses.toBuilder() : null;
        }

        @Override
        public final Builder eyeglasses(Eyeglasses eyeglasses) {
            this.eyeglasses = eyeglasses;
            return this;
        }

        public final void setEyeglasses(Eyeglasses.BuilderImpl eyeglasses) {
            this.eyeglasses = eyeglasses != null ? eyeglasses.build() : null;
        }

        public final Sunglasses.Builder getSunglasses() {
            return sunglasses != null ? sunglasses.toBuilder() : null;
        }

        @Override
        public final Builder sunglasses(Sunglasses sunglasses) {
            this.sunglasses = sunglasses;
            return this;
        }

        public final void setSunglasses(Sunglasses.BuilderImpl sunglasses) {
            this.sunglasses = sunglasses != null ? sunglasses.build() : null;
        }

        public final Gender.Builder getGender() {
            return gender != null ? gender.toBuilder() : null;
        }

        @Override
        public final Builder gender(Gender gender) {
            this.gender = gender;
            return this;
        }

        public final void setGender(Gender.BuilderImpl gender) {
            this.gender = gender != null ? gender.build() : null;
        }

        public final Beard.Builder getBeard() {
            return beard != null ? beard.toBuilder() : null;
        }

        @Override
        public final Builder beard(Beard beard) {
            this.beard = beard;
            return this;
        }

        public final void setBeard(Beard.BuilderImpl beard) {
            this.beard = beard != null ? beard.build() : null;
        }

        public final Mustache.Builder getMustache() {
            return mustache != null ? mustache.toBuilder() : null;
        }

        @Override
        public final Builder mustache(Mustache mustache) {
            this.mustache = mustache;
            return this;
        }

        public final void setMustache(Mustache.BuilderImpl mustache) {
            this.mustache = mustache != null ? mustache.build() : null;
        }

        public final EyeOpen.Builder getEyesOpen() {
            return eyesOpen != null ? eyesOpen.toBuilder() : null;
        }

        @Override
        public final Builder eyesOpen(EyeOpen eyesOpen) {
            this.eyesOpen = eyesOpen;
            return this;
        }

        public final void setEyesOpen(EyeOpen.BuilderImpl eyesOpen) {
            this.eyesOpen = eyesOpen != null ? eyesOpen.build() : null;
        }

        public final MouthOpen.Builder getMouthOpen() {
            return mouthOpen != null ? mouthOpen.toBuilder() : null;
        }

        @Override
        public final Builder mouthOpen(MouthOpen mouthOpen) {
            this.mouthOpen = mouthOpen;
            return this;
        }

        public final void setMouthOpen(MouthOpen.BuilderImpl mouthOpen) {
            this.mouthOpen = mouthOpen != null ? mouthOpen.build() : null;
        }

        public final Collection<Emotion.Builder> getEmotions() {
            return emotions != null ? emotions.stream().map(Emotion::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder emotions(Collection<Emotion> emotions) {
            this.emotions = EmotionsCopier.copy(emotions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder emotions(Emotion... emotions) {
            emotions(Arrays.asList(emotions));
            return this;
        }

        public final void setEmotions(Collection<Emotion.BuilderImpl> emotions) {
            this.emotions = EmotionsCopier.copyFromBuilder(emotions);
        }

        public final Collection<Landmark.Builder> getLandmarks() {
            return landmarks != null ? landmarks.stream().map(Landmark::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder landmarks(Collection<Landmark> landmarks) {
            this.landmarks = LandmarksCopier.copy(landmarks);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder landmarks(Landmark... landmarks) {
            landmarks(Arrays.asList(landmarks));
            return this;
        }

        public final void setLandmarks(Collection<Landmark.BuilderImpl> landmarks) {
            this.landmarks = LandmarksCopier.copyFromBuilder(landmarks);
        }

        public final Pose.Builder getPose() {
            return pose != null ? pose.toBuilder() : null;
        }

        @Override
        public final Builder pose(Pose pose) {
            this.pose = pose;
            return this;
        }

        public final void setPose(Pose.BuilderImpl pose) {
            this.pose = pose != null ? pose.build() : null;
        }

        public final ImageQuality.Builder getQuality() {
            return quality != null ? quality.toBuilder() : null;
        }

        @Override
        public final Builder quality(ImageQuality quality) {
            this.quality = quality;
            return this;
        }

        public final void setQuality(ImageQuality.BuilderImpl quality) {
            this.quality = quality != null ? quality.build() : null;
        }

        public final Float getConfidence() {
            return confidence;
        }

        @Override
        public final Builder confidence(Float confidence) {
            this.confidence = confidence;
            return this;
        }

        public final void setConfidence(Float confidence) {
            this.confidence = confidence;
        }

        @Override
        public FaceDetail build() {
            return new FaceDetail(this);
        }
    }
}
