/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.rekognition.transform.S3ObjectMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the S3 bucket name and object name.
 * </p>
 * <p>
 * The region for the S3 bucket containing the S3 object must match the region you use for Amazon Rekognition
 * operations.
 * </p>
 * <p>
 * For Amazon Rekognition to process an S3 object, the user must have permission to access the S3 object. For more
 * information, see <a>manage-access-resource-policies</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class S3Object implements StructuredPojo, ToCopyableBuilder<S3Object.Builder, S3Object> {
    private final String bucket;

    private final String name;

    private final String version;

    private S3Object(BuilderImpl builder) {
        this.bucket = builder.bucket;
        this.name = builder.name;
        this.version = builder.version;
    }

    /**
     * <p>
     * Name of the S3 bucket.
     * </p>
     * 
     * @return Name of the S3 bucket.
     */
    public String bucket() {
        return bucket;
    }

    /**
     * <p>
     * S3 object key name.
     * </p>
     * 
     * @return S3 object key name.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * If the bucket is versioning enabled, you can specify the object version.
     * </p>
     * 
     * @return If the bucket is versioning enabled, you can specify the object version.
     */
    public String version() {
        return version;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((bucket() == null) ? 0 : bucket().hashCode());
        hashCode = 31 * hashCode + ((name() == null) ? 0 : name().hashCode());
        hashCode = 31 * hashCode + ((version() == null) ? 0 : version().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3Object)) {
            return false;
        }
        S3Object other = (S3Object) obj;
        if (other.bucket() == null ^ this.bucket() == null) {
            return false;
        }
        if (other.bucket() != null && !other.bucket().equals(this.bucket())) {
            return false;
        }
        if (other.name() == null ^ this.name() == null) {
            return false;
        }
        if (other.name() != null && !other.name().equals(this.name())) {
            return false;
        }
        if (other.version() == null ^ this.version() == null) {
            return false;
        }
        if (other.version() != null && !other.version().equals(this.version())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (bucket() != null) {
            sb.append("Bucket: ").append(bucket()).append(",");
        }
        if (name() != null) {
            sb.append("Name: ").append(name()).append(",");
        }
        if (version() != null) {
            sb.append("Version: ").append(version()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Bucket":
            return Optional.of(clazz.cast(bucket()));
        case "Name":
            return Optional.of(clazz.cast(name()));
        case "Version":
            return Optional.of(clazz.cast(version()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        S3ObjectMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, S3Object> {
        /**
         * <p>
         * Name of the S3 bucket.
         * </p>
         * 
         * @param bucket
         *        Name of the S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * S3 object key name.
         * </p>
         * 
         * @param name
         *        S3 object key name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * If the bucket is versioning enabled, you can specify the object version.
         * </p>
         * 
         * @param version
         *        If the bucket is versioning enabled, you can specify the object version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);
    }

    static final class BuilderImpl implements Builder {
        private String bucket;

        private String name;

        private String version;

        private BuilderImpl() {
        }

        private BuilderImpl(S3Object model) {
            bucket(model.bucket);
            name(model.name);
            version(model.version);
        }

        public final String getBucket() {
            return bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getVersion() {
            return version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public S3Object build() {
            return new S3Object(this);
        }
    }
}
