/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Structure containing details about the detected label, including the name, detected instances, parent labels, and
 * level of confidence.
 * </p>
 * <p>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Label implements SdkPojo, Serializable, ToCopyableBuilder<Label.Builder, Label> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Label::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Float> CONFIDENCE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .getter(getter(Label::confidence)).setter(setter(Builder::confidence))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Confidence").build()).build();

    private static final SdkField<List<Instance>> INSTANCES_FIELD = SdkField
            .<List<Instance>> builder(MarshallingType.LIST)
            .getter(getter(Label::instances))
            .setter(setter(Builder::instances))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Instances").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Instance> builder(MarshallingType.SDK_POJO)
                                            .constructor(Instance::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Parent>> PARENTS_FIELD = SdkField
            .<List<Parent>> builder(MarshallingType.LIST)
            .getter(getter(Label::parents))
            .setter(setter(Builder::parents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parents").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Parent> builder(MarshallingType.SDK_POJO)
                                            .constructor(Parent::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, CONFIDENCE_FIELD,
            INSTANCES_FIELD, PARENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Float confidence;

    private final List<Instance> instances;

    private final List<Parent> parents;

    private Label(BuilderImpl builder) {
        this.name = builder.name;
        this.confidence = builder.confidence;
        this.instances = builder.instances;
        this.parents = builder.parents;
    }

    /**
     * <p>
     * The name (label) of the object or scene.
     * </p>
     * 
     * @return The name (label) of the object or scene.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * Level of confidence.
     * </p>
     * 
     * @return Level of confidence.
     */
    public Float confidence() {
        return confidence;
    }

    /**
     * Returns true if the Instances property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasInstances() {
        return instances != null && !(instances instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If <code>Label</code> represents an object, <code>Instances</code> contains the bounding boxes for each instance
     * of the detected object. Bounding boxes are returned for common object labels such as people, cars, furniture,
     * apparel or pets.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasInstances()} to see if a value was sent in this field.
     * </p>
     * 
     * @return If <code>Label</code> represents an object, <code>Instances</code> contains the bounding boxes for each
     *         instance of the detected object. Bounding boxes are returned for common object labels such as people,
     *         cars, furniture, apparel or pets.
     */
    public List<Instance> instances() {
        return instances;
    }

    /**
     * Returns true if the Parents property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasParents() {
        return parents != null && !(parents instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The parent labels for a label. The response includes all ancestor labels.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasParents()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The parent labels for a label. The response includes all ancestor labels.
     */
    public List<Parent> parents() {
        return parents;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(confidence());
        hashCode = 31 * hashCode + Objects.hashCode(instances());
        hashCode = 31 * hashCode + Objects.hashCode(parents());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Label)) {
            return false;
        }
        Label other = (Label) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(confidence(), other.confidence())
                && Objects.equals(instances(), other.instances()) && Objects.equals(parents(), other.parents());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Label").add("Name", name()).add("Confidence", confidence()).add("Instances", instances())
                .add("Parents", parents()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Confidence":
            return Optional.ofNullable(clazz.cast(confidence()));
        case "Instances":
            return Optional.ofNullable(clazz.cast(instances()));
        case "Parents":
            return Optional.ofNullable(clazz.cast(parents()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Label, T> g) {
        return obj -> g.apply((Label) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Label> {
        /**
         * <p>
         * The name (label) of the object or scene.
         * </p>
         * 
         * @param name
         *        The name (label) of the object or scene.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Level of confidence.
         * </p>
         * 
         * @param confidence
         *        Level of confidence.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confidence(Float confidence);

        /**
         * <p>
         * If <code>Label</code> represents an object, <code>Instances</code> contains the bounding boxes for each
         * instance of the detected object. Bounding boxes are returned for common object labels such as people, cars,
         * furniture, apparel or pets.
         * </p>
         * 
         * @param instances
         *        If <code>Label</code> represents an object, <code>Instances</code> contains the bounding boxes for
         *        each instance of the detected object. Bounding boxes are returned for common object labels such as
         *        people, cars, furniture, apparel or pets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instances(Collection<Instance> instances);

        /**
         * <p>
         * If <code>Label</code> represents an object, <code>Instances</code> contains the bounding boxes for each
         * instance of the detected object. Bounding boxes are returned for common object labels such as people, cars,
         * furniture, apparel or pets.
         * </p>
         * 
         * @param instances
         *        If <code>Label</code> represents an object, <code>Instances</code> contains the bounding boxes for
         *        each instance of the detected object. Bounding boxes are returned for common object labels such as
         *        people, cars, furniture, apparel or pets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instances(Instance... instances);

        /**
         * <p>
         * If <code>Label</code> represents an object, <code>Instances</code> contains the bounding boxes for each
         * instance of the detected object. Bounding boxes are returned for common object labels such as people, cars,
         * furniture, apparel or pets.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Instance>.Builder} avoiding the need to
         * create one manually via {@link List<Instance>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Instance>.Builder#build()} is called immediately and its
         * result is passed to {@link #instances(List<Instance>)}.
         * 
         * @param instances
         *        a consumer that will call methods on {@link List<Instance>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instances(List<Instance>)
         */
        Builder instances(Consumer<Instance.Builder>... instances);

        /**
         * <p>
         * The parent labels for a label. The response includes all ancestor labels.
         * </p>
         * 
         * @param parents
         *        The parent labels for a label. The response includes all ancestor labels.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parents(Collection<Parent> parents);

        /**
         * <p>
         * The parent labels for a label. The response includes all ancestor labels.
         * </p>
         * 
         * @param parents
         *        The parent labels for a label. The response includes all ancestor labels.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parents(Parent... parents);

        /**
         * <p>
         * The parent labels for a label. The response includes all ancestor labels.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Parent>.Builder} avoiding the need to
         * create one manually via {@link List<Parent>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Parent>.Builder#build()} is called immediately and its
         * result is passed to {@link #parents(List<Parent>)}.
         * 
         * @param parents
         *        a consumer that will call methods on {@link List<Parent>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parents(List<Parent>)
         */
        Builder parents(Consumer<Parent.Builder>... parents);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Float confidence;

        private List<Instance> instances = DefaultSdkAutoConstructList.getInstance();

        private List<Parent> parents = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Label model) {
            name(model.name);
            confidence(model.confidence);
            instances(model.instances);
            parents(model.parents);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Float getConfidence() {
            return confidence;
        }

        @Override
        public final Builder confidence(Float confidence) {
            this.confidence = confidence;
            return this;
        }

        public final void setConfidence(Float confidence) {
            this.confidence = confidence;
        }

        public final Collection<Instance.Builder> getInstances() {
            return instances != null ? instances.stream().map(Instance::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder instances(Collection<Instance> instances) {
            this.instances = InstancesCopier.copy(instances);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instances(Instance... instances) {
            instances(Arrays.asList(instances));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instances(Consumer<Instance.Builder>... instances) {
            instances(Stream.of(instances).map(c -> Instance.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setInstances(Collection<Instance.BuilderImpl> instances) {
            this.instances = InstancesCopier.copyFromBuilder(instances);
        }

        public final Collection<Parent.Builder> getParents() {
            return parents != null ? parents.stream().map(Parent::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parents(Collection<Parent> parents) {
            this.parents = ParentsCopier.copy(parents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parents(Parent... parents) {
            parents(Arrays.asList(parents));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parents(Consumer<Parent.Builder>... parents) {
            parents(Stream.of(parents).map(c -> Parent.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setParents(Collection<Parent.BuilderImpl> parents) {
            this.parents = ParentsCopier.copyFromBuilder(parents);
        }

        @Override
        public Label build() {
            return new Label(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
