/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a body part detected by <a>DetectProtectiveEquipment</a> that contains PPE. An array of
 * <code>ProtectiveEquipmentBodyPart</code> objects is returned for each person detected by
 * <code>DetectProtectiveEquipment</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProtectiveEquipmentBodyPart implements SdkPojo, Serializable,
        ToCopyableBuilder<ProtectiveEquipmentBodyPart.Builder, ProtectiveEquipmentBodyPart> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ProtectiveEquipmentBodyPart::nameAsString)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Float> CONFIDENCE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("Confidence").getter(getter(ProtectiveEquipmentBodyPart::confidence)).setter(setter(Builder::confidence))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Confidence").build()).build();

    private static final SdkField<List<EquipmentDetection>> EQUIPMENT_DETECTIONS_FIELD = SdkField
            .<List<EquipmentDetection>> builder(MarshallingType.LIST)
            .memberName("EquipmentDetections")
            .getter(getter(ProtectiveEquipmentBodyPart::equipmentDetections))
            .setter(setter(Builder::equipmentDetections))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EquipmentDetections").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EquipmentDetection> builder(MarshallingType.SDK_POJO)
                                            .constructor(EquipmentDetection::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, CONFIDENCE_FIELD,
            EQUIPMENT_DETECTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Float confidence;

    private final List<EquipmentDetection> equipmentDetections;

    private ProtectiveEquipmentBodyPart(BuilderImpl builder) {
        this.name = builder.name;
        this.confidence = builder.confidence;
        this.equipmentDetections = builder.equipmentDetections;
    }

    /**
     * <p>
     * The detected body part.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link BodyPart#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The detected body part.
     * @see BodyPart
     */
    public BodyPart name() {
        return BodyPart.fromValue(name);
    }

    /**
     * <p>
     * The detected body part.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link BodyPart#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The detected body part.
     * @see BodyPart
     */
    public String nameAsString() {
        return name;
    }

    /**
     * <p>
     * The confidence that Amazon Rekognition has in the detection accuracy of the detected body part.
     * </p>
     * 
     * @return The confidence that Amazon Rekognition has in the detection accuracy of the detected body part.
     */
    public Float confidence() {
        return confidence;
    }

    /**
     * Returns true if the EquipmentDetections property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasEquipmentDetections() {
        return equipmentDetections != null && !(equipmentDetections instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of Personal Protective Equipment items detected around a body part.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEquipmentDetections()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of Personal Protective Equipment items detected around a body part.
     */
    public List<EquipmentDetection> equipmentDetections() {
        return equipmentDetections;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(confidence());
        hashCode = 31 * hashCode + Objects.hashCode(equipmentDetections());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProtectiveEquipmentBodyPart)) {
            return false;
        }
        ProtectiveEquipmentBodyPart other = (ProtectiveEquipmentBodyPart) obj;
        return Objects.equals(nameAsString(), other.nameAsString()) && Objects.equals(confidence(), other.confidence())
                && Objects.equals(equipmentDetections(), other.equipmentDetections());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ProtectiveEquipmentBodyPart").add("Name", nameAsString()).add("Confidence", confidence())
                .add("EquipmentDetections", equipmentDetections()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(nameAsString()));
        case "Confidence":
            return Optional.ofNullable(clazz.cast(confidence()));
        case "EquipmentDetections":
            return Optional.ofNullable(clazz.cast(equipmentDetections()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProtectiveEquipmentBodyPart, T> g) {
        return obj -> g.apply((ProtectiveEquipmentBodyPart) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProtectiveEquipmentBodyPart> {
        /**
         * <p>
         * The detected body part.
         * </p>
         * 
         * @param name
         *        The detected body part.
         * @see BodyPart
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BodyPart
         */
        Builder name(String name);

        /**
         * <p>
         * The detected body part.
         * </p>
         * 
         * @param name
         *        The detected body part.
         * @see BodyPart
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BodyPart
         */
        Builder name(BodyPart name);

        /**
         * <p>
         * The confidence that Amazon Rekognition has in the detection accuracy of the detected body part.
         * </p>
         * 
         * @param confidence
         *        The confidence that Amazon Rekognition has in the detection accuracy of the detected body part.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confidence(Float confidence);

        /**
         * <p>
         * An array of Personal Protective Equipment items detected around a body part.
         * </p>
         * 
         * @param equipmentDetections
         *        An array of Personal Protective Equipment items detected around a body part.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder equipmentDetections(Collection<EquipmentDetection> equipmentDetections);

        /**
         * <p>
         * An array of Personal Protective Equipment items detected around a body part.
         * </p>
         * 
         * @param equipmentDetections
         *        An array of Personal Protective Equipment items detected around a body part.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder equipmentDetections(EquipmentDetection... equipmentDetections);

        /**
         * <p>
         * An array of Personal Protective Equipment items detected around a body part.
         * </p>
         * This is a convenience that creates an instance of the {@link List<EquipmentDetection>.Builder} avoiding the
         * need to create one manually via {@link List<EquipmentDetection>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<EquipmentDetection>.Builder#build()} is called immediately
         * and its result is passed to {@link #equipmentDetections(List<EquipmentDetection>)}.
         * 
         * @param equipmentDetections
         *        a consumer that will call methods on {@link List<EquipmentDetection>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #equipmentDetections(List<EquipmentDetection>)
         */
        Builder equipmentDetections(Consumer<EquipmentDetection.Builder>... equipmentDetections);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Float confidence;

        private List<EquipmentDetection> equipmentDetections = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ProtectiveEquipmentBodyPart model) {
            name(model.name);
            confidence(model.confidence);
            equipmentDetections(model.equipmentDetections);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public final Builder name(BodyPart name) {
            this.name(name == null ? null : name.toString());
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Float getConfidence() {
            return confidence;
        }

        @Override
        public final Builder confidence(Float confidence) {
            this.confidence = confidence;
            return this;
        }

        public final void setConfidence(Float confidence) {
            this.confidence = confidence;
        }

        public final Collection<EquipmentDetection.Builder> getEquipmentDetections() {
            return equipmentDetections != null ? equipmentDetections.stream().map(EquipmentDetection::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder equipmentDetections(Collection<EquipmentDetection> equipmentDetections) {
            this.equipmentDetections = EquipmentDetectionsCopier.copy(equipmentDetections);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder equipmentDetections(EquipmentDetection... equipmentDetections) {
            equipmentDetections(Arrays.asList(equipmentDetections));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder equipmentDetections(Consumer<EquipmentDetection.Builder>... equipmentDetections) {
            equipmentDetections(Stream.of(equipmentDetections).map(c -> EquipmentDetection.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setEquipmentDetections(Collection<EquipmentDetection.BuilderImpl> equipmentDetections) {
            this.equipmentDetections = EquipmentDetectionsCopier.copyFromBuilder(equipmentDetections);
        }

        @Override
        public ProtectiveEquipmentBodyPart build() {
            return new ProtectiveEquipmentBodyPart(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
